/*
 * bltGrBar.c --
 *
 *	This module implements a elements in the graph widget
 *	for the Tk toolkit.
 *
 * Copyright 1991-1996 by AT&T Bell Laboratories.
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the
 * names of AT&T Bell Laboratories any of their entities not be used
 * in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * AT&T disclaims all warranties with regard to this software, including
 * all implied warranties of merchantability and fitness.  In no event
 * shall AT&T be liable for any special, indirect or consequential
 * damages or any damages whatsoever resulting from loss of use, data
 * or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or
 * performance of this software.
 *
 */

#include "bltGraph.h"
#include <ctype.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "bltGrElem.h"

static int BarModeParseProc _ANSI_ARGS_((ClientData, Tcl_Interp *, Tk_Window,
	char *, char *, int));
static char *BarModePrintProc _ANSI_ARGS_((ClientData, Tk_Window, char *, int,
	Tcl_FreeProc **));

Tk_CustomOption bltBarModeOption =
{
    BarModeParseProc, BarModePrintProc, (ClientData)0
};

extern Tk_CustomOption bltXDataOption;
extern Tk_CustomOption bltYDataOption;
extern Tk_CustomOption bltDataPairsOption;
extern Tk_CustomOption bltMapXAxisOption;
extern Tk_CustomOption bltMapYAxisOption;
extern Tk_CustomOption bltLengthOption;

#define DEF_NUM_BARS	8	/* Default size of the static array *defSpace*.
				 * Each entry is a bar of the element. If more
				 * bars are needed for an element, they must be
				 * malloc-ed. */

typedef struct {
    Graph *graphPtr;		/* Graph widget of element*/
    ElementType type;		/* Type of element is BAR_ELEMENT */
    int flags;			/* Indicates if the entire bar is active,
				 * or if coordinates need to be calculated */
    Tk_Uid nameId;		/* Identifier to refer the element.
				 * Used in the "insert", "delete", or
				 * "show", commands. */
    int mapped;			/* If non-zero, element is currently
				 * visible.*/
    Tk_ConfigSpec *configSpecs;	/* Configuration specifications */
    char *label;		/* Label displayed in legend */
    unsigned int axisMask;	/* Indicates which axes to map element's
				 * coordinates onto */
    Vector x, y;		/* Contains array of numeric values */
    DataStatus *statusArr;	/* Array of flags (malloc-ed) which
				 * indicate which data points are active
				 * (drawn with "active" colors. */
    int arraySize;		/* Number of active data points. Special
				 * case: if numActivePoints==0 and the active
				 * bit is set in "flags", then all data
				 * points are drawn active. */

    ElemConfigProc *configProc;
    ElemDestroyProc *destroyProc;
    ElemDrawProc *drawNormalProc;
    ElemDrawProc *drawActiveProc;
    ElemLimitsProc *limitsProc;
    ElemClosestProc *closestProc;
    ElemCoordsProc *coordsProc;
    ElemPrintProc *printNormalProc;
    ElemPrintProc *printActiveProc;
    ElemDrawSymbolsProc *drawSymbolsProc;
    ElemPrintSymbolsProc *printSymbolsProc;

    int numDataPoints;		/* Number of data points (x-y cooridinates)
				 * in the element.  This is the minimum
				 * size of the x and y vectors. */
    /*
     * Bar specific attributes
     */
    Tk_3DBorder border;		/* 3D border and background color */
    XColor *normalFg;
    GC normalGC;
    Pixmap normalPixmap;
    XColor *activeFg;
    GC activeGC;
    Pixmap activePixmap;

    int borderWidth;		/* 3D border width of bar */
    int relief;			/* Relief of the bar */


    XRectangle defSpace[DEF_NUM_BARS];
    XRectangle *segArr;		/* Array of rectangles comprising the bar
				 * segments of the element. By default, this
				 * is set to *defSpace*.  If more segments are
				 * needed, the array will point to malloc-ed
				 * memory */
    int numSegs;		/* Number of bars to be drawn */
    int padX;			/* Spacing on either side of bar */
    double barWidth;

} Bar;

#define DEF_BAR_ACTIVE_BG_COLOR "red"
#define DEF_BAR_ACTIVE_BG_MONO	WHITE
#define DEF_BAR_ACTIVE_FG_COLOR "pink"
#define DEF_BAR_ACTIVE_FG_MONO 	BLACK
#define DEF_BAR_BG_COLOR	"navyblue"
#define DEF_BAR_BG_MONO		BLACK
#define DEF_BAR_BORDERWIDTH	"2"
#define DEF_BAR_DATA		(char *)NULL
#define DEF_BAR_FG_COLOR     	"blue"
#define DEF_BAR_FG_MONO		WHITE
#define DEF_BAR_LABEL		(char *)NULL
#define DEF_BAR_MAPPED		"1"
#define DEF_BAR_RELIEF		"raised"
#define DEF_BAR_NORMAL_STIPPLE	""
#define DEF_BAR_ACTIVE_STIPPLE	""
#define DEF_BAR_AXIS_X		"x"
#define DEF_BAR_X_DATA		(char *)NULL
#define DEF_BAR_AXIS_Y		"y"
#define DEF_BAR_Y_DATA		(char *)NULL
#define DEF_BAR_WIDTH		"0.0"

static Tk_ConfigSpec configSpecs[] =
{
    {TK_CONFIG_COLOR, "-activeforeground", "elemActiveForeground", "Foreground",
	DEF_BAR_ACTIVE_FG_COLOR, Tk_Offset(Bar, activeFg),
	TK_CONFIG_COLOR_ONLY | TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-activeforeground", "elemActiveForeground", "Foreground",
	DEF_BAR_ACTIVE_FG_MONO, Tk_Offset(Bar, activeFg),
	TK_CONFIG_MONO_ONLY | TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-activestipple", "elemActiveStipple", "Stipple",
	DEF_BAR_ACTIVE_STIPPLE, Tk_Offset(Bar, activePixmap), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BORDER, "-background", "elemBackground", "Background",
	DEF_BAR_BG_COLOR, Tk_Offset(Bar, border), TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-background", "elemBackground", "Background",
	DEF_BAR_BG_COLOR, Tk_Offset(Bar, border), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_DOUBLE, "-barwidth", "elemBarWidth", "BarWidth",
	DEF_BAR_WIDTH, Tk_Offset(Bar, barWidth), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_SYNONYM, "-bd", "elemBorderwidth", (char *)NULL,
	(char *)NULL, 0, 0},
    {TK_CONFIG_SYNONYM, "-bg", "elemBackground", (char *)NULL,
	(char *)NULL, 0, 0},
    {TK_CONFIG_CUSTOM, "-borderwidth", "elemBorderwidth", "Borderwidth",
	DEF_BAR_BORDERWIDTH, Tk_Offset(Bar, borderWidth), 0, &bltLengthOption},
    {TK_CONFIG_SYNONYM, "-fg", "elemForeground", (char *)NULL,
	(char *)NULL, 0, 0},
    {TK_CONFIG_CUSTOM, "-data", "elemData", "Data",
	(char *)NULL, 0, 0, &bltDataPairsOption},
    {TK_CONFIG_COLOR, "-foreground", "elemForeground", "Foreground",
	DEF_BAR_FG_COLOR, Tk_Offset(Bar, normalFg),
	TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_COLOR, "-foreground", "elemForeground", "Foreground",
	DEF_BAR_FG_COLOR, Tk_Offset(Bar, normalFg),
	TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_STRING, "-label", "elemLabel", "Label",
	DEF_BAR_LABEL, Tk_Offset(Bar, label), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BOOLEAN, "-mapped", "elemMapped", "Mapped",
	DEF_BAR_MAPPED, Tk_Offset(Bar, mapped), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-mapx", "elemMapX", "MapX",
	DEF_BAR_AXIS_X, Tk_Offset(Bar, axisMask),
	TK_CONFIG_DONT_SET_DEFAULT, &bltMapXAxisOption},
    {TK_CONFIG_CUSTOM, "-mapy", "elemMapY", "MapY",
	DEF_BAR_AXIS_Y, Tk_Offset(Bar, axisMask),
	TK_CONFIG_DONT_SET_DEFAULT, &bltMapYAxisOption},
    {TK_CONFIG_RELIEF, "-relief", "elemRelief", "Relief",
	DEF_BAR_RELIEF, Tk_Offset(Bar, relief), 0},
    {TK_CONFIG_BITMAP, "-stipple", "elemStipple", "Stipple",
	DEF_BAR_NORMAL_STIPPLE, Tk_Offset(Bar, normalPixmap), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-xdata", "elemXdata", "Xdata",
	DEF_BAR_X_DATA, 0, 0, &bltXDataOption},
    {TK_CONFIG_CUSTOM, "-ydata", "elemYdata", "Ydata",
	DEF_BAR_Y_DATA, 0, 0, &bltYDataOption},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

static INLINE int
Abs(x)
    register int x;
{
    return ((x < 0) ? -x : x);
}

/* ----------------------------------------------------------------------
 * Custom option parse and print procedures
 * ----------------------------------------------------------------------
 */
/*
 * ----------------------------------------------------------------------
 *
 * NameOfBarMode --
 *
 *	Converts the integer representing the mode style into a string.
 *
 * ----------------------------------------------------------------------
 */
static char *
NameOfBarMode(mode)
    BarMode mode;
{
    switch (mode) {
    case MODE_NORMAL:
	return "normal";
    case MODE_STACKED:
	return "stacked";
    case MODE_ALIGNED:
	return "aligned";
    default:
	return "unknown mode value";
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ModeParseProc --
 *
 *	Converts the mode string into its numeric representation.
 *
 *	Valid mode strings are:
 *
 *      "normal"    Draw a full bar at each point in the element.
 *
 * 	"stacked"   Stack bar segments vertically. Each stack is defined
 *		    by each ordinate at a particular abscissa. The height
 *		    of each segment is represented by the sum the previous
 *		    ordinates.
 *
 *	"aligned"   Align bar segments as smaller slices one next to
 *		    the other.  Like "stacks", aligned segments are
 *		    defined by each ordinate at a particular abscissa.
 *
 * Results:
 *	A standard Tcl result.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
BarModeParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* not used */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* not used */
    char *value;		/* Mode style string */
    char *widgRec;		/* Cubicle structure record */
    int offset;			/* Offset of style in record */
{
    BarMode *modePtr = (BarMode *)(widgRec + offset);
    unsigned int length;
    char c;

    c = value[0];
    length = strlen(value);
    if ((c == 'n') && (strncmp(value, "normal", length) == 0)) {
	*modePtr = MODE_NORMAL;
    } else if ((c == 's') && (strncmp(value, "stacked", length) == 0)) {
	*modePtr = MODE_STACKED;
    } else if ((c == 'a') && (strncmp(value, "aligned", length) == 0)) {
	*modePtr = MODE_ALIGNED;
    } else {
	Tcl_AppendResult(interp, "bad mode argument \"", value,
	    "\": should be \"normal\", \"stacked\", or \"aligned\"",
	    (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * BarModePrintProc --
 *
 *	Returns the mode style string based upon the mode flags.
 *
 * Results:
 *	The mode style string is returned.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
BarModePrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* not used */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* Row/column structure record */
    int offset;			/* Offset of mode in Partition record */
    Tcl_FreeProc **freeProcPtr;	/* not used */
{
    BarMode mode = *(BarMode *)(widgRec + offset);

    return (NameOfBarMode(mode));
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureBar --
 *
 *	Sets up the appropriate configuration parameters in the GC.
 *      It is assumed the parameters have been previously set by
 *	a call to Tk_ConfigureWidget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information such as bar foreground/background
 *	color and stipple etc. get set in a new GC.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ConfigureBar(graphPtr, elemPtr)
    Graph *graphPtr;
    register Element *elemPtr;
{
    XGCValues gcValues;
    unsigned long gcMask;
    Bar *barPtr = (Bar *)elemPtr;
    XColor *colorPtr;
    GC newGC;

    gcMask = GCForeground | GCBackground;
    colorPtr = (barPtr->activeFg != NULL) ? barPtr->activeFg : barPtr->normalFg;
    gcValues.foreground = colorPtr->pixel;
    gcValues.background = (Tk_3DBorderColor(barPtr->border))->pixel;
    if (barPtr->activePixmap != None) {
	gcValues.stipple = barPtr->activePixmap;
	gcValues.fill_style = FillOpaqueStippled;
	gcMask |= (GCStipple | GCFillStyle);
    }
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (barPtr->activeGC != NULL) {
	Tk_FreeGC(graphPtr->display, barPtr->activeGC);
    }
    barPtr->activeGC = newGC;

    gcMask = GCForeground | GCBackground;
    gcValues.foreground = barPtr->normalFg->pixel;
    if (barPtr->normalPixmap != None) {
	gcValues.stipple = barPtr->normalPixmap;
	gcValues.fill_style = FillOpaqueStippled;
	gcMask |= (GCStipple | GCFillStyle);
    }
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (barPtr->normalGC != NULL) {
	Tk_FreeGC(graphPtr->display, barPtr->normalGC);
    }
    barPtr->normalGC = newGC;
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * ComputeLimits --
 *
 *	Returns the limits of the bar data
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ComputeLimits(graphPtr, elemPtr, axisPtr, minPtr, maxPtr)
    Graph *graphPtr;
    Element *elemPtr;
    GraphAxis *axisPtr;		/* Axis information */
    double *minPtr, *maxPtr;
{
    Bar *barPtr = (Bar *)elemPtr;
    Vector *vecPtr;
    double min, max;

    min = bltPosInfinity, max = bltNegInfinity;
    if ((barPtr->x.length < 1) || (barPtr->y.length < 1)) {
	return 0;		/* No data points */
    }
    if (!(elemPtr->axisMask & AXIS_MASK(axisPtr))) {
	return 0;		/* Bar is not mapped to this axis */
    }
    if (AXIS_MASK(axisPtr) & AXIS_MASK_X) {
	double middle;

	vecPtr = &(barPtr->x);
	/* This doesn't help much if the width of the bars on the
	 * ends have been configured differently. But I don't know
	 * that information until I lay out the bar elements */
	middle = graphPtr->barWidth * 0.5;
	min = vecPtr->min - middle;
	max = vecPtr->max + middle;

	/* Warning: You get what you deserve if the x-axis is logScale */
	if (axisPtr->logScale) {
	    min = Blt_FindVectorMinimum(vecPtr, DBL_MIN) + middle;
	}
    } else {
	vecPtr = &(barPtr->y);
	min = barPtr->y.min;
	if ((min <= 0.0) && (axisPtr->logScale)) {
	    min = Blt_FindVectorMinimum(vecPtr, DBL_MIN);
	} else {
	    if (min > 0.0) {
		min = 0.0;
	    }
	}
	max = barPtr->y.max;
	if (max < 0.0) {
	    max = 0.0;
	}
    }
    *minPtr = min, *maxPtr = max;
    return (vecPtr->length);
}

/*
 * ----------------------------------------------------------------------
 *
 * ClosestBar --
 *
 *	Find the bar segment closest to the window coordinates	point
 *	specified.
 *
 *	Note:  This does not return the height of the stacked segment
 *	       (in graph coordinates) properly.
 *
 * Results:
 *	Returns 1 if the point is width any bar segment, otherwise 0.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
ClosestBar(graphPtr, elemPtr, infoPtr)
    Graph *graphPtr;		/* Graph widget record */
    Element *elemPtr;		/* Bar element */
    ClosestInfo *infoPtr;	/* Index of closest point in element */
{
    Bar *barPtr = (Bar *)elemPtr;
    double dummy;
    double dist, prevMin;
    XPoint pointArr[5], *pointPtr;
    register int i, side;
    register XRectangle *rectPtr;

    prevMin = infoPtr->dist;
    rectPtr = barPtr->segArr;
    for (i = 0; i < barPtr->numDataPoints; i++) {
	if (IsClipped(barPtr->statusArr[i])) {
	    continue;		/* Ignore bar segments that aren't visible */
	}
	if ((infoPtr->x >= rectPtr->x) &&
	    (infoPtr->x <= (rectPtr->x + rectPtr->width)) &&
	    (infoPtr->y >= rectPtr->y) &&
	    (infoPtr->y <= (rectPtr->y + rectPtr->height))) {

	    /* The point is interior to this bar segment. */

	    infoPtr->index = i;
	    infoPtr->dist = 0.0;
	    break;
	}
	pointArr[4].x = pointArr[3].x = pointArr[0].x = rectPtr->x;
	pointArr[4].y = pointArr[1].y = pointArr[0].y = rectPtr->y;
	pointArr[2].x = pointArr[1].x = rectPtr->x + rectPtr->width;
	pointArr[3].y = pointArr[2].y = rectPtr->y + rectPtr->height;

	pointPtr = pointArr;
	for (side = 0; side < 4; side++) {
	    dist = Blt_GetDistanceToSegment(pointPtr, infoPtr->x, infoPtr->y,
		&dummy, &dummy);
	    if (dist < infoPtr->dist) {
		infoPtr->index = i;
		infoPtr->dist = dist;
	    }
	    pointPtr++;
	}
	rectPtr++;
    }
    if (infoPtr->dist < prevMin) {
	infoPtr->elemId = elemPtr->nameId;
	/* Provide the segment value, not the stacked sum */
	infoPtr->closest.x = barPtr->x.data[infoPtr->index];
	infoPtr->closest.y = barPtr->y.data[infoPtr->index];
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * ComputeBarCoordinates --
 *
 *	Calculates the actual window coordinates of the bar element.
 *	The window coordinates are saved in the bar element structure.
 *
 * Results:
 *	None.
 *
 * Notes:
 *	A bar can have multiple segments (more than one x,y pairs).
 *	In this case, the bar can be represented as either a set of
 *	non-contiguous bars or a single multi-segmented (stacked) bar.
 *
 *	The x-axis layout for a barchart may be presented in one of
 *	two ways.  If abscissas are used, the bars are placed at those
 *	coordinates.  Otherwise, the range will represent the number
 *	of values.
 *
 * ----------------------------------------------------------------------
 */
static void
ComputeBarCoordinates(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    AxisPair axes;
    Bar *barPtr = (Bar *)elemPtr;
    int numDataPoints;
    double x1, y1, x2, y2;	/* Two opposite corners of the rectangle
				 * in graph coordinates. */
    XPoint corner1, corner2;
    double barWidth;
    FreqKey key;
    register XRectangle *rectPtr;
    register int i, count;

    if ((barPtr->x.length < 1) || (barPtr->y.length < 1)) {
	return;			/* No data points */
    }
    if (barPtr->segArr != barPtr->defSpace) {
	free((char *)barPtr->segArr);
    }
    barPtr->numDataPoints = barPtr->numSegs = 0;
    barPtr->segArr = barPtr->defSpace;
    numDataPoints = BLT_MIN(barPtr->x.length, barPtr->y.length);
    if (numDataPoints > DEF_NUM_BARS) {
	XRectangle *segArr;

	segArr = (XRectangle *)malloc(numDataPoints * sizeof(XRectangle));
	if (segArr == NULL) {
	    return;		/* Can't allocated bar array */
	}
	barPtr->segArr = segArr;
    }
    if (numDataPoints != elemPtr->arraySize) {
	Blt_ResizeStatusArray(elemPtr, numDataPoints);
    }
    Blt_GetAxisMapping(graphPtr, barPtr->axisMask, &axes);
    barWidth = graphPtr->barWidth;
    if (barPtr->barWidth > 0.0) {
	barWidth = barPtr->barWidth;
    }
    key.dummy = 0;
    rectPtr = barPtr->segArr;
    count = 0;
    for (i = 0; i < numDataPoints; i++) {
	SetClipped(barPtr->statusArr[i]);	/* Turn off visibility */
	if (((barPtr->x.data[i] - barWidth) > axes.x->max) ||
	    ((barPtr->x.data[i] + barWidth) < axes.x->min)) {
	    continue;		/* Abscissa is out of range of the x-axis */
	}
	x1 = barPtr->x.data[i] - (barWidth * 0.5);
	x2 = x1 + barWidth;
	y1 = barPtr->y.data[i];
	y2 = 0.0;

	if ((graphPtr->numStacks > 0) && (graphPtr->mode != MODE_NORMAL)) {
	    Tcl_HashEntry *hPtr;

	    key.value = barPtr->x.data[i];
	    key.mask = barPtr->axisMask;
	    hPtr = Tcl_FindHashEntry(&(graphPtr->freqTable), (char *)&key);
	    if (hPtr != NULL) {
		FreqInfo *infoPtr;

		infoPtr = (FreqInfo *)Tcl_GetHashValue(hPtr);
		if (graphPtr->mode == MODE_STACKED) {
		    y2 = infoPtr->lastY;
		    y1 += y2;
		    infoPtr->lastY = y1;
		} else {	/* MODE_ALIGNED */
		    double slice;

		    slice = barWidth / (double)infoPtr->freq;
		    x1 += (slice * infoPtr->count);
		    x2 = x1 + slice;
		    infoPtr->count++;
		}
	    }
	}
	if (y1 < y2) {
	    double temp;

	    /* Handle negative bar values by swapping ordinates */
	    temp = y1, y1 = y2, y2 = temp;
	}
	/*
	 * Get the two corners of the bar segment and compute the rectangle
	 */
	corner1 = Blt_TransformPt(graphPtr, x1, y1, &axes);
	corner2 = Blt_TransformPt(graphPtr, x2, y2, &axes);

	/* Clip the bars vertically at the size of the graph window */
	if (corner1.y < 0) {
	    corner1.y = 0;
	} else if (corner1.y > graphPtr->height) {
	    corner1.y = graphPtr->height;
	}
	if (corner2.y < 0) {
	    corner2.y = 0;
	} else if (corner2.y > graphPtr->height) {
	    corner2.y = graphPtr->height;
	}
	rectPtr->x = BLT_MIN(corner1.x, corner2.x);
	rectPtr->y = BLT_MIN(corner1.y, corner2.y);
	rectPtr->width = Abs(corner1.x - corner2.x);
	rectPtr->height = Abs(corner1.y - corner2.y);
	SetVisible(barPtr->statusArr[i]);
	rectPtr++;
	count++;
    }
    barPtr->numSegs = count;
    barPtr->numDataPoints = numDataPoints;
}

/*
 * -----------------------------------------------------------------
 *
 * DrawSymbols --
 *
 * 	Draw a symbol centered at the given x,y window coordinate
 *	based upon the element symbol type and size.
 *
 * Results:
 *	None.
 *
 * Problems:
 *	Most notable is the round-off errors generated when
 *	calculating the centered position of the symbol.
 * -----------------------------------------------------------------
 */
/*ARGSUSED*/
static void
DrawSymbols(graphPtr, elemPtr, size, pointArr, numPoints, active)
    Graph *graphPtr;
    Element *elemPtr;
    int size;
    XPoint *pointArr;
    int numPoints;
    int active;			/* unused */
{
    Bar *barPtr = (Bar *)elemPtr;
    int radius;
    register int i;

    radius = (size / 2);
    size--;
    for (i = 0; i < numPoints; i++) {
	XFillRectangle(graphPtr->display, graphPtr->pixwin, barPtr->normalGC,
	    pointArr[i].x - radius, pointArr[i].y - radius, size, size);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * DrawNormalBar --
 *
 *	Draws the rectangle representing the bar element.  If the
 *	relief option is set to "raised" or "sunken" and the bar
 *	borderwidth is set (borderwidth > 0), a 3D border is drawn
 *	around the bar.
 *
 *	Don't draw bars that aren't visible (i.e. within the limits
 *	of the axis).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	X drawing commands are output.
 *
 * ----------------------------------------------------------------------
 */
static void
DrawNormalBar(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    Bar *barPtr = (Bar *)elemPtr;

    XFillRectangles(graphPtr->display, graphPtr->pixwin, barPtr->normalGC,
	barPtr->segArr, barPtr->numSegs);
    if ((barPtr->borderWidth > 0) && (barPtr->relief != TK_RELIEF_FLAT)) {
	register XRectangle *rectPtr;
	register int i;
	int twiceBW;

	twiceBW = (2 * barPtr->borderWidth);
	rectPtr = barPtr->segArr;
	for (i = 0; i < barPtr->numSegs; i++) {
	    if ((twiceBW < rectPtr->width) && (twiceBW < rectPtr->height)) {
#if (TK_MAJOR_VERSION == 3)
		Tk_Draw3DRectangle(graphPtr->display, graphPtr->pixwin,
		    barPtr->border, rectPtr->x, rectPtr->y,
		    rectPtr->width, rectPtr->height,
		    barPtr->borderWidth, barPtr->relief);
#else
		Tk_Draw3DRectangle(graphPtr->tkwin, graphPtr->pixwin,
		    barPtr->border, rectPtr->x, rectPtr->y,
		    rectPtr->width, rectPtr->height,
		    barPtr->borderWidth, barPtr->relief);
#endif /* TK_MAJOR_VERSION == 3 */
	    }
	    rectPtr++;
	}
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * DrawActiveBar --
 *
 *	Draws the rectangle representing the bar element.  If the
 *	-relief option is set to "raised" or "sunken" and the bar
 *	borderwidth is set (borderwidth > 0), a 3D border is drawn
 *	around the bar.
 *
 *	Don't draw bars that aren't visible (i.e. within the limits
 *	of the axis).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	X drawing commands are output.
 *
 * ----------------------------------------------------------------------
 */
static void
DrawActiveBar(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    Bar *barPtr = (Bar *)elemPtr;
    int twiceBW;
    register XRectangle *rectPtr;
    register int i;

    twiceBW = (2 * barPtr->borderWidth);
    rectPtr = barPtr->segArr;
    for (i = 0; i < barPtr->numDataPoints; i++) {
	if (IsClipped(barPtr->statusArr[i])) {
	    continue;		/* Data point isn't visible */
	}
	if (IsActive(barPtr->statusArr[i])) {
	    XFillRectangle(graphPtr->display, graphPtr->pixwin,
		barPtr->activeGC, rectPtr->x, rectPtr->y,
		rectPtr->width, rectPtr->height);
	    /*
	     * Don't draw a 3D border if it will cover the surface of the bar
	     */
	    if ((barPtr->borderWidth > 0) &&
		(barPtr->relief != TK_RELIEF_FLAT) &&
		(twiceBW < rectPtr->width) && (twiceBW < rectPtr->height)) {
#if (TK_MAJOR_VERSION == 3)
		Tk_Draw3DRectangle(graphPtr->display, graphPtr->pixwin,
		    barPtr->border, rectPtr->x, rectPtr->y,
		    rectPtr->width, rectPtr->height,
		    barPtr->borderWidth, barPtr->relief);
#else
		Tk_Draw3DRectangle(graphPtr->tkwin, graphPtr->pixwin,
		    barPtr->border, rectPtr->x, rectPtr->y,
		    rectPtr->width, rectPtr->height,
		    barPtr->borderWidth, barPtr->relief);
#endif /* TK_MAJOR_VERSION == 3 */
	    }
	}
	rectPtr++;
    }
}

/*
 * -----------------------------------------------------------------
 *
 * PrintSymbols --
 *
 * 	Draw a symbol centered at the given x,y window coordinate
 *	based upon the element symbol type and size.
 *
 * Results:
 *	None.
 *
 * Problems:
 *	Most notable is the round-off errors generated when
 *	calculating the centered position of the symbol.
 *
 * -----------------------------------------------------------------
 */
/*ARGSUSED*/
static void
PrintSymbols(graphPtr, elemPtr, size, pointArr, numPoints, active)
    Graph *graphPtr;
    Element *elemPtr;
    int size;
    XPoint *pointArr;
    int numPoints;
    int active;
{
    Bar *barPtr = (Bar *)elemPtr;
    Pixmap stipple;
    register int i;

    stipple = (active) ? barPtr->activePixmap : barPtr->normalPixmap;
    if (stipple != None) {
	int width, height;

	Blt_BackgroundToPostScript(graphPtr, Tk_3DBorderColor(barPtr->border));
	Tcl_AppendResult(graphPtr->interp, "/StippleProc {\ngsave\n",
	    (char *)NULL);
	Tk_SizeOfBitmap(graphPtr->display, stipple, &width, &height);
	Blt_StippleToPostScript(graphPtr, stipple, width, height, 1);
	Tcl_AppendResult(graphPtr->interp, "} def\n", (char *)NULL);
    } else {
	Blt_ForegroundToPostScript(graphPtr, barPtr->normalFg);
    }
    for (i = 0; i < numPoints; i++) {
	sprintf(graphPtr->scratchArr, "%d %d %d Sq\n", pointArr[i].x,
	    pointArr[i].y, size);
	Tcl_AppendResult(graphPtr->interp, graphPtr->scratchArr, (char *)NULL);
	if (stipple != None) {
	    Tcl_AppendResult(graphPtr->interp, "gsave\n", (char *)NULL);
	    Blt_ForegroundToPostScript(graphPtr, barPtr->normalFg);
	    Tcl_AppendResult(graphPtr->interp,
		"/StippleProc cvx exec\ngrestore\n", (char *)NULL);
	}
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * PrintActiveBar --
 *
 *	Similar to the DrawElement procedure, prints PostScript
 *	related commands to form rectangle representing the bar
 *	element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript pen width, dashes, and color settings are changed.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
PrintBar(graphPtr, barPtr, fgColorPtr, stipple)
    Graph *graphPtr;
    Bar *barPtr;
    XColor *fgColorPtr;
    Pixmap stipple;
{
    register XRectangle *rectPtr;
    register int i;

    rectPtr = barPtr->segArr;
    for (i = 0; i < barPtr->numDataPoints; i++) {
	if (IsClipped(barPtr->statusArr[i])) {
	    continue;
	}
	if (stipple != None) {
	    int width, height;

	    Blt_BackgroundToPostScript(graphPtr,
		Tk_3DBorderColor(barPtr->border));
	    Blt_RectangleToPostScript(graphPtr, rectPtr->x, rectPtr->y,
		(int)rectPtr->width, (int)rectPtr->height);
	    Tk_SizeOfBitmap(graphPtr->display, stipple, &width, &height);
	    Blt_ForegroundToPostScript(graphPtr, fgColorPtr);
	    Blt_StippleToPostScript(graphPtr, stipple, width, height, True);
	} else {
	    Blt_ForegroundToPostScript(graphPtr, fgColorPtr);
	    Blt_RectangleToPostScript(graphPtr, rectPtr->x, rectPtr->y,
		(int)rectPtr->width, (int)rectPtr->height);
	}
	if ((barPtr->borderWidth > 0) &&
	    (barPtr->relief != TK_RELIEF_FLAT)) {
	    Blt_Print3DRectangle(graphPtr, barPtr->border, rectPtr->x,
		rectPtr->y, (int)rectPtr->width, (int)rectPtr->height,
		barPtr->borderWidth, barPtr->relief);
	}
	rectPtr++;
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * PrintActiveBar --
 *
 *	Similar to the DrawElement procedure, prints PostScript
 *	related commands to form rectangle representing the bar
 *	element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript pen width, dashes, and color settings are changed.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
PrintActiveBar(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    Bar *barPtr = (Bar *)elemPtr;

    PrintBar(graphPtr, barPtr, barPtr->activeFg, barPtr->activePixmap);
}

/*
 * ----------------------------------------------------------------------
 *
 * PrintNormalBar --
 *
 *	Similar to the DrawElement procedure, prints PostScript
 *	related commands to form rectangle representing the bar
 *	element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	PostScript pen width, dashes, and color settings are changed.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
PrintNormalBar(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    Bar *barPtr = (Bar *)elemPtr;

    PrintBar(graphPtr, barPtr, barPtr->normalFg, barPtr->normalPixmap);
}

/*
 * ----------------------------------------------------------------------
 *
 * DestroyBar --
 *
 *	Release memory and resources allocated for the bar element.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the bar element is freed up.
 *
 * ----------------------------------------------------------------------
 */
static void
DestroyBar(graphPtr, elemPtr)
    Graph *graphPtr;
    Element *elemPtr;
{
    Bar *barPtr = (Bar *)elemPtr;

    Tk_FreeOptions(barPtr->configSpecs, (char *)barPtr, graphPtr->display, 0);

    if (barPtr->normalGC != NULL) {
	Tk_FreeGC(graphPtr->display, barPtr->normalGC);
    }
    if (barPtr->activeGC != NULL) {
	Tk_FreeGC(graphPtr->display, barPtr->activeGC);
    }
    if (barPtr->segArr != barPtr->defSpace) {
	free((char *)barPtr->segArr);
    }
    if (barPtr->x.clientId != NULL) {
	Blt_FreeVectorId(barPtr->x.clientId);
    } else if (barPtr->x.data != NULL) {
	free((char *)barPtr->x.data);
    }
    if (barPtr->y.clientId != NULL) {
	Blt_FreeVectorId(barPtr->y.clientId);
    } else if (barPtr->y.data != NULL) {
	free((char *)barPtr->y.data);
    }
    if (barPtr->statusArr != NULL) {
	free((char *)barPtr->statusArr);
    }
    free((char *)barPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_BarElement --
 *
 *	Allocate memory and initialize methods for the new bar element.
 *
 * Results:
 *	The pointer to the newly allocated element structure is returned.
 *
 * Side effects:
 *	Memory is allocated for the bar element structure.
 *
 * ----------------------------------------------------------------------
 */
Element *
Blt_BarElement()
{
    register Bar *barPtr;

    barPtr = (Bar *)calloc(1, sizeof(Bar));
    if (barPtr == NULL) {
	panic("can't allocate a bar element structure");
    }
    barPtr->configSpecs = configSpecs;
    barPtr->configProc = ConfigureBar;
    barPtr->destroyProc = DestroyBar;
    barPtr->drawNormalProc = DrawNormalBar;
    barPtr->drawActiveProc = DrawActiveBar;
    barPtr->limitsProc = ComputeLimits;
    barPtr->closestProc = ClosestBar;
    barPtr->coordsProc = ComputeBarCoordinates;
    barPtr->printNormalProc = PrintNormalBar;
    barPtr->printActiveProc = PrintActiveBar;
    barPtr->drawSymbolsProc = DrawSymbols;
    barPtr->printSymbolsProc = PrintSymbols;
    barPtr->type = ELEM_BAR;
    barPtr->relief = TK_RELIEF_RAISED;
    barPtr->borderWidth = 2;
    barPtr->numDataPoints = barPtr->numSegs = 0;
    barPtr->segArr = barPtr->defSpace;
    return ((Element *)barPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_InitFreqTable --
 *
 *	Generate a table of abscissa frequencies.  Duplicate
 *	x-coordinates (depending upon the bar drawing mode) indicate
 *	that something special should be done with each bar segment
 *	mapped to the same abscissa (i.e. it should be stacked,
 *	aligned, or overlay-ed with other segments)
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Memory is allocated for the bar element structure.
 *
 * ----------------------------------------------------------------------
 */
void
Blt_InitFreqTable(graphPtr)
    Graph *graphPtr;
{
    register Element *elemPtr;
    Blt_ListItem *itemPtr;
    Tcl_HashEntry *hPtr;
    Tcl_HashSearch cursor;
    Bar *barPtr;
    register int i;
    int isNew;
    int count;
    int numStacks, numSegs;
    int numDataPoints;
    FreqKey key, *keyPtr;

    /*
     * Free resources associated with a previous frequency table. This
     * includes the array of frequency information and the table itself
     */
    if (graphPtr->freqArr != NULL) {
	free((char *)graphPtr->freqArr);
	graphPtr->freqArr = NULL;
    }
    if (graphPtr->numStacks > 0) {
	Tcl_DeleteHashTable(&(graphPtr->freqTable));
	graphPtr->numStacks = 0;
    }
    if (graphPtr->mode == MODE_NORMAL) {
	return;			/* No frequency table is needed for
				 * normal mode */
    }
    /*
     * Re-initialize the table and fill it with unique abscissas.
     * Keep track of the frequency of each x-coordinate and how many
     * abscissas have duplicate mappings.
     */
    Tcl_InitHashTable(&(graphPtr->freqTable), sizeof(FreqKey) / sizeof(int));
    numSegs = numStacks = 0;
    key.dummy = 0;
    for (itemPtr = Blt_FirstListItem(&(graphPtr->elemList));
	itemPtr != NULL; itemPtr = Blt_NextItem(itemPtr)) {
	elemPtr = (Element *)Blt_GetItemValue(itemPtr);
	if ((!elemPtr->mapped) || (elemPtr->type != ELEM_BAR)) {
	    continue;
	}
	numSegs++;
	barPtr = (Bar *)elemPtr;
	numDataPoints = BLT_MIN(barPtr->x.length, barPtr->y.length);
	for (i = 0; i < numDataPoints; i++) {
	    key.value = barPtr->x.data[i];
	    key.mask = barPtr->axisMask;
	    hPtr = Tcl_CreateHashEntry(&(graphPtr->freqTable), (char *)&key,
		&isNew);
	    if (hPtr == NULL) {
		panic("can't allocate freqTable entry");
	    }
	    if (isNew) {
		count = 1;
	    } else {
		count = (int)Tcl_GetHashValue(hPtr);
		if (count == 1) {
		    numStacks++;
		}
		count++;
	    }
	    Tcl_SetHashValue(hPtr, (ClientData)count);
	}
    }
    if (numSegs == 0) {
	return;			/* No bar elements to be displayed */
    }
    if (numStacks > 0) {
	FreqInfo *freqArr, *infoPtr;

	/*
	 * Prune out singleton entries.  The only consideration here is
	 * memory consumption.
	 */
	for (hPtr = Tcl_FirstHashEntry(&(graphPtr->freqTable), &cursor);
	    hPtr != NULL; hPtr = Tcl_NextHashEntry(&cursor)) {
	    count = (int)Tcl_GetHashValue(hPtr);
	    keyPtr = (FreqKey *)Tcl_GetHashKey(&(graphPtr->freqTable), hPtr);
	    if (count == 1) {
		Tcl_DeleteHashEntry(hPtr);
	    }
	}
	freqArr = (FreqInfo *)calloc(numStacks, sizeof(FreqInfo));
	if (freqArr == NULL) {
	    panic("can't allocate dup info array");
	}
	/*
	 * Convert the data of each hash entry into an information
	 * structure.  Save the frequency count.
	 */
	infoPtr = freqArr;
	for (hPtr = Tcl_FirstHashEntry(&(graphPtr->freqTable), &cursor);
	    hPtr != NULL; hPtr = Tcl_NextHashEntry(&cursor)) {
	    count = (int)Tcl_GetHashValue(hPtr);
	    infoPtr->freq = count;
	    keyPtr = (FreqKey *)Tcl_GetHashKey(&(graphPtr->freqTable), hPtr);
	    infoPtr->mask = keyPtr->mask;
	    Tcl_SetHashValue(hPtr, (ClientData)infoPtr);
	    infoPtr++;
	}
	graphPtr->freqArr = freqArr;

    } else {
	Tcl_DeleteHashTable(&(graphPtr->freqTable));
    }
    graphPtr->numStacks = numStacks;
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_ComputeStacks --
 *
 *	Determine the height of each stack of bar segments.  A stack
 *	is created by designating two or more points with the same
 *	abscissa.  Each ordinate defines the height of a segment in
 *	the stack.  This procedure simply looks at all the data points
 *	summing the heights of each stacked segment. The sum is saved
 *	in the frequency information table.  This value will be used
 *	to calculate the y-axis limits (data limits aren't sufficient).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The heights of each stack is computed. Blt_CheckStacks will
 *	use this information to adjust the y-axis limits if necessary.
 *
 * ----------------------------------------------------------------------
 */
void
Blt_ComputeStacks(graphPtr)
    Graph *graphPtr;
{
    Element *elemPtr;
    Bar *barPtr;
    FreqKey key;
    Blt_ListItem *itemPtr;
    Tcl_HashEntry *hPtr;
    int numDataPoints;
    register int i;
    register FreqInfo *infoPtr;

    if ((graphPtr->mode != MODE_STACKED) || (graphPtr->numStacks == 0)) {
	return;
    }
    /* Reset the sums for all duplicate values to zero. */

    infoPtr = graphPtr->freqArr;
    for (i = 0; i < graphPtr->numStacks; i++) {
	infoPtr->sum = 0.0;
	infoPtr++;
    }

    /* Look at each bar point, adding the ordinates of duplicate abscissas */

    key.dummy = 0;
    for (itemPtr = Blt_FirstListItem(&(graphPtr->elemList));
	itemPtr != NULL; itemPtr = Blt_NextItem(itemPtr)) {
	elemPtr = (Element *)Blt_GetItemValue(itemPtr);
	if ((!elemPtr->mapped) || (elemPtr->type != ELEM_BAR)) {
	    continue;
	}
	barPtr = (Bar *)elemPtr;
	numDataPoints = BLT_MIN(barPtr->x.length, barPtr->y.length);
	for (i = 0; i < numDataPoints; i++) {
	    key.value = barPtr->x.data[i];
	    key.mask = barPtr->axisMask;
	    hPtr = Tcl_FindHashEntry(&(graphPtr->freqTable), (char *)&key);
	    if (hPtr == NULL) {
		continue;
	    }
	    infoPtr = (FreqInfo *)Tcl_GetHashValue(hPtr);
	    infoPtr->sum += barPtr->y.data[i];
	}
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_CheckStacks --
 *
 *	Check that the data limits are not superseded by the heights
 *	of stacked bar segments.  The heights are calculated by
 *	Blt_ComputeStacks.
 *
 * Results:
 *	If the y-axis limits need to be adjusted for stacked segments,
 *	*minPtr* or *maxPtr* are updated.
 *
 * Side effects:
 *	Autoscaling of the y-axis is affected.
 *
 * ----------------------------------------------------------------------
 */
void
Blt_CheckStacks(graphPtr, axisMask, minPtr, maxPtr)
    Graph *graphPtr;
    int axisMask;
    double *minPtr, *maxPtr;	/* Current minimum maximum for y-axis */
{
    FreqInfo *infoPtr;
    register int i;

    if ((graphPtr->mode != MODE_STACKED) || (graphPtr->numStacks == 0)) {
	return;
    }
    infoPtr = graphPtr->freqArr;
    for (i = 0; i < graphPtr->numStacks; i++) {
	if (infoPtr->mask & axisMask) {
	    /*
	     * Check if any of the y-values (because of stacking) are
	     * greater than the current limits of the graph.
	     */
	    if (infoPtr->sum < 0.0) {
		if (*minPtr > infoPtr->sum) {
		    *minPtr = infoPtr->sum;
		}
	    } else {
		if (*maxPtr < infoPtr->sum) {
		    *maxPtr = infoPtr->sum;
		}
	    }
	}
	infoPtr++;
    }
}


void
Blt_ResetStacks(graphPtr)
    Graph *graphPtr;
{
    register FreqInfo *infoPtr;
    register int i;

    infoPtr = graphPtr->freqArr;
    for (i = 0; i < graphPtr->numStacks; i++) {
	infoPtr->lastY = 0.0;
	infoPtr->count = 0;
	infoPtr++;
    }
}
