/*
 * bltList.c --
 *
 *	Generic linked list management routines.
 *
 * Copyright 1991-1996 by AT&T Bell Laboratories.
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the
 * names of AT&T Bell Laboratories any of their entities not be used
 * in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * AT&T disclaims all warranties with regard to this software, including
 * all implied warranties of merchantability and fitness.  In no event
 * shall AT&T be liable for any special, indirect or consequential
 * damages or any damages whatsoever resulting from loss of use, data
 * or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or
 * performance of this software.
 *
 */

#include "bltInt.h"

/*
 *----------------------------------------------------------------------
 *
 * Blt_InitList --
 *
 *	Initialized a linked list.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_InitList(listPtr, type)
    Blt_List *listPtr;
    int type;
{

    listPtr->numEntries = 0;
    listPtr->headPtr = listPtr->tailPtr = (Blt_ListItem *)NULL;
    listPtr->type = type;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_NewList --
 *
 *	Creates a new linked list structure and initializes its pointers
 *
 * Results:
 *	Returns a pointer to the newly created list structure.
 *
 *----------------------------------------------------------------------
 */
Blt_List *
Blt_NewList(type)
    int type;
{
    Blt_List *listPtr;

    listPtr = (Blt_List *)malloc(sizeof(Blt_List));
    if (listPtr != NULL) {
	Blt_InitList(listPtr, type);
    }
    return (listPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_NewItem --
 *
 *	Creates a list entry holder.  This routine does not insert
 *	the entry into the list, nor does it no attempt to maintain
 *	consistency of the keys.  For example, more than one entry
 *	may use the same key.
 *
 * Results:
 *	The return value is the pointer to the newly created entry.
 *
 * Side Effects:
 *	The key is not copied, only the Uid is kept.  It is assumed
 *	this key will not change in the life of the entry.
 *
 *----------------------------------------------------------------------
 */
Blt_ListItem *
Blt_NewItem(key)
    char *key;			/* Unique key to reference object */
{
    register Blt_ListItem *itemPtr;

    itemPtr = (Blt_ListItem *)malloc(sizeof(Blt_ListItem));
    if (itemPtr == (Blt_ListItem *)NULL) {
	panic("can't allocate list item structure");
    }
    itemPtr->keyPtr = key;
    itemPtr->clientData = (ClientData)NULL;
    itemPtr->nextPtr = itemPtr->prevPtr = (Blt_ListItem *)NULL;
    itemPtr->listPtr = (Blt_List *)NULL;
    return (itemPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_FreeItem --
 *
 *	Free the memory allocated for the entry.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_FreeItem(itemPtr)
    Blt_ListItem *itemPtr;
{
    free((char *)itemPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_ResetList --
 *
 *	Removes all the entries from a list, removing pointers to the
 *	objects and keys (not the objects or keys themselves).
 *	The entry counter is reset to zero.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_ResetList(listPtr)
    Blt_List *listPtr;		/* List to clear */
{
    register Blt_ListItem *oldPtr;
    register Blt_ListItem *itemPtr = listPtr->headPtr;

    while (itemPtr != (Blt_ListItem *)NULL) {
	oldPtr = itemPtr;
	itemPtr = itemPtr->nextPtr;
	Blt_FreeItem(oldPtr);
    }
    Blt_InitList(listPtr, listPtr->type);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DeleteList
 *
 *     Frees all list structures
 *
 * Results:
 *	Returns a pointer to the newly created list structure.
 *
 *----------------------------------------------------------------------
 */
void
Blt_DeleteList(listPtr)
    Blt_List *listPtr;
{
    Blt_ResetList(listPtr);
    free((char *)listPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_LinkAfter --
 *
 *	Inserts an entry following a given entry.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_LinkAfter(listPtr, itemPtr, afterPtr)
    Blt_List *listPtr;
    Blt_ListItem *itemPtr;
    Blt_ListItem *afterPtr;
{
    /*
     * If the list keys are strings, change the key to a Tk_Uid
     */
    if (listPtr->type == TCL_STRING_KEYS) {
	itemPtr->keyPtr = Tk_GetUid(itemPtr->keyPtr);
    }
    if (listPtr->headPtr == (Blt_ListItem *)NULL) {
	listPtr->tailPtr = listPtr->headPtr = itemPtr;
    } else {
	if (afterPtr == (Blt_ListItem *)NULL) {
	    afterPtr = listPtr->tailPtr;
	}
	itemPtr->nextPtr = afterPtr->nextPtr;
	itemPtr->prevPtr = afterPtr;
	if (afterPtr == listPtr->tailPtr) {
	    listPtr->tailPtr = itemPtr;
	} else {
	    afterPtr->nextPtr->prevPtr = itemPtr;
	}
	afterPtr->nextPtr = itemPtr;
    }
    itemPtr->listPtr = listPtr;
    listPtr->numEntries++;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_LinkBefore --
 *
 *	Inserts an entry preceding a given entry.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_LinkBefore(listPtr, itemPtr, beforePtr)
    Blt_List *listPtr;		/* List to contain new entry */
    Blt_ListItem *itemPtr;	/* New entry to be inserted */
    Blt_ListItem *beforePtr;	/* Entry to link before */
{
    /*
     * If the list keys are strings, change the key to a Tk_Uid
     */
    if (listPtr->type == TCL_STRING_KEYS) {
	itemPtr->keyPtr = Tk_GetUid(itemPtr->keyPtr);
    }
    if (listPtr->headPtr == (Blt_ListItem *)NULL) {
	listPtr->tailPtr = listPtr->headPtr = itemPtr;
    } else {
	if (beforePtr == (Blt_ListItem *)NULL) {
	    beforePtr = listPtr->headPtr;
	}
	itemPtr->prevPtr = beforePtr->prevPtr;
	itemPtr->nextPtr = beforePtr;
	if (beforePtr == listPtr->headPtr) {
	    listPtr->headPtr = itemPtr;
	} else {
	    beforePtr->prevPtr->nextPtr = itemPtr;
	}
	beforePtr->prevPtr = itemPtr;
    }
    itemPtr->listPtr = listPtr;
    listPtr->numEntries++;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_UnlinkItem --
 *
 *	Unlinks an entry from the given list. The entry itself is
 *	not deallocated, but only removed from the list.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_UnlinkItem(itemPtr)
    Blt_ListItem *itemPtr;
{
    Blt_List *listPtr;

    listPtr = itemPtr->listPtr;
    if (listPtr != NULL) {
	if (listPtr->headPtr == itemPtr) {
	    listPtr->headPtr = itemPtr->nextPtr;
	}
	if (listPtr->tailPtr == itemPtr) {
	    listPtr->tailPtr = itemPtr->prevPtr;
	}
	if (itemPtr->nextPtr != NULL) {
	    itemPtr->nextPtr->prevPtr = itemPtr->prevPtr;
	}
	if (itemPtr->prevPtr != NULL) {
	    itemPtr->prevPtr->nextPtr = itemPtr->nextPtr;
	}
	itemPtr->listPtr = NULL;
	listPtr->numEntries--;
    }
}

#ifdef notdef
/*
 *----------------------------------------------------------------------
 *
 * Blt_SetItemValue --
 *
 *	Sets the entry data pointer to the given value.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The data is not copied, only the pointer is kept.  It is assumed
 *	this data will remain valid as long as the entry.
 *
 *----------------------------------------------------------------------
 */
void
Blt_SetItemValue(itemPtr, clientData)
    Blt_ListItem *itemPtr;	/* Pointer to the entry */
    ClientData clientData;	/* Data attached to key */
{
    itemPtr->clientData = clientData;
}

#endif


#ifdef notdef
/*
 *----------------------------------------------------------------------
 *
 * Blt_List_Index --
 *
 *	Find the position of an entry in the list.
 *
 * Results:
 *	Returns the index to the entry.  If no entry matching
 *	the key given is found, then -1 is returned.
 *
 *----------------------------------------------------------------------
 */
int
Blt_ItemIndex(listPtr, searchPtr)
    Blt_List *listPtr;		/* List to search */
    Blt_ListItem *searchPtr;	/* Entry to match */
{
    register int count = 0;
    register Blt_ListItem *itemPtr;	/* Entry to match */

    for (itemPtr = listPtr->headPtr; itemPtr != NULL;
	itemPtr = itemPtr->nextPtr) {
	if (searchPtr == itemPtr)
	    return (count);
	count++;
    }
    return (-1);
}

#endif

/*
 *----------------------------------------------------------------------
 *
 * Blt_FindItem --
 *
 *	Find the first entry matching the key given.
 *
 * Results:
 *	Returns the pointer to the entry.  If no entry matching
 *	the key given is found, then NULL is returned.
 *
 *----------------------------------------------------------------------
 */
Blt_ListItem *
Blt_FindItem(listPtr, searchKey)
    Blt_List *listPtr;		/* List to search */
    char *searchKey;		/* Key to match */
{
    register Blt_ListItem *itemPtr;
    Tk_Uid newPtr;

    newPtr = searchKey;
    if (listPtr->type == TCL_STRING_KEYS) {
	newPtr = Tk_GetUid(searchKey);
    }
    for (itemPtr = listPtr->headPtr; itemPtr != NULL;
	itemPtr = itemPtr->nextPtr) {
	if (newPtr == itemPtr->keyPtr)
	    return (itemPtr);
    }
    return (Blt_ListItem *) NULL;
}

#ifdef notdef
/*
 *----------------------------------------------------------------------
 *
 * Blt_FirstListItem --
 *
 *	Find the first entry in the list and return its pointer.
 *	In addition, update the given search pointer.
 *
 * Results:
 *	Returns a pointer to the first entry in the list. If the
 *      list is empty, NULL is returned.  The search pointer (used in
 *	subsequent searches) is set to the appropriate value.
 *
 *----------------------------------------------------------------------
 */
Blt_ListItem *
Blt_FirstListItem(listPtr, itemPtrPtr)
    Blt_List *listPtr;		/* The list we are searching */
    Blt_ListItem **itemPtrPtr;	/* Search pointer to set */
{
    if (listPtr == (Blt_List *)NULL)
	return (Blt_ListItem *) NULL;

    if (listPtr->headPtr == (Blt_ListItem *)NULL)
	return (Blt_ListItem *) NULL;

    if (itemPtrPtr != (Blt_ListItem **)NULL) {
	*itemPtrPtr = listPtr->headPtr;
    }
    return (listPtr->headPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_LastListItem --
 *
 *	Find the last entry in the list using the given entry as
 *	a cursor to the last entry and return its pointer.
 *	In addition, the cursor position is updated.
 *
 * Results:
 *	A pointer to the last object in the list is returned. If the
 *      list is at end, NULL is returned.  The search pointer (used in
 *	subsequent searches) is set to the appropriate value.
 *
 *----------------------------------------------------------------------
 */
Blt_ListItem *
Blt_LastListItem(itemPtrPtr)
    Blt_ListItem **itemPtrPtr;	/* Search pointer of current position */
{
    if ((itemPtrPtr == (Blt_ListItem **)NULL) ||
	(*itemPtrPtr == (Blt_ListItem *)NULL)) {
	return (Blt_ListItem *) NULL;
    }
    *itemPtrPtr = (*itemPtrPtr)->prevPtr;
    return (*itemPtrPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_NextItem --
 *
 *	Find the next entry in the list using the given search pointer
 *	as the current location and return its pointer.
 *	In addition, update the given search pointer.
 *
 * Results:
 *	A pointer to the next object in the list is returned. If the
 *      list is at end, NULL is returned.  The search pointer (used in
 *	subsequent searches) is set to the appropriate value.
 *
 *----------------------------------------------------------------------
 */
Blt_ListItem *
Blt_NextItem(itemPtrPtr)
    Blt_ListItem **itemPtrPtr;	/* Search pointer indicates current position */
{
    if ((itemPtrPtr == NULL) || (*itemPtrPtr == NULL)) {
	return (Blt_ListItem *) NULL;
    }
    *itemPtrPtr = (*itemPtrPtr)->nextPtr;
    return (*itemPtrPtr);
}

#endif
/*
 *----------------------------------------------------------------------
 *
 * Blt_DeleteItem --
 *
 *	Find the entry and free the memory allocated for the entry.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_DeleteItem(itemPtr)
    Blt_ListItem *itemPtr;
{
    Blt_UnlinkItem(itemPtr);
    Blt_FreeItem(itemPtr);
}
