#
# Module for file I/O for Misc. widgets
#


# Returns pathname p with . and .. expressions evaluated
proc th_dedot_file {p} {
# First eliminate .
  while {[string match {\./*} $p]} {set p [string range $p 2 end]}
  while {[string match {*/\.} $p]} {set p [string range $p 0 [expr [string length $p] - 2]]}
  while {[regsub -- {/\./} $p {/} q]} {set p $q}

# Now eliminate ..
  while {[regsub -- {[^/]*/\.\.} $p {} q]} {regsub -- {//} $q {/} p}
  return $p
}

# Prompts user for a file to write to, before saving file.
proc th_save_file_prompt {w} {
  th_file_auto_warn $w
  th_file_prompt $w "th_write_file $w" "File to Save:"
}

# Saves file w/o prompting, prompts user only if necessary
proc th_save_file {w} {
  global TH
  if {[catch "set TH(File,Last,$w)"]} {set TH(File,Last,$w) ""}
  if {([catch "set TH(File,$w)"] || ($TH(File,$w) == ""))} {
    th_save_file_prompt $w
  } elseif {([string first "|" $TH(File,$w)] >= 0) && ($TH(File,Last,$w) != "w")} {
    th_save_file_prompt $w
  } elseif {[th_file_auto_warn $w]} {
    th_save_file_prompt $w 
  } else {
    set TH(File,New,$w) $TH(File,$w)
    th_write_file $w
}}

# Writes contents of w out to TH(File,%W)
proc th_write_file {w} {
  global TH auto_index
  if {[catch "set TH(Pipe,Enabled)"]} {set TH(Pipe,Enabled) 0}
  if {($TH(Pipe) || ([string first "|" $TH(File,New,$w)] >= 0)) && !$TH(Pipe,Enabled)} {
    error "Pipe writing (to $TH(File,New,$w)) is not allowed in this application."
  }

  set pwd [pwd]
  if {([set index [string first "|" $TH(File,New,$w)]] >= 0)} {
    cd [string range $TH(File,New,$w) 0 [expr $index - 2]]
    set file [open [string range $TH(File,New,$w) $index end] w]
  } elseif $TH(Pipe) {
    cd [th_file_dirname $TH(File,New,$w)]
    set file [open [th_file_tail $TH(File,New,$w)] w]
  } else {  set file [open $TH(File,New,$w) w]}

  th_file_update_widgets $w $TH(File,New,$w)
  th_[winfo class $w]_write_file $w $file
  close $file
  if {![catch "set TH(File,Auto,Suffix,$w)"]} {
    if {$TH(File,Auto,Suffix,$w) != ""} {
      set backup "[set TH(File,New,$w)][set TH(File,Auto,Suffix,$w)]"
      if {[file exists $backup]} {catch {exec rm $backup}}
  }}

  catch {set TH(Mtime,$w) [file mtime $TH(File,New,$w)]}
  set TH(Modified,$w) 0
  if {([info procs th_kill_undos] != "") || \
     ([lsearch [array names auto_index] th_kill_undos] >= 0)} {
    th_kill_undos $w}
  set TH(File,Last,$w) "w"
  set TH(File,$w) [th_dedot_file $TH(File,New,$w)]
  cd $pwd
  th_file_update_widgets $w
}


# Prompts user for a new file to read.
proc th_read_file_prompt {w} {
  th_confirm_save $w "th_file_prompt $w \"th_load_file $w\" {File to Read:}"
}

# Reads (reverts to) current file w/o prompt, unless necessary.
proc th_read_file {w} {
  global TH
  if {[catch "set TH(File,Last,$w)"]} {set TH(File,Last,$w) ""}
  if {([catch "set TH(File,$w)"] || ($TH(File,$w) == ""))} {
    th_read_file_prompt $w
  } elseif {([string first "|" $TH(File,$w)] >= 0) && ($TH(File,Last,$w) != "r")} {
    th_read_file_prompt $w
  } elseif {!([string first "|" $TH(File,$w)] >= 0) &&
      (![file exists $TH(File,$w)] || ![file readable $TH(File,$w)])} {
    th_read_file_prompt $w
  } else {
    set TH(File,New,$w) $TH(File,$w)
    global auto_index
    if {([info procs th_[winfo class $w]_save_view] != "") || \
       ([lsearch [array names auto_index] th_[winfo class $w]_save_view] >= 0)} {
      th_confirm_save $w "th_[winfo class $w]_save_view $w \"th_load_file $w\""
    } else {
      th_confirm_save $w "th_load_file $w"
}}}

# Inserts contents of prompted file into w, in usual editing fashion.
proc th_insert_file {w} {
  th_file_prompt $w "th_load_file $w 1" {File to Insert:}
}

# Loads w with contents of TH(File,$w) if insert_flag is 1, then insert,
# otherwise load.
proc th_load_file {w {insert_flag 0}} {
  global TH auto_index
  if {[catch "set TH(Pipe)"]} {set TH(Pipe) 0}
  if {[catch "set TH(Pipe,Enabled)"]} {set TH(Pipe,Enabled) 0}
  if {($TH(Pipe) || ([string first "|" $TH(File,New,$w)] >= 0)) && !$TH(Pipe,Enabled)} {
    error "Pipe reading (from $TH(File,New,$w)) is not allowed in this application."
  }

  set pwd [pwd]
  if {([set index [string first "|" $TH(File,New,$w)]] >= 0)} {
    cd [string range $TH(File,New,$w) 0 [expr $index - 2]]
    set name [string range $TH(File,New,$w) $index end]
    set pipe 1
  } elseif $TH(Pipe) {
    cd [th_file_dirname $TH(File,New,$w)]
    set name [th_file_tail $TH(File,New,$w)]
    set pipe 1
  } else {  set name $TH(File,New,$w) ; set pipe 0}
  if $pipe {
    if {([info procs th_string_complete_multiple] != "") || \
       ([lsearch [array names auto_index] th_string_complete_multiple] >= 0)} {
      if {[set comp [lindex [th_string_complete_multiple $name \
		{{th_substring_replace th_string_glob_files { }}}] 0]] != ""} {
        set name $comp
  }}}
  set file [open $name r]

  if {![file writable $TH(File,New,$w)] && [file exists $TH(File,New,$w)]} {
    th_flash_label $w -text "File is not writable!"
  }

  if $insert_flag {
    if {![catch "set TH(File,New,Old,$w)"]} {set TH(File,New,$w) $TH(File,New,Old,$w)
    } else {unset TH(File,New,$w)}
    th_[winfo class $w]_insert_file $w $file $pipe
    set TH(Modified,$w) 1
  } else {
    th_file_update_widgets $w $TH(File,New,$w)
    th_[winfo class $w]_read_file $w $file $pipe
    catch {set TH(Mtime,$w) [file mtime $TH(File,New,$w)]}
    set TH(Modified,$w) 0
    if {([info procs th_kill_undos] != "") || \
       ([lsearch [array names auto_index] th_kill_undos] >= 0)} {
      th_kill_undos $w}
    set TH(File,New,Last,$w) "r"
    set TH(File,$w) [th_dedot_file $TH(File,New,$w)]
    th_file_update_widgets $w
    }
  cd $pwd
  close $file
  if {![catch "set TH(File,Auto,Suffix,$w)"]} {
    if {$TH(File,Auto,Suffix,$w) != ""} {
      set backup "[set TH(File,New,$w)][set TH(File,Auto,Suffix,$w)]"
      if {[file exists $backup]} {catch {exec rm $backup}}
  }}
}

# Prompts user for a file, doing cmd if OK.
proc th_file_prompt {w cmd {msg "File:"}} {
  global TH
  if {[catch "set TH(File,$w)"]} {set TH(File,$w) ""}

  if {[set f [th_show_entry $w file]] == ""} {bell ; return}
  set TH(Completions,$f.e) [list {th_line_complete th_filter_glob_pwd none}]
  if {![catch "set TH(Completions,File,Commands)"]} {
    lappend TH(Completions,$f.e) [list th_substring_complete [list th_list_completions $TH(Completions,File,Commands)] {|}]
  }
  $f.l configure -text $msg
  if {![catch "set TH(File,$w)"] || ($TH(File,$w) == "")} {
    $f.e insert 0 $TH(File,$w)
    if {[set index [string first "|" $TH(File,$w)]] < 0} {
      set index [string last "/" $TH(File,$w)] ; incr index
    }
    $f.e icursor $index
    $f.e xview $index
  }
  th_bind $f.e OK "th_file_prompt_aux $w \[string trimright \[$f.e get\]\]; $cmd"
  return $f
}

proc th_file_prompt_aux {w file} {
  global TH
  if {[catch "set TH(Pipe)"]} {set TH(Pipe) 0}
  th_hide_entry $w file
  if {$TH(Pipe) && ([string first "|" $file] < 0)} {
    set TH(File,New,$w) "[th_file_dirname $file]/| [th_file_tail $file]"
  } elseif {([string first [string index $file 0] {~/}] < 0)} {
    set TH(File,New,$w) "[pwd]/$file"
  } else {set TH(File,New,$w) $file}
}

# If w's contents differ from TH(File,$w), prompt user to save contents.
# Executes cmd if user does not cancel.
proc th_confirm_save {w cmd} {
  global TH
  if {![catch "set TH(Modified,$w)"] && $TH(Modified,$w)} {
    if {([catch "set TH(File,$w)"] || ($TH(File,$w) == ""))} {set name "This"
    } else {set name [th_file_tail $TH(File,$w)]}
    set message "$name has been modified. Do you want to save it first?"
    set choice [tk_dialog "$w.conf" "Save Dialog" $message question 0 "Save" "Don't Save" "Cancel"]
    if {$choice == 2} {return}
    if {$choice == 0} {
      th_save_file $w
      if $TH(Modified,$w) { tkwait variable TH(Modified,$w)}
  }}
  uplevel #0 $cmd
}

# Updates any widgets that contain the file patn/name (usually window title)
proc th_file_update_widgets {w {filename ""}} {
  global TH
  if {$filename == ""} {
    if {([catch "set TH(File,$w)"] || ($TH(File,$w) == ""))} {return}
    set filename $TH(File,$w)
  }
  if {[string first [string index $filename 0] {~/}] >= 0} {
    set path [th_file_dirname $filename] ; set name [th_file_tail $filename]
  } else {set path [pwd] ; set name $filename}
  set f [th_frame $w]
  if {[winfo exists $f.fpl]} {
   $f.fpl configure -onvalue $path
   if {$TH(File,fpl,$w) != "-"} {set TH(File,fpl,$w) $path}}
  if {[winfo exists $f.fnl]} {
    $f.fnl configure -onvalue $name 
    if {$TH(File,fnl,$w) != "-"} {set TH(File,fnl,$w) $name}}
  if {[info procs th_file_update_titles] != ""} {
    th_file_update_titles $w $filename}
}

# Returns the dirname of f (even if it is a pipe)
proc th_file_dirname {f} {
  if {[set pipe [string first "|" $f]] >= 0} {
    return [string range $f 0 [expr $pipe - 2]]
  } else {return [file dirname $f]}
}

# Returns the tail of f.
proc th_file_tail {f} {
  if {[set pipe [string first "|" $f]] >= 0} {
    return [string range $f $pipe end]
  } else {return [file tail $f]}
}


# Auto routines

# This is called for every auto cycle.
proc th_file_auto {w} {
  if {![winfo exists $w]} {return cycle_deactivate}

  global TH
  th_file_auto_validate $w
  th_file_auto_warn $w
  if {$TH(File,Auto,Type,$w) == 0} {
    return [th_file_auto_read $w]
  } else {
    return [th_file_auto_save $w]
}}

# Make sure auto variables are consistant with file.
proc th_file_auto_validate {w} {
  global TH auto_index
  set pipe 0
  if {![catch "set TH(File,$w)"] && ($TH(File,$w) != "") &&
	([string first "|" $TH(File,$w)] >= 0)} {set pipe 1}

  if $pipe {
    set TH(File,Auto,Type,$w) 0
    if {([info procs th_[winfo class $w]_contents] != "") || \
       ([lsearch [array names auto_index] th_[winfo class $w]_contents] >= 0)} {
      set TH(File,Auto,Warn,$w) 1
    } else {set TH(File,Auto,Warn,$w) 0}
}}

# If file has been modified, warn user. (and return 1)
proc th_file_auto_warn {w} {
  global TH auto_index
  set result [th_external_change $w]
  if {(![catch "set TH(Cycle,Active,$w)"] && !$TH(Cycle,Active,$w)) ||
	(![catch "set TH(File,Auto,Warn,$w)"] && !$TH(File,Auto,Warn,$w))} {return 0}

  if $result {th_file_auto_notify $w}
  return $result
}

proc th_file_auto_notify {w} {
  catch {th_flash_label $w -text "File content has been changed!!!"}
  catch {th_raise_window $w}
  bell
}

# Returns 1 if file has been changed since we last accessed it.
# Set modified variable if file doesn't exist or is unwritable
proc th_external_change {w} {
  global TH

# don't bother if file is a pipe.
  if {[catch "set TH(File,$w)"]} {return 0}
  if {$TH(File,$w) == ""} {return 0}
  if {([string first "|" $TH(File,$w)] >= 0)} {return 0}

  if {![file exists $TH(File,$w)]} {set TH(Modified,$w) 1 ; return 0}
  if {![file writable $TH(File,$w)]} {return 0}

  if {[catch "set TH(Mtime,$w)"]} {return 0}
  if {([file mtime $TH(File,$w)] <= $TH(Mtime,$w))} {return 0}
  if {[catch "file mtime $TH(File,$w)"]} {return 0}

  set TH(Modified,$w) 1
  return 1
}

proc th_file_auto_save {w} {
  global TH
  if {!$TH(Cycle,Active,$w) || ($TH(File,Auto,Type,$w) != 1)} {return 0}

# don't bother if file is a pipe or hasn't been modified or is unwritable
  if {[catch "set TH(File,$w)"]} {return 0}
  if {$TH(File,$w) == ""} {return 0}
  if {([string first "|" $TH(File,$w)] >= 0)} {return 0}
  if {[catch "set TH(Modified,$w)"]} {return 0}
  if {!$TH(Modified,$w)} {return 0}
  if {[catch "set TH(File,Auto,Suffix,$w)"]} {return 0}
  if {![file writable $TH(File,$w)]} {return 0}

  set name "[set TH(File,$w)][set TH(File,Auto,Suffix,$w)]"
  catch {th_flash_label $w -text "Auto-saving [th_file_tail $name]"}
  if {$TH(File,Auto,Suffix,$w) == ""} {
    th_write_file $w
  } else {
    set file [open $name w]
    th_[winfo class $w]_write_file $w $file
    close $file
  }
  return 1
}

proc th_file_auto_read {w} {
  global TH auto_index
  if {!$TH(Cycle,Active,$w) || ($TH(File,Auto,Type,$w) != 0)} {return 0}

# If file is not defined, don't bother
  if {[catch "set TH(File,$w)"]} {return 0}
  if {$TH(File,$w) == ""} {return 0}

  set pipe 0
  if {([string first "|" $TH(File,$w)] >= 0)} {set pipe 1}

# For a regular file, only reread if modified and readable.
  if {!$pipe} {
    if {[catch "set TH(Modified,$w)"]} {return 0}
    if {!$TH(Modified,$w)} {return 0}
    if {![file readable $TH(File,$w)]} {return 0}
  }

# If a pipe, we want to do a diff on the old contents
  if {$pipe && (([info procs th_[winfo class $w]_contents] != "") || \
       ([lsearch [array names auto_index] th_[winfo class $w]_contents] >= 0))} {
    set data [th_[winfo class $w]_contents $w]
  }

  set TH(Modified,$w) 0
  set TH(File,New,$w) $TH(File,$w)
  th_load_file $w
  catch {th_flash_label $w -text "Rereading [th_file_tail $TH(File,$w)]"}

  if {$pipe && (([info procs th_[winfo class $w]_contents] != "") || \
       ([lsearch [array names auto_index] th_[winfo class $w]_contents] >= 0))} {
    set data2 [th_[winfo class $w]_contents $w]
    if {$data != $data2} {th_file_auto_notify $w}
  }
  return 1
}
