#
# Procedure for executing a command periodically
#

# A cycle is simply some command that executes periodically. It is started with
# th_cycle, which takes a time interval, and a command. The cycle can be paused
# or reactivated, but it cannot be killed, unless some error occurs in the
# command. So if you want to kill a cycle, give it a command that will 
# deactivate it at the proper time. A command returning the string
# "deactivate_cycle" will deactivate the cycle.

# To turn off the cycle, set TH(Cycle,Active,$handle) to 0
# To turn on the cycle, set TH(Cycle,Active,$handle) to 1
# Setting TH(Cycle,Active,$handle) to 1 while it's already 1 'resets' the cycle.
# While it is 0, you can execute the command indpendently, by doing:
# eval $TH(Cycle,Command,$handle)
# Cycle lives on until an error occurs in cycle's command, or program quits.


# Args is a Tcl command. Sets up command to execute after every $time
# milliseconds. Handle is some unqiue word that identifies this cycle.
proc th_cycle {handle time args} {
  global TH
  set TH(Cycle,Time,$handle) $time
  set TH(Cycle,Command,$handle) $args
  set TH(Cycle,Active,$handle) 1
  set TH(Cycle,After,$handle) [after $time "th_cycle_bg $handle"]
  trace variable TH(Cycle,Active,$handle) "w" th_cycle_trace 
}

proc th_cycle_bg {handle} {
  global TH
  set TH(Cycle,After,$handle) ""

  if $TH(Cycle,Active,$handle) {
    if {([uplevel #0 $TH(Cycle,Command,$handle)] == "cycle_deactivate")} {
      set TH(Cycle,Active,$handle) 0
  }}

# Cycle was requested to terminate.
  if {$TH(Cycle,After,$handle) == "cancel"} {return} else {
    set TH(Cycle,After,$handle) [after $TH(Cycle,Time,$handle) th_cycle_bg $handle]
}}

proc th_cycle_trace {th index op} {
  if {$th != "TH"} {return}
  if {![scan $index "Cycle,Active,%s" handle]} {return}
  if {$op != "w"} {return}
  global TH

  if $TH(Cycle,Active,$handle) {
# If Cmd is running, just make sure it won't cancel
    if {$TH(Cycle,After,$handle) == "cancel"} {set TH(Cycle,After,$handle) ""}

# If Cmd is sleeping, cancel it & restart.
    if {$TH(Cycle,After,$handle) != ""} {
      after cancel $TH(Cycle,After,$handle)
      set TH(Cycle,After,$handle) [after 1 "th_cycle_bg $handle"]
  }} else {
# If Cmd is running...cancel when it's finished.
    if {$TH(Cycle,After,$handle) == ""} {set TH(Cycle,After,$handle) "cancel"}

# If Cmd is sleeping, cancel it.
    if {$TH(Cycle,After,$handle) != "cancel"} {
      after cancel $TH(Cycle,After,$handle)
}}}


# Cycle checkbuttons.

# These get associated with a cycle...they determine when a cycle is on or off.
# Conceptually, they serve as a wrapper around the cycle. The button name
# is used as the cycle handle.

# To turn a cycle off, you turn off its checkbutton
# To turn the cycle on, turn on its checkbutton.
# If a cycle is off, you can double-click its checkbutton to
# 	independently execute its command.
# If a cycle is on, you can double-click its checkbutton to
#	reset its counter (and execute its command)
# If the checkbutton gets destroyed, the cycle deactivates.


# Activates button w while executing command
proc th_Checkbutton_command {w cmd} {
  if {![winfo exists $w]} {return "cycle_deactivate"}

  $w configure -state disabled
  update
  global errorInfo
  set error [catch {uplevel #0 $cmd} result]
  set info $errorInfo
  $w configure -state normal
  if $error {error $result $info} else {return $result}
}

# Starts a cycle associated with w, a checkbutton. 
proc th_Checkbutton_cycle {w time args} {
  global TH
  th_cycle $w $time th_Checkbutton_command $w $args
  $w configure -variable TH(Cycle,Active,$w)
  bind $w <Double-Button-1> "$w flash"
}


#
# Menubutton cycles
#
# These work just like checkbutton cycles, but they're done with menubuttons.
# Except they don't flash when you double-click on them. The handle is "$m.$i"

# Activates menuentry number i in menu m while executing command
proc th_menucheckbutton_command {m i cmd} {
  if {![winfo exists $m]} {return "cycle_deactivate"}
  if {[$m type $i] != "checkbutton"} {return "cycle_deactivate"}
  $m entryconfigure $i -state disabled
  update
  global errorInfo
  set error [catch {uplevel #0 $cmd} result]
  set info $errorInfo
  $m entryconfigure $i -state normal
  if $error {error $result $info} else {return $result}
}

# Starts a cycle associated with i, a checkbutton entry in m.
proc th_menucheckbutton_cycle {m i time args} {
  global TH
  set handle "$m.[$m index $i]"
  th_cycle $handle $time th_menucheckbutton_command $m $i $args
  $m entryconfigure $i -variable TH(Cycle,Active,$handle)
}
