/* 
** A system performance monitor for Linux written in ET.
*/
#include <stdio.h>

/*
** This is the main procedure.
*/
int main(int argc,char **argv){
  Et_Init(&argc,argv);             /* Initialize ET */
  ET_INSTALL_COMMANDS;             /* Install Tcl/Tk defined below */
  ET_INCLUDE( perfmon.tcl );       /* Setup the main window */
  ET( UpdateCPU; UpdateMem );      /* Start the ball rolling */
  Et_MainLoop();                   /* Process events forever */
}

/*
** Read CPU usage statistics from the /proc/stat file and update
** the screen accordingly.
**
** After the display has been updated, this function arranges for
** itself to be called again after the number of milliseconds specified
** by the variable "CpuUpdateTime".
*/
ET_PROC( UpdateCPU ){
  /* These 4 variables hold the user, nice, system and idle times
  ** from the previous invocations of this function.  These values
  ** are subtracted from the current times to get the amount of CPU
  ** time spent in each mode.  The times are kept in a circular
  ** buffer whose size is determined by the size of the averaging
  ** window */
  static int *prevUsr;
  static int *prevNice;
  static int *prevSys;
  static int *prevIdle;

  /* This is the index of the next slot of the circular buffers to use */
  static int index;

  /* This is the size of the circular buffers */
  static int limit = 0;

  /* These variables hold the CPU times obtained at this function call */
  int thisUsr, thisNice, thisSys, thisIdle;

  /* These variables hold the amount of CPU type devoted to user, nice,
  ** system and idle, over the averaging window time. "total"
  ** is the total of the four, and is used for normalization */
  int usr, nice, sys, idle, total;

  /* These variables hold the X-coordinate of the right edge of the line
  ** which is drawn to represent each of the times */
  int usrEnd, niceEnd, sysEnd, idleEnd;

  /* This is the input stream for the /proc/stat file */
  FILE *pIn;

  /* The current correct value for "limit" */
  int newLimit;

  /* The next value for "index" */
  int nextIndex;

  /* open the /proc/stat file and extract the four CPU times contained
  ** therein */
  pIn = fopen("/proc/stat","r");
  if( pIn==0 ) return ET_OK;
  if( fscanf(pIn,"cpu %d %d %d %d",&thisUsr,&thisNice,&thisSys,&thisIdle)<4 ){
    fclose(pIn);
    return ET_OK;
  }
  fclose(pIn);

  /* Update the size of the circular buffer, if necessary */
  newLimit = ET_INT( expr {$CpuAveragingWindow/$CpuUpdateTime} );
  if( newLimit<1 ) newLimit = 1;
  if( newLimit>100 ) newLimit = 100;
  if( newLimit!=limit ){
    if( limit>0 ){
      free( prevUsr );
    }
    limit = newLimit;
    prevUsr = (int*)malloc( sizeof(int)*limit*4 );
    memset(prevUsr,0,sizeof(int)*limit*4);
    prevNice = &prevUsr[limit];
    prevSys = &prevNice[limit];
    prevIdle = &prevSys[limit];
    index = 0;
  }

  /* Get the CPU statistics averaged over time */
  nextIndex = index + 1;
  if( nextIndex>=limit ) nextIndex = 0;
  usr = prevUsr[nextIndex] - thisUsr;
  prevUsr[nextIndex] = thisUsr;
  nice = prevNice[nextIndex] - thisNice;
  prevNice[nextIndex] = thisNice;
  sys = prevSys[nextIndex] - thisSys;
  prevSys[nextIndex] = thisSys;
  idle = prevIdle[nextIndex] - thisIdle;
  prevIdle[nextIndex] = thisIdle;
  total = usr + nice + sys + idle;
  index = nextIndex;

  /* Update the display */
  usrEnd = 50 + usr*240/total;
  niceEnd = usrEnd + nice*240/total;
  sysEnd = niceEnd + sys*240/total;
  ET(
    .c coords usr 50 40 %d(usrEnd) 40
    .c coords nice %d(usrEnd) 40 %d(niceEnd) 40
    .c coords sys %d(niceEnd) 40 %d(sysEnd) 40
    .c coords idle %d(sysEnd) 40 290 40
    after $CpuUpdateTime UpdateCPU
  );
  return ET_OK;
}

/*
** Read memory usage statistics from the /proc/meminfo file and update
** the screen accordingly.
**
** After the display has been updated, this function arranges for
** itself to be called again after the number of milliseconds specified
** by the variable "MemUpdateTime".
*/
ET_PROC( UpdateMem )
{
  /* The following variables hold the values of the memory statistics
  ** as taken from the /proc/meminfo file */
  int coreFree;
  int coreUsed;
  int coreBuffer;
  int swapFree;
  int swapUsed;

  /* The next variables are intermediate results */
  int coreNormalizer;   /* Used to normalize the core memory sizes */
  int coreUsedEnd;      /* End of the bar showing core in use */
  int coreBufferEnd;    /* End coordinate of the bar showing core used for
                        ** as disk buffer space */
  int swapUsedEnd;      /* End coordinate of the bar showing swap used */

  /* This is the input stream for the /proc/stat file */
  FILE *pIn;

  /* Open the /proc/stat file and extract the four times */
  pIn = fopen("/proc/meminfo","r");
  if( pIn==0 ) return ET_OK;
  if( fscanf(pIn,"%*[^\n]\nMem: %*d %d %d %*d %d\nSwap: %*d %d %d",
                 &coreUsed,&coreFree,&coreBuffer,&swapUsed,&swapFree)!=5 ){
    fclose(pIn);
    return ET_OK;
  }
  fclose(pIn);

  /* Update the display */
  coreNormalizer = (coreUsed + coreFree)/240;
  coreUsedEnd = (coreUsed-coreBuffer)/coreNormalizer + 50;
  coreBufferEnd = coreBuffer/coreNormalizer + coreUsedEnd;
  if( swapUsed>0 ){
    swapUsedEnd = swapUsed/((swapUsed+swapFree)/240) + 50;
  }else{
    swapUsedEnd = 50;
  }
  ET(
    .c coords coreUsed 50 10 %d(coreUsedEnd) 10
    .c coords coreBuffer %d(coreUsedEnd) 10 %d(coreBufferEnd) 10
    .c coords coreFree %d(coreBufferEnd) 10 290 10
    .c coords swapUsed 50 25 %d(swapUsedEnd) 25
    .c coords swapFree %d(swapUsedEnd) 25 290 25
    after $MemUpdateTime UpdateMem
  );
  return ET_OK;
}
