# This Tcl/Tk script implements a simple text editor.
#
# For the public domain.
# Originally generated by...
#
#         D. Richard Hipp
#         Hipp, Wyrick & Company, Inc.
#         6200 Maple Cove Lane
#         Charlotte, NC 28269
#         704.948.4565
#         drh@vnet.net
#
option add *highlightThickness 0

# The next pair of functions implement a dialog box that tells about
# this program.  The dialog box is the one that appears when the user
# selects Help/AboutThisProgram from the main menu.
#
proc AboutMessage {w text} {
  label $w.title -text TkEdit \
    -font -adobe-times-bold-i-normal--24-240-75-75-p-128-iso8859-1
  pack $w.title -side top -pady 15
  message $w.subtitle -width 10c -justify center \
    -font -adobe-times-bold-i-normal-*-14-140-75-75-p-77-iso8859-1 \
    -text "An ASCII text editor built upon\nthe Tcl/Tk Text widget"
  pack $w.subtitle -side top -pady 10 -padx 15
  message $w.msg -width 10c -text "
By D. Richard Hipp
Hipp, Wyrick & Company, Inc.
6200 Maple Cove Lane
Charlotte, NC 28269
704-948-4565
drh@vnet.net" \
    -font -adobe-times-medium-r-normal-*-12-120-75-75-p-64-iso8859-1
  pack $w.msg -padx 15 -anchor w
  label $w.spacer -text { }
  pack $w.spacer
}
proc HelpAboutThisProgram {} {
  ModalDialogBox . {About} AboutMessage {} {{Dismiss {}}}
}

# Get the name of the file to be edited.
#
if {$argc>1} {
  # If more than one filename is listed on the command line, invoke
  # a separate editor for each one.
  foreach i $argv {
    catch {exec $argv0 $i &}
  }
  exit 0
}
if {$argc==1} {
  # This is the case where a file is specified on the command line.
  # Bring up the editor on that single file
  #
  set path $argv
  if {"[string index $path 0]"=="/"} {
    set full_path $path
  } else {
    set full_path [pwd]/$path
  }
  set FullPath {}

  # See if any other editor is currently working on this file.  If it is,
  # then deiconify and raise that editor and exit, because we don't need
  # more than a single editor running on each file.
  #
  wm withdraw .
  foreach other [winfo interps] {
    if {![string match tkedit* $other]} continue
    catch {
      if {[send $other {set tkedit_FullPath}]==$full_path} {
        send $other {wm deiconify .; raise .}
        exit 0
      }
    }
  }

  set FullPath $full_path
  set tkedit_FullPath $full_path
  wm deiconify .
} else {
  # This is the case where no filename is given.  Simply start a new
  # editor.
  #
  set FullPath {}
  set tkedit_FullPath {}
}

# Make sure the name doesn't coorespond to a directory.  It doesn't have
# to be a file which exists, but it can't be a directory.
#
if {"$FullPath"!="" && [file exists $FullPath] && ![file readable $FullPath]} {
  puts stderr "tkedit: Can't open $FullPath for reading"
  exit 1
}

# Construct the editor
#
wm title . [file tail $FullPath]
wm iconname . [file tail $FullPath]
wm minsize . 320 240
wm protocol . WM_DELETE_WINDOW FileQuit

# Construct the menu bar across the top of the application
#
frame .mbar -bd 2 -relief raised
pack .mbar -side top -fill x
foreach i {file edit view help} {
  set name [string toupper [string range $i 0 0]][string range $i 1 end]
  menubutton .mbar.$i -text $name -underline 0 -menu .mbar.$i.menu -pady 0
  pack .mbar.$i -side left -padx 8
}
pack .mbar -side top -fill x

menu .mbar.file.menu
.mbar.file.menu add command -label Revert -command FileRevert
.mbar.file.menu add command -label Open... -command FileOpen
.mbar.file.menu add command -label New -command FileNew
.mbar.file.menu add command -label Save -command FileSave
.mbar.file.menu add command -label {Save as...} -command FileSaveAs
#.mbar.file.menu add command -label Print... -command FilePrint
.mbar.file.menu add separator
.mbar.file.menu add command -label Quit -command FileQuit

menu .mbar.edit.menu
.mbar.edit.menu add command -label Cut -command EditCut
.mbar.edit.menu add command -label Copy -command EditCopy
.mbar.edit.menu add command -label Paste -command EditPaste
.mbar.edit.menu add command -label Delete -command EditDelete
.mbar.edit.menu add separator
.mbar.edit.menu add command -label Search... -command EditSearch

menu .mbar.view.menu
.mbar.view.menu add cascade -label {Font Size} -menu .mbar.view.menu.fontsize
.mbar.view.menu add cascade -label Height -menu .mbar.view.menu.height
.mbar.view.menu add cascade -label Width -menu .mbar.view.menu.width
.mbar.view.menu add cascade -label Wrap -menu .mbar.view.menu.wrap
menu .mbar.view.menu.fontsize
foreach i {
  {Tiny -schumacher-clean-medium-r-normal--6-60-75-75-c-40-iso8859-1}
  {Small -schumacher-clean-medium-r-normal--8-80-75-75-c-50-iso8859-1}
  {Short -schumacher-clean-medium-r-normal-*-10-100-75-75-c-60-iso8859-1}
  {Normal -misc-fixed-medium-r-semicondensed-*-13-120-75-75-c-60-iso8859-1}
  {Big -misc-fixed-medium-r-normal--14-130-75-75-c-70-iso8859-1}
  {Huge -misc-fixed-medium-r-normal--20-200-75-75-c-100-iso8859-1}
} {
  .mbar.view.menu.fontsize add radiobutton \
     -label [lindex $i 0] -variable Font -value [lindex $i 1] \
     -command "ChangeFont [lindex $i 1]"
}
set Font -misc-fixed-medium-r-semicondensed-*-13-120-75-75-c-60-iso8859-1
menu .mbar.view.menu.height
foreach i {16 20 24 30 36 40 48 60 72} {
  .mbar.view.menu.height add radiobutton -label "$i lines" \
    -variable Height -value $i -command "ChangeHeight $i"
}
set Height 24
menu .mbar.view.menu.width
foreach i {40 64 80 92 112 124 140 160} {
  .mbar.view.menu.width add radiobutton -label "$i chars" \
    -variable Width -value $i -command "ChangeWidth $i"
}
set Width 80
menu .mbar.view.menu.wrap
.mbar.view.menu.wrap add radiobutton -label Word\
  -variable Wrap -value word -command {ChangeWrap word}
.mbar.view.menu.wrap add radiobutton -label Character\
  -variable Wrap -value char -command {ChangeWrap char}
.mbar.view.menu.wrap add radiobutton -label None\
  -variable Wrap -value none -command {ChangeWrap none}
set Wrap char


menu .mbar.help.menu
.mbar.help.menu add command -label {About this program...} \
  -command HelpAboutThisProgram

# Put a speed bar directly below the menu bar
#
# To change the contents of the speedbar, just add (or remove) entrys
# to the list on the 2nd argument to the "foreach" statement below.  The
# first field of each entry is the text to go in the speed button, the
# second field is the procedure to call when the speed button is pressed.
#
frame .sbar -bd 1  -relief raised
pack .sbar -side top -expand 1 -fill x
set i 0
foreach btn {
  {Save FileSave}
  {Copy EditCopy}
  {Paste EditPaste}
  {Search EditSearch}
  {{24} {ChangeHeight 24}}
  {{36} {ChangeHeight 36}}
  {{48} {ChangeHeight 48}}
  {{60} {ChangeHeight 60}}
  {Small {ChangeFont -schumacher-clean-medium-r-normal--8-80-75-75-c-50-iso8859-1}}
  {Normal {ChangeFont -misc-fixed-medium-r-semicondensed-*-13-120-75-75-c-60-iso8859-1}}
     
} {
  incr i
  button .sbar.b$i -text [lindex $btn 0] \
    -font -adobe-helvetica-medium-r-normal-*-8-80-75-75-p-46-iso8859-1 \
    -command [lindex $btn 1] \
    -padx 1 -pady 0
  pack .sbar.b$i -side left -padx 1 -pady 1
}

# The ".mesg" frame is used to display information about the current
# file and our position within that file, at the bottom of the screen.
#
frame .mesg -bd 1 -relief raised
pack .mesg -side bottom -fill x
label .mesg.name -text $FullPath -font $Font
pack .mesg.name -side left
frame .mesg.spacer
pack .mesg.spacer -side left -fill x -expand 1
label .mesg.line -width 20 -font $Font
pack .mesg.line -side left

# The main text widget with its scroll bar.
#
scrollbar .sb -orient vertical -command {.t yview} \
  -highlightthickness 0 -bd 1 -relief raised
pack .sb -side right -fill y
text .t -font $Font -width $Width -height $Height -yscrollcommand {.sb set} \
   -wrap $Wrap -bd 1 -relief raised -padx 3 -pady 2 \
   -exportselection 0 -highlightthickness 0
bindtags .t .t
pack .t -fill both -expand 1
.t insert end {}
#update
#update idletasks

# Change the height of the editor.  The argument is the number of rows
# of text to display at one time.
#
proc ChangeHeight newheight {
  global Width Height
  .t config -height $newheight -width $Width
  set Height $newheight
}

# Change the width of the editor.  The argument is the number of columns
# of text to display.
#
proc ChangeWidth newwidth {
  global Height Width
  .t config -height $Height -width $newwidth
  set Width $newwidth
}

# Change the word-wrap mode for the editor.  The argument should be
# one of:   word   char   none
#
proc ChangeWrap newmode {
  .t config -wrap $newmode
}

# Change the font used by the editor.  The argument is the name of the
# font to use.  For best results, use a constant-width font.
#
proc ChangeFont newfont {
  global Font
  .t config -font $newfont
  .mesg.name config -font $newfont
  .mesg.line config -font $newfont
  set Font $newfont
}

# The procedure defined below implements a generic dialog box.  The
# arguments are as follows:
#
#   position      The new dialog box is centered over the window given
#                 by this argument
#
#   title         This is the title for the dialog box
#
#   build         This procedure is called to construct the top-most
#                 panel of the dialog box.  The first argument to the
#                 procedure is the name of the frame widget which
#                 is the top panel.  Subsequent arguments are given by
#                 the "buildargs" parameter.
#
#   buildargs     This is arguments to the "build" command which come
#                 after the name of the top panel widget.
#
#   btns          This is a list of button descriptions.  Each button
#                 description consists of the name of the button and
#                 some text to be displayed beside that button.
#
# The procedure builds a model dialog box and waits for a button to be
# pressed.  When a button is pressed, the dialog box goes away and the
# procedure returns an integer which is the index of the selected button.
# The first button is numbered 0.
#
proc ModalDialogBox {pos title build buildargs btns} {
  global dialog_button
  if [winfo exists .d] {destroy .d}
  toplevel .d -class Dialog
  wm title .d $title
  wm iconname .d Dialog
  frame .d.msg -relief raised -bd 1
  $build .d.msg $buildargs
  pack .d.msg -side top -fill both -expand 1
  set cnt -1
  foreach btn $btns {
    incr cnt
    set btnname [lindex $btn 0]
    set btntext [lindex $btn 1]
    frame .d.x$cnt -relief raised -bd 1
    if [llength $btn]==3 {
      set cmd "[lindex $btn 2] .d; set dialog_button $cnt"
    } else {
      set cmd "set dialog_button $cnt"
    }
    button .d.x$cnt.btn -text $btnname -command $cmd -width 9
    pack .d.x$cnt.btn -side left -padx 5 -pady 5
    message .d.x$cnt.msg -text $btntext -width 10c
    pack .d.x$cnt.msg -anchor w -padx 5 -pady 5
    pack .d.x$cnt -side top -fill x
  }
  wm withdraw .d
  update idletasks
  set x [expr [winfo rootx $pos] + ([winfo width $pos]-[winfo reqwidth .d])/2]
  set y [expr [winfo rooty $pos] + ([winfo height $pos]-[winfo reqheight .d])/2]
  wm geometry .d +$x+$y
  wm deiconify .d
  set old_focus [focus]
  focus .d
  grab set .d
  bind .d <ButtonPress> {
    if {![string match .d.* %W]} {
      bell
      wm withdraw .d
      wm deiconify .d
    }
  }
  tkwait variable dialog_button
  grab release .d
  destroy .d
  focus $old_focus
  return $dialog_button
}

#
# The following procedures are used to construct a dialog box header which
# contains an icon and a message.  The difference is in the icon.
#
proc QuestionMessage {w text} {
  label $w.icon -bitmap info
  pack $w.icon -side left -padx 15 -pady 15
  message $w.msg -text $text -width 10c
  pack $w.msg -padx 15 -pady 15 -anchor w
}
proc WarningMessage {w text} {
  label $w.icon -bitmap warning
  pack $w.icon -side left -padx 15 -pady 15
  message $w.msg -text $text -width 10c
  pack $w.msg -padx 15 -pady 15 -anchor w
}

#
# The following procedure creates a search dialog box.  Or, if the dialog
# box already exists, it raises and deiconifies it.
#
proc EditSearch {} {
  global FullPath
  if [winfo exists .srch] {
    wm deiconify .srch
    raise .srch
    return
  }
  toplevel .srch -class Dialog
  wm title .srch [file tail $FullPath]
  wm iconname .srch [file tail $FullPath]
  wm withdraw .srch
  frame .srch.pat
  pack .srch.pat -side top -padx 5 -pady 5 -fill x
  label .srch.pat.l -text {Search for:}
  pack .srch.pat.l -side left
  entry .srch.pat.e -width 40 -bd 2 -relief sunken
  pack .srch.pat.e -side left -fill x
  bind .srch.pat.e <Return> FindNext
  frame .srch.to
  pack .srch.to -side top -padx 5 -pady 5 -fill x
  label .srch.to.l -text {Change to:}
  pack .srch.to.l -side left
  entry .srch.to.e -width 40 -bd 2 -relief sunken
  pack .srch.to.e -side left -fill x
  frame .srch.btn
  pack .srch.btn -side top -padx 5 -pady 5 -fill x
  foreach b {
    {close    Close     {destroy .srch}}
    {prev     Previous  FindPrev}
    {next     Next      FindNext}
    {replace  Replace   Replace}
    {all      {Replace All} ReplaceAll}
  } {
    button .srch.btn.[lindex $b 0] -text [lindex $b 1] -command [lindex $b 2]
    pack .srch.btn.[lindex $b 0] -side left -padx 5 -pady 5 -expand 1
  }
  update idletasks
  set y [expr {[winfo rooty .] - [winfo reqheight .srch] - 25}]
  if {$y<0} {set y 0}
  wm geometry .srch +[winfo rootx .]+$y
  wm deiconify .srch
  focus .srch.pat.e
}

#
# Using the pattern string located in the entry widget .srch.pat.e,
# find the next occurance of the pattern in the text widget.
#
proc FindNext {} {
  set pattern [.srch.pat.e get]
  set i [string first $pattern [.t get {insert +1 chars} end]]
  if {$i>=0} {
    incr i
    .t mark set insert "insert +$i chars"
    textFixup
  } else {
    set i [string first $pattern [.t get 1.0 insert]]
    if {$i>=0} {.t mark set insert "1.0 +$i chars"; textFixup}
  }
  focus .t
}
proc FindPrev {} {
  set pattern [.srch.pat.e get]
  set i [string last $pattern [.t get 1.0 insert]]
  if {$i>=0} {
    .t mark set insert "1.0 +$i chars"
    textFixup
  } else {
    set i [string last $pattern [.t get insert end]]
    if {$i>=0} {.t mark set insert "insert +$i chars"; textFixup}
  }
  focus .t
}

#
# If the text at the insertion cursor matches the pattern in the
# search dialog, then replace the pattern with the substitution string
# and find the next occurance of the pattern.
#
proc Replace {} {
  global Dirty
  set pattern [.srch.pat.e get]
  set n [string length $pattern]
  set substitute [.srch.to.e get]
  if {[.t get insert "insert +$n chars"]==$pattern} {
    .t delete insert "insert +$n chars"
    .t insert insert $substitute
    set Dirty 1
    FindNext
    return 1
  } else {
    return 0
  }
}
proc ReplaceAll {} {
  FindNext
  while {[Replace]} {}
}

############################################################################
# The following code implements the file selection dialog box
#

#
# These variables are used internally by the GetFileName procedure.
# They should not be accessed or changed by any outside procedure.
#
set GFN(path) [pwd]           ;# The directory we are currently reading
set GFN(name)   {}            ;# Name of a selected file
set GFN(result) {}            ;# Write to this to trigger end-of-file
set GFN(filter) {*}           ;# Filter filenames using this pattern

#
# This is the main routine -- the only routine which should be accessed
# or used from outside this file.  This routine causes a file chooser
# dialog box to appear centered in the window "pos"  The title of the
# dialog box will be "title".  File names will be filtered using "filter".
#
# The procedure will return either the name of a selected file or {} if
# the cancel button is pressed.  The name returned does not necessary
# refer to a file that exists -- the user may have typed in a bogus
# name
#
proc GetFileName {pos title filter} {
  global GFN

#  set GFN(filter) $filter
  if [winfo exists .gfn] {destroy .gfn}
  toplevel .gfn -class Dialog
  wm title .gfn $title
  wm iconname .gfn Dialog
  wm withdraw .gfn
  frame .gfn.f2
  pack .gfn.f2 -side top -padx 2 -pady 2 -fill x
  label .gfn.f2.l -text Name
  pack .gfn.f2.l -side left
  entry .gfn.f2.e -bd 2 -relief sunken
  bind .gfn.f2.e <Return> {GFN_ReturnPressedInEntry}
  pack .gfn.f2.e -side left -expand 1 -fill x
  set old_focus [focus]
  focus .gfn.f2.e
  frame .gfn.f3
  listbox .gfn.f3.ll -yscrollcommand {.gfn.f3.sl set} -bd 2 -relief sunken \
     -exportselection 0 -font fixed -selectmode single
  bind .gfn.f3.ll <Button-1> {GFN_SingleClickLeftListbox %y; break}
  bind .gfn.f3.ll <Double-Button-1> {set GFN(result) $GFN(path)/$GFN(name)}
  scrollbar .gfn.f3.sl -command {.gfn.f3.ll yview} -orient vertical
  frame .gfn.f3.spacer -width 5
  listbox .gfn.f3.lr -yscrollcommand {.gfn.f3.sr set} -bd 2 -relief sunken \
     -exportselection 0 -font fixed -selectmode single
  bind .gfn.f3.lr <Double-Button-1> {GFN_DoubleClickRightListbox %y}
  scrollbar .gfn.f3.sr -command {.gfn.f3.lr yview} -orient vertical
  pack .gfn.f3.ll .gfn.f3.sl .gfn.f3.spacer .gfn.f3.lr .gfn.f3.sr \
     -side left -fill y
  pack .gfn.f3 -padx 2 -pady 5
  frame .gfn.f4
  button .gfn.f4.ok -text Ok -command {set GFN(result) $GFN(path)/$GFN(name)}
  button .gfn.f4.cancel -text Cancel -command {set GFN(result) {}}
  pack .gfn.f4.ok .gfn.f4.cancel -expand 1 -side left -pady 5
  pack .gfn.f4 -fill x
  update idletasks
  GFN_FillListboxes
  set x [expr [winfo rootx $pos] + ([winfo width $pos]-[winfo reqwidth .gfn])/2]
  set y [expr [winfo rooty $pos] + ([winfo height $pos]-[winfo reqheight .gfn])/2]
  wm geometry .gfn +$x+$y
  wm deiconify .gfn
  grab set .gfn
  set GFN(result) no
  bind .gfn <ButtonPress> {
    if {![string match .gfn.* %W]} {
      bell
      wm withdraw .gfn
      wm deiconify .gfn
    }
  }
  tkwait variable GFN(result)
  grab release .gfn
  destroy .gfn
  focus $old_focus
  return $GFN(result)
}

#
# This routine is called to refill the listboxes.  The left listbox
# (.gfn.f3.ll) gets the names of all files in the directory $path
# which match the filter given by $filter.  The right listbox (.gfn.f3.lr)
# gets the name of every directory in the directory $path, plus the
# special directories "." and "..".  The text in the entry box is also
# updated.
#
proc GFN_FillListboxes {} {
  global GFN
  .gfn config -cursor watch
  update
  .gfn.f3.lr delete 0 end
  .gfn.f3.ll delete 0 end
  .gfn.f3.lr insert end .
  if {"$GFN(path)"!=""} {.gfn.f3.lr insert end ..}
  GFN_ChangeEntryText $GFN(path)/$GFN(filter)
  .gfn.f4.ok config -state disabled
  if {[file isdir /$GFN(path)] && [file readable /$GFN(path)]} {
    foreach fn [lsort [glob -nocomplain $GFN(path)/*]] {
      if [file isdir $fn] {
       .gfn.f3.lr insert end [file tail $fn]
      } elseif [string match $GFN(filter) $fn] {
        .gfn.f3.ll insert end [file tail $fn]
      }
    }
  }
  .gfn config -cursor {}
}

#
# This routine is called to change the text in the entry box.
#
proc GFN_ChangeEntryText {text} {
  .gfn.f2.e delete 0 end
  .gfn.f2.e insert 0 $text
  .gfn.f2.e icursor end
  # TBD: Scroll the entry widget so that the insertion cursor is visible
}

#
# This routine is called whenever the user double-clicks on the right
# listbox -- the listbox which contains directory names.  This means
# that the user wants to change directories
#
proc GFN_DoubleClickRightListbox {y} {
  global GFN
  set dir [.gfn.f3.lr get [.gfn.f3.lr nearest $y]]
  if {"$dir"=="."} {
    # do nothing
  } elseif {"$dir"==".."} {
    set GFN(path) [file dirname $GFN(path)]
  } else {
    set GFN(path) $GFN(path)/$dir
  }
  if {"$GFN(path)"=="/"} {set GFN(path) {}}
  GFN_FillListboxes
}

#
# This routine is called whenever the user single-clicks any of the
# files named in the left listbox.  The effect is to set the variable
# GFN(name) to the file name so that the Ok button will know how to
# set the result if and when it is pressed
#
proc GFN_SingleClickLeftListbox {y} {
  global GFN
  #.gfn.f3.ll select from [.gfn.f3.ll nearest $y]
  set index [.gfn.f3.ll nearest $y]
  .gfn.f3.ll select clear 0 end
  .gfn.f3.ll select set $index
  set GFN(name) [.gfn.f3.ll get $index]
  GFN_ChangeEntryText $GFN(path)/$GFN(name)
  .gfn.f4.ok config -state normal
}

#
# This routine is called whenever the user presses Return inside
# the entry box.  If the filename in the entry box does not contain
# any wildcards and isn't a directory, then return that filename.  
# If the box does contain wildcard, then set the filter to the 
# wildcard name and refill the listbox.  If the entry contains the
# name of a directory, then refill the list boxes from that directory.
#
proc GFN_ReturnPressedInEntry {} {
  global GFN
  set fn [.gfn.f2.e get]
  set basename [file tail $fn]
  if [regexp {[*?[]} $basename] {
    set GFN(filter) $basename
    set GFN(path) [file dirname $fn]
    GFN_FillListboxes
  } elseif [file isdir $fn] {
    set GFN(path) $fn
    GFN_FillListboxes
  } else {
    set GFN(result) $fn
  }
  if {"$GFN(path)"=="/"} {set GFN(path) {}}
}

# Here ends the code for implementing the file dialog box
##############################################################################

#
# The following function loads the text of a file into the text widget.
#
proc FileLoad {filename} {
  . config -cursor watch
  .t delete 1.0 end
  if [file readable $filename] {
    set f [open $filename]
    .t insert end [read $f]
    close $f
  }
  focus .t
  .t mark set insert 1.0
  textFixup
  . config -cursor {}
  global Dirty
  set Dirty 0
}

#
# This routine is called in response to the "File/Quit" menu action.
#
proc FileQuit {} {
  global Dirty FullPath
  while {$Dirty} {
    set response [ModalDialogBox . {Save Changes?} QuestionMessage \
      "You have made changes to the file \"$FullPath\". Do you want to save those changes before exiting?" {
         {Save     {Save the changes to the file before exiting}}
         {Discard  {Don't save the changes, but exit anyway}}
         {Cancel   {Don't exit.  Return to the editor}}
      }]
    if $response==2 return
    if $response==0 FileSave
    if $response==1 {set Dirty 0}
  }
  destroy .
}

#
# Write the editor contents into the file FullPath.
#
# This routine is never called unless we are sure that the file has
# write permission.
#
proc FileWrite {} {
  global FullPath Dirty
  set f [open $FullPath w]
  puts $f [string trimright [.t get 1.0 end]]
  close $f
  set Dirty 0
}

#
# This routine is called in order to save the editor contents to its
# default filename.
#
proc FileSave {} {
  global FullPath Dirty
  if {"$FullPath"==""} {
    FileSaveAs
    return
  }
  if {[file exists $FullPath] && ![file writable $FullPath]} {
    set response [ModalDialogBox . {Can't Write} WarningMessage \
      "The file \"$FullPath\" is not writable.  What do you want to do?" {
      {{Save As...}  {Save the editor contents to a different filename}}
      {{Try Again}   {Attempt to change permissions to allow the file to be written}}
      {{Cancel}      {Don't save the contents of the editor}}
    }]
    if {$response==2} return
    if {$response==0} {FileSaveAs; break}
    catch {exec chmod +w $FullPath}
    if {![file writable $FullPath]} {
      set response [ModalDialogbox . {Still Can't Write} WarningMessage \
        "Unable to enable write permission for the file \"$FullPath\".  What now?" {
        {{Save As...}  {Save the editor contents to a different filename}}
        {{Cancel}      {Don't save the contents of the editor}}
      }]
      if {$response==1} return
      FileSaveAs
    } else {
      FileWrite
    }
  } else {
    FileWrite
  }
}

#
# This routine is called query the user for a new filename in which
# to write the contents of a file.
#
proc FileSaveAs {} {
  global FullPath
  set loop 1
  while {$loop} {
    set loop 0
    set newname [GetFileName . {Save As} {*}]
    if {"$newname"==""} return
    if {[file writable $newname]} {
      set response [ModalDialogBox . {File Exists} WarningMessage \
        "The file \"$newname\" already exists.  What do you want to do?" {
          {Overwrite  {Replace the existing contents of the file with the current contents of the editor}}
          {Rename     {Choose a new name to which to write the file}}
          {Cancel     {Don't save the editor contents}}
        }]
      if $response==1 {set loop 1; continue}
      if $response==2 return
      if $response==0 break
    }   
  }
  set FullPath $newname; 
  .mesg.name config -text $FullPath
  wm title . [file tail $FullPath]
  wm iconname . [file tail $FullPath]
  FileSave
}

#
# Create a new file editor
#
proc FileNew {} {
  global argv0
  catch "exec $argv0 &"
}

#
# Open a new file
#
proc FileOpen {} {
  set newname [GetFileName . {Open} {*}]
  if {"$newname"!=""} {
    global argv0
    catch "exec $argv0 $newname &"
  }
}

#
# This routine is called in response to the File/Revert menu selection
#
proc FileRevert {} {
  global FullPath Dirty
  set oldname $FullPath
  while {$Dirty} {
    set response [ModalDialogBox . {Undo Changes?} WarningMessage \
      "The contents of the editor have been modified. If you reload the editor you will lose your changes!" {
        {Revert  {Go ahead and overwrite the changes in the editor}}
        {Save    {Save editor contents to a different file before reloading}}
        {Cancel  {Don't reload the editor}}
      }]
    if $response==0 break
    if $response==1 FileSaveAs
    if $response==2 return
  }
  set FullPath $oldname
  .mesg.name config -text $FullPath
  FileLoad $FullPath
}

#
# This routine is called to compute and display the line information
# in the message frame.  It should be called whenever the insertion
# cursor moves.
#
proc ComputeLine {} {
  scan [.t index insert] {%d.%*d} now
  scan [.t index end] {%d.%*d} last
  incr last -1
  .mesg.line config -text "line $now of $last"
}

# A routine for dispensing the selection.  The selection is always owned
# by the window ".".  Its value is stored in the variable "Selection"
#
set Selection {}
selection handle . RetrieveSelection
proc RetrieveSelection {offset max} {
  global Selection
  return [string range $Selection $offset [expr {$offset+$max}]]
}

# This routine is called whenever "." owns the selection but another
# window claims ownership.
#
proc LoseSelection {} {
  global Selection
  set Selection {}
}

# Copy the text selected in the text widget into the Selection variable,
# then claim ownership of the selection.
#
proc EditCopy {} {
  global Selection
  catch {
    set Selection [.t get sel.first sel.last]
    selection own . LoseSelection
  }
}
proc EditCut {} {
  EditCopy
  EditDelete
}
proc EditPaste {} {
  catch {
    .t insert insert [selection get]
    textChange
  }
}
proc EditDelete {} {
  catch {
    .t delete sel.first sel.last
    textChange
  }
}

# The default bindings for the text widget are woefully inadequate.
# The following section implements improved bindings.
#
bind .t <KeyPress> {if {"%A"!=""} {.t insert insert %A; textChange}}
bind .t <Return> {.t insert insert \n; textChange}
bind .t <Delete> {.t delete {insert -1 char}; textChange}
#bind .t <Delete> {.t delete insert; textChange}
bind .t <BackSpace> {.t delete {insert -1 char}; textChange}
bind .t <Control-h> {.t delete {insert -1 char}; textChange}
bind .t <Control-x> EditCut
bind .t <Control-c> EditCopy
bind .t <Control-v> EditPaste
bind .t <Control-u> {}
bind .t <Control-d> {.t delete insert; textChange}
bind .t <Control-w> {}
bind .t <Control-k> {.t delete insert {insert lineend}; textChange}
bind .t <Control-p> {textAnchor {insert -1 line}}
bind .t <Control-a> {textAnchor {insert linestart}}
bind .t <Control-e> {textAnchor {insert lineend}}
bind .t <Control-n> {textAnchor {insert +1 line}}
bind .t <Control-b> {textAnchor {insert -1 char}}
bind .t <Control-f> {textAnchor {insert +1 char}}
bind .t <Left> {textAnchor {insert -1 char}}
bind .t <Right> {textAnchor {insert +1 char}}
bind .t <Home> {textAnchor {insert linestart}}
bind .t <End> {textAnchor {insert lineend}}
bind .t <Up> {textAnchor {insert -1 line}}
bind .t <Down> {textAnchor {insert +1 line}}
bind .t <Prior> {textAnchor "insert -$Height lines"}
bind .t <Next> {textAnchor "insert +$Height lines"}
bind .t <Control-Left> {textInsert {insert -1 char}; textSel tk_anchor insert}
bind .t <Control-Right> {textInsert {insert +1 char}; textSel tk_anchor insert}
bind .t <Control-Home> {textInsert {insert linestart}; textSel tk_anchor insert}
bind .t <Control-End> {textInsert {insert lineend}; textSel tk_anchor insert}
bind .t <Control-Up> {textInsert {insert -1 line}; textSel tk_anchor insert}
bind .t <Control-Down> {textInsert {insert +1 line}; textSel tk_anchor insert}
bind .t <Control-Prior> {
  textInsert "insert -$Height lines"
  textSel tk_anchor insert
}
bind .t <Control-Next> {
  textInsert "insert +$Height lines"
  textSel tk_anchor insert
}
bind .t <1> {.t tag remove sel 0.0 end; textAnchor @%x,%y; focus .t}
bind .t <B1-Motion> {textInsert @%x,%y; textSel tk_anchor insert}
bind .t <Double-1> {textSel {insert wordstart} {insert wordend}}
bind .t <Triple-1> {textSel {insert linestart} {insert lineend}}
bind .t <Shift-1> {textInsert @%x,%y; focus .t; textSel tk_anchor insert}
bind .t <Control-B1-Motion> {textSel tk_anchor @%x,%y}
bind .t <2> {.t scan mark %y}
bind .t <B2-Motion> {.t scan dragto %y}

#
# The next group of functions are used by the bindings.
#
proc textChange {} {textFixup; global Dirty; set Dirty 1}
proc textFixup {} {.t yview -pickplace insert; ComputeLine }
proc textSel {a b} {
  if [.t compare $a < $b] {
    .t tag remove sel 0.0 $a
    .t tag add sel $a $b
    .t tag remove sel $b end
  } else {
    .t tag remove sel 0.0 $b
    .t tag add sel $b $a
    .t tag remove sel $a end
  }
}
proc textInsert {w} {.t mark set insert $w; textFixup}
proc textAnchor {w} {.t mark set tk_anchor $w; textInsert $w}

#
# Finally, load the file into the editor and begin editing
#
FileLoad $FullPath
