# [[ Header ]]

# Copyright (c) 1996 by SoftWorks.  All Rights Reserved.
# File:   main.tcl
# Author: Richard Schwaninger
# Date:   20/02/1996
# System: chiron.standalone
# RCS:    $Id: font.tcl,v 1.1.1.1 1996/02/26 06:06:45 risc Exp $

# [endfold]

# XLFD
# registry
# foundry
# family
# weight
# slant
# setwidth
# style
# pixel
# size
# xres
# yres
# spacing
# width
# charset
# encoding

# [[ globals ]]

# standard template for outgoing real font defs
set Font+Tmpl+Default \
      "%registry-%foundry-%family-%weight-%slant-%width-%style-%pixel-%size-%xres-%yres-%spacing-%avgwidth-%charset-%encoding"

# conversion table for different attributes like italic and oblique, 
# also defines things that do not exist at all (like light fonts).

array set Font+Prop+Default {
   slant.italic i
   slant.normal r
   weight.light normal
   width.wide normal
   width.condensed normal
   size {[expr \$val*10]}
}

array set Font+Prop+adobe-helvetica {
   slant.italic o
   weight.normal medium
   weight.light medium
}

array set Font+Prop+adobe-times {
   weight.normal medium
   weight.light medium
}

array set Font+Prop+misc-fixed {
   weight.normal medium
   weight.light medium
   style {}
   width.condensed semicondensed
}

# [endfold]
# [[ font_Init ]]

# this routine has to be called before you can use any of the
# predefined font schemes. Of course you can define your own fonts
# anywhere in your code but this is cumbersome and nonportable.

proc font_Init { args } {
   Font def Helvetica \
	 {registry= foundry=adobe family=helvetica charset=iso8859 \
	 encoding=1 spacing=p pixel=0 } \
	 {slant=normal width=normal weight=normal  size=14}

   Font def Times \
	 {registry= foundry=adobe family=times charset=iso8859 \
	 encoding=1 spacing=p pixel=0 } \
	 {weight.normal=medium} \
	 {slant=normal width=normal weight=normal size=14}

   Font def Terminal \
	 {registry= foundry=misc family=fixed charset=iso8859 \
	 encoding=1 spacing=c pixel=0 slant=r} \
	 {slant=normal width=normal weight=normal size=14}
}

# [endfold]

# [[ Font ]]

# + font management
# the generic toplevel font manage routine. Follows standard call
# syntax like eg the info command. 

# Input is a logical font name, output is a standard X
# font description. 
# 
# The idea behind this routine:
# we define a logical font name (eg Times, or LabelFont) with the basic
# properties of the font (eg registry, foundry, charset, but also
# default size and slant). The definition itself is not very portable but at
# least it can be kept in a single place.
# A definition looks like this:
#         Font def LabelFont \
#	    {registry= foundry=adobe family=helvetica charset=iso8859 \
#	     encoding=1 spacing=p pixel=0 } \
#	    {slant=normal width=normal weight=normal  size=14}
# As you can see I used the standard XLFD conventions (don't know
# enough about Windows or Mac). The first list gives things that
# should be constant for this font (that is you can no longer change
# eg. the encoding for the logical font you create). The second list
# states the default values used if you don't override them when you
# get the font. All values not mentioned in one of the lists will be
# replaced by '*' in the output.

# Logical fontnames make it easy to change fonts througout the source
# code as the use looks always something like this:
#         button .b -text "huhu?" -font [Font get LabelFont]
# 
# To give the button a bold appearance you can define a new logical
# font or (if this is just a onetime change) do this
#         button .b config -font [Font get LabelFont weight=bold]

# Of course you can do more complicated things:
#         button .b -font [Font get LabelFont weight=bold \
#                  slant=italic size=24]

# 'Font set' may be used to incrementally define a logical font (for the
# really complicated things...)

# 'Font exists <font>' returns 1 if a logical font with this name
# exists, 0 otherwise
#          Font exists LabelFont

# 'Font names' returns a list of all defined logical fonts.

# 'Font copy <old> <new' copies a logical font definition and makes
# that font availabe under a new name. You may now use 'Font set' to
# change the attributes of the new font. NOTE: Attributes defined
# statically with 'Font def' (the first list - see above) cannot be
# changed!

# Implementation note: 
# I could have used the standard option syntax '-xxx yyy', but it
# would have been tricky to easily integrate them into XLFDs because
# they also contain dashes. I dont say it is not possible, but...

# here is a description of the arguments:
# > cmd - the subcommand to execute, one of get, set, def, info,
# exists, names or copy
# > name - generic fontname
# > args - arguments, depending on cmd
# < returns a standard X font name most of the time (depending on
# subcommand). 

proc Font { cmd {name Default} args } {
   switch $cmd {
      get {
	 return [FontGen $name $args]
      }
      set {
	 upvar \#0 Font+Def+$name var
	 upvar \#0 Font+Tmpl+$name tmpl
	 if { ![info exists tmpl] } {
	    error "Font '$name' does not exist"
	 }
	 foreach i $args {
	    regexp {^([^=]+)=(.*)} $i dummy spec val
	    set var($spec) $val
	 }
      }
      def {
	 set template [lindex $args 0]
	 set default [lindex $args 1]
	 FontTemplate $name $template
	 upvar \#0 Font+Def+$name var
	 foreach i $default {
	    regexp {^([^=]+)=(.*)} $i dummy spec val
	    set var($spec) $val
	 }
      }
      info {
	 upvar \#0 Font+Def+Default var
	 foreach i [array names var] {
	    set x($i) $var($i)
	 }
	 upvar \#0 Font+Def+$name var
	 foreach i [array names var] {
	    set x($i) $var($i)
	 }
	 set y {}
	 foreach i [array names x] {
	    lappend y "$i=$x($i)"
	 }
	 return $y
      }
      names {
	 set x {}
	 foreach i [info globals Font+Tmpl+$name] {
	    lappend x [crange $i 10 end]
	 }
	 return $x
      }	 
      exists {
	 global Font+Tmpl+$name
	 if {[info exists Font+Tmpl+$name]} {
	    return 1
	 } 
	 return 0
      }
      copy {
	 upvar \#0 Font+Tmpl+$name srctmpl
	 if {![info exists srctmpl]} {
	    error "Source font '$name' does not exist"
	 }
	 set args [lassign $args dst]
	 upvar \#0 Font+Tmpl+$dst dsttmpl
	 if {[info exists dsttmpl]} {
	    error "Destination font '$name' does exist"
	 }
	 set dsttmpl $srctmpl
	 upvar \#0 Font+Def+$name srcdef
	 upvar \#0 Font+Def+$dst dstdef
	 array set dstdef [array get srcdef] 
	 eval Font set $dst $args
      }
      default {
	 error "Font <cmd> <args>"
      }
   }
   return {}
}

# [endfold]
# [[ FontGen ]]

# - generate X font
# this routine is local

# takes a logical font name and a list of attributes and constructs a
# proper XLFD.

# > name - logical font name
# > par - a list of attributes in the form 'attr=value'
# < returns the X font name

proc FontGen { name par } {
   upvar \#0 Font+Tmpl+$name orig
   upvar \#0 Font+Def+$name def

   # do not change the original, use a copy!
   if { ![info exists orig] } {
      error "Font '$name' does not exist"
   }
   set template $orig

   foreach i $par {
      regexp {^([^=]+)=(.*)} $i dummy spec val
      set val [FontSub $name $spec $val]
      regsub %$spec $template $val template
   }
   foreach spec [array names def] {
      set val [FontSub $name $spec $def($spec)]
      regsub %$spec $template $val template
   }
   regsub -all {%[^-]+} $template * template
   return $template
}

# [endfold]
# [[ FontSub ]]

# - substitute attributes
# this routine is local

# substitute real attribute names for logical ones. This copes with
# the fact that we do want a layer in between our own attribute names
# and the names of X. ('italic' is mapped to either 'i' or 'o' for
# example). This is a two stage process - if we don't find a mapping
# in the current font we look into the 'Default' font. 

# The substitutions can even handle executable tcl code (see the size
# definition where we multiply the point value by 10 to get the X value.

# > name - logical font name
# > spec - attribute name (eg slant)
# > val - attribute value (eg italic)
# < returns the definition that can be inserted into the XLFD

proc FontSub { name spec val } {
   upvar \#0 Font+Tmpl+$name tmpl

   regexp -- {-([^-]+-[^-]+)} $tmpl dummy propname
   upvar \#0 Font+Prop+$propname prop

   set form \$val
   if { [info exists prop($spec.$val)] } {
      set form $prop($spec.$val)
   } else {
      if { [info exists prop($spec)] } {
	 set form $prop($spec)
      } else {
	 upvar \#0 Font+Prop+Default prop
	 if { [info exists prop($spec.$val)] } {
	    set form $prop($spec.$val)
	 } else {
	    if { [info exists prop($spec)] } {
	       set form $prop($spec)
	    }
	 }
      }
   }
   eval set ret $form
   return $ret
}

# [endfold]
# [[ FontTemplate ]]

# - make font template
# this is a local routine

# create a XLFD template for the new font with the constant items
# substituted. Start with the default template.

# > name - new logical name
# > par - a list of default parameters
# < returns nothing, sets the corresponding global vars

proc FontTemplate { name par } {
   upvar \#0 Font+Tmpl+Default orig
   upvar \#0 Font+Tmpl+$name template
   # don't use the original here
   set template $orig

   # alas: we need foundry and family set in template to allow for
   # FontSub to be called
   set foundry adobe
   set family helvetica
   regexp {foundry=([^ 	]+)} $par dummy foundry
   regexp {family=([^ 	]+)} $par dummy family
   regsub %foundry $template $foundry template
   regsub %family $template $family template   

   foreach i $par {
      regexp {^([^=]+)=(.*)} $i dummy spec val
      set val [FontSub $name $spec $val]
      regsub %$spec $template $val template

   }
}

# [endfold]

# EOF
