--------------------------------------------------------------------
--
-- freq.adb --
--
-- Copyright (c) 1995 Terry J. Westley
--
-- See the file "license.terms" for information on usage and redistribution
-- of this file, and for a DISCLAIMER OF ALL WARRANTIES.
--
--
--------------------------------------------------------------------

with C_Aux;
with Interfaces.C.Strings;
with Tcl;
with Text_IO;

procedure Freq is -- Frequency counter

   package C renames Interfaces.C;

   function "=" (Left, Right : in Tcl.Integer_Hash.HashEntry_Ptr)
      return Boolean renames Tcl.Integer_Hash."=";
   function "=" (Left, Right : in C.Int) return Boolean renames C."=";

   Line		 : C.Strings.Chars_Ptr;
   Freq_Count	 : Integer;
   Item		 : C.Strings.Chars_Ptr;
   Freq_Hash     : aliased Tcl.Integer_Hash.HashTable_Rec;
   Entry_Ptr     : Tcl.Integer_Hash.HashEntry_Ptr;
   Is_New_Entry  : aliased C.Int;
   Search        : aliased Tcl.Integer_Hash.HashSearch_Rec;

   procedure Get_Line (Line : in out C.Strings.Chars_Ptr) is
   -- This procedure gets a line from standard input and converts
   -- it to a "C" string.
      Input_Line : String (1..1024);
      Length	 : Natural;
   begin -- Get_Line
      Text_IO.Get_Line (Input_Line, Length);
      C.Strings.Free (Line);
      Line := C.Strings.New_String (Input_Line (1..Length));
   end Get_Line;

begin -- Freq

   -- create a hash table for holding frequency counts
   Tcl.Integer_Hash.InitHashTable (Freq_Hash'unchecked_access,
      Tcl.STRING_KEYS);

   -- read lines from standard input until
   -- end of file encountered
   while not Text_IO.End_of_File loop
      Get_Line (Line);
      -- create (or find, if already created) an entry for this line
      Entry_Ptr := Tcl.Integer_Hash.CreateHashEntry (
         Freq_Hash'unchecked_access, Line, Is_New_Entry'unchecked_access);
      if Is_New_Entry = 1 then
         Freq_Count := 1;
      else
         -- get the frequency count from the hash
         Freq_Count := Tcl.Integer_Hash.GetHashValue (Entry_Ptr) + 1;
      end if;
      -- Store the updated frequency count in the table.
      -- WARNING: We take advantage of the fact that an integer is the
      -- same size as a C pointer and store the count in the table,
      -- rather than a pointer to it.
      Tcl.Integer_Hash.SetHashValue (Entry_Ptr, Freq_Count);
   end loop;

   -- iterate through every item and print it and its frequency count
   Entry_Ptr := Tcl.Integer_Hash.FirstHashEntry (Freq_Hash'unchecked_access,
      Search'unchecked_access);
   while Entry_Ptr /= Null loop
      Freq_Count := Tcl.Integer_Hash.GetHashValue (Entry_Ptr);
      Item       := Tcl.Integer_Hash.GetHashKey (
      	 Freq_Hash'unchecked_access, Entry_Ptr);
      Text_IO.Put_Line (C_Aux.Value (Item) & Integer'image (Freq_Count));
      Entry_Ptr  := Tcl.Integer_Hash.NextHashEntry (Search'unchecked_access);
   end loop;

   -- delete the frequency counter hash table
   Tcl.Integer_Hash.DeleteHashTable (Freq_Hash'unchecked_access);

end Freq;
