--------------------------------------------------------------------
--
-- tcl.ads --
--
--	This is an Ada 95 interface to Tcl version 7.5a2.
--
-- Copyright (c) 1987-1994 The Regents of the University of California.
-- Copyright (c) 1994-1995 Sun Microsystems, Inc.
-- Copyright (c) 1995-1996 Terry J. Westley
--
-- See the file "license.terms" for information on usage and redistribution
-- of this file, and for a DISCLAIMER OF ALL WARRANTIES.
--
-- Notes on Ada interface:
--
-- For each Tcl function available in the public Tcl interface in tcl.h,
-- a direct pragma Import function (as close to the C prototype as we can
-- get in Ada) is available.  Also, more conventional Ada style interfaces
-- which uses Ada data types and exceptions in place of the Tcl command
-- completion code were added.
--
-- Things to do (not in any particluar order):
--
-- 1) Continue adding Ada-style subprograms
-- 2) Check for memory leaks, especially when returning C.Strings.Chars_Ptr,
--    then calling another function, such as C_Aux.Value, without the
--    possibility of freeing the C string.
-- 3) Examine all record types for conformance to C layout in Tcl C library.
-- 4) Convert TclTest to Ada.
-- 5) Comprehensive test of calling all procedures:
--    a) Are pragma's subprograms accurate interfaces?
--    b) Do all Ada-style subprograms call pragma versions correctly and
--       prevent memory leaks?
-- 6) Document each subprogram from Ousterhout's book or from code...
--    Don't assume user knows how to use each function already.
--
--------------------------------------------------------------------

with C_Aux;
with Interfaces.C.Strings;

package Tcl is

   Version       : constant String := "7.5";
   Major_Version : constant String := "7";
   Minor_Version : constant String := "5";
   Ada_Version	 : constant String := "b";

   package C renames Interfaces.C;

   --------------------------------------------------------------------
   --
   --	Private data types
   --
   --------------------------------------------------------------------

   type Interp_Rec     is private;
   type Interp_Ptr     is access all Interp_Rec;
   type Interp_Ptr_Ptr is access all Interp_Ptr;
   pragma Convention (C, Interp_Ptr);
   pragma Convention (C, Interp_Ptr_Ptr);
   -- A Tcl interpreter keeps the current state of Tcl script execution,
   -- including procedures, variables, and a stack.

   function Result  (Interp : in Interp_Ptr) return C.Strings.Chars_Ptr;
   function SResult (Interp : in Interp_Ptr) return String;
   pragma Inline (Result);
   -- Get the result or error string after completion of some function.

   function ErrorLine (Interp : in Interp_Ptr) return C.Int;
   pragma Inline (ErrorLine);
   -- Get the line on which the error occured during most recent
   -- script evaluation.

   type Trace_Ptr        is access all C.Int;
   type Command_Ptr      is access all C.Int;
   type AsyncHandler_Rec is private;
   type AsyncHandler_Ptr is access all AsyncHandler_Rec;
   type RegExp_Rec       is private;
   type RegExp_Ptr       is access all RegExp_Rec;

   --------------------------------------------------------------------
   --
   --	Command return codes
   --
   --------------------------------------------------------------------

   -- When a TCL command returns, the result string contains return
   -- information from the command.  In addition, the command procedure
   -- returns an integer value, which is one of the following:

   OK	    : constant :=  0; -- Command completed normally; the result string
                              -- contains the command's result.
   ERROR    : constant :=  1; -- The command couldn't be completed
                              -- successfully; the result string describes
                              -- what went wrong.
   RETRN    : constant :=  2; -- The command requests that the current
                              -- procedure return; the result string contains
                              -- the procedure's return value.
   BREAK    : constant :=  3; -- The command requests that the innermost loop
                              -- be exited;  the result string is meaningless.
   CONTINUE : constant :=  4; -- Go on to the next iteration of the current
                              -- loop; the result string is meaningless.

   RESULT_SIZE : constant := 200;

   --------------------------------------------------------------------
   --
   --	Argument descriptors for math function callbacks in expressions
   --
   --------------------------------------------------------------------

   type ValueType is (INT, DOUBLE, EITHER);
   for ValueType'size use 32;
   type ValueType_Ptr is access all ValueType;
   pragma Convention (C, ValueType_Ptr);

   type Value_Rec is
      record
         Typ         : ValueType;
         intValue    : C.Int;
         doubleValue : C.Double;
      end record;
   pragma Convention (C, Value_Rec);
   type Value_Ptr is access all Value_Rec;
   pragma Convention (C, Value_Ptr);

   --------------------------------------------------------------------
   --
   --	Procedure types
   --
   --------------------------------------------------------------------

   type AppInitProc_Ptr is access function (
      Interp        : in Interp_Ptr) return C.Int;
   pragma Convention (C, AppInitProc_Ptr);

   type FreeProc_Ptr is access procedure (
      blockPtr      : in C.Strings.Chars_Ptr);
   pragma Convention (C, FreeProc_Ptr);

   type PackageInitProc_Ptr is access function (
      Interp        : in Interp_Ptr) return C.Int;
   pragma Convention (C, PackageInitProc_Ptr);

   type PanicProc is access procedure (
      String1         : in C.Strings.Chars_Ptr;
      String2         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String3         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String4         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String5         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String6         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String7         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String8         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String9         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr);

   --------------------------------------------------------------------
   --
   --	Tcl dynamic strings
   --
   --------------------------------------------------------------------

   type DString_Rec     is private;
   type DString_Ptr is access all DString_Rec;
   pragma Convention (C, DString_Ptr);

   function DStringLength (dsPtr : in DString_Ptr) return C.Int;
   pragma Inline (DStringLength);
   -- Number of non-null characters in the string.

   function DStringValue (dsPtr : in DString_Ptr) return C.Strings.Chars_Ptr;
   pragma Inline (DStringValue);
   -- Returns the string value.

   --------------------------------------------------------------------
   --
   --	Miscellaneous constant declarations
   --
   --------------------------------------------------------------------

   -- Definitions for the maximum number of digits of precision that may
   -- be specified in the "tcl_precision" variable, and the number of
   -- characters of buffer space required by Tcl.PrintDouble.

   MAX_PREC             : constant := 17;
   DOUBLE_SPACE         : constant := MAX_PREC + 10;

   -- Flag that may be passed to Tcl.ConvertElement to force it not to
   -- output braces (careful!  if you change this flag be sure to change
   -- the definitions at the front of tclUtil.c).

   DONT_USE_BRACES	: constant := 1;

   -- Flag values passed to Tcl.RecordAndEval.
   -- WARNING: these bit choices must not conflict with the bit choices
   -- for evalFlag bits in tclInt.h!!

   NO_EVAL		: constant := 16#10000#;
   EVAL_GLOBAL		: constant := 16#20000#;

   -- Special freeProc values that may be passed to Tcl.SetResult (see
   -- the man page for details):

   VOLATILE	        : constant := 1;
   STATIC	        : constant := 0;
   DYNAMIC	        : constant := 3;

   -- Flag values passed to variable-related procedures.

   GLOBAL_ONLY		: constant := 1;
   APPEND_VALUE	        : constant := 2;
   LIST_ELEMENT	        : constant := 4;
   TRACE_READS		: constant := 16#10#;
   TRACE_WRITES	        : constant := 16#20#;
   TRACE_UNSETS	        : constant := 16#40#;
   TRACE_DESTROYED	: constant := 16#80#;
   INTERP_DESTROYED	: constant := 16#100#;
   LEAVE_ERR_MSG	: constant := 16#200#;

   -- Types for linked variables:

   LINK_INT		: constant := 1;
   LINK_DOUBLE		: constant := 2;
   LINK_BOOLEAN	        : constant := 3;
   LINK_STRING		: constant := 4;
   LINK_READ_ONLY	: constant := 16#80#;

   -- Permission flags for files:

   FILE_READABLE	: constant := 1;
   FILE_WRITABLE	: constant := 2;

   STRING_KEYS		: constant := 0;
   ONE_WORD_KEYS	: constant := 1;

   --------------------------------------------------------------------
   --
   --	Generics for all subprograms which require Client Data
   --
   --------------------------------------------------------------------

   generic
      type ClientData_Ptr is private;
   package Generic_AsyncCommands is

      type AsyncProc_Ptr is access function (
	 ClientData      : in ClientData_Ptr;
	 Interp          : in Interp_Ptr;
	 Code            : in C.Int) return C.Int;
      pragma Convention (C, AsyncProc_Ptr);

      function AsyncCreate (
	 proc            : in AsyncProc_Ptr;
	 clientData      : in ClientData_Ptr) return AsyncHandler_Ptr;
      pragma Import (C, AsyncCreate, "Tcl_AsyncCreate");

   end Generic_AsyncCommands;

   generic
      type ClientData_Ptr is private;
   package Generic_CreateCommands is

      type CmdDeleteProc_Ptr is access procedure (
	 ClientData      : in out ClientData_Ptr);
      pragma Convention (C, CmdDeleteProc_Ptr);

      type CmdProc_Ptr is access function (
	 ClientData      : in ClientData_Ptr;
	 Interp          : in Interp_Ptr;
	 Argc	         : in C.Int;
	 Argv	         : in C_Aux.Chars_Ptr_Ptr) return C.Int;
      pragma Convention (C, CmdProc_Ptr);

      type CmdTraceProc_Ptr is access procedure (
	 ClientData      : in ClientData_Ptr;
	 Interp          : in Interp_Ptr;
	 Level           : in C.Int;
	 Command         : in C.Strings.Chars_Ptr;
	 Proc            : in CmdProc_Ptr;
	 cmdClientData   : in ClientData_Ptr;
	 Argc	         : in C.Int;
	 Argv	         : in C_Aux.Chars_Ptr_Ptr);
      pragma Convention (C, CmdTraceProc_Ptr);

      type CmdInfo_Rec is
	 record
	    proc       : CmdProc_Ptr;	-- Procedure that implements command.
	    ClientData : ClientData_Ptr;-- ClientData passed to proc.
	    deleteProc : CmdDeleteProc_Ptr; -- Procedure to call when command
					-- is deleted.
	    deleteData : ClientData_Ptr;-- Value to pass to deleteProc (usually
					-- the same as ClientData).
	 end record;
      pragma Convention (C, CmdInfo_Rec);

      type CmdInfo_Ptr is access all CmdInfo_Rec;
      pragma Convention (C, CmdInfo_Ptr);

      function CreateCommand (
	 interp          : in Interp_Ptr;
	 cmdName         : in C.Strings.Chars_Ptr;
	 proc            : in CmdProc_Ptr;
	 clientData      : in ClientData_Ptr;
	 deleteProc      : in CmdDeleteProc_Ptr) return Command_Ptr;
      pragma Import (C, CreateCommand, "Tcl_CreateCommand");

      function CreateCommand (
	 interp          : in Interp_Ptr;
	 CmdName         : in String;
	 Proc            : in CmdProc_Ptr;
	 ClientData      : in ClientData_Ptr;
	 DeleteProc      : in CmdDeleteProc_Ptr) return Command_Ptr;

      procedure CreateCommand (
	 interp          : in Interp_Ptr;
	 CmdName         : in String;
	 Proc            : in CmdProc_Ptr;
	 ClientData      : in ClientData_Ptr;
	 DeleteProc      : in CmdDeleteProc_Ptr);

      function CreateTrace (
	 interp          : in Interp_Ptr;
	 level           : in C.Int;
	 proc            : in CmdTraceProc_Ptr;
	 clientData      : in ClientData_Ptr) return Trace_Ptr;
      pragma Import (C, CreateTrace, "Tcl_CreateTrace");

      function GetCommandInfo (
	 interp          : in Interp_Ptr;
	 cmdName         : in C.Strings.Chars_Ptr;
	 infoPtr         : in CmdInfo_Ptr) return C.Int;
      pragma Import (C, GetCommandInfo, "Tcl_GetCommandInfo");

      function SetCommandInfo (
	 interp          : in Interp_Ptr;
	 cmdName         : in C.Strings.Chars_Ptr;
	 infoPtr         : in CmdInfo_Ptr) return C.Int;
      pragma Import (C, SetCommandInfo, "Tcl_SetCommandInfo");

   end Generic_CreateCommands;

   generic
      type ClientData_Ptr is private;
   package Generic_CallWhenDeleted is

      type InterpDeleteProc_Ptr is access procedure (
	 ClientData      : in ClientData_Ptr;
	 Interp          : in Interp_Ptr);
      pragma Convention (C, InterpDeleteProc_Ptr);

      procedure CallWhenDeleted (
	 interp          : in Interp_Ptr;
	 proc            : in InterpDeleteProc_Ptr;
	 clientData      : in ClientData_Ptr);
      pragma Import (C, CallWhenDeleted, "Tcl_CallWhenDeleted");

      procedure DontCallWhenDeleted (
	 interp          : in Interp_Ptr;
	 proc            : in InterpDeleteProc_Ptr;
	 clientData      : in ClientData_Ptr);
      pragma Import (C, DontCallWhenDeleted, "Tcl_DontCallWhenDeleted");

      function GetAssocData (
	 interp          : in Interp_Ptr;
	 name            : in C.Strings.Chars_Ptr;
	 procPtr         : in InterpDeleteProc_Ptr) return ClientData_Ptr;
      pragma Import (C, GetAssocData, "Tcl_GetAssocData");

      procedure SetAssocData (
	 interp          : in Interp_Ptr;
	 name            : in C.Strings.Chars_Ptr;
	 proc            : in InterpDeleteProc_Ptr;
	 clientData      : in ClientData_Ptr);
      pragma Import (C, SetAssocData, "Tcl_SetAssocData");

   end Generic_CallWhenDeleted;

   generic
      type ClientData_Ptr is private;
   package Generic_MathFunc is

      type MathProc_Ptr is access function (
	 ClientData      : in ClientData_Ptr;
	 Interp          : in Interp_Ptr;
	 Args	         : in Value_Ptr;
	 resultPtr       : in Value_Ptr) return C.Int;
      pragma Convention (C, MathProc_Ptr);

      procedure CreateMathFunc (
	 interp          : in Interp_Ptr;
	 name            : in C.Strings.Chars_Ptr;
	 numArgs         : in C.Int;
	 argTypes        : in ValueType_Ptr;
	 proc            : in MathProc_Ptr;
	 clientData      : in ClientData_Ptr);
      pragma Import (C, CreateMathFunc, "Tcl_CreateMathFunc");

   end Generic_MathFunc;

   generic
      type ClientData_Ptr is private;
   package Generic_Hash is

      type HashEntry_Rec  is private;
      type HashEntry_Ptr  is access all HashEntry_Rec;
      type HashTable_Rec  is private;
      type HashTable_Ptr  is access all HashTable_Rec;
      type HashSearch_Rec is private;
      type HashSearch_Ptr is access all HashSearch_Rec;

      function CreateHashEntry (
	 tablePtr : in HashTable_Ptr;
	 Key      : in C.Strings.Chars_Ptr;
	 NewPtr   : in C_Aux.Int_Ptr) return HashEntry_Ptr;
      pragma Inline (CreateHashEntry);

      procedure DeleteHashEntry (
	 entryPtr : in HashEntry_Ptr);
      pragma Import (C, DeleteHashEntry, "Tcl_DeleteHashEntry");

      procedure DeleteHashTable (
	 tablePtr : in HashTable_Ptr);
      pragma Import (C, DeleteHashTable, "Tcl_DeleteHashTable");

      function GetHashValue (
	 EntryPtr : in HashEntry_Ptr) return ClientData_Ptr;
      pragma Inline (GetHashValue);

      function GetHashKey (
	 TablePtr : in HashTable_Ptr;
	 EntryPtr : in HashEntry_Ptr) return C.Strings.Chars_Ptr;
      pragma Inline (GetHashKey);

      function FindHashEntry (
	 TablePtr : in HashTable_Ptr;
	 Key      : in C.Strings.Chars_Ptr) return HashEntry_Ptr;
      pragma Inline (FindHashEntry);

      function FirstHashEntry (
	 tablePtr : in HashTable_Ptr;
	 searchPtr: in HashSearch_Ptr) return HashEntry_Ptr ;
      pragma Import (C, FirstHashEntry, "Tcl_FirstHashEntry");

      function HashStats (
	 tablePtr : in HashTable_Ptr) return C.Strings.Chars_Ptr ;
      pragma Import (C, HashStats, "Tcl_HashStats");

      procedure InitHashTable (
	 tablePtr        : in HashTable_Ptr;
	 keyType         : in C.Int);
      pragma Import (C, InitHashTable, "Tcl_InitHashTable");

      function NextHashEntry (
	 searchPtr       : in HashSearch_Ptr) return HashEntry_Ptr ;
      pragma Import (C, NextHashEntry, "Tcl_NextHashEntry");

      procedure SetHashValue (
	 EntryPtr : in HashEntry_Ptr;
	 Value    : in ClientData_Ptr);
      pragma Inline (SetHashValue);

   private

      type HashEntry_access is access all HashEntry_Ptr;

      type HashEntry_Rec is
	 record
	    NextPtr   : HashEntry_Ptr;	 -- Pointer to next entry in this
					 -- hash bucket, or NULL for end of
					 -- chain.
	    TablePtr  : HashTable_Ptr;	 -- Pointer to table containing entry.
	    BucketPtr : HashEntry_access;-- Pointer to bucket that points to
					 -- first entry in this entry's chain:
					 -- used for deleting the entry.
	    ClientData: ClientData_Ptr;	 -- Application stores something here
					 -- with Tcl_SetHashValue.
	    strng     : aliased C.Char_Array(0..3);
					 -- String for key.  The actual size
					 -- will be as large as needed to hold
					 -- the key.
	 end record;
      pragma Convention (C, HashEntry_Rec);

      type HashEntry_Array is array (C.Int range <>) of HashEntry_Ptr;

      TCL_SMALL_HASH_TABLE : constant := 4;

      type HashEntryFindProc is access function (
	 tablePtr : in HashTable_Ptr;
	 Key      : in C.Strings.Chars_Ptr) return HashEntry_Ptr;

      type HashEntryCreateProc is access function (
	 tablePtr : in HashTable_Ptr;
	 Key      : in C.Strings.Chars_Ptr;
	 NewPtr   : in C_Aux.Int_Ptr) return HashEntry_Ptr;

      type HashTable_Rec is
	 record
	    Buckets : HashEntry_access;	-- Pointer to bucket array.  Each
					-- element points to first entry in
					-- bucket's hash chain, or NULL.
	    StaticBuckets : HashEntry_Array (0..TCL_SMALL_HASH_TABLE-1);
					-- Bucket array used for small tables
					-- (to avoid mallocs and frees).
	    NumBuckets  : C.Int;	-- Total number of buckets allocated
					-- at **bucketPtr.
	    NumEntries  : C.Int;	-- Total number of entries present
					-- in table.
	    RebuildSize : C.Int;	-- Enlarge table when numEntries gets
					-- to be this large.
	    DownShift   : C.Int;	-- Shift count used in hashing
					-- function.  Designed to use high-
					-- order bits of randomized keys.
	    Mask        : C.Int;	-- Mask value used in hashing
					-- function.
	    KeyType     : C.Int;	-- Type of keys used in this table. 
					-- It's either TCL_STRING_KEYS,
					-- TCL_ONE_WORD_KEYS, or an integer
					-- giving the number of ints in a
	    FindProc    : HashEntryFindProc;
	    CreateProc  : HashEntryCreateProc;
	 end record;
      pragma Convention (C, HashTable_Rec);

      type HashSearch_Rec is
	 record
	    TablePtr     : HashTable_Ptr;-- Table being searched.
	    NextIndex    : C.Int;	 -- Index of next bucket to be
				         -- enumerated after present one.
	    NextEntryPtr : HashEntry_Ptr;-- Next entry to be enumerated in the
	 end record;		         -- the current bucket.
      pragma Convention (C, HashSearch_Rec);

   end Generic_Hash;

   package Integer_Hash is new Generic_Hash (Integer);

   generic
      type ClientData_Ptr is private;
   package Generic_TraceCommands is

      type VarTraceProc_Ptr is access function (
	 ClientData    : in ClientData_Ptr;
	 Interp        : in Interp_Ptr;
	 Part1         : in C.Strings.Chars_Ptr;
	 Part2         : in C.Strings.Chars_Ptr;
	 Flags         : in C.Int) return C.Strings.Chars_Ptr;
      pragma Convention (C, VarTraceProc_Ptr);

      function TraceVar (
	 interp          : in Interp_Ptr;
	 varName         : in C.Strings.Chars_Ptr;
	 flags           : in C.Int;
	 proc            : in VarTraceProc_Ptr;
	 clientData      : in ClientData_Ptr) return C.Int;
      pragma Import (C, TraceVar, "Tcl_TraceVar");

      function TraceVar2 (
	 interp          : in Interp_Ptr;
	 part1           : in C.Strings.Chars_Ptr;
	 part2           : in C.Strings.Chars_Ptr;
	 flags           : in C.Int;
	 proc            : in VarTraceProc_Ptr;
	 clientData      : in ClientData_Ptr) return C.Int;
      pragma Import (C, TraceVar2, "Tcl_TraceVar2");

      procedure UntraceVar (
	 interp          : in Interp_Ptr;
	 varName         : in C.Strings.Chars_Ptr;
	 flags           : in C.Int;
	 proc            : in VarTraceProc_Ptr;
	 clientData      : in ClientData_Ptr);
      pragma Import (C, UntraceVar, "Tcl_UntraceVar");

      procedure UntraceVar2 (
	 interp          : in Interp_Ptr;
	 part1           : in C.Strings.Chars_Ptr;
	 part2           : in C.Strings.Chars_Ptr;
	 flags           : in C.Int;
	 proc            : in VarTraceProc_Ptr;
	 clientData      : in ClientData_Ptr);
      pragma Import (C, UntraceVar2, "Tcl_UntraceVar2");

      function VarTraceInfo (
	 interp          : in Interp_Ptr;
	 varName         : in C.Strings.Chars_Ptr;
	 flags           : in C.Int;
	 procPtr         : in VarTraceProc_Ptr;
	 prevClientData  : in ClientData_Ptr) return ClientData_Ptr;
      pragma Import (C, VarTraceInfo, "Tcl_VarTraceInfo");

      function VarTraceInfo2 (
	 interp          : in Interp_Ptr;
	 part1           : in C.Strings.Chars_Ptr;
	 part2           : in C.Strings.Chars_Ptr;
	 flags           : in C.Int;
	 procPtr         : in VarTraceProc_Ptr;
	 prevClientData  : in ClientData_Ptr) return ClientData_Ptr;
      pragma Import (C, VarTraceInfo2, "Tcl_VarTraceInfo2");

   end Generic_TraceCommands;

   --------------------------------------------------------------------
   --
   --	Exported Tcl procedures (alphabetic order except for those
   --   in the generics above)
   --
   --------------------------------------------------------------------

   Tcl_Error : exception;

   procedure AddErrorInfo (
      interp          : in Interp_Ptr;
      message         : in C.Strings.Chars_Ptr);
   pragma Import (C, AddErrorInfo, "Tcl_AddErrorInfo");

   procedure AddErrorInfo (
      interp  : in Interp_Ptr;
      message : in String);

   procedure AllowExceptions (
      interp          : in Interp_Ptr);
   pragma Import (C, AllowExceptions, "Tcl_AllowExceptions");

   procedure AppendElement (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr);
   pragma Import (C, AppendElement, "Tcl_AppendElement");

   procedure AppendElement (
      interp : in Interp_Ptr;
      strng  : in String);

   procedure AppendResult (
      interp          : in Interp_Ptr;
      String1         : in C.Strings.Chars_Ptr;
      String2         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String3         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String4         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String5         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String6         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String7         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String8         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String9         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr);
   procedure AppendResult (
      interp          : in Interp_Ptr;
      String1         : in String;
      String2         : in String := "";
      String3         : in String := "";
      String4         : in String := "";
      String5         : in String := "";
      String6         : in String := "";
      String7         : in String := "";
      String8         : in String := "";
      String9         : in String := "");
   -- Appends each string to the current result string in interp.  The C
   -- version of this procedure takes variable number of arguments.  This
   -- is simulated here by providing 9 string arguments.  This method
   -- is not quite as flexible as C's indefinite number of arguments,
   -- but we believe that in practice, this will be sufficient.  You can
   -- always call AppendResult more than once.  The body of this procedure
   -- adds a Null argument before calling the C function, so you may use
   -- all 9 arguments.

   procedure AsyncDelete (
      async           : in AsyncHandler_Ptr);
   pragma Import (C, AsyncDelete, "Tcl_AsyncDelete");

   function AsyncInvoke (
      interp          : in Interp_Ptr;
      code            : in C.Int) return C.Int;
   pragma Import (C, AsyncInvoke, "Tcl_AsyncInvoke");

   procedure AsyncInvoke (
      interp : in Interp_Ptr;
      code   : in C.Int);

   procedure AsyncMark (
      async           : in AsyncHandler_Ptr);
   pragma Import (C, AsyncMark, "Tcl_AsyncMark");

   function AsyncReady return C.Int;
   pragma Import (C, AsyncReady, "Tcl_AsyncReady");

   procedure AsyncReady;

   function Backslash (
      src             : in C.Strings.Chars_Ptr;
      readPtr         : in C_Aux.Int_Ptr) return C.Char;
   pragma Import (C, Backslash, "Tcl_Backslash");

   function Ckalloc (
      size            : in C.Unsigned) return C_Aux.Void_Ptr ;
   pragma Import (C, Ckalloc, "Tcl_Ckalloc");

   procedure Ckfree (
      ptr             : in C.Strings.Chars_Ptr);
   pragma Import (C, Ckfree, "Tcl_Ckfree");

   function CommandComplete (
      cmd             : in C.Strings.Chars_Ptr) return C.Int;
   pragma Import (C, CommandComplete, "Tcl_CommandComplete");

   function CommandComplete (
      Cmd : in C.Strings.Chars_Ptr) return Boolean;
   function CommandComplete (
      Cmd : in String) return Boolean;
   -- Returns True if Cmd holds one or more complete commands, False if
   -- the last command in Cmd is incomplete due to open braces, etc.

   function Concat (
      argc            : in C.Int;
      argv            : in C_Aux.Chars_Ptr_Ptr) return C.Strings.Chars_Ptr ;
   pragma Import (C, Concat, "Tcl_Concat");

   function ConvertElement (
      src             : in C.Strings.Chars_Ptr;
      dst             : in C.Strings.Chars_Ptr;
      flags           : in C.Int) return C.Int;
   pragma Import (C, ConvertElement, "Tcl_ConvertElement");

   function CreateAlias (
      slave           : in Interp_Ptr;
      slaveCmd        : in C.Strings.Chars_Ptr;
      target          : in Interp_Ptr;
      targetCmd       : in C.Strings.Chars_Ptr;
      argc            : in C.Int;
      argv            : in C_Aux.Chars_Ptr_Ptr) return C.Int;
   pragma Import (C, CreateAlias, "Tcl_CreateAlias");

   function CreateInterp return Interp_Ptr ;
   pragma Import (C, CreateInterp, "Tcl_CreateInterp");

   function CreatePipeline (
      interp          : in Interp_Ptr;
      argc            : in C.Int;
      argv            : in C_Aux.Chars_Ptr_Ptr;
      pidArrayPtr     : in C_Aux.Int_Ptr_Ptr;
      inPipePtr       : in C_Aux.Int_Ptr;
      outPipePtr      : in C_Aux.Int_Ptr;
      errFilePtr      : in C_Aux.Int_Ptr) return C.Int;
   pragma Import (C, CreatePipeline, "Tcl_CreatePipeline");

   procedure CreatePipeline (
      interp      : in     Interp_Ptr;
      Argc        : in     C.Int; 
      Argv        : in     C_Aux.Chars_Ptr_Ptr;
      NumPids     :    out C.Int;
      PidArrayPtr :    out C_Aux.Int_Ptr_Ptr;
      InPipePtr   :    out C.Int;
      OutPipePtr  :    out C.Int;
      ErrFilePtr  :    out C.Int);
   -- Creates a process pipeline, returns a count of the number
   -- of processes created in NumPids, and stores at PidArrayPtr the
   -- address of a malloc-ed array of process identifiers.
   -- If an error occurs, raises Tcl_Error.  Then, the error
   -- string can be obtained by calling Ada.Exceptions.Exception_Message
   -- or Result.  InPipePtr, OutPipePtr, and ErrFilePtr are used to
   -- control default I/O redirection.

   function CreateSlave (
      interp          : in Interp_Ptr;
      slaveName       : in C.Strings.Chars_Ptr;
      isSafe          : in C.Int;
      withTk          : in C.Int) return Interp_Ptr ;
   pragma Import (C, CreateSlave, "Tcl_CreateSlave");

   function DbCkalloc (
      size            : in C.Unsigned;
      file            : in C.Strings.Chars_Ptr;
      line            : in C.Int) return C.Strings.Chars_Ptr ;
   pragma Import (C, DbCkalloc, "Tcl_DbCkalloc");

   function DbCkfree (
      ptr             : in C.Strings.Chars_Ptr;
      file            : in C.Strings.Chars_Ptr;
      line            : in C.Int) return C.Int;
   pragma Import (C, DbCkfree, "Tcl_DbCkfree");

   function DbCkrealloc (
      ptr             : in C.Strings.Chars_Ptr;
      size            : in C.Unsigned;
      file            : in C.Strings.Chars_Ptr;
      line            : in C.Int) return C.Strings.Chars_Ptr ;
   pragma Import (C, DbCkrealloc, "Tcl_DbCkrealloc");

   procedure DeleteAssocData (
      interp          : in Interp_Ptr;
      name            : in C.Strings.Chars_Ptr);
   pragma Import (C, DeleteAssocData, "Tcl_DeleteAssocData");

   function DeleteCommand (
      interp          : in Interp_Ptr;
      cmdName         : in C.Strings.Chars_Ptr) return C.Int;
   pragma Import (C, DeleteCommand, "Tcl_DeleteCommand");

   procedure DeleteCommand (
      interp          : in Interp_Ptr;
      cmdName         : in C.Strings.Chars_Ptr);
   procedure DeleteCommand (
      interp          : in Interp_Ptr;
      cmdName         : in String);
   -- If CmdName is a command or procedure in Interp, then it is deleted,
   -- otherwise exception Tcl_Error is raised.

   procedure DeleteInterp (
      interp          : in Interp_Ptr);
   pragma Import (C, DeleteInterp, "Tcl_DeleteInterp");

   procedure DeleteTrace (
      interp          : in Interp_Ptr;
      trace           : in Trace_Ptr);
   pragma Import (C, DeleteTrace, "Tcl_DeleteTrace");

   procedure DetachPids (
      numPids         : in C.Int;
      pidPtr          : in C_Aux.Int_Ptr);
   pragma Import (C, DetachPids, "Tcl_DetachPids");

   function DStringAppend (
      dsPtr           : in DString_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      length          : in C.Int) return C.Strings.Chars_Ptr ;
   pragma Import (C, DStringAppend, "Tcl_DStringAppend");

   function DStringAppend (
      DsPtr  : in DString_Ptr;
      strng  : in String;
      Length : in C.Int := -1) return C.Strings.Chars_Ptr;
   -- Appends Length characters from strng to DsPtr's value and returns
   -- the new value of DsPtr.  If Length is less than zero, appends
   -- all of strng up to terminating Nul character (where applicable).

   procedure DStringAppend (
      DsPtr  : in DString_Ptr;
      strng  : in C.Strings.Chars_Ptr;
      Length : in C.Int := -1);
   procedure DStringAppend (
      DsPtr  : in DString_Ptr;
      strng  : in String;
      Length : in C.Int := -1);
   -- Appends Length characters from strng to DsPtr's value.  If Length
   -- is less than zero, appends all of strng up to terminating Nul
   -- character (where applicable).

   function DStringAppendElement (
      dsPtr           : in DString_Ptr;
      strng           : in C.Strings.Chars_Ptr) return C.Strings.Chars_Ptr ;
   pragma Import (C, DStringAppendElement, "Tcl_DStringAppendElement");

   function DStringAppendElement (
      DsPtr  : in DString_Ptr;
      strng  : in String) return C.Strings.Chars_Ptr;
   -- Converts strng to proper list element and appends to DsPtr's value
   -- (with separator space if needed).  Returns new value of DsPtr.

   procedure DStringAppendElement (
      DsPtr  : in DString_Ptr;
      strng  : in C.Strings.Chars_Ptr);
   procedure DStringAppendElement (
      DsPtr  : in DString_Ptr;
      strng  : in String);
   -- Converts strng to proper list element and appends to DsPtr's value
   -- (with separator space if needed).

   procedure DStringEndSublist (
      dsPtr           : in DString_Ptr);
   pragma Import (C, DStringEndSublist, "Tcl_DStringEndSublist");

   procedure DStringFree (
      dsPtr           : in DString_Ptr);
   pragma Import (C, DStringFree, "Tcl_DStringFree");

   procedure DStringGetResult (
      interp          : in Interp_Ptr;
      dsPtr           : in DString_Ptr);
   pragma Import (C, DStringGetResult, "Tcl_DStringGetResult");

   procedure DStringInit (
      dsPtr           : in DString_Ptr);
   pragma Import (C, DStringInit, "Tcl_DStringInit");

   procedure DStringResult (
      interp          : in Interp_Ptr;
      dsPtr           : in DString_Ptr);
   pragma Import (C, DStringResult, "Tcl_DStringResult");

   procedure DStringSetLength (
      dsPtr           : in DString_Ptr;
      length          : in C.Int);
   pragma Import (C, DStringSetLength, "Tcl_DStringSetLength");

   procedure DStringTrunc (
      dsPtr           : in DString_Ptr;
      length          : in C.Int) renames DStringSetLength;

   procedure DStringStartSublist (
      dsPtr           : in DString_Ptr);
   pragma Import (C, DStringStartSublist, "Tcl_DStringStartSublist");

   procedure EnterFile (
      interp          : in Interp_Ptr;
      file            : in C_Aux.File_Ptr;
      permissions     : in C.Int);
   pragma Import (C, EnterFile, "Tcl_EnterFile");

   function ErrnoId return C.Strings.Chars_Ptr;
   pragma Import (C, ErrnoId, "Tcl_ErrnoId");

   function ErrnoId return String;
   procedure ErrnoId (
      strng : out String;
      last  : out Natural);
   -- Returns a symbolic name corresponding to the current value of
   -- errno, such as ENOENT.

   function ErrnoMsg (
      err             : in C.Int) return C.Strings.Chars_Ptr;
   pragma Import (C, ErrnoMsg, "Tcl_ErrnoMsg");

   function Eval (
      interp          : in Interp_Ptr;
      cmd             : in C.Strings.Chars_Ptr) return C.Int;
   pragma Import (C, Eval, "Tcl_Eval");

   function Eval (
      interp : in Interp_Ptr; 
      Script : in C.Strings.Chars_Ptr) return C.Strings.Chars_Ptr;
   function Eval (
      interp : in Interp_Ptr; 
      Script : in String) return C.Strings.Chars_Ptr;
   procedure Eval (
      interp : in Interp_Ptr; 
      Script : in C.Strings.Chars_Ptr);
   procedure Eval (
      interp : in Interp_Ptr; 
      Script : in String);
   -- Evaluate Script in Interp.
   -- The functions return the result of the evaluation.  Call Result to get
   -- the result of the procedure call.  In either case, the exception
   -- Tcl_Error is raised if evaluation is not successful.  Then, the error
   -- string can be obtained by calling Ada.Exceptions.Exception_Message
   -- or Result.

   function EvalFile (
      interp          : in Interp_Ptr;
      fileName        : in C.Strings.Chars_Ptr) return C.Int;
   pragma Import (C, EvalFile, "Tcl_EvalFile");

   function EvalFile (
      interp   : in Interp_Ptr; 
      FileName : in C.Strings.Chars_Ptr) return C.Strings.Chars_Ptr;
   function EvalFile (
      interp   : in Interp_Ptr; 
      FileName : in String) return C.Strings.Chars_Ptr;
   procedure EvalFile (
      interp   : in Interp_Ptr; 
      FileName : in C.Strings.Chars_Ptr);
   procedure EvalFile (
      interp   : in Interp_Ptr; 
      FileName : in String);
   -- Evaluate the contents of file Filename in Interp.
   -- The functions return the result of the evaluation.  Call Result to get
   -- the result of the procedure call.  In either case, the exception Tcl_Error
   -- is raised if evaluation is not successful.  Then, the error string can be
   -- obtained by calling Ada.Exceptions.Exception_Message or Result.

   function ExprBoolean (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      ptr             : in C_Aux.Int_Ptr) return C.Int;
   pragma Import (C, ExprBoolean, "Tcl_ExprBoolean");

   generic
      type Enum is (<>);
   package Generic_ExprEnum is

      function ExprEnum (
	 interp : in Interp_Ptr;
	 strng  : in C.Strings.Chars_Ptr) return Enum;

      function ExprEnum (
	 interp : in Interp_Ptr;
	 strng  : in String) return Enum;

   end Generic_ExprEnum;

   package ExprBool is new Generic_ExprEnum (Boolean);

   function ExprBoolean (
      interp : in Interp_Ptr;
      strng  : in C.Strings.Chars_Ptr) return Boolean
         renames ExprBool.ExprEnum;

   function ExprBoolean  (
      interp : in Interp_Ptr;
      strng  : in String) return Boolean
         renames ExprBool.ExprEnum;

   function ExprDouble (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      ptr             : in C_Aux.Double_Ptr) return C.Int;
   pragma Import (C, ExprDouble, "Tcl_ExprDouble");

   generic
      type Num is digits <>;
   package Generic_ExprFloat is

      function ExprFloat (
	 interp : in Interp_Ptr;
	 strng  : in C.Strings.Chars_Ptr) return Num;

      function ExprFloat (
	 interp : in Interp_Ptr;
	 strng  : in String) return Num;

   end Generic_ExprFloat;

   package ExprFlt is new Generic_ExprFloat (Float);

   function ExprDouble (
      interp : in Interp_Ptr;
      strng  : in C.Strings.Chars_Ptr) return Float
         renames ExprFlt.ExprFloat;

   function ExprDouble (
      interp : in Interp_Ptr;
      strng  : in String) return Float
         renames ExprFlt.ExprFloat;

   generic
      type Num is delta <>;
   package Generic_ExprFixed is

      function ExprFixed (
	 interp : in Interp_Ptr;
	 strng  : in C.Strings.Chars_Ptr) return Num;

      function ExprFixed (
	 interp : in Interp_Ptr;
	 strng  : in String) return Num;

   end Generic_ExprFixed;

   function ExprLong (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      ptr             : in C_Aux.Long_Ptr) return C.Int;
   pragma Import (C, ExprLong, "Tcl_ExprLong");

   generic
      type Num is range <>;
   package Generic_ExprInteger is

      function ExprInteger (
	 interp : in Interp_Ptr;
	 strng  : in C.Strings.Chars_Ptr) return Num;

      function ExprInteger (
	 interp : in Interp_Ptr;
	 strng  : in String) return Num;

   end Generic_ExprInteger;

   package ExprInteger is new Generic_ExprInteger (Integer);

   function ExprInt (
      interp : in Interp_Ptr;
      strng  : in C.Strings.Chars_Ptr) return Integer
         renames ExprInteger.ExprInteger;

   function ExprInt  (
      interp : in Interp_Ptr;
      strng  : in String) return Integer
         renames ExprInteger.ExprInteger;

   generic
      type Num is mod <>;
   package Generic_ExprModulus is

      function ExprMod (
	 interp : in Interp_Ptr;
	 strng  : in C.Strings.Chars_Ptr) return Num;

      function ExprMod (
	 interp : in Interp_Ptr;
	 strng  : in String) return Num;

   end Generic_ExprModulus;

   function ExprString (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr) return C.Int;
   pragma Import (C, ExprString, "Tcl_ExprString");

   function ExprString (
      interp : in Interp_Ptr;
      strng  : in C.Strings.Chars_Ptr) return C.Strings.Chars_Ptr;
   function ExprString (
      interp : in Interp_Ptr;
      strng  : in String) return C.Strings.Chars_Ptr;
   procedure ExprString (
      interp : in Interp_Ptr;
      strng  : in C.Strings.Chars_Ptr);
   procedure ExprString (
      interp : in Interp_Ptr;
      strng  : in String);
   -- Evaluates the string strng as an expression.
   -- The functions return the result of the evaluation, whereas, you
   -- must call Result to get the result of the procedure call.
   -- In either case, the exception Tcl_Error is raised if evaluation
   -- is not successful.  Then, the error string can be obtained by
   -- calling Ada.Exceptions.Exception_Message or Result.

   function FilePermissions (
      interp          : in Interp_Ptr;
      file            : in C_Aux.File_Ptr) return C.Int;
   pragma Import (C, FilePermissions, "Tcl_FilePermissions");

   procedure FindExecutable (
      argv0           : in C.Strings.Chars_Ptr);
   pragma Import (C, FindExecutable, "Tcl_FindExecutable");

   function GetAlias (
      interp          : in Interp_Ptr;
      slaveCmd        : in C.Strings.Chars_Ptr;
      targetInterpPtr : in Interp_Ptr_Ptr;
      targetCmdPtr    : in C_Aux.Chars_Ptr_Ptr;
      argcPtr         : in C_Aux.Int_Ptr;
      argvPtr         : in C_Aux.Chars_Ptr_Ptr_Ptr) return C.Int;
   pragma Import (C, GetAlias, "Tcl_GetAlias");

   function GetBoolean (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      boolPtr         : in C_Aux.Int_Ptr) return C.Int;
   pragma Import (C, GetBoolean, "Tcl_GetBoolean");

   function GetCommandName (
      interp          : in Interp_Ptr;
      command         : in Command_Ptr) return C.Strings.Chars_Ptr ;
   pragma Import (C, GetCommandName, "Tcl_GetCommandName");

   function GetDouble (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      doublePtr       : in C_Aux.Double_Ptr) return C.Int;
   pragma Import (C, GetDouble, "Tcl_GetDouble");

   function GetInt (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      intPtr          : in C_Aux.Int_Ptr) return C.Int;
   pragma Import (C, GetInt, "Tcl_GetInt");

   function GetMaster (
      interp          : in Interp_Ptr) return Interp_Ptr ;
   pragma Import (C, GetMaster, "Tcl_GetMaster");

   function GetOpenFile (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      write           : in C.Int;
      checkUsage      : in C.Int;
      filePtr         : in C_Aux.File_Ptr_Ptr) return C.Int;
   pragma Import (C, GetOpenFile, "Tcl_GetOpenFile");

   function GetSlave (
      interp          : in Interp_Ptr;
      slaveName       : in C.Strings.Chars_Ptr) return Interp_Ptr ;
   pragma Import (C, GetSlave, "Tcl_GetSlave");

   function GetVar (
      interp          : in Interp_Ptr;
      varName         : in C.Strings.Chars_Ptr;
      flags           : in C.Int) return C.Strings.Chars_Ptr ;
   pragma Import (C, GetVar, "Tcl_GetVar");

   function GetVar (
      interp   : in Interp_Ptr;
      VarName  : in String;
      Flags    : in C.Int := 0) return C.Strings.Chars_Ptr;
   -- Returns the value of the named variable, or the Null pointer if
   -- an error occurs.

   function GetVar2 (
      interp          : in Interp_Ptr;
      part1           : in C.Strings.Chars_Ptr;
      part2           : in C.Strings.Chars_Ptr;
      flags           : in C.Int) return C.Strings.Chars_Ptr ;
   pragma Import (C, GetVar2, "Tcl_GetVar2");

   function GetVar2 (
      interp : in Interp_Ptr;
      Name1  : in String; 
      Name2  : in C.Strings.Chars_Ptr; 
      Flags  : in C.Int := 0) return C.Strings.Chars_Ptr;
   function GetVar2 (
      interp : in Interp_Ptr;
      Name1  : in C.Strings.Chars_Ptr; 
      Name2  : in String; 
      Flags  : in C.Int := 0) return C.Strings.Chars_Ptr;
   function GetVar2 (
      interp : in Interp_Ptr;
      Name1  : in String; 
      Name2  : in String; 
      Flags  : in C.Int := 0) return C.Strings.Chars_Ptr;
   -- Returns the value of the named variable, or the Null pointer if
   -- an error occurs.

   function GlobalEval (
      interp          : in Interp_Ptr;
      command         : in C.Strings.Chars_Ptr) return C.Int;
   pragma Import (C, GlobalEval, "Tcl_GlobalEval");

   function GlobalEval (
      interp   : in Interp_Ptr; 
      Script   : in C.Strings.Chars_Ptr) return C.Strings.Chars_Ptr;
   function GlobalEval (
      interp   : in Interp_Ptr; 
      Script   : in String) return C.Strings.Chars_Ptr;
   procedure GlobalEval (
      interp   : in Interp_Ptr; 
      Script   : in C.Strings.Chars_Ptr);
   procedure GlobalEval (
      interp   : in Interp_Ptr; 
      Script   : in String);
   -- Evaluate Script in interp at global level.
   -- The functions return the result of the evaluation.  Call Result to get
   -- the result of the procedure call.  In either case, the exception Tcl_Error
   -- is raised if evaluation is not successful.  Then, the error string can be
   -- obtained by calling Ada.Exceptions.Exception_Message or Result.

   function Init (
      interp          : in Interp_Ptr) return C.Int;
   pragma Import (C, Init, "Tcl_Init");

   procedure InitMemory (
      interp          : in Interp_Ptr);
   pragma Import (C, InitMemory, "Tcl_InitMemory");

   function GetInterpPath (
      askInterp       : in Interp_Ptr;
      slaveInterp     : in Interp_Ptr) return C.Int;
   pragma Import (C, GetInterpPath, "Tcl_GetInterpPath");

   function IsSafe (
      interp          : in Interp_Ptr) return C.Int;
   pragma Import (C, IsSafe, "Tcl_IsSafe");

   function LinkVar (
      interp          : in Interp_Ptr;
      varName         : in C.Strings.Chars_Ptr;
      addr            : in C.Strings.Chars_Ptr;
      typ             : in C.Int) return C.Int;
   pragma Import (C, LinkVar, "Tcl_LinkVar");

   procedure Main (
      argc            : in C.Int;
      argv            : in C_Aux.Chars_Ptr_Ptr;
      appInitProc     : in AppInitProc_Ptr);
   pragma Import (C, Main, "Tcl_Main");

   function MakeSafe (
      interp          : in Interp_Ptr) return C.Int;
   pragma Import (C, MakeSafe, "Tcl_MakeSafe");

   function Merge (
      argc            : in C.Int;
      argv            : in C_Aux.Chars_Ptr_Ptr) return C.Strings.Chars_Ptr ;
   pragma Import (C, Merge, "Tcl_Merge");

   function ParseVar (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      termPtr         : in C_Aux.Chars_Ptr_Ptr) return C.Strings.Chars_Ptr ;
   pragma Import (C, ParseVar, "Tcl_ParseVar");

   function PosixError (
      interp          : in Interp_Ptr) return C.Strings.Chars_Ptr ;
   pragma Import (C, PosixError, "Tcl_PosixError");

   function PosixError (
      interp : in Interp_Ptr) return String;
   -- Sets the errorCode variable in interp based on the current value
   -- of errno, and returns a string identifying the error.

   procedure PrintDouble (
      interp          : in Interp_Ptr;
      value           : in C.Double;
      dst             : in C.Strings.Chars_Ptr);
   pragma Import (C, PrintDouble, "Tcl_PrintDouble");

   function PutEnv (
      strng           : in C.Strings.Chars_Ptr) return C.Int;
   pragma Import (C, PutEnv, "Tcl_PutEnv");

   procedure ReapDetachedProcs;
   pragma Import (C, ReapDetachedProcs, "Tcl_ReapDetachedProcs");

   function RecordAndEval (
      interp          : in Interp_Ptr;
      cmd             : in C.Strings.Chars_Ptr;
      flags           : in C.Int) return C.Int;
   pragma Import (C, RecordAndEval, "Tcl_RecordAndEval");

   function RecordAndEval (
      interp : in Interp_Ptr;
      Script : in C.Strings.Chars_Ptr;
      Eval   : in BOOLEAN := True) return C.Strings.Chars_Ptr;
   function RecordAndEval (
      interp : in Interp_Ptr;
      Script : in String;
      Eval   : in BOOLEAN := True) return C.Strings.Chars_Ptr;
   procedure RecordAndEval (
      interp : in Interp_Ptr;
      Script : in C.Strings.Chars_Ptr;
      Eval   : in BOOLEAN := True);
   procedure RecordAndEval (
      interp : in Interp_Ptr;
      Script : in String;
      Eval   : in BOOLEAN := True);
   -- Records Script as an event in Interp's history list and evaluates
   -- it if Eval is True.
   -- The functions return the result of the evaluation.  Call Result to get
   -- the result of the procedure call.  In either case, the exception Tcl_Error
   -- is raised if evaluation is not successful.  Then, the error string can be
   -- obtained by calling Ada.Exceptions.Exception_Message or Result.

   function RegExpCompile (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr) return RegExp_Rec;
   pragma Import (C, RegExpCompile, "Tcl_RegExpCompile");

   function RegExpExec (
      interp          : in Interp_Ptr;
      regexp          : in RegExp_Rec;
      strng           : in C.Strings.Chars_Ptr;
      start           : in C.Strings.Chars_Ptr) return C.Int;
   pragma Import (C, RegExpExec, "Tcl_RegExpExec");

   function RegExpMatch (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      pattern         : in C.Strings.Chars_Ptr) return C.Int;
   pragma Import (C, RegExpMatch, "Tcl_RegExpMatch");

   procedure RegExpRange (
      regexp          : in RegExp_Rec;
      index           : in C.Int;
      startPtr        : in C_Aux.Chars_Ptr_Ptr;
      endPtr          : in C_Aux.Chars_Ptr_Ptr);
   pragma Import (C, RegExpRange, "Tcl_RegExpRange");

   procedure ResetResult (
      interp          : in Interp_Ptr);
   pragma Import (C, ResetResult, "Tcl_ResetResult");

   function ScanElement (
      strng           : in C.Strings.Chars_Ptr;
      flagPtr         : in C_Aux.Int_Ptr) return C.Int;
   pragma Import (C, ScanElement, "Tcl_ScanElement");

   procedure SetErrorCode (
      interp          : in Interp_Ptr;
      Field1          : in C.Strings.Chars_Ptr;
      Field2          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field3          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field4          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field5          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field6          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field7          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field8          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field9          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr);
   -- Creates a list whose elements are the arguments Field1..Field9 and
   -- sets the Tcl errorCode variable in interpr to the list.
   -- The C version of this procedure takes variable number of arguments.
   -- This is simulated here by providing 9 string arguments.  This method
   -- is not quite as flexible as C's indefinite number of arguments,
   -- but we believe that in practice, this will be sufficient.  You can
   -- always call SetErrorCode more than once.  The body of this procedure
   -- adds a Null argument before calling the C function, so you may use
   -- all 9 arguments.

   procedure SetErrorCode (
      interp          : in Interp_Ptr;
      Field1          : in String;
      Field2          : in String := "";
      Field3          : in String := "";
      Field4          : in String := "";
      Field5          : in String := "";
      Field6          : in String := "";
      Field7          : in String := "";
      Field8          : in String := "";
      Field9          : in String := "");
   -- Creates a list whose elements are the arguments Field1..Field9 and
   -- sets the Tcl errorCode variable in interp to the list.

   procedure SetFileCloseProc (
      proc            : in C_Aux.Void_Ptr);
   pragma Import (C, SetFileCloseProc, "Tcl_SetFileCloseProc");

   procedure SetPanicProc (
      proc            : in PanicProc);
   pragma Import (C, SetPanicProc, "Tcl_SetPanicProc");

   function SetRecursionLimit (
      interp          : in Interp_Ptr;
      depth           : in C.Int) return C.Int;
   pragma Import (C, SetRecursionLimit, "Tcl_SetRecursionLimit");

   procedure SetResult (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      freeProc        : in FreeProc_Ptr);
   procedure SetResult (
      interp          : in Interp_Ptr;
      strng           : in C.Strings.Chars_Ptr;
      FreeProc        : in C.Int);
   pragma Import (C, SetResult, "Tcl_SetResult");

   procedure SetResult (
      interp   : in Interp_Ptr;
      strng    : in String);
   -- Copy the string strng into interp so that it is returned as its
   -- result string.  FreeProc is not needed, since the Ada string must
   -- be converted to a C string anyway before calling the Tcl library.

   function SetVar (
      interp          : in Interp_Ptr;
      varName         : in C.Strings.Chars_Ptr;
      newValue        : in C.Strings.Chars_Ptr;
      flags           : in C.Int) return C.Strings.Chars_Ptr ;
   pragma Import (C, SetVar, "Tcl_SetVar");

   procedure SetVar (
      interp   : in Interp_Ptr;
      VarName  : in C.Strings.Chars_Ptr;
      NewValue : in C.Strings.Chars_Ptr;
      Flags    : in C.Int := 0);
   procedure SetVar (
      interp   : in Interp_Ptr;
      VarName  : in String;
      NewValue : in C.Strings.Chars_Ptr;
      Flags    : in C.Int := 0);
   procedure SetVar (
      interp   : in Interp_Ptr;
      VarName  : in C.Strings.Chars_Ptr;
      NewValue : in String;
      Flags    : in C.Int := 0);
   procedure SetVar (
      interp   : in Interp_Ptr;
      VarName  : in String;
      NewValue : in String;
      Flags    : in C.Int := 0);
   -- Sets the value of the named Tcl variable to NewValue, creating the
   -- variable if it did not previously exist.  The exception Tcl_Error
   -- is raised if not successful.  Then, the error string can be
   -- obtained by calling Ada.Exceptions.Exception_Message or Result.

   function SetVar2 (
      interp          : in Interp_Ptr;
      part1           : in C.Strings.Chars_Ptr;
      part2           : in C.Strings.Chars_Ptr;
      newValue        : in C.Strings.Chars_Ptr;
      flags           : in C.Int) return C.Strings.Chars_Ptr ;
   pragma Import (C, SetVar2, "Tcl_SetVar2");

   procedure SetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr; 
      Name2    : in C.Strings.Chars_Ptr; 
      NewValue : in C.Strings.Chars_Ptr; 
      Flags    : in C.Int := 0);
   procedure SetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr; 
      Name2    : in String; 
      NewValue : in C.Strings.Chars_Ptr; 
      Flags    : in C.Int := 0);
   procedure SetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr; 
      Name2    : in C.Strings.Chars_Ptr; 
      NewValue : in String; 
      Flags    : in C.Int := 0);
   procedure SetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr; 
      Name2    : in String; 
      NewValue : in String; 
      Flags    : in C.Int := 0);
   procedure SetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in String;
      Name2    : in C.Strings.Chars_Ptr; 
      NewValue : in C.Strings.Chars_Ptr; 
      Flags    : in C.Int := 0);
   procedure SetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in String;
      Name2    : in String; 
      NewValue : in C.Strings.Chars_Ptr; 
      Flags    : in C.Int := 0);
   procedure SetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in String;
      Name2    : in C.Strings.Chars_Ptr; 
      NewValue : in String; 
      Flags    : in C.Int := 0);
   procedure SetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in String;
      Name2    : in String; 
      NewValue : in String; 
      Flags    : in C.Int := 0);
   -- Sets the value of the named Tcl variable to NewValue, creating the
   -- variable if it did not previously exist.  The exception Tcl_Error
   -- is raised if not successful.  Then, the error string can be
   -- obtained by calling Ada.Exceptions.Exception_Message or Result.

   function ShareChannel (
      channelId       : in C.Strings.Chars_Ptr;
      interp          : in Interp_Ptr;
      srcInterp       : in Interp_Ptr;
      destInterp      : in Interp_Ptr) return C.Int;
   pragma Import (C, ShareChannel, "Tcl_ShareChannel");

   function SignalId (
      sig             : in C.Int) return C.Strings.Chars_Ptr ;
   pragma Import (C, SignalId, "Tcl_SignalId");

   function SignalId (
      Sig : in C.Int) return String;
   -- Returns the symbolic name for Sig, such as SIGINT.

   function SignalMsg (
      sig             : in C.Int) return C.Strings.Chars_Ptr ;
   pragma Import (C, SignalMsg, "Tcl_SignalMsg");

   function SignalMsg (
      Sig : in C.Int) return String;
   -- Returns a human-readable message describing signal Sig.

   function SplitList (
      interp          : in Interp_Ptr;
      list            : in C.Strings.Chars_Ptr;
      argcPtr         : in C_Aux.Int_Ptr;
      argvPtr         : in C_Aux.Chars_Ptr_Ptr_Ptr) return C.Int;
   pragma Import (C, SplitList, "Tcl_SplitList");

   procedure SplitList (
      interp  : in     Interp_Ptr;
      List    : in     C.Strings.Chars_Ptr;
      Argc    :    out C.Int;
      ArgvPtr :    out C_Aux.Chars_Ptr_Ptr);
   procedure SplitList (
      interp  : in     Interp_Ptr;
      List    : in     String;
      Argc    :    out C.Int;
      ArgvPtr :    out C_Aux.Chars_Ptr_Ptr);
   -- Parses List as a Tcl list and creates an array of strings whose
   -- values are the elements of the list.  The resulting list is
   -- dynamically allocated and a pointer to it is stored in ArgvPtr.
   -- This space must eventually be freed by the caller using C_Aux.Free.
   -- The exception Tcl_Error is raised if SplitList fails.  Then, the error
   -- string can be obtained by calling Ada.Exceptions.Exception_Message or Result.

   function StringMatch (
      strng           : in C.Strings.Chars_Ptr;
      pattern         : in C.Strings.Chars_Ptr) return C.Int;
   pragma Import (C, StringMatch, "Tcl_StringMatch");

   function StringMatch (
      strng   : in C.Strings.Chars_Ptr;
      Pattern : in C.Strings.Chars_Ptr) return Boolean;
   function StringMatch (
      strng   : in String;
      Pattern : in C.Strings.Chars_Ptr) return Boolean;
   function StringMatch (
      strng   : in C.Strings.Chars_Ptr;
      Pattern : in String) return Boolean;
   function StringMatch (
      strng   : in String;
      Pattern : in String) return Boolean;
   -- Returns True if strng matches Pattern using glob-style rules for
   -- pattern matchine, False otherwise.

   function TildeSubst (
      interp          : in Interp_Ptr;
      name            : in C.Strings.Chars_Ptr;
      bufferPtr       : in DString_Ptr) return C.Strings.Chars_Ptr ;
   pragma Import (C, TildeSubst, "Tcl_TildeSubst");

   function TildeSubst (
      interp  : in Interp_Ptr;
      Name    : in String;
      TempPtr : in DString_Ptr) return C.Strings.Chars_Ptr;
   -- If Name starts with ~, returns a new name with the ~ and
   -- following characters replaced with the corresponding home
   -- directory name.  If Name doesn't start with ~, returns Name.
   -- Uses TempPtr if needed to hold new name (caller need not
   -- initialize it, but must free if by calling DstringFree).
   -- If an error occurs, returns Null_Ptr and leaves an error
   -- message in Interp's Result string.

   procedure TildeSubst (
      interp  : in Interp_Ptr;
      Name    : in C.Strings.Chars_Ptr;
      TempPtr : in DString_Ptr);
   procedure TildeSubst (
      interp  : in Interp_Ptr;
      Name    : in String;
      TempPtr : in DString_Ptr);
   -- If Name starts with ~, returns a new name with the ~ and
   -- following characters replaced with the corresponding home
   -- directory name.  If Name doesn't start with ~, returns Name.
   -- Uses TempPtr if needed to hold new name (caller need not
   -- initialize it, but must free if by calling DstringFree).
   -- If an error occurs, raises Tcl_Error.  Then, the error
   -- string can be obtained by calling Ada.Exceptions.Exception_Message
   -- or Result.

   function TransferChannel (
      channelId       : in C.Strings.Chars_Ptr;
      interp          : in Interp_Ptr;
      srcInterp       : in Interp_Ptr;
      destInterp      : in Interp_Ptr) return C.Int;
   pragma Import (C, TransferChannel, "Tcl_TransferChannel");

   procedure UnlinkVar (
      interp          : in Interp_Ptr;
      varName         : in C.Strings.Chars_Ptr);
   pragma Import (C, UnlinkVar, "Tcl_UnlinkVar");

   function UnsetVar (
      interp          : in Interp_Ptr;
      varName         : in C.Strings.Chars_Ptr;
      flags           : in C.Int) return C.Int;
   pragma Import (C, UnsetVar, "Tcl_UnsetVar");

   procedure UnsetVar (
      interp   : in Interp_Ptr;
      VarName  : in C.Strings.Chars_Ptr; 
      Flags    : in C.Int := 0);
   procedure UnsetVar (
      interp   : in Interp_Ptr;
      VarName  : in String; 
      Flags    : in C.Int := 0);
   -- Removes the variable from Interp.  Raises Tcl_Error if the variable
   -- does not exist or has an active trace.

   function UnsetVar2 (
      interp          : in Interp_Ptr;
      part1           : in C.Strings.Chars_Ptr;
      part2           : in C.Strings.Chars_Ptr;
      flags           : in C.Int) return C.Int;
   pragma Import (C, UnsetVar2, "Tcl_UnsetVar2");

   procedure UnsetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr; 
      Name2    : in C.Strings.Chars_Ptr; 
      Flags    : in C.Int := 0);
   procedure UnsetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in String; 
      Name2    : in C.Strings.Chars_Ptr; 
      Flags    : in C.Int := 0);
   procedure UnsetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr; 
      Name2    : in String; 
      Flags    : in C.Int := 0);
   procedure UnsetVar2 (
      interp   : in Interp_Ptr;
      Name1    : in String; 
      Name2    : in String; 
      Flags    : in C.Int := 0);
   -- Removes the variable from Interp.  Raises Tcl_Error if the variable
   -- does not exist or has an active trace.

   function UpVar (
      interp          : in Interp_Ptr;
      frameName       : in C.Strings.Chars_Ptr;
      varName         : in C.Strings.Chars_Ptr;
      localName       : in C.Strings.Chars_Ptr;
      flags           : in C.Int) return C.Int;
   pragma Import (C, UpVar, "Tcl_UpVar");

   function UpVar2 (
      interp          : in Interp_Ptr;
      frameName       : in C.Strings.Chars_Ptr;
      part1           : in C.Strings.Chars_Ptr;
      part2           : in C.Strings.Chars_Ptr;
      localName       : in C.Strings.Chars_Ptr;
      flags           : in C.Int) return C.Int;
   pragma Import (C, UpVar2, "Tcl_UpVar2");

   function VarEval (
      Interp  : in Interp_Ptr;
      Script1 : in C.Strings.Chars_Ptr;
      Script2 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script3 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script4 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script5 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script6 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script7 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script8 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script9 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr) return C.Int;
   -- Concatenate all the String prameters into a single string and
   -- evaluate the resulting script in Interp.  Returns OK or ERROR.
   -- The result or error string can be obtained by calling function
   -- Result above.

   function VarEval (
      Interp  : in Interp_Ptr;
      Script1 : in C.Strings.Chars_Ptr;
      Script2 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script3 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script4 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script5 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script6 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script7 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script8 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script9 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr) return C.Strings.Chars_Ptr;
   function VarEval (
      Interp  : in Interp_Ptr;
      Script1 : in String;
      Script2 : in String := "";
      Script3 : in String := "";
      Script4 : in String := "";
      Script5 : in String := "";
      Script6 : in String := "";
      Script7 : in String := "";
      Script8 : in String := "";
      Script9 : in String := "") return C.Strings.Chars_Ptr;
   procedure VarEval (
      Interp  : in Interp_Ptr;
      Script1 : in C.Strings.Chars_Ptr;
      Script2 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script3 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script4 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script5 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script6 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script7 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script8 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script9 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr);
   procedure VarEval (
      Interp  : in Interp_Ptr;
      Script1 : in String;
      Script2 : in String := "";
      Script3 : in String := "";
      Script4 : in String := "";
      Script5 : in String := "";
      Script6 : in String := "";
      Script7 : in String := "";
      Script8 : in String := "";
      Script9 : in String := "");
   -- Concatenate Script1..String9 arguments into a single string and
   -- evaluate the resulting script in Interp.
   -- The functions return the result of the evaluation.  Call Result to
   -- get the result of the procedure call.  In either case, the exception
   -- Tcl_Error is raised if evaluation is not successful.  Then, the
   -- error string can be obtained by calling
   -- Ada.Exceptions.Exception_Message or Result.

   --------------------------------------------------------------------
   --
   --	Exported Procedures in Ada binding only
   --
   --------------------------------------------------------------------

   procedure Main_Exit (
      Interp  : in Interp_Ptr;
      Code    : in C.Int);

private

   -- Interp_Rec is is a dummy type which models the publicly available
   -- interface (from the C Tcl library) to the Tcl interpreter.
   -- It is not the full type definition, which doesn't
   -- matter as the Tcl library allocates and "owns" the interpreter space.

   type Interp_Rec is
      record
         result : C.Strings.Chars_Ptr; -- Points to result string returned by
				  -- last command.
         freeProc  : FreeProc_Ptr;   -- Null means result is statically allocated.
				  -- If non-null, gives address of procedure
				  -- to invoke to free the result.  Must be
				  -- freed by Tcl_Eval before executing next
				  -- command.
         errorLine : C.Int;	  -- When TCL_ERROR is returned, this gives
				  -- the line number within the command where
				  -- the error occurred (1 means first line).
      end record;
   pragma Convention (C, Interp_Rec);

   type AsyncHandler_Rec is null record;
   pragma Convention (C, AsyncHandler_Rec);

   type RegExp_Rec is null record;
   pragma Convention (C, RegExp_Rec);

   DSTRING_STATIC_SIZE : constant := 200;

   type DString_Rec is
      record
      	 strng : C.Strings.Chars_Ptr;-- Points to beginning of string:  either
				   -- staticSpace below or a malloc'ed array.
         Length   : C.Int;	   -- Number of non-NULL characters in the
				   -- string.
         spaceAvl : C.Int;	   -- Total number of bytes available for the
				   -- string and its terminating NULL char.
         staticSpace : aliased String (1..DSTRING_STATIC_SIZE);
				   -- Space to use in common case where string
      end record;		   -- is small.

end Tcl;
