--------------------------------------------------------------------
--
-- tcl.adb --
--
-- Copyright (c) 1995-1996 Terry J. Westley
--
-- See the file "license.terms" for information on usage and redistribution
-- of this file, and for a DISCLAIMER OF ALL WARRANTIES.
--
--
--------------------------------------------------------------------

with Ada.Exceptions;

package body Tcl is

   function "=" (Left, Right : in C.Int) return Boolean renames C."=";

   function Result (Interp : in Interp_Ptr) return C.Strings.Chars_Ptr is
   begin -- Result
      return Interp.Result;
   end Result;

   function SResult (Interp : in Interp_Ptr) return String is
   begin -- SResult
      return C_Aux.Value (Interp.Result);
   end SResult;

   function Result (
      Interp : in Interp_Ptr;
      Expr   : in Boolean) return C.Strings.Chars_Ptr is
   -- If Expr is True, this function returns the current Result string in
   -- Interp.  Otherwise, it raises the exception Tcl_Error.
   begin -- Result
      if Expr then
         return Result (Interp);
      else
         -- raise Tcl_Error;
	 -- not yet implemented in GNAT:
	 Ada.Exceptions.Raise_Exception (Tcl_Error'Identity, SResult (Interp));
      end if; 
   end Result;

   function Result (
      Interp : in Interp_Ptr;
      Expr   : in Boolean) return String is
   -- If Expr is True, this function returns the current Result string in
   -- Interp.  Otherwise, it raises the exception Tcl_Error.
   begin -- Result
      if Expr then
         return SResult (Interp);
      else
         -- raise Tcl_Error;
	 -- not yet implemented in GNAT:
	 Ada.Exceptions.Raise_Exception (Tcl_Error'Identity, SResult (Interp));
      end if; 
   end Result;

   procedure Result (
      Interp : in Interp_Ptr;
      Expr   : in Boolean) is
   -- If Expr is False, this procedure raises the exception Tcl_Error.
   begin -- Result
      if not Expr then
         -- raise Tcl_Error;
	 -- not yet implemented in GNAT:
	 Ada.Exceptions.Raise_Exception (Tcl_Error'Identity, SResult (Interp));
      end if; 
   end Result;

   function ErrorLine (Interp : in Interp_Ptr) return C.Int is
   begin -- ErrorLine
      return Interp.ErrorLine;
   end ErrorLine;

   function DStringLength (dsPtr : in DString_Ptr) return C.Int is
   begin -- DStringLength
      return dsPtr.Length;
   end DStringLength;

   function DStringValue (dsPtr : in DString_Ptr) return C.Strings.Chars_Ptr is
   begin -- DStringValue
      return dsPtr.strng;
   end DStringValue;

   package body Generic_CreateCommands is

      function CreateCommand (
	 Interp     : in Interp_Ptr;
	 CmdName    : in String;
	 Proc       : in CmdProc_Ptr;
	 ClientData : in ClientData_Ptr;
	 DeleteProc : in CmdDeleteProc_Ptr) return Command_Ptr is
      --
	 CmdName_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (CmdName);
	 Return_Cmd  : Command_Ptr;
      begin -- CreateCommand
	 Return_Cmd := CreateCommand (
	    Interp, CmdName_Ptr, Proc, ClientData, DeleteProc);
	 C.Strings.Free (CmdName_Ptr);
	 return Return_Cmd;
      end CreateCommand;

      procedure CreateCommand (
	 Interp     : in Interp_Ptr;
	 CmdName    : in String;
	 Proc       : in CmdProc_Ptr;
	 ClientData : in ClientData_Ptr;
	 DeleteProc : in CmdDeleteProc_Ptr) is
      --
	 CmdName_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (CmdName);
	 Return_Cmd  : Command_Ptr;
      begin -- CreateCommand
	 Return_Cmd := CreateCommand (
	    Interp, CmdName_Ptr, Proc, ClientData, DeleteProc);
	 C.Strings.Free (CmdName_Ptr);
      end CreateCommand;

   end Generic_CreateCommands;

   package body Generic_Hash is

      function CreateHashEntry (
	 tablePtr : in HashTable_Ptr;
	 Key      : in C.Strings.Chars_Ptr;
	 NewPtr   : in C_Aux.Int_Ptr) return HashEntry_Ptr is
      begin -- CreateHashEntry
	 return TablePtr.CreateProc (TablePtr, Key, NewPtr);
      end CreateHashEntry;

      function FindHashEntry (
	 TablePtr : in HashTable_Ptr;
	 Key      : in C.Strings.Chars_Ptr) return HashEntry_Ptr is
      begin -- FindHashEntry
	 return TablePtr.FindProc (TablePtr, Key);
      end FindHashEntry;

      function GetHashKey (
	 TablePtr : in HashTable_Ptr;
	 EntryPtr : in HashEntry_Ptr) return C.Strings.Chars_Ptr is
      begin -- GetHashKey
	 return C.Strings.To_Chars_Ptr (EntryPtr.Strng'access);
      end GetHashKey;

      function GetHashValue (
	 EntryPtr : in HashEntry_Ptr) return ClientData_Ptr is
      begin -- GetHashValue
	 return EntryPtr.ClientData;
      end GetHashValue;

      procedure SetHashValue (
	 EntryPtr : in HashEntry_Ptr;
	 Value    : in ClientData_Ptr) is
      begin -- SetHashValue
	 EntryPtr.ClientData := Value;
      end SetHashValue;

   end Generic_Hash;

   package body Generic_ExprEnum is

      function ExprEnum (
	 Interp : in Interp_Ptr;
	 strng  : in C.Strings.Chars_Ptr) return Enum is
      --
	 function "=" (Left, Right : in C.Int) return Boolean renames C."=";
	 Completion_Code : C.Int;
      begin -- ExprEnum
	 Completion_Code := ExprString (Interp, strng);
	 return Enum'value (Result (Interp, Completion_Code /= ERROR));
      end ExprEnum;

      function ExprEnum (
	 Interp : in Interp_Ptr;
	 strng  : in String) return Enum is
      --
	 function "=" (Left, Right : in C.Int) return Boolean renames C."=";
	 Str_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
	 Completion_Code : C.Int;
      begin -- ExprEnum
	 Completion_Code := ExprString (Interp, Str_Ptr);
	 C.Strings.Free (Str_Ptr);
	 return Enum'value (Result (Interp, Completion_Code /= ERROR));
      end ExprEnum;

   end Generic_ExprEnum;

   package body Generic_ExprFloat is

      function ExprFloat (
	 Interp : in Interp_Ptr;
	 strng  : in C.Strings.Chars_Ptr) return Num is
      --
	 function "=" (Left, Right : in C.Int) return Boolean renames C."=";
	 Completion_Code : C.Int;
      begin -- ExprFloat
	 Completion_Code := ExprString (Interp, strng);
	 return Num'value (Result (Interp, Completion_Code /= ERROR));
      end ExprFloat;

      function ExprFloat (
	 Interp : in Interp_Ptr;
	 strng  : in String) return Num is
      --
	 function "=" (Left, Right : in C.Int) return Boolean renames C."=";
	 Str_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
	 Completion_Code : C.Int;
      begin -- ExprFloat
	 Completion_Code := ExprString (Interp, Str_Ptr);
	 C.Strings.Free (Str_Ptr);
	 return Num'value (Result (Interp, Completion_Code /= ERROR));
      end ExprFloat;

   end Generic_ExprFloat;

   package body Generic_ExprFixed is

      function ExprFixed (
	 Interp : in Interp_Ptr;
	 strng  : in C.Strings.Chars_Ptr) return Num is
      --
	 function "=" (Left, Right : in C.Int) return Boolean renames C."=";
	 Completion_Code : C.Int;
      begin -- ExprFixed
	 Completion_Code := ExprString (Interp, strng);
	 return Num'value (Result (Interp, Completion_Code /= ERROR));
      end ExprFixed;

      function ExprFixed (
	 Interp : in Interp_Ptr;
	 strng  : in String) return Num is
      --
	 function "=" (Left, Right : in C.Int) return Boolean renames C."=";
	 Str_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
	 Completion_Code : C.Int;
      begin -- ExprFixed
	 Completion_Code := ExprString (Interp, Str_Ptr);
	 C.Strings.Free (Str_Ptr);
	 return Num'value (Result (Interp, Completion_Code /= ERROR));
      end ExprFixed;

   end Generic_ExprFixed;

   package body Generic_ExprInteger is

      function ExprInteger (
	 Interp : in Interp_Ptr;
	 strng  : in C.Strings.Chars_Ptr) return Num is
      --
	 function "=" (Left, Right : in C.Int) return Boolean renames C."=";
	 Completion_Code : C.Int;
      begin -- ExprInteger
	 Completion_Code := ExprString (Interp, strng);
	 return Num'value (Result (Interp, Completion_Code /= ERROR));
      end ExprInteger;

      function ExprInteger (
	 Interp : in Interp_Ptr;
	 strng  : in String) return Num is
      --
	 function "=" (Left, Right : in C.Int) return Boolean renames C."=";
	 Str_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
	 Completion_Code : C.Int;
      begin -- ExprInteger
	 Completion_Code := ExprString (Interp, Str_Ptr);
	 C.Strings.Free (Str_Ptr);
	 return Num'value (Result (Interp, Completion_Code /= ERROR));
      end ExprInteger;

   end Generic_ExprInteger;

   package body Generic_ExprModulus is

      function ExprMod (
	 Interp : in Interp_Ptr;
	 strng  : in C.Strings.Chars_Ptr) return Num is
      --
	 function "=" (Left, Right : in C.Int) return Boolean renames C."=";
	 Completion_Code : C.Int;
      begin -- ExprMod
	 Completion_Code := ExprString (Interp, strng);
	 return Num'value (Result (Interp, Completion_Code /= ERROR));
      end ExprMod;

      function ExprMod (
	 Interp : in Interp_Ptr;
	 strng  : in String) return Num is
      --
	 function "=" (Left, Right : in C.Int) return Boolean renames C."=";
	 Str_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
	 Completion_Code : C.Int;
      begin -- ExprMod
	 Completion_Code := ExprString (Interp, Str_Ptr);
	 C.Strings.Free (Str_Ptr);
	 return Num'value (Result (Interp, Completion_Code /= ERROR));
      end ExprMod;

   end Generic_ExprModulus;

   procedure AddErrorInfo (
      Interp  : in Interp_Ptr;
      Message : in String) is
   --
      Message_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Message);
   begin -- AddErrorInfo
      AddErrorInfo (Interp, Message_Ptr);
      C.Strings.Free (Message_Ptr);
   end AddErrorInfo;

   procedure AppendElement (
      Interp   : in Interp_Ptr;
      strng    : in String) is
   --
      Str_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
   begin -- AppendElement
      AppendElement (Interp, Str_Ptr);
      C.Strings.Free (Str_Ptr);
   end AppendElement;

   procedure Tcl_AppendResult (
      interp          : in Interp_Ptr;
      String1         : in C.Strings.Chars_Ptr;
      String2         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String3         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String4         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String5         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String6         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String7         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String8         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String9         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      nullarg         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr);
   pragma Import (C, Tcl_AppendResult, "Tcl_AppendResult");

   procedure AppendResult (
      interp          : in Interp_Ptr;
      String1         : in C.Strings.Chars_Ptr;
      String2         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String3         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String4         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String5         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String6         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String7         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String8         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      String9         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr) is
   begin -- AppendResult
      Tcl_AppendResult (interp, String1, String2, String3, String4,
         String5, String6, String7, String8, String9);
   end AppendResult;

   procedure AppendResult (
      Interp  : in Interp_Ptr;
      String1 : in String;
      String2 : in String := "";
      String3 : in String := "";
      String4 : in String := "";
      String5 : in String := "";
      String6 : in String := "";
      String7 : in String := "";
      String8 : in String := "";
      String9 : in String := "") is
   --
      String1_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (String1);
      String2_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (String2);
      String3_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (String3);
      String4_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (String4);
      String5_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (String5);
      String6_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (String6);
      String7_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (String7);
      String8_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (String8);
      String9_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (String9);
   begin -- AppendResult
      Tcl_AppendResult (Interp,
         String1_Ptr,
         String2_Ptr,
         String3_Ptr,
         String4_Ptr,
         String5_Ptr,
         String6_Ptr,
         String7_Ptr,
         String8_Ptr,
         String9_Ptr);
      C.Strings.Free (String1_Ptr);
      C.Strings.Free (String2_Ptr);
      C.Strings.Free (String3_Ptr);
      C.Strings.Free (String4_Ptr);
      C.Strings.Free (String5_Ptr);
      C.Strings.Free (String6_Ptr);
      C.Strings.Free (String7_Ptr);
      C.Strings.Free (String8_Ptr);
      C.Strings.Free (String9_Ptr);
   end AppendResult;

   procedure AsyncInvoke (
      interp : in Interp_Ptr;
      code   : in C.Int) is
   --
      Completion_Code : C.Int;
   begin -- AsyncInvoke
      Completion_Code := AsyncInvoke (Interp, Code);
      Result (Interp, Completion_Code /= ERROR);
   end AsyncInvoke;

   procedure AsyncReady is
   --
      Completion_Code : C.Int;
   begin -- AsyncReady
      Completion_Code := AsyncReady;
   end AsyncReady;

   function CommandComplete (
      Cmd : in C.Strings.Chars_Ptr) return Boolean is
   begin -- CommandComplete
      return CommandComplete (Cmd) = 1;
   end CommandComplete;

   function CommandComplete (
      Cmd : in String) return Boolean is
   --
      Cmd_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Cmd);
      Completion_Code : C.Int;
   begin -- CommandComplete
      Completion_Code := CommandComplete (Cmd_Ptr);
      C.Strings.Free (Cmd_Ptr);
      return Completion_Code = 1;
   end CommandComplete;

   procedure CreatePipeline (
      Interp      : in Interp_Ptr;
      Argc        : in C.Int; 
      Argv        : in C_Aux.Chars_Ptr_Ptr;
      NumPids     :    out C.Int;
      PidArrayPtr :    out C_Aux.Int_Ptr_Ptr;
      InPipePtr   :    out C.Int;
      OutPipePtr  :    out C.Int;
      ErrFilePtr  :    out C.Int) is
   --
      C_PidArrayPtr   : C_Aux.Int_Ptr_Ptr;
      C_InPipePtr     : aliased C.Int;
      C_OutPipePtr    : aliased C.Int;
      C_ErrFilePtr    : aliased C.Int;
   begin -- CreatePipeline
      NumPids := CreatePipeline (
         Interp, Argc, Argv, C_PidArrayPtr,
	 C_InPipePtr'unchecked_access,
	 C_OutPipePtr'unchecked_access,
	 C_ErrFilePtr'unchecked_access);
      PidArrayPtr := C_PidArrayPtr;
      InPipePtr   := C_InPipePtr;
      OutPipePtr  := C_OutPipePtr;
      ErrFilePtr  := C_ErrFilePtr;
   end CreatePipeline;

   procedure DeleteCommand (
      Interp  : in Interp_Ptr;
      CmdName : in C.Strings.Chars_Ptr) is
   --
      Completion_Code : C.Int;
   begin -- DeleteCommand
      Completion_Code := DeleteCommand (Interp, CmdName);
      Result (Interp, Completion_Code /= ERROR);
   end DeleteCommand;

   procedure DeleteCommand (
      Interp  : in Interp_Ptr;
      CmdName : in String) is
   --
      CmdName_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (CmdName);
      Completion_Code : C.Int;
   begin -- DeleteCommand
      Completion_Code := DeleteCommand (Interp, CmdName_Ptr);
      C.Strings.Free (CmdName_Ptr);
      Result (Interp, Completion_Code /= ERROR);
   end DeleteCommand;

   function DStringAppend (
      DsPtr  : in DString_Ptr;
      strng  : in String;
      Length : in C.Int := -1) return C.Strings.Chars_Ptr is
   --
      Str_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
      Result_Ptr : C.Strings.Chars_Ptr;
   begin -- DStringAppend
      Result_Ptr := DStringAppend (DsPtr, Str_Ptr, Length);
      C.Strings.Free (Str_Ptr);
      return Result_Ptr;
   end DStringAppend;

   procedure DStringAppend (
      DsPtr  : in DString_Ptr;
      strng  : in C.Strings.Chars_Ptr;
      Length : in C.Int := -1) is
   --
      Result_Ptr : C.Strings.Chars_Ptr;
   begin -- DStringAppend
      Result_Ptr := DStringAppend (DsPtr, strng, Length);
   end DStringAppend;

   procedure DStringAppend (
      DsPtr  : in DString_Ptr;
      strng  : in String;
      Length : in C.Int := -1) is
   --
      Str_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
      Result_Ptr : C.Strings.Chars_Ptr;
   begin -- DStringAppend
      Result_Ptr := DStringAppend (DsPtr, Str_Ptr, Length);
      C.Strings.Free (Str_Ptr);
   end DstringAppend;

   function DStringAppendElement (
      DsPtr  : in DString_Ptr;
      strng  : in String) return C.Strings.Chars_Ptr is
   --
      Str_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
      Result_Ptr : C.Strings.Chars_Ptr;
   begin -- DStringAppendElement
      Result_Ptr := DStringAppendElement (DsPtr, Str_Ptr);
      C.Strings.Free (Str_Ptr);
      return Result_Ptr;
   end DStringAppendElement;

   procedure DStringAppendElement (
      DsPtr  : in DString_Ptr;
      strng  : in C.Strings.Chars_Ptr) is
   --
      Result_Ptr : C.Strings.Chars_Ptr;
   begin -- DStringAppendElement
      Result_Ptr := DStringAppendElement (DsPtr, strng);
   end DStringAppendElement;

   procedure DStringAppendElement (
      DsPtr  : in DString_Ptr;
      strng  : in String) is
   --
      Str_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
      Result_Ptr : C.Strings.Chars_Ptr;
   begin -- DStringAppendElement
      Result_Ptr := DStringAppendElement (DsPtr, Str_Ptr);
      C.Strings.Free (Str_Ptr);
   end DstringAppendElement;

   function ErrnoId return String is
   begin -- ErrnoId
      return C_Aux.Value (ErrnoId);
   end ErrnoId;

   procedure ErrnoId (
      strng : out String;
      last  : out Natural) is
   --
      Error : constant String := ErrnoId;
   begin -- ErrnoId
      last := strng'first+Error'last-1;
      strng (strng'first..last) := Error;
   end ErrnoId;

   function Eval (
      Interp : in Interp_Ptr; 
      Script : in C.Strings.Chars_Ptr) return C.Strings.Chars_Ptr is
   --
      Completion_Code : C.Int;
   begin -- Eval
      Completion_Code := Eval (Interp, Script);
      return Result (Interp, Completion_Code /= ERROR);
   end Eval;

   function Eval (
      Interp : in Interp_Ptr; 
      Script : in String) return C.Strings.Chars_Ptr is
   --
      Script_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script);
      Completion_Code : C.Int;
   begin -- Eval
      Completion_Code := Eval (Interp, Script_Ptr);
      C.Strings.Free (Script_Ptr);
      return Result (Interp, Completion_Code /= ERROR);
   end Eval;

   procedure Eval (
      Interp : in Interp_Ptr; 
      Script : in C.Strings.Chars_Ptr) is
   --
      Completion_Code : C.Int;
   begin -- Eval
      Completion_Code := Eval (Interp, Script);
      Result (Interp, Completion_Code /= ERROR);
   end Eval;

   procedure Eval (
      Interp : in Interp_Ptr; 
      Script : in String) is
   --
      Script_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script);
      Completion_Code : C.Int;
   begin -- Eval
      Completion_Code := Eval (Interp, Script_Ptr);
      C.Strings.Free (Script_Ptr);
      Result (Interp, Completion_Code /= ERROR);
   end Eval;

   function EvalFile (
      Interp   : in Interp_Ptr; 
      FileName : in C.Strings.Chars_Ptr) return C.Strings.Chars_Ptr is
   --
      Completion_Code : C.Int;
   begin -- EvalFile
      Completion_Code := EvalFile (Interp, FileName);
      return Result (Interp, Completion_Code /= ERROR);
   end EvalFile;

   function EvalFile (
      Interp   : in Interp_Ptr; 
      FileName : in String) return C.Strings.Chars_Ptr is
   --
      FileName_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (FileName);
      Completion_Code : C.Int;
   begin -- EvalFile
      Completion_Code := EvalFile (Interp, FileName_Ptr);
      C.Strings.Free (FileName_Ptr);
      return Result (Interp, Completion_Code /= ERROR);
   end EvalFile;

   procedure EvalFile (
      Interp   : in Interp_Ptr; 
      FileName : in C.Strings.Chars_Ptr) is
   --
      Completion_Code : C.Int;
   begin -- EvalFile
      Completion_Code := EvalFile (Interp, FileName);
      Result (Interp, Completion_Code /= ERROR);
   end EvalFile;

   procedure EvalFile (
      Interp   : in Interp_Ptr; 
      FileName : in String) is
   --
      FileName_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (FileName);
      Completion_Code : C.Int;
   begin -- EvalFile
      Completion_Code := EvalFile (Interp, FileName_Ptr);
      C.Strings.Free (FileName_Ptr);
      Result (Interp, Completion_Code /= ERROR);
   end EvalFile;

   function ExprString (
      Interp : in Interp_Ptr;
      strng  : in C.Strings.Chars_Ptr) return C.Strings.Chars_Ptr is
   --
      Completion_Code : C.Int;
   begin -- ExprString
      Completion_Code := ExprString (Interp, strng);
      return Result (Interp, Completion_Code /= ERROR);
   end ExprString;

   function ExprString (
      Interp : in Interp_Ptr;
      strng  : in String) return C.Strings.Chars_Ptr is
   --
      Str_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
      Completion_Code : C.Int;
   begin -- ExprString
      Completion_Code := ExprString (Interp, Str_Ptr);
      C.Strings.Free (Str_Ptr);
      return Result (Interp, Completion_Code /= ERROR);
   end ExprString;

   procedure ExprString (
      Interp : in Interp_Ptr;
      strng  : in C.Strings.Chars_Ptr) is
   --
      Completion_Code : C.Int;
   begin -- ExprString
      Completion_Code := ExprString (Interp, strng);
      Result (Interp, Completion_Code /= ERROR);
   end ExprString;

   procedure ExprString (
      Interp : in Interp_Ptr;
      strng  : in String) is
   --
      Str_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
      Completion_Code : C.Int;
   begin -- ExprString
      Completion_Code := ExprString (Interp, Str_Ptr);
      C.Strings.Free (Str_Ptr);
      Result (Interp, Completion_Code /= ERROR);
   end ExprString;

   function GetVar (
      Interp   : in Interp_Ptr;
      VarName  : in String; 
      Flags    : in C.Int := 0) return C.Strings.Chars_Ptr is
   --
      VarName_Ptr  : C.Strings.Chars_Ptr := C.Strings.New_String (VarName);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- GetVar
      Result_Ptr := GetVar (Interp, VarName_Ptr, Flags);
      C.Strings.Free (VarName_Ptr);
      return Result_Ptr;
   end GetVar;

   function GetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in String; 
      Name2    : in C.Strings.Chars_Ptr; 
      Flags    : in C.Int := 0) return C.Strings.Chars_Ptr is
   --
      Name1_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name1);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- GetVar2
      Result_Ptr := GetVar2 (Interp, Name1_Ptr, Name2, Flags);
      C.Strings.Free (Name1_Ptr);
      return Result_Ptr;
   end GetVar2;

   function GetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr; 
      Name2    : in String; 
      Flags    : in C.Int := 0) return C.Strings.Chars_Ptr is
   --
      Name2_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name2);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- GetVar2
      Result_Ptr := GetVar2 (Interp, Name1, Name2_Ptr, Flags);
      C.Strings.Free (Name2_Ptr);
      return Result_Ptr;
   end GetVar2;

   function GetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in String; 
      Name2    : in String; 
      Flags    : in C.Int := 0) return C.Strings.Chars_Ptr is
   --
      Name1_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name1);
      Name2_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name2);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- GetVar2
      Result_Ptr := GetVar2 (Interp, Name1_Ptr, Name2_Ptr, Flags);
      C.Strings.Free (Name1_Ptr);
      C.Strings.Free (Name2_Ptr);
      return Result_Ptr;
   end GetVar2;

   function GlobalEval (
      Interp : in Interp_Ptr; 
      Script : in C.Strings.Chars_Ptr) return C.Strings.Chars_Ptr is
   --
      Completion_Code : C.Int;
   begin -- GlobalEval
      Completion_Code := GlobalEval (Interp, Script);
      return Result (Interp, Completion_Code /= ERROR);
   end GlobalEval;

   function GlobalEval (
      Interp : in Interp_Ptr; 
      Script : in String) return C.Strings.Chars_Ptr is
   --
      Script_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script);
      Completion_Code : C.Int;
   begin -- GlobalEval
      Completion_Code := GlobalEval (Interp, Script_Ptr);
      C.Strings.Free (Script_Ptr);
      return Result (Interp, Completion_Code /= ERROR);
   end GlobalEval;

   procedure GlobalEval (
      Interp : in Interp_Ptr; 
      Script : in C.Strings.Chars_Ptr) is
   --
      Completion_Code : C.Int;
   begin -- GlobalEval
      Completion_Code := GlobalEval (Interp, Script);
      Result (Interp, Completion_Code /= ERROR);
   end GlobalEval;

   procedure GlobalEval (
      Interp : in Interp_Ptr; 
      Script : in String) is
   --
      Script_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script);
      Completion_Code : C.Int;
   begin -- GlobalEval
      Completion_Code := GlobalEval (Interp, Script_Ptr);
      C.Strings.Free (Script_Ptr);
      Result (Interp, Completion_Code /= ERROR);
   end GlobalEval;

   function PosixError (
      Interp : in Interp_Ptr) return String is
   begin -- PosixError
      return C_Aux.Value (PosixError (Interp));
   end PosixError;

   function RecordAndEval (
      Interp : in Interp_Ptr; 
      Script : in C.Strings.Chars_Ptr;
      Eval   : in BOOLEAN := True) return C.Strings.Chars_Ptr is
   --
      Eval_Int        : C.Int := 0;
      Completion_Code : C.Int;
   begin -- RecordAndEval
      if not Eval then
         Eval_Int := NO_EVAL;
      end if;
      Completion_Code := RecordAndEval (Interp, Script, Eval_Int);
      return Result (Interp, Completion_Code /= ERROR);
   end RecordAndEval;

   function RecordAndEval (
      Interp : in Interp_Ptr; 
      Script : in String;
      Eval   : in BOOLEAN := True) return C.Strings.Chars_Ptr is
   --
      Script_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script);
      Eval_Int        : C.Int := 0;
      Completion_Code : C.Int;
   begin -- RecordAndEval
      if not Eval then
         Eval_Int := NO_EVAL;
      end if;
      Completion_Code := RecordAndEval (Interp, Script_Ptr, Eval_Int);
      C.Strings.Free (Script_Ptr);
      return Result (Interp, Completion_Code /= ERROR);
   end RecordAndEval;

   procedure RecordAndEval (
      Interp : in Interp_Ptr; 
      Script : in C.Strings.Chars_Ptr;
      Eval   : in BOOLEAN := True) is
   --
      Eval_Int        : C.Int := 0;
      Completion_Code : C.Int;
   begin -- RecordAndEval
      if not Eval then
         Eval_Int := NO_EVAL;
      end if;
      Completion_Code := RecordAndEval (Interp, Script, Eval_Int);
      Result (Interp, Completion_Code /= ERROR);
   end RecordAndEval;

   procedure RecordAndEval (
      Interp : in Interp_Ptr; 
      Script : in String;
      Eval   : in BOOLEAN := True) is
   --
      Script_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script);
      Eval_Int        : C.Int := 0;
      Completion_Code : C.Int;
   begin -- RecordAndEval
      if not Eval then
         Eval_Int := NO_EVAL;
      end if;
      Completion_Code := RecordAndEval (Interp, Script_Ptr, Eval_Int);
      C.Strings.Free (Script_Ptr);
      Result (Interp, Completion_Code /= ERROR);
   end RecordAndEval;

   procedure Tcl_SetErrorCode (
      interp          : in Interp_Ptr;
      Field1          : in C.Strings.Chars_Ptr;
      Field2          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field3          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field4          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field5          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field6          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field7          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field8          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field9          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      nullarg         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr);
   pragma Import (C, Tcl_SetErrorCode, "Tcl_SetErrorCode");

   procedure SetErrorCode (
      interp          : in Interp_Ptr;
      Field1          : in C.Strings.Chars_Ptr;
      Field2          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field3          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field4          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field5          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field6          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field7          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field8          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Field9          : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr) is
   begin -- SetErrorCode
      Tcl_SetErrorCode (interp, Field1, Field2, Field3, Field4,
         Field5, Field6, Field7, Field8, Field9);
   end SetErrorCode;

   procedure SetErrorCode (
      Interp : in Interp_Ptr;
      Field1 : in String;
      Field2 : in String := "";
      Field3 : in String := "";
      Field4 : in String := "";
      Field5 : in String := "";
      Field6 : in String := "";
      Field7 : in String := "";
      Field8 : in String := "";
      Field9 : in String := "") is
   --
      Field1_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Field1);
      Field2_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Field2);
      Field3_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Field3);
      Field4_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Field4);
      Field5_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Field5);
      Field6_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Field6);
      Field7_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Field7);
      Field8_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Field8);
      Field9_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Field9);
   begin -- SetErrorCode
      Tcl_SetErrorCode (Interp,
         Field1_Ptr,
         Field2_Ptr,
         Field3_Ptr,
         Field4_Ptr,
         Field5_Ptr,
         Field6_Ptr,
         Field7_Ptr,
         Field8_Ptr,
         Field9_Ptr);
      C.Strings.Free (Field1_Ptr);
      C.Strings.Free (Field2_Ptr);
      C.Strings.Free (Field3_Ptr);
      C.Strings.Free (Field4_Ptr);
      C.Strings.Free (Field5_Ptr);
      C.Strings.Free (Field6_Ptr);
      C.Strings.Free (Field7_Ptr);
      C.Strings.Free (Field8_Ptr);
      C.Strings.Free (Field9_Ptr);
   end SetErrorCode;

   procedure Tcl_SetPanicProc (
      proc            : in C_Aux.Void_Ptr;
      varargs         : in C.Strings.Chars_Ptr;
      nullarg         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr);
   pragma Import (C, Tcl_SetPanicProc, "Tcl_SetPanicProc");

   procedure SetPanicProc (
      proc            : in C_Aux.Void_Ptr;
      varargs         : in C.Strings.Chars_Ptr) is
   begin -- SetPanicProc
      Tcl_SetPanicProc (proc, varargs);
   end SetPanicProc;

   procedure SetResult (
      Interp   : in Interp_Ptr;
      strng    : in String) is
   --
      Str_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
   begin -- SetResult
      SetResult (Interp, Str_Ptr, Tcl.VOLATILE);
      C.Strings.Free (Str_Ptr);
   end SetResult;

   procedure SetVar (
      Interp   : in Interp_Ptr;
      VarName  : in C.Strings.Chars_Ptr;
      NewValue : in C.Strings.Chars_Ptr;
      Flags    : in C.Int := 0) is
   --
      Result_Ptr  : C.Strings.Chars_Ptr;
   begin -- SetVar
      Result_Ptr := SetVar (Interp, VarName, NewValue, Flags);
   end SetVar;

   procedure SetVar (
      Interp   : in Interp_Ptr;
      VarName  : in String; 
      NewValue : in C.Strings.Chars_Ptr;
      Flags    : in C.Int := 0) is
   --
      VarName_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (VarName);
      Result_Ptr  : C.Strings.Chars_Ptr;
   begin -- SetVar
      Result_Ptr := SetVar (Interp, VarName_Ptr, NewValue, Flags);
      C.Strings.Free (VarName_Ptr);
   end SetVar;

   procedure SetVar (
      Interp   : in Interp_Ptr;
      VarName  : in C.Strings.Chars_Ptr; 
      NewValue : in String;
      Flags    : in C.Int := 0) is
   --
      NewValue_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (NewValue);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- SetVar
      Result_Ptr := SetVar (Interp, VarName, NewValue_Ptr, Flags);
      C.Strings.Free (NewValue_Ptr);
   end SetVar;

   procedure SetVar (
      Interp   : in Interp_Ptr;
      VarName  : in String; 
      NewValue : in String;
      Flags    : in C.Int := 0) is
   --
      VarName_Ptr  : C.Strings.Chars_Ptr := C.Strings.New_String (VarName);
      NewValue_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (NewValue);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- SetVar
      Result_Ptr := SetVar (Interp, VarName_Ptr, NewValue_Ptr, Flags);
      C.Strings.Free (VarName_Ptr);
      C.Strings.Free (NewValue_Ptr);
   end SetVar;

   procedure SetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr;
      Name2    : in C.Strings.Chars_Ptr;
      NewValue : in C.Strings.Chars_Ptr;
      Flags    : in C.Int := 0) is
   --
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- SetVar2
      Result_Ptr := SetVar2 (Interp, Name1, Name2, NewValue, Flags);
   end SetVar2;

   procedure SetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr; 
      Name2    : in String; 
      NewValue : in C.Strings.Chars_Ptr;
      Flags    : in C.Int := 0) is
   --
      Name2_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name2);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- SetVar2
      Result_Ptr := SetVar2 (Interp, Name1, Name2_Ptr, NewValue, Flags);
      C.Strings.Free (Name2_Ptr);
   end SetVar2;

   procedure SetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr; 
      Name2    : in C.Strings.Chars_Ptr; 
      NewValue : in String;
      Flags    : in C.Int := 0) is
   --
      NewValue_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (NewValue);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- SetVar2
      Result_Ptr := SetVar2 (Interp, Name1, Name2, NewValue_Ptr, Flags);
      C.Strings.Free (NewValue_Ptr);
   end SetVar2;

   procedure SetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr;
      Name2    : in String; 
      NewValue : in String;
      Flags    : in C.Int := 0) is
   --
      Name2_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name2);
      NewValue_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (NewValue);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- SetVar2
      Result_Ptr := SetVar2 (Interp, Name1, Name2_Ptr, NewValue_Ptr, Flags);
      C.Strings.Free (Name2_Ptr);
      C.Strings.Free (NewValue_Ptr);
   end SetVar2;

   procedure SetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in String; 
      Name2    : in C.Strings.Chars_Ptr; 
      NewValue : in C.Strings.Chars_Ptr;
      Flags    : in C.Int := 0) is
   --
      Name1_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name1);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- SetVar2
      Result_Ptr := SetVar2 (Interp, Name1_Ptr, Name2, NewValue, Flags);
      C.Strings.Free (Name1_Ptr);
   end SetVar2;

   procedure SetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in String; 
      Name2    : in String; 
      NewValue : in C.Strings.Chars_Ptr;
      Flags    : in C.Int := 0) is
   --
      Name1_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name1);
      Name2_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name2);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- SetVar2
      Result_Ptr := SetVar2 (Interp, Name1_Ptr, Name2_Ptr, NewValue, Flags);
      C.Strings.Free (Name1_Ptr);
      C.Strings.Free (Name2_Ptr);
   end SetVar2;

   procedure SetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in String;
      Name2    : in C.Strings.Chars_Ptr; 
      NewValue : in String;
      Flags    : in C.Int := 0) is
   --
      Name1_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name1);
      NewValue_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (NewValue);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- SetVar2
      Result_Ptr := SetVar2 (Interp, Name1_Ptr, Name2, NewValue_Ptr, Flags);
      C.Strings.Free (Name1_Ptr);
      C.Strings.Free (NewValue_Ptr);
   end SetVar2;

   procedure SetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in String;
      Name2    : in String; 
      NewValue : in String;
      Flags    : in C.Int := 0) is
   --
      Name1_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name1);
      Name2_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name2);
      NewValue_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (NewValue);
      Result_Ptr   : C.Strings.Chars_Ptr;
   begin -- SetVar2
      Result_Ptr := SetVar2 (Interp, Name1_Ptr, Name2_Ptr, NewValue_Ptr, Flags);
      C.Strings.Free (Name1_Ptr);
      C.Strings.Free (Name2_Ptr);
      C.Strings.Free (NewValue_Ptr);
   end SetVar2;

   function SignalId (
      Sig : in C.Int) return String is
   begin -- SignalId
      return C_Aux.Value (SignalId (Sig));
   end SignalId;

   function SignalMsg (
      Sig : in C.Int) return String is
   begin -- SignalMsg
      return C_Aux.Value (SignalMsg (Sig));
   end SignalMsg;

   procedure SplitList (
      Interp  : in     Interp_Ptr;
      List    : in     C.Strings.Chars_Ptr;
      Argc    :    out C.Int;
      ArgvPtr :    out C_Aux.Chars_Ptr_Ptr) is
   --
      Local_Argc      : aliased C.Int;
      Local_Argv      : aliased C_Aux.Chars_Ptr_Ptr;
      Completion_Code : C.Int;
   begin -- SplitList
      Completion_Code := SplitList (
         Interp, List, Local_Argc'unchecked_access,
	 Local_Argv'unchecked_access);
      ArgvPtr := Local_Argv;
      Result (Interp, Completion_Code /= ERROR);
   end SplitList;

   procedure SplitList (
      Interp  : in     Interp_Ptr;
      List    : in     String;
      Argc    :    out C.Int;
      ArgvPtr :    out C_Aux.Chars_Ptr_Ptr) is
   --
      List_Ptr        : C.Strings.Chars_Ptr := C.Strings.New_String (List);
      Local_Argc      : aliased C.Int;
      Local_Argv      : aliased C_Aux.Chars_Ptr_Ptr;
      Completion_Code : C.Int;
   begin -- SplitList
      Completion_Code := SplitList (
         Interp, List_Ptr, Local_Argc'unchecked_access,
	 Local_Argv'unchecked_access);
      ArgvPtr := Local_Argv;
      C.Strings.Free (List_Ptr);
      Result (Interp, Completion_Code /= ERROR);
   end SplitList;

   function StringMatch (
      strng   : in C.Strings.Chars_Ptr;
      Pattern : in C.Strings.Chars_Ptr) return Boolean is
   begin -- StringMatch
      return StringMatch (strng, Pattern) = 1;
   end StringMatch;

   function StringMatch (
      strng   : in String;
      Pattern : in C.Strings.Chars_Ptr) return Boolean is
   --
      Str_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
      Completion_Code : C.Int;
   begin -- StringMatch
      Completion_Code := StringMatch (Str_Ptr, Pattern);
      C.Strings.Free (Str_Ptr);
      return Completion_Code = 1;
   end StringMatch;

   function StringMatch (
      strng   : in C.Strings.Chars_Ptr;
      Pattern : in String) return Boolean is
   --
      Pattern_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Pattern);
      Completion_Code : C.Int;
   begin -- StringMatch
      Completion_Code := StringMatch (strng, Pattern_Ptr);
      C.Strings.Free (Pattern_Ptr);
      return Completion_Code = 1;
   end StringMatch;

   function StringMatch (
      strng   : in String;
      Pattern : in String) return Boolean is
   --
      Str_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (strng);
      Pattern_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Pattern);
      Completion_Code : C.Int;
   begin -- StringMatch
      Completion_Code := StringMatch (Str_Ptr, Pattern_Ptr);
      C.Strings.Free (Str_Ptr);
      C.Strings.Free (Pattern_Ptr);
      return Completion_Code = 1;
   end StringMatch;

   function TildeSubst (
      Interp  : in Interp_Ptr;
      Name    : in String;
      TempPtr : in DString_Ptr) return C.Strings.Chars_Ptr is
   --
      Name_Ptr   : C.Strings.Chars_Ptr := C.Strings.New_String (Name);
      Result_Ptr : C.Strings.Chars_Ptr;
   begin -- TildeSubst
      Result_Ptr := TildeSubst (Interp, Name_Ptr, TempPtr);
      C.Strings.Free (Name_Ptr);
      return Result_Ptr;
   end TildeSubst;

   procedure TildeSubst (
      Interp  : in Interp_Ptr;
      Name    : in C.Strings.Chars_Ptr;
      TempPtr : in DString_Ptr) is
   --
      Result_Ptr : C.Strings.Chars_Ptr;
   begin -- TildeSubst
      Result_Ptr := TildeSubst (Interp, Name, TempPtr);
   end TildeSubst;

   procedure TildeSubst (
      Interp  : in Interp_Ptr;
      Name    : in String;
      TempPtr : in DString_Ptr) is
   --
      Name_Ptr   : C.Strings.Chars_Ptr := C.Strings.New_String (Name);
      Result_Ptr : C.Strings.Chars_Ptr;
   begin -- TildeSubst
      Result_Ptr := TildeSubst (Interp, Name_Ptr, TempPtr);
      C.Strings.Free (Name_Ptr);
   end TildeSubst;

   procedure UnsetVar (
      Interp   : in Interp_Ptr;
      VarName  : in C.Strings.Chars_Ptr; 
      Flags    : in C.Int := 0) is
   --
      Completion_Code : C.Int;
   begin -- UnsetVar
      Completion_Code := UnsetVar (Interp, VarName, Flags);
      if Completion_Code = ERROR then
         -- raise Tcl_Error;
	 -- not yet implemented in GNAT:
	 Ada.Exceptions.Raise_Exception (Tcl_Error'Identity, SResult (Interp));
      end if; 
   end UnsetVar;

   procedure UnsetVar (
      Interp   : in Interp_Ptr;
      VarName  : in String; 
      Flags    : in C.Int := 0) is
   --
      VarName_Ptr  : C.Strings.Chars_Ptr := C.Strings.New_String (VarName);
      Completion_Code : C.Int;
   begin -- UnsetVar
      Completion_Code := UnsetVar (Interp, VarName_Ptr, Flags);
      C.Strings.Free (VarName_Ptr);
      if Completion_Code = ERROR then
         -- raise Tcl_Error;
	 -- not yet implemented in GNAT:
	 Ada.Exceptions.Raise_Exception (Tcl_Error'Identity, SResult (Interp));
      end if; 
   end UnsetVar;

   procedure UnsetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr; 
      Name2    : in C.Strings.Chars_Ptr; 
      Flags    : in C.Int := 0) is
   --
      Completion_Code : C.Int;
   begin -- UnsetVar2
      Completion_Code := UnsetVar2 (Interp, Name1, Name2, Flags);
      if Completion_Code = ERROR then
         -- raise Tcl_Error;
	 -- not yet implemented in GNAT:
	 Ada.Exceptions.Raise_Exception (Tcl_Error'Identity, SResult (Interp));
      end if; 
   end UnsetVar2;

   procedure UnsetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in String; 
      Name2    : in C.Strings.Chars_Ptr; 
      Flags    : in C.Int := 0) is
   --
      Name1_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name1);
      Completion_Code : C.Int;
   begin -- UnsetVar2
      Completion_Code := UnsetVar2 (Interp, Name1_Ptr, Name2, Flags);
      C.Strings.Free (Name1_Ptr);
      if Completion_Code = ERROR then
         -- raise Tcl_Error;
	 -- not yet implemented in GNAT:
	 Ada.Exceptions.Raise_Exception (Tcl_Error'Identity, SResult (Interp));
      end if; 
   end UnsetVar2;

   procedure UnsetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in C.Strings.Chars_Ptr; 
      Name2    : in String; 
      Flags    : in C.Int := 0) is
   --
      Name2_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name2);
      Completion_Code : C.Int;
   begin -- UnsetVar2
      Completion_Code := UnsetVar2 (Interp, Name1, Name2_Ptr, Flags);
      C.Strings.Free (Name2_Ptr);
      if Completion_Code = ERROR then
         -- raise Tcl_Error;
	 -- not yet implemented in GNAT:
	 Ada.Exceptions.Raise_Exception (Tcl_Error'Identity, SResult (Interp));
      end if; 
   end UnsetVar2;

   procedure UnsetVar2 (
      Interp   : in Interp_Ptr;
      Name1    : in String; 
      Name2    : in String; 
      Flags    : in C.Int := 0) is
   --
      Name1_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name1);
      Name2_Ptr    : C.Strings.Chars_Ptr := C.Strings.New_String (Name2);
      Completion_Code : C.Int;
   begin -- UnsetVar2
      Completion_Code := UnsetVar2 (Interp, Name1_Ptr, Name2_Ptr, Flags);
      C.Strings.Free (Name1_Ptr);
      C.Strings.Free (Name2_Ptr);
      if Completion_Code = ERROR then
         -- raise Tcl_Error;
	 -- not yet implemented in GNAT:
	 Ada.Exceptions.Raise_Exception (Tcl_Error'Identity, SResult (Interp));
      end if; 
   end UnsetVar2;

   function Tcl_VarEval (
      interp          : in Interp_Ptr;
      Script1 : in C.Strings.Chars_Ptr;
      Script2 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script3 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script4 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script5 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script6 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script7 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script8 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script9 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      nullarg         : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr)
         return C.Int;
   pragma Import (C, Tcl_VarEval, "Tcl_VarEval");

   function VarEval (
      interp          : in Interp_Ptr;
      Script1 : in C.Strings.Chars_Ptr;
      Script2 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script3 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script4 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script5 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script6 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script7 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script8 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script9 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr) return C.Int is
   begin -- VarEval
      return Tcl_VarEval (interp, Script1, Script2, Script3, Script4,
         Script5, Script6, Script7, Script8, Script9);
   end VarEval;

   function VarEval (
      Interp  : in Interp_Ptr;
      Script1 : in C.Strings.Chars_Ptr;
      Script2 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script3 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script4 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script5 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script6 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script7 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script8 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script9 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr) return C.Strings.Chars_Ptr is
   --
      Completion_Code : C.Int;
   begin -- VarEval
      Completion_Code := Tcl_VarEval (Interp,
         Script1,
         Script2,
         Script3,
         Script4,
         Script5,
         Script6,
         Script7,
         Script8,
         Script9);
      return Result (Interp, Completion_Code /= ERROR);
   end VarEval;

   function VarEval (
      Interp  : in Interp_Ptr;
      Script1 : in String;
      Script2 : in String := "";
      Script3 : in String := "";
      Script4 : in String := "";
      Script5 : in String := "";
      Script6 : in String := "";
      Script7 : in String := "";
      Script8 : in String := "";
      Script9 : in String := "") return C.Strings.Chars_Ptr is
   --
      Script1_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script1);
      Script2_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script2);
      Script3_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script3);
      Script4_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script4);
      Script5_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script5);
      Script6_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script6);
      Script7_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script7);
      Script8_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script8);
      Script9_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script9);
      Completion_Code : C.Int;
   begin -- VarEval
      Completion_Code := Tcl_VarEval (Interp,
         Script1_Ptr,
         Script2_Ptr,
         Script3_Ptr,
         Script4_Ptr,
         Script5_Ptr,
         Script6_Ptr,
         Script7_Ptr,
         Script8_Ptr,
         Script9_Ptr);
      C.Strings.Free (Script1_Ptr);
      C.Strings.Free (Script2_Ptr);
      C.Strings.Free (Script3_Ptr);
      C.Strings.Free (Script4_Ptr);
      C.Strings.Free (Script5_Ptr);
      C.Strings.Free (Script6_Ptr);
      C.Strings.Free (Script7_Ptr);
      C.Strings.Free (Script8_Ptr);
      C.Strings.Free (Script9_Ptr);
      return Result (Interp, Completion_Code /= ERROR);
   end VarEval;

   procedure VarEval (
      Interp  : in Interp_Ptr;
      Script1 : in C.Strings.Chars_Ptr;
      Script2 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script3 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script4 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script5 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script6 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script7 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script8 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr;
      Script9 : in C.Strings.Chars_Ptr := C.Strings.Null_Ptr) is
   --
      Completion_Code : C.Int;
   begin -- VarEval
      Completion_Code := Tcl_VarEval (Interp,
         Script1,
         Script2,
         Script3,
         Script4,
         Script5,
         Script6,
         Script7,
         Script8,
         Script9);
      Result (Interp, Completion_Code /= ERROR);
   end VarEval;

   procedure VarEval (
      Interp  : in Interp_Ptr;
      Script1 : in String;
      Script2 : in String := "";
      Script3 : in String := "";
      Script4 : in String := "";
      Script5 : in String := "";
      Script6 : in String := "";
      Script7 : in String := "";
      Script8 : in String := "";
      Script9 : in String := "") is
   --
      Script1_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script1);
      Script2_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script2);
      Script3_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script3);
      Script4_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script4);
      Script5_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script5);
      Script6_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script6);
      Script7_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script7);
      Script8_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script8);
      Script9_Ptr : C.Strings.Chars_Ptr := C.Strings.New_String (Script9);
      Completion_Code : C.Int;
   begin -- VarEval
      Completion_Code := Tcl_VarEval (Interp,
         Script1_Ptr,
         Script2_Ptr,
         Script3_Ptr,
         Script4_Ptr,
         Script5_Ptr,
         Script6_Ptr,
         Script7_Ptr,
         Script8_Ptr,
         Script9_Ptr);
      C.Strings.Free (Script1_Ptr);
      C.Strings.Free (Script2_Ptr);
      C.Strings.Free (Script3_Ptr);
      C.Strings.Free (Script4_Ptr);
      C.Strings.Free (Script5_Ptr);
      C.Strings.Free (Script6_Ptr);
      C.Strings.Free (Script7_Ptr);
      C.Strings.Free (Script8_Ptr);
      C.Strings.Free (Script9_Ptr);
      Result (Interp, Completion_Code /= ERROR);
   end VarEval;

   procedure Main_Exit (
      Interp  : in Interp_Ptr;
      Code    : in C.Int) is
   begin -- Main_Exit
      Eval (Interp, "exit" & C.Int'image (Code));
   end Main_Exit;

end Tcl;
