--------------------------------------------------------------------
--
-- testapp.adb --
--
-- Copyright (c) 1995 Terry J. Westley
--
-- See the file "license.terms" for information on usage and redistribution
-- of this file, and for a DISCLAIMER OF ALL WARRANTIES.
--
--
--------------------------------------------------------------------

with Ada.Strings.Fixed;
with C_Aux;
with Text_IO;
with Unchecked_Deallocation;

package body TestApp is

   function "+" (Left, Right : in C.Int) return C.Int   renames C."+";
   function "=" (Left, Right : in C.Int) return Boolean renames C."=";

   package CreateCommands is new Tcl.Generic_CreateCommands (Integer);

   type Counter_Rec is
      record
         Value : Integer := 0;
      end record;
   type Counter_Ptr is access Counter_Rec;

   procedure Free_Counter is new Unchecked_Deallocation (
      Object => Counter_Rec,
      Name   => Counter_Ptr);

   package CounterCommands is new Tcl.Generic_CreateCommands (Counter_Ptr);

   Counter_Id : Natural := 0;

   procedure Put_Argv (
      Msg  : in String;
      Argc : in C.Int;
      Argv : in C_Aux.Chars_Ptr_Ptr) is
   begin -- Put_Argv
      Text_IO.Put_Line (Msg & ": c=" & C.Int'image (Argc) &
         "  v=" & C_Aux.Value (Tcl.Merge (Argc, Argv)));
   end Put_Argv;

   function EqCmd (
      ClientData    : in Integer;
      Interp        : in Tcl.Interp_Ptr;
      Argc	    : in C.Int;
      Argv	    : in C_Aux.Chars_Ptr_Ptr) return C.Int is
   -- From Section 30.2 of _Tcl_and_the_Tk_Toolkit_ by John Ousterhout.
   -- Compares two arguments for equality using string comparision.
   -- Returns 1 if equal, 0 if not.
      Vector : C_Aux.Arg_Vector (1..Argc);
   begin -- EqCmd
      if Argc /= 3 then
         Tcl.SetResult (Interp, "wrong # args");
         return Tcl.ERROR;
      end if;
      Vector := C_Aux.Argv.Value (Argv, C.Ptrdiff_t(Argc));
      if C_Aux."=" (Vector(Vector'first+1), Vector(Vector'first+2)) then
         Tcl.SetResult (Interp, "1");
      else
         Tcl.SetResult (Interp, "0");
      end if;
      return Tcl.OK;
   end EqCmd;
   pragma Convention (C, EqCmd);

   function ConcatCmd (
      ClientData    : in Integer;
      Interp        : in Tcl.Interp_Ptr;
      Argc	    : in C.Int;
      Argv	    : in C_Aux.Chars_Ptr_Ptr) return C.Int is
   -- From Section 30.4 of _Tcl_and_the_Tk_Toolkit_ by John Ousterhout.
   -- This is a simple implementation of the concat command using AppendResult.
      Vector : C_Aux.Arg_Vector (1..Argc);
   begin -- ConcatCmd
      if Argc = 1 then
         return Tcl.OK;
      end if;
      Vector := C_Aux.Argv.Value (Argv, C.Ptrdiff_t(Argc));
      Tcl.AppendResult (Interp, Vector(2));
      for i in 3..Argc loop
         Tcl.AppendResult (Interp, " ");
	 Tcl.AppendResult (Interp, Vector(i));
      end loop;
      return Tcl.OK;
   end ConcatCmd;
   pragma Convention (C, ConcatCmd);

   function ListCmd (
      ClientData    : in Integer;
      Interp        : in Tcl.Interp_Ptr;
      Argc	    : in C.Int;
      Argv	    : in C_Aux.Chars_Ptr_Ptr) return C.Int is
   -- From Section 30.4 of _Tcl_and_the_Tk_Toolkit_ by John Ousterhout.
   -- This is a simple implementation of the list command using AppendElement.
      Vector : C_Aux.Arg_Vector (1..Argc);
   begin -- ListCmd
      Vector := C_Aux.Argv.Value (Argv, C.Ptrdiff_t(Argc));
      for i in 2..Argc loop
	 Tcl.AppendElement (Interp, Vector(i));
      end loop;
      return Tcl.OK;
   end ListCmd;
   pragma Convention (C, ListCmd);

   function "&" (Left : in String; Right : in Integer) return String is
   begin -- "&"
      return Left & Ada.Strings.Fixed.Trim (
      	 Integer'image (Right), Ada.Strings.Left);
   end "&";

   procedure DeleteCounter (
      Counter : in out Counter_Ptr) is
   begin -- DeleteCounter
      Free_Counter (Counter);
   end DeleteCounter;
   pragma Convention (C, DeleteCounter);

   function ObjectCmd (
      Counter       : in Counter_Ptr;
      Interp        : in Tcl.Interp_Ptr;
      Argc	    : in C.Int;
      Argv	    : in C_Aux.Chars_Ptr_Ptr) return C.Int is
   -- From Section 30.5 of _Tcl_and_the_Tk_Toolkit_ by John Ousterhout.
   -- This is an Ada implementation of the counter object command.  It
   -- demonstrates the use of Client Data and deletion callbacks.
      Vector : C_Aux.Arg_Vector (1..Argc);
   begin -- ObjectCmd
      if Argc /= 2 then
         Tcl.SetResult (Interp, "wrong # args");
         return Tcl.ERROR;
      end if;
      Vector := C_Aux.Argv.Value (Argv, C.Ptrdiff_t(Argc));
      declare
         Command : constant String := C_Aux.Value (Vector(2));
      begin
      	 if Command = "get" then
            Tcl.SetResult (Interp, Integer'image (Counter.Value));
	 elsif Command = "next" then
	    Counter.Value := Counter.Value + 1;
	 else
	    Tcl.AppendResult (Interp, "bad counter command """ &
	       Command & """: should be get or next");
	    return Tcl.ERROR;
	 end if;
      end;
      return Tcl.OK;
   end ObjectCmd;
   pragma Convention (C, ObjectCmd);

   function CounterCmd (
      ClientData    : in Counter_Ptr;
      Interp        : in Tcl.Interp_Ptr;
      Argc	    : in C.Int;
      Argv	    : in C_Aux.Chars_Ptr_Ptr) return C.Int is
   -- From Section 30.5 of _Tcl_and_the_Tk_Toolkit_ by John Ousterhout.
   -- This is an Ada implementation of the Object command.  It demonstrates
   -- the use of Client Data and deletion callbacks.
      Counter : Counter_Ptr;
   begin -- CounterCmd
      if Argc /= 1 then
         Tcl.SetResult (Interp, "wrong # args");
         return Tcl.ERROR;
      end if;
      Counter := new Counter_Rec;
      Tcl.SetResult (Interp, "ctr" & Counter_Id);
      Counter_Id := Counter_Id + 1;
      CounterCommands.CreateCommand (
         interp, Tcl.SResult (Interp), ObjectCmd'access,
	 Counter, DeleteCounter'access);
      return Tcl.OK;
   end CounterCmd;
   pragma Convention (C, CounterCmd);

   function SumCmd (
      ClientData    : in Integer;
      Interp        : in Tcl.Interp_Ptr;
      Argc	    : in C.Int;
      Argv	    : in C_Aux.Chars_Ptr_Ptr) return C.Int is
   -- From Section 32.1 of _Tcl_and_the_Tk_Toolkit_ by John Ousterhout.
   -- Adds its two integer arguments.
      Vector : C_Aux.Arg_Vector (1..Argc);
      Left, Right : aliased C.Int;
   begin -- SumCmd
      if Argc /= 3 then
         Tcl.SetResult (Interp, "wrong # args");
         return Tcl.ERROR;
      end if;
      Vector := C_Aux.Argv.Value (Argv, C.Ptrdiff_t(Argc));
      begin
      	 Left  := C.Int'value (C_Aux.Value (Vector(2)));
      exception
         when others =>
	    if Tcl.GetInt (Interp, Vector(2), Left'unchecked_access) /= Tcl.OK then
	       return Tcl.ERROR;
            end if;
      end;
      begin
      	 Right  := C.Int'value (C_Aux.Value (Vector(3)));
      exception
         when others =>
	    if Tcl.GetInt (Interp, Vector(3), Right'unchecked_access) /= Tcl.OK then
	       return Tcl.ERROR;
            end if;
      end;
      Tcl.SetResult (Interp, C.Int'image (Left + Right));
      return Tcl.OK;
   end SumCmd;
   pragma Convention (C, SumCmd);

   function ExprCmd (
      ClientData    : in Integer;
      Interp        : in Tcl.Interp_Ptr;
      Argc	    : in C.Int;
      Argv	    : in C_Aux.Chars_Ptr_Ptr) return C.Int is
   -- From Section 32.2 of _Tcl_and_the_Tk_Toolkit_ by John Ousterhout.
   -- This is a simple implementation of the expr command using ExprString.
      Vector : C_Aux.Arg_Vector (1..Argc);
   begin -- ExprCmd
      if Argc /= 2 then
         Tcl.SetResult (Interp, "wrong # args");
         return Tcl.ERROR;
      end if;
      Vector := C_Aux.Argv.Value (Argv, C.Ptrdiff_t(Argc));
      return Tcl.ExprString (Interp, Vector(2));
   end ExprCmd;
   pragma Convention (C, ExprCmd);

   function Init (
      Interp : in Tcl.Interp_Ptr) return C.Int is

   begin -- Init

      if Tcl.Init(interp) = Tcl.ERROR then
	 return Tcl.ERROR;
      end if;

      -- Call the init procedures for included packages.  Each call should
      -- look like this:
      -- 
      -- if Mod.Init(interp) = Tcl.ERROR then
      --    return Tcl.ERROR;
      -- end if;
      --
      -- where "Mod" is the name of the module.

      -- Call CreateCommand for application-specific commands, if
      -- they weren't already created by the init procedures called above.

      CreateCommands.CreateCommand (
         interp, "eq", EqCmd'access, 0, NULL);

      CreateCommands.CreateCommand (
         interp, "concat", ConcatCmd'access, 0, NULL);

      CreateCommands.CreateCommand (
         interp, "list", ListCmd'access, 0, NULL);

      CounterCommands.CreateCommand (
         interp, "counter", CounterCmd'access, NULL, NULL);

      CreateCommands.CreateCommand (
         interp, "sum", SumCmd'access, 0, NULL);

      CreateCommands.CreateCommand (
         interp, "simple_expr", ExprCmd'access, 0, NULL);

      -- Specify a user-specific startup file to invoke if the application
      -- is run interactively.  Typically the startup file is "~/.apprc"
      -- where "app" is the name of the application.  If this line is deleted
      -- then no user-specific startup file will be run under any conditions.

      Tcl.SetVar(interp, "tcl_rcFileName", "~/.tashrc", Tcl.GLOBAL_ONLY);
      return Tcl.OK;

   end Init;

end TestApp;
