--------------------------------------------------------------------
--
-- c_aux.adb --
--
-- Copyright (c) 1995 Terry J. Westley
--
-- See the file "license.terms" for information on usage and redistribution
-- of this file, and for a DISCLAIMER OF ALL WARRANTIES.
--
--
--------------------------------------------------------------------

with Ada.Strings.Fixed;
with Ada.Command_Line;
with Unchecked_Deallocation;

package body C_Aux is

   package ASF renames Ada.Strings.Fixed;

   function "+" (Left, Right : in C.Size_t) return C.Size_t renames C."+";
   function "+" (Left, Right : in C.Int)    return C.Int    renames C."+";

   procedure Get_Argv (argv : out Arg_Vector) is
   begin -- Get_Argv
      Argv(Argv'first) := CS.New_String (Ada.Command_Line.Command_Name);
      for i in 1..C.Int(Ada.Command_Line.Argument_Count) loop
         Argv(Argv'first+i) := CS.New_String (
	    Ada.Command_Line.Argument (Integer(i)));
      end loop;
   end Get_Argv;

   procedure Free (Argv : in out Arg_Vector) is
   begin -- Free
      for i in Argv'range loop
         CS.Free (Argv (i));
      end loop;
   end Free;

   function Value (Item : in CS.Chars_Ptr) return String is
   begin -- Value
      -- We use fully qualified C.Strings."=" to avoid recursive
      -- call that would result from calling C_Aux."=".
      if CS."=" (Item, CS.Null_Ptr) then
	 return "";
      else
	 return CS.Value (Item);
      end if;
   end Value;

   function "&" (Left, Right : in CS.Chars_Ptr) return String is
   begin -- "&"
      return Value (Left) & Value (Right);
   end "&";

   function "&" (Left : in CS.Chars_Ptr; Right : String) return String is
   begin -- "&"
      return Value (Left) & Right;
   end "&";

   function "&" (Left : String; Right : in CS.Chars_Ptr) return String is
   begin -- "&"
      return Left & Value (Right);
   end "&";

   ---------
   -- "=" --
   ---------

   function "=" (Left, Right : in CS.Chars_Ptr) return Boolean is
   begin -- "="
      return Value (Left) = Value (Right);
   end "=";

   function "=" (Left : in CS.Chars_Ptr; Right : in String) return  Boolean is
   begin -- "="
      return Value (Left) = Right;
   end "=";

   function "=" (Left : in String; Right : in CS.Chars_Ptr) return  Boolean is
   begin -- "="
      return Left = Value (Right);
   end "=";

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : in CS.Chars_Ptr) return Boolean is
   begin -- "<"
      return Value (Left) = Value (Right);
   end "<";

   function "<" (Left : in CS.Chars_Ptr; Right : in String) return Boolean is
   begin -- "<"
      return Value (Left) < Right;
   end "<";

   function "<"
     (Left  : in String;
      Right : in CS.Chars_Ptr)
      return  Boolean
   is
   begin -- "<"
      return Left < Value (Right);
   end "<";

   ----------
   -- "<=" --
   ----------

   function "<=" (Left, Right : in CS.Chars_Ptr) return Boolean is
   begin -- "<="
      return Value (Left) <= Value (Right);
   end "<=";

   function "<="
     (Left  : in CS.Chars_Ptr;
      Right : in String)
      return  Boolean
   is
   begin -- "<="
      return Value (Left) <= Right;
   end "<=";

   function "<="
     (Left  : in String;
      Right : in CS.Chars_Ptr)
      return  Boolean
   is
   begin -- "<="
      return Left <= Value (Right);
   end "<=";

   ---------
   -- ">" --
   ---------

   function ">"  (Left, Right : in CS.Chars_Ptr) return Boolean is
   begin -- ">"
      return Value (Left) > Value (Right);
   end ">";

   function ">"
     (Left  : in CS.Chars_Ptr;
      Right : in String)
      return  Boolean
   is
   begin -- ">"
      return Value (Left) > Right;
   end ">";

   function ">"
     (Left  : in String;
      Right : in CS.Chars_Ptr)
      return  Boolean
   is
   begin -- ">"
      return Left > Value (Right);
   end ">";

   ----------
   -- ">=" --
   ----------

   function ">=" (Left, Right : in CS.Chars_Ptr) return Boolean is
   begin -- ">="
      return Value (Left) >= Value (Right);
   end ">=";

   function ">="
     (Left  : in CS.Chars_Ptr;
      Right : in String)
      return  Boolean
   is
   begin -- ">="
      return Value (Left) >= Right;
   end ">=";

   function ">="
     (Left  : in String;
      Right : in CS.Chars_Ptr)
      return  Boolean
   is
   begin -- ">="
      return Left >= Value (Right);
   end ">=";

   ---------
   -- "*" --
   ---------

   function "*"
     (Left  : Natural;
      Right : Character)
      return  CS.Chars_Ptr
   is
      Result : CS.char_array_access :=
         new C.char_array (1..C.Size_t (Left+1));
   begin -- "*"
      Result.all := C.To_C (ASF."*" (Left, Right), Append_Nul => True);
      return CS.To_Chars_Ptr (Result);
   end "*";

   function "*"
     (Left  : Natural;
      Right : String)
     return   CS.Chars_Ptr
   is
      Result : CS.char_array_access :=
         new C.char_array (1..C.Size_t (Left * Right'length + 1));
   begin -- "*"
      Result.all := C.To_C (ASF."*" (Left, Right), Append_Nul => True);
      return CS.To_Chars_Ptr (Result);
   end "*";

   function "*"
     (Left  : Natural;
      Right : CS.Chars_Ptr)
      return  CS.Chars_Ptr
   is
      Result : CS.char_array_access :=
         new C.char_array (1..C.Size_t (Left * Length (Right) + 1));
   begin -- "*"
      Result.all := C.To_C (ASF."*" (Left, Value (Right)), Append_Nul => True);
      return CS.To_Chars_Ptr (Result);
   end "*";

   -----------
   -- Count --
   -----------

   function Count
     (Source   : CS.Chars_Ptr;
      Pattern  : String;
      Mapping  : Ada.Strings.Maps.Character_Mapping :=
                    Ada.Strings.Maps.Identity)
      return     Natural
   is
   begin -- Count
      return Ada.Strings.Fixed.Count (Value (Source), Pattern, Mapping);
   end Count;

   function Count
     (Source   : in CS.Chars_Ptr;
      Pattern  : in String;
      Mapping  : in Ada.Strings.Maps.Character_Mapping_Function)
      return     Natural
   is
   begin -- Count
      return Ada.Strings.Fixed.Count (Value (Source), Pattern, Mapping);
   end Count;

   function Count
     (Source   : CS.Chars_Ptr;
      Set      : Ada.Strings.Maps.Character_Set)
      return     Natural
   is
   begin -- Count
      return Ada.Strings.Fixed.Count (Value (Source), Set);
   end Count;

   ------------
   -- Delete --
   ------------

   function Delete
     (Source  : CS.Chars_Ptr;
      From    : Positive;
      Through : Natural)
      return    CS.Chars_Ptr
   is
   begin -- Delete
      return
        CS.New_String
          (Ada.Strings.Fixed.Delete (Value (Source), From, Through));
   end Delete;

   procedure Delete
     (Source  : in out CS.Chars_Ptr;
      From    : in Positive;
      Through : in Natural)
   is
      Temp : CS.Chars_Ptr := Source;
   begin -- Delete
      Source := CS.New_String
        (Ada.Strings.Fixed.Delete (Value (Source), From, Through));
      CS.Free (Temp);
   end Delete;

   -------------
   -- Element --
   -------------

   function Element
     (Source : CS.Chars_Ptr;
      Index  : Positive)
      return   Character
   is
   begin -- Element
      if Index <= Length (Source) then
         return Value (Source) (Index);
      else
         raise Ada.Strings.Index_Error;
      end if;
   end Element;

   ----------------
   -- Find_Token --
   ----------------

   procedure Find_Token
     (Source : CS.Chars_Ptr;
      Set    : Ada.Strings.Maps.Character_Set;
      Test   : Ada.Strings.Membership;
      First  : out Positive;
      Last   : out Natural)
   is
   begin -- Find_Token
      Ada.Strings.Fixed.Find_Token (Value (Source), Set, Test, First, Last);
   end Find_Token;

   ----------
   -- Head --
   ----------

   function Head
     (Source : CS.Chars_Ptr;
      Count  : Natural;
      Pad    : Character := Ada.Strings.Space)
      return   CS.Chars_Ptr
   is
   begin -- Head
      return
        CS.New_String (Ada.Strings.Fixed.Head (Value (Source), Count, Pad));
   end Head;

   procedure Head
     (Source : in out CS.Chars_Ptr;
      Count  : in Natural;
      Pad    : in Character := Ada.Strings.Space)
   is
      Temp : CS.Chars_Ptr := Source;
   begin -- Head
      Source := CS.New_String
        (Ada.Strings.Fixed.Head (Value (Source), Count, Pad));
   end Head;

   -----------
   -- Index --
   -----------

   function Index
     (Source   : CS.Chars_Ptr;
      Pattern  : String;
      Going    : Ada.Strings.Direction := Ada.Strings.Forward;
      Mapping  : Ada.Strings.Maps.Character_Mapping := Ada.Strings.Maps.Identity)
      return     Natural
   is
   begin -- Index
      return Ada.Strings.Fixed.Index (Value (Source), Pattern, Going, Mapping);
   end Index;

   function Index
     (Source   : in CS.Chars_Ptr;
      Pattern  : in String;
      Going    : in Ada.Strings.Direction := Ada.Strings.Forward;
      Mapping  : in Ada.Strings.Maps.Character_Mapping_Function)
      return Natural
   is
   begin -- Index
      return Ada.Strings.Fixed.Index (Value (Source), Pattern, Going, Mapping);
   end Index;

   function Index
     (Source : CS.Chars_Ptr;
      Set    : Ada.Strings.Maps.Character_Set;
      Test   : Ada.Strings.Membership := Ada.Strings.Inside;
      Going  : Ada.Strings.Direction  := Ada.Strings.Forward)
      return   Natural
   is
   begin -- Index
      return Ada.Strings.Fixed.Index (Value (Source), Set, Test, Going);
   end Index;

   function Index_Non_Blank
     (Source : CS.Chars_Ptr;
      Going  : Ada.Strings.Direction := Ada.Strings.Forward)
      return   Natural
   is
   begin -- Index_Non_Blank
      return Ada.Strings.Fixed.Index_Non_Blank (Value (Source), Going);
   end Index_Non_Blank;

   ------------
   -- Insert --
   ------------

   function Insert
     (Source   : CS.Chars_Ptr;
      Before   : Positive;
      New_Item : String)
      return     CS.Chars_Ptr
   is
   begin -- Insert
      return
        CS.New_String
          (Ada.Strings.Fixed.Insert (Value (Source), Before, New_Item));
   end Insert;

   procedure Insert
     (Source   : in out CS.Chars_Ptr;
      Before   : in Positive;
      New_Item : in String)
   is
      Temp : CS.Chars_Ptr := Source;
   begin -- Insert
      Source := CS.New_String
        (Ada.Strings.Fixed.Insert (Value (Source), Before, New_Item));
      CS.Free (Temp);
   end Insert;

   ------------
   -- Length --
   ------------

   function Length (Source : CS.Chars_Ptr) return Natural is
   begin -- Length
      return Natural (CS.Strlen (Source));
   end Length;

   ---------------
   -- Overwrite --
   ---------------

   function Overwrite
     (Source    : CS.Chars_Ptr;
      Position  : Positive;
      New_Item  : String)
      return      CS.Chars_Ptr is

   begin -- Overwrite
      return CS.New_String
        (Ada.Strings.Fixed.Overwrite (Value (Source), Position, New_Item));
   end Overwrite;

   procedure Overwrite
     (Source    : in out CS.Chars_Ptr;
      Position  : in Positive;
      New_Item  : in String)
   is
      Temp : CS.Chars_Ptr := Source;
   begin -- Overwrite
      Source := CS.New_String
        (Ada.Strings.Fixed.Overwrite (Value (Source), Position, New_Item));
      CS.Free (Temp);
   end Overwrite;

   ---------------------
   -- Replace_Element --
   ---------------------

   procedure Replace_Element
     (Source : in out CS.Chars_Ptr;
      Index  : Positive;
      By     : Character)
   is
   begin -- Replace_Element
      if Index <= Length (Source) then
         CS.Update (Source, C.Size_t (Index), By & "", Check => False);
      else
         raise Ada.Strings.Index_Error;
      end if;
   end Replace_Element;

   -------------------
   -- Replace_Slice --
   -------------------

   function Replace_Slice
     (Source   : CS.Chars_Ptr;
      Low      : Positive;
      High     : Natural;
      By       : String)
      return     CS.Chars_Ptr
   is
   begin -- Replace_Slice
      return
        CS.New_String
          (Ada.Strings.Fixed.Replace_Slice (Value (Source), Low, High, By));
   end Replace_Slice;

   procedure Replace_Slice
     (Source   : in out CS.Chars_Ptr;
      Low      : in Positive;
      High     : in Natural;
      By       : in String)
   is
      Temp : CS.Chars_Ptr := Source;
   begin -- Replace_Slice
      Source := CS.New_String
        (Ada.Strings.Fixed.Replace_Slice (Value (Source), Low, High, By));
      CS.Free (Temp);
   end Replace_Slice;

   -----------
   -- Slice --
   -----------

   function Slice
     (Source : CS.Chars_Ptr;
      Low    : Positive;
      High   : Natural)
      return   String
   is
      Result : String (1..High - Low + 1);

   begin -- Slice
      Result := Value (Source) (Low..High);
      return Result;
   end Slice;

   ----------
   -- Tail --
   ----------

   function Tail
     (Source : CS.Chars_Ptr;
      Count  : Natural;
      Pad    : Character := Ada.Strings.Space)
      return   CS.Chars_Ptr is

   begin -- Tail
      return
        CS.New_String (Ada.Strings.Fixed.Tail (Value (Source), Count, Pad));
   end Tail;

   procedure Tail
     (Source : in out CS.Chars_Ptr;
      Count  : in Natural;
      Pad    : in Character := Ada.Strings.Space)
   is
      Temp : CS.Chars_Ptr := Source;
   begin -- Tail
      Source := CS.New_String
        (Ada.Strings.Fixed.Tail (Value (Source), Count, Pad));
      CS.Free (Temp);
   end Tail;

   ------------
   -- Append --
   ------------

   procedure Append
     (Source   : in out CS.Chars_Ptr;
      New_Item : in CS.Chars_Ptr)
   is
      S_Length : constant C.Size_t := C.Size_t (Length (Source));
      T_Length : constant C.Size_t := S_Length + C.Size_t (Length (New_Item));
      Result   : CS.char_array_access := new C.char_array (1..T_Length+1);
   begin -- Append
      Result.all (1..S_Length+1) := CS.Value (Source);
      Result.all (S_Length + 1..T_Length+1) := CS.Value (New_Item);
      CS.Free (Source);
      Source := CS.To_Chars_Ptr (Result);
   end Append;

   procedure Append
     (Source   : in out CS.Chars_Ptr;
      New_Item : in String)
   is
      S_Length : constant C.Size_t := CS.Strlen (Source);
      Length   : constant C.Size_t := S_Length + New_Item'Length;
      Result   : CS.char_array_access := new C.char_array (1..Length+1);
   begin -- Append
      Result.all (1..S_Length+1) := CS.Value (Source);
      Result.all (S_Length + 1..Length+1) := C.To_C (New_Item);
      CS.Free (Source);
      Source := CS.To_Chars_Ptr (Result);
   end Append;

   procedure Append
     (Source   : in out CS.Chars_Ptr;
      New_Item : in Character)
   is
      S_Length : constant C.Size_t := CS.Strlen (Source);
      Length   : constant C.Size_t := S_Length + 1;
      Result   : CS.char_array_access := new C.char_array (1..Length+1);
   begin -- Append
      Result.all (1..S_Length+1) := CS.Value (Source);
      Result.all (Length) := C.To_C (New_Item);
      Result.all (Length+1)  := C.To_C (ASCII.Nul);
      CS.Free (Source);
      Source := CS.To_Chars_Ptr (Result);
   end Append;

   ---------------
   -- Translate --
   ---------------

   function Translate
     (Source  : CS.Chars_Ptr;
      Mapping : Ada.Strings.Maps.Character_Mapping)
      return    CS.Chars_Ptr
   is
   begin -- Translate
      return
        CS.New_String (Ada.Strings.Fixed.Translate (Value (Source), Mapping));
   end Translate;

   procedure Translate
     (Source  : in out CS.Chars_Ptr;
      Mapping : Ada.Strings.Maps.Character_Mapping)
   is
      Temp : CS.Chars_Ptr := Source;
   begin -- Translate
      Source := CS.New_String (Ada.Strings.Fixed.Translate (Value (Source), Mapping));
      CS.Free (Temp);
   end Translate;

   function Translate
     (Source  : in CS.Chars_Ptr;
      Mapping : in Ada.Strings.Maps.Character_Mapping_Function)
      return    CS.Chars_Ptr
   is
   begin -- Translate
      return
        CS.New_String (Ada.Strings.Fixed.Translate (Value (Source), Mapping));
   end Translate;

   procedure Translate
     (Source  : in out CS.Chars_Ptr;
      Mapping : in Ada.Strings.Maps.Character_Mapping_Function)
   is
      Temp : CS.Chars_Ptr := Source;
   begin -- Translate
      Source := CS.New_String (Ada.Strings.Fixed.Translate (Value (Source), Mapping));
      CS.Free (Temp);
   end Translate;

   ----------
   -- Trim --
   ----------

   function Trim
     (Source : in CS.Chars_Ptr;
      Side   : in Ada.Strings.Trim_End)
      return   CS.Chars_Ptr
   is
   begin -- Trim
      return CS.New_String (Ada.Strings.Fixed.Trim (Value (Source), Side));
   end Trim;

   procedure Trim
     (Source : in out CS.Chars_Ptr;
      Side   : in Ada.Strings.Trim_End)
   is
      Temp : CS.Chars_Ptr := Source;
   begin -- Trim
      Source := CS.New_String (Ada.Strings.Fixed.Trim (Value (Source), Side));
      CS.Free (Temp);
   end Trim;

   function Trim
     (Source : in CS.Chars_Ptr;
      Left   : in Ada.Strings.Maps.Character_Set;
      Right  : in Ada.Strings.Maps.Character_Set)
      return   CS.Chars_Ptr
   is
   begin -- Trim
      return
        CS.New_String (Ada.Strings.Fixed.Trim (Value (Source), Left, Right));
   end Trim;

   procedure Trim
     (Source : in out CS.Chars_Ptr;
      Left   : in Ada.Strings.Maps.Character_Set;
      Right  : in Ada.Strings.Maps.Character_Set)
   is
      Temp : CS.Chars_Ptr := Source;
   begin -- Trim
      Source := CS.New_String (Ada.Strings.Fixed.Trim (Value (Source), Left, Right));
      CS.Free (Temp);
   end Trim;

end C_Aux;
