--------------------------------------------------------------------
--
-- c_aux.ads --
--
-- Copyright (c) 1995 Terry J. Westley
--
-- See the file "license.terms" for information on usage and redistribution
-- of this file, and for a DISCLAIMER OF ALL WARRANTIES.
--
--
--------------------------------------------------------------------
with Ada.Strings.Maps;
with Interfaces.C.Pointers;
with Interfaces.C.Strings;

package C_Aux is

   package C  renames Interfaces.C;
   package CS renames Interfaces.C.Strings;

   subtype CPositive is C.Int range 0..C.Int'last;
   type Arg_Vector is array (CPositive range <>) of aliased CS.Chars_Ptr;

   package Argv is new C.Pointers (
      Index              => CPositive,
      Element            => CS.Chars_Ptr,
      Element_Array      => Arg_Vector,
      Default_Terminator => CS.Null_Ptr);

   subtype Chars_Ptr_Ptr  is Argv.Pointer;                  -- char **

   type Chars_Ptr_Ptr_Ptr is access all Chars_Ptr_Ptr;	    -- char ***
   type Int_Ptr           is access all C.Int;		    -- int *
   type Int_Ptr_Ptr       is access all Int_Ptr;	    -- int **
   type Long_Ptr          is access all C.Long;		    -- long *
   type Double_Ptr        is access all C.Double;	    -- double *
   subtype Void_Ptr       is System.Address;		    -- void *
   type File_Ptr          is new System.Address;	    -- FILE *
   type File_Ptr_Ptr      is access all File_Ptr;	    -- FILE **

   function To_C (Str : in String) return C.Strings.Chars_Ptr
      renames C.Strings.New_String;

   -- Get command line arguments from Ada.Command_Line and make
   -- a traditional C (null-terminated) argv vector.  Caller must
   -- allocate the space before call to Get_Argv.
   procedure Get_Argv (argv : out Arg_Vector);

   -- Free a (null-terminated) argv-style vector
   procedure Free (Argv : in out Arg_Vector);

   function "&" (Left, Right : in CS.Chars_Ptr) return String;

   function "&" (
      Left  : in CS.Chars_Ptr;
      Right : String) return String;

   function "&" (
      Left  : String;
      Right : in CS.Chars_Ptr) return String;

   function Length (Source : CS.Chars_Ptr) return Natural;

   function Value (Item : in CS.Chars_Ptr) return String;
   pragma Inline (Value);

   --------------------------------------------------------
   -- Conversion, Concatenation, and Selection Functions --
   --------------------------------------------------------

   procedure Append
     (Source   : in out CS.Chars_Ptr;
      New_Item : in CS.Chars_Ptr);

   procedure Append
     (Source   : in out CS.Chars_Ptr;
      New_Item : in String);

   procedure Append
     (Source   : in out CS.Chars_Ptr;
      New_Item : in Character);

   function Element
     (Source : in CS.Chars_Ptr;
      Index  : in Positive)
      return   Character;

   procedure Replace_Element
     (Source : in out CS.Chars_Ptr;
      Index  : in Positive;
      By     : Character);

   function Slice
     (Source : in CS.Chars_Ptr;
      Low    : in Positive;
      High   : in Natural)
      return   String;

   function "=" (Left, Right : in CS.Chars_Ptr) return Boolean;

   function "="
     (Left  : in CS.Chars_Ptr;
      Right : in String)
      return  Boolean;

   function "="
     (Left  : in String;
      Right : in CS.Chars_Ptr)
      return  Boolean;

   function "<" (Left, Right : in CS.Chars_Ptr) return Boolean;

   function "<"
     (Left  : in CS.Chars_Ptr;
      Right : in String)
      return  Boolean;

   function "<"
     (Left  : in String;
      Right : in CS.Chars_Ptr)
      return  Boolean;

   function "<=" (Left, Right : in CS.Chars_Ptr) return Boolean;

   function "<="
     (Left  : in CS.Chars_Ptr;
      Right : in String)
      return  Boolean;

   function "<="
     (Left  : in String;
      Right : in CS.Chars_Ptr)
      return  Boolean;

   function ">" (Left, Right : in CS.Chars_Ptr) return Boolean;

   function ">"
     (Left  : in CS.Chars_Ptr;
      Right : in String)
      return  Boolean;

   function ">"
     (Left  : in String;
      Right : in CS.Chars_Ptr)
      return  Boolean;

   function ">=" (Left, Right : in CS.Chars_Ptr) return Boolean;

   function ">="
     (Left  : in CS.Chars_Ptr;
      Right : in String)
      return  Boolean;

   function ">="
     (Left  : in String;
      Right : in CS.Chars_Ptr)
      return  Boolean;

   ------------------------
   -- Search Subprograms --
   ------------------------

   function Index
     (Source   : in CS.Chars_Ptr;
      Pattern  : in String;
      Going    : in Ada.Strings.Direction := Ada.Strings.Forward;
      Mapping  : in Ada.Strings.Maps.Character_Mapping :=
                                             Ada.Strings.Maps.Identity)
      return     Natural;

   function Index
     (Source   : in CS.Chars_Ptr;
      Pattern  : in String;
      Going    : in Ada.Strings.Direction := Ada.Strings.Forward;
      Mapping  : in Ada.Strings.Maps.Character_Mapping_Function)
      return     Natural;

   function Index
     (Source : in CS.Chars_Ptr;
      Set    : in Ada.Strings.Maps.Character_Set;
      Test   : in Ada.Strings.Membership := Ada.Strings.Inside;
      Going  : in Ada.Strings.Direction  := Ada.Strings.Forward)
      return   Natural;

   function Index_Non_Blank
     (Source : in CS.Chars_Ptr;
      Going  : in Ada.Strings.Direction := Ada.Strings.Forward)
      return   Natural;

   function Count
     (Source  : in CS.Chars_Ptr;
      Pattern : in String;
      Mapping : in Ada.Strings.Maps.Character_Mapping := Ada.Strings.Maps.Identity)
      return    Natural;

   function Count
     (Source   : in CS.Chars_Ptr;
      Pattern  : in String;
      Mapping  : in Ada.Strings.Maps.Character_Mapping_Function)
      return     Natural;

   function Count
     (Source : in CS.Chars_Ptr;
      Set    : in Ada.Strings.Maps.Character_Set)
      return   Natural;

   procedure Find_Token
     (Source : in CS.Chars_Ptr;
      Set    : in Ada.Strings.Maps.Character_Set;
      Test   : in Ada.Strings.Membership;
      First  : out Positive;
      Last   : out Natural);

   ------------------------------------
   -- String Translation Subprograms --
   ------------------------------------

   function Translate
     (Source  : in CS.Chars_Ptr;
      Mapping : in Ada.Strings.Maps.Character_Mapping)
      return    CS.Chars_Ptr;

   procedure Translate
     (Source  : in out CS.Chars_Ptr;
      Mapping : Ada.Strings.Maps.Character_Mapping);

   function Translate
     (Source  : in CS.Chars_Ptr;
      Mapping : in Ada.Strings.Maps.Character_Mapping_Function)
      return    CS.Chars_Ptr;

   procedure Translate
     (Source  : in out CS.Chars_Ptr;
      Mapping : in Ada.Strings.Maps.Character_Mapping_Function);

   ---------------------------------------
   -- String Transformation Subprograms --
   ---------------------------------------

   function Replace_Slice
     (Source : in CS.Chars_Ptr;
      Low    : in Positive;
      High   : in Natural;
      By     : in String)
      return   CS.Chars_Ptr;

   procedure Replace_Slice
     (Source   : in out CS.Chars_Ptr;
      Low      : in Positive;
      High     : in Natural;
      By       : in String);

   function Insert
     (Source   : in CS.Chars_Ptr;
      Before   : in Positive;
      New_Item : in String)
      return     CS.Chars_Ptr;

   procedure Insert
     (Source   : in out CS.Chars_Ptr;
      Before   : in Positive;
      New_Item : in String);

   function Overwrite
     (Source   : in CS.Chars_Ptr;
      Position : in Positive;
      New_Item : in String)
      return     CS.Chars_Ptr;

   procedure Overwrite
     (Source    : in out CS.Chars_Ptr;
      Position  : in Positive;
      New_Item  : in String);

   function Delete
     (Source  : in CS.Chars_Ptr;
      From    : in Positive;
      Through : in Natural)
      return    CS.Chars_Ptr;

   procedure Delete
     (Source  : in out CS.Chars_Ptr;
      From    : in Positive;
      Through : in Natural);

   function Trim
     (Source : in CS.Chars_Ptr;
      Side   : in Ada.Strings.Trim_End)
      return   CS.Chars_Ptr;

   procedure Trim
     (Source : in out CS.Chars_Ptr;
      Side   : in Ada.Strings.Trim_End);

   function Trim
     (Source : in CS.Chars_Ptr;
      Left   : in Ada.Strings.Maps.Character_Set;
      Right  : in Ada.Strings.Maps.Character_Set)
      return   CS.Chars_Ptr;

   procedure Trim
     (Source : in out CS.Chars_Ptr;
      Left   : in Ada.Strings.Maps.Character_Set;
      Right  : in Ada.Strings.Maps.Character_Set);

   function Head
     (Source : in CS.Chars_Ptr;
      Count  : in Natural;
      Pad    : in Character := Ada.Strings.Space)
      return   CS.Chars_Ptr;

   procedure Head
     (Source : in out CS.Chars_Ptr;
      Count  : in Natural;
      Pad    : in Character := Ada.Strings.Space);

   function Tail
     (Source : in CS.Chars_Ptr;
      Count  : in Natural;
      Pad    : in Character := Ada.Strings.Space)
      return   CS.Chars_Ptr;

   procedure Tail
     (Source : in out CS.Chars_Ptr;
      Count  : in Natural;
      Pad    : in Character := Ada.Strings.Space);

   function "*"
     (Left  : in Natural;
      Right : in Character)
      return  CS.Chars_Ptr;

   function "*"
     (Left  : in Natural;
      Right : in String)
      return  CS.Chars_Ptr;

   function "*"
     (Left  : in Natural;
      Right : in CS.Chars_Ptr)
      return  CS.Chars_Ptr;

end C_Aux;
