/*-
 * Copyright (c) 1993, 1994, 1995, 1996 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifdef USE_TCLMDRIVER
#include <assert.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strstream.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>

#ifdef SVR4
#include <sys/filio.h>
#endif

#ifdef linux
#include <linux/termios.h>
#endif

#include <sys/midiioctl.h>

#include "TclmDr.h"
#include "EvntUtil.h"
#include "Note.h"

const int MaxEventSize = 256;

typedef void Sigfunc(int);

typedef struct active_list {
	TclmDriver		*dev;
	struct active_list	*next;
} AL;

static AL *ActiveList = 0;
static Sigfunc *posixsignal(int signo, Sigfunc *func);

TclmDriver::TclmDriver() : last_play_time(0), last_rec_time(0), curr_event(0),
    play_song(0), rec_song(0), fd(-1), finished(0), last_record_rs(0)
{
}

TclmDriver::TclmDriver(const TclmDriver &td) : last_play_time(0),
    last_rec_time(0), curr_event(0), play_song(0), rec_song(0), fd(-1),
    finished(0), last_record_rs(0)
{
	const TclmDriver *dummy;

	// shut up a warning
	dummy = &td;
}

TclmDriver::~TclmDriver()
{

	if (fd != -1) {
		Stop();
		Close();
		RemoveFromActiveList();
	}
	if (curr_event != 0) {
		delete curr_event;
		curr_event = 0;
	}
}

int
TclmDriver::Open(const char *dev)
{
	ostrstream err;
	char *str;

	if (fd != -1) {
		SetError("Device already open");
		return (0);
	}
	SetName(dev);
	fd = open(dev, O_RDWR);
	if (fd == -1) {
		err << "open failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
	return (1);
}

int
TclmDriver::Close(void)
{

	delete name;
	name = 0;
	if (fd != -1) {
		if (!Stop())
			return (0);
		RemoveFromActiveList();
	}
	if (curr_event != 0) {
		delete curr_event;
		curr_event = 0;
	}
	return (1);
}

int
TclmDriver::Play(Song *s, int r)
{
	ostrstream err;
	char *str;
	int arg, i, raw;

	if (fd == -1) {
		SetError("Device is not open");
		return (0);
	}

	if (ioctl(fd, MGRAW, &raw) == -1) {
		err << "MGRAW ioctl failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	if (raw) {
		SetError("Can't play song to raw device - use timed device "
		    "instead.");
		return (0);
	}

	if (rec_song != 0) {
		SetError("Already recording");
		return (0);
	}
	if (play_song != 0) {
		SetError("Already playing");
		return (0);
	}

	// set repeat
	SetRepeat(r);

	play_song = s;

	// initialize track/event info
	last_play_time = 0;
	finished = 0;
	curr_event = new Event *[play_song->GetNumTracks()];
	if (curr_event == 0) {
		SetError("No more memory");
		return (0);
	}
	for (i = 0; i < play_song->GetNumTracks(); i++)
		curr_event[i] = play_song->GetTrack(i).GetEventsNoMod(0);

	AddToActiveList();

	// Reset the device so the timer is close to current
	if (ioctl(fd, MRESET, 0) == -1) {
		err << "ioctl MRESET failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		RemoveFromActiveList();
		return (0);
	}

	// set division
	arg = play_song->GetDivision();
	if (ioctl(fd, MSDIVISION, &arg) == -1) {
		err << "ioctl MSDIVISION failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		RemoveFromActiveList();
		return (0);
	}

	arg = GetMidiThru();
	if (ioctl(fd, MTHRU, &arg) == -1) {
		err << "ioctl MTHRU failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		RemoveFromActiveList();
		return (0);
	}

	// the following should be a fcntl FIOASYNC call, but the
	// FIOASYNC ioctl doesn't always get passed down to the driver
	// properly under all OSes (read Linux), thus we use the MASYNC
	// entry point.
	arg = 1;
	if (ioctl(fd, MASYNC, &arg) == -1) {
		err << "ioctl MASYNC failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		RemoveFromActiveList();
		return (0);
	}

	return (1);
}

int
TclmDriver::Record(Song *rs, Song *ps, int r)
{
	ostrstream err;
	char *str;
	int arg, i, raw;

	if (fd == -1) {
		SetError("Device is not open");
		return (0);
	}

	if (ioctl(fd, MGRAW, &raw) == -1) {
		err << "MGRAW ioctl failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	if (raw) {
		SetError("Can't record song from raw device - use timed device "
		    "instead.");
		return (0);
	}

	if (rec_song != 0) {
		SetError("Already recording");
		return (0);
	}
	if (play_song != 0) {
		SetError("Already playing");
		return (0);
	}

	AddToActiveList();

	// Reset the device so the timer is close to current
	if (ioctl(fd, MRESET, 0) == -1) {
		err << "ioctl MRESET failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		RemoveFromActiveList();
		return (0);
	}

	rec_song = rs;
	last_play_time = 0;
	last_rec_time = 0;
	last_record_rs = 0;
	if (ps == 0) {
		play_song = 0;
		arg = rec_song->GetDivision();
		if (ioctl(fd, MSDIVISION, &arg) == -1) {
			err << "ioctl MSDIVISION failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			RemoveFromActiveList();
			return (0);
		}
	} else {
		play_song = ps;

		// initialize track/event info
		curr_event = new Event *[play_song->GetNumTracks()];
		if (curr_event == 0) {
			SetError("No more memory");
			RemoveFromActiveList();
			return (0);
		}
		for (i = 0; i < play_song->GetNumTracks(); i++)
			curr_event[i] =
			    play_song->GetTrack(i).GetEventsNoMod(0);

		// set division
		arg = play_song->GetDivision();
		if (ioctl(fd, MSDIVISION, &arg) == -1) {
			err << "ioctl MSDIVISION failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			RemoveFromActiveList();
			return (0);
		}

		// start record timer when first event is scheduled to play
		arg = 1;
		if (ioctl(fd, MRECONPLAY, &arg) == -1) {
			err << "ioctl MRECONPLAY failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			RemoveFromActiveList();
			return (0);
		}

		finished = 0;
	}

	// set repeat
	SetRepeat(r);

	arg = GetMidiThru();
	if (ioctl(fd, MTHRU, &arg) == -1) {
		err << "ioctl MTHRU failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		RemoveFromActiveList();
		return (0);
	}

	// the following should be a fcntl FIOASYNC call, but the
	// FIOASYNC ioctl doesn't always get passed down to the driver
	// properly under all OSes (read Linux), thus we use the MASYNC
	// entry point.
	arg = 1;
	if (ioctl(fd, MASYNC, &arg) == -1) {
		err << "ioctl MASYNC failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		RemoveFromActiveList();
		return (0);
	}

	return (1);
}

int
TclmDriver::Stop(void)
{
	ostrstream err;
	char *str;
	int arg;

	finished = 1;
	if (fd == -1)
		return (1);

	posixsignal(SIGIO, SIG_IGN);

	arg = 0;
	if (ioctl(fd, MASYNC, &arg) == -1) {
		err << "ioctl MASYNC failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	if (ioctl(fd, MGPLAYQ, &arg) == -1) {
		err << "ioctl MGPLAYQ failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
	// flush queues
	if (arg != 0) {
		if (ioctl(fd, MDRAIN, NULL) == -1) {
			err << "ioctl MDRAIN failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
	}

	RemoveFromActiveList();

	if (curr_event != 0) {
		delete curr_event;
		curr_event = 0;
	}
	play_song = 0;
	rec_song = 0;
	return (1);
}

int
TclmDriver::Wait(void)
{
	sigset_t sigset;

	finished = 0;
	sigemptyset(&sigset);
	do
		sigsuspend(&sigset);
	while (!finished);
	return (1);
}

int
TclmDriver::GetMidiThru(void)
{
	ostrstream err;
	const char *name;
	char *str;
	int d, mt;

	// open the device
	if (fd != -1)
		d = fd;
	else {
		if ((name = GetName()) == 0) {
			SetError("No device set");
			return (-1);
		}
		// open device
		if ((d = open(name, O_RDONLY)) == -1) {
			err << "open failed: " << strerror(errno) << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (-1);
		}
	}
	if (ioctl(d, MGTHRU, &mt) == -1) {
		err << "ioctl failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (-1);
	}
	if (fd == -1)
		close(d);
	return (mt);
}

int
TclmDriver::SetMidiThru(int mt)
{
	ostrstream err;
	const char *name;
	char *str;
	int d;

	// open the device
	if (fd != -1)
		d = fd;
	else {
		if ((name = GetName()) == 0) {
			SetError("No device set");
			return (0);
		}
		// open device
		if ((d = open(name, O_RDONLY)) == -1) {
			err << "open failed: " << strerror(errno) << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
	}
	if (ioctl(d, MTHRU, &mt) == -1) {
		err << "ioctl failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
	if (fd == -1)
		close(d);
	return (1);
}

ostrstream *
TclmDriver::Feature(const char *command, const char *argv[], int argc)
{
	struct midi_feature feature;
	struct SMPTE_frame frame;
	ostrstream err, *res;
	char *str;
	const char *dummy1;
	int dummy2;

	dummy1 = argv[0];
	dummy2 = argc;

	if (strcmp(command, "kernel_timing") == 0) {
		feature.type = MFEAT_KERNEL_TIMING;
		feature.data = 0;
	} else if (strcmp(command, "smpte_timing") == 0) {
		feature.type = MFEAT_SMPTE_TIMING;
		feature.data = 0;
	} else if (strcmp(command, "mpu401_timing") == 0) {
		feature.type = MFEAT_MPU401_TIMING;
		feature.data = 0;
	} else if (strcmp(command, "get_smpte") == 0) {
		feature.type = MFEAT_GET_SMPTE;
		feature.data = &frame;
	} else {
		err << "unsupported feature: " << command << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
	if (ioctl(fd, MFEATURE, &feature) == -1) {
		err << "feature ioctl failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	res = new ostrstream;
	if (strcmp(command, "get_smpte") != 0)
		*res << ends;
	else {
		/* convert data we received from SMPTE command*/
		if (!(frame.status & SMPTE_SYNC))
			*res << "NOSYNC";
		else
			*res << setw(2) << setfill('0') << (int)frame.hour << ":"
			    << setw(2) << setfill('0') << (int)frame.minute << ":"
			    << setw(2) << setfill('0') << (int)frame.second << ":"
			    << setw(2) << setfill('0') << (int)frame.frame << ":"
			    << setw(2) << setfill('0') << (int)frame.fraction;
		*res << ends;
	}
	return (res);
}

int
TclmDriver::Slave(const MidiDevice &md)
{
	ostrstream err;
	int id;
	TclmDriver *master;
	char *str;

	master = (TclmDriver *)&md;

	/* get the master's id */
	if (ioctl(master->fd, MGETID, &id) == -1) {
		err << "id ioctl failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	/* now make ourselves its slave */
	if (ioctl(fd, MSLAVE, &id) == -1) {
		err << "slave ioctl failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	return (1);
}

int
TclmDriver::GetTime(unsigned long *t)
{
	ostrstream err;
	char *str;

	if (ioctl(fd, MGSMFTIME, t) == -1) {
		err << "ioctl MGSMFTIME failed: " << strerror(errno)
		    << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
	return (1);
}

int
TclmDriver::Send(Event **events, int num)
{
	SMFTrack smf;
	ostrstream buf;
	unsigned long junk;
	long bytes_written, len;
	int i, raw;
	const char *errstr;
	char *str;


	if (fd == -1) {
		SetError("Device not open");
		return (0);
	}

	if (ioctl(fd, MGRAW, &raw) == -1) {
		buf << "MGRAW ioctl failed: " << strerror(errno) << ends;
		str = buf.str();
		SetError(str);
		delete str;
		return (0);
	}

	if (!raw) {
		SetError("Can't send events to timed device - use raw device "
		    "instead.");
		return (0);
	}

	for (i = 0; i < num; i++) {
		if (!WriteEventToSMFTrack(smf, junk, events[i], 0, errstr)) {
			SetError(errstr);
			return (0);
		}
	}

	len = smf.GetLength();
	bytes_written = write(fd, smf.GetData(len), len);
	if (bytes_written != len) {
		buf << "TclmDriver: write wrote wrong amount";
		if (bytes_written == -1)
			buf << ": " << strerror(errno) << ends;
		else
			buf << " " << bytes_written <<
			    " of " << len << ends;
		str = buf.str();
		SetError(str);
		delete str;
		return (0);
	}
	return (1);
}

int
TclmDriver::Recv(Event ***events, int *num)
{
	SMFTrack track, dup_track;
	Event **event_ptr;
	ostrstream buf;
	unsigned long junk;
	int i, num_events, num_read, raw;
	unsigned char event[MaxEventSize];
	const char *errstr;
	char *str;

	if (fd == -1) {
		SetError("Device not open");
		return (0);
	}

	if (ioctl(fd, MGRAW, &raw) == -1) {
		buf << "MGRAW ioctl failed: " << strerror(errno) << ends;
		str = buf.str();
		SetError(str);
		delete str;
		return (0);
	}

	if (!raw) {
		SetError("Can't recv events to timed device - use raw device "
		    "instead.");
		return (0);
	}

	do {
		if ((num_read = read(fd, event, MaxEventSize)) == -1) {
			buf << "error reading TclmDriver event: " <<
			    strerror(errno) << ends;
			str = buf.str();
			SetError(str);
			delete str;
			return (0);
		}
		track.PutData(event, num_read);
	} while (num_read == MaxEventSize);

	dup_track = track;

	// count how many events are in the track;
	num_events = 0;
	while (ReadEventFromSMFTrack(dup_track, junk, 0, errstr) != 0)
		num_events++;

	event_ptr = new Event *[num_events];
	if (event_ptr == 0) {
		SetError("Out of memory");
		return (0);
	}

	// get the actual events
	for (i = 0; i < num_events; i++)
		event_ptr[i] = ReadEventFromSMFTrack(track, junk, 0, errstr);

	*events = event_ptr;
	*num = num_events;
	return (1);
}

int
TclmDriver::SetChannelMask(int mask)
{
	ostrstream buf;
	char *str;
	unsigned short us_mask;

	if (fd == -1) {
		SetError("Device not open");
		return (0);
	}
	us_mask = mask;
	if (ioctl(fd, MSCMASK, &us_mask) == -1) {
		buf << "MSCMASK ioctl failed: " << strerror(errno) << ends;
		str = buf.str();
		SetError(str);
		delete str;
		return (0);
	}
	return (1);
}

int
TclmDriver::GetChannelMask(void)
{
	ostrstream buf;
	char *str;
	unsigned short us_mask;

	if (fd == -1) {
		SetError("Device not open");
		return (-1);
	}
	if (ioctl(fd, MGCMASK, &us_mask) == -1) {
		buf << "MGCMASK ioctl failed: " << strerror(errno) << ends;
		str = buf.str();
		SetError(str);
		delete str;
		return (-1);
	}
	return (us_mask);
}

/*
 * Since this routine is called ASYNC, we just print out errors
 * instead of putting them in SetError
 */
void
TclmDriverSigIO(int notused)
{
	struct timeval t;
	fd_set rset, wset;
	struct active_list *al;
	int largest_fd;
	TclmDriver *mpu;
	int dummy;

	// shut up a warning
	dummy = notused;

	FD_ZERO(&rset);
	FD_ZERO(&wset);
	largest_fd = 0;
	for (al = ActiveList; al != 0; al = al->next) {
		mpu = al->dev;
		if (mpu->fd > largest_fd)
			largest_fd = mpu->fd;
		FD_SET(mpu->fd, &rset);
		FD_SET(mpu->fd, &wset);
	}
	t.tv_sec = 0;
	t.tv_usec = 0;

	// poll to find out what caused the SIGIO
	if (select(largest_fd + 1, &rset, &wset, 0, &t) == -1) {
		cerr << "error selecting TclmDriver: " << strerror(errno)
		    << "\n";
		return;
	}

	for (al = ActiveList; al != 0; al = al->next) {
		mpu = al->dev;
		// check for record event
		if (FD_ISSET(mpu->fd, &rset)) {
			if (mpu->rec_song != 0) {
				mpu->ReadEvents();
			}
		}

		if (FD_ISSET(mpu->fd, &wset)) {
			if (mpu->play_song != 0 && !mpu->finished) {
				if (mpu->WriteEvents() == 0)
					mpu->finished = 1;
			}
		}
	}
}

/*
 * We get SIGURGs when we are using a SMPTE device and the time
 * has changed so much that we need to reposition ourselves
 */
void
TclmDriverSigUrg(int notused)
{
	unsigned long new_time;
	fd_set eset;
	struct timeval t;
	struct active_list *al;
	int i, largest_fd;
	TclmDriver *mpu;
	int dummy;

	// shut up a warning
	dummy = notused;

	FD_ZERO(&eset);
	largest_fd = 0;
	for (al = ActiveList; al != 0; al = al->next) {
		mpu = al->dev;
		if (mpu->fd > largest_fd)
			largest_fd = mpu->fd;
		FD_SET(mpu->fd, &eset);
	}
	t.tv_sec = 0;
	t.tv_usec = 0;

	// poll to find out what caused the SIGURG
	if (select(largest_fd + 1, 0, 0, &eset, &t) == -1) {
		cerr << "error selecting TclmDriver: " << strerror(errno)
		    << "\n";
		return;
	}

	for (al = ActiveList; al != 0; al = al->next) {
		mpu = al->dev;
		// check for record event
		if (FD_ISSET(mpu->fd, &eset)) {
			if (ioctl(mpu->fd, MGSMFTIME, &new_time) == -1) {
				/*
				 * we can't update, but we still need
				 * to send events
				 */
				mpu->WriteEvents();
				return;
			}

			/* redo curr_event pointers based on new time */
			for (i = 0; i < mpu->play_song->GetNumTracks(); i++)
				mpu->curr_event[i] = mpu->play_song->
				    GetTrack(i).GetEventsNoMod(new_time);
			mpu->last_play_time = new_time;
			mpu->last_rec_time = new_time;
			/* now write the new events */
			mpu->WriteEvents();
		}
		/*
		 * We might have finished playing, but then rewound the smpte
		 * tape, in which case, we're not finished anymore.
		 */
		mpu->finished = 0;
	}
}

int
TclmDriver::WriteEvents(void)
{
	SMFTrack smf;
	unsigned long low_time;
	long len;
	int bytes_written, i, low_index, numtowrite, numwritten, pq;
	const char *errstr;

	if (ioctl(fd, MGQAVAIL, &numtowrite) == -1) {
		cerr << "TclmDriverThread: ioctl MGQAVAIL failed: "
		    << strerror(errno) << "\n";
		return (-1);
	}
	for (numwritten = 0; numwritten < numtowrite;) {
		// find track with lowest timed event
		low_index = -1;
		low_time = 0;
		for (i = 0; i < play_song->GetNumTracks(); i++) {
			if (curr_event[i] == 0)
				continue;
			if (low_index == -1) {
				low_time = curr_event[i]->GetTime();
				low_index = i;
				continue;
			} else {
				if (curr_event[i]->GetTime() < low_time) {
					low_time = curr_event[i]->GetTime();
					low_index = i;
				}
			}
		}
		// Check to see if we're done
		if (low_index == -1) {
			if (!GetRepeat()) {
				// write any remaining events
				if (numwritten != 0) {
					len = smf.GetLength();
					bytes_written = write(fd,
					    smf.GetData(len), len);
					if (bytes_written != len) {
						cerr << "TclmDriver: write "
						    "wrote wrong amount";
						if (bytes_written == -1)
							cerr << ": " <<
							    strerror(errno) <<
							    "\n";
						else {
							cerr << " " <<
							    bytes_written
							    << " of " <<
							    len << "\n";
						}
					}
					return (numwritten);
				} else {
					/*
					 * if play queue is empty, we're
					 * done, otherwise do nothing
					 */
					if (ioctl(fd, MGPLAYQ, &pq) == -1) {
						cerr << "TclmDriver: MGPLAYQ "
						    "failed: " <<
						    strerror(errno) << "\n";
						return (-1);
					}
					if (pq == 0)
						return (0);
					else
						return (-1);
				}
			}
			// otherwise reset event pointers
			for (i = 0; i < play_song->GetNumTracks(); i++)
				curr_event[i] = play_song->GetTrack(i).
				    GetEventsNoMod(0);
			last_play_time = 0;
			continue;
		}
		// Write events until time changes
		while (curr_event[low_index] != 0 &&
		    curr_event[low_index]->GetTime() == low_time &&
		    numwritten < numtowrite) {
			if (!WriteEventToSMFTrack(smf, last_play_time,
			    curr_event[low_index], 1, errstr)) {
				cerr << "error playing: " << errstr << "\n";
				return (-1);
			}
			numwritten++;
			curr_event[low_index] = play_song->GetTrack(low_index).
			    NextEvent(curr_event[low_index]);
		}
	}
	if (numwritten != 0) {
		len = smf.GetLength();
		bytes_written = write(fd, smf.GetData(len), len);
		if (bytes_written != len) {
			cerr << "TclmDriver: write wrote wrong amount";
			if (bytes_written == -1)
				cerr << ": " << strerror(errno) << "\n";
			else
				cerr << " " << bytes_written <<
				    " of " << len << "\n";
		}
	}
	return (numwritten);
}

void
TclmDriver::ReadEvents(void)
{
	SMFTrack track;
	int num_read;
	EventType etype;
	unsigned char event[MaxEventSize];
	const char *errstr;
	Event *e, *first_e;

	do {
		if ((num_read = read(fd, event, MaxEventSize)) == -1) {
			cerr << "error reading TclmDriver event: " <<
			    strerror(errno) << "\n";
			return;
		}
		track.PutData(event, num_read);
	} while (num_read == MaxEventSize);

	track.SetRunningState(last_record_rs);
	first_e = 0;
	while ((e = ReadEventFromSMFTrack(track, last_rec_time, 1, errstr))
	    != 0) {
		Event *new_e;

		if (first_e == 0)
			first_e = e;
		new_e = rec_song->PutEvent(0, *e);
		delete e;
		if (new_e == 0)
			continue;
		etype = new_e->GetType();
		// put links on noteoffs
		if ((etype == NOTEON && ((NoteEvent *)new_e)->GetVelocity()
		    == 0) || etype == NOTEOFF)
			rec_song->SetNotePair(0, new_e);
	}
	if (errstr != 0)
		cerr << "Error while recording: " << errstr << "\n";
	last_record_rs = track.GetRunningState();
}

void
TclmDriver::AddToActiveList(void)
{
	struct active_list *al;

	// set up async stuff
	if (ActiveList == 0) {
		posixsignal(SIGIO, TclmDriverSigIO);
		posixsignal(SIGURG, TclmDriverSigUrg);
	}

	al = new AL;
	assert(al != 0);
	al->dev = this;
	al->next = ActiveList;
	ActiveList = al;
}

void
TclmDriver::RemoveFromActiveList(void)
{
	struct active_list *al, *alprev;

	alprev = 0;
	for (al = ActiveList; al != 0; al = al->next) {
		if (al->dev == this) {
			if (alprev == 0)
				ActiveList = al->next;
			else
				alprev->next = al->next;
			break;
		}
	}
	delete al;
	if (ActiveList == 0) {
		posixsignal(SIGIO, SIG_DFL);
		posixsignal(SIGURG, SIG_DFL);
	}
}

ostream &
operator<<(ostream &os, const TclmDriver &mpu)
{

	os << mpu.GetName();
	if (mpu.name != 0)
		os << "Open " << mpu.name;
	if (mpu.play_song != 0)
		os << " Playing";
	if (mpu.rec_song != 0)
		os << " Recording";
	return (os);
}

Sigfunc *
posixsignal(int signo, Sigfunc *func)
{
	struct sigaction act, oact;

	sigemptyset(&oact.sa_mask);
	sigemptyset(&act.sa_mask);
#ifdef __FreeBSD__
	/* FreeBSD is not POSIX in this respect */
	act.sa_handler = (void (*)())func;
#else
	act.sa_handler = func;
#endif
	act.sa_flags = 0;

	if (signo == SIGALRM) {
#ifdef SA_INTERRUPT
		act.sa_flags |= SA_INTERRUPT;	/* SunOS */
#endif
	} else {
#ifdef SA_RESTART
		act.sa_flags |= SA_RESTART;	/* SVR4, 4.4BSD */
#endif
	}

	if (sigaction(signo, &act, &oact) < 0)
		return (SIG_ERR);
	return ((Sigfunc *)oact.sa_handler);
}
#endif
