/*-
 * Copyright (c) 1995 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "midios.h"
#include "midi-iface.h"
#include "midi-gus.h"

static u_char gus_irq_bitmap(int irq);
static int gus_ping(unsigned short port);
static void gus_delay(unsigned short);
static void gus_poke_data(unsigned short port, unsigned long address,
    unsigned char data);
static unsigned char gus_peek_data(unsigned short port, unsigned long address);

struct midi_iface *
gus_probe(struct midi_softc *softc, MIDI_IFACE_TYPE type, void *vloc)
{
	struct midi_iface_gus *iface;
	struct midi_isa_iface *loc;
	int irqmask;

	if (type != MIDI_ISA_IFACE)
		return (NULL);
	loc = (struct midi_isa_iface *)vloc;
	/* if not irq specified, try to find one we can use. */
	if (loc->irq == -1) {
		irqmask = (1 << 2) | (1 << 3) | (1 << 5) | (1 << 7) |
		    (1 << 11) | (1 << 12) | (1 << 15);
		loc->irq = FIND_IRQ(irqmask);
		if (loc->irq == -1)
			return (NULL);
	}

	/* reset the GF1 */
	OUTB(loc->io_port + GF1_REG_SELECT, MASTER_RESET);
	OUTB(loc->io_port + GF1_DATA_HI, 0x00);

	/* wait a little while ... */
	gus_delay(loc->io_port + GF1_DRAM);
	gus_delay(loc->io_port + GF1_DRAM);

	/* Release Reset */
	OUTB(loc->io_port + GF1_REG_SELECT, MASTER_RESET);
	OUTB(loc->io_port + GF1_DATA_HI, GF1_MASTER_RESET);

	gus_delay(loc->io_port + GF1_DRAM);
	gus_delay(loc->io_port + GF1_DRAM);

	if (!gus_ping(loc->io_port))
		return (NULL);

	iface = MALLOC_NOWAIT(sizeof(struct midi_iface_gus));
	if (iface == NULL)
		return (NULL);

	iface->iface.loc = MALLOC_NOWAIT(sizeof(struct midi_isa_iface));
	if (iface->iface.loc == NULL) {
		FREE(iface, sizeof(struct midi_iface_gus));
		return (NULL);
	}

	/* copy location */
	iface->iface.type = type;
	*(struct midi_isa_iface *)iface->iface.loc = *loc;

	iface->iface.softc = softc;
	iface->iface.name = gus_name;
	iface->iface.size = gus_size;
	iface->iface.gen_intr = gus_gen_intr;
	iface->iface.reset = gus_big_reset;
	iface->iface.open = gus_open;
	iface->iface.close = gus_close;
	iface->iface.intr = gus_intr;
	iface->iface.data_avail = gus_data_avail;
	iface->iface.write = gus_write;
	iface->iface.read = gus_read;
	iface->iface.feature = gus_feature;
	iface->iface.free = gus_free;

	gus_reset(iface);
	return ((struct midi_iface *)iface);
}

const char *
gus_name(struct midi_iface *mif)
{

	return ("Gravis Ultra Sound");
}

void
gus_size(struct midi_iface *mif, void *size)
{
	int *s;

	/*
	 * This is a lie.  The GUS uses all sorts of I/O ports, but
	 * unfortunately, they are not contiguous.  Thus I'm just saying
	 * 0.
	 */
	s = (int *)size;
	*s = 0;
}

void
gus_gen_intr(struct midi_iface *mif)
{
	struct midi_iface_gus *iface;
	struct midi_isa_iface *loc;

	iface = (struct midi_iface_gus *)mif;
	loc = (struct midi_isa_iface *)mif->loc;

	/* allow xmit irqs */
	/* default to irq 15 for now. */
	OUTB(loc->io_port + GUS_IRQ_CONTROL, gus_irq_bitmap(15));
	OUTB(loc->io_port + GUS_MIDI_CONTROL, 0x20);
	/* reset board - this should generate an xmit buffer emptry irq */
	gus_reset(iface);
	/* disable xmit irqs */
	OUTB(loc->io_port + GUS_MIDI_CONTROL, 0x00);
}

int
gus_big_reset(struct midi_iface *mif)
{
	struct midi_iface_gus *iface;
	struct midi_isa_iface *loc;

	iface = (struct midi_iface_gus *)mif;
	loc = (struct midi_isa_iface *)mif->loc;
	gus_reset(iface);
	/* set receive IRQ and clear xmit IRQ */
	/* default to irq 15 for now */
	OUTB(loc->io_port + GUS_IRQ_CONTROL, gus_irq_bitmap(15));
	OUTB(loc->io_port + GUS_MIDI_CONTROL, 0x80);
	return (1);
}

int
gus_open(struct midi_iface *mif)
{

	/*
	 * don't do anything.  midiopen already calls fullreset which
	 * puts the board in UART mode.
	 */
	return (0);
}

void
gus_close(struct midi_iface *mif)
{
}

int
gus_intr(struct midi_iface *mif)
{
	struct midi_iface_gus *iface;
	struct midi_isa_iface *loc;
	u_char code;

	iface = (struct midi_iface_gus *)mif;
	loc = (struct midi_isa_iface *)mif->loc;

	code = INB(loc->io_port + GUS_MIDI_DATA);
	return (code);
}

int
gus_data_avail(struct midi_iface *mif)
{
	struct midi_isa_iface *loc;
	int flag;

	loc = (struct midi_isa_iface *)mif->loc;

	flag = INB(loc->io_port + GUS_MIDI_STATUS);
	return (flag & MIDI_DATA_AVL);
}

void
gus_write(struct midi_iface *mif, struct event *event)
{
	struct midi_iface_gus *iface;
	struct midi_isa_iface *loc;
	int i;
	u_char byte;

	iface = (struct midi_iface_gus *)mif;
	loc = (struct midi_isa_iface *)mif->loc;

	for (i = 0; i < stynamic_len(&event->data); i++) {
		byte = stynamic_get_byte(&event->data, i);
		gus_wait_rdy_rcv(iface);
		OUTB(loc->io_port + GUS_MIDI_DATA, byte);
	}
}

u_char
gus_read(struct midi_iface *mif)
{
	struct midi_iface_gus *iface;
	struct midi_isa_iface *loc;
	u_char byte;

	iface = (struct midi_iface_gus *)mif;
	loc = (struct midi_isa_iface *)mif->loc;

	byte = INB(loc->io_port + GUS_MIDI_DATA);
	return (byte);
}

int
gus_feature(struct midi_iface *iface, struct midi_feature *feature)
{

	/* no features yet */
	return (ENOTTY);
}

void
gus_free(struct midi_iface *mif)
{
	struct midi_iface_gus *iface;

	FREE(mif->loc, sizeof(struct midi_isa_iface));

	iface = (struct midi_iface_gus *)mif;
	FREE(iface, sizeof(struct midi_iface_gus));
}


int
gus_wait_rdy_rcv(struct midi_iface_gus *iface)
{
	struct midi_isa_iface *loc;
	int flag, i;

	loc = (struct midi_isa_iface *)iface->iface.loc;

	for (i = 0; i < MIDI_TRIES; i++) {
		flag = INB(loc->io_port + GUS_MIDI_STATUS) & MIDI_RDY_RCV;
		if (flag)
			break;
	}
	if (i == MIDI_TRIES)
		return (0);
	/*
	 * One more small delay to insure the GUS can accept data.  Sometimes
	 * it's a bit premature.
	 */
	(void)INB(loc->io_port + GF1_DRAM);     /* GJW */
	return (1);
}

void
gus_reset(struct midi_iface_gus *iface)
{
	struct midi_isa_iface *loc;

	loc = (struct midi_isa_iface *)iface->iface.loc;

	/* bit 0 & 1 high ... */
	OUTB(loc->io_port + GUS_MIDI_CONTROL, 0x03);
	/* then low */
	OUTB(loc->io_port + GUS_MIDI_CONTROL, 0x00);
}

/*
 * converts an irq number to a byte suitable for the irq control register.
 * 0xff means invalid irq.
 */
u_char
gus_irq_bitmap(int irq)
{
	u_char b;

	b = 0x40;
	switch (irq) {
	case 2:
		b |= 0x01;
		break;
	case 3:
		b |= 0x03;
		break;
	case 5:
		b |= 0x02;
		break;
	case 7:
		b |= 0x04;
		break;
	case 11:
		b |= 0x05;
		break;
	case 12:
		b |= 0x06;
		break;
	case 15:
		b |= 0x07;
		break;
	default:
		b = 0xff;
		break;
	}
	return (b);
}

static int
gus_ping(unsigned short port)
{
	unsigned char save_val0, save_val1, val0, val1;

	/* save current values  ... */
	save_val0 = gus_peek_data(port, 0L);
	save_val1 = gus_peek_data(port, 1L);

	gus_poke_data(port, 0L, 0xaa);
	gus_poke_data(port, 1L, 0x55);
	val0 = gus_peek_data(port, 0L);
	val1 = gus_peek_data(port, 1L);

	/* restore data to old values ... */
	gus_poke_data(port, 0L, save_val0);
	gus_poke_data(port, 1L, save_val1);

	if ((val0 == 0xaa) && (val1 == 0x55))
		return (1);
	else
		return (0);
}

static void
gus_delay(unsigned short port)
{
	int i;

	for (i = 0; i < 7; i++)
		(void)INB(port);
}

static void
gus_poke_data(unsigned short port, unsigned long address, unsigned char data)
{
	int s;

	s = BLOCK_INTR();

	OUTB(port + GF1_REG_SELECT, SET_DRAM_LOW);
	OUTW(port + GF1_DATA_LOW, (unsigned short)(address & 0xffffUL));
	OUTB(port + GF1_REG_SELECT, SET_DRAM_HIGH);
	OUTB(port + GF1_DATA_HI, (unsigned short)((address >> 16) & 0xffffUL));
	OUTB(port + GF1_DRAM, data);

	UNBLOCK_INTR(s);
}

static unsigned char
gus_peek_data(unsigned short port, unsigned long address)
{
	int s;
	unsigned char ret;

	s = BLOCK_INTR();

	OUTB(port + GF1_REG_SELECT, SET_DRAM_LOW);
	OUTW(port + GF1_DATA_LOW, (unsigned short)(address & 0xffffUL));
	OUTB(port + GF1_REG_SELECT, SET_DRAM_HIGH);
	OUTB(port + GF1_DATA_HI, (unsigned short)((address >> 16) & 0xffffUL));
	ret = INB(port + GF1_DRAM);

	UNBLOCK_INTR(s);
	return (ret);
}
