;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:TCLTK; Base:10 -*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                      Usage & Distribution Agreement                      ;;
;;                            Tk/Lisp Interface                             ;;
;;                                                                          ;;
;; This file is part of the Tk/Lisp Interface (TLI) developed by the        ;;
;; Lockheed Artificial Intelligence Center.  Use, duplication,              ;;
;; distribution, and disclosure of the TLI is subject to the following      ;;
;; restrictions and limitations.                                            ;;
;;                                                                          ;;
;; 1. Lockheed makes no warranty regarding the merchantability or fitness   ;;
;;    of the TLI for any particular purpose.                                ;;
;;                                                                          ;;
;; 2. Lockheed assumes no responsibility for the TLI and assumes no         ;;
;;    obligation to support or maintain the TLI.                            ;;
;;                                                                          ;;
;; 3. The TLI can be duplicated and redistributed without restriction,      ;;
;;    provided that this Usage & Distribution Agreement header is left      ;;
;;    intact and unaltered in the source code.                              ;;
;;                                                                          ;;
;; 4. The TLI can be modified and/or extended without restriction, provided ;;
;;    that all modifications and their authors are clearly noted in a       ;;
;;    header preceding this header in the source code files.                ;;
;;                                                                          ;;
;;    Lockheed asks that the authors of any such modifications or           ;;
;;    extensions to the TLI share them with the Lockheed AI Center in the   ;;
;;    same open manner as the Lockheed AI Center is sharing the TLI.        ;;
;;                                                                          ;;
;; Please send comments about or enhancements to the TLI to Dan Kuokka      ;;
;; (kuokka@aic.lockheed.com)  or Larry Harada (harada@aic.lockheed.com).    ;;
;;                                                                          ;;
;; For more information about the Lockheed AI Center, see our               ;;
;; World-Wide Web page @ http://www.rdd.lmsc.lockheed.com.                  ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;-----------------------------------------------------------------------------
;;; Tk Utilities
;;;-----------------------------------------------------------------------------
(eval-when (:load-toplevel :compile-toplevel :execute)
  (unless (find-package :tcltk)
    (defpackage :tcltk))

  (in-package :tcltk)
					
  (import 'user::*tk-foreign-file-directory*)

  (require :process)			;For Tk loop.
  (use-package :multiprocessing)

  (require :foreign)			;Foreign function definitions.
  (use-package :foreign-functions))

(eval-when (:load-toplevel :execute)
  (load ""
	:foreign-files `(,(make-pathname 
			   :directory *tk-foreign-file-directory*
			   :name      "tkutils.o"))))

;;;-----------------------------------------------------------------------------
;;; Exported symbols
;;;-----------------------------------------------------------------------------
(eval-when (:load-toplevel :execute)
  (export '(*tk-loop-exit*
	    *tk-process-name*
	    def-tcl-command
	    register-tcl-command
	    Tclist->lisp
	    Tkcmd
	    Tkeval
	    Tkget-basename
	    Tkget-interp-result
	    Tkget-name
	    Tkget-parent
	    Tk-init-session
	    Tkset-interp-result
	    Tkwidget-exists-p
	    Tkwidget-name
	    unpack-argv)))
	    
;;;-----------------------------------------------------------------------------
;;; Globals
;;;-----------------------------------------------------------------------------
(defparameter *tk-init-script* "wish.tcl"    "Tk initialization script.")
(defparameter *tk-loop-exit*   nil           "Exit flag for tk-loop.")
(defparameter *tk-main-window* nil           "Tk main window.")
(defparameter *tk-process-name* "TkMainLoop" "Tk Lisp Process name.")

;;;-----------------------------------------------------------------------------
;;; tkloop 
;;;-----------------------------------------------------------------------------
(defun tkloop (&optional (loopname *tk-process-name*) (priority 0))

  ;; Need some way of making sure we don't get multiple main loops running?
  ;; this will do for now ...
  (setq *tk-loop-exit* t)
  (sleep 1)
  (setq *tk-loop-exit* nil)

  (process-run-function ;; Start a Lisp process to handle the loop.
   `(:name ,loopname :priority ,priority)
   #'(lambda ()
       (do () (*tk-loop-exit*)  ;;Busy loop forever unless *tk-loop-exit*
	 (tk-dooneevent *tk-dont-wait*)
	 ;; This gives us fair interactive response time.
	 (process-allow-schedule)))))

;;;-----------------------------------------------------------------------------
;;; tkeval sends the string cmd to the Tcl intepreter
;;;-----------------------------------------------------------------------------
(defun tkeval (interp cmd)
  (tcl-eval interp cmd))

;;;-----------------------------------------------------------------------------
;;; C convenience functions to unpack arrays of strings.
;;;-----------------------------------------------------------------------------
(defforeign 'LFI-unpack-argv		;args: array index
    :entry-point (convert-to-lang "unpack_argv" :language :c)
    :arguments '(integer integer)
    :return-type :integer) 

(defun unpack-argv (argv index)
  (char*-to-string (LFI-unpack-argv argv index)))

;;;-----------------------------------------------------------------------------
;;; set & get-interp-result C convenience functions to set the Tcl_Interp *
;;; result field.
;;;-----------------------------------------------------------------------------
(defforeign 'LFI-get-interp-result		;args: interp 
  :entry-point (convert-to-lang "get_interp_result" :language :c)
  :arguments '(integer)
  :return-type :integer)

(defun tkget-interp-result (interp)
  (let ((res (LFI-get-interp-result interp)))
       (if (zerop res)
	   ""
	 (char*-to-string res))))

(defforeign 'tkset-interp-result		;args: interp result_string
  :entry-point (convert-to-lang "set_interp_result" :language :c)
  :arguments '(integer string)
  :return-type :void)

;;;-----------------------------------------------------------------------------
;;; register-tcl-command 
;;; A Tcl interpreter must be created prior to calling this function.
;;; lisp-name is the name of the command given to def-tcl-command.
;;; tcl-namestring is the name used in Tcl scripts for this command.
;;;-----------------------------------------------------------------------------
(defun register-tcl-command (interp lisp-name tcl-namestring 
				    &optional (clientdata 0))
  (multiple-value-bind (ptr index old)
      (register-function lisp-name)
    (declare (ignore index old))
    (tcl-createcommand interp tcl-namestring ptr clientdata 0))) ;; Same problem.

;;;-----------------------------------------------------------------------------
;;; def-tcl-command creates a c-callable Lisp function with the function 
;;; signature required for Tcl_CreateCommand. These commands are registered
;;; with Lisp through register-tcl-command(s).
;;;-----------------------------------------------------------------------------
(defmacro def-tcl-command (name arglist &body body)
  (let ((argcount 0)
	(clientdata (gensym))
	(interp (gensym)) ;; If interp is referenced, the user should supply it.
	(argc (gensym))
	(argv (gensym)))
    `(defun-c-callable ,name ((,clientdata :unsigned-long)
			      (,interp     :unsigned-long)
			      (,argc       :signed-long)
			      (,argv       :unsigned-long))

       (declare (ignore ,clientdata ,interp))

       ;;defun-c-callable doesn't establish an implicit block for a return.
       (block nil 
	 (unless (= ,argc ,(1+ (length arglist)))
	   (format *error-output* 
		   "Error: Wrong number of arguments - received ~a - expected ~a.~%"
		   ,argc
		   ,(1+ (length arglist)))
	   (format *error-output* "Ignoring Tcl command ~a.~%"
		   (unpack-argv ,argv 0))
	   (return *tcl-error*))

	 (let ,(mapcar #'(lambda (p) 
			   (list p `(unpack-argv ,argv ,(incf argcount))))
		arglist)
	   ,@body)

	 *tcl-ok*))))

;;;-----------------------------------------------------------------------------
;;; Tk-init-session loads the default Tk script and creates a main window.
;;; The main window is mapped and the window event loop is entered.
;;; This is just a convenience macro. 
;;;-----------------------------------------------------------------------------
(defmacro Tk-init-session (interp window name)
  `(progn 
     (setf ,interp (tcl-createinterp))
     (setf ,window 
	   (tk-createmainwindow ,interp (system:getenv "DISPLAY") ,name "Tk"))
     (tcl-evalfile ,interp *tk-init-script*)
     (tk-mapwindow ,window)
     (tkloop)))

;;;-----------------------------------------------------------------------------
;;; with-lower-case 
;;;-----------------------------------------------------------------------------
(defmacro with-lower-case (&body body)
  `(let ((*print-case* :downcase))
     ,@body))

;;;-----------------------------------------------------------------------------
;;; Tkcmd
;;;-----------------------------------------------------------------------------
(defun Tkcmd (interp &rest args)
  (labels ((translate (arg)
	     (cond ((stringp arg) arg)
		   ((keywordp arg)
		    (with-lower-case (format nil "-~a" (princ-to-string arg))))
		   ((atom arg)
		    (with-lower-case (princ-to-string arg)))
		   ((listp arg)
		    (format nil "{~{~a ~}}" (mapcar #'translate arg)))
		   (t ""))))

    (Tkeval interp (format nil "~{~a ~}" (mapcar #'translate args)))
    (Tkget-interp-result interp)))

;;;-----------------------------------------------------------------------------
;;; Miscellaneous widget name functions.
;;;-----------------------------------------------------------------------------
;;;-----------------------------------------------------------------------------
;;; Tkget-parent returns the parent widget of the widget "name". 
;;;  (The parent of the "." widget is "").
;;;-----------------------------------------------------------------------------
(defun Tkget-parent (name &optional (interp *interp*))
  (declare (special *interp*))
  (Tkcmd interp "winfo parent" name))

;;;-----------------------------------------------------------------------------
;;; Tkget-name returns the last suffix of the widget "name".
;;;  (The name of the "." widget is the application name).
;;;-----------------------------------------------------------------------------
(defun Tkget-name (name &optional (interp *interp*))
  (declare (special *interp*))
  (Tkcmd interp "winfo name" name))

;;;-----------------------------------------------------------------------------
;;; Tkget-basename returns the rootname of the widget.
;;;  Except for the original root window. These will be toplevel windows.
;;;-----------------------------------------------------------------------------
(defun Tkget-basename (name)
  (subseq name 1 (position #\. name :start 1)))

;;;-----------------------------------------------------------------------------
;;; Tclist->lisp converts a Tcl list "{}" to a Lisp list.
;;;-----------------------------------------------------------------------------
(defun Tclist->lisp (tclist)
  (read-from-string (substitute #\) #\} (substitute #\( #\{ tclist))))

;;;-----------------------------------------------------------------------------
;;; Tkwidget-exists-p returns t if a widget with this name has already been 
;;; created.
;;;-----------------------------------------------------------------------------
(defun Tkwidget-exists-p (name &optional (interp *interp*))
  (declare (special *interp*))
  (string= (Tkcmd interp "winfo exists" name) "1"))

;;;-----------------------------------------------------------------------------
;;; Tkwidget-name creates a new widget name for child.
;;;-----------------------------------------------------------------------------
(defun Tkwidget-name (parent child)
  (with-lower-case
      (if (string= parent ".")
	  (format nil "~a~a" parent child)
	(format nil "~a.~a" parent child))))

;; End of file