/*
 * tkPictImgFmtVIEW.c --
 *
 * A Pict image file handler for VIEW files. 
 * At present there is no write function. 
 *
 * Copyright (c) 1995 The Regents of the University of California.
 *
 * Author: Pierre-Louis Bossart
 * Date: November 17, 1995
 *
 * Derived from tkImgFmtPPM.c in the tk4.0b2 distribution 
 * copyrighted as follows:
 * Copyright (c) 1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "tkpict.h"

#define MAX_MEMORY	1000000		/* don't allocate > 1MB */
#define VIEW 1

/*
 * The format record for the VIEW file format:
 */

static int		FileMatchVIEW (FILE *f, char *fileName,
			    char *formatString, int *widthPtr,
			    int *heightPtr);
static int		FileReadVIEW  (Tcl_Interp *interp,
			    FILE *f, char *fileName, char *formatString,
			    Tk_PictHandle imageHandle, int destX, int destY,
			    int width, int height, int srcX, int srcY);
static int		FileWriteVIEW (Tcl_Interp *interp,
			    char *fileName, char *formatString,
			    Tk_PictImageBlock *blockPtr);

Tk_PictImageFormat tkImgFmtVIEW = {
    "VIEW",			/* name */
    FileMatchVIEW,		/* fileMatchProc */
    NULL,			/* stringMatchProc */
    FileReadVIEW,		/* fileReadProc */
    NULL,			/* stringReadProc */
    FileWriteVIEW,		/* fileWriteProc */
    NULL,			/* stringWriteProc */
};

/*
 * Prototypes for local procedures defined in this file:
 */
static int		ReadVIEWFileHeader (char *filename, int *widthPtr,
			    int *heightPtr, int *data_size,int *data_type);



/*
 *----------------------------------------------------------------------
 *
 * FileMatchVIEW --
 *
 *	This procedure is invoked by the Pict image type to see if
 *	a file contains image data in VIEW format.
 *
 * Results:
 *	The return value is >0 if the first characters in file "f" look
 *	like VIEW data, and 0 otherwise.
 *
 * Side effects:
 *	The access position in f may change.
 *
 *----------------------------------------------------------------------
 */

static int
FileMatchVIEW(f, fileName, formatString, widthPtr, heightPtr)
    FILE *f;			/* The image file, open for reading. */
    char *fileName;		/* The name of the image file. */
    char *formatString;		/* User-specified format string, or NULL. */
    int *widthPtr, *heightPtr;	/* The dimensions of the image are
				 * returned here if the file is a valid
				 * raw VIEW file. */
{
  int data_size;
  int data_type;

  return ReadVIEWFileHeader(fileName, widthPtr, heightPtr,&data_size,&data_type);
}

/*
 *----------------------------------------------------------------------
 *
 * FileReadVIEW --
 *
 *	This procedure is called by the Pict image type to read
 *	VIEW format data from a file and write it into a given
 *	Pict image.
 *
 * Results:
 *	A standard TCL completion code.  If TCL_ERROR is returned
 *	then an error message is left in interp->result.
 *
 * Side effects:
 *	The access position in file f is changed, and new data is
 *	added to the image given by imageHandle.
 *
 *----------------------------------------------------------------------
 */

static int
FileReadVIEW(interp, f, fileName, formatString, imageHandle, destX, destY,
	width, height, srcX, srcY)
    Tcl_Interp *interp;		/* Interpreter to use for reporting errors. */
    FILE *f;			/* The image file, open for reading. */
    char *fileName;		/* The name of the image file. */
    char *formatString;		/* User-specified format string, or NULL. */
    Tk_PictHandle imageHandle;	/* The Pict image to write into. */
    int destX, destY;		/* Coordinates of top-left pixel in
				 * Pict image to be written to. */
    int width, height;		/* Dimensions of block of Pict image to
				 * be written to. */
    int srcX, srcY;		/* Coordinates of top-left pixel to be used
				 * in image being read. */
{
    int fileWidth, fileHeight;
    int nBytes, type, count;
    unsigned char *pixelPtr;
    Tk_PictImageBlock block;
    int data_size;
    int data_type;

    type = ReadVIEWFileHeader(fileName, &fileWidth, &fileHeight,&data_size,&data_type);

    if (type == 0) {
	Tcl_AppendResult(interp, "couldn't read raw VIEW header from file \"",
		fileName, "\"", NULL);
	return TCL_ERROR;
    }
    if ((fileWidth <= 0) || (fileHeight <= 0)) {
	Tcl_AppendResult(interp, "VIEW image file \"", fileName,
		"\" has dimension(s) <= 0", (char *) NULL);
	return TCL_ERROR;
    }

    if ((srcX + width) > fileWidth) {
	width = fileWidth - srcX;
    }
    if ((srcY + height) > fileHeight) {
	height = fileHeight - srcY;
    }
    if ((width <= 0) || (height <= 0)
	|| (srcX >= fileWidth) || (srcY >= fileHeight)) {
	return TCL_OK;
    }

    block.datatype = data_type;
    block.pixelSize = data_size;
    block.width = width;
    block.height = height;
    block.pitch = fileWidth;
    block.skip = 0;

    Tk_PictExpand(imageHandle, destX + width, destY + height);
    /* goto the correct slice */
    fseek(f, (long) data_size*slice_nb*width*height, SEEK_SET); 
   
    if (srcY > 0) {
	fseek(f, (long) (srcY * block.pitch*block.pixelSize), 1);
    }
   
    nBytes = height*block.pitch*block.pixelSize;

    pixelPtr = (unsigned char *) ckalloc((unsigned) nBytes);
    if(pixelPtr == NULL) {
      (void)fprintf(stderr,"FileReadVIEW: Could not allocate memory\n");
      return;
    }

    block.pixelPtr = pixelPtr + srcX * block.pixelSize;
    if ( destX != 0 || destY != 0 || width != fileWidth )
      block.copy = COPY;
    else
      block.copy = NO_COPY; 
  
    count = fread(pixelPtr, 1, (unsigned) nBytes, f);
    if (count != nBytes) {
      Tcl_AppendResult(interp, "error reading VIEW image file \"",
		       fileName, "\": ",
		       feof(f) ? "not enough data" : Tcl_PosixError(interp),
		       (char *) NULL);
      ckfree((char *) pixelPtr);
      return TCL_ERROR;
    }
  
    Tk_PictPutBlock(imageHandle, &block, destX, destY, width, height);
    
    /* free data if necessary */
    if( block.copy == COPY )
	free((void*)pixelPtr);
      
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FileWriteVIEW --
 *
 *	This procedure is invoked to write image data to a file in VIEW
 *	format.
 *
 * Results:
 *	A standard TCL completion code.  If TCL_ERROR is returned
 *	then an error message is left in interp->result.
 *
 * Side effects:
 *	Data is written to the file given by "fileName".
 *
 *----------------------------------------------------------------------
 */

static int
FileWriteVIEW(interp, fileName, formatString, blockPtr)
    Tcl_Interp *interp;
    char *fileName;
    char *formatString;
    Tk_PictImageBlock *blockPtr;
{
#define MAXSTRING 1000
  FILE *f;
  int w, h;
  int nBytes;
  unsigned char *pixelPtr, *pixLinePtr;
  int len;
  char temp_filename[MAXSTRING] ;
  char filename[MAXSTRING] ;

  (void)strcpy(filename,fileName);
  len=strlen(filename);
  if( strstr(filename,".sdt") || strstr(filename,".spr") )
    {
      (void)strncpy(temp_filename,filename,len-4);
      temp_filename[len-4] = '\0'; /*add null terminating character */
    }
  else (void)strcpy(temp_filename,filename);
  (void)strcpy(filename,temp_filename);

  (void)strcat(temp_filename,".spr");  
  f = fopen(temp_filename,"w") ;
  if (f == NULL)
    return TCL_ERROR;
  else      
    {
      if( fprintf(f,"%d\n",2) == 0 ||
	 fprintf(f,"%d\n",blockPtr->width) == 0 ||
	 fprintf(f,"%d\n",0) == 0 ||
	 fprintf(f,"%d\n",1) == 0 ||
	 fprintf(f,"%d\n",blockPtr->height) == 0 ||
	 fprintf(f,"%d\n",0) == 0 ||
	 fprintf(f,"%d\n",1) == 0 ||
	 fprintf(f,"%d\n",blockPtr->datatype) == 0 ) {
	(void)fclose(f);
	return TCL_ERROR;
      }
      (void)fclose(f);
      (void)strcpy(temp_filename,filename);
      (void)strcat(temp_filename,".sdt");  
      if ((f = fopen(temp_filename, "w")) == NULL) {
	Tcl_AppendResult(interp, temp_filename, ": ", Tcl_PosixError(interp),
			 (char *)NULL);
	return TCL_ERROR;
      }
      pixLinePtr = blockPtr->pixelPtr;
      nBytes = blockPtr->height * blockPtr->pitch * blockPtr->pixelSize ;
      if (fwrite(pixLinePtr, 1, (unsigned) nBytes, f) != nBytes) {
	Tcl_AppendResult(interp, "error writing \"", temp_filename, "\": ",
			 Tcl_PosixError(interp), (char *) NULL);
	if (f != NULL) {
	  fclose(f);
	}
	return TCL_ERROR;
      }
      if (fclose(f) == 0) {
	return TCL_OK;
      }
      f = NULL;
    } 
}

/*
 *----------------------------------------------------------------------
 *
 * ReadVIEWFileHeader --
 *
 *	This procedure reads the VIEW header from the beginning of a
 *	VIEW file and returns the dimensions of the image.
 *
 * Results:
 *	The return value is PGM if file "f" appears to start with
 *	a valid PGM header, VIEW if "f" appears to start with a valid
 *      VIEW header, and 0 otherwise.  If the header is valid,
 *	then *widthPtr and *heightPtr are modified to hold the
 *	dimensions of the image.
 *
 * Side effects:
 *	The access position in f advances.
 *
 *----------------------------------------------------------------------
 */

static int
ReadVIEWFileHeader(filename, widthPtr, heightPtr, data_size, data_type)
    char *filename;		/* Image filename to read the header from */
    int *widthPtr, *heightPtr;  /* The dimensions of the image are
					   * returned here. */
     int *data_size;
     int *data_type;
{
   
#define MAXSTRING 1000
  FILE *pfp ;
  char temp_filename[MAXSTRING] ;
  int i;
  int type;
  int ndim, dim[3];
  float ori[3],inter[3];
  int len;

  len=strlen(filename);
  if( strstr(filename,".sdt") || strstr(filename,".spr") )
    {
      (void)strncpy(temp_filename,filename,len-4);
      temp_filename[len-4] = '\0'; /*add null terminating character */
    }
  else (void)strcpy(temp_filename,filename);
 
  (void)strcat(temp_filename,".spr");
  pfp = fopen(temp_filename,"r") ;
 
  if (pfp == NULL)
    return 0;
  else      
    {
      if (fscanf(pfp,"%d",&ndim) != 1)
	{
	  (void)fclose(pfp);return(0);
	}
      if(ndim != 2 &&  ndim != 3)
	{ 
	  (void)fclose(pfp);return(0);
	}
      for (i=0; i<ndim; i++) {
	if (fscanf(pfp,"%d",&dim[i]) != 1) 
	  { (void)fclose(pfp);return(0);}
        if (fscanf(pfp,"%f",&ori[i]) != 1) 
	  { (void)fclose(pfp);return(0);}
        if (fscanf(pfp,"%f",&inter[i]) != 1) 
	  { (void)fclose(pfp);return(0);}
      }
      if (fscanf(pfp,"%d",&type) != 1) 
	{ (void)fclose(pfp);return(0);}
    }
  (void)fclose(pfp) ;

  *heightPtr = dim[1];
  *widthPtr = dim[0];
  switch(type) 
    {
    case BYTE:
       *data_size=sizeof(unsigned char); break;
     case WORD:
       *data_size=sizeof(short); break;
     case LWORD:
       *data_size=sizeof(int); break;
     case REAL:
       *data_size=sizeof(float); break;
     default:
       printf(" Error: can only read byte, short, int, float data \n");
       return(0);
     }
  *data_type = type;
  if( ndim == 2 )
    nb_slices = 0;
  else if( ndim == 3)
    nb_slices = dim[2];

  return(1);
}














