/*
 * lut.c --
 *
 * A source file for Pict images 
 *
 * Copyright (c) 1995 The Regents of the University of California.
 *
 * Author: Pierre-Louis Bossart
 * Date: November 17, 1995
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "tkpict.h"

static int red[256], green[256], blue[256] ;
static int random_walk(int *color);

void EnableOverlays(PictInstance *instancePtr)
{
  int color_nb;
  int i,r;
  
  if(instancePtr->colormap_level != NEW_PRIVATE_COLORMAP ) {
    (void)fprintf(stderr,"Create Private Colormap first \n");
    return;
  }
  if(instancePtr->has_overlay != False)
    return;

  instancePtr->has_overlay = True;
  color_nb = instancePtr->ncolors -1;

  get_lut(chColor,instancePtr->ncolors,instancePtr->lut_start,
	  red,green,blue);

  put_lut(instancePtr->display,
	  instancePtr->colormap,
	  chColor,
	  instancePtr->ncolors,
	  instancePtr->lut_start,
	  instancePtr->has_overlay,
	  red,green,blue);

  for (i = 0; i < 256; ++i) {
    r = i*color_nb/255;
    if( r%2 == 0 )
       r += instancePtr->lut_start;
    else r+= (instancePtr->lut_start-1);
    instancePtr->redValues[i] = lut_colorcell_defs[r].pixel;
  }

  instancePtr->overlay_gc = XCreateGC(instancePtr->display,
				      instancePtr->pixels,0,NULL);
  XSetForeground(instancePtr->display,instancePtr->overlay_gc,
		 lut_colorcell_defs[instancePtr->lut_start+1].pixel);
  XSetPlaneMask(instancePtr->display,instancePtr->overlay_gc,1L);
  XSetFunction(instancePtr->display,instancePtr->overlay_gc,GXor); 
  XSetFillRule(instancePtr->display,instancePtr->overlay_gc,WindingRule);
  XSetLineAttributes(instancePtr->display,instancePtr->overlay_gc,3,
		     LineSolid,CapButt,JoinMiter);
  DitherInstance(instancePtr,0,0,instancePtr->width,instancePtr->height);

} /* end SetUpOverlays */

void DisableOverlays(PictInstance *instancePtr)
{
  int color_nb;
  int i,r;

  if(instancePtr->has_overlay == False)
    return;

  XFreeGC(instancePtr->display,instancePtr->overlay_gc);
  instancePtr->has_overlay = False;

  color_nb = instancePtr->ncolors -1;
  put_lut(instancePtr->display,
	  instancePtr->colormap,
	  chColor,
	  instancePtr->ncolors,
	  instancePtr->lut_start,
	  instancePtr->has_overlay,
	  red,green,blue);

  for (i = 0; i < 256; ++i) {
    r = i*color_nb/255+instancePtr->lut_start;
    instancePtr->redValues[i] = lut_colorcell_defs[r].pixel;
  }

  DitherInstance(instancePtr,0,0,instancePtr->width,instancePtr->height);
} /* end DestroyOverlays */

void non_linear_lut(Display *disp,Colormap cmap,XColor *chColor,
	      int ncolors,int lut_start,char overlay)
{
  /* get_lut(chColor,ncolors,lut_start,red,green,blue) ;*/
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue);
}

void spectrum(Display *disp,Colormap cmap,XColor *chColor,
	      int ncolors,int lut_start,char overlay)
{
  int 		i;
  double 	pi_over_4, f1, f2, aa, bb, wavelength, s, delta;
  
  /* Compute some necessary values */
  pi_over_4 = atan(1.0);	/* for later */
  aa = (2. * S1 - S2) / (S1 * S2);
  bb =      (S2 - S1) / (S1 * S2);
  delta = 1.0 / (ncolors - 1.0);
  
  /* Go thru each color of the spectrum and load with the proper values */
  for (i=0; i<ncolors; i++) {

    /* Compute the distance along a contour in RGB space */
    wavelength = i * delta;
    s = wavelength / (aa * wavelength + bb);
    
    if (s <= 0.0) {		/* Black floor */
      red[i] = 0;
      green[i] = 0;
      blue[i] = 0;
      
    } 
    else if (s <= 1.0) {	/* Black to red */
      red[i] = s * MAXLUTVALUE ;
      green[i] = 0;
      blue[i] = 0;
    } 
    else if (s <= 2.0) {	/* Red to yellow */
      red[i] = MAXLUTVALUE ;
      green[i] = (s - 1.0) * MAXLUTVALUE ;
      blue[i] = 0;
    } 
    else if (s <= 3.0) {	/* Yellow to green */
      red[i] = MAXLUTVALUE  - ((s - 2.0) * MAXLUTVALUE );
      green[i] = MAXLUTVALUE ;
      blue[i] = 0;
    } 
    else if (s <= 4.0) {	/* Green to cyan */
      f1 = (s - 3.0) * pi_over_4;
      red[i] = 0;
      green[i] = cos(f1) * MAXLUTVALUE ;
      blue[i] = sin(f1) * MAXLUTVALUE ;
    } 
    else if (s <= 5.0) {	/* Cyan to blue */
      f1 = (s - 3.0) * pi_over_4;	/* Yes, s-3, not s-4 here */
      red[i] = 0;
      green[i] = cos(f1) * MAXLUTVALUE ;
      blue[i] = sin(f1) * MAXLUTVALUE ;
    } 
    else if (s <= 6.0) {	/* Blue to magenta */
      f1 = (s - 5.0) * pi_over_4;
      red[i] = sin(f1) * MAXLUTVALUE ;
      green[i] = 0;
      blue[i] = cos(f1) * MAXLUTVALUE ;
    } 
    else if (s <= 7.0) {	/* Magenta to white */
      f1 = s - 6.0;
      f2 = (f1 + (1.0-f1)/sqrt(2.0));
      red[i] = f2 * MAXLUTVALUE ;
      green[i] = f1 * MAXLUTVALUE ;
      blue[i] = f2 * MAXLUTVALUE ;
    } 
    else {			/* Saturate to white */
      red[i] = MAXLUTVALUE ;
      green[i] = MAXLUTVALUE ;
      blue[i] = MAXLUTVALUE ;
    }

  } /* end for each color in spectrum */

  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue);
} /* end spectrum */


void rgb(Display *disp,Colormap cmap,XColor *chColor,
	 int ncolors,int lut_start,char overlay)
{
  float step, c;
  int i;

  c = 0;
  step = (ncolors - 1) / 3.0;
  
  for(i=0; i<ncolors; i++){
    if ( c < ncolors )
      blue[i] = c;
    else {
      c = 0;
      blue[i] = c;
    }
    c += step;
  }
  
  step = (ncolors - 1) / 7.0;
  c = 0 ;
  for(i=0; i<ncolors; i++){
    if ( c < ncolors )
      green[i] = c;
    else {
      c = 0;
      green[i] = c;
    }
    c += step;
  }
  c=0;
  for(i=0; i<ncolors; i++){
    if ( c < ncolors )
      red[i] = c;
    else {
      c = 0;
      red[i] = c;
    }
    c += step;
  }
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue);
} /* end rgb */


void gray(Display *disp,Colormap cmap,XColor *chColor,
	 int ncolors,int lut_start,char overlay)
{
  lut_ramp(red,0,0.0,ncolors-1,1.0) ;
  lut_ramp(green,0,0.0,ncolors-1,1.0) ;
  lut_ramp(blue,0,0.0,ncolors-1,1.0) ;
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue);
 
} /* end gray */


void hot(Display *disp,Colormap cmap,XColor *chColor,
	 int ncolors,int lut_start,char overlay) 
     /* this table is currently very similar to ct */
{
  float mult = (float)(ncolors-1) / (float)(MAXLUTVALUE) ;

  lut_ramp(red,   (int)(  0*mult), 0.0, (int)(120*mult), 1.0) ;
  lut_ramp(red,   (int)(120*mult), 1.0, (int)(255*mult), 1.0) ;
  lut_ramp(green, (int)(  0*mult), 0.0, (int)( 85*mult), 0.0) ;
  lut_ramp(green, (int)( 85*mult), 0.0, (int)(255*mult), 1.0) ;
  lut_ramp(blue,  (int)(  0*mult), 0.0, (int)(170*mult), 0.0) ;
  lut_ramp(blue,  (int)(170*mult), 0.0, (int)(255*mult), 1.0) ;
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue);
}/* end hot */


void cold(Display *disp,Colormap cmap,XColor *chColor,
	 int ncolors,int lut_start,char overlay) /* this table is currently ct + invert + flip	*/
{
  float mult = (float)(ncolors-1) / (float)(MAXLUTVALUE) ;

  lut_ramp(red,   (int)(  0*mult), 0.0, (int)( 75*mult), 0.0) ;
  lut_ramp(red,   (int)( 75*mult), 0.0, (int)(195*mult), 0.1) ;
  lut_ramp(red,   (int)(195*mult), 0.1, (int)(255*mult), 1.0) ;
  lut_ramp(green, (int)(  0*mult), 0.0, (int)( 55*mult), 0.0) ;
  lut_ramp(green, (int)( 55*mult), 0.0, (int)(245*mult), 1.0) ;
  lut_ramp(green, (int)(245*mult), 1.0, (int)(255*mult), 1.0) ;
  lut_ramp(blue,  (int)(  0*mult), 0.0, (int)(135*mult), 1.0) ;
  lut_ramp(blue,  (int)(135*mult), 1.0, (int)(255*mult), 1.0) ; 
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue);
} /* end cold */



void hls(Display *disp,Colormap cmap,XColor *chColor,
	 int ncolors,int lut_start,char overlay)
{
  set_hls(red,green,blue) ;
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue); 
} /* end hls */



void ct(Display *disp,Colormap cmap,XColor *chColor,
	 int ncolors,int lut_start,char overlay)
{
  float mult = (float)(ncolors-1) / (float)(MAXLUTVALUE) ;
  
  lut_ramp(red,   (int)(  0*mult), 0.0, (int)( 60*mult), 0.9) ;
  lut_ramp(red,   (int)( 60*mult), 0.9, (int)(180*mult), 1.0) ;
  lut_ramp(red,   (int)(180*mult), 1.0, (int)(255*mult), 1.0) ;
  lut_ramp(green, (int)(  0*mult), 0.0, (int)( 10*mult), 0.0) ;
  lut_ramp(green, (int)( 10*mult), 0.0, (int)(200*mult), 1.0) ;
  lut_ramp(green, (int)(200*mult), 1.0, (int)(255*mult), 1.0) ;
  lut_ramp(blue,  (int)(  0*mult), 0.0, (int)(120*mult), 0.0) ; 
  lut_ramp(blue,  (int)(120*mult), 0.0, (int)(255*mult), 1.0) ;
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue); 
} /* end ct */

void invert(Display *disp,Colormap cmap,XColor *chColor,
	 int ncolors,int lut_start,char overlay)
{
  int nred[MAX_COLORS], ngreen[MAX_COLORS], nblue[MAX_COLORS] ;
  int i, start ;

  get_lut(chColor,ncolors,lut_start,red,green,blue) ;
  start = ncolors - 1 ;
  
  for (i=0; i<ncolors; i++) {
    nred[i] = red[start-i] ;
    ngreen[i] = green[start-i] ;
    nblue[i] = blue[start-i] ;
  }
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,nred,ngreen,nblue); 
  for (i=0; i<ncolors; i++) {
    red[i] = nred[i] ;
    green[i] = ngreen[i] ;
    blue[i] = nblue[i] ;
  }
} /* end invert */

void bowlerhat(Display *disp,Colormap cmap,XColor *chColor,
	 int ncolors,int lut_start,char overlay)
{
  int halfway ;

  halfway = ncolors/2 - 1 ;
  
  /* Up ramp */

  lut_ramp(red,   0, 0.0, halfway, 1.0) ;
  lut_ramp(green, 0, 0.0, halfway, 1.0) ;
  lut_ramp(blue,  0, 0.0, halfway, 1.0) ;
  
  /* Down ramp */
  
  lut_ramp(red,   halfway+1, 1.0, ncolors-1, 0.0) ;
  lut_ramp(green, halfway+1, 1.0, ncolors-1, 0.0) ;
  lut_ramp(blue,  halfway+1, 1.0, ncolors-1, 0.0) ;
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue);
} /* end bowlerhat */

void tophat(Display *disp,Colormap cmap,XColor *chColor,
	 int ncolors,int lut_start,char overlay)
{
  int third, twothird ;
  int j ;
  
  third = ncolors/3  ;
  twothird = 2*third ;
  
  for (j = 0; j < third; j++) {
    red[j] = 0.0 ;
    green[j] = 0.0 ;
    blue[j] = 0.0 ;
  }

  for (j = third; j < twothird; j++) {
    red[j] = MAXLUTVALUE ;
    green[j] = MAXLUTVALUE ;
    blue[j] = MAXLUTVALUE ;
  }
  
  for (j = twothird; j < ncolors; j++) {
    red[j] = 0.0 ;
    green[j] = 0.0 ;
    blue[j] = 0.0 ;
  } 
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue);
} /* end tophat */

void randwalk_spectrum(Display *disp,Colormap cmap,XColor *chColor,
		       int ncolors,int lut_start,char overlay)
{
   int j;
   int rval, gval, bval;

   /* Set up default values for random walk function (from Bob Sherwood). */
   /* Start with red; this could be any starting color  */
   
   rval = (MAXLUTVALUE) ;
   gval = 0;
   bval = 0;

   for (j=0; j<ncolors; j++) {
     red[j] = random_walk(&rval);
     green[j] = random_walk(&gval);
     blue[j] = random_walk(&bval);
   }
   put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue);
} /* end randwalk_spectrum */


int random_walk(int *color)
{
   register long idelta;
   int mask = 0x1f ;

   /*   NOTE: This algorithm adds or subtracts AT MOST the low order
    *   5 bits to the previous value.  To change this, change the
    *   mask.  To make it change more rapidly, use 0x3f
    *   or 0x7f.  To change more slowly, use 0x0f or 0x07.  */

   idelta = random();		/* Get an RV */

   if (idelta & 0x80) {		/* Use eighth bit as a sign */
     *color = abs(*color - (idelta & mask));
   } else {
     *color = abs(*color + (idelta & mask));
   }

   if (*color > 255) {	/* Reflect off 255 */
     *color = (2*255) - *color;
   }
   
   return( *color & 0xff );
} /* end random_walk */

void lut_ramp(int *lut,int begin,float beginv,int end,float endv)
{
  int intensity, i ;
  float increment, value ;
  
  if ((begin < 0) || (begin > 255)) 
    return;
  if ((end < 0) || (end > 255))
    return;    

  if ((beginv < 0.0) || (beginv > 1.0)) 
    return;
  if ((endv < 0.0) || (endv > 1.0)) 
    return;

  if (end == begin) {
    intensity = beginv * 255 + 0.5 ;
    lut[begin] = intensity ;
    return;
  }
  
  increment = (endv - beginv) / (end - begin) ;
  value = beginv ;
  for (i=begin; i<= end; i++) {
    intensity = value * 255 + 0.5 ;
    lut[i] = intensity ;
    value += increment ;
  }
} /* end lut_ramp */

void set_hls(int *red,int *green,int *blue)
{
  float H,L,S ;
  int r,g,b ;
  int n ;
  

  /* set background blue ... n=0 */
  
  H = 0 ;  L = 0.5 ;  S = 0.5 ;
  convert_HLS_rgb(H,L,S,&r,&g,&b) ;
  red[0] = r ;  green[0] = g ;  blue[0] = b ;
  

  /* set red, orange, yellow, and green ranges */

  for (n=1; n<=255; n++) {
    if (n < 64) {
      H = 105.0 ;
      L = 0.3 + 0.00968 * (n - 1) ;
      S = 0.4 + 0.00645 * (n - 1) ;
    }
    else if (n < 128) {
      H = 155.0 ;
      L = 0.3 + 0.00952 * (n - 64) ;
      S = 0.4 + 0.00635 * (n - 64) ;
    }
    else if (n < 192) {
      H = 190.0 ;
      L = 0.3 + 0.00968 * (n - 128) ;
      S = 0.4 + 0.00645 * (n - 128) ;
    }
    else {
      H = 240.0 ;
      L = 0.3 + 0.00968 * (n - 192) ;
      S = 0.4 + 0.00645 * (n - 192) ;
    }
    convert_HLS_rgb(H,L,S,&r,&g,&b) ;
    red[n] = r ;  green[n] = g ;  blue[n] = b ;
  }
} /* end set_hls */


void convert_HLS_rgb(float H,float L,float S,int *r,int *g,int *b)
{
  float R,G,B ;
  float M,m ;
  

  /* Setup equations */
  
  if (L <= 0.5)
    M = L * (1 + S) ;
  else M = L + S - L*S ;
  m = 2*L - M ;
  
  /* Calculate R */
  
  if (H < 60)
    R = m + (M - m) * (H/60.0) ;
  else if (H < 180)
    R = M ;
  else if (H < 240)
    R = m + (M - m) * ((240 - H)/60) ;
  else R = m ;
  
  
  /* calculate G */
  
  if (H < 120)
        G = m ;
  else if (H < 180)
    G = m + (M - m) * ((H - 120)/60) ;
  else if (H < 300)
    G = M ;
  else 
    G = m + (M - m) * ((360 - H)/60) ;

  /* calculate B */
  
  if (H < 60)
    B = M ;
  else if (H < 120)
    B = m + (M - m) * ((120 - H)/60) ;
  else if (H < 240)
    B = m ;
  else if (H < 300)
    B = m + (M - m) * (( H - 240)/60) ;
  else 
    B = M ;


  /* scale R,G,B to 0-255 */
  
  *r = 255 * R ;
  *g = 255 * G ;
  *b = 255 * B ;

} /* end convert_HLS_rgb */

void hatgray(Display *disp,Colormap cmap,XColor *chColor,
	     int ncolors,int lut_start,char overlay)
{
  int tred[256],tgreen[256],tblue[256];
  int i,j;

  /* compute the gray color map */
  
  lut_ramp(red,0,0.0,ncolors-1,1.0) ;
  lut_ramp(green,0,0.0,ncolors-1,1.0) ;
  lut_ramp(blue,0,0.0,ncolors-1,1.0) ;

  /* Up ramp */
  i = 0;
  for (j=1; j<ncolors-1; j+=2) {
    tred[i] = red[j];
    tgreen[i] = green[j];
    tblue[i] = blue[j];
    i++;
  }
  
  /* Down ramp */
  for (j=ncolors-1; j>0; j-=2) {
    tred[i] = red[j];
    tgreen[i] = green[j];
    tblue[i] = blue[j];
    i++;
  }
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,tred,tgreen,tblue);
} /* end hatgray */
 
void hatct(Display *disp,Colormap cmap,XColor *chColor,
	   int ncolors,int lut_start,char overlay)
{
  int tred[256],tgreen[256],tblue[256];
  int i,j;
  float mult = (float)(ncolors-1) / 255.0;

  /* compute the ct color map */
  lut_ramp(red, 0, 0.0, (int)(60*mult), 0.9) ;
  lut_ramp(red, (int)(60*mult), 0.9, (int)(180*mult), 1.0) ;
  lut_ramp(red, (int)(180*mult), 1.0, (int)(255*mult), 1.0) ;
  lut_ramp(green, 0, 0.0, (int)(10*mult), 0.0) ;
  lut_ramp(green, (int)(10*mult), 0.0, (int)(200*mult), 1.0) ;
  lut_ramp(green, (int)(200*mult), 1.0, (int)(255*mult), 1.0) ;
  lut_ramp(blue,  0, 0.0, (int)(120*mult), 0.0) ; 
  lut_ramp(blue, (int)(120*mult), 0.0, (int)(255*mult), 1.0) ;
  
  /* Up ramp */
  i = 0;
  for (j=1; j<ncolors-1; j+=2) {
    tred[i] = red[j];
    tgreen[i] = green[j];
    tblue[i] = blue[j];
    i++;
  }

  /* Down ramp */
  for (j=ncolors-1; j>0; j-=2) {
    tred[i] = red[j];
    tgreen[i] = green[j];
    tblue[i] = blue[j];
    i++;
  } 
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,tred,tgreen,tblue);
} /* end hat_ct */
 



void lut_thres(Display *disp,Colormap cmap,XColor *chColor,
	   int ncolors,int lut_start,char overlay,int loval,int hival)
{
  int i;
 
  if((loval < 0) || (hival>255))
    return;
  if( loval>= hival )
    return;

  /* first discard values set by colortool */
  n1 = fn1 = n2 = fn2 = 0;

  loval = loval*ncolors/255;
  hival = hival*ncolors/255;
 
  for (i=0; i<loval; i++) {
   red[i]=blue[i]=green[i]=0;
  }
  for (i=loval; i<=hival; i++) {
    red[i]=blue[i]=green[i]=255;
  }
  for (i=hival+1; i<ncolors; i++) {
    red[i]=blue[i]=green[i]=0;
  }
  put_lut(disp,cmap,chColor,ncolors,lut_start,overlay,red,green,blue);
} /* end lut_thres */

