/*
 * shared_colors.c  --
 *
 * A source file for Pict images 
 *
 * Copyright (c) 1995 The Regents of the University of California.
 *
 * Author: Pierre-Louis Bossart
 * Date: November 17, 1995
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "tkpict.h"

/* definition of variables for the entire module */

static struct XColorSharedStruct { /* Structure for posting color lut info
				    for other clients to share */
   Colormap        cmap;      /* The colormap ID */
   unsigned char   lutStart;  /* The first available LUT value */
   unsigned char   nPC;       /* The number of pseudocolor values used */
};

static Window root_window;
static Bool   tfWeWroteColorAtom = False; /* True if we allocated colors, 
					     else False */
static int    color_def;

/* definition of global variables */
XColor	 lut_colorcell_defs[MAX_COLORS];
XColor	 chColor[MAX_COLORS];
int      n1,fn1,n2,fn2;	

/* forward declaration of static functions */
static void show_installed_colormaps(Display *disp); 
static void print_visual_class (Visual *visual);
static void writeSharedColorAtom (Display *disp,Colormap cmap,
				  int lutStart,int nPC);
static Status readSharedColorAtom (Display *disp,Colormap *cmap,
				   int *lutStart,int *nPC);     
static void destroySharedColorAtom(Display *disp);  


int init_colors(Display *disp,Colormap *colormap,Visual *visual,
		char *colormap_level,
		int *numcolors,int *start_lut,char *atom)
{
  int                  depth;
  unsigned long	       *plane_masks;
  unsigned long	       *pixels;
  int                  i;               /* Local looping variables */
  int                  lutStartIndex;      /* Index in pixels for lut_start */
  int                  lutEnd;             /* Last value we will use in LUT */
  Bool                 tfGotColors;        /* True if we got colors from X */
  Colormap             cmap;
  Status	       status;
  unsigned long        blackPix, whitePix; /* Pixel value for black, white */
  int                  screenIndex;   /* Index of screen (0, 1, etc.) */
  int                  lut_start;
  int                  ncolors;
  XColor               *colors;
  int                  colormap_size;
 

  screenIndex = DefaultScreen(disp);  
  root_window = RootWindow(disp, screenIndex);
  
  plane_masks = (unsigned long *) malloc(MAXPLANES*sizeof(*plane_masks));
  pixels = (unsigned long *) malloc(MAX_COLORS*sizeof(*pixels));
  if (plane_masks == NULL || pixels == NULL) {
    fprintf(stderr,
	    "\n Unable to allocate storage for init colors\n");
    return 0;
  }

  /* Determine pixel values for black, white in default colormap */
  blackPix = BlackPixel (disp, screenIndex);
  whitePix = WhitePixel (disp, screenIndex);
  depth = DisplayPlanes(disp, screenIndex); 
 
  switch( *colormap_level ) {
  case READ_SHARED_COLORMAP:
    /* Get LUT start and length from other client */  
    status = readSharedColorAtom (disp,&cmap, &lut_start, &ncolors);
    
    if (status != Success) 
      return 0;
    else break;
  case DEFAULT_SCREEN_COLORMAP:
    cmap = DefaultColormap(disp, screenIndex); 
    break;
  case DEFAULT_PRIVATE_COLORMAP:
  case NEW_PRIVATE_COLORMAP: 
    /* Allocate our own colormap */
    /* get 40 colors from default colormap to avoid flashing */
    colormap_size = DisplayCells(disp,screenIndex);
    colors = (XColor*)malloc(colormap_size*sizeof(XColor));

    for(i=0;i<colormap_size;i++) {
      pixels[i] = colors[i].pixel = i;
      colors[i].flags = DoRed | DoGreen | DoBlue;
    }

    XQueryColors(disp,DefaultColormap(disp, screenIndex),
		 colors,colormap_size);
    
    cmap = XCreateColormap (disp, root_window, visual, AllocNone); 
    /* AllocAll doesn't work quite correctly for X11/NeWS 1.0 */
     
    if (!cmap) {
      printf ("ERROR in init_colors: XCreateColormap returned %x\n", 
	      (unsigned int)cmap);
      return 0;
    }

    colormap_size = 40;
    XAllocColorCells(disp,cmap,True,plane_masks,0,pixels,colormap_size); 
    XStoreColors(disp,cmap,colors,colormap_size);
    free((void*)colors);
    break;
  } /* end switch statement */

  if( *colormap_level != READ_SHARED_COLORMAP ) {
    /* Try to get a set of contiguous color table entries for our use */
    /* Decrement by 10 cells until we succeed */
  
    switch (depth) {    
    case 8:
      tfGotColors = False;           /* Will be set True when we succeed */
      ncolors = 212;
      
      while (ncolors > 10) {
	status = XAllocColorCells(disp,cmap,True,plane_masks,0,pixels,ncolors);
	if (status != 0) {
	  tfGotColors = True;   /* Success.  Break out of the while loop */
	  break;
	}
	ncolors -= 10;      /* Failure.  Decrement request and try again */
      } /* End of while loop */
      break;
    default:
      fprintf(stderr,
	      "\nERROR in init_colors: Not enough graphics planes, depth is %d\n",depth) ;
      exit(0);
    }
  
  
    if(tfGotColors == False) {/* Case 1: Got no colors */
      free(plane_masks);  plane_masks = NULL;
      free(pixels);       pixels = NULL;
      return 0;
    }
    
    else if((depth == 8) && (ncolors < 50)) {  /* Case 2: not enough colors */
      
      /* Free the colors we got */      
      XFreeColors (disp, cmap, pixels, ncolors, 0);
      free(plane_masks);  plane_masks = NULL;
      free(pixels);       pixels = NULL;
      return 0;
    }
    else	{         /* Case 3: We got enough colors */
      
      /* We got colors, but they may not be contiguous (SGI, for example...).
       * Find the endpoints of the set of contiguous colors ending at the top
       * value of those allocated, and use that segment.
       * Free up the rest of the allocated colors, if any.
       */
      lutStartIndex = ncolors - 1;
      
      for (i=ncolors-1; i>0; i--) {
	if (pixels[i-1] != (pixels[i] - 1)) break;
	lutStartIndex = i-1;
      }
      
      lut_start = pixels[lutStartIndex]; 
      lutEnd = pixels[ncolors-1];
      ncolors = lutEnd - lut_start + 1;  

      /* Free the short, unused block of colors */
      if (lutStartIndex != 0) {   
	XFreeColors (disp, cmap, pixels, lutStartIndex, 0);
      } 

      /* Send default lut data to server for other clients to use */
      if( *colormap_level == DEFAULT_SCREEN_COLORMAP && 
	 tfWeWroteColorAtom == False  ) {
	writeSharedColorAtom (disp,cmap, lut_start, ncolors);
	*atom = 1;
	tfWeWroteColorAtom = True;  /* We wrote color atom, and should delete
				       it upon exiting */
      }
    }    /* End of third case */
  }

  for (color_def=lut_start; color_def<lut_start+ncolors;
       color_def++) {
    lut_colorcell_defs[color_def].pixel = color_def;
    lut_colorcell_defs[color_def].flags = (DoRed | DoGreen | DoBlue);
  }

  *colormap = cmap;
  *start_lut = lut_start;
  *numcolors = ncolors;

  free(plane_masks);  plane_masks = NULL;
  free(pixels);       pixels = NULL;
  
  /* Load the gray color pattern */
  gray(disp,cmap,chColor,ncolors,lut_start,False);                          
  
  return 1;
} /* End of init_colors */

void deinit_disp(Display *disp)
{
  /* This routine deinitializes the display.
    If this copy of the program wrote the X server information about
    the shared colormap, it destroys that information so that the
    colortool and other clients will be aware that no colors are
    reserved for them.
    */

  /* Destroy X server info re shared colormap */
#ifdef DEBUG
  printf(" deinit_disp \n");
#endif

   if (tfWeWroteColorAtom == True) {
#ifdef DEBUG
     printf("destroying \n");
#endif
     destroySharedColorAtom(disp);
   }
   XFlush(disp);

} /* end deinit_disp */

XVisualInfo *get_visual(Display *disp) 
     /* Pointer to list of matching visuals */
{
  XVisualInfo vTemplate;   /* Template of the visual we want */
  XVisualInfo *visualList; /* Pointer to list of matching visuals */
  int visualsMatched;      /* Number of visuals matched */
  int screenIndex;
  int depth;
 
  screenIndex = DefaultScreen(disp);       /* Screen index */

  depth = DisplayPlanes(disp, screenIndex); 
  root_window	= RootWindow(disp, screenIndex);
 
  /* Set up the template for getting visual info.  Select some or
     all of these depending on the VisualMask. */
  
  vTemplate.screen = screenIndex;  /* Must be on this screen */
  vTemplate.depth = depth;         /* Must have depth of default vis. */
  vTemplate.class = PseudoColor;   /* Must be PseudoColor */
  
  /* Find all PseudoColor visuals of depth "depth".  Accept the first. */
  visualList = XGetVisualInfo (disp,
			       VisualScreenMask | VisualDepthMask | VisualClassMask,
			       &vTemplate,
			       &visualsMatched);

  if (visualList == NULL) {
    printf ("get_visual:  No matching visuals.  Exiting.\n");
    exit (0);
  }
  
  if (visualsMatched != 1) {
    printf ("init_disp:  Too many matching visuals.  Taking only the first\n");
  }
 
   return(visualList);        
} /* End of "get_visual" */


/* get lut entries for subsequent lut manipulation */
void get_lut(XColor *chColor,
	     int ncolors,int lut_start,
	     int *red,int *green,int *blue)
{
  int        i;

  for (i=0; i<ncolors; i++) {
    red[i] = (int) (chColor[i].red) ;
    green[i] = (int) (chColor[i].green) ;
    blue[i] = (int) (chColor[i].blue) ;
  }
  return;
} /* end of "get_lut" */



/* Copy new lut into cmap */
/* Copy ncolors colors from argument arrays to color cells, and
   load into cmap to change the colors on the X display */

void put_lut(Display *disp,Colormap cmap,XColor *chColor,
	     int ncolors,int lut_start,char overlay,
	     int *red,int *green,int *blue)
{
   int 	i,j;
   int index;
   int background_index;

   if(overlay == False) {
     for (i=lut_start,j=0; j<ncolors; i++,j++) {
       if(j<n1) {
	 index = j*fn1/n1;
	 lut_colorcell_defs[i].pixel = i;
	 lut_colorcell_defs[i].red   = (unsigned short) (red[index]<<8) ;
	 lut_colorcell_defs[i].green = (unsigned short) (green[index]<<8) ;
	 lut_colorcell_defs[i].blue  = (unsigned short) (blue[index]<<8) ;
	 lut_colorcell_defs[i].flags = DoRed | DoBlue | DoGreen;
       }
       else if (j<n2) {
	 index = (j-n1)*(fn2-fn1)/(n2-n1)+fn1;
	 lut_colorcell_defs[i].pixel = i;
	 lut_colorcell_defs[i].red   = (unsigned short) (red[index]<<8) ;
	 lut_colorcell_defs[i].green = (unsigned short) (green[index]<<8) ;
	 lut_colorcell_defs[i].blue  = (unsigned short) (blue[index]<<8) ;
	 lut_colorcell_defs[i].flags = DoRed | DoBlue | DoGreen; 
       }
       else {
	 index = (j-n2)*(ncolors-fn2)/(ncolors-n2)+fn2;
	 lut_colorcell_defs[i].pixel = i;
	 lut_colorcell_defs[i].red   = (unsigned short) (red[index]<<8) ;
	 lut_colorcell_defs[i].green = (unsigned short) (green[index]<<8) ;
	 lut_colorcell_defs[i].blue  = (unsigned short) (blue[index]<<8) ;
	 lut_colorcell_defs[i].flags = DoRed | DoBlue | DoGreen;
       }
       
       chColor[j].red   = (unsigned short) red[j] ;
       chColor[j].green = (unsigned short) green[j] ;
       chColor[j].blue  = (unsigned short) blue[j] ;
     }
   } else {
     for (i=lut_start,j=0; j<ncolors; i++,j++) {
       if(j<n1) {
	 index = j*fn1/n1;
	 lut_colorcell_defs[i].pixel = i;
	 if( i%2==0) {
	   lut_colorcell_defs[i].red   = (unsigned short) (red[index]<<8) ;
	   lut_colorcell_defs[i].green = (unsigned short) (green[index]<<8) ;
	   lut_colorcell_defs[i].blue  = (unsigned short) (blue[index]<<8) ;
	 } else {
	   lut_colorcell_defs[i].red   = (unsigned short) 0xfffffff;
	   if( index < 50 )
	     background_index = ncolors-1-50;
	   else background_index = ncolors-1-index;
	   lut_colorcell_defs[i].green = (unsigned short) (green[background_index]<<8) ;
	   lut_colorcell_defs[i].blue  = (unsigned short) (blue[background_index]<<8) ;
	 }
	 lut_colorcell_defs[i].flags = DoRed | DoBlue | DoGreen;
       }
       else if (j<n2) {
	 index = (j-n1)*(fn2-fn1)/(n2-n1)+fn1;
	 lut_colorcell_defs[i].pixel = i;
	 if( i%2==0) {
	   lut_colorcell_defs[i].red   = (unsigned short) (red[index]<<8) ;
	   lut_colorcell_defs[i].green = (unsigned short) (green[index]<<8) ;
	   lut_colorcell_defs[i].blue  = (unsigned short) (blue[index]<<8) ;
	 } else {
	   lut_colorcell_defs[i].red   = (unsigned short) 0xfffffff;
	   if( index < 50 )
	     background_index = ncolors-1-50;
	   else background_index = ncolors-1-index;
	   lut_colorcell_defs[i].green = (unsigned short) (green[background_index]<<8) ;
	   lut_colorcell_defs[i].blue  = (unsigned short) (blue[background_index]<<8) ;
	 }
	 lut_colorcell_defs[i].flags = DoRed | DoBlue | DoGreen; 
       }
       else {
	 index = (j-n2)*(ncolors-fn2)/(ncolors-n2)+fn2;
	 lut_colorcell_defs[i].pixel = i;
	 if( i%2==0) {
	   lut_colorcell_defs[i].red   = (unsigned short) (red[index]<<8) ;
	   lut_colorcell_defs[i].green = (unsigned short) (green[index]<<8) ;
	   lut_colorcell_defs[i].blue  = (unsigned short) (blue[index]<<8) ;
	 } else {
	   lut_colorcell_defs[i].red   = (unsigned short) 0xfffffff ;
	   if( index < 50 )
	     background_index = ncolors-1-50;
	   else background_index = ncolors-1-index;
	   lut_colorcell_defs[i].green = (unsigned short) (green[background_index]<<8) ;
	   lut_colorcell_defs[i].blue  = (unsigned short) (blue[background_index]<<8) ;
	 }
	 lut_colorcell_defs[i].flags = DoRed | DoBlue | DoGreen;
       }
       chColor[j].red   = (unsigned short) red[j] ;
       chColor[j].green = (unsigned short) green[j] ;
       chColor[j].blue  = (unsigned short) blue[j] ;
     }
   }
   XStoreColors(disp,cmap,&lut_colorcell_defs[lut_start],ncolors) ;
   XFlush(disp) ;
   return;  
}

static void show_installed_colormaps(Display *disp)
{
  Colormap *colorMapList;  /* Pointer to list of installed colormaps */
  int i;                   /* Local looping variable */
  int num;                 /* Number of returned colormaps */

  printf ("In show_installed_colormaps\n");
  XFlush(disp) ;
  colorMapList = XListInstalledColormaps (disp, root_window, &num);
  XFlush(disp) ;
  printf ("Number of installed colormaps in root_window (%x) =%d\n",
	  (unsigned int)root_window, num);
  for (i=0; i<num; i++) {
    printf("colorMapList[%d]=%x\n", i,(unsigned int)( colorMapList[i]));
  }
  XFree (colorMapList);    /* Free the list of colormaps */
  return;
}




/* Print the type of a visual to stdout */
static void print_visual_class (Visual *visual)
{
  switch (visual->class) {
  case PseudoColor:
    printf ("Visual class is PseudoColor\n");
    break;
  case StaticColor:
    printf ("Visual class is StaticColor\n");
    break;
  case DirectColor:
    printf ("Visual class is DirectColor\n");
    break;
  case TrueColor:
    printf ("Visual class is TrueColor\n");
    break;
  case GrayScale:
    printf ("Visual class is GrayScale\n");
    break;
  case StaticGray:
    printf ("Visual class is StaticGray\n");
    break;
  default:
    printf ("Visual class is not known\n");
    break;
  }
  return;
}/* end of "print_visual_class" */



/* Write the shared color atom to give info about the shared colorlut */
void writeSharedColorAtom (Display *disp,Colormap cmap,int lutStart,int nPC)
{
   Atom       colormapAtom;  /* Atom for specifying colormap info to server */
   struct XColorSharedStruct XColorShared;

   XColorShared.cmap = cmap;
   XColorShared.lutStart = lutStart;
   XColorShared.nPC = nPC;
 
#ifdef DEBUG
   printf("writing color atom \n");
#endif

   /* Post the colormap in atom VIEW_COLORMAP */
   /* Create the atom */
   colormapAtom = XInternAtom (disp, "VIEW_COLORMAP", False);

   if (colormapAtom == None) {
     fprintf(stderr,
	     "ERROR in writeSharedColorAtom: XInternAtom returned None (%d)\n",
	     (int)colormapAtom);
   }

   /* Store XColorShared in the atom */
   XChangeProperty (disp, root_window, colormapAtom, XA_STRING, 8,
		    PropModeReplace, (unsigned char*)&XColorShared, sizeof(XColorShared));
   return;
}
/* End of "writeSharedColorAtom" */



/* Read the shared color atom to get information about the shared colorlut */
Status readSharedColorAtom (Display *disp,
			    Colormap *cmap, /* The colormap ID */
			    int *lutStart, /* The first available LUT value */
			    int *nPC)      /* The number of pseudocolor values used */
{
   Atom       colormapAtom;  /* Atom for specifying colormap info to server */
   int status;
   Atom actualType;
   int actualFormat;
   unsigned long nitems;
   unsigned long bytesAfter;
   struct XColorSharedStruct *theColorShared=NULL;

   /* Create the atom */
   colormapAtom = XInternAtom (disp, "VIEW_COLORMAP", True);

   if (colormapAtom == None) {
     printf("ERROR in readSharedColorAtom: XInternAtom returned None (%d)\n",
	    (unsigned int)colormapAtom);
     cmap = NULL;
     *lutStart = *nPC = 0;
     return BadAtom;
   }

   status = XGetWindowProperty (disp, root_window, colormapAtom,
				0L, 1000L, False, AnyPropertyType,
				&actualType, &actualFormat, &nitems,
				&bytesAfter,(unsigned char **)&theColorShared);
   
   if ((status == Success) && (theColorShared != NULL)) {
     *cmap = theColorShared->cmap;
     *lutStart = theColorShared->lutStart;
     *nPC = theColorShared->nPC;
     
     XFree (theColorShared); 
     theColorShared=NULL;
     return status;
   }
   else {
      
     switch (status) {
     case Success:
       if (actualType == None) {
	 /* printf ("actualType=None\n"); */
	 return BadAtom;
       }
       break;
     case BadAtom :
       printf("bad atom\n");
       break;
     case BadMatch :
       printf("bad match\n");
       break;
     case BadValue :
       printf("bad value\n");
       break;
     case BadWindow :
       printf("bad window\n");
       break;
     default:
       printf("bad other\n");
       break;
     }              /* End of switch (status) */
	 
     printf("ERROR in readSharedColorAtom: XGetWindowProperty returned %d\n",
	    status);
     cmap = NULL;
     *lutStart = *nPC = 0;
     return status;
   }
 }
/* End of readSharedColorAtom */


/* Destroy the shared colormap atom, in preparation to exit the program */
void destroySharedColorAtom(Display *disp) 
{
  Atom       colormapAtom;  /* Atom for specifying colormap info to server */

  /* Create the atom */
  colormapAtom = XInternAtom (disp, "VIEW_COLORMAP", True);

  if (colormapAtom == None) {
    printf("ERROR in destroySharedColorAtom: XInternAtom returned None (%d)\n",
	   (int)colormapAtom);
  }

  /* Destroy the property, in preparation to exit */
  XDeleteProperty (disp, root_window, colormapAtom);
  return;
}
/* End of "destroySharedColorAtom" */



