/*
 * dpIO.c --
 *
 *	Handle the some I/O callbacks from TCL.
 *
 * Copyright (c) 1993 The Regents of the University of California.
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#ifdef _WINDOWS
# include <windows.h>
#endif

#include <fcntl.h>
#include <tk.h>
#include "dpInt.h"


DP_SockInfo *dpSockInfoList;

/*
 *----------------------------------------------------------------------
 * Tdp_RemoveSocket --
 *
 *	This procedure is used to remove a DP_SockInfo from the DP_SockInfo
 *	table.
 *
 * Results:
 *	Standard TCL result
 *----------------------------------------------------------------------
 */
int
Tdp_RemoveSocket(interp, infoPtr)
    Tcl_Interp *interp;
    DP_SockInfo *infoPtr;
{
    DP_SockInfo *curr, *prev, *next;

    for (prev = NULL, curr = dpSockInfoList; curr != NULL;
	 prev = curr, curr = next)
    {
	next = curr->next;
	if (curr == infoPtr) {
#ifdef _WINDOWS
	    extern void Tknt_UnregisterSocket(int);
	    Tknt_UnregisterSocket(curr->sockId);
#endif
	    if (prev == NULL) {
		dpSockInfoList = next;
	    } else {
		prev->next = next;
	    }
	    ckfree((char *) curr);
	    return TCL_OK;
	}
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tdp_EnterSocket --
 *
 *	This procedure is used to enter an already-open file into the
 *	dp_sockinfo table for an interpreter so that the socket can be read
 *	and written with Tcl commands.
 *
 * Results:
 *	The standard TCL result and interp->result is set to
 *	hold Tdp's id for the open socket, such as "socket1039".
 *
 * Side effects:
 *	"File" is added to the files accessible from interp.
 *
 *----------------------------------------------------------------------
 */

int
Tdp_EnterSocket(interp, socket, permissions, infoPtrPtr)
    Tcl_Interp *interp;		/* Interpreter in which to make file
				 * available. */
    DP_SOCKET socket;		/* Socket to make available in interp */
    int permissions;		/* Ops that may be done on socket:  OR-ed
				 * combinination of TCL_READABLE and
				 * TCL_WRITABLE. */
    DP_SockInfo **infoPtrPtr;	/* The dp_sockinfo is returned through this */
{
    FILE *fp;
    int fd;
    DP_SockInfo *infoPtr;
    Tcl_Channel chan;
#ifdef _WINDOWS
    extern void Tknt_RegisterSocket(int, SOCKET);
    static int UseOsfOpenHandle = -1;

    if (UseOsfOpenHandle == -1) {
	OSVERSIONINFO version;
	version.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	if (GetVersionEx(&version)) {
	    if (version.dwPlatformId == VER_PLATFORM_WIN32_NT &&
		((version.dwMajorVersion > 3) ||
		 (version.dwMajorVersion == 3 && version.dwMinorVersion > 10)))
	    {
		UseOsfOpenHandle = 1;
	    } else {
		UseOsfOpenHandle = 0;
	    }
	}
    }

    /*
     * In Windows NT 3.1 and Windows 95, you cannot open a socket as a
     * file handle.  So we use a kludge to get a unique file identifier
     */
    if (!UseOsfOpenHandle) {
#define TCL_TMPDIR "\\temp"
	char *fakeName;

	fakeName = (char *) tempnam(TCL_TMPDIR, "dp");
	fp = fopen(fakeName, "w");
	unlink(fakeName);
	free(fakeName);
	fd = fileno(fp);
    } else {
	fd = _open_osfhandle(socket, _O_RDWR);
	fp = fdopen (fd, "r+");
    }
    Tknt_RegisterSocket(fd, socket);
#else
    /*
     * Open the file with the correct type
     */
    fd = socket;
    fp = fdopen (fd, "r+");
#endif
    clearerr(fp);

    /*
     * Turn off buffering.  Otherwise, we run into nasty interaction
     * problems with gets/puts/read and our transmission commands below.
     */
    setbuf (fp, (char *) NULL);

    /* Tcl_EnterFile(interp, fp, TCL_READABLE|TCL_WRITABLE); */
    chan = Tcl_MakeFileChannel(fd, fd, TCL_READABLE|TCL_WRITABLE);
    Tcl_RegisterChannel(interp, chan);
    Tcl_SetChannelOption(interp, chan, "-buffering", "none");
    
    infoPtr = (DP_SockInfo *) ckalloc(sizeof(DP_SockInfo));
    infoPtr->socket = socket;
    infoPtr->sockId = fd;
    infoPtr->filePtr = fp;
    infoPtr->optFlags = 0;
    infoPtr->partial = NULL;
    infoPtr->handlers = NULL;
    infoPtr->permissions = permissions;
    infoPtr->error = 0;

    infoPtr->rpcWaiting = 0;
    infoPtr->rpcCheck = NULL;
    infoPtr->rpcResult = TCL_OK;
    infoPtr->rpcValue = NULL;
    infoPtr->rpcFlush = 0;
    infoPtr->rpcTimeoutReturn = NULL;
    infoPtr->timerToken = (ClientData) -1;
    infoPtr->timerInterp = NULL;

    infoPtr->next = dpSockInfoList;
    dpSockInfoList = infoPtr;
    *infoPtrPtr = infoPtr;
    sprintf(interp->result, "file%u", (unsigned int) fd);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tdp_GetOpenSocket --
 *
 *	Given a string identifier for an open socket, find the corresponding
 *	open socket structure, if there is one.
 *
 * Results:
 *	A standard Tcl return value.  If the open socket is successfully
 *	located and meets any usage check requested by checkUsage, TCL_OK
 *	is returned and *infoPtrPtr is modified to hold a pointer to its
 *	FILE structure.  If an error occurs then TCL_ERROR is returned
 *	and interp->result contains an error message.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tdp_GetOpenSocket(interp, string, forWriting, checkUsage, infoPtrPtr)
    Tcl_Interp *interp;		/* Interpreter in which to find socket. */
    char *string;		/* String that identifies socket. */
    int forWriting;		/* 1 means the socket is going to be used
				 * for writing, 0 means for reading. */
    int checkUsage;		/* 1 means verify that the socket was opened
				 * in a mode that allows the access specified
				 * by "forWriting". */
    DP_SockInfo **infoPtrPtr;	/* Store pointer to DP_SockInfo here. */
{
    DP_SockInfo *infoPtr;
    unsigned int sockid;

    static int last_sockid = -1;
    static DP_SockInfo *last_infoPtr = NULL;


    if ((string[0] == 'f') && (string[1] == 'i') && (string[2] == 'l')
	    & (string[3] == 'e')) {
	char *end;

	sockid = strtoul(string+4, &end, 10);
	if ((end == string+4) || (*end != 0)) {
	    goto badId;
	}
    } else {
	badId:
	Tcl_AppendResult(interp, "bad socket identifier \"", string,
		"\"", (char *) NULL);
	return TCL_ERROR;
    }

#if 0    
    /* A quick cache check of the last found socket */
    if (sockid == last_sockid && sockid != -1) {
	*infoPtrPtr = last_infoPtr;
	return TCL_OK;
    }
#endif

    for (infoPtr = dpSockInfoList; infoPtr != NULL; infoPtr = infoPtr->next) {
	if (infoPtr->sockId == sockid) {
	    *infoPtrPtr = infoPtr;
	    break;
	}
    }

    if (infoPtr == NULL) {
	sprintf(interp->result, "file \"%s\" isn't open", string);
	return TCL_ERROR;
    }

    if (forWriting) {
	if (checkUsage && !(infoPtr->permissions & TCL_WRITABLE)) {
	    Tcl_AppendResult(interp, "\"", string,
		    "\" wasn't opened for writing", (char *) NULL);
	    return TCL_ERROR;
	}
    } else {
	if (checkUsage && !(infoPtr->permissions & TCL_READABLE)) {
	    Tcl_AppendResult(interp, "\"", string,
		    "\" wasn't opened for reading", (char *) NULL);
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}
