/*
 * dpnetwork.c --
 *
 * This file implements most of the network connection management
 * functions of Tcl-DP.  The following comments are inherited from
 * the progenitors of Tcl-DP.
 * This file contains a simple Tcl "dp_connect" command
 * that returns an standard Tcl File descriptor (as would
 * be returned by Tcl_OpenCmd).  This part of the file was written by
 * Pekka Nikander <pnr@innopoli.ajk.tele.fi>
 *
 * Tim MacKenzie <tym@dibbler.cs.monash.edu.au> extended it to
 * create servers, accept connections, shutdown parts of full
 * duplex connections and handle UNIX domain sockets.
 *
 * Brian Smith <bsmith@cs.berkeley.edu> further modified it to
 * add support for various send/receive primitives, and connectionless
 * sockets.
 *
 * Copyright 1992 Telecom Finland
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  Telecom Finland
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 * Copyright (c) 1993 The Regents of the UNIVERSITY OF CALIFORNIA.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 * Copyright (c) The Regents of 1995 Cornell University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 *
 * IN NO EVENT SHALL CORNELL UNIVERSITY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * CORNELL UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, CORNELL UNIVERSITY HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */
#ifdef _WINDOWS
#    include <windows.h>
#    include <io.h>
#endif
#include <stdio.h>
#include <assert.h>
#include <ctype.h>
#include "tk.h"
#include "dpInt.h"
#include <netinet/in.h>

#ifndef MULTICAST
#    include "compat/in.h"
#endif

#include <arpa/inet.h>
#include <netdb.h>

#ifdef HAVE_SYS_UN_H
#    include <sys/un.h>
#endif

#include <sys/param.h>
#include <sys/ioctl.h>
#ifndef NO_WRITEV
#    include <sys/uio.h>
#endif

#ifdef HAVE_FCNTL_H
#    include <fcntl.h>
#endif

#ifndef _WINDOWS
extern int errno;
#    if HAVE_ERRNO_H
#        include <errno.h>
#else
#        define EAGAIN 11
#        define EINVAL 22
#        define EWOULDBLOCK 35
#        define EPIPE 32
#	 define	ENOPROTOOPT 42
#    endif
#endif

#ifdef SOLARIS
#    include <stropts.h>
#    include <sys/conf.h>
#    include <sys/systeminfo.h>
#    define MAXHOSTNAMELEN	256	/* from the man page */
#    define gethostname(BUF,MAXLEN)	our_gethostname(BUF,MAXLEN)
#endif

#ifndef max
#    define max(a,b)    ((a)>(b)?(a):(b))
#endif
#ifndef min
#    define min(a,b)    ((a)<(b)?(a):(b))
#endif
#ifndef _WINDOWS
#    ifndef abs
#        define abs(a)  ((a)>0?(a):-(a))
#    endif
#endif

/*
 * This is a "magic number" prepended to the beginning of the packet
 * It's used to help resync the packet machanism in the event of errors.
 */
#define PACKET_MAGIC    0x6feeddcc

DP_SOCKET Tdp_inet_connect	_ANSI_ARGS_((char *host, int port,
						     int server, int udp,
						     int reuseAddr,
						     int lingerTime));
#ifdef HAVE_SYS_UN_H
static int Tdp_unix_connect		_ANSI_ARGS_((char *path,
						     int server, int udp));
#endif

static void Tdp_HandleEvent		_ANSI_ARGS_((ClientData clientData,
						     int mask));
static void Tdp_FreeHandler		_ANSI_ARGS_((ClientData clientData));

/*
 * For every file descriptor handler created, a structure of
 * the following type is maintained.
 */
struct DP_FileHandle {
    Tcl_Interp *interp;
    DP_SockInfo *infoPtr;	/* Open file descriptor (file or socket) */
    int mask;			/* Mask of file descriptor conditions */
    char *rCmd;			/* Command to call on readable condition */
    char *wCmd;			/* Command to call on writable condition */
    char *eCmd;			/* Command to call on exception condition */
    char *fileId;		/* Represents filePtr */

};

static DP_FileHandle *handlers[MAX_OPEN_FILES];	/* Indexed by fd. */

/*
 * We keep around a single, large buffer which we can receive data into.
 * The size of this buffer is the maximum size of any of the receive buffers
 * on any open sockets, stored in bufferSize.
 */
static char *buffer;	/* Buffer for receiving data */
static int bufferSize;	/* Size of buffer */

/*
 * For TCP, it's possible to get a line in pieces.  In case everything we
 * want isn't there (e.g., in dp_packetReceive), we need a place to store
 * partial results when we're in non-blocking mode or peeking at the data.
 * The partial buffers below are created dynamically to store incomplete
 * data in these cases.
 */
struct DP_PartialRead {
    char *buffer;		/* Buffer of characters */
    int bufSize;		/* Size of buffer */
    int offset;			/* Offset of current character within the
			         * buffer */
    struct DP_PartialRead *next;/* Next buffer in chain */
};

/* static PartialRead *partial[MAX_OPEN_FILES]; */

/*
 * The next array stores state about each socket.  The optFlags is an or'd
 * combination of the following state:
 *  FD_BLOCKING   -- Blocking I/O mode is on
 *  FD_GOTPARTIAL -- Have received a partial message (only applicable for TCP)
 *  FD_TCP        -- Is a TCP/IP line (otherwise udp)
 *  FD_UNIX       -- Is a unix domain sokcet (otherwise internet)
 *  FD_SERVER     -- Was created with -server
 *  FD_AUTO_CLOSE -- Socket should auto close on error.
 */
/* static unsigned char optFlags[MAX_OPEN_FILES]; */
#define FD_BLOCKING 1
#define FD_GOTPARTIAL   2
#define FD_TCP      4
#define FD_UNIX     8
#define FD_SERVER   16
#define FD_AUTO_CLOSE   32

#ifdef SOLARIS
/*
 *--------------------------------------------------------------
 *
 * our_gethostname
 *
 *	Mimic the BSD gethostname call with the POSIX sysinfo call.
 *
 * Results:
 *	0 if successful, -1 on error.  (See gethostname(2).)
 *
 * Side effects:
 *	Set name to the hostname.
 *
 * Notes:
 *
 *	Didn't use a macro to preserve return value.
 *
 *--------------------------------------------------------------
 */
int our_gethostname(char *name, int namelen)
{
  int count;

  count = sysinfo(SI_HOSTNAME, name, namelen);
  return (count == -1) ? (-1) : (0);
}
#endif

/*
 *--------------------------------------------------------------
 *
 * Tdp_SetBlocking --
 *
 *	Make the socket blocking (or non-blocking) as specified,
 *	and be efficient about it (i.e., cache the current state).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The socket whose file descriptor is passed in will be either
 *	blocking or not, as specified, after this call.
 *
 *--------------------------------------------------------------
 */
static void Tdp_SetBlocking(infoPtr, noblock)
    DP_SockInfo *infoPtr;	/* Socket info */
    int noblock;		/* 0 if we should block from now on, 1 if not */
{
/*
 * HPUX need the flag O_NONBLOCK to get the right non-blocking I/O
 * semantics, while most other systems need O_NDELAY.  Define the
 * constant NBIO_FLAG to be one of these
 */
#ifdef HPUX
#  define NBIO_FLAG O_NONBLOCK
#else
#  define NBIO_FLAG O_NDELAY
#endif

#ifndef _WINDOWS
    int flags;
    if (noblock) {
	if ((infoPtr->optFlags & FD_BLOCKING) == 0) {
	    /* Set blocking mode */
	    flags = fcntl(infoPtr->socket, F_GETFL, 0);
	    fcntl(infoPtr->socket, F_SETFL, flags & ~NBIO_FLAG);
	    infoPtr->optFlags |= FD_BLOCKING;
	}
    } else {
	if (infoPtr->optFlags & FD_BLOCKING) {
	    /* Set non-blocking mode */
	    flags = fcntl(infoPtr->socket, F_GETFL, 0);
	    fcntl(infoPtr->socket, F_SETFL, flags | NBIO_FLAG);
	    infoPtr->optFlags &= ~FD_BLOCKING;
	}
    }
#else
    if (noblock) {
	if ((infoPtr->optFlags & FD_BLOCKING) == 0) {
	    /* Set blocking mode */
	    u_long val = 0;
	    infoPtr->optFlags |= FD_BLOCKING;
	    ioctlsocket(infoPtr->socket, FIONBIO, (u_long *) &val);
	}
    } else {
	if (infoPtr->optFlags & FD_BLOCKING) {
	    /* Set non-blocking mode */
	    u_long val = 1;
	    ioctlsocket(infoPtr->socket, FIONBIO, (u_long *) &val);
	    infoPtr->optFlags &= ~FD_BLOCKING;
	}
    }
#endif
}

/*
 *--------------------------------------------------------------
 *
 * Tdp_GetBufferSize --
 *
 *	Get the size of the receive buffer on a socket.
 *
 * Results:
 *	The size of the receive buffer of the specified socket, in bytes,
 *	or -1 on error.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static int
Tdp_GetBufferSize(infoPtr)
    DP_SockInfo *infoPtr;
{
    int optlen, optval, result;
    optlen = sizeof(int);
    result = getsockopt(infoPtr->socket, SOL_SOCKET, SO_RCVBUF, (char *) &optval, &optlen);
    if (result == -1) {
	return -1;
    } else {
	return optval;
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tdp_AllocateBuffer --
 *
 *	This command is called to allocate (or reallocate) the global
 *	receive buffer when the file descriptor passed in is created or
 *	modified.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The global variable "buffer" is (re)allocated
 *
 *--------------------------------------------------------------
 */
static void
Tdp_AllocateBuffer(infoPtr)
    DP_SockInfo *infoPtr;	/* Info about socket created/modified */
{
    /*
     * Get the size of the send/receive buffer, and make sure the buffer we
     * have is big enough to receive the largest possible message.
     */
    if (buffer == NULL) {
	bufferSize = Tdp_GetBufferSize(infoPtr) + 32;
	buffer = ckalloc(bufferSize);
    } else if (Tdp_GetBufferSize(infoPtr) > bufferSize) {
	bufferSize = Tdp_GetBufferSize(infoPtr) + 32;
	buffer = ckrealloc(buffer, bufferSize);
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tdp_SocketIsReady --
 *
 *	This function determines if a file descriptor is readable
 *	or writeable.
 *
 * Results:
 *	An or'd combination of TCL_READABLE and TCL_WRITABLE
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
int
Tdp_SocketIsReady(socket)
    DP_SOCKET socket;
{
    fd_set readFdset;
    fd_set writeFdset;
    struct timeval tv;
    struct timeval *tv_ptr;
    int rv, srv;

    FD_ZERO(&readFdset);
    FD_SET(socket, &readFdset);
    FD_ZERO(&writeFdset);
    FD_SET(socket, &writeFdset);

    tv.tv_sec = 0;
    tv.tv_usec = 0;
    tv_ptr = &tv;

    srv = select(socket + 1, (SELECT_MASK *) & readFdset,
		 (SELECT_MASK *) & writeFdset, (SELECT_MASK *) NULL, tv_ptr);
    if (FD_ISSET(socket, &readFdset)) {
	rv = TCL_READABLE;
    } else {
	rv = 0;
    }
    if (FD_ISSET(socket, &writeFdset)) {
	rv |= TCL_WRITABLE;
    }
    return rv;
}

/*
 *--------------------------------------------------------------
 *
 * Tdp_FreeReadBuffer --
 *
 *	This function is called to free up all the memory associated
 *	with a file once the file is closed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Any data buffered locally will be lost.
 *
 *--------------------------------------------------------------
 */
static void
Tdp_FreeReadBuffer(infoPtr)
    DP_SockInfo *infoPtr;
{
    DP_PartialRead *readList;
    while (infoPtr->partial != NULL) {
	readList = infoPtr->partial;
	infoPtr->partial = readList->next;
	ckfree(readList->buffer);
	ckfree((char *) readList);
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tdp_BlockSocket --
 *
 *	This function is needed under Windows NT to get proper
 *	blocking behavior before with calls to recv().  This
 *	is because the call to WSAAsyncSelect() in tkEvent.c
 *	causes the socket to become non-blocking.  dpsh.exe does
 *	not have this problem because it uses an event loop that
 *	does not need to call WSAAsyncSelect().  The call to
 *	select() will only occur if the socket is a blocking socket.
 *	if the socket is not blocking, this function just returns.
 *
 * Results: None
 *
 * Side Effects:
 *	This function will return when an event occurs on the
 *	socket
 *
 *--------------------------------------------------------------
 */
#ifdef _WINDOWS
static __inline void
Tdp_BlockSocket(infoPtr, mask)
    DP_SockInfo *infoPtr;
    int mask;
{
    /* This next piece of code gets around a problem that occurs with
     * WSAAsyncSelect(): the socket becomes non-blocking when this call
     * is used.  Wait for select() to tell us something is available
     */
    if (infoPtr->optFlags & FD_BLOCKING) {
	fd_set ready[3];
	fd_set *readPtr, *writePtr, *exceptPtr;

	FD_ZERO(&ready[0]);
	FD_ZERO(&ready[1]);
	FD_ZERO(&ready[2]);

	if (mask & TK_READABLE) {
	    FD_SET(infoPtr->socket, &ready[0]);
	    readPtr = &ready[0];
	} else {
	    readPtr = NULL;
	}

	if (mask & TK_WRITABLE) {
	    FD_SET(infoPtr->socket, &ready[1]);
	    writePtr = &ready[1];
	} else {
	    writePtr = NULL;
	}

	if (mask & TK_EXCEPTION) {
	    FD_SET(infoPtr->socket, &ready[2]);
	    exceptPtr = &ready[2];
	} else {
	    exceptPtr = NULL;
	}

	select(1, readPtr, writePtr, exceptPtr, NULL);
    }
}
#else
#define Tdp_BlockSocket(x,y)
#endif /* ifdef _WINDOWS */
    
/*
 *--------------------------------------------------------------
 *
 * Tdp_Unread --
 *
 *	This function puts data back into the read chain on a
 *	file descriptor.  It's basically an extended "ungetc".
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Subsequent calls to Tdp_Read on the socket will get this data.
 *
 *--------------------------------------------------------------
 */
static void
Tdp_Unread(infoPtr, buffer, numBytes, copy, push)
    DP_SockInfo *infoPtr;	/* Socket info */
    char *buffer;		/* Data to unget */
    int numBytes;		/* Number of bytes to unget */
    int copy;			/* Should we copy the data, or use this
				 * buffer? */
    int push;			/* Should we use stack or list semantics.
				   If push=1, then the next call to Tdp_Read
				   will get this data.  If push=0, then
				   this data will be the last thing returned
				   by Tdp_Read() */
{
    DP_PartialRead *new;
    DP_PartialRead *readList;

    if (numBytes == 0)
	return;
    new = (DP_PartialRead *) ckalloc(sizeof(DP_PartialRead));
    if (copy) {
	new->buffer = ckalloc(numBytes);
	memcpy(new->buffer, buffer, numBytes);
    } else {
	new->buffer = buffer;
    }
    new->bufSize = numBytes;
    new->offset = 0;

    if (push || (infoPtr->partial == NULL)) {
        new->next = infoPtr->partial;
        infoPtr->partial = new;
    } else {
        readList = infoPtr->partial;
        while (readList->next != NULL) {
	    readList = readList->next;
        }
        readList->next = new;
        new->next = NULL;
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tdp_Read --
 *
 *	This function impplements a "recv"-like command, but
 *	buffers partial reads.  The semantics are the same as
 *	with recv.
 *
 * Results:
 *	Number of bytes read, or -1 on error (with errno set).
 *
 * Side effects:
 *	All available data is read from the socket
 *
 *--------------------------------------------------------------
 */
static int
Tdp_Read(infoPtr, buffer, numReq, flags)
    DP_SockInfo *infoPtr;	/* Socket info */
    char *buffer;		/* Place to put the data */
    int numReq;			/* Number of bytes to get */
    int flags;			/* Flags for receive */
{
    int peek;
    DP_PartialRead *readList;
    DP_PartialRead *tmp;
    int numRead;
    int numToCopy;
    readList = infoPtr->partial;

    /*
     * If there's no data left over from a previous read, then just do a recv
     * This is the common case.
     */
    if (readList == NULL) {
#if defined(SOLARIS) && (OS_MAJOR_VERSION == 5) && (OS_MINOR_VERSION <= 4)
	/* The following block of code is a workaround for a bug in 
	 * Solaris 2.4. It should be removed once this bugs is fixed
	 * (i.e., confident that everyone is running 2.5.
	 */

	Tdp_BlockSocket(infoPtr, TK_READABLE|TK_EXCEPTION);
	numRead = read(infoPtr->sockId, buffer, numReq);
	if (flags & MSG_PEEK) {
	    Tdp_Unread(infoPtr, buffer, numRead, 1, 0);
	}
	return numRead;
#else
	Tdp_BlockSocket(infoPtr, TK_READABLE|TK_EXCEPTION);
	numRead = recv(infoPtr->socket, buffer, numReq, flags);
	return numRead;
#endif
    }

    /*
     * There's data left over from a previous read.  Yank it in and only call
     * recv() if we didn't get enough data (this keeps the fd readable if
     * they only request as much data as is in the buffers).
     */
    peek = flags & MSG_PEEK;
    numRead = 0;
    while ((readList != NULL) && (numRead < numReq)) {
	numToCopy = readList->bufSize - readList->offset;
	if (numToCopy + numRead > numReq) {
	    numToCopy = numReq - numRead;
	}
	memcpy(buffer + numRead, readList->buffer + readList->offset, numToCopy);

	/*
	 * Consume the data if we're not peeking at it
	 */
	tmp = readList;
	readList = readList->next;
	if (!peek) {
	    tmp->offset += numToCopy;
	    if (tmp->offset == tmp->bufSize) {
		ckfree(tmp->buffer);
		ckfree((char *) tmp);
		infoPtr->partial = readList;
	    }
	}
	numRead += numToCopy;
    }

    /*
     * Only call recv if we reached the end of previously read data and they
     * didn't get enough and the socket has data to be consumed.
     */
    if ((numRead < numReq) && (Tdp_SocketIsReady(infoPtr->socket) & TCL_READABLE)) {

#if defined(SOLARIS) && (OS_MAJOR_VERSION == 5) && (OS_MINOR_VERSION <= 4)
	/* Solaris 2.4 bug workaround. */
	   
	int offset;

	Tdp_BlockSocket(infoPtr, TK_READABLE|TK_EXCEPTION);

	numToCopy = numReq - numRead;
	offset = numRead;

	numRead += read(infoPtr->sockId, buffer+offset, numToCopy);
	if (flags & MSG_PEEK) {
	    Tdp_Unread(infoPtr, buffer+offset, numRead-offset, 1, 0);
	}
#else
	Tdp_BlockSocket(infoPtr, TK_READABLE|TK_EXCEPTION);
	numToCopy = numReq - numRead;
	numRead += recv(infoPtr->socket, buffer + numRead, numToCopy, flags);
#endif

    }
    return numRead;
}


/*
 *------------------------------------------------------------------
 *
 * Tdp_GetPort --
 *
 *	Converts a string representing a service name or number to an integer.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *------------------------------------------------------------------
 */
int
Tdp_GetPort(interp, portstr, proto, portp)
    Tcl_Interp *interp;
    char *portstr, *proto;
    int *portp;
{
    int ret = TCL_ERROR;
    char *cp;
    for (cp = portstr; *cp != '\0'; cp++) {
	if (!isascii(*cp) || !isdigit(*cp))
	    break;
    }
    if (*cp != '\0') {
	/* string has a non-digit, must be a service name */
	struct servent *sp = getservbyname(portstr, proto);
	if (sp != NULL) {
	    *portp = ntohs(sp->s_port);
	    ret = TCL_OK;
	}
    } else {
	ret = Tcl_GetInt(interp, portstr, portp);
	if (*portp > 0xFFFF)
	    return (TCL_ERROR);
    }
    return ret;
}

/*
 *----------------------------------------------------------------
 *
 * Tdp_multicast_connect --
 *
 *	Create a multicast IP socket connection to given group and port
 *	with a given ttl.
 *
 * Results:
 *	An open socket or DP_INVALID_SOCKET.
 *
 * Side effects:
 *	None.
 *----------------------------------------------------------------
 */

static DP_SOCKET
Tdp_multicast_connect(group, port, ttl)
    char *group;		/* Group to connect to, name or IP address */
    int port;			/* Port number to use */
    unsigned char ttl;		/* Time to live */
{
    struct hostent *hostent, _hostent;
    struct sockaddr_in sockaddr;
    struct ip_mreq mreq;
    DP_SOCKET sock;
    int status;
    int hostaddr, hostaddrPtr[2];
    char localhost[MAXHOSTNAMELEN];
    unsigned char loop = 1;
    int one = 1;
    char *bp;
    int dotNotation;

    /*
     * Set default group to be localhost
     */
    if (group == NULL) {
	gethostname(localhost, MAXHOSTNAMELEN);
	group = localhost;
    }
    dotNotation = 1;
    bp = group;
    for (bp = group; *bp; bp++) {
	if (! isdigit(*bp) && *bp != '.') {
	    dotNotation = 0;
	    break;
	}
    }

    if (dotNotation == 0) {
	hostent = gethostbyname(group);
    } else {
	hostent == NULL;
    }
    if (hostent == NULL) {
	if (strlen(group) == 0) {
	    hostaddr = 0;
	} else {
	    hostaddr = inet_addr(group);
	}
	if (hostaddr == -1) {
	    DP_SetLastError(DP_EINVAL);
	    return DP_INVALID_SOCKET;
	}
	_hostent.h_addr_list = (char **) hostaddrPtr;
#ifdef CRAY_HACKS
	hostaddr <<= 32;
#endif
	_hostent.h_addr_list[0] = (char *) &hostaddr;
	_hostent.h_addr_list[1] = NULL;
	_hostent.h_length = sizeof(hostaddr);
	_hostent.h_addrtype = AF_INET;
	hostent = &_hostent;
    }
    sock = socket(PF_INET, SOCK_DGRAM, 0);
    if (sock == DP_INVALID_SOCKET) {
	return DP_INVALID_SOCKET;
    }
    memset(&sockaddr, 0, sizeof(sockaddr));
    sockaddr.sin_family = AF_INET;

/*
#ifdef CRAY_HACKS
    {
    unsigned long foo;

    memcpy((char *)&foo,
       (char *)hostent->h_addr_list[0],
       (size_t)hostent->h_length);

    sockaddr.sin_addr.s_addr = foo>>32;
}
#else
    memcpy((char *)&(sockaddr.sin_addr.s_addr),
       (char *)hostent->h_addr_list[0],
       (size_t)hostent->h_length);
#endif
*/
    sockaddr.sin_port = htons((unsigned short) port);

    sockaddr.sin_addr.s_addr = htonl(INADDR_ANY);

    if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *)&one,
		   sizeof(one)) != 0) {
	closesocket(sock);
	return DP_INVALID_SOCKET;
    }

    status = bind(sock, (struct sockaddr *) & sockaddr, sizeof(sockaddr));

    if (status != 0) {
	closesocket(sock);
	return DP_INVALID_SOCKET;
    }
    sockaddr.sin_addr.s_addr = inet_addr(group);

    if (IN_MULTICAST(ntohl(sockaddr.sin_addr.s_addr))) {
	int size, len;
	u_char loop = 1;
	u_char buf[sizeof(int) * 2];
	u_char *p;

	memset(buf, 0, sizeof(int) * 2);
	mreq.imr_multiaddr.s_addr = inet_addr(group);
	mreq.imr_interface.s_addr = htonl(INADDR_ANY);
	if (setsockopt(sock, IPPROTO_IP, IP_ADD_MEMBERSHIP, (char *)&mreq,
		       sizeof(mreq)) != 0) {
	    closesocket(sock);
	    return DP_INVALID_SOCKET;
	}

	/*
	 * NT 3.5 seems to have a bug when passing a size of char down.
	 * It comes back with an error indicating a segfault.  When
	 * increased to the size of an int, everything seems to be OK.
	 * Pad on both sides of the value so NT doesn't segfault. 
	 */

	p = &buf[sizeof(int)];
	*p = ttl;
	if (setsockopt(sock, IPPROTO_IP, IP_MULTICAST_TTL, (char *)p, sizeof(ttl)) != 0) {
	    int err;
	    err = DP_GetLastError();
	    closesocket(sock);
	    return DP_INVALID_SOCKET;
	}
	ttl = *p;

	*p = loop;
	if (setsockopt(sock, IPPROTO_IP, IP_MULTICAST_LOOP, (char *)p, sizeof(loop)) != 0) {
	    int err;
	    err = DP_GetLastError();
	    /*
	     * If the machine does not support this option (like Windows NT),
	     * then ignore this error
	     */
	    if (err != DP_ENOPROTOOPT) {
		closesocket(sock);
		return DP_INVALID_SOCKET;
	    }
	}
	loop = *p;

	/*
	 * Make sure the send/recv buffers are of a reasonable size
	 * (this is a problem under Solaris 2.x) -- the default size is 0.
	 */
	size = 8192;
	len = sizeof(int);
	getsockopt(sock, SOL_SOCKET, SO_SNDBUF, (char *)&size, &len);
	if (size < 8192) {
	    size = 8192;
	}
	len = sizeof(int);
	setsockopt(sock, SOL_SOCKET, SO_SNDBUF, (char *)&size, sizeof(int));
	getsockopt(sock, SOL_SOCKET, SO_RCVBUF, (char *)&size, &len);
	if (size < 8192) {
	    size = 8192;
	}
	setsockopt(sock, SOL_SOCKET, SO_RCVBUF, (char *)&size, sizeof(int));
    } else {
	fprintf(stderr, "There was no such multicast address!\n");
	closesocket(sock);
	return DP_INVALID_SOCKET;
    }
    sockaddr.sin_addr.s_addr = htonl(INADDR_ANY);

    return sock;
}


/*
 *------------------------------------------------------------------
 *
 * Tdp_ConnectCmd --
 *
 *	This procedure is the C interface to the "dp_connect"
 *	command. See the user documentation for a description.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	An open socket connection.
 *
 *------------------------------------------------------------------
 */
int
Tdp_ConnectCmd(notUsed, interp, argc, argv)
    ClientData notUsed;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    DP_SOCKET socket;		/* Open file descriptor */
    DP_SockInfo *infoPtr;	/* Socket info */
    int port;			/* User specified port number */
    char *host;			/* Hostname (inet) */
#ifdef HAVE_SYS_UN_H
    Tcl_DString buffer;		/* Buffer for tilde substitution */
    char *pathname;		/* Pathname (unix) */
    int unixSocket;		/* Unix domain socket? */
#endif

    int mudp = 0;		/* UDP on multicast sockets? */
    int mserver = 0;		/* Set up listening multicast socket? */
    int mclient = 0;		/* Multicast client socket? */
    unsigned char ttl;		/* TTL for multicast services */

    int udp;			/* UDP protocol? */
    char *protoname = "tcp";	/* "udp" or "tcp" */
    int server;			/* Set up listening socket? */
    char tmp[256];
    int lingerTime;		/* For linger socket option */
    int reuseAddr;		/* Allow local reuse of TCP addresses */
    host = NULL;
    udp = 0;
    server = 0;
    lingerTime = 0;
    reuseAddr = 0;
#ifdef HAVE_SYS_UN_H
    unixSocket = 0;
    pathname = NULL;
    Tcl_DStringInit(&buffer);
#endif
    if (argc < 2) {
error:
	Tcl_SetResult(interp, argv[0], TCL_VOLATILE);
	Tcl_AppendResult(interp, ": should be one of the forms:\n",
			 (char *) NULL);
	Tcl_AppendResult(interp,
		     " \"dp_connect -server port ?-linger? ?-reuseAddr\"\n",
			 (char *) NULL);

	Tcl_AppendResult(interp, " \"dp_connect -mudp group port ttl\"\n",
			 (char *) NULL);
	Tcl_AppendResult(interp, " \"dp_connect host port\"\n",
			 (char *) NULL);
	Tcl_AppendResult(interp, "\"dp_connect -udp ?port? \"\n",
			 (char *) NULL);
#ifdef HAVE_SYS_UN_H
	Tcl_AppendResult(interp, " \"dp_connect -server path\"\n",
			 (char *) NULL);
	Tcl_AppendResult(interp, " or \"dp_connect path\"\n",
			 (char *) NULL);
#endif
	return TCL_ERROR;
    }

    /*
     * Break into one of three catergories: udp sockets, server setup client
     * setup multicast udp sockets
     */
    if (strcmp(argv[1], "-udp") == 0) {
	udp = 1;
	protoname = "udp";
    } else if (strcmp(argv[1], "-server") == 0) {
	server = 1;
    } else if (strcmp(argv[1], "-mudp") == 0) {
	mudp = 1;
    }
    if (udp) {
	/*
	 * Must be "dp_connect -udp ?port?"
	 */
	host = "";		/* Allow packets from any source */
	if (argc == 3) {
	    if (Tdp_GetPort(interp, argv[2], protoname, &port) != TCL_OK) {
		Tcl_AppendResult(interp, argv[0], ": bad port number!", (char *) NULL);
		return TCL_ERROR;
	    }
	} else if (argc == 2) {
	    port = 0;
	} else {
	    goto error;
	}
    } else if (server) {
	/*
	 * Must be either "dp_connect -server port ?-linger? ?-reuseAddr?" or
	 * "dp_connect -server path"
	 */
	if (argc < 3) {
	    goto error;
	}
	host = "";		/* Allow packets from any source */
	if (Tdp_GetPort(interp, argv[2], protoname, &port) != TCL_OK) {
#ifdef HAVE_SYS_UN_H
	    if (argc != 3) {
		goto error;
	    }
	    pathname = Tcl_TildeSubst(interp, argv[2], &buffer);
	    if (pathname == NULL) {
		return TCL_ERROR;
	    }
	    unixSocket = 1;
#else
	    goto error;
#endif
	} else if (argc > 3) {
	    int arg;
	    for (arg = 3; arg < argc; arg++) {
		/* This is a tcp connection */
		if (strcmp(argv[arg], "-linger") == 0) {
		    lingerTime = 1;
		} else if (strcmp(argv[arg], "-reuseAddr") == 0) {
		    reuseAddr = 1;
		} else {
		    goto error;
		}
	    }

	}
    } else if (mudp) {
	/*
	 * Must be "dp_connect -mudp group port ttl"
	 */

	int temp_ttl;
	if (argc != 5) {
	    goto error;
	}
	host = argv[2];
	if (Tcl_GetInt(interp, argv[3], &port) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Tcl_GetInt(interp, argv[4], &temp_ttl) != TCL_OK) {
	    return TCL_ERROR;
	}
	ttl = (unsigned char) temp_ttl;

    } else {
	/*
	 * Client setup. Must be one of: "dp_connect host port" or
	 * "dp_connect path"
	 */
	if (argc == 3) {
	    host = argv[1];
	    if (Tdp_GetPort(interp, argv[2], protoname, &port) != TCL_OK) {
		Tcl_AppendResult(interp, argv[0], ": bad port number!", (char *) NULL);
		return TCL_ERROR;
	    }
#ifdef HAVE_SYS_UN_H
	} else if (argc == 2) {
	    pathname = Tcl_TildeSubst(interp, argv[1], &buffer);
	    if (pathname == NULL) {
		return TCL_ERROR;
	    }
	    unixSocket = 1;
#endif
	} else {
	    goto error;
	}

    }

    /*
     * Create the connection
     */
#ifdef HAVE_SYS_UN_H
    if (unixSocket) {
	socket = Tdp_unix_connect(pathname, server, udp);
    } else
#endif
    if (mudp) {
	socket = Tdp_multicast_connect(host, port, ttl);
    } else {
	socket = Tdp_inet_connect(host, port, server, udp, reuseAddr, lingerTime);
    }

    if (socket == DP_INVALID_SOCKET) {

	/* Tell them why it fell apart */
#ifdef HAVE_SYS_UN_H
	if (unixSocket) {
	    if (server) {
		Tcl_AppendResult(interp,
			     "Couldn't setup listening socket with path \"",
				 pathname, "\": ", Tdp_Error(interp),
				 (char *) NULL);
	    } else {
		Tcl_AppendResult(interp,
				 "Couldn't connect to \"", pathname, "\": ",
				 Tdp_Error(interp), (char *) NULL);
	    }
	    Tcl_DStringFree(&buffer);
	} else
#endif
	if (server) {
	    if (port == 0) {
		Tcl_AppendResult(interp,
			    "Couldn't setup listening socket on any port: ",
				 Tdp_Error(interp), (char *) NULL);
	    } else {
		sprintf(tmp, "%d", port);
		Tcl_AppendResult(interp,
			    "Couldn't setup listening socket on port ", tmp,
			       ": ", Tdp_Error(interp), (char *) NULL);
	    }
	} else if (udp) {
	    sprintf(tmp, "%d", port);
	    Tcl_AppendResult(interp,
			     "Couldn't open udp socket ", tmp, " : ",
			     Tdp_Error(interp), (char *) NULL);
	} else {
	    sprintf(tmp, "%d", port);
	    Tcl_AppendResult(interp,
		      "Couldn't open connection to ", host, ":", tmp, " : ",
			     Tdp_Error(interp), (char *) NULL);
	}
	return TCL_ERROR;
    }
#ifdef HAVE_SYS_UN_H
    if (unixSocket) {
	Tcl_DStringFree(&buffer);
    }
#endif
    if (Tdp_EnterSocket(interp, socket, TCL_READABLE|TCL_WRITABLE,
			&infoPtr) != TCL_OK)
    {
	return TCL_ERROR;
    }

    /*
     * Clear up any leftover data that might not have been cleaned up, just
     * in case.
     */
    Tdp_FreeReadBuffer(infoPtr);
#ifdef HAVE_SYS_UN_H
    if (!unixSocket)
#endif
    {
	struct sockaddr_in sockaddr;
	int res, len;
	/* Find the local port we're using for the connection. */

	len = sizeof(sockaddr);
	res = getsockname(socket, (struct sockaddr *) & sockaddr, &len);

	if (res < 0) {
	    sprintf(tmp, "%s %d", interp->result, DP_GetLastError());
	} else {
	    sprintf(tmp, "%s %d", interp->result, ntohs(sockaddr.sin_port));
	}
    }
#ifdef HAVE_SYS_UN_H
    else {
	sprintf(tmp, "file%d", infoPtr->socket);
    }
#endif
    Tcl_SetResult(interp, tmp, TCL_VOLATILE);

    Tdp_AllocateBuffer(infoPtr);


    if (mudp) {
	infoPtr->optFlags = FD_BLOCKING | FD_AUTO_CLOSE;
    } else if (udp) {
	infoPtr->optFlags = FD_BLOCKING | FD_AUTO_CLOSE;
    } else {
	infoPtr->optFlags = FD_TCP | FD_BLOCKING | FD_AUTO_CLOSE;
    }
#ifdef HAVE_SYS_UN_H
    if (unixSocket) {
	infoPtr->optFlags |= FD_UNIX;
    }
#endif
    if (server) {
	infoPtr->optFlags |= FD_SERVER;
    }
    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * Tdp_SocketOptionCmd --
 *
 *	This function implements the tcl "dp_socketOption" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	The system level properties of the socket may be changed
 *	by this call.
 *
 *--------------------------------------------------------------
 */
int
Tdp_SocketOptionCmd(clientData, interp, argc, argv)
    ClientData *clientData;	/* Often ignored */
    Tcl_Interp *interp;		/* tcl interpreter */
    int argc;			/* Number of arguments */
    char *argv[];		/* Arg list */
{
    char c;
    DP_SockInfo *infoPtr;
    int len;
    int optname;
    int intVal;
    unsigned char byteVal;
    int result;
    int optlen;
    char tmp[256];

    char *mcastAddr;
    struct ip_mreq mreq;
    struct sockaddr_in sockaddr;

    if ((argc != 3) && (argc != 4)) {
	Tcl_AppendResult(interp, "wrong# args: should be \"", argv[0],
			 " socket option ?value?\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 0, 0, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }

    c = argv[2][0];
    if (isupper(c)) {
	c = tolower(c);
    }
    len = strlen(argv[2]);

    /*
     * Step 1:  Get arguments, check syntax, etc.  At
     * the end of this step, optname will be the socket option,
     * and either byteVal or intVal will be set to the value passed
     * in (if any).
     */
    /* ------------------------ SEND BUFFER ---------------------------- */
    if ((c == 's') && (strncasecmp(argv[2], "sendBuffer", len) == 0)) {
	optname = SO_SNDBUF;
	if (argc == 4) {
	    if (Tcl_GetInt(interp, argv[3], &intVal) != TCL_OK) {
		return TCL_ERROR;
	    }
	}

    /* ------------------------ RECV BUFFER ---------------------------- */
    } else if ((c == 'r') && (strncasecmp(argv[2], "recvBuffer", len) == 0)) {
	optname = SO_RCVBUF;
	if (argc == 4) {
	    if (Tcl_GetInt(interp, argv[3], &intVal) != TCL_OK) {
		return TCL_ERROR;
	    }
	}

    /* ------------------------ NON BLOCKING --------------------------- */
    } else if ((c == 'n') && (strncasecmp(argv[2], "noblock", len) == 0)) {
	if (argc == 3) {
	    sprintf (tmp, "%s", ((infoPtr->optFlags&FD_BLOCKING)?"no":"yes"));
	    Tcl_SetResult(interp, tmp, TCL_VOLATILE);
	} else {
	    if (strcmp(argv[3], "yes") == 0) {
		Tdp_SetBlocking(infoPtr, 0);
	    } else if (strcmp(argv[3], "no") == 0) {
		Tdp_SetBlocking(infoPtr, 1);
	    } else {
		Tcl_AppendResult(interp, argv[0], ": Bad value \"",
				 argv[3], "\"", NULL);
		return TCL_ERROR;
	    }
	}
	return TCL_OK;

    /* ------------------------ KEEP ALIVE ----------------------------- */
    } else if ((c == 'k') && (strncasecmp(argv[2], "keepAlive", len) == 0)) {
	optname = SO_KEEPALIVE;
	intVal = 1;
	if (argc == 4) {
	    if (strcmp(argv[3], "yes") == 0) {
		intVal = 1;
	    } else if (strcmp(argv[3], "no") == 0) {
		intVal = 0;
	    } else {
		Tcl_AppendResult(interp, argv[0], ": Bad value \"",
				 argv[3], "\"", NULL);
		return TCL_ERROR;
	    }
	}

    /* ------------------------ AUTO CLOSE ----------------------------- */
    } else if ((c == 'a') && (strncasecmp(argv[2], "autoClose", len) == 0)) {
	if (argc == 3) {
	    sprintf (tmp, "%s",((infoPtr->optFlags&FD_AUTO_CLOSE)?"yes":"no"));
	    Tcl_SetResult(interp, tmp, TCL_VOLATILE);
	} else {
	    if (strcmp(argv[3], "yes") == 0) {
		infoPtr->optFlags |= FD_AUTO_CLOSE;
	    } else if (strcmp(argv[3], "no") == 0) {
		infoPtr->optFlags &= ~FD_AUTO_CLOSE;
	    } else {
		Tcl_AppendResult(interp, argv[0], ": Bad value \"",
				 argv[3], "\"", NULL);
		return TCL_ERROR;
	    }
	}
	return TCL_OK;

    /* -------------------------- TTL----------------------------------- */
    } else if ((c == 't') && (strncasecmp(argv[2], "ttl", len) == 0)) {
	optname = IP_MULTICAST_TTL;
	intVal = 1;

	if (argc == 4) {
	    if (Tcl_GetInt(interp, argv[3], &intVal) != TCL_OK) {
		Tcl_AppendResult(interp, argv[0], ": Bad ttl value \"",
				 argv[3], "\"", NULL);
		return TCL_ERROR;
	    }
	    if ((intVal < 0) || (intVal > 255)) {
		Tcl_AppendResult(interp, argv[0], ": Bad ttl value ",
				 argv[3], " Must be in [0..255]", NULL);
		return TCL_ERROR;
	    }
	}
	byteVal = intVal;

    /* -------------------------- LOOPBACK----------------------------- */
    } else if ((c == 'l') && (strncasecmp(argv[2], "loopBack", len) == 0)) {
	optname = IP_MULTICAST_LOOP;
	if (argc == 4) {
	    if (strcmp(argv[3], "yes") == 0) {
		byteVal = 1;
	    } else if (strcmp(argv[3], "no") == 0) {
		byteVal = 0;
	    } else {
		Tcl_AppendResult(interp, argv[0], ": Bad value \"",
				 argv[3], "\"", NULL);
		return TCL_ERROR;
	    }
	}
	intVal = byteVal;

    /* ------------------- ADD MEMBERSHIP ----------------------- */
    } else if ((c == 'a') && (strncasecmp(argv[2], "addMbr", len) == 0)) {
	optname = IP_ADD_MEMBERSHIP;
	if (argc == 4) {
	    mcastAddr = argv[3];
	    sockaddr.sin_addr.s_addr = inet_addr(mcastAddr);
	    if (IN_MULTICAST(ntohl(sockaddr.sin_addr.s_addr))) {
		mreq.imr_multiaddr.s_addr = inet_addr(mcastAddr);
		mreq.imr_interface.s_addr = htonl(INADDR_ANY);
	    } else {
		Tcl_AppendResult(interp, argv[0], ": Bad multicast address \"",
				 argv[3], "\"", NULL);
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, "Wrong number of args -- usage: \"",
			     argv[0], "file addMbr addr\"", NULL);
	    return TCL_ERROR;
	}

    /* ------------------------ DROP MEMBERSHIP ------------------------ */
    } else if ((c == 'd') && (strncasecmp(argv[2], "dropMbr", len) == 0)) {
	optname = IP_DROP_MEMBERSHIP;

	if (argc == 4) {
	    mcastAddr = argv[3];
	    sockaddr.sin_addr.s_addr = inet_addr(mcastAddr);
	    if (IN_MULTICAST(ntohl(sockaddr.sin_addr.s_addr))) {
		mreq.imr_multiaddr.s_addr = inet_addr(mcastAddr);
		mreq.imr_interface.s_addr = htonl(INADDR_ANY);
	    } else {
		Tcl_AppendResult(interp, argv[0], ": Bad multicast address \"",
				 argv[3], "\"", NULL);
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, "Wrong number of args -- usage: \"",
			     argv[0], "file addMbr addr\"", NULL);
	    return TCL_ERROR;
	}

    /* ------------------------ ERROR ---------------------------------- */
    } else {
	Tcl_AppendResult(interp, argv[0], "unknown option \"",
			 argv[2], "\"", (char *) NULL);
	return TCL_ERROR;
    }


    /*
     * Step 2: Set or get the socket option.  The parameters to
     * setsockopt/getsockopt depend on which option we're  processing
     */
    if ((optname == IP_ADD_MEMBERSHIP) || (optname == IP_DROP_MEMBERSHIP)) {
	optlen = sizeof(mreq);
	if (argc == 4) {
	    result = setsockopt(infoPtr->socket, IPPROTO_IP, optname,
				(char *)&mreq, optlen);
	} else {
	    result = getsockopt(infoPtr->socket, IPPROTO_IP, optname,
				(char *)&mreq, &optlen);
	}
    } else if ((optname == IP_MULTICAST_LOOP) ||
	       (optname == IP_MULTICAST_TTL)) {
	u_char buf[sizeof(int) * 2] = {0};
	u_char *p;

	/*
	 * This is here to handle a bug in Windows NT
	 */
	optlen = sizeof(byteVal);
	p = &buf[sizeof(int)];
	*p = byteVal;
	if (argc == 4) {
	    result = setsockopt(infoPtr->socket, IPPROTO_IP, optname, p, optlen);
	} else {
	    result = getsockopt(infoPtr->socket, IPPROTO_IP, optname, p, &optlen);
	}
	byteVal = *p;
    } else {
	optlen = sizeof(intVal);
	if (argc == 4) {
	    result = setsockopt(infoPtr->socket, SOL_SOCKET, optname,
				(char *)&intVal, optlen);
	} else {
	    result = getsockopt(infoPtr->socket, SOL_SOCKET, optname,
				(char *)&intVal, &optlen);
	}
    }

    if (result != 0) {
	goto dpError;
    }

    if (optname == IP_MULTICAST_LOOP) {
	sprintf(tmp, "%s", byteVal ? "yes" : "no");
    } else if (optname == IP_MULTICAST_TTL) {
	sprintf(tmp, "%d", (int)byteVal);
    } else if (optname == SO_KEEPALIVE) {
	sprintf(tmp, "%s", intVal ? "yes" : "no");
    } else if ((optname == IP_ADD_MEMBERSHIP) ||
	       (optname == IP_DROP_MEMBERSHIP)) {
	tmp[0] = 0;
    } else {
	sprintf(tmp, "%d", intVal);
    }

    if ((optname == SO_RCVBUF) && (intVal > bufferSize)) {
	Tdp_AllocateBuffer(infoPtr);
    }

    Tcl_SetResult(interp, tmp, TCL_VOLATILE);

    return TCL_OK;

dpError:
    Tcl_AppendResult(interp, argv[0], ": ", Tdp_Error(interp),
		     (char *) NULL);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * Tdp_FindFileHandler --
 *
 *	Find the filehandler associated with the
 *	descriptor passed in.
 *
 * Results:
 *	A pointer to the handler, or NULL if there is none.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

#define Tdp_FindFileHandler(fd)         \
    (((fd) < 0 || (fd) >= MAX_OPEN_FILES)   \
     ? ((DP_FileHandle *)NULL)      \
     : handlers[fd])


/*
 *------------------------------------------------------------------
 *
 * Tdp_ShutdownCmd --
 *
 *	This procedure is the C interface to the "dp_shutdown"
 *	command. See the user documentation for a description.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Modifies the OpenFile structure appropriately.
 *	Delete any created filehandlers.
 *
 *------------------------------------------------------------------
 */
int
Tdp_ShutdownCmd(notUsed, interp, argc, argv)
    ClientData notUsed;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    DP_SockInfo *infoPtr;
    DP_FileHandle *handler;
    int permissions;

    /*
     * Check args, find file
     */
    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong# args: should be \"", argv[0],
			 " fileid <option>\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 0, 0, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }

    permissions = infoPtr->permissions;
    if (permissions == -1) {
	Tcl_AppendResult(interp, "unable to determine access for socket \"",
			 argv[1], "\"", (char *) NULL);
	return TCL_ERROR;
    }
    handler = infoPtr->handlers;

    /*
     * Call shutdown with correct args, update file handler
     */
    if (!strcmp(argv[2], "0") ||
	!strcmp(argv[2], "receives") ||
	!strcmp(argv[2], "read")) {
	if ((permissions & TCL_READABLE) == 0) {
	    Tcl_AppendResult(interp, "File is not readable", (char *) NULL);
	    return TCL_ERROR;
	}
	if (shutdown(infoPtr->socket, 0)) {
	    Tcl_AppendResult(interp, "shutdown: ", Tdp_Error(interp),
			     (char *) NULL);
	    return TCL_ERROR;
	}
	permissions &= ~TCL_READABLE;
    } else if (!strcmp(argv[2], "1") ||
	       !strcmp(argv[2], "sends") ||
	       !strcmp(argv[2], "write")) {
	if ((permissions & TCL_WRITABLE) == 0) {
	    Tcl_AppendResult(interp, "File is not writable", (char *) NULL);
	    return TCL_ERROR;
	}
	fflush(infoPtr->filePtr);
	if (shutdown(infoPtr->socket, 1)) {
	    Tcl_AppendResult(interp, "shutdown: ", Tdp_Error(interp),
			     (char *) NULL);
	    return TCL_ERROR;
	}
	permissions &= ~TCL_WRITABLE;
    } else if (!strcmp(argv[2], "2") ||
	       !strcmp(argv[2], "all") ||
	       !strcmp(argv[2], "both")) {
	fflush(infoPtr->filePtr);
	if (shutdown(infoPtr->socket, 2)) {
	    Tcl_AppendResult(interp, "shutdown: ", Tdp_Error(interp),
			     (char *) NULL);
	    return TCL_ERROR;
	}
	permissions = 0;
    } else {
	Tcl_AppendResult(interp, "Bad argument: should be \"read\", \"write\" or \"both\".", (char *) NULL);
	return TCL_ERROR;
    }
    infoPtr->permissions = permissions;

    /*
     * Update the handler, freeing it if it's dead.
     */
    if (handler) {
	if (((permissions & TCL_READABLE) == 0) &&
	    (handler->rCmd != NULL)) {
	    ckfree(handler->rCmd);
	    handler->rCmd = NULL;
	}
	if (((permissions & TCL_WRITABLE) == 0) &&
	    (handler->wCmd != NULL)) {
	    ckfree(handler->wCmd);
	    handler->wCmd = NULL;
	}
	if ((permissions & (TCL_READABLE | TCL_WRITABLE)) == 0) {
	    if (handler->eCmd != NULL) {
		ckfree(handler->eCmd);
		handler->eCmd = NULL;
	    }

	    /*
	     * Delete handler.
	     */
	    Tk_DeleteFileHandler(infoPtr->socket);

	    infoPtr->handlers = (DP_FileHandle *) NULL;
	    Tk_EventuallyFree((ClientData) handler,
			      (Tk_FreeProc *) Tdp_FreeHandler);
	}
    }
    return TCL_OK;
}


/*
 *------------------------------------------------------------------
 *
 * Tdp_AcceptCmd --
 *
 *	This procedure is the C interface to the "dp_accept"
 *	command. See the user documentation for a description.
 *	It accepts a connection on a listening socket.
 *
 * Results:
 *	a standard tcl result
 *
 * Side effects:
 *	Opens a new file.
 *
 *------------------------------------------------------------------
 */
int
Tdp_AcceptCmd(notUsed, interp, argc, argv)
    ClientData notUsed;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    struct sockaddr_in sockaddr;
    int len = sizeof sockaddr;
    DP_SOCKET socket;
    DP_SockInfo *infoPtr1, *infoPtr2;
    int addr, f1, f2, f3, f4;
    char tmp[128];
    int size;

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong# args: should be \"", argv[0],
			 " listening_socket\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (Tdp_GetOpenSocket(interp, argv[1], 0, 1, &infoPtr1) != TCL_OK) {
  	return TCL_ERROR;
    }

    if (infoPtr1->optFlags & FD_SERVER) {
	socket = accept(infoPtr1->socket, (struct sockaddr *)&sockaddr, &len);
    } else {
	Tcl_AppendResult(interp, argv[0], ": must be a server socket", NULL);
	return TCL_ERROR;
    }

    if (socket == DP_INVALID_SOCKET) {
	Tcl_AppendResult(interp, "accept: ", dp_strerror(DP_GetLastError()),
			 (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Create the DP_SockInfo structure.
     */
    if (Tdp_EnterSocket(interp, socket, TCL_READABLE|TCL_WRITABLE,
			&infoPtr2) != TCL_OK)
    {
	return TCL_ERROR;
    }
    if (sockaddr.sin_family == AF_INET) {
	addr = htonl(sockaddr.sin_addr.s_addr);
	f1 = (addr >> 24) & 0xff;
	f2 = (addr >> 16) & 0xff;
	f3 = (addr >> 8) & 0xff;
	f4 = addr & 0xff;
    } else {
	f1 = f2 = f3 = f4 = 0;
    }

    sprintf(tmp, "file%d %d.%d.%d.%d", infoPtr2->sockId, f1, f2, f3, f4);
    Tcl_SetResult(interp, tmp, TCL_VOLATILE);

    /*
     * Make sure the send/recv buffers are of a reasonable size
     * (this is a problem under Solaris 2.x) -- the default size is 0.
     */
    size = 8192;
    len = sizeof(int);
    getsockopt(socket, SOL_SOCKET, SO_SNDBUF, (char *)&size, &len);
    if (size < 8192) {
	size = 8192;
    }
    len = sizeof(int);
    setsockopt(socket, SOL_SOCKET, SO_SNDBUF, (char *)&size, sizeof(int));
    getsockopt(socket, SOL_SOCKET, SO_RCVBUF, (char *)&size, &len);
    if (size < 8192) {
	size = 8192;
    }
    setsockopt(socket, SOL_SOCKET, SO_RCVBUF, (char *)&size, sizeof(int));

    Tdp_AllocateBuffer(infoPtr2);
    infoPtr2->optFlags = FD_AUTO_CLOSE | FD_TCP |
	FD_BLOCKING | (infoPtr1->optFlags & FD_UNIX);
    return TCL_OK;
}


#ifdef HAVE_SYS_UN_H

/*
 *----------------------------------------------------------------
 *
 * Tdp_unix_connect --
 *
 *	Create a (unix_domain) fd connection using given rendevous
 *
 * Results:
 *	An open socket or DP_INVALID_SOCKET.
 *
 * Side effects:
 *	None.
 *----------------------------------------------------------------
 */
static int
Tdp_unix_connect(path, server, udp)
    char *path;			/* Path name to create or use */
    int server;			/* 1->make server, 0->connect to server */
    int udp;			/* Make it a udp protocol socket */
{
    struct sockaddr_un sockaddr;
    DP_SOCKET sock;
    int status;
    if (udp) {
	sock = socket(PF_UNIX, SOCK_DGRAM, 0);
    } else {
	sock = socket(PF_UNIX, SOCK_STREAM, 0);
    }
    if (sock == DP_INVALID_SOCKET) {
	return DP_INVALID_SOCKET;
    }
    memset((char *) &sockaddr, 0, sizeof(sockaddr));
    sockaddr.sun_family = AF_UNIX;
    strncpy(sockaddr.sun_path, path, sizeof(sockaddr.sun_path) - 1);

    /* Just in case addr is too long... */
    sockaddr.sun_path[sizeof(sockaddr.sun_path) - 1] = 0;

    if (server | udp) {
	status = bind(sock, (struct sockaddr *) & sockaddr, sizeof(sockaddr));
    } else {
	status = connect(sock, (struct sockaddr *) & sockaddr, sizeof(sockaddr));
    }
    if (status < 0) {
	closesocket(sock);
	return DP_INVALID_SOCKET;
    }
    if (server && !udp) {
	if (listen(sock, 5) == DP_INVALID_SOCKET) {
	    int err = DP_GetLastError();
	    closesocket(sock);
	    DP_SetLastError(err);
	    return DP_INVALID_SOCKET;
	}
    }
    return sock;
}
#endif


/*
 *----------------------------------------------------------------
 *
 * Tdp_inet_connect --
 *
 *	Create a (inet domain) socket connection to given host and port.
 *
 * Results:
 *	An open socket or DP_INVALID_SOCKET.
 *
 * Side effects:
 *	None.
 *----------------------------------------------------------------
 */

DP_SOCKET
Tdp_inet_connect(host, port, server, udp, reuseAddr, lingerTime)
    char *host;			/* Host to connect, name or IP address */
    int port;			/* Port number to use */
    int server;			/* 1->make server, 0->connect to server */
    int udp;			/* Make it a udp protocol socket */
    int reuseAddr;		/* Allow local reuse of addresses */
    int lingerTime;		/* Time to linger on close */
{
    struct hostent *hostent, _hostent;
    struct sockaddr_in sockaddr;
    DP_SOCKET sock;
    int status;
#ifdef __DGUX__
    struct sockaddr_in hostaddr, hostaddrPtr[2];
#else
    int hostaddr, hostaddrPtr[2];
#endif
    char localhost[MAXHOSTNAMELEN];
    int result;
    int linger[2];

#if 0
    /*
     * Translate the hostname. Use the hostname of the localhost if none is
     * passed in.
     */
#else
    /*
     * Translate the hostname
     * Don't use "localhost" if none is passed in, since this locks
     * an RPC server into using one interface.  Also, don't use ""
     * because, on NT, a name lookup of "" will go out for lunch.
     */
#endif
    if (host == NULL) {
        gethostname(localhost,MAXHOSTNAMELEN);
        host = localhost;
    }
    if (host != NULL && host[0] != 0) {
	int dotNotation = 1;
	char *bp;

	for (bp = host; *bp; bp++) {
	    if (! isdigit(*bp) && *bp != '.') {
		dotNotation = 0;
		break;
	    }
	}

	if (dotNotation == 0) {
	    hostent = gethostbyname(host);
	} else {
	    hostent = NULL;
	}
    } else {
	hostent = NULL;
    }

    if (hostent == NULL) {
	if (strlen(host) == 0) {
	    memset(&hostaddr, 0, sizeof(hostaddr));
	} else {
#ifdef __DGUX__
	    hostaddr.sin_addr = inet_addr(host);
	    if ((int) hostaddr.sin_addr.s_addr == -1) {
#else
	    hostaddr = inet_addr(host);
	    if (hostaddr == -1) {
#endif
		return DP_INVALID_SOCKET;
	    }
	}

	_hostent.h_addr_list = (char **) hostaddrPtr;
#ifdef CRAY_HACKS
	hostaddr <<= 32;
#endif
#ifdef __DGUX__
	_hostent.h_addr_list[0] = (char *) &hostaddr.sin_addr;
	_hostent.h_length = sizeof(hostaddr.sin_addr);
#else
	_hostent.h_addr_list[0] = (char *) &hostaddr;
	_hostent.h_length = sizeof(hostaddr);
#endif
	_hostent.h_addr_list[1] = NULL;
	_hostent.h_addrtype = AF_INET;
	hostent = &_hostent;
    }

    /*
     * Create the socket
     */
    if (udp) {
	sock = socket(PF_INET, SOCK_DGRAM, 0);
    } else {
	sock = socket(PF_INET, SOCK_STREAM, 0);
    }
    if (sock == DP_INVALID_SOCKET) {
	return DP_INVALID_SOCKET;
    }

    /*
     * Set the linger and reuseAddr socket options
     */
#ifdef SO_LINGER
    linger[0] = lingerTime > 0;
    linger[1] = lingerTime;
    result = setsockopt(sock, SOL_SOCKET, SO_LINGER,
			(char *) linger, sizeof(linger));
#endif
#ifdef SO_REUSEADDR
    if (reuseAddr) {
	int one = 1;
	result = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR,
			    (char *) &one, sizeof(int));
    }
#endif

    /*
     * Make sure the send/recv buffers are of a reasonable size
     * (this is a problem under Solaris 2.x) -- the default size is 0.
     */
    {
    int size, len;
    size = 8192;
    len = sizeof(int);
    getsockopt(sock, SOL_SOCKET, SO_SNDBUF, (char *)&size, &len);
    if (size < 8192) {
	size = 8192;
    }
    setsockopt(sock, SOL_SOCKET, SO_SNDBUF, (char *)&size, sizeof(int));
    len = sizeof(int);
    getsockopt(sock, SOL_SOCKET, SO_RCVBUF, (char *)&size, &len);
    if (size < 8192) {
	size = 8192;
    }
    setsockopt(sock, SOL_SOCKET, SO_RCVBUF, (char *)&size, sizeof(int));
    }

    /*
     * Bind the socket.
     */
    memset((char *) &sockaddr, 0, sizeof(sockaddr));
    sockaddr.sin_family = AF_INET;
#ifdef CRAY_HACKS
    {
	unsigned long foo;
	memcpy((char *) &foo,
	       (char *) hostent->h_addr_list[0],
	       (size_t) hostent->h_length);

	sockaddr.sin_addr.s_addr = foo >> 32;
    }
#else
    memcpy((char *) &(sockaddr.sin_addr.s_addr),
	   (char *) hostent->h_addr_list[0],
	   (size_t) hostent->h_length);
#endif
    sockaddr.sin_port = htons((unsigned short) port);
    /* sockaddr.sin_addr.s_addr = INADDR_ANY; */

    if (server | udp) {
	status = bind(sock, (struct sockaddr *)&sockaddr, sizeof(sockaddr));
    } else {
	status = connect(sock, (struct sockaddr *)&sockaddr, sizeof(sockaddr));
    }

    if (status < 0) {
	int err = DP_GetLastError();
	closesocket(sock);
	DP_SetLastError(err);
	return DP_INVALID_SOCKET;
    }

    /*
     * Finally, listen on the socket if it's a server.
     */
    if (server && !udp) {
	if (listen(sock, 5) == DP_INVALID_SOCKET) {
	    int err = DP_GetLastError();
	    closesocket(sock);
	    DP_SetLastError(err);
	    return DP_INVALID_SOCKET;
	}
    }
    return sock;
}



/*
 *--------------------------------------------------------------
 *
 * Tdp_IsReadyCmd --
 *
 *	This procedure implements the "dp_isready" function, which
 *	returns whether a file has input pending.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
int
Tdp_IsReadyCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Ignored */
    Tcl_Interp *interp;		/* Tcl interpreter */
    int argc;			/* Number of arguments */
    char *argv[];		/* Arg list */
{
    DP_SockInfo *infoPtr;
    FILE *filePtr;
    int state, readable, writeable;
    char tmp[32];
    int sockId;
    if (argc != 2)
	goto syntaxError;

    if (Tdp_GetOpenSocket(interp, argv[1], 0, 0, &infoPtr) != TCL_OK) {
	if (Tcl_GetOpenFile(interp, argv[1], 0, 0, &filePtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	sockId = fileno(filePtr);
    } else {
	sockId = infoPtr->socket;
    }
    state = Tdp_SocketIsReady(sockId);
    readable = (state & TCL_READABLE) != 0;
    writeable = (state & TCL_WRITABLE) != 0;
    sprintf(tmp, "%d %d", readable, writeable);
    Tcl_SetResult(interp, tmp, TCL_VOLATILE);
    return TCL_OK;

syntaxError:
    Tcl_AppendResult(interp, "wrong# args: should be \"",
		     argv[0], " fileId\"", (char *) NULL);
    return TCL_ERROR;
}


/*
 *----------------------------------------------------------------
 *
 * Tdp_FileHandlerCmd --
 *
 *	This procedure is the C interface to the "dp_filehandler"
 *	command. See the user documentation for a description.
 *	Register a file handler with an open fileId.  If there is
 *	already an existing handler, it will be no longer called.
 *	If no mask and command are given, any existing handler
 *	will be deleted.
 *
 * Results:
 *	A standard Tcl result. (Always OK).
 *
 * Side effects:
 *	A new file handler is associated with a give TCL open file.
 *	Whenever the file is readable, writeable and/or there is
 *	an expection condition on the file, a user supplied TCL
 *	command is called.
 *
 *----------------------------------------------------------------
 */
int
Tdp_FileHandlerCmd(notUsed, interp, argc, argv)
    ClientData notUsed;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    FILE *filePtr;
    DP_SockInfo *infoPtr;
    int fd, mask;
    DP_FileHandle *handler;
    int file = 0;

    /*
     * Checks args.
     */
    if (argc != 2 && argc != 4) {
	Tcl_AppendResult(interp, "wrong# args: should be \"", argv[0],
			 " fileId ?mode command?\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 0, 0, &infoPtr) != TCL_OK) {
	if (Tcl_GetOpenFile(interp, argv[1], 0, 0, &filePtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	file = 1;
	fd = fileno(filePtr);
	assert(fd < MAX_OPEN_FILES);
	handler = Tdp_FindFileHandler(fd);
	if (handler != NULL) {
	    handlers[fd] = NULL;
	}
    } else {
	handler = infoPtr->handlers;
	infoPtr->handlers = NULL;
    }

    if (handler != NULL) {
	Tk_EventuallyFree((ClientData) handler,
			  (Tk_FreeProc *) Tdp_FreeHandler);
	handler = NULL;
    }
    if (file) {
	Tk_DeleteFileHandler(fd);
    } else {
	Tk_DeleteFileHandler(infoPtr->socket);
    }
    if (argc == 2) {
	return TCL_OK;
    }

    /*
     * Find out on what situations the user is interested in. This is not the
     * most elegant or efficient way to do this, but who cares?
     */
    mask = 0;
    if (strchr(argv[2], 'r')) {
	mask |= TK_READABLE;
    }
    if (strchr(argv[2], 'w')) {
	mask |= TK_WRITABLE;
    }
    if (strchr(argv[2], 'e')) {
	mask |= TK_EXCEPTION;
    }
    if (mask == 0 || (strlen(argv[2]) != strspn(argv[2], "rwe"))) {
	Tcl_AppendResult(interp, "bad mask argument \"", argv[2],
		  "\": should be any combination of \"r\", \"w\" and \"e\"",
			 (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Create a new handler.
     */
    handler = (DP_FileHandle *) ckalloc(sizeof(DP_FileHandle));
    handler->interp = interp;
    if (file) {
	handler->infoPtr = (DP_SockInfo *) filePtr;
    } else {
	handler->infoPtr = infoPtr;
    }
    handler->fileId = ckalloc(strlen(argv[1]) + 1);
    handler->rCmd = NULL;
    handler->wCmd = NULL;
    handler->eCmd = NULL;
    handler->mask = 0;
    strcpy(handler->fileId, argv[1]);

    if (file) {
	handlers[fd] = handler;
    } else {
	infoPtr->handlers = handler;
    }

    if (mask & TK_READABLE) {
	handler->rCmd = ckalloc(strlen(argv[3]) + 1);
	strcpy(handler->rCmd, argv[3]);
    }
    if (mask & TK_WRITABLE) {
	handler->wCmd = ckalloc(strlen(argv[3]) + 1);
	strcpy(handler->wCmd, argv[3]);
    }
    if (mask & TK_EXCEPTION) {
	handler->eCmd = ckalloc(strlen(argv[3]) + 1);
	strcpy(handler->eCmd, argv[3]);
    }
    handler->mask = mask;

    /*
     * Finally, get Tk to call Tdp_HandleEvent whenever there is a file
     * descriptor condition.
     */
#ifdef TCM
    Tcm_CreateFileHandler ("rpc", infoPtr->socket, mask, (Tk_FileProc *)Tdp_HandleEvent,
                        (ClientData) handler);
#else
#ifdef TK_EXTENDED
    Tk_CreateFileHandler(infoPtr->socket, (FILE *) NULL, mask,
		     (Tk_FileProc *) Tdp_HandleEvent, (ClientData) handler);
#else
    Tk_CreateFileHandler(infoPtr->socket, mask, (Tk_FileProc *) Tdp_HandleEvent,
			 (ClientData) handler);
#endif
#endif
    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * Tdp_FreeHandler --
 *
 *	Free up a file handler and all it's parts.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static void
Tdp_FreeHandler(clientData)
    ClientData clientData;
{
    DP_FileHandle *handler = (DP_FileHandle *) clientData;
    if (handler->rCmd != NULL) {
	ckfree(handler->rCmd);
    }
    if (handler->wCmd != NULL) {
	ckfree(handler->wCmd);
    }
    if (handler->eCmd != NULL) {
	ckfree(handler->eCmd);
    }
    if (handler->fileId != NULL) {
	ckfree((char *) handler->fileId);
    }
    ckfree((char *) handler);
}


/*
 *----------------------------------------------------------------
 *
 * Tdp_HandleEvent --
 *
 *	This procedure is called from Tk_DoOneEvent whenever there is
 *	a file descriptor condition on a given file descriptor.  It is
 *	installed by the "dp_filehandler" command.  A Tcl command
 *	given by the user is executed to handle the condition.  If
 *	an EOF or ERROR condition is noticed, the file descriptor
 *	is closed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The user supplied command can do anything.
 *
 *----------------------------------------------------------------
 */

static void
Tdp_HandleEvent(clientData, mask)
    ClientData clientData;
    int mask;
{
    int result;
    DP_FileHandle *handler = (DP_FileHandle *) clientData;
    Tcl_Interp *interp;
    DP_SockInfo *infoPtr;
    DP_SockInfo *dummy;
    FILE *filePtr;
    int fd;
    int delete;
    int evalError = 0;
    int file = 0;
    if (!handler)
	return;

    interp = handler->interp;
    infoPtr = handler->infoPtr;
    filePtr = (FILE *) infoPtr;
    if ((interp == NULL) || (infoPtr == NULL)) {
	return;
    }
    Tk_Preserve((ClientData) handler);

    delete = 0;
    if (Tdp_GetOpenSocket(interp, handler->fileId, 0, 0, &dummy) != TCL_OK) {
	if (Tcl_GetOpenFile(interp, handler->fileId, 0, 0, (FILE **) &dummy) != TCL_OK) {
	    /*  File descriptor is closed. */
	    Tcl_ResetResult(interp);
	    delete = 1;
	} else {
	    file = 1;
	}
    } else {
	filePtr = infoPtr->filePtr;
    }

    if (delete == 0) {
	Tcl_DString cmd;
	assert(dummy == handler->infoPtr);
	Tcl_DStringInit(&cmd);

	if (mask & TK_EXCEPTION) {
	    if (handler->eCmd != NULL) {
		Tcl_DStringAppend(&cmd, handler->eCmd, -1);
		Tcl_DStringAppend(&cmd, " e ", 3);
		Tcl_DStringAppend(&cmd, handler->fileId, -1);
		result = Tcl_GlobalEval(interp, Tcl_DStringValue(&cmd));
		Tcl_DStringFree(&cmd);
		if (result != TCL_OK) {
		    goto close;
		}
	    } else {
		goto close;
	    }
	} else if ((mask & TK_READABLE) && (handler->rCmd != NULL)) {
	    Tcl_DStringAppend(&cmd, handler->rCmd, -1);
	    Tcl_DStringAppend(&cmd, " r ", 3);
	    Tcl_DStringAppend(&cmd, handler->fileId, -1);
	    result = Tcl_GlobalEval(interp, Tcl_DStringValue(&cmd));
	    Tcl_DStringFree(&cmd);
	    if (result != TCL_OK) {
		evalError = 1;
		goto freeHandler;
	    }
	} else if ((mask & TK_WRITABLE) && (handler->wCmd != NULL)) {
	    Tcl_DStringAppend(&cmd, handler->wCmd, -1);
	    Tcl_DStringAppend(&cmd, " w ", 3);
	    Tcl_DStringAppend(&cmd, handler->fileId, -1);
	    result = Tcl_GlobalEval(interp, Tcl_DStringValue(&cmd));
	    Tcl_DStringFree(&cmd);
	    if (result != TCL_OK) {
		evalError = 1;
		goto freeHandler;
	    }
	} else if (feof(filePtr) || ferror(filePtr)) {
	  close:
	    if (file) {
		if (Tcl_VarEval(interp, "close ", handler->fileId,
				(char *) NULL) != TCL_OK) {
		    Tcl_AppendResult(interp, "Unexpected EOF on ",
				     handler->fileId, (char *) NULL);
		    Tk_BackgroundError(interp);
		}
	    }
	    delete = 1;
	}
    }

    Tk_Release((ClientData) handler);

    if (delete) {

freeHandler:
	fd = fileno(filePtr);
	assert(fd < MAX_OPEN_FILES);

	if ((handler = Tdp_FindFileHandler(fd)) != NULL) {
	    handlers[fd] = (DP_FileHandle *) NULL;
	    Tk_EventuallyFree((ClientData) handler,
			      (Tk_FreeProc *) Tdp_FreeHandler);
	}
	if (file) {
	    Tk_DeleteFileHandler(fd);
	} else {
	    Tk_DeleteFileHandler(infoPtr->socket);
	}
	if (evalError) {
	    char str[64];
	    sprintf (str, "file%d", fd);
	    Tcl_AppendResult(interp, " -- file handler removed for ",
			     str, (char *) NULL);
	    Tk_BackgroundError(interp);
	}
    }
}


/*
 *--------------------------------------------------------------
 *
 * Tdp_ReceiveCmd --
 *
 *	This procedure is invoked to process the "dp_receive" Tcl/Tk
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	The file descriptor passed in is read.
 *
 *--------------------------------------------------------------
 */
int
Tdp_ReceiveCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Ignored */
    Tcl_Interp *interp;		/* Tcl interpreter */
    int argc;			/* Number of arguments */
    char *argv[];		/* Arg list */
{
    DP_SockInfo *infoPtr;
    int count;
    int flags;
    int i, len;
    int error;
    if ((argc < 2) || (argc > 4)) {
	goto syntaxError;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 0, 1, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Make sure this is a non-server TCP socket
     */
    if ((infoPtr->optFlags & FD_TCP) == 0) {
	Tcl_AppendResult(interp, "can't use ", argv[0], " on non TCP socket",
			 (char *) NULL);
	return TCL_ERROR;
    }
    if ((infoPtr->optFlags & FD_SERVER) != 0) {
	Tcl_AppendResult(interp, "can't use ", argv[0], " on server socket",
			 (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Get the extra parameters, if specified
     */
    count = bufferSize;
    flags = 0;
    for (i = 2; i < argc; i++) {
	len = strlen(argv[i]);
	if (strncmp(argv[i], "-peek", len) == 0) {
	    flags |= MSG_PEEK;
	} else if (Tcl_GetInt(interp, argv[i], &count) != TCL_OK) {
	    goto syntaxError;
	}
    }

    /*
     * Read the message into the global buffer and put on trailing 0 at end
     * of string in case we received a partial message.
     */
    count = Tdp_Read(infoPtr, buffer, count, flags);
    Tcl_ResetResult(interp);
    if (count == -1) {
	/*
	 * If the file is in non-blocking mode, return null string
	 */
	error = DP_GetLastError();
	if (error == DP_EWOULDBLOCK || error == DP_EAGAIN) {
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, "error reading ", argv[1], ": ",
			     Tdp_Error(interp), (char *) NULL);
	    return TCL_ERROR;
	}
    }

    /*
     * If we get an eof, the connection is closed and we should do some
     * cleanup.
     */
    if (count == 0) {
	if (infoPtr->optFlags & FD_AUTO_CLOSE) {
	    Tdp_CleanupFile(interp, argv[1], infoPtr);
	    Tcl_ResetResult(interp);
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp,
			     "error reading socket (connection closed) ",
			     argv[1], (char *) NULL);
	    return TCL_ERROR;
	}
    }

    /*
     * Ok, we got what we got.  Return it.
     */
    buffer[count] = 0;
    Tcl_SetResult(interp, buffer, TCL_STATIC);
    return TCL_OK;

syntaxError:
    Tcl_AppendResult(interp,
		     "syntax error: should be \"", argv[0],
		     " fileId ?numBytes? ?-peek?\"",
		     (char *) NULL);
    return TCL_ERROR;
}


/*
 *--------------------------------------------------------------
 *
 * Tdp_CleanupFile --
 *
 *	Clean up a socket on error.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	Will close the file and remove the handler if auto close
 *	is on.  This is the default action.
 *
 *--------------------------------------------------------------
 */
void
Tdp_CleanupFile(interp, file, clientData)
    Tcl_Interp *interp;
    char *file;
    ClientData clientData;
{
    DP_SockInfo *infoPtr = (DP_SockInfo *) clientData;
    if (infoPtr->optFlags & FD_AUTO_CLOSE) {
	Tcl_VarEval(interp, "dp_filehandler ", file, (char *) NULL);
	Tcl_VarEval(interp, "close ", file, (char *) NULL);
    }
}


/*
 *--------------------------------------------------------------
 *
 * Tdp_SendCmd --
 *
 *	This procedure is invoked to process the "dp_send" Tcl/Tk
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	The specified string is written to the file descriptor passed
 *	in.
 *
 *--------------------------------------------------------------
 */
int
Tdp_SendCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Ignored */
    Tcl_Interp *interp;		/* Tcl interpreter */
    int argc;			/* Number of arguments */
    char *argv[];		/* Arg list */
{
    DP_SockInfo *infoPtr;
    int count;
    int newline;
    int error;
    char tmp[256];
    if ((argc < 3) || (argc > 4)) {
	Tcl_AppendResult(interp, "wrong# args: should be \"", argv[0],
			 " fileId string ?nonewline?\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 1, 1, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Make sure this is a non-server TCP socket
     */
    if ((infoPtr->optFlags & FD_TCP) == 0) {
	Tcl_AppendResult(interp, "can't use ", argv[0], " on non TCP socket",
			 (char *) NULL);
	return TCL_ERROR;
    }
    if ((infoPtr->optFlags & FD_SERVER) != 0) {
	Tcl_AppendResult(interp, "can't use ", argv[0], " on server socket",
			 (char *) NULL);
	return TCL_ERROR;
    }
    newline = 1;
    if (argc == 4) {
	if (strncmp(argv[3], "nonewline", strlen(argv[3])) != 0) {
	    Tcl_AppendResult(interp, "bad argument \"", argv[3],
			     "\": should be \"nonewline\"", (char *) NULL);
	    return TCL_ERROR;
	}
	newline = 0;
    }
#ifndef NO_WRITEV
    {
	struct iovec iov[2];
	register int iovcnt = 1;
	iov[0].iov_len = strlen(argv[2]);
	iov[0].iov_base = argv[2];
	if (newline) {
	    ++iovcnt;
	    iov[1].iov_len = 1;
	    iov[1].iov_base = "\n";
	}
	/* Use writev to reduce number of kernel calls */
	count = writev(infoPtr->socket, iov, iovcnt);
    }
#else

    /*
     * Simulate writev() call with two write() calls, returning correct
     * value.
     */
    {
	char buf = '\n';
	int len = strlen(argv[2]);
	Tdp_BlockSocket(infoPtr, TK_WRITABLE|TK_EXCEPTION);
	count = send(infoPtr->socket, argv[2], len, 0);
	if (count != -1 && newline) {
	    Tdp_BlockSocket(infoPtr, TK_WRITABLE|TK_EXCEPTION);
	    count = send(infoPtr->socket, &buf, 1, 0);
	    if (count != -1) {
		count = len + 1;
	    }
	}
    }
#endif
    if (count == -1) {
	error = DP_GetLastError();
	if (error == DP_EPIPE) {
	    /*
	     * Got a broken pipe signal.  Close the file, delete the file
	     * handler, and return 0 bytes written.
	     */
	    Tdp_CleanupFile(interp, argv[1], infoPtr);
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	    return TCL_OK;
	}
	if (error == DP_EWOULDBLOCK || error == DP_EAGAIN) {
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	    return TCL_OK;
	}
	Tcl_AppendResult(interp, "error writing ", argv[1], ": ",
			 Tdp_Error(interp), (char *) NULL);
	return TCL_ERROR;
    }
    sprintf(tmp, "%d", count);
    Tcl_SetResult(interp, tmp, TCL_VOLATILE);
    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * Tdp_PacketReceive --
 *
 *	This procedure is the C interface to the "dp_packetReceive"
 *	command. See the user documentation for a description.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	The file descriptor passed in is read.
 *
 *--------------------------------------------------------------
 */
int
Tdp_PacketReceive(interp, fileHandle, peek)
    Tcl_Interp *interp;		/* Tcl interpreter */
    char *fileHandle;
    int peek;
{
    int numRead;
    int packetLen;
    int headerSize;
    DP_SockInfo *infoPtr;
    unsigned char hbuf[8];
    int header[2];
    char *errMsg;
    int flags;
    int badPacket = 0;
    int error;
    if (Tdp_GetOpenSocket(interp, fileHandle, 0, 1, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Make sure this is a non-server TCP socket
     */
    if ((infoPtr->optFlags & FD_TCP) == 0) {
	Tcl_AppendResult(interp, "can't use \"dp_packetReceive\" on non TCP socket",
			 (char *) NULL);
	return TCL_ERROR;
    }
    if ((infoPtr->optFlags & FD_SERVER) != 0) {
	Tcl_AppendResult(interp, "can't use \"dp_packetReceive\" on server socket",
			 (char *) NULL);
	return TCL_ERROR;
    }
    if (peek) {
	flags = MSG_PEEK;
    } else {
	flags = 0;
    }

    /*
     * Read in the header (8 bytes)
     */
    headerSize = 8;
    numRead = Tdp_Read(infoPtr, hbuf, headerSize, flags);

    if (numRead <= 0) {
	goto readError;
    }

    /*
     * Check for incomplete read.  If so, put it back (only if we consumed
     * it!) and return.
     */
    if (numRead < headerSize) {
	if (!peek) {
	    Tdp_Unread(infoPtr, hbuf, numRead, 1, 1);
	}
	Tcl_ResetResult(interp);
	return TCL_OK;
    }

    /*
     * Convert header character stream into ints.  This works when the
     * connecting machine has a different size int and takes care of endian
     * problems.  It is also mostly backward compatible since network byte
     * ordering (big endian) is used.
     */
    header[0] = 0;
    header[0] |= (unsigned int) hbuf[0] << 24;
    header[0] |= (unsigned int) hbuf[1] << 16;
    header[0] |= (unsigned int) hbuf[2] << 8;
    header[0] |= (unsigned int) hbuf[3];

    header[1] = 0;
    header[1] |= (unsigned int) hbuf[4] << 24;
    header[1] |= (unsigned int) hbuf[5] << 16;
    header[1] |= (unsigned int) hbuf[6] << 8;
    header[1] |= (unsigned int) hbuf[7];

    /*
     * Format of each packet:
     * 
     * First 4 bytes are PACKET_MAGIC. Next 4 bytes are packetLen. Next
     * packetLen-headerSize is zero terminated string
     */
    if (header[0] != PACKET_MAGIC) {
	Tcl_AppendResult(interp, "Error reading ", fileHandle,
			 ": badly formatted packet", (char *) NULL);
	badPacket = 1;
	goto readError;
    }
    packetLen = header[1] - headerSize;

    /*
     * Expand the size of the global buffer, as needed.
     */
    if (header[1] > bufferSize) {
	ckfree(buffer);
	bufferSize = header[1] + 32;
	buffer = ckalloc(bufferSize);
    }

    /*
     * Read in the packet.  If it's only partially there, unread it and
     * return.  If we're peeking, we need to be careful since the header is
     * still in the queue.
     */
    if (peek) {
	numRead = Tdp_Read(infoPtr, buffer, header[1], flags);
	if (numRead <= 0) {
	    goto readError;
	}

	/*
	 * Only partially there.  Return a null string.
	 */
	if (numRead != header[1]) {
	    Tcl_ResetResult(interp);
	    return TCL_OK;
	}
	buffer[numRead] = 0;
	Tcl_SetResult(interp, buffer + headerSize, TCL_STATIC);
	return TCL_OK;
    }

    /*
     * We're not peeking, so we've consumed the header (this is normal mode).
     * Read in the packet, and if it's not all there, put it back.
     * 
     * We have to be careful here, because we could block when if just the
     * header came in (making the file readable at the beginning of this
     * function) but the rest of the packet is still out on the network.
     */
    if (Tdp_SocketIsReady(infoPtr->socket) & TCL_READABLE) {
	numRead = Tdp_Read(infoPtr, buffer, packetLen, flags);
    } else {
	Tdp_Unread(infoPtr, hbuf, headerSize, 1, 1);
	Tcl_ResetResult(interp);
	return TCL_OK;
    }
    if (numRead < 0) {
	goto readError;
    }
    if (numRead != packetLen) {
	Tdp_Unread(infoPtr, hbuf, headerSize, 1, 0);
	Tdp_Unread(infoPtr, buffer, numRead, 1, 0);
	Tcl_ResetResult(interp);
	return TCL_OK;
    }
    buffer[numRead] = 0;
    Tcl_SetResult(interp, buffer, TCL_STATIC);
    return TCL_OK;

readError:

    /*
     * If we're in non-blocking mode, and this would block, return. If the
     * connection is closed (numRead == 0), don't return an error message.
     * Otherwise, return one.
     * 
     * In either case, we close the file, delete the file handler, and
     * return a null string.
     */
    error = DP_GetLastError();
    if (error == DP_EWOULDBLOCK || error == DP_EAGAIN) {
	Tcl_ResetResult(interp);
	return TCL_OK;
    }

    /*
     * If there was an error and it wasn't "Connection reset by peer"
     * then record the error before closing the file
     */
    if ((error != DP_ECONNRESET) && (numRead != 0)) {
	errMsg = Tdp_Error(interp);
    } else {
	numRead = 0;
	errMsg = NULL;		/* Suppresses spurious compiler warning */
    }

    /*
     * Remove the file handler and close the file.  We want to go through tcl
     * in case the user has overridden the close procedure
     */
    Tdp_CleanupFile(interp, fileHandle, infoPtr);
    Tdp_FreeReadBuffer(infoPtr);

    Tcl_ResetResult(interp);
    if (numRead == 0) {
	return TCL_OK;
    } else if (badPacket) {
	Tcl_AppendResult(interp, "Error reading ", fileHandle,
			 ": badly formatted packet", (char *) NULL);
	return TCL_ERROR;
    } else {
	Tcl_AppendResult(interp, "Tdp_PacketReceive -- error reading ",
			 fileHandle, ": ", errMsg, (char *) NULL);
	return TCL_ERROR;
    }
}


/*
 *--------------------------------------------------------------
 *
 * Tdp_PacketReceiveCmd --
 *
 *	This procedure is invoked to process the "dp_packetReceive" Tcl/Tk
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	The specified string is written to the file descriptor passed
 *	in.
 *
 *--------------------------------------------------------------
 */
int
Tdp_PacketReceiveCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Ignored */
    Tcl_Interp *interp;		/* Tcl interpreter */
    int argc;			/* Number of arguments */
    char *argv[];		/* Arg list */
{
    char *fileHandle;
    int len, peek;
    if ((argc < 2) || (argc > 3)) {
syntaxError:
	Tcl_AppendResult(interp, "wrong# args: should be \"", argv[0],
			 "\" fileId ?-peek?", (char *) NULL);
	return TCL_ERROR;
    }
    fileHandle = argv[1];

    if (argc == 3) {
	len = strlen(argv[2]);
	if (strncmp(argv[2], "-peek", len) == 0) {
	    peek = 1;
	} else {
	    goto syntaxError;
	}
    } else {
	peek = 0;
    }
    return (Tdp_PacketReceive(interp, fileHandle, peek));
}


/*
 *--------------------------------------------------------------
 *
 * Tdp_PacketSend --
 *
 *	This procedure is the C interface to the "dp_packetSend" command.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	The specified string is written to the file descriptor passed
 *	in.
 *
 *--------------------------------------------------------------
 */

int
Tdp_PacketSend(interp, fileHandle, message)
    Tcl_Interp *interp;		/* Tcl interpreter */
    char *fileHandle;
    char *message;
{
    DP_SockInfo *infoPtr;
    int strLen;
    int packetLen;
    int numSent;
    int error;
    unsigned char hbuf[8];
    unsigned long header[2];
    char tmp[256];
    if (Tdp_GetOpenSocket(interp, fileHandle, 1, 1, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Make sure this is a non-server TCP socket
     */
    if ((infoPtr->optFlags & FD_TCP) == 0) {
	Tcl_AppendResult(interp, "can't use \"dp_packetSend\" on non TCP socket",
			 (char *) NULL);
	return TCL_ERROR;
    }
    if ((infoPtr->optFlags & FD_SERVER) != 0) {
	Tcl_AppendResult(interp, "can't use \"dp_packetSend\" on server socket",
			 (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Format up the packet: First 4 bytes are PACKET_MAGIC. Next 4 bytes are
     * packetLen. Next packetLen-(sizeof(int)) bytes are zero terminated
     * message.
     */
    strLen = strlen(message);
    packetLen = strLen + 8;

    header[0] = PACKET_MAGIC;
    header[1] = packetLen;

    /*
     * Convert header ints to character stream. Network byte ordering (big
     * endian) is used.
     */
    hbuf[0] = (unsigned char)((header[0] & (unsigned long) 0xFF000000) >> 24);
    hbuf[1] = (unsigned char)((header[0] & (unsigned long) 0x00FF0000) >> 16);
    hbuf[2] = (unsigned char)((header[0] & (unsigned long) 0x0000FF00) >> 8);
    hbuf[3] = (unsigned char) (header[0] & (unsigned long) 0x000000FF);

    hbuf[4] = (unsigned char)((header[1] & (unsigned long) 0xFF000000) >> 24);
    hbuf[5] = (unsigned char)((header[1] & (unsigned long) 0x00FF0000) >> 16);
    hbuf[6] = (unsigned char)((header[1] & (unsigned long) 0x0000FF00) >> 8);
    hbuf[7] = (unsigned char) (header[1] & (unsigned long) 0x000000FF);
#ifndef NO_WRITEV
    {
	struct iovec iov[2];
	/* Set up scatter/gather vector */
	iov[0].iov_len = 8;
	iov[0].iov_base = (char *) hbuf;
	iov[1].iov_len = strLen;
	iov[1].iov_base = message;

	/* Send it off, with error checking */
	numSent = writev(infoPtr->socket, iov, 2);
    }
#else

    /*
     * Again, simulate writev (this time using memcpy to put together the msg
     * so it can go out in a single write() call
     */
    {
	int len;
	char *buffer;
	len = strLen + 8;
	buffer = (char *) ckalloc(len);

	memcpy(buffer, hbuf, 8);
	memcpy(buffer + 8, message, strLen);

	Tdp_BlockSocket(infoPtr, TK_WRITABLE|TK_EXCEPTION);
	numSent = send(infoPtr->socket, buffer, len, 0);

	ckfree(buffer);
    }
#endif
    if (numSent != packetLen) {
	error = DP_GetLastError();
	if ((error == 0) || (error == DP_EWOULDBLOCK || error == DP_EAGAIN)) {

	    /*
	     * Non-blocking I/O: return number of bytes actually sent.
	     */
	    Tcl_ResetResult(interp);
	    sprintf(tmp, "%d", numSent - 8);
	    Tcl_SetResult(interp, tmp, TCL_VOLATILE);
	    return TCL_OK;
	} else if (error == DP_EPIPE) {

	    /*
	     * Got a broken pipe signal, which means the far end closed the
	     * connection.  Close the file, delete the file handler, and
	     * return 0 bytes sent.
	     */
	    Tdp_CleanupFile(interp, fileHandle, infoPtr);
	    sprintf(tmp, "0");
	    Tcl_SetResult(interp, tmp, TCL_VOLATILE);
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, "Tdp_PacketSend -- error writing ",
			     fileHandle, ": ",
			     Tdp_Error(interp), (char *) NULL);
	}

	return TCL_ERROR;
    }

    /*
     * Return the number of bytes sent (minus the header).
     */
    sprintf(tmp, "%d", numSent - 8);
    Tcl_SetResult(interp, tmp, TCL_VOLATILE);
    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * Tdp_PacketSendCmd --
 *
 *	This procedure is invoked to process the "dp_packetSend" Tcl/Tk
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	The specified string is written to the file descriptor passed
 *	in.
 *
 *--------------------------------------------------------------
 */
int
Tdp_PacketSendCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Ignored */
    Tcl_Interp *interp;		/* Tcl interpreter */
    int argc;			/* Number of arguments */
    char *argv[];		/* Arg list */
{
    char *fileHandle;
    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong# args: should be \"", argv[0],
			 " fileId string\"", (char *) NULL);
	return TCL_ERROR;
    }
    fileHandle = argv[1];

    return (Tdp_PacketSend(interp, fileHandle, argv[2]));
}

/*
 *--------------------------------------------------------------
 *
 * Tdp_ReceiveFromCmd --
 *
 *	This procedure is invoked to process the "dp_receiveFrom" Tcl/Tk
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	The file descriptor passed in is read.
 *
 *--------------------------------------------------------------
 */
int
Tdp_ReceiveFromCmd(notUsed, interp, argc, argv)
    ClientData notUsed;
    Tcl_Interp *interp;		/* Tcl interpreter */
    int argc;			/* Number of arguments */
    char *argv[];		/* Arg list */
{
    DP_SockInfo *infoPtr;
    int flags, numBytes, bytesAvail;
    int len, i;
    int error;
    char *addrName;
    int count, addrLen;
    struct sockaddr_in addr;
    int noaddr;

    if ((argc < 2) || (argc > 5)) {
	goto syntaxError;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 0, 1, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Parse flag parameters;
     */
    flags = 0;
    numBytes = bufferSize;
    noaddr = 0;
    for (i = 2; i < argc; i++) {
	len = strlen(argv[i]);
	if (strncmp(argv[i], "-peek", len) == 0) {
	    flags |= MSG_PEEK;
	} else if (strncmp(argv[i], "-noaddr", len) == 0) {
	    noaddr = 1;
	} else if (Tcl_GetInt(interp, argv[i], &numBytes) != TCL_OK) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "bad argument -- should be \"", argv[0],
			     " fileId ?numBytes? ?-noaddr? ?-peek?\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
    }

    addrLen = sizeof(addr);
    memset((char *) &addr, 0, addrLen);

    /*
     * Make sure this is a UDP socket
     */
    if (infoPtr->optFlags & FD_TCP) {
	Tcl_AppendResult(interp, "can't use ", argv[0], " on a TCP socket",
			 (char *) NULL);
	return TCL_ERROR;
    }

#if defined(SOLARIS) && (OS_MAJOR_VERSION == 5) && (OS_MINOR_VERSION <= 4)
    /*
     * Solaris doesn't truncate the datagram when you do a
     * recvfrom wiht numBytes < bytes available. The code below
     * is a work-around to flush the rest of the datagram out of
     * the socket
     */
    bytesAvail = recvfrom(infoPtr->socket, buffer, bufferSize, MSG_PEEK, NULL, 0);
#endif

    /*
     * Read the message and put on trailing 0 at end of string in case we
     * received a partial message.
     */
    Tdp_BlockSocket(infoPtr, TK_READABLE|TK_EXCEPTION);
    count = recvfrom(infoPtr->socket, buffer, numBytes, flags,
		     (struct sockaddr *) & addr, &addrLen);
    if (count == -1) {
	/*
	 * Under Windows NT, if you don't read the whole packet, it
	 * returns a WSAEMSGSIZE error.  Ignore this message.
	 */
	error = DP_GetLastError();
	if (error != DP_EMSGSIZE) {
	    Tcl_ResetResult(interp);
	    if (error == DP_EWOULDBLOCK || error == DP_EAGAIN) {
		return TCL_OK;
	    }
	    Tcl_AppendResult(interp, "error reading ", argv[1], ": ",
			     Tdp_Error(interp), (char *) NULL);

	    return TCL_ERROR;
	}
    }

#if defined(SOLARIS) && (OS_MAJOR_VERSION == 5) && (OS_MINOR_VERSION <= 4)
    /*
     * Flush the rest of the datagram (see note above on truncation
     * under Solaris).
     */
    bytesAvail -= count;
    while (bytesAvail) {
	char dummy[4096];
	int bytesRead;
	bytesRead = recvfrom(infoPtr->socket, dummy, sizeof(dummy), 0, NULL, 0);
	bytesAvail -= bytesRead;
    }
#endif

    buffer[count] = 0;
    if (!noaddr) {
	unsigned long theAddr;
	int thePort;
	theAddr = addr.sin_addr.s_addr;
	thePort = ntohs(addr.sin_port);

	addrName = Tdp_CreateAddress(theAddr, thePort);
	Tcl_SetResult(interp, addrName, TCL_STATIC);
	Tcl_AppendElement(interp, buffer);
    } else {
	Tcl_SetResult(interp, buffer, TCL_STATIC);
    }
    return TCL_OK;

syntaxError:
    Tcl_AppendResult(interp,
		     "wrong# args: should be \"", argv[0],
		     " fileId ?numBytes? ?-noaddr? ?-peek?\"",
		     (char *) NULL);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * Tdp_SendToCmd --
 *
 *	This procedure is invoked to process the "dp_sendTo" Tcl/Tk
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	The specified string is written to the file descriptor passed
 *	in.
 *
 *--------------------------------------------------------------
 */
int
Tdp_SendToCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Ignored */
    Tcl_Interp *interp;		/* Tcl interpreter */
    int argc;			/* Number of arguments */
    char *argv[];		/* Arg list */
{
    DP_SockInfo *infoPtr;
    int len, status;
    struct sockaddr_in *addrPtr;
    int sndBufSize, dummy;

    if (argc != 4) {
	Tcl_AppendResult(interp, "wrong# args: should be \"", argv[0],
			 " fileId string address\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 1, 1, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    len = strlen(argv[2]) + 1;

    /*
     * Make sure this is a UDP socket
     */
    if (infoPtr->optFlags & FD_TCP) {
	Tcl_AppendResult(interp, "can't use ", argv[0], " on a TCP socket",
			 (char *) NULL);
	return TCL_ERROR;
    }
    addrPtr = (struct sockaddr_in *) Tdp_FindAddr(argv[3]);
    if (addrPtr == NULL) {
	Tcl_AppendResult(interp, argv[0], ": invalid address \"", argv[3],
			 "\"", (char *) NULL);
	return TCL_ERROR;
    }

#if defined(SOLARIS) || defined(LINUX)
    /*
     * Under SOLARIS, you can send any size message, regardless of the
     * size of the send buffer.  So we have to simulate an error
     * for compatibility.
     *
     * Under linux, dp_sendTo will just hang if the send buffer
     * if too small, so we simulate an error for compatibility.
     */
    dummy = sizeof(int);
    getsockopt(infoPtr->socket, SOL_SOCKET, SO_SNDBUF, (char *)&sndBufSize, &dummy);
    if (len > sndBufSize) {
	Tcl_AppendResult(interp, "error writing ", argv[1],
			 ": Message too long", (char *) NULL);
	return TCL_ERROR;
    }
#endif

    
    Tdp_BlockSocket(infoPtr, TK_WRITABLE|TK_EXCEPTION);
    status = sendto(infoPtr->socket, argv[2], len, 0, (struct sockaddr *) addrPtr,
		    sizeof(struct sockaddr_in));
    if (status != len) {
	Tcl_AppendResult(interp, "error writing ", argv[1], ": ",
			 Tdp_Error(interp), (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *------------------------------------------------------------------
 *
 * Tdp_FdUpdateCmd --
 *
 *	This procedure is invoked to process the "dp_fupdate" Tcl/Tk
 *	command.  Fd is a file descriptor on which a connection is already
 *	establised, but the optflags aren't set for this fd.
 *
 * Results:
 *	a standard tcl result
 *
 * Side effects:
 *	Unpredicted operation may result if the application uses this fd
 *	for anything other than dp_RPC.
 *
 *
 *------------------------------------------------------------------
 */
int
Tdp_FdUpdateCmd(notUsed, interp, argc, argv)
    ClientData notUsed;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    DP_SockInfo *infoPtr;
    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong# args: should be \"", argv[0],
			 " tcp_socket\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 0, 1, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }

#if 0
    /* XXX: Brian, I'm not sure if I'm doing the right thing here--Gordon */
    /*
     * Create the fileId structure.
     */
    Tdp_MakeOpenFile(interp, fd);
#endif

    Tdp_AllocateBuffer(infoPtr);
    infoPtr->optFlags = FD_AUTO_CLOSE | FD_TCP | FD_BLOCKING;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tdp_SocketIsReadyToRead --
 *
 *	This function determines if a file descriptor is readable
 *	and times out after "timeout" interval..
 *
 *----------------------------------------------------------------------
 */
int
Tdp_SocketIsReadyToRead(infoPtr, timeout)
    DP_SockInfo *infoPtr;
    int timeout;
{
    fd_set readFdset;
    struct timeval tv;
    int rv;
    FD_ZERO(&readFdset);
    FD_SET(infoPtr->socket, &readFdset);

    tv.tv_sec = timeout;
    tv.tv_usec = 0;

    select(infoPtr->sockId + 1, (SELECT_MASK *) & readFdset, (SELECT_MASK *) NULL,
	   (SELECT_MASK *) NULL, &tv);
    if (FD_ISSET(infoPtr->socket, &readFdset)) {
	rv = TCL_READABLE;
    } else {
	rv = 0;
    }
    return rv;
}

/*
 *----------------------------------------------------------------------
 *
 * Tdp_GetsCmd --
 *
 *	This procedure is invoked to process the "dp_gets" Tcl command.
 *
 * Results:
 *	A standard Tcl result.
 *
 *----------------------------------------------------------------------
 */
int
Tdp_GetsCmd(notUsed, interp, argc, argv)
    ClientData notUsed;		/* Not used. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
#define BUF_SIZE 200
    char buffer[BUF_SIZE + 1];
    int totalCount, done, flags;
    DP_SockInfo *infoPtr;
    FILE *f;
    int error;
    if ((argc != 2) && (argc != 3)) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " fileId ?varName?\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 0, 1, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Timeout value is 10 seconds
     */
    if (Tdp_SocketIsReadyToRead(infoPtr, 10) == 0) {
	Tcl_AppendResult(interp, "Server not responding", (char *) NULL);
	return TCL_ERROR;
    }
    f = infoPtr->filePtr;

    /*
     * We can't predict how large a line will be, so read it in pieces,
     * appending to the current result or to a variable.
     */
    totalCount = 0;
    done = 0;
    flags = 0;
    clearerr(f);
    while (!done) {
	register int c, count;
	register char *p;
	for (p = buffer, count = 0; count < BUF_SIZE - 1; count++, p++) {
	    c = getc(f);
	    if (c == EOF) {
		if (ferror(f)) {
		    /*
		     * If the file is in non-blocking mode, return any bytes
		     * that were read before a block would occur.
		     */
		    error = DP_GetLastError();
		    if ((error == DP_EWOULDBLOCK)
			&& ((count > 0 || totalCount > 0))) {
			done = 1;
			break;
		    }
		    Tcl_ResetResult(interp);
		    Tcl_AppendResult(interp, "error reading \"", argv[1],
			     "\": ", Tdp_Error(interp), (char *) NULL);
		    return TCL_ERROR;
		} else if (feof(f)) {
		    if ((totalCount == 0) && (count == 0)) {
			totalCount = -1;
		    }
		    done = 1;
		    break;
		}
	    }
	    if (c == '\n') {
		done = 1;
		break;
	    }
	    *p = c;
	}
	*p = 0;
	if (argc == 2) {
	    Tcl_AppendResult(interp, buffer, (char *) NULL);
	} else {
	    if (Tcl_SetVar(interp, argv[2], buffer, flags | TCL_LEAVE_ERR_MSG)
		== NULL) {
		return TCL_ERROR;
	    }
	    flags = TCL_APPEND_VALUE;
	}
	totalCount += count;
    }

    if (argc == 3) {
	sprintf(interp->result, "%d", totalCount);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tdp_CloseSocketCmd --
 *
 *	This procedure is invoked to process the "dp_closesocket" Tcl command.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Tdp_CloseSocketCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int result = TCL_OK;
    DP_SockInfo *infoPtr;
    DP_SOCKET socket;

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" sockId\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 0, 0, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }

    infoPtr->handlers = NULL;
    Tk_DeleteFileHandler(infoPtr->socket);

    socket = infoPtr->socket;
    Tdp_RemoveSocket(interp, infoPtr);
#ifdef _WINDOWS
    closesocket(socket);
#endif

    return result;
}

/*
 * ----------------------------------------------------------------------
 *
 * Tdp_ReadBinaryCmd -
 *
 *	This procedure implements a new command, "dp_readbinary".  The
 *	command line has two file descriptors - one for reading from
 *	a socket, and the other is intended for writing to a temporary
 *	file.  No checks are made as to whether the read is non-blocking
 *	or not.
 *
 *  Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *
 */
	/* ARGSUSED */
int
Tdp_ReadBinaryCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
#define BUFSIZE 4048    
    DP_SockInfo *infoPtr;
    FILE *inFile;
    FILE *outFile;
    int numBytes;
    int numRead = 0;
    char buf[BUFSIZE];
    
    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " sockId FileFd\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 0, 0, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Tcl_GetOpenFile(interp, argv[2], 1, 0, &outFile) != TCL_OK) {
	return TCL_ERROR;
    }

    inFile = infoPtr->filePtr;

    /* We have no idea, how large the file will be.  Just read as much
     * as we can into the buffer, until we can read no more
     */
    while (numBytes = fread (buf, sizeof(char), BUFSIZE, inFile)) {
	numRead += numBytes;
	fwrite (buf, sizeof(char), numBytes, outFile);
    }

    /* Return the number of bytes read
     */
    sprintf(interp->result, "%d", numRead);
    return TCL_OK;
}


/*
 * ----------------------------------------------------------------------
 *
 * Tdp_WriteBinaryCmd -
 *
 *	This procedure implements a new command, "dp_writebinary".  The
 *	command line has two file descriptors - one for reading from
 *	a file, and the other is intended for writing to a socket
 *	No checks are made as to whether the write is non-blocking
 *	or not.
 *
 *  Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *
 */
	/* ARGSUSED */
int
Tdp_WriteBinaryCmd(dummy, interp, argc, argv)
    ClientData dummy;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    DP_SockInfo *infoPtr;
    FILE *inFile;
    FILE *outFile;
    int numBytes;
    int numRead = 0;
    char buf[BUFSIZE];
    
    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " sockId FileFd\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tdp_GetOpenSocket(interp, argv[1], 0, 0, &infoPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Tcl_GetOpenFile(interp, argv[2], 0, 0, &inFile) != TCL_OK) {
	return TCL_ERROR;
    }

    outFile = infoPtr->filePtr;

    /* We have no idea, how large the file will be.  Just read as much
     * as we can into the buffer, until we can read no more
     */
    while (numBytes = fread (buf, sizeof(char), BUFSIZE, inFile)) {
	numRead += numBytes;
	fwrite (buf, sizeof(char), numBytes, outFile);
    }

    /* Return the number of bytes read
     */
    sprintf(interp->result, "%d", numRead);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tdp_Tcp_Init -
 *
 *	Initialize the connection management level functions of
 *	Tcl-DP and register them with the given interpreter.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Serveral new commands are added to the interpreter.
 *
 *--------------------------------------------------------------
 */

void
Tdp_Tcp_Init(interp)
    Tcl_Interp *interp;		/* Tcl interpreter */
{
    Tcl_CreateCommand(interp, "dp_isready",
		      (Tcl_CmdProc *) Tdp_IsReadyCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_socketOption",
		      (Tcl_CmdProc *) Tdp_SocketOptionCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_connect",
		      (Tcl_CmdProc *) Tdp_ConnectCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_shutdown",
		      (Tcl_CmdProc *) Tdp_ShutdownCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_accept",
		      (Tcl_CmdProc *) Tdp_AcceptCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_filehandler",
		      (Tcl_CmdProc *) Tdp_FileHandlerCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_send",
		      (Tcl_CmdProc *) Tdp_SendCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_receive",
		      (Tcl_CmdProc *) Tdp_ReceiveCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_packetSend",
		      (Tcl_CmdProc *) Tdp_PacketSendCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_packetReceive",
		      (Tcl_CmdProc *) Tdp_PacketReceiveCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_sendTo",
		      (Tcl_CmdProc *) Tdp_SendToCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_receiveFrom",
		      (Tcl_CmdProc *) Tdp_ReceiveFromCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_gets",
		      (Tcl_CmdProc *) Tdp_GetsCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_fupdate",
		      (Tcl_CmdProc *) Tdp_UpdateCmd,
	      (ClientData) NULL, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_closesocket", Tdp_CloseSocketCmd,
	(ClientData)0, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_readbinary", Tdp_ReadBinaryCmd,
	(ClientData)0, (void (*) _ANSI_ARGS_((ClientData))) NULL);
    Tcl_CreateCommand(interp, "dp_writebinary", Tdp_WriteBinaryCmd,
	(ClientData)0, (void (*) _ANSI_ARGS_((ClientData))) NULL);
}

