/*
 * ns.c --
 *
 *  Introduction -- This file implements the extra Tcl/Tk commands
 *	needed by the name server.
 *
 * Copyright (c) 1995 Cornell University
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for research and educational purposes, without fee, and
 * without written agreement is hereby granted, provided that the above
 * copyright notice and the following two paragraphs appear in all copies
 * of this software.
 *
 * IN NO EVENT SHALL CORNELL UNIVERSITY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF CORNELL
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * CORNELL UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND CORNELL UNIVERSITY HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <tcl.h>
#include <tk.h>
#include <stdio.h>
#include <signal.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include "dpInt.h"
extern int errno;
#include <errno.h>
#if HAVE_ERRNO_H
#    include <errno.h>
#endif
#include <sys/file.h>
#include <sys/ioctl.h>
#ifdef HAVE_FCNTL_H
#    include <fcntl.h>
#endif
#ifdef HAVE_SYS_STREAM_H
#    include <sys/stream.h>
#endif


/*
 *--------------------------------------------------------------
 *
 * Tns_DupCmd --
 *
 *	Implements the "ns_dup" Tcl command.  This duplicates a
 *	file descriptors, specifying the value of the new file
 *	descriptor.
 *
 * Results:
 *	A standard Tcl result
 *
 * Side effects:
 *	From now on, there will be two file descriptors
 *
 *--------------------------------------------------------------
 */
int
Tns_DupCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    int i;
    int fd[2];
    
    /*
     * Initial value needed only to stop compiler warnings.
     */
    fd[0] = 0;
    fd[1] = 0;

    /*
     * Check syntax
     */
    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"",
			  argv[0], "fileid filed",
			  (char *) NULL);
	return TCL_ERROR;
    }

    for (i = 1; i <= 2; i++) {
	if ((argv[i][0] == 'f') && (argv[i][1] == 'i') && (argv[i][2] == 'l')
	    & (argv[i][3] == 'e')) {
	    char *end;

	    fd[i-1] = strtoul(argv[i]+4, &end, 10);
	    if ((end == argv[i]+4) || (*end != 0)) {
		goto badId;
	    }
	} else if ((argv[i][0] == 's') && (argv[i][1] == 't')
		   && (argv[i][2] == 'd')) {
	    if (strcmp(argv[i]+3, "in") == 0) {
		fd[i-1] = 0;
	    } else if (strcmp(argv[i]+3, "out") == 0) {
		fd[i-1] = 1;
	    } else if (strcmp(argv[i]+3, "err") == 0) {
		fd[i-1] = 2;
	    } else {
		goto badId;
	    }
	} else {
badId:
	    Tcl_AppendResult(interp, "bad file identifier \"", argv[i],
			     "\"", (char *) NULL);
	    return TCL_ERROR;
	}
    }

    if (dup2(fd[0], fd[1]) == -1) {
	Tcl_AppendResult(interp, "couldn't dup \"", argv[1], " onto ", argv[2],
			 "\": ", Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * Tns_DaemonizeCmd --
 *
 *	Implements the Tcl "ns_daemonize" command.  This detaches a daemon
 *	process from login session context by forking.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	A new process is started.
 *
 *--------------------------------------------------------------
 */
int
Tns_DaemonizeCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    register int childpid, fd;
    register int i;
    Tcl_DString dstr;

    if (argc != 1) {
	Tcl_AppendResult (interp, "wrong # args: should be \"",
			  argv[0], 
			  (char *) NULL);
	return TCL_ERROR;
    }

    if (getppid() == 1) {
	fprintf(stderr, "I think my parent is inetd");
	goto out;
    }

    /*
     * Ignore the terminal stop signals (BSD)
     */
#ifdef SIGTTOU
    signal (SIGTTOU, SIG_IGN);
#endif
#ifdef SIGTTIN
    signal (SIGTTIN, SIG_IGN);
#endif
#ifdef SIGTSTP
    signal (SIGTSTP, SIG_IGN);
#endif

    /*
     * If we were not started in the background, fork and let the parent
     * exit.  This also guarantees the first child is not a process
     * group leader
     */

    if ((childpid = fork()) < 0) {
	Tcl_AppendResult (interp, "%s: Unable to fork process",
			  argv[0], 
			  (char *) NULL);
	return TCL_ERROR;
    } else if (childpid > 0) {
	exit (0);		/* Parent process exits */
    }

    /*
     * If we get here, we must be the first child process.
     * We now execute the following steps:
     *
     *   1. Become the session leader (setsid()).
     *   2. Change to the dp library directory so the filesystem
     *      we're on can be unmounted.
     *   3. Set the file creation mask to 0.
     *   4. Ensure the process can't reacquire a new controlling terminal.
     *      Under SVR4, this means another fork.  Otherwise, an ioctl()
     *   5. Close any open file descriptors
     */
    setsid();

    /* Construct string for where the library is. */

    Tcl_DStringInit(&dstr);
    Tcl_DStringAppend(&dstr, Tcl_GetVar(interp, "dp_library", TCL_GLOBAL_ONLY), -1);
    Tcl_DStringAppend(&dstr, "/ns", -1);

    chdir (Tcl_DStringValue(&dstr));
    Tcl_DStringFree(&dstr);

    umask(0);

    /*
     * Use this code if we're running BSD...
     */
#if (defined TIOCNOTTY)

    if ( (fd = open("/dev/tty", O_RDWR)) >= 0) {
	ioctl(fd, TIOCNOTTY, (char *) NULL); /* Lose controlling tty */
	close(fd);
    }

    /*
     * Use this code if we're running a System V variant
     */
#else
    signal (SIGHUP, SIG_IGN);
    if ( (childpid = fork()) < 0) {
	Tcl_AppendResult (interp, "%s: Unable to fork process",
			  argv[0], 
			  (char *) NULL);
	return TCL_ERROR;
    } else if (childpid > 0) {
	exit (0);		/* Exit first child */
    }
#endif

out:
    /*
     * Close all file descriptors in the child.
     * We will reopen 0,1,2 below...
     */
    for (i = 0; i < 1024; i++) {
	close (i);
    }

    /*
     * Attach the file descriptors 0, 1, and 2 to /dev/null.
     * Applications expect them to be open.
     */
    errno = 0;
    if ((fd = open("/dev/null", O_RDWR)) != 0) {
	Tcl_AppendResult (interp, "%s: Unable to open /dev/null",
			  argv[0], 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ((fd = dup(0)) != 1) {
	Tcl_AppendResult (interp, "%s: Unable to dup() onto stdout",
			  argv[0], 
			  (char *) NULL);
	return TCL_ERROR;
    }
    if ((fd = dup(0)) != 2) {
	Tcl_AppendResult (interp, "%s: Unable to dup() onto stderr",
			  argv[0], 
			  (char *) NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * Tns_SystimeCmd --
 *
 *	Implement the Tcl "ns_systime" command.  This returns a
 *	string that indicates the current value on the system clock
 *	as a double precision value.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
int
Tns_SystimeCmd (clientData, interp, argc, argv)
    ClientData clientData;         /* ignored */
    Tcl_Interp *interp;            /* tcl interpreter */
    int argc;                      /* Number of arguments */
    char **argv;                   /* Arg list */
{
    struct timeval tv;

    if (argc != 1) {
        Tcl_AppendResult(interp, argv[0], " doesn't take any args", NULL);
        return TCL_ERROR;
    }

    (void) gettimeofday(&tv, NULL);
    sprintf (interp->result, "%d.%06d", tv.tv_sec, tv.tv_usec);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tns_Init --
 *
 *	Initialize the full Tcl-ISIS package.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	ISIS related commands are bound to the interpreter.
 *
 *--------------------------------------------------------------
 */
int
Tns_Init (interp)
    Tcl_Interp *interp;
{
    Tcl_CreateCommand(interp, "ns_daemonize", Tns_DaemonizeCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "ns_dup", Tns_DupCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "ns_systime", Tns_SystimeCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    return TCL_OK;
}
