/*
 * neoXdb.c
 *
 * dbopen interface for Tcl
 * 
 * based on work by
 * Poul-Henning Kamp, phk@data.fls.dk
 * 921027 0.00
 * 921222 0.10	New syntax based on descriptors
 *
 * $Id: neoXdb.c,v 1.2 1996/09/26 18:32:17 kunkee Exp $
 */

/* This file will not be compiled unless TCLXDB_O=tclXdb.c in Makefile */

#ifdef NEO_DB

#include "neo.h"

#define NEO_DBOPEN

#ifdef NEO_DBOPEN

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <memory.h>

#include <db.h>

/* Randy Kunkee Wed Oct 11 11:30:12 CDT 1995 */
/* O_SHLOCK and O_EXLOCK are bsd4.4 specific and may not be in all O/S */
/* below code ripped out of compat.h, which is included by db.h, but */
/* does not define below constants unless __DBINTERFACE_PRIVATE is defined */
#ifndef O_SHLOCK
# define O_SHLOCK 0
#endif
#ifndef O_EXLOCK
# define O_EXLOCK 0
#endif

#define ERR_N_ARG(syntax) { Tcl_AppendResult(interp, "bad # arg: ", \
	"\nSyntax is: ", syntax, 0) ; return TCL_ERROR; }
#define CHKNARG(min, max, syntax) if (argc < min || argc > max) ERR_N_ARG(syntax)

/*
 * t_cldat
 * -------
 */

typedef struct {
    void_pt		handles;
} t_cldat;

/*
 * t_desc
 * ------
 */

typedef struct {
    DB			*db;
    Tcl_HashTable	*cache;
} t_desc;

/*
 * DbProc(cd, interp, argc, argv)
 * ============================
 *
 * Do the actual work.
 *
 */

static int
DbProc (cd, interp, argc, argv)
    t_cldat	*cd;
    Tcl_Interp	*interp;
    int		argc;
    char	**argv;
{
    int i;
    char *p;
    char buf[64];
    int flags = O_RDONLY;
    u_int dbflags;
    int mode = 0644;
    DB *d;
    DBT key, content;
    t_desc *td;
    Tcl_HashEntry *hash;
    int result;
    DBTYPE dbtype;
    Tcl_DString dbPathBuf;

    static char *badDbSeqArgs = "db seq <flag> <sess> [<key>] [<var>]";
    static char *badDbCmdArgs = "db {open|close|get|put|seq|forall|searchall|sync}";
    int varNameIndex;
    int keyIndex;

    argv++; argc--;

    if(!argc)
	ERR_N_ARG(badDbCmdArgs);

/*
 *
 * db open <file> <type> [<flags> [<mode>]]		# return <sess>
 *
 *     Opens the specified <file> using <mode>.
 *
 *     <flags> is a subset of these chars: (default: r)
 *         c  -- create
 *         r -- read mode
 *         w -- read-write mode     
 *	   t -- truncate
 *         l -- shared lock
 *	   L -- exclusive lock
 *	   ? -- non-blocking
 *          
 *      <mode> is a octal integer used for creating files. (default 644)
*/
    if (STREQU (*argv, "open")) {
	CHKNARG (3, 5, "db open <file> <type> [<flags> [<mode>]]");

	if (STREQU (argv[2], "hash")) {
	    dbtype = DB_HASH;
	} else if (STREQU (argv[2], "btree")) {
	    dbtype = DB_BTREE;
	} else {
	    Tcl_AppendResult(interp,
		argv[0],
		": type must be 'hash' or 'btree'",
		(char *)NULL);

	    return TCL_ERROR;
	}

	if (argc > 3) {
	    for (p = argv[3]; *p; p++) {
		switch(*p) {
		    case 'c': flags |= (O_CREAT|O_RDWR); break;
		    case 'r': flags |= O_RDONLY; break;
		    case 'w': flags |= O_RDWR; break;
		    case 't': flags |= (O_TRUNC|O_RDWR); break;
		    case 'l': flags |= (O_SHLOCK); break;
		    case 'L': flags |= (O_EXLOCK); break;
		    case '?': flags |= (O_NONBLOCK); break;
          
		    default:
			Tcl_AppendResult(interp, "what ??: ",
			    "\n<flags> must be a subset of 'rwct'", 0);
			return TCL_ERROR;
		}
	    }
	}

	if (argc > 4) {
	    if (sscanf (argv[4], "%o", &mode) != 1) {
		Tcl_AppendResult (interp, 
		    argv[0],
		    ": mode must be an octal integer",
		    (char *)NULL);
		return TCL_ERROR;
	    }
	}
	mode &= 0777;

        Tcl_DStringInit (&dbPathBuf);
	d = dbopen (Tcl_TildeSubst(interp, argv[1], &dbPathBuf),
	    flags,
	    mode,
	    dbtype,
	    (const void *)NULL);
	Tcl_DStringFree (&dbPathBuf);

	if (d == (DB *)NULL) {
	    Tcl_AppendResult (interp, "couldn't open \"", argv[1],
		"\": ", Tcl_PosixError(interp), (char *)NULL);
	    return TCL_ERROR;
	}
	td = Tcl_HandleAlloc(cd->handles, buf);
	memset (td, 0, sizeof *td);
	td->db = d;
	Tcl_AppendResult (interp, buf, (char *)NULL);
	return TCL_OK;
    }

    /* everything from here needs a handle, but we want #args messages to come
     * from each one in turn to get the syntax presented
     */
    if (argc > 1 && !(td = Tcl_HandleXlate (interp, cd->handles, argv[1])))
	return TCL_ERROR;
/*
 
  db close <sess>
      closes the specified db file.
*/
    if (STREQU (*argv, "close")) {
	CHKNARG (2, 2, "db close <sess>");
	td->db->close (td->db);
	Tcl_HandleFree (cd->handles, td);
	return TCL_OK;
    }
/*
 
  db cache <sess> 
      cache the specified db file.
*/
    if (STREQU (*argv, "cache")) {
	CHKNARG (2, 2, "db cache <sess>");
	if (td->cache) return TCL_OK;
	td->cache = ckalloc(sizeof *(td->cache));
	Tcl_InitHashTable (td->cache, TCL_STRING_KEYS);
	return TCL_OK;
    }
/*
 * db get <sess> <key> [<var>]	
 *    will try to get the value addressed by <key>
 *    if <var> is present it will contain the result, and the return
 *        value will be a boolean: 1 on success, 0 on failure.
 *    else the found value will be returned and failures will give an
 *        error return;
 */
    if (STREQU (*argv, "get")) {
	CHKNARG (3, 4, "db get <sess> <key> [<var>]");
	if (td->cache) {
	    hash = Tcl_FindHashEntry (td->cache, argv[2]);
	    if (hash) {
		content.data = hash->clientData;
		goto gotit;
	    }
	}

	key.data = argv[2];
	key.size = strlen (key.data) + 1;

	result = td->db->get (td->db, &key, &content, 0);

	if (result < 0) {
	    Tcl_AppendResult(interp,
		"Couldn't get: ", Tcl_PosixError(interp), 0);
	    return TCL_ERROR;
	}

	if (td->cache) {
	    hash = Tcl_CreateHashEntry (td->cache, argv[2], &i);
	    if (content.data) {
		hash->clientData = ckalloc (strlen(content.data) + 1);
		strcpy (hash->clientData, content.data);
	    } else {
		hash->clientData = 0;
	    }
	}

      gotit:

	if (argc > 3) {
	    if (result == 1) {
		Tcl_SetResult (interp, "0", TCL_VOLATILE);
		Tcl_SetVar (interp, argv[3], "", 0);
	    } else {
		Tcl_SetResult (interp, "1", TCL_VOLATILE);
		Tcl_SetVar (interp, argv[3], content.data, 0);
	    }
	} else {
	    if (result == 1) {
		Tcl_AppendResult(interp, "no match on key \"",argv[2],"\"",0);
		return TCL_ERROR;
	    }
	    Tcl_SetResult (interp, content.data, TCL_VOLATILE);
	}
	return TCL_OK;
    }
/*
 * db put <sess> <key> <cont> [insert|replace]
 *     puts the <cont> under the <key>. 'replace' is the default mode.
 *     returns an error on failure.
 */
    if (STREQU (*argv, "put")) {
	CHKNARG (4, 5, "db put <sess> <key> <cont> [insert|replace]");
	dbflags = 0;
	if (argc > 4) {
	    if (STREQU(argv[4], "insert"))
		dbflags = R_NOOVERWRITE;
	    else if (STREQU (argv[4], "replace"))
		dbflags = 0;
	    else {
		Tcl_AppendResult (interp,
		   "what ?? either 'insert' or 'replace'",
		   (char *)NULL);
	 	return TCL_ERROR;
	    }
	}

	key.data = argv[2];
	key.size = strlen (key.data) + 1;

	content.data = argv[3];
	content.size = strlen (content.data) + 1;

	result = td->db->put (td->db, &key, &content, dbflags);
	if (result < 0) {
	    Tcl_AppendResult (
		interp,
		"Couldn't put: ",
		Tcl_PosixError(interp),
		(char *)NULL);
	    return TCL_ERROR;
	}

	if (td->cache) {
	    hash = Tcl_CreateHashEntry (td->cache, argv[2], &i);
	    hash->clientData = ckalloc (strlen (content.data) + 1);
	    strcpy (hash->clientData, content.data);
	}

	if (result) {
	    Tcl_AppendResult (
		interp,
		"Duplicate key: \"",
		argv[2],
		"\"",
		(char *)NULL);
	    return TCL_ERROR;
	}
	return TCL_OK;
    }

/*
 *
 * db del <sess> <key>
 *     Deletes the entry with the specified <key> 
 *     If the record is deleted return value is "1" 
 *     else it returns "0" 
 *     probably because <key> wasn't there to begin with 
*/

    if (STREQU (*argv, "del")) {
	CHKNARG (3, 3, "db del <sess> <key>");

	if (td->cache) {
	    hash = Tcl_FindHashEntry (td->cache, argv[2]);

	    if (hash->clientData)
		ckfree (hash->clientData);

	    if (hash)
		Tcl_DeleteHashEntry (hash);
	}

	key.data = argv[2];
	key.size = strlen (key.data) + 1;

	result = td->db->del (td->db, &key, 0);

	if (result) {
	    Tcl_SetResult (interp, "0", TCL_VOLATILE);
	} else {
	    Tcl_SetResult (interp, "1", TCL_VOLATILE);
	}
	return TCL_OK;
    }

/*
 * db seq <sess> <flag> [<key>] [<var>]
 *     	Sequential read.
 *
 *       <flag> can be:
 *          cursor - data associated with key is returned,
 *                   partial matches allowed.
 *
 *          first - the first key/data pair is returned.
 *
 *          last - the last key/data pair is returned.
 *
 *          next - retrieve the next key/data pair after the
 *                 cursor, or first if customer hasn't been set.
 *
 *          prev - retrieve the key/data pair immediately before
 *                 the cursor.
 *
 *     	if <var> is specified:
 *
 *     	    if no keys found:
 *     	        <var> is set to "" and "0" is returned
 *     	    else
 *     	        <var> is set to the key and "1" is returned.
 *     	else 
 *     	    if no keys found:
 *     	        "" is returned
 *     	    else
 *     	        the key is returned
 */
    if (STREQU (*argv, "seq")) {

	CHKNARG (3, 5, badDbSeqArgs);

	varNameIndex = -1;
	keyIndex = -1;

	if (STREQU (argv[2], "cursor")) {
	    CHKNARG (4, 5, badDbSeqArgs);
	    keyIndex = 3;
	    if (argc == 5) {varNameIndex = 4;}
	    dbflags = R_CURSOR;
	} else {
	    CHKNARG (3, 4, badDbSeqArgs);
	    if (argc == 4) varNameIndex = 3;

	    if (STREQU (argv[2], "first")) {
		dbflags = R_FIRST;
	    } else if (STREQU (argv[2], "last")) {
		dbflags = R_LAST;
	    } else if (STREQU (argv[2], "next")) {
		dbflags = R_NEXT;
	    } else if (STREQU (argv[2], "prev")) {
		dbflags = R_PREV;
	    } else {
	    Tcl_AppendResult (
		interp,
		"db seq options are: cursor, first, last, next, prev",
		(char *)NULL);
	    return TCL_ERROR;
	    }
	}

        if (keyIndex >= 0)  {
	    key.data = argv[keyIndex];
	    key.size = strlen (key.data) + 1;
	}

	result = td->db->seq (td->db, &key, &content, dbflags);

	if (result < 0) {
	    Tcl_AppendResult (
		interp,
		"Couldn't seq: ",
		Tcl_PosixError(interp),
		(char *)NULL);
	    return TCL_ERROR;
	}
	if (varNameIndex != -1) {
	    if (result == 0) {
		Tcl_SetVar (interp, argv[varNameIndex], key.data, 0);
		Tcl_SetResult (interp, "1", TCL_VOLATILE);
	    } else {
		Tcl_SetVar(interp, argv[varNameIndex], "", 0);
		Tcl_SetResult(interp, "0", TCL_VOLATILE);
	    }
	} else {
	    if (result == 0)
		Tcl_SetResult (interp, key.data, TCL_VOLATILE);
	}
	return TCL_OK;
    }
/*
 * db forall <sess> <key_var> <proc>
 *     	executes <proc> for all keys found in <tag>
 *     	The actual key_value is available in $<key_var>
 *     	Example:
 *     		xx forall key {
 *     			puts stdout "$key: [xx get $key]"
 *     		}
 *
 */
    if STREQU (*argv, "forall") {
	CHKNARG (4, 4, "db forall <sess> <key_var> <proc>");
        result = td->db->seq (td->db, &key, &content, R_FIRST);

	if (result < 0) {
	    Tcl_AppendResult (
		interp,
		"Couldn't forall: ",
		Tcl_PosixError(interp),
		(char *)NULL);
	    return TCL_ERROR;
	}

	if (result == 1)
	    return TCL_OK;

	for (;;) {
	    Tcl_SetVar (interp, argv[2], key.data, 0);
	    result = Tcl_VarEval (interp, argv[3], 0);

	    if (result == TCL_BREAK) 
		break;
	    else if (result != TCL_OK && result != TCL_CONTINUE)
		return result;

	    result = td->db->seq (td->db, &key, &content, R_NEXT);
	    if (result == 1) break;
	}
	return TCL_OK;
    }

/*
 * db searchall <sess> <keyvar> [-<searchtype>] <pattern> <proc>
 *     	executes <proc> for all keys found in <tag>
 *     	The actual key_value is available in $<key_var>
 *     	Example:
 *     		db searchall $db key -glob "*foo*" {
 *     			puts stdout "$key: [db get $key]"
 *     		}
 *
 */
    if STREQU (*argv, "searchall") {
	int searchMatch, searchMode;
#define EXACT   0
#define GLOB    1
#define REGEXP  2

	CHKNARG (5, 6, "db searchall <sess> <key_var> [-<searchtype>] <pattern> <proc>");
        result = td->db->seq (td->db, &key, &content, R_FIRST);

	if (result < 0) {
	    Tcl_AppendResult (
		interp,
		"Couldn't searchall: ",
		Tcl_PosixError(interp),
		(char *)NULL);
	    return TCL_ERROR;
	}

	if (result == 1)
	    return TCL_OK;

                                         
	searchMode = GLOB;
	if (argc == 6) {
	    if (strcmp(argv[3], "-glob") == 0) {
		searchMode = GLOB;
	    } else if (strcmp(argv[3], "-regexp") == 0) {
		searchMode = REGEXP;
	    } else {
		Tcl_AppendResult(interp, "bad search mode \"", argv[3],
			"\": must be -glob or -regexp", (char *) NULL);
		return TCL_ERROR;
	    }
	}

	for (;;) {
	    searchMatch = 0;
	    switch (searchMode) {
		case GLOB:
		    searchMatch = Tcl_StringMatch(key.data, argv[argc-2]);
		    break;
		case REGEXP:
		    searchMatch = Tcl_RegExpMatch(interp, key.data, argv[argc-2]);
		    if (searchMatch < 0) {
			return TCL_ERROR;
		    }
		    break;
	   }

           if (searchMatch) {
	        Tcl_SetVar (interp, argv[2], key.data, 0);
	        result = Tcl_VarEval (interp, argv[argc-1], 0);

	        if (result == TCL_BREAK) 
		    break;
	        else if (result != TCL_OK && result != TCL_CONTINUE)
		    return result;
           }

	    result = td->db->seq (td->db, &key, &content, R_NEXT);
	    if (result == 1) break;
	}
	return TCL_OK;
    }

    if STREQU (*argv, "sync") {
	CHKNARG (2, 2, "db sync <sess>");
        result = td->db->sync (td->db, 0);

	if (result < 0) {
	    Tcl_AppendResult (
		interp,
		"Couldn't sync: ",
		Tcl_PosixError(interp),
		(char *)NULL);
	    return TCL_ERROR;
	}
	return TCL_OK;
    }

    Tcl_AppendResult(interp, "what ??: ",
	"\nSyntax is:", 
	badDbCmdArgs,
	0 );
    return TCL_ERROR;
}

/*
 * Neo_dbInit(interp)
 * ====================
 *
 * Initialize the db interface for the interpreter 'interp'
 *
 */

void
Neo_dbInit (interp)
Tcl_Interp	*interp;
{
    t_cldat *cd;

    cd = (t_cldat *)ckalloc (sizeof *cd);
    memset ((void*)cd, 0, sizeof *cd);
    cd->handles = Tcl_HandleTblInit ("db", sizeof(t_desc), 10);
    Tcl_CreateCommand (interp, "db", DbProc, (ClientData)cd, 0);

    /* Postgres 95 */
    /* Pg_Init(interp); */
}

#else

void
Neo_dbInit (interp)
Tcl_Interp	*interp;
{
}

#endif
#endif /* NEO_DB */
