'\"
'\" Copyright (c) 1995 AT&T Bell Laboratories
'\"
'\" See the file "license.terms" for information on usage and redistribution
'\" of this file, and for a DISCLAIMER OF ALL WARRANTIES.
'\" 
'\" $Id: namespace.n,v 1.1.1.1 1996/09/10 06:54:31 karl Exp $
'\" 
.so man.macros
'\"	# CS - begin display of code example
.de CS
.in +0.4i
.nf
\fC
..
'\"	# CE - end display of code example
.de CE
\fP
.fi
.in -0.4i
..
.TH namespace n "" Tcl "Tcl With Namespaces"
.BS
'\" Note:  do not modify the .SH NAME line immediately below!
.SH NAME
namespace \- evaluate scripts in a particular namespace context
.SH SYNOPSIS
\fBnamespace \fIname \fR?\fB-local\fR? \fR?\fB-enforced \fIstate\fR? ?\fB--\fR? \fIscript\fR?
.BE

.SH DESCRIPTION
.PP
The \fInamespace\fR command finds or creates a namespace \fIname\fR
and executes an optional \fIscript\fR code string in that context.
This command is usually used to add commands and variables, or to
redefine commands and variables, in a namespace.
When the \fIscript\fR is executed, the \fInamespace\fR command adds
a call frame to the stack, so the \fInamespace\fR command counts as
another level for the \fIuplevel\fR and \fIupvar\fR commands.
.PP
If the \fB-local\fR flag is included, the namespace \fIname\fR will be
found or created in the local namespace context.  Otherwise, \fIname\fR
could reference any namespace found by following the import list
(see below for rules regarding name resolution).  The \fB-local\fR
flag allows a child namespace to be created even if another namespace
can be found with the same name.
.PP
If the \fB-enforced\fR flag is included, its argument \fIstate\fR is
a boolean value enabling or disabling special rules for command/variable
name resolution.  This makes it possible to create "safe" namespaces
with restricted access to commands.  See the section on safe namespaces
below for more details.
.PP
The optional "\fB--\fR" marks the end of the option list.  The
next argument is treated as a command script, even if it starts
with a "\fB-\fR" character.

.SH OVERVIEW
.PP
A namespace is a collection of commands and global variables that
is kept apart from the usual global scope.  This allows Tcl code
libraries to be packaged in a well-defined manner, and prevents
unwanted interactions with other libraries.  A namespace can also
have child namespaces within it, so one library can contain its
own private copy of many other libraries.  The global scope
(named "\fC::\fR") is the root namespace for an interpreter; all
other namespaces are contained within it.
.PP
Commands and variables local to a namespace can be referenced
using simple names within the namespace.  But in another context,
they must be qualified by the namespace name.  Namespace qualifiers
leading up to a command or variable name are separated by the
"\fC::\fR" string.
.PP
If a command, variable or namespace name is absolutely qualified
(i.e., it starts with "::") then it is accessed directly.
For example, the name "::foo::bar::x" refers to the variable "x",
which is in the namespace "bar", which is in the namespace "foo",
which is in the global namespace.  If the name does not start with
"::", it is treated relative to the current namespace context.
Lookup starts in the current namespace, then continues through all other
namespaces included on the "import" list.  The "\fBinfo which\fR" command
can be used to check the results of name resolution.  Whenever a name
is resolved, the result is cached to keep namespace performance on par
with vanilla Tcl.
.PP
By default, the import list for each namespace includes
the parent namespace, so commands at the global scope can be accessed
transparently.  The import list can be modified by using the \fIimport\fR
command within a namespace.
.PP
By default, commands and variables are "public" and can be accessed
from any other namespace context.  The commands "protected" and "private"
can be used when defining commands and variables to restrict access to them.
These commands designate which parts of a library are meant to be accessed
by users, and which parts are not.
.PP
A namespace can be deleted using the "\fBdelete namespace\fR" command.
This deletes all commands and variables in the namespace, and deletes
all child namespaces as well.

.SH EXAMPLE
The following namespace implements a simple counter facility:
.CS
namespace counter {
    variable x 0

    proc restart {{by 1}} {
        global x
        set x 0
        next $by
    }

    proc next {{by 1}} {
        global x
        incr x $by
        return $x
    }
}

puts "Count up by 2..."
puts "  count: [counter::next 2]"
puts "  count: [counter::next 2]"
puts "  count: [counter::next 2]"
puts "restart: [counter::restart 2]"
puts "  count: [counter::next 2]" \fR
.CE

The \fCcounter\fR namespace contains two commands (\fCrestart\fR and
\fCnext\fR) and one global variable (\fCx\fR).  All of these elements
are protected by the namespace, so there is no conflict, for example,
between "counter::x" and another variable named "x" in the global
namespace.
.PP
Procedures execute in the context of the namespace that contains them.
Within \fCcounter::restart\fR and \fCcounter::next\fR, the variable name
"x" refers to "::counter::x".  Within \fCcounter::restart\fR, the command
"next" refers to "::counter::next".  Outside of the namespace, these names
must be explicitly qualified with the "counter::" namespace path.

.SH SAFE NAMESPACES
Namespaces include a special enforcement feature that can be activated
using the \fB-enforced\fR flag.  When enforcement is turned on, command
and variable references can be intercepted, and the usual lookup rules
can be modified.  This supports the construction of "safe" namespaces,
which interpret code from an untrusted source and deny access to commands
which could damage the system.
.PP
Whenever a command name is encountered, the namespace facility checks
to see if the current namespace context is enforced.  If it is not,
the usual name resolution rules are carried out.  If it is, the namespace
facility executes the following command in that context:
.CS
enforce_cmd \fIname\fP
.CE
If this procedure returns an error, access to that command is denied.
If it returns a null string, name resolution continues according to
the usual rules.  Otherwise, it should return the same string \fIname\fR,
or the name of another command to be substituted in its place.  This
procedure is only invoked the first time a command \fIname\fR is
encountered.  The results are cached in the name resolution tables,
so performance is not adversely affected.
.PP
Variable references are handled the same way, except that the following
command is invoked to resolve the reference:
.CS
enforce_var \fIname\fP
.CE
Note that enforcement is carried out before any of the usual name
resolution rules come into play.  Because of this, even absolute
references like "::exec" or "::counter::next" can be intercepted
and dealt with.
.PP
Because the enforcement procedures apply to all of the command/variable
references in a namespace, it can be difficult to define procedures in
an enforced namespace and have them work correctly.  If you deny access
to the "proc" command, for example, you will not be able to define any
procedures in the namespace.  To avoid problems like this, it is usually
better to use enforced namespaces as follows.  Set up a namespace
containing the \fCenforce_cmd\fR and \fCenforce_var\fR procedures,
along with any other code needed to enforce the namespace.  Within
that namespace, include a child namespace that is empty, but has
enforcement turned on.  Commands can be fed to the child namespace,
which will automatically look to its parent for the enforcement
procedures and all other commands/variables.  Procedures may be
referenced from the child, but they will actually execute in the
parent namespace, which is not enforced.
.PP
In the following example, a "safe" namespace is constructed which
will interpret any command string, but will guard access to commands
like "exec" and "open" which are considered harmful.  Calls to "exec"
are intercepted and sent to "safe_exec" for execution.  This logs
the offending command in a file "security.log" and returns the null
string.  Calls to "open" are intercepted and sent to "safe_open".
This allows read access to ordinary files, but blocks write operations
and execution of processes.  Note that the interception and redirection
of commands happens only when commands are interpreted in the namespace
"safe::isolated".  In procedures like "safe_exec" and "safe_open", which
are interpreted in namespace "safe", access to "exec" and "open" is
allowed.
.CS
namespace safe {

    proc interpret {cmds} {
        namespace isolated $cmds
    }

    proc safe_exec {args} {
        set mesg "access denied: $args"
        puts stderr $mesg
        catch {
            set fid [open "security.log" a]
            puts $fid $mesg
            close $fid
        }
    }

    proc safe_open {args} {
        set file [lindex $args 0]
        if {[string match |* $file]} {
            error "cannot open process: $file"
        }
        set access [lindex $args 1]
        if {$access == "r"} {
            return [eval open $args]
        }
        error "cannot open with write access: [lindex $args 0]"
    }

    proc enforce_cmd {name} {
        global commands
        if {[info exists commands($name)]} {
            return $commands($name)
        }
        return $name
    }
    set commands(exec) safe_exec
    set commands(::exec) safe_exec
    set commands(open) safe_open
    set commands(::open) safe_open

    proc enforce_var {name} {
        if {[string match *::* $name]} {
            error "variable access denied: $name"
        }
        return $name
    }

    namespace isolated -local -enforced yes
}

#
# Use this to interpret a nasty script:
#
safe::interpret {
    #
    # Files can be read but not written.
    #
    set cmd {
        set fid [open "/etc/passwd" r]
        set info [read $fid]
        close $fid
    }
    puts "read: [catch $cmd result] => $result"

    set cmd {
        set fid [open "$env(HOME)/.cshrc" w]
        puts $fid "# ha! ha!
        puts $fid "# make the user think his files have been erased"
        puts $fid "alias ls 'echo "total 0"'
        close $fid
    }
    puts "write: [catch $cmd result] => $result"

    #
    # Kill all of the jobs we can find!
    #
    set processInfo [lrange [split [exec ps -gx] \n] 1 end]
    foreach line $processInfo {
        set pid [lindex $line 0]
        exec kill -9 $pid
    }
}
.CE

.SH KEYWORDS
delete, import, private, protected, public, variable
