/* 
 * tclLoadDl.c --
 *
 *	This procedure provides a version of the TclLoadFile that
 *	works with the "dlopen" and "dlsym" library procedures for
 *	dynamic loading.
 *
 * Copyright (c) 1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tclLoadDl.c 1.7 96/03/14 09:03:33
 */

#include "tclInt.h"
#ifdef NO_DLFCN_H
#   include "../compat/dlfcn.h"
#else
#   include <dlfcn.h>
#endif

extern char *		strcpy _ANSI_ARGS_((char *dst, CONST char *src));
extern char *		strcat _ANSI_ARGS_((char *dst, CONST char *src));

/*
 * In some systems, like SunOS 4.1.3, the RTLD_NOW flag isn't defined
 * and this argument to dlopen must always be 1.  The RTLD_GLOBAL
 * flag is needed on some systems (e.g. SCO and UnixWare) but doesn't
 * exist on others;  if it doesn't exist, set it to 0 so it has no effect.
 */

#ifndef RTLD_NOW
#   define RTLD_NOW 1
#endif

#ifndef RTLD_GLOBAL
#   define RTLD_GLOBAL 0
#endif

/*
 *----------------------------------------------------------------------
 *
 * TclLoadFile --
 *
 *	Dynamically loads a binary code file into memory and returns
 *	the addresses of two procedures within that file, if they
 *	are defined.
 *
 * Results:
 *	A standard Tcl completion code.  If an error occurs, an error
 *	message is left in interp->result.  *proc1Ptr and *proc2Ptr
 *	are filled in with the addresses of the symbols given by
 *	*sym1 and *sym2, or NULL if those symbols can't be found.
 *
 * Side effects:
 *	New code suddenly appears in memory.
 *
 *----------------------------------------------------------------------
 */

int
TclLoadFile(interp, fileName, sym1, sym2, proc1Ptr, proc2Ptr)
    Tcl_Interp *interp;		/* Used for error reporting. */
    char *fileName;		/* Name of the file containing the desired
				 * code. */
    char *sym1, *sym2;		/* Names of two procedures to look up in
				 * the file's symbol table. */
    Tcl_PackageInitProc **proc1Ptr, **proc2Ptr;
				/* Where to return the addresses corresponding
				 * to sym1 and sym2. */
{
    VOID *handle = NULL;
    char *buffer,*dirName;
    Tcl_DString buf;

    buffer = Tcl_TranslateFileName(interp, fileName, &buf);
    if (buffer[0]!='/') {
	dirName = TclGetCwd(interp);
	Tcl_DStringSetLength(&buf,0);
	Tcl_DStringAppend(&buf,dirName,-1);
	Tcl_DStringAppend(&buf,"/",-1);
	Tcl_DStringAppend(&buf,fileName,-1);
	buffer = Tcl_DStringValue(&buf);
    }
    handle = dlopen(buffer, RTLD_NOW | RTLD_GLOBAL);
    Tcl_DStringFree(&buf);
    
    if (handle == NULL) {
	Tcl_AppendResult(interp, "couldn't load file \"", fileName,
		"\": ", dlerror(), (char *) NULL);
	return TCL_ERROR;
    }
    *proc1Ptr = (Tcl_PackageInitProc *) dlsym(handle, sym1);
    if (*proc1Ptr == NULL) {
	buffer = ckalloc(strlen(sym1)+18);
	buffer[0] = '_';
	strcpy(buffer+1, sym1);
	*proc1Ptr = (Tcl_PackageInitProc *) dlsym(handle, buffer);
	if (*proc1Ptr == NULL) {
	    strcat(buffer,"__FP10Tcl_Interp");
	    *proc1Ptr = (Tcl_PackageInitProc *) dlsym(handle, buffer+1);
	    if (*proc1Ptr == NULL) {
		*proc1Ptr = (Tcl_PackageInitProc *) dlsym(handle, buffer);
	    }
	}
	ckfree(buffer);
    }
    *proc2Ptr = (Tcl_PackageInitProc *) dlsym(handle, sym2);
    if (*proc2Ptr == NULL) {
	buffer = ckalloc(strlen(sym2)+18);
	buffer[0] = '_';
	strcpy(buffer+1, sym2);
	*proc2Ptr = (Tcl_PackageInitProc *) dlsym(handle, buffer);
	if (*proc2Ptr == NULL) {
	    strcat(buffer,"__FP10Tcl_Interp");
	    *proc2Ptr = (Tcl_PackageInitProc *) dlsym(handle, buffer+1);
	    if (*proc2Ptr == NULL) {
		*proc2Ptr = (Tcl_PackageInitProc *) dlsym(handle, buffer);
	    }
	}
	ckfree(buffer);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TclGuessPackageName --
 *
 *	If the "load" command is invoked without providing a package
 *	name, this procedure is invoked to try to figure it out.
 *
 * Results:
 *	Always returns 0 to indicate that we couldn't figure out a
 *	package name;  generic code will then try to guess the package
 *	from the file name.  A return value of 1 would have meant that
 *	we figured out the package name and put it in bufPtr.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TclGuessPackageName(fileName, bufPtr)
    char *fileName;		/* Name of file containing package (already
				 * translated to local form if needed). */
    Tcl_DString *bufPtr;	/* Initialized empty dstring.  Append
				 * package name to this if possible. */
{
    return 0;
}
