/* 
 * tkMacInit.c --
 *
 *	This file contains Mac-specific interpreter initialization
 *	functions.
 *
 * Copyright (c) 1995-1996 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkMacInit.c 1.19 96/04/05 10:30:43
 */

#include <Files.h>
#include <TextUtils.h>
#include <Strings.h>
#include "tkInt.h"
#include <tclInt.h>
#include "tkMacInt.h"
#include "tclMacInt.h"

#ifndef TCL_ACTIVE
#define TCL_ACTIVE (1<<4)
#endif

static void		DisplayCheckProc _ANSI_ARGS_((ClientData clientData,
			    int flags));
static void		DisplaySetupProc _ANSI_ARGS_((ClientData clientData,
			    int flags));

/*
 *----------------------------------------------------------------------
 *
 * DisplaySetupProc --
 *
 *	This procedure is part of the event source for Mac displays.
 *	It is invoked by Tcl_DoOneEvent before it calls select to check
 *	for events on all displays. Because for the Mac this kind of
 *	events is handled already in Tcl, all that needs to be done is
 *	check if there are any windows open.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Tells the notifier whether the Tk event-handler is active or not.
 *
 *----------------------------------------------------------------------
 */

static void
DisplaySetupProc(clientData, flags)
    ClientData clientData;		/* Not used. */
    int flags;				/* Flags passed to Tk_DoOneEvent:
					 * if it doesn't include
					 * TCL_WINDOW_EVENTS then we do
					 * nothing. */
{
    if ((flags & TCL_WINDOW_EVENTS) && (Tk_GetNumMainWindows()>0)) {
	Tcl_WatchFile((Tcl_File) NULL, TCL_ACTIVE);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DisplayCheckProc --
 *
 *	This procedure is just a dummy function, because Mac events
 *	are handled in Tcl.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
DisplayCheckProc(clientData, flags)
    ClientData clientData;		/* Not used. */
    int flags;				/* Not used. */
{
    return;
}

/*
 *----------------------------------------------------------------------
 *
 * TkPlatformInit --
 *
 *	Performs Mac-specific interpreter initialization related to the
 *      tk_library variable.
 *
 * Results:
 *	A standard Tcl completion code (TCL_OK or TCL_ERROR).  Also
 *	leaves information in interp->result.
 *
 * Side effects:
 *	Sets "tk_library" Tcl variable, runs initialization scripts
 *	for Tk.
 *
 *----------------------------------------------------------------------
 */

static char **initScripts = NULL;

int
TkPlatformInit(interp)
    Tcl_Interp *interp;
{
    char *libDir, *tempPath;
    Tcl_DString path;
    FSSpec macDir;
    int result;
    static char initResCmd[] =
    	"source -rsrc tk\n\
    	source -rsrc button\n\
    	source -rsrc entry\n\
    	source -rsrc listbox\n\
    	source -rsrc menu\n\
    	source -rsrc scale\n\
    	source -rsrc scrollbar\n\
    	source -rsrc text\n\
    	source -rsrc dialog\n\
    	source -rsrc focus\n\
    	source -rsrc optionMenu\n\
    	source -rsrc palette\n\
    	source -rsrc tearoff\n\
    	source -rsrc tkerror\n\
    	";
    static char initCmd[] =
	"if [file exists [file join $tk_library tk.tcl]] {\n\
	    source [file join $tk_library tk.tcl]\n\
	    source [file join $tk_library button.tcl]\n\
	    source [file join $tk_library entry.tcl]\n\
	    source [file join $tk_library listbox.tcl]\n\
	    source [file join $tk_library menu.tcl]\n\
	    source [file join $tk_library scale.tcl]\n\
	    source [file join $tk_library scrlbar.tcl]\n\
	    source [file join $tk_library text.tcl]\n\
	} else {\n\
	    set msg \"can't find tk resource or [file join $tk_library tk.tcl];\"\n\
	    append msg \" perhaps you need to\\ninstall Tk or set your \"\n\
	    append msg \"TK_LIBRARY environment variable?\"\n\
	    error $msg\n\
	}";
    static int initialized = 0;

    if (!initialized) {
	Tcl_CreateEventSource(DisplaySetupProc, DisplayCheckProc,
		(ClientData) NULL);
	initialized = 1;
    }

    Tcl_DStringInit(&path);

    /*
     * The tk_library path can be found in several places.  Here is the order
     * in which the are searched.
     *		1) the variable may already exist
     *		2) env array
     *		3) System Folder:Extensions:Tool Command Language:
     *		4) use TK_LIBRARY - which probably won't work
     */
     
    libDir = Tcl_GetVar(interp, "tk_library", TCL_GLOBAL_ONLY);
    if (libDir == NULL) {
	libDir = Tcl_GetVar2(interp, "env", "TK_LIBRARY", TCL_GLOBAL_ONLY);
    }
    if (libDir == NULL) {
	tempPath = Tcl_GetVar2(interp, "env", "EXT_FOLDER", TCL_GLOBAL_ONLY);
	if (tempPath != NULL) {
	    Tcl_DString libPath;
	    
	    Tcl_JoinPath(1, &tempPath, &path);
	    
	    Tcl_DStringInit(&libPath);
	    Tcl_DStringAppend(&libPath, ":Tool Command Language:lib:itcl:tk", -1);
	    Tcl_DStringAppend(&libPath, TK_VERSION, -1);
	    Tcl_JoinPath(1, &libPath.string, &path);
	    Tcl_DStringFree(&libPath);
	    
	    if (FSpLocationFromPath(path.length, path.string, &macDir) == noErr) {
	    	libDir = path.string;
	    }
	}
    }
    if (libDir == NULL) {
	libDir = TK_LIBRARY;
    }

    /*
     * Assign path to the global Tcl variable tcl_library.
     */
    Tcl_SetVar(interp, "tk_library", libDir, TCL_GLOBAL_ONLY);
    Tcl_DStringFree(&path);

    if (initScripts != NULL) {
	char **p = initScripts;
	Tcl_DString data;

	Tcl_SetVar(interp, "tk_library", "", TCL_GLOBAL_ONLY);
	Tcl_DStringInit(&data);
	while(*p) {
	    /* Copy the constant into a dynamic string. This */
	    /* is necessary because Tcl7.5 doesn't accept    */
	    /* constants as an argument to Tcl_Eval()        */
	    Tcl_DStringSetLength(&data,0);
	    Tcl_DStringAppend(&data,*p++,-1);
	    if(Tcl_Eval(interp,Tcl_DStringValue(&data)) == TCL_ERROR) {
		Tcl_DStringFree(&data);
		return TCL_ERROR;
	    }
	}
	Tcl_DStringFree(&data);
	return TCL_OK;
    }

    result = Tcl_Eval(interp, initResCmd);
    if (result != TCL_OK) {
	result = Tcl_Eval(interp, initCmd);
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * TkSetInitScript --
 *
 *	This procedure sets the initialization script used in Tk_Init().
 *	It will be used instead of the file "tk.tcl" in all future calls
 *	to Tk_Init. If the argument is NULL, tk.tcl will be used again.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None
 *
 *----------------------------------------------------------------------
 */

void
TkSetInitScript(script)
    char  **script;		/* Script to be executed. */
{
    initScripts = script;
}
