#@package: array_utilities copy_array array_to_keyed_list keyed_list_to_array are_arrays_identical
#
# Array Utility Functions
#
#
# $Id: arrayutil.tcl,v 1.1 1996/08/24 19:48:49 karl Exp $

#
# Copy the contents of one array into another.
#
proc copy_array {sourceArrayName destArrayName} {
    upvar $sourceArrayName sourceArray $destArrayName destArray

    set searchId [array startsearch sourceArray]
    while {[array anymore sourceArray $searchId]} {
	set elementName [array nextelement sourceArray $searchId]
	set destArray($elementName) sourceArray($elementName)
    }
    array donesearch sourceArray $searchId
}

#
# Convert an array to a keyed list.  Only handles "flat" keyed lists.
#
proc array_to_keyed_list {arrayName} {
    upvar $arrayName array

    set list ""
    foreach key [array names array] {
        keylset list $key $array($key)
    }
    return $list
}

#
# Convert a keyed list to an array.  Only handles "flat" keyed lists.
#
proc keyed_list_to_array {list arrayName} {
    upvar $arrayName array

    foreach key [keylkeys list] {
        set array($key) [keylget list $key]
    }
}

#
# Return true if two arrays are identical, else false.
#
proc are_arrays_identical {array1Name array2Name} {
    upvar $array1Name array1 $array2Name array2

    # if they don't have the same number of elements, no way are they identical
    if {[array size array1] != [array size array2]} {return 0}

    # For each element in the first array,
    #    If the element isn't in the second array, they're not identical.
    #    If the contents of both elements aren't the same, the arrays aren't
    #      identical.
    # If you get to the end, the arrays are identical.

    set searchId [array startsearch array1]
    while {[array anymore array1 $searchId]} {
	set elementName [array nextelement array1 $searchId]
	if ![info exists array2($elementName)] {return 0}
	if {$array1($elementName) != $array2($elementName)} {
	    return 0
	}
    }
    array donesearch sourceArray $searchId
    return 1
}

#@package: catchers fail_exit die

#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

# $Id: catchers.tcl,v 1.2 1996/09/28 20:54:25 kunkee Exp $

#
#  fail_exit command [exitStatus]
#      execute command and, if it gets a tcl error, write the program
#      name and error message to stderr and exit the program.
#      If exit is taken, exit status is 1 unless specified.
#
proc fail_exit {command {exitStatus 1}} {
    if {[catch {uplevel $command} result] == 1} {
        global argv0
	puts stderr "$argv0: $result"
	exit $exitStatus
    }
}

proc die {message {exitStatus 1}} {
    puts stderr $message
    exit $exitStatus
}
#@package: EventLoop EventLoop

package require Itcl

#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

### 
#
#
# $Id: eventloop.tcl,v 1.2 1996/09/28 20:54:26 kunkee Exp $
#
# Incr Tcl Event Loop class.
#
# For use with Extended Tcl, but not Wishx, which has its
# own mechanism (addinput).  A functionally identical EventLoop 
# class for Wishx would be really nice.
#

itcl_class EventLoop {

    constructor {config} {
    }

    destructor {
    }

    # make a list containing the file handles of all the sockets,
    # pipes, fifos, etc, we are waiting for data from.
    method make_select_list {} {
	set readSelectList [array names readSelectorCallouts]
    }

    #
    # run an event loop, waiting for input from any of the file handles
    # we've been told to look at, and calling the timeoutCallout,
    # if it exists, every selectTimeout floating point seconds.
    #
    method run {} {
	set running 1
	make_select_list

	while {$running} {

	    set selectResult \
		[select $readSelectList "" "" $selectTimeout]
	    set readReadyList [lindex $selectResult 0]

            if {$readReadyList == ""} {
		eval $timeoutCallout
	    }
	    foreach readReady $readReadyList {
		set command $readSelectorCallouts($readReady)
		if {[gets $readReady line] < 0} {
		    close $readReady
		    continue
		}
		[lindex $command 0] [lindex $command 1] $line
	    }
	}
    }

    # stop the event loop after the event that invoked this method
    # finishes
    method stop {} {
	set running 0
    }

    # add a trigger whereby data on the specified filehandle causes
    # command to execute
    method add_trigger {file command} {
	set readSelectorCallouts($file) $command
    }

    # remove the trigger on the specified filehandle
    method remove_trigger {file} {
	unset readSelectorCallouts($file)
    }

    # true while the event loop is running
    protected running 0

    # array where keys are file handles and data are commands to execute
    protected readSelectorCallouts

    # list of file handles to select on as a list
    protected readSelectList

    # timeout interval, default 10 seconds
    public selectTimeout 10

    # timeout callout code, default is to do nothing
    public timeoutCallout ""
}


#@package: getargs getargs

#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

# $Id: getargs.tcl,v 1.2 1996/09/28 20:54:26 kunkee Exp $

#
# parses an arglist of key-value pairs where the key starts with a
# dash and the value must always be present.
#
# we want to use it in this manner 
#
# dialog_box -bitmap @/bitmap -text "hi there" -suppress 1
#
# in dialog_box,
#
# proc dialog_box {blah blah blah args} {
#     getargs $args parms
# }
#
# and have the parms array set as follows:
#
# bitmap=@/bitmap
# text="hi there"
# suppress="1"
#
#

proc getargs {arrayName argList} {
    upvar $arrayName array

    set length [llength $argList]
    if {$length % 2 != 0} {
        error "list of key-value pairs is missing a value"
    }
    for {set i 0} {$i < $length} {incr i 2} {
        set key [lindex $argList $i]
        if {[cindex $key 0] != "-"} {
            error "key '$key' of key-value pairs doesn't start with '-'"
        }
        set array([crange $key 1 end]) [lindex $argList [expr $i + 1]]
    }
}
#@package: libmanager require

#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

#
# This little thing will force the autoloading of the specified command,
# without actually executing it.
#
proc require {commandName} {

    # already loaded?
    if {[info commands $commandName] != ""} return

    # get it
    if ![auto_load $commandName] {
	error "couldn't auto_load $commandName"
    }
}
#@package: neo_misc_procs force_width dollar_format

#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

#
# return a string forceed to a specified width either by padding, if it
# is too short, or by truncation, if it is too long.
#
proc force_width {string width} {
    return [crange [format "%-${width}s" $string] 0 $width-1]
}

#
# format a number and return it as a dollar field (two digits precision
# to the right of the decimal place.)
#
proc dollar_format {number} {
    return [format "%.2f" $number]
}

#@package: prompts prompt prompt_yn prompt_choices

#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

#
# prompt - given some prompt text, the name of a variable, and a possible
# default value, emit the text, read a line into the named variable and,
# if an empty string is entered, use the default instead, and set the
# result into the named variable.
#
proc prompt {promptText varName {default ""}} {
    upvar $varName line
    set line ""

    if {$default == ""} {
        puts stdout "$promptText: " nonewline
    } else {
        puts stdout "$promptText \[$default\]: " nonewline
    }
    if {[gets stdin line] < 0} return
    if {$line == ""} {
        set line $default
    }
}

#
# prompt_yn - emit some prompt text and get a yes or no response,
# returning 1 for yes and 0 for no.
#
proc prompt_yn {promptText {default ""}} {
    while 1 {
        if {$default == ""} {
            puts stdout "$promptText (y/n): " nonewline
        } else {
            puts stdout "$promptText (y/n) \[$default\]: " nonewline
        }
        if {[gets stdin line] < 0} return
        if {$line == ""} {set line $default}
        set char [string tolower [cindex $line 0]]
        if {$char == "n"} {return 0}
        if {$char == "y"} {return 1}
        echo "Please answer y)es or n)o"
    }
}

#
# prompt_choices - emit some prompt text and get a selection of one
# of a number of responses.
#
proc prompt_choices {promptText choiceList {default ""}} {
    while 1 {
	set nChoices 0
	echo $promptText
	foreach element $choiceList {
	    echo "  $nChoices $element"
	    incr nChoices
	}
	prompt "Your selection?" input $default
	if {([catch {set choice [lindex $choiceList $input]} result] == 1) || \
	    ($choice == "" && $default != "")} {
	    echo "Please enter a number between 1 and $nChoices."
	    continue
	}
	if {$choice == ""} {return $default}
	return $choice
    }
}


#@package: rfc822 parse_rfc822_line read_rfc822_group read_rfc822 write_rfc822_group write_rfc822 parse_rfc822_mail_address

#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

#
# we really need to rewrite this in C and be truly RFC-822 compliant,
# it's a pig.
#
# RFC noncompliance issues: Line order isn't maintained, multiple
# lines with the same key aren't supported.
# There may be stuff in there about quoting, too.
#

#
# parse_rfc822_line
#
# Given a line of RFC822 header text and the names of two variables,
# parse the key into one and the value into the other.
#
# Warning, can't do multiple line messages unless line argument contains
# all necessary lines.
#
#
proc parse_rfc822_line {line keyVar valueVar} {
    upvar $keyVar key
    upvar $valueVar value

    set key ""
    set colon [string first ":" $line]
    if {$colon < 2} {
	return 0
    }
    set key [string trim [string range $line 0 [expr $colon-1]]]
    set value [string trim [string range $line [expr $colon+1] end]]
    return 1
}

#
# read_rfc822_group
#
# given a filehandle and an array name, read the file, pulling key-value
# pairs and putting them into the named array, returning when an empty
# line or EOF is found
#
#
proc read_rfc822_group {fp arrayVarName} {
    upvar $arrayVarName rfcArray
    set success 0
    
    while {[gets $fp line] >= 0} {
        if {$line == ""} break

	set firstchar [cindex $line 0]
	if {$firstchar == " " || $firstchar == "\t"} {
	    if {$key != ""} {
	        append rfcArray($key) "\n$line"
	    } else {
		puts stderr "rfc-822 line starts with whitespace with no prior key '$line'"
	    }
	    continue
	}
        set key ""
        set colon [string first ":" $line]
        if {$colon < 2} {
            puts stderr "malformed rfc-822 line '$line'"
        }
        set success 1
        set key [string trim [string range $line 0 [expr $colon-1]]]
        set value [string trim [string range $line [expr $colon+1] end]]
        set rfcArray($key) $value
    }
    return $success
}

#
# read_rfc822
#
# load in a file containing one rfc822 group
#
proc read_rfc822 {fileName arrayVarName} {
    upvar $arrayVarName rfcArray
    
    set fp [open $fileName]
    read_rfc822_group $fp rfcArray
    close $fp
}

#
# write_rfc822_group
#
# given a filehandle and an array name, write the file, pulling key-value
# pairs from the named array, returning when finished.
#
#
proc write_rfc822_group {fp arrayVarName} {
    upvar $arrayVarName rfcArray 

    foreach name [array names rfcArray] {
        set data [split $rfcArray($name) "\n"]
        puts $fp "$name: [lindex $data 0]"
        foreach additionalLine [lrange $data 1 end] {
            puts $fp "\t$additionalLine"
        }
    }
    puts $fp ""
}

#
# write_rfc822
#
# write out a file containing one rfc822 group
#
proc write_rfc822 {fileName arrayVarName} {
    upvar $arrayVarName rfcArray
    
    set fp [open $fileName w]
    write_rfc822_group $fp rfcArray
    close $fp
}

#
# parse_rfc822_mail_address
#
# take the body of an address line and convert it into a name and address
#
proc parse_rfc822_mail_address {line emailAddressVar fullNameVar} {
    upvar $emailAddressVar emailAddress
    upvar $fullNameVar fullName
 
    # in the form "Ellyn Jones <ellyn@NeoSoft.com>"?
    if {[regexp {(.*) <(.*)>} $line dummy fullName emailAddress]} {
        set emailAddress [string tolower $emailAddress]
        return
    }   
    
    # in the form "ellyn@NeoSoft.com (Ellyn Jones)"?
    if {[regexp {(.*) \((.*)\)} $line dummy emailAddress fullName]} {
        set emailAddress [string tolower $emailAddress]
        return
    }
 
    # hmm, take whatever's there as the reply address 
    set emailAddress [string tolower $line]
    set fullName $emailAddress
}

#@package: rfc931 rfc931

#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

#
# Perform an rfc931 authentication request on the socket bound to the
# specified filehandle.  Returns username and hostname if found,
# username and ip if no hostname is found, or an empty string if
# the remote site isn't running an rfc931 authentication server.
#
# $Id: rfc931.tcl,v 1.2 1996/09/28 20:54:27 kunkee Exp $
#

proc rfc931 {fp resultVar} {
    upvar $resultVar result

    lassign [fstat $fp localhost] localAddr localPort
    lassign [fstat $fp remotehost] remoteAddr remotePort

    if {[catch {lassign [server_open -myip $localAddr $remoteAddr ident] readIdent writeIdent} result] == 1} {
	global errorCode errorInfo
	if {[lindex $errorCode 1] == "ECONNREFUSED"} {return 0}
	error $result $errorInfo $errorCode
    }
    set remoteName [server_info address_name $remoteAddr]

    puts $writeIdent "$remotePort,$localPort\r"
    close $writeIdent

    if {[gets $readIdent authLine] < 0} {
	close $readIdent
	set result "read failed"
	return 0
    }

    set result $authLine
    if {[scan $authLine {%u , %u : USERID :%*[^:]:%s} remote local user] != 3} {
	close $readIdent
	return 0
    }

    if {$remote != $remotePort || $local != $localPort} {
	close $readIdent
	set result "ports didn't match"
	return 0
    }

    if {[cindex $user end] == "\r"} {
	set user [crange $user 0 end-1]
    }
    set result $user@$remoteName
    close $readIdent
    return 1
}
#@package: spawn_child spawn_child

#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

#
# spawn_child - fork a tcl subprocess, redirecting standard input,
# output and error to pipes.
#
# Three variable names are specified.
# Within those variables spawn_child writes the filehandles of pipes that
# correspond to standard input, standard output and standard error of
# the subprocess.
#

proc spawn_child {command stdinPipeVarName stdoutPipeVarName stderrPipeVarName} {

    upvar $stdinPipeVarName stdinPipe \
	  $stdoutPipeVarName stdoutPipe \
	  $stderrPipeVarName stderrPipe

    pipe childStdinPipe stdinPipe
    pipe stdoutPipe childStdoutPipe
    pipe stderrPipe childStderrPipe

    if {[set childPid [fork]] == 0} {

        dup $childStdinPipe stdin
        close $childStdinPipe

        dup $childStdoutPipe stdout
        close $childStdoutPipe

        dup $childStderrPipe stderr
        close $childStderrPipe

        eval "execl $command"
    }
    return $childPid
}


#@package: stanza read_stanza_header read_stanza_body read_stanza_file load_stanza_file write_stanza save_stanza_file

#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

# $Id: stanza.tcl,v 1.2 1996/09/28 20:54:27 kunkee Exp $

#
# we really need to rewrite this in C and be truly stanza compliant,
# it's kind of a pig.
#
# stanza noncompliance issues: lots
#
# seems to work, though.
#


# read_stanza_header
#
# given a filehandle and an array name, read the file, pulling key-value
# pairs and putting them into the named array, returning when an empty
# line or EOF is found
#
#
proc read_stanza_header {fp headerVarName} {
    upvar $headerVarName headerVar
    
    if {[gets $fp line] <= 0} {return 0}
    if {[string index $line 0] != "\["} {
        error "malformed stanza file - no '[' in header"
    }
    if {[cindex $line [clength $line]-1] != "]"} {
        error "malformed stanza file - no ']' in header"
    }
    set headerVar [crange $line 1 [clength $line]-2]
    return 1
}

# parse_stanza_line
#
# given a line and the names of key and value variables, put
# the key and value in variables and return 1, else 0 or error
#
proc parse_stanza_line {line keyVarName dataVarName} {
    upvar $keyVarName key
    upvar $dataVarName value
    
    if {[string trim $line] == ""} {return 0}
    if {[cindex $line 0] == "\["} {
        error "called parse_stanza_line with a header line '$line'"
    }

    set separator [string first "=" $line]
    if {$separator < 2} {
        error "malformed stanza line '$line'"
    }
    set key [string trim [string range $line 0 [expr $separator-1]]]
    set value [string trim [string range $line [expr $separator+1] end]]
    return 1
}

# read_stanza_body
#
# given a filehandle and an array name, read the file, pulling key-value
# pairs and putting them into the named array, returning when an empty
# line or EOF is found
#
#
proc read_stanza_body {fp arrayVarName} {
    upvar $arrayVarName stanzaArray
    set success 0
    
    while {[gets $fp line] >= 0} {
        if ![parse_stanza_line $line key value] break
        set success 1
        set stanzaArray($key) $value
    }
    return $success
}

#
# write_stanza
#
# given a filehandle, an array name containing stanza elements
# and a stanza header name, write it out
#
#
proc write_stanza {fp headerName arrayVarName} {
    upvar $arrayVarName stanzaArray
    
    puts $fp "\[$headerName\]"
    foreach name [array names stanzaArray] {
        puts $fp "$name=$rfcArray($name)"
    }
    puts $fp ""
}

proc store_stanza_field {varName upLevel key value} {
    upvar $upLevel $varName myArray

    set myArray($key) $value
}

proc read_stanza_file {fileName arrayVarPrefix {global ""}} {
    set upLevel "2"
    if {$global != ""} {set upLevel "#0"}
    set fp [open $fileName]
    while 1 {
        if {[read_stanza_header $fp section] == 0} {
            close $fp
            return
        }
        store_stanza_field $arrayVarPrefix$section $upLevel ID $section
        while {[gets $fp line] >= 0} {
            if ![parse_stanza_line $line key value] break
            store_stanza_field $arrayVarPrefix$section $upLevel $key $value
        }
    }
}


#
# load_stanza_file and save_stanza_file work differently from the
# above.  these guys read the stanza bodies into keyed lists indexed
# by the header names through an array
#
#

#
# load_stanza_file
#
# load in a stanza file where the stanza headers becomne array names
# and the bodies become keyed lists
#
# slight optimization hack was to append up the list instead of
# keylsetting it, don't know if that really helped or not
#
proc load_stanza_file {fileName arrayVarName} {
    upvar $arrayVarName stanzaArray
    
    set fp [open $fileName]
    while 1 {
        if {[read_stanza_header $fp section] == 0} {
            close $fp
            return
        }
        set keyedList ""
        while {[gets $fp line] >= 0} {
            if ![parse_stanza_line $line key value] break
            lappend keyedList "$key $value"
        }
        set stanzaArray($section) $keyedList
    }
}

#
# save_stanza_file
#
# save a stanza file where the stanza headers come from an array's keys
# and the bodies are from keyed lists indexed by the keys
#
proc save_stanza_file {fileName arrayVarPrefix {global ""}} {
    set upVar "1"
    if {$global != ""} {set upVar "#0"}
    set varPrefixLength [string length $arrayVarPrefix]

    set fp [open $fileName w]
    foreach arrayVarName [uplevel $upVar "info vars $arrayVarPrefix*"] {
        upvar $upVar $arrayVarName stanzaArray

        set setName [string range $arrayVarName $varPrefixLength end]
    
	puts $fp "\[$setName\]"
        foreach key [array names stanzaArray] {
            if {[cindex $key 0] != "_"} {
	        puts $fp "$key=$stanzaArray($key)"
	    }
	}
        puts $fp ""
    }
    close $fp
}

#@package: timedate-utility next_month_year days_in_month calculate_ratio_of_month_left day_month_calculations

#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

#
# Time/Date utility functions
#
# $Id: timedate.tcl,v 1.2 1996/09/28 20:54:27 kunkee Exp $
#
#


#
# given the current month and year and the names of two variables,
# put in the first the numeric month following this one and in the
# second, the year of that month.
#
proc next_month_year {currentMonth currentYear nextMonthVar nextMonthYearVar} {
    upvar $nextMonthVar nextMonth
    upvar $nextMonthYearVar nextMonthYear

    set nextMonthYear $currentYear
    set nextMonth $currentMonth

    if {$nextMonth < 12} {
	incr nextMonth
    } else {
	set nextMonth 1
	incr nextMonthYear
    }
}

#
# How many days are in the specified month, in the specified year?
#
proc days_in_month {month year} {
    switch $month {
	1 {return 31}
	2 {
	    if {$year % 4 == 0} {
		return 29
	    } else {
		return 28
	    }
	}
	3 {return 31}
	4 {return 30}
	5 {return 31}
	6 {return 30}
	7 {return 31}
	8 {return 31}
	9 {return 30}
       10 {return 31}
       11 {return 30}
       12 {return 31}
    }
    error "month out of range ($month)"
}

#
# What is the floating point ratio of days left from the specified
# day, in the specified month, of the specified year?
#
proc calculate_ratio_of_month_left {month day year} {
    set daysInThisMonth [days_in_month $month $year]

    set daysLeftInThisMonth [expr $daysInThisMonth - $day + 1]
    return [expr $daysLeftInThisMonth.0/$daysInThisMonth]
}

#
# Given a integer-seconds-since-1970 and the names of some variables,
# return in those variables the month number (1-12), the year number,
# the days in the current month (28-31), the days left in this month,
# the ratio of days left in the month, the month number (1-12) of
# the month following this date's month, and the year number of the month
# following this date's month.
#
proc day_month_calculations {clock thisMonthVar thisYearVar daysInThisMonthVar 
                             daysLeftInThisMonthVar ratioLeftVar 
			     nextMonthMonthVar nextMonthYearVar} \
{
    upvar $thisMonthVar thismonth
    upvar $thisYearVar thisyear
    upvar $daysInThisMonthVar daysInThisMonth
    upvar $daysLeftInThisMonthVar daysLeftInThisMonth
    upvar $ratioLeftVar ratioLeft
    upvar $nextMonthMonthVar nextmonth
    upvar $nextMonthYearVar nextyear

    lassign [fmtclock $clock "%m %d %Y"] thismonth thisday thisyear
    scan $thismonth %d thismonth
    scan $thisday %d thisday

    next_month_year $thismonth $thisyear nextmonth nextyear
    set monthStartSeconds [convertclock $nextmonth/1/$nextyear]

    set daysInThisMonth [days_in_month $thismonth $thisyear]

    set daysLeftInThisMonth [expr $daysInThisMonth - $thisday + 1]
    set ratioLeft [expr $daysLeftInThisMonth.0/$daysInThisMonth]
}

#
#
# Copyright (C) 1992-1996 NeoSoft.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted, provided
# that the above copyright notice appear in all copies.  NeoSoft makes no 
# representations about the suitability of this software for any purpose.
# It is provided "as is" without express or implied warranty.

#
# $Id: xkeylget.tcl,v 1.2 1996/09/28 20:54:28 kunkee Exp $
#

#@package: xkeylget xkeylget


proc xkeylget {keylistName args} {
    upvar $keylistName keylist
    if {[llength $args] % 2 != 0} {error "odd number of key-varname pairs"}
    while {$args != ""} {
        set elementName [lindex $args 0]
        set varName [lindex $args 1]
        set args [lrange $args 2 end]
        keylget keylist $elementName value
        uplevel set $varName [list $value]
    }
}
