#
# NeoScript - Server Side Programming based on Safe Tcl
#
# Copyright (C) NeoSoft, All Rights Reserved.  See NeoWebScript LICENSE
# files with this distribution for conditions on use and redistribution.
#
# $Id: common.tcl,v 1.16 1997/07/29 16:41:46 karl Exp $
#

#
# PROPERTY LIST STORAGE AND RETRIEVAL
#
set parallelBase "$server(SERVER_ROOT)/neoscript-data/users"

proc setup_data_access {} {
    global parallelDir webpageOwnerName webenv

    if [info exists parallelDir] return

    global webenv parallelBase

    set webpageOwnerName [id convert userid $webenv(NEO_DOCUMENT_UID)]
    set parallelDir $parallelBase/[cindex $webpageOwnerName 0]/$webpageOwnerName/
}

proc db_name_check {dbname dbtype fileNameVar} {
    if ![regexp {^[a-zA-Z0-9_]*$} $dbname] {
	error "illegal database name: $dbname, upper/lowercase and 0-9 only"
    }
    upvar $fileNameVar dbFileName
    global parallelDir

    set dbFileName ${parallelDir}$dbtype/$dbname.$dbtype
}

proc ul_name_check {dbname dbtype fileNameVar} {
    if ![regexp {^[-a-zA-Z0-9_=+.]*$} $dbname] {
	error "illegal database name: $dbname, lowercase, uppercase, 0-9, _, -, +, =, and .  only"
    }
    upvar $fileNameVar dbFileName
    global parallelDir

    set dbFileName ${parallelDir}$dbtype/$dbname.$dbtype
}

proc create_user_dir {plusSubDir} {
    global parallelDir errorCode

    setup_data_access

    if {[catch {mkdir -path $parallelDir$plusSubDir} result] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX EEXIST"} {
	    error "$errorCode"
	}
    }
}

proc dbdelkey {database id} {
    global errorCode

    setup_data_access

    db_name_check $database db dbFileName

    if {[catch {set db [db open $dbFileName hash rw 0664]}] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    error "$dbFileName: $errorCode"
	} else {
	    return 0
	}
    }
    set result [db del $db $id]
    db close $db
    return $result
}

proc dbfetch {database id arrayName {singleVar ""}} {
    global errorCode

    setup_data_access
    db_name_check $database db dbFileName

    if {[catch {db open $dbFileName hash rwl 0664} db] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX EEXIST"} {
	   error $errorCode
	}
	return 0
    }

    db get $db $id list
    db close $db

    set haveData [expr ![lempty $list]]

	if {$singleVar == "-singleVar"} {
		set $arrayName $list
	} else {
		upvar $arrayName array
		array set array $list
	}

    return $haveData
}

proc dbstore {database id arrayName {singleVar ""}} {
    global errorCode

    setup_data_access
    db_name_check $database db dbFileName

	if {$singleVar == "-singleVar"} {
		set list $arrayName
	} else {
		upvar $arrayName array
		set list [array get array]
	}

    if {[catch {set db [db open $dbFileName hash rwL 0664]}] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    error "$dbFileName: $errorCode"
	} else {
        create_user_dir db
	    set db [db open $dbFileName hash ctL 0664]
	}
    }
    db put $db $id $list
    db close $db
    return
}

proc dbexists {database} {
    setup_data_access
    db_name_check $database db dbFileName

    return [file readable $dbFileName]
}

proc filerm {type database} {
    setup_data_access
    db_name_check $database $type dbFileName

    unlink -nocomplain $dbFileName
}

proc filemv {type old_database new_database} {
    setup_data_access
    db_name_check $old_database $type old_dbFileName
    db_name_check $new_database $type new_dbFileName

    if ![file exists $old_dbFileName] {
	error "attempt to move non-existent $type file $old_database"
    }
    if [file exists $new_dbFileName] {
	unlink $new_dbFileName
    }
    link $old_dbFileName $new_dbFileName
    unlink $old_dbFileName
}

proc dbkeys {database {pattern *}} {
    global errorCode

    setup_data_access

    db_name_check $database db dbFileName

    if {[catch {set db [db open $dbFileName hash rw 0664]}] == 1} {
	if {[lrange $errorCode 0 1] != "POSIX ENOENT"} {
	    error "$dbFileName: $errorCode"
	} else {
	    return ""
	}
    }
    set keys ""
    db searchall $db key -glob $pattern {
	lappend keys $key
    }
    db close $db

    return $keys
}

proc dbdump {database {pattern *}} {
    setup_data_access

    db_name_check $database db dbFileName

    set db [db open $dbFileName hash rw 0664]
    set keys ""
    db searchall $db key -glob $pattern {
	html "\[$key\]\n"
	db get $db $key list
	foreach pair $list {
	    html "[lindex $pair 0]=[lindex $pair 1]\n"
	}
	html "\n"
    }
    db close $db

    return
}

proc dbfiles {{pattern *}} {
    global parallelDir errorCode
    setup_data_access

    set result {}
    foreach file [glob -nocomplain $parallelDir/db/$pattern.db] {
	lappend result [file root [file tail $file]]
    }
    return $result
}

# proc to import key value pairs of the form "-color blue"
# into an array in the caller's context.
#
# and dialog box thingie that uses it
#
# i am really missing incr tcl
#
#
#neo_dialog -text "This is an alert mesesage... or something."
#
#neo_dialog -text "This is another alert message."
#

proc import_keyvalue_pairs {arrayName string} {
    upvar $arrayName array

    set len [llength $string]
    if {$len % 2 != 0} {
        error "unmatched key-value pair"
    }

    for {set i 0} {$i < $len} {incr i 2} {
        set key [lindex $string $i]
        if {[string index $key 0] != "-"} {
            error "key $key of key-value pairs doesn't start with a dash"
        }
        set array([string range $key 1 end]) [lindex $string [expr $i + 1]]
    }
}

#
# List the data files in the server-maintained data directory
# for the user.
#
#
proc list_data_files {{pattern *}} {
    global parallelDir errorCode
    setup_data_access

    set result {}
    foreach file [glob -nocomplain $parallelDir/data/$pattern.data] {
	lappend result [file root [file tail $file]]
    }
    return $result
}

#
#Return 1 if a datafile exists in the server-maintained data directory.
#
#
proc data_file_exists {dataFile} {
	global parallelDir errorCode
	setup_data_access

	return [file exists $parallelDir/data/$dataFile.data]
}

# Authorization routine for PostGres95.  Maintains a db cache locally
# which expires in 15 minutes.
# Use the following in .htaccess to reference this authentication
# routine:
#
# TclAuthBasic dbname table userCol passwordCol
#
proc postgres_auth {database table userCol passwordCol username} {
	global server
	set password *
	set dbFileName $server(SERVER_ROOT)/neoscript-data/system/pg95cache.db
	if {[catch {set db [db open $dbFileName hash rwL 0644]}] == 1} {
	    set db [db open $dbFileName hash ctL 0664]
	}
	if [db get $db $database/$table/$username list] {
		lassign $list password expire
		if {[clock seconds] < $expire} {
			db close $db
			return $password
		}
	}
	# Postgres is flakey.  If it is not there, then return cached
	# password.
	if [catch {set conn [pg_connect $database]}] {return $password}
	set result [pg_exec $conn \
		"select $passwordCol from $table where $userCol = '$username'"]
	if {[pg_result $result -numTuples] == 1} {
		set password [lindex [pg_result $result -getTuple 0] 0]
		db put $db $database/$table/$username [list $password [expr [clock seconds]+900]]
	}
	pg_result $result -clear
	pg_disconnect $conn
	db close $db
	return $password
}

proc postgres_access {user type} {
	if {$type == "valid-user"} { return OK }
	return AUTH_REQUIRED
}

# Authorization routine for DB Auth.
proc tcl_db_auth {dbOwner dbName username} {
	global server
	upvar _${dbOwner}_${dbName}_auth_cache authCache
	if [info exists authCache($username)] {
		lassign $authCache($username) password expire
		if {$expire > [clock seconds]} {return $password}
	}
	set password *
	set dbFileName $server(SERVER_ROOT)/neoscript-data/users/[cindex $dbOwner 0]/$dbOwner/db/$dbName.db
	if {[catch {set db [db open $dbFileName hash r]}] == 1} {
		return
	}
	set found [db get $db $username password]
	db close $db
	if $found {
		set authCache($username) "$password [expr [clock seconds] + 60]"
		return $password
	}
	return
}

proc tcl_db_access {user type} {
	if {$type == "valid-user"} { return OK }
	return AUTH_REQUIRED
}

# Authorization routine for /etc/passwd access.  Requires the 'getpass'
# program to be built.

proc tcl_passwd_auth {username} {
	global server
	if [catch {exec $server(SERVER_ROOT)/bin/getpass $username} password] {
		return
	}
	return $password
}

proc tcl_passwd_access {user type} {
	if {$type == "valid-user"} { return OK }
	return AUTH_REQUIRED
}

# Routines for the test authentication based on Tcl
proc test_tcl_auth {args} {
	return [neo_crypt test ab]
}

proc test_tcl_access {args} {
	return OK
}
