#! /usr/local/bin/scotty -inf
##
## IP Layout Tool for [TK]INED.
##
## This file contains code to query a nameserver and to automatically
## layout an IP network. This is not finished yet. This tool also 
## provides lots of utilities to set the name, address, icon ... 
## of a network node.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

set gridx 65
set gridy 50
set columns 8

set icon_table(SUN.*4/370) SUN-Server
set icon_table(SUN.*10)   SparcStation
set icon_table(SUN.*SS2)  SparcStation
set icon_table(SUN.*4/75) SparcStation
set icon_table(SUN.*IPX)  IPC
set icon_table(SUN.*IPC)  IPC
set icon_table(SUN.*4/65) IPC
set icon_table(SUN.*4/60) IPC
set icon_table(SUN.*4/45) IPC
set icon_table(SUN.*4/40) IPC
set icon_table(SUN.*ELC)  SLC
set icon_table(SUN.*SLC)  SLC
set icon_table(SUN.*4/25) SLC
set icon_table(SUN.*4/20) SLC
set icon_table(SUN.*3/50) SUN3
set icon_table(SUN.*3/60) SUN3
set icon_table(SUN.*3/80) SUN3
set icon_table(PYRAMID.*90) locker
set icon_table(X.*Terminal) Xstation
set icon_table(PC.*Kompatibel) pc
set icon_table(PC.*AT) pc
set icon_table(PC.*\[34\]86) pc
set icon_table(i\[34\]86) unixpc
set icon_table(hp.*) HP
set icon_table(cisco.*) Router

##
## Search for tkined.defaults files following the auto_path and add 
## icon mappings to the allow easy user customization.
##

proc load_defaults {name} {

    global auto_path
    global icon_table

    set reverse_path ""
    foreach dir $auto_path {
	set reverse_path "$dir $reverse_path"
    }

    foreach dir $reverse_path {
        if [file readable $dir/$name] {
	    set fh [open $dir/$name r]
	    while {![eof $fh]} {
		gets $fh line
		set line [string trim $line]
		if {($line=="") || ([regexp "^#|^!" $line])} continue
		if {[string match layout.* $line]} {
		    set line [split $line ":"]
		    set attname [lindex [split [lindex $line 0] "."] 1]
		    set attval  [string trim [lindex $line 1]]
		    if {$attname == "icon"} {
			set attval [split $attval]
			if {[llength $attval] == 2} {
			    regsub "\.bm$" [lindex $attval 1] "" bm
			    set icon_table([lindex $attval 0]) $bm
			}
		    }
		}
	    }
	    close $fh
	}
    }
    return ""
}

load_defaults tkined.defaults

## Get the IP Address of a node. Query the name server, if the
## address attribute is not set to something that looks like a
## valid IP address.

proc Get_IP_Address {node} {
    if {[ined_type $node]=="NODE"} {
	set host [ined_name $node]
	set ip [ined_address $node]
	if {[regexp "^\[0-9\]+\.\[0-9\]+\.\[0-9\]+\.\[0-9\]+$" $ip]>0} {
	    return $ip
	}
	if {[catch {nslook $host} ip]==0} {
	    return [lindex $ip 0]
	}
    }
    return ""
}

##
## Set the name of all node objects.
##

proc "set name" {list} {
    foreach comp $list {
	if {[ined_type $comp]=="NODE"} {
            set id [ined_id $comp]
            set host [ined_name $comp]
            set ip [ined_address $comp]
            if {$ip==""} {
                ined acknowledge "No IP Address set for $host."
                continue;
            }
	    if {[catch {dns ptr $ip} host]} {
		if {[catch {nslook $ip} host]} {
		    ined acknowledge "Can not lookup name for \[$ip\]."
		    continue
		}
	    }
	    ined -noupdate name $id [string tolower [lindex $host 0]]
	    ined label $id name
        }
    }
}

##
## Set the address of all node objects.
##

proc "set address" {list} {
    foreach comp $list {
	if {[ined_type $comp]=="NODE"} {
	    set id [ined_id $comp]
            set host [ined_name $comp]
	    if {$host==""} {
                ined acknowledge "No name set for $host."
                continue;
            }
	    if {[catch {nslook $host} ip]==0} {
		ined -noupdate address $id [lindex $ip 0]
		ined label $id address
	    } else {
		ined acknowledge "Can not lookup address for \[$host\]."
	    }
	}
    }
}

##
## Try to set the icon of all node objects. This procedure needs
## more support from scotty to query the name servers directly.
##

proc "set icon" {list} {

    global icon_table

    set failures ""

    foreach comp $list {
	if {[ined_type $comp] != "NODE"} continue
	set id [ined_id $comp]
	set host [lindex [ined_name $comp] 0]
	set ip [Get_IP_Address $comp]
	if {[catch {dns ptr $ip} name]} continue
	if {$name==""} { set name $host }
	if {[catch {dns hinfo $name} hinfo]} continue
	set cpu [lindex $hinfo 0]
	set icon ""
	foreach regex [array names icon_table] {
	    if {[regexp -nocase [string tolower $regex] $cpu]} {
		set icon $icon_table($regex)
		break
	    }
	}
	if {$icon!=""} {
	    ined icon $id $icon
	} else {
	    lappend failures "$host ($cpu)"
	}
    }

    if {$failures!=""} {
	ined browse "No vaild icon mapping for the following hosts:" $failures
    }
}

##
## For all nodes try to reduce the domain name to the unique prefix.
##

proc "unique name" {list} {
    set idlist ""
    foreach comp $list {
	set type [ined_type $comp]
	if {$type!="NODE"} continue
	append idlist " [ined_id $comp]"
    }
    if {[llength $idlist]==0} return
    foreach nodeid $idlist {
	set name($nodeid) ""
	foreach p [split [lindex [ined name $nodeid] 0] "."] {
	    set name($nodeid) "$p $name($nodeid)"
	}
    }
    set unique 1
    while {$unique} {
	if {[llength $name($nodeid)]<=1} break
	set pfx [lindex $name($nodeid) 0]
	foreach id [array names name] {
	    if {$pfx != [lindex $name($id) 0]} {
		set unique 0
		break
	    }
	}
	if {$unique} {
	    foreach id [array names name] {
		set name($id) [lrange $name($id) 1 end]
	    }
	} 
    }
    foreach id [array names name] {
	set new ""
	foreach p $name($id) { set new "$p.$new" }
	set new [string trimright $new "."]
	ined name $id $new
    }
}

##
## Make groups. Get all nodes connected to a network and throw
## them into a group named like the network. Exclude nodes with
## more than one link.
##

proc "group network" {list} {
    foreach comp $list {
	set id [ined_id $comp]
	set type [ined_type $comp]
	if {$type != "NETWORK"} continue
	set member $id
	foreach n [neighbours $id] {
	    if {[llength [ined_links [ined retrieve $n]]] == 1} {
		lappend member $n
	    }
	}
	set group [eval ined -noupdate create GROUP $member]
	ined -noupdate name $group [ined_name $comp]
	ined -noupdate icon $group Bus
	ined -noupdate label $group name
	ined -noupdate color $group [ined color $id]
	ined -noupdate font  $group [ined font  $id]
    }
}

##
## Get all neighbours of a NETWORK or a NODE object that are
## connected using a direct link. Return the list of ids.
##

proc neighbours {id} {
    set object [ined retrieve $id]
    set type [ined_type $object]
    if {($type!="NETWORK") && ($type!="NODE")} { 
	error "illegal object type"
    }
    set links [ined_links $object]
    set result ""
    foreach link $links {
	set comp [ined retrieve $link]
	set ida [ined_ida $comp]
	set idb [ined_idb $comp]
	if {$ida==$id} {
	    append result " $idb"
	} else {
	    append result " $ida"
	}
    }
    return $result
}

##
## Move the object given by id to a new absolute position.
##

proc position {id abs_x abs_y} {
    set xy [ined -noupdate move $id]
    ined -noupdate move $id \
	[expr {$abs_x-[lindex $xy 0]}] [expr {$abs_y-[lindex $xy 1]}]
}

##
## move all objects to a grid centered around net_x and net_y
## with a grid spacing set to grid_x and grid_y. Put columns
## objects in one row.
##

proc layout_grid {objects net_x net_y grid_x grid_y columns} {
    set y1 $net_y
    set y2 $net_y
    set i 0
    set x $net_x
    foreach node $objects {
	if {! ($i%(2*$columns))} {
	    set net_x [expr {$net_x+2}]
	    set y1 [expr {$y1+$grid_y}]
	    set y2 [expr {$y2-$grid_y}]
	    set x1 [expr {$net_x+($grid_x/2)}]
	    set x2 [expr {$net_x-($grid_x/2)}]
	    set x $x1
	    set y $y1
	}
	position $node $x $y
	if {$y==$y1} {
	    set y $y2
	} else {
	    set y $y1
	    if {$x==$x1} {
		set x $x2
	    } else {
		set x1 [expr {$x1+$grid_x}]
		set x2 [expr {$x2-$grid_x}]
		set x $x1
	    }
	}
	incr i
    }
}

##
## Move all nodes connected to network in the area of the
## network. We simply put them in a grid around the network
## position.
##

proc "layout network" {list} {
    global gridx gridy columns
    foreach comp $list {
	if {[ined_type $comp]=="NETWORK"} {
	    set id [ined_id $comp]
	    ined icon $id
	    set size [ined -noupdate size $id]
	    set net_x [expr {([lindex $size 0]+[lindex $size 2])/2}]
	    set net_y [expr {([lindex $size 1]+[lindex $size 3])/2}]
	    set nodes [neighbours $id]
	    set single ""
	    set gates  ""
	    foreach node $nodes {
		set links [ined_links [ined retrieve $node]]
		if {[llength $links]==1} {
		    lappend single $node
		} else {
		    lappend gates $node
		}
	    }
	    set nodes [join [list $single $gates]]
	    layout_grid $nodes $net_x $net_y $gridx $gridy $columns
	}
    }
}

##
## This simple dialog allows us to modify the layout
## parameters on the fly.
##

proc "set parameter" {list} {
    global gridx gridy columns

    set result [ined request "Layout Parameter" \
	" { {Horizontal grid space:} $gridx scale 20 160} \
	  { {Vertical grid space:}   $gridy scale 20 160} \
	  { {Nodes per row:}       $columns scale 10 40} "]

    if {$result==""} return

    set gridx   [lindex $result 0]
    set gridy   [lindex $result 1]
    set columns [lindex $result 2]
}

##
## Display some help about this tool.
##

proc "help IP Layout" {list} {
    ined browse "Help about IP Layout" {
	"set name:" 
	"    Query the Domain Name Service and set the name of all " 
	"    selected nodes." 
	"" 
	"set address:" 
	"    Query the Domain Name Service and set the address of all " 
	"    selected nodes." 
	"" 
	"set icon:" 
	"    Query the Domain Name Service and set the icon of all " 
	"    selected nodes based on the HINFO record." 
	"" 
	"unique name:" 
	"    Shorten the names of all selected objects by removing " 
	"    common domain name endings."  
	"" 
	"group networks:" 
	"    Throw all objects connected to network into a group." 
	"" 
	"layout network:" 
	"    Position all nodes connected to a network around it." 
	"" 
	"set parameter:" 
	"    Set the parameter that control the layout algorithm." 
    } 
}

##
## Delete the tools created by this interpreter.
##

proc "delete IP Layout" {list} {
    global tools
    foreach id $tools {	ined delete $id }
    exit
}

set tools [ ined create TOOL "IP Layout" \
	    "set name" "set address" "set icon" "" \
	    "unique name" "group network" "" \
	    "layout network" "" \
	    "set parameter" "" \
	    "help IP Layout" "delete IP Layout" ]
