#! /usr/local/bin/scotty -inf
##
## Simple Troubleshooting interpreter for [TK]INED.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

##
## These are the global parameters that control the discovering
## process. They are manipulated by the "Discover Parameter" proc.
##

set icmp_retries 3
set icmp_timeout 3
set icmp_delay 5
set max_route_len 16

##
## The the defaults for the icmp command (see scotty(1)).
##

icmp -retries $icmp_retries
icmp -timeout $icmp_timeout
icmp -delay $icmp_delay

##
## Write a report to a log object.
##

proc writeln {{txt ""}} {
    static log

    if {![info exists log]} {
	set log [ined -noupdate create LOG]
	ined -noupdate name $log "IP Trouble Log"
    }

    if {[ined -noupdate retrieve $log] == ""} {
	set log [ined -noupdate create LOG]
	ined -noupdate name $log "IP Trouble Log"
    }

    ined append $log $txt
}

## Get the IP Address of a node. Query the name server, if the
## address attribute is not set to something that looks like a
## valid IP address.

proc Get_IP_Address {node} {
    if {[ined_type $node]=="NODE"} {
	set host [ined_name $node]
	set ip [ined_address $node]
	if {[regexp "^\[0-9\]+\.\[0-9\]+\.\[0-9\]+\.\[0-9\]+$" $ip]>0} {
	    return $ip
	}
	if {[catch {nslook $host} ip]==0} {
	    return [lindex $ip 0]
	}
    }
    return ""
}

proc netping {network} {
    if {[regexp "^\[0-9\]+\.\[0-9\]+\.\[0-9\]+$" $network]>0} {
	set hosts ""
	for {set a4 1} {$a4<255} {incr a4} {
	    set hosts "$hosts $network.$a4"
	}
	return [icmp -timeout 15 $hosts]
    }
    if {[regexp "^\[0-9\]+\.\[0-9\]+$" $network]>0} {
	set result ""
	for {set a3 0} {$a3<256} {incr a3} {
	    lappend result [netping $network.$a3]
	}
	return "[join $result]"
    }
}

##
## IP Troubleshooter Tool for INED
##

proc ping {list} {

    set nodes ""
    set networks ""
    foreach comp $list {
	set type [ined_type $comp]
        if {$type=="NODE"}    { lappend nodes $comp }
	if {$type=="NETWORK"} { lappend networks $comp }
    }

    if {[llength $nodes]>0} {
	foreach comp $nodes {
	    set host [lindex [ined_name $comp] 0]
	    set ip [Get_IP_Address $comp]
	    if {$ip==""} {
		ined acknowledge "Can not lookup IP Address for $host."
		continue;
	    }
	    set rtt [lindex [join [icmp -size 0 -timeout 5 $ip]] 1]
	    if {$rtt>=0} {
		writeln "Round trip time for $host \[$ip\]: $rtt ms"
	    } else {
		writeln "$host \[$ip\] not reachable."
	    }
	}
	writeln
    }

    if {[llength $networks]>0} {
	foreach comp $networks {
	    set net [ined_name $comp]
	    set ip [ined_address $comp]
	    if {![regexp "^\[0-9\]+\.\[0-9\]+\.\[0-9\]+$" $ip]} {
		ined acknowledge "Can not lookup IP Address for $net."
		continue
	    }
	    writeln "Network ping for $net \[$ip\]:"
	    foreach pr [netping $ip] {
		set pr_ip [lindex $pr 0]
		set pr_time [lindex $pr 1]
		if {$pr_time>=0} {
		    if [catch {nslook $pr_ip} pr_host] {
			set pr_host $pr_ip
		    }
		    writeln "  Round trip time for [lindex $pr_host 0]: $pr_time ms"
		}
	    }
	    writeln
	}
    }
}

##
## Ping a host with a series of packets with varying packet sizes.
##

proc "multi ping" {list} {
    static increment
    static maxsize

    if ![info exists increment] { set increment "128" }
    if ![info exists maxsize]   { set maxsize "2048" }

    set result [ined request "Send icmp Packets with varying sizes" \
		" { {Increment \[bytes\]:} $increment scale 1 256 } \
		  { {Max. packet  size \[bytes\]:} $maxsize scale 64 2048 }" ]

    if {$result==""} return

    set increment [lindex $result 0]
    set maxsize   [lindex $result 1]

    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
	    set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
	    writeln "Multi ping for $host \[$ip\]:"
	    for {set size 0} {$size <= $maxsize} {incr size $increment} {
		set time [lindex [lindex  [icmp -size $size $ip] 0] 1]
		writeln [format "  Round trip time (%5s bytes) %5s ms" \
			 $size $time]
	    }
	    writeln
	}
    }
}

##
## Trace the route to a host using the van Jakobsen method.
##

proc "trace route" {list} {
    global max_route_len

    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
	    writeln "Routing trace for $host \[$ip\]:"
	    set ttl 1
	    set trace ""
	    set trace_ip ""
	    while {$trace_ip!=$ip} {
		set trace [lindex [icmp -trace $ttl $ip] 0]
		set trace_ip [lindex $trace 0]
		if {[catch {nslook $trace_ip} trace_name]} {
		    set trace_name ""
		}
		set trace_time [lindex $trace 1]
		if {$trace_time<0} {
		    writeln [format "%2d  %s" $ttl "* * *"]
		} else {
		    writeln [format "%2d  %-35s %17s %5d ms" $ttl \
			     $trace_name "\[$trace_ip\]" $trace_time]
		}
		incr ttl
		if {$ttl > $max_route_len} break;
	    }
	    writeln
	}
    }
}

##
## Get the netmask using a special icmp request.
##

proc "netmask" {list} {
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
            set result [lindex [join [icmp -mask $ip]] 1]
	    writeln "Netmask for $host \[$ip\]: $result"
	}
    }
    writeln
}

##
## Open an xterm with a telnet for every selected node.
##

proc telnet {list} {
    global env

    set xterm "/usr/bin/X11/xterm"

    if {[info exists env(PATH)]} {
	foreach d [split $env(PATH) :] {
	    if {[file exists $d/xterm]} {
		set xterm $d/xterm
	    }
	}
    }
    
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
	    set host [lindex [ined_name $comp] 0]
	    set ip [Get_IP_Address $comp]
	    if {$ip==""} {
		ined acknowledge "Can not lookup IP Address for $host."
		continue;
	    }
	    catch {exec $xterm -title $host -e telnet $ip &}
	}
    }
}

##
## Open an xterm with a rlogin for every selected node.
##

proc rlogin {list} {
    global env

    set xterm "/usr/bin/X11/xterm"

    if {[info exists env(PATH)]} {
	foreach d [split $env(PATH) :] {
	    if {[file exists $d/xterm]} {
		set xterm $d/xterm
	    }
	}
    }
    
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
	    set host [lindex [ined_name $comp] 0]
	    set ip [Get_IP_Address $comp]
	    if {$ip==""} {
		ined acknowledge "Can not lookup IP Address for $host."
		continue;
	    }
	    catch {exec $xterm -title $host -e rlogin $ip &}
	}
    }
}

##
## Ask the inetd of the selected nodes for the daytime.
##

proc daytime {list} {
    foreach comp $list {
        if {[ined_type $comp] == "NODE"} {
	    set host [lindex [ined_name $comp] 0]
	    set ip [Get_IP_Address $comp]
	    if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
	    if {[catch {connect $ip daytime} f]} {
		ined acknowledge "Can not connect to $host."
	    } else {
		set time [string trimright [gets $f]]
		writeln "Daytime for $host \[$ip\]: $time"
		shutdown $f all
	    }
	}
    }
    writeln
}

##
## Connect to the finger port and get what they are willing to tell us.
## This is also nice for cisco routers.
##

proc finger {list} {
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
	    if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
	    if {[catch {connect $ip finger} f]} {
		ined acknowledge "Can not connect to $host."
		continue
	    }
	    puts $f ""
	    writeln "Finger for $host \[$ip\]:"
	    while {! [eof $f]} {
		writeln [string trimright [gets $f] "\r\n"]
	    }
	    shutdown $f all
	    writeln
	}
    }
}

##
## Get the host information from the name server. This should really
## go to the troubleshooter. And it should use the buildin hostname
## lookup command.
##

proc "DNS info" {list} {
    foreach comp $list {
	if {[ined_type $comp]!="NODE"} continue;
	set host [ined_name $comp]
	set ip [Get_IP_Address $comp]
	if {[catch {dns ptr $ip} name]} { set name $host }
        set name [lindex $name 0]
	if {[catch {dns a $name} a]}             { set a "" }
	if {[catch {dns ptr [lindex $a 0]} ptr]} { set ptr "" }
	if {[catch {dns hinfo $name} hinfo]}     { set hinfo "" }
	if {[catch {dns mx $name} mx]}           { set mx "" }

	set soa ""
	while {([llength [split $name .]]>0) && ($soa=="")} {
	    if {[catch {dns soa $name} soa]} { set soa "" }
	    set name [join [lrange [split $name .] 1 end] .]
	}

	writeln "Domain Name Server information for $host \[$ip\]:"
	if {$ptr != ""} { writeln "Name: $ptr" }
	if {$a   != ""} { writeln "Address: $a" }
	if {[lindex $hinfo 0] != ""} { writeln "Machine: [lindex $hinfo 0]" }
	if {[lindex $hinfo 1] != ""} { writeln "OS: [lindex $hinfo 1]" }
	if {$mx != ""}  { writeln "MX: $mx" }
	if {$soa != ""} { writeln "SOA: $soa" }

	writeln
    }
}

##
## Scan the /etc/services file and try to connect to all ports
## listed there. This will not really tell us, if we could you
## a service but it may be a hint.
##

proc "TCP services" {list} {

    if {[catch {open /etc/services r} s]} {
	ined acknowledge "Can not open /etc/services."
	return
    }

    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
	    set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
	    writeln "TCP services on $host \[$ip\]:"
	    seek $s 0
	    set service "X11  6000/tcp"
	    while {! [eof $s]} {
		if {[regexp "\[0-9\]*/tcp" $service]>0} {
		    set name [lindex $service 0]
		    if {[catch {connect $ip $name} f]} {
			set service [gets $s]
			continue
		    }
		    shutdown $f all
		    writeln [format "  %-12s %8s %s" $name \
			     [lindex $service 1] [lrange $service 2 end]]
		}
                set service [gets $s]
	    }
	    writeln
	}
    }
    close $s
}

##
## Ask the portmapper of registered RPC services for the selected nodes.
##

proc "RPC services" {list} {
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
	    if {[catch {rpc info $ip} server]} {
		ined acknowledge "Can not connect to $host."
		continue
	    }
	    if {$server == "" } {
		ined acknowledge "No RPC servers registered for $host"
		continue
	    }

	    writeln "Registered RPC server for $host \[$ip\]:"
	    foreach probe [lsort -ascii $server] {
		set probe [join $probe]
		if {[catch {eval rpc probe $ip $probe} res]} {
		    writeln "  $probe (probe failed)"
		} else {
		    writeln "  $probe (probe successful)"
		}
	    }
	    writeln
        }
    }
}

##
## Get the NFS exports list by querying the mountd of the selected nodes.
##

proc "NFS exports" {list} {
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
            if {[catch {rpc exports $ip} inf]} {
                ined acknowledge "Can not connect to $host."
                continue
            }
	    if {$inf == ""} {
		ined acknowledge "No filesystems exported by $host."
		continue
	    }
	    catch {unset fstab}
	    writeln "Exported NFS Filesystems for $host \[$ip\]:"
	    foreach fs $inf {
		writeln "  [lindex $fs 0] -> [join [lrange $fs 1 end]]"
	    }
	    writeln
        }
    }
}

##
## Show who has mounted partitions of the selected nodes.
##

proc "NFS mounts" {list} {
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
            if {[catch {rpc mount $ip} inf]} {
                ined acknowledge "Can not connect to $host."
                continue
            }
	    if {$inf == ""} {
                ined acknowledge "No filesystems mounted from $host."
                continue
            }
	    catch {unset fstab}
            foreach fs $inf {
		set fsname [lindex $fs 0]
		set mname  [join [lrange $fs 1 end]]
		if {![info exists fstab($fsname)]} {
		    set fstab($fsname) $mname
		} else {
		    lappend fstab($fsname) $mname
		}
            }
	    writeln "Mounted NFS Filesystems from $host \[$ip\]:"
	    foreach fsname [lsort -ascii [array names fstab]] {
		writeln "  $fsname -> $fstab($fsname)"
	    }
	    writeln
        }
    }
}

##
## Make a call to the rstat daemon and report the results for
## a time intervall which is queried from the user.
##

proc rstat_diff {l1 l2 period} {
    set len [llength $l1]
    set res ""
    for {set i "0"} {$i<$len} {incr i} {
        set el1 [lindex $l1 $i]
        set el2 [lindex $l2 $i]
        set tmp [lindex $el1 2]
        if {[lindex $el1 1]=="Counter"} {
            set tmp [expr {[lindex $el1 2]-[lindex $el2 2]}]
            set tmp [expr {"$tmp.0" / $period}]
	}
        if {[lindex $el1 1]=="Gauge"} {
            set tmp [expr {[lindex $el1 2]/256.0}]
        }
        lappend res [format "%-12s %-12s %16.3f" \
		     [lindex $el1 0] [lindex $el1 1] $tmp]
    }
    return $res
}

proc rstat {host period} {
    set stat [rpc stat $host]
    sleep $period
    set newstat [rpc stat $host]
    set res ""
    foreach s [rstat_diff $newstat $stat $period] { lappend res $s }
    return $res
}

proc "rstat RPC" {list} {
    static intervall
    if {![info exists intervall]} { set intervall 1 }
    set intervall [ined request "Show rstat results (rstat)" \
		   "{{Intervall \[s\]:} $intervall scale 1 10}" ]
    if {$intervall==""} return
    foreach comp $list {
	if {[ined_type $comp]=="NODE"} {
	    set host [lindex [ined_name $comp] 0]
	    set ip [Get_IP_Address $comp]
	    if {$ip==""} {
		ined acknowledge "Can not lookup IP Address for $host."
		continue;
	    }
	    if {[catch {rstat $ip $intervall} result]==0} {
		ined browse "rstat RPC result for $host ($intervall seconds)" $result
	    } else {
		ined acknowledge "rstat RPC failed for $host"
	    }
	}
    }
}

##
## Make a call to the ether daemon and report the results for
## a time intervall which is queried from the user.
##

proc "ether RPC" {list} {
    static intervall
    if {![info exists intervall]} { set intervall 1 }
    set intervall [ined request "Show ether results (etherd)" \
                   "{{Intervall \[s\]:} $intervall scale 1 10}" ]
    if {$intervall==""} return
    foreach comp $list {
	if {[ined_type $comp]=="NODE"} {
	    set host [lindex [ined_name $comp] 0]
	    set ip [Get_IP_Address $comp]
	    if {$ip==""} {
		ined acknowledge "Can not lookup IP Address for $host."
		continue;
	    }
	    if {[catch {
		rpc ether open $ip
		sleep $intervall
		set res [rpc ether $ip]
		rpc ether close $ip
		set result ""
		foreach elem $res {
		    lappend result [eval "format {%-12s %-12s %12d} $elem"]
		}
	    }]} {
		ined acknowledge "ether RPC failed for $host"
	    } else {
		ined browse "ether RPC result for $host ($intervall seconds)" $result
	    }
	}
    }
}

##
## Set some constants that control the tool.
##

proc "set parameter" {list} {
    global icmp_retries icmp_timeout icmp_delay
    global max_route_len

    set result [ined request "IP Discover Parameter" \
	"{ {# of ICMP retries:} $icmp_retries scale 1 10} \
         { {ICMP timeout \[s\]:} $icmp_timeout scale 1 42} \
         { {Delay between ICMP packets \[ms\]:} $icmp_delay scale 1 100} \
         { {Max. length of a route:} $max_route_len scale 1 30} "]

    if {$result==""} return

    set icmp_retries  [lindex $result 0]
    set icmp_timeout  [lindex $result 1]
    set icmp_delay    [lindex $result 2]
    set max_route_len [lindex $result 3]

    icmp -retries $icmp_retries
    icmp -timeout $icmp_timeout
    icmp -delay   $icmp_delay
}

##
## Display some help about this tool.
##

proc "help IP Trouble" {list} {
    ined browse "Help about IP Trouble" {
	"ping:" 
	"    Send an ICMP echo request to all selected objects. If you" 
	"    select a network with a valid address, then all address of" 
	"    address space will be queried (be careful!)." 
	"" 
	"multi ping:" 
	"    Send ICMP echo request with varying packet size." 
	"" 
	"netmask:" 
	"    Query the netmask of all selected nodes." 
	"" 
	"trace route:" 
	"    Trace the route to the selected nodes." 
	"" 
	"telnet:" 
	"    Open a telnet session to the selected nodes." 
	"" 
	"rlogin:" 
	"    Open a rlogin session to the selected nodes." 
	"" 
	"daytime:" 
	"    Get the daytime of all selected nodes." 
	"" 
	"finger:" 
	"    Finger all selected nodes." 
	"" 
	"DNS info:" 
	"    Show information stored in the Domain Name Service." 
	"" 
	"TCP services:" 
	"    Find available TCP services on all selected nodes." 
	"" 
	"RPC services:" 
	"    Find available RPC services on all selected nodes." 
	"" 
	"NFS exports:" 
	"    List the NFS export list of all selected nodes." 
	"" 
	"NFS mounts:" 
	"    Show who has mounted the selected nodes." 
	"" 
	"rstat RPC:" 
	"    Display the result of a call to the rstat RPC service." 
	"" 
	"ether RPC:" 
	"    Display the result of a call to the etherstat RPC service." 
	"" 
	"set parameter:" 
	"    Display and change parameters of the tools." 
    }
}

##
## Delete the tools created by this interpreter.
##

proc "delete IP Trouble" {list} {
    global tools
    foreach id $tools { ined delete $id }
    exit
}

set tools [ ined create TOOL "IP Trouble" \
	    ping "multi ping" "netmask" "trace route" "" \
	    telnet rlogin daytime finger "DNS info" "" \
	    "TCP services" "RPC services" "" \
	    "NFS exports" "NFS mounts" "" \
	    "set parameter" "" \
	    "help IP Trouble" "delete IP Trouble" ]
