#! /usr/local/bin/scotty -inf
##
## Communication utilities for the [TK]INED editor.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

set role "slave"
set port 6336
set master_name "localhost"
set master_accept "automatically"
set master_dump "true"

set slave_list ""

##
## Delete an element from a list.
##

proc ldelete {list element} {
    upvar $list mylist
    set result ""
    foreach e $mylist {
        if {$e != $element} { lappend result $e }
    }
    return $result
}

##
## Test if we are at home. Otherwise issue a warning message.
##

proc warning {} {
    if {[catch {dns ptr [nslook [exec hostname]]} name]} {
        return 0
    }
    if {![string match "*cs.tu-bs.de" $name]} {
	ined acknowledge \
	    "Communication s still under construction. Use with care!!"
    }
}

##
## This is the hard part. Here we must parse all commands to eliminate
## simple query commands and to map the ids to the ids used by our slave
## tkined.
##

proc process {cmd result} {
    static myid

    if {[llength $cmd] <= 2} return
    case [lindex $cmd 1] in {
	{select} {
	    puts "select discarded: $cmd"; flush stdout
	    return
	}
	{create} { 
	    set id [string trim $result]
	    switch [lindex $cmd 2] {
		LINK {
		    set ida [lindex $cmd 3]
		    if {![info exists myid($ida)]} return
		    set cmd [lreplace $cmd 3 3 $myid($ida)]
		    set idb [lindex $cmd 4]
		    if {![info exists myid($idb)]} return
		    set cmd [lreplace $cmd 4 4 $myid($idb)]
		}
		GROUP {
		    set member [lrange $cmd 3 end]
		    set cmd [lrange $cmd 0 2]
		    foreach guy $member {
			if {![info exists myid($guy)]} return
			lappend cmd $myid($guy)
		    }
		}
	    }
	    regsub ined $cmd "ined -noupdate" cmd
	    if {[catch {eval $cmd} res]} return
	    set myid($id) $res
	    return
	}
	{delete} {
	    set ids [join [lrange $cmd 2 end]]
	    set myids ""
	    foreach id $ids {
		if {[info exists myid($id)]} {
		    lappend myids $myid($id)
		}
	    }
	    set cmd [lreplace $cmd 2 end $myids]
	    regsub ined $cmd "ined -noupdate" cmd
	    catch {eval $cmd}
	}
	{   retrieve 
	    move 
	    icon 
	    font 
	    color 
	    label 
	    name 
	    address 
	    oid 
	    collapse
	    expand
	    lower 
	    raise} {
		set id [lindex $cmd 2]
		if {![info exists myid($id)]} return
		set cmd [lreplace $cmd 2 2 $myid($id)]
		regsub ined $cmd "ined -noupdate" cmd
		catch {eval $cmd}
		return
	    }
	
	{stripchart piechart barchart} {
	    set id [lindex $cmd 2]
	    if {![info exists myid($id)]} return
	    set cmd [lreplace $cmd 2 2 $myid($id)]
	    regsub ined $cmd "ined -noupdate" cmd
	    catch {eval $cmd}
	    return
	}
	
	{default} {
	    puts stderr "** process: unkown command $cmd"; flush stderr
	}
    }
}

##
## This proc sets up incoming connections. It is scheduled by the
## job manager and checks for requests on the control socket. It
## accepts a connection to get the remote address and queries the
## user if he really wants this connection. If so, the new socket
## is appended to the slave_list.
##

proc accept_connection {} {
    global slave_list
    global control_socket
    global master_accept
    global master_dump
    global connect_info

    if {[catch {select $control_socket {} {} 0.01} socket]} {
	puts stderr "** accept_connection select failed: $socket"
	return
    }
    set socket [lindex $socket 0]

    if {$socket == $control_socket} {
	set new_socket [accept $socket]

	# Check if we want this connection.

	if {$master_accept != "automatically"} {

	    set rem_ip [lindex $connect_info($new_socket) 0]
	    if {[catch {nslook $rem_ip} rem_name]} {
		set rem_name ""
	    }
	    set rem_name [lindex $rem_name 0]

	    set res [ined confirm \
	      "Accept a slave request coming in from $rem_name \[$rem_ip\] ?"]

	    if {$res != "yes"} {
		catch {shutdown $new_socket all}
		catch {close $new_socket}
		return
	    }
	}

	# Put the socket to our distribution list.

	if {$master_dump} {
	    download $new_socket
	}

	lappend slave_list $new_socket

	# Start the tkined trace.

	ined trace trace_callback
    }
}

##
## Download the whole image to the connection given by socket.
##

proc download {socket} {
    set ok true
    set todo ""
    while {$ok == "true"} {
	set newtodo ""
	foreach aa [eval ined retrieve $todo] {
	    set id [ined_id $aa]
	    switch [ined_type $aa] {
		NODE {
		    set node($id) $aa
		}
		NETWORK {
		    set network($id) $aa
		}
		LINK {
		    set link($id) $aa
		}
		GROUP {
		    set group($id) $aa
		}
	    }
	    if {[ined_type $aa] == "GROUP"} {
		lappend newtodo [ined_member $aa]
	    }
	}
	set todo [join $newtodo]
	if {$todo == ""} { set ok "false" }
    }

    if [info exists node] {
	foreach id [array names node] {
	    puts $socket "ined create NODE"; flush $socket
	}
    }
    if [info exists network] {
	foreach id [array names network] {
	    puts $socket "ined create NETWORK"; flush $socket
	}
    }
    if [info exists link] {
	foreach id [array names link] {
	    puts $socket "ined create LINK [ined_ida $link($id)] [ined_idb $link($id)]"; flush $socket
	}
    }
    if [info exists group] {
	foreach id [array names group] {
	    puts $socket "ined create GROUP [ined_members $group($id)]"; flush $socket
	}
    }
}

##
## This is the callback that absorbs the tkined trace. It simply forwards
## the trace to all the slaves in slave_list.
##

proc trace_callback {args} {
    global slave_list
    global connect_info

    set args [eval list $args]

    foreach slave $slave_list {
	if {[catch {puts $slave $args; flush $slave}]} {
	    set ip   [lindex $connect_info($slave) 0]
	    set port [lindex $connect_info($slave) 1]
	    if {[catch {nslook $ip} name]} {
		set name ""
	    }
	    set name [lindex $name 0]
	    ined acknowledge "Connection from $name \[$ip\] closed."
	    catch {shutdown $slave}
	    catch {close $slave}
	    set slave_list [ldelete slave_list $slave]

	    if {[llength $slave_list] == 0} {
		ined trace ""
	    }
	}
    }
}

##
## Start the server side. Create a control socket and create a job
## that checks for incoming requests. Also tell tkined that we want
## a trace of all operations.
##

proc start_master {} {
    global port
    global slave_list
    global control_socket
    global accept_job
    
    if {![info exists control_socket]} {
	if {[catch {connect -server "" $port} control_socket]} {
	    ined acknowledge "Can not create server socket:" $control_socket
	    unset control_socket
	    exit
	}
	set accept_job [ined_create_job accept_connection 2]
    }
}

##
## Become a slave. Connect to a server. If succesful read until eof
## and process everything we get. This should not block the command
## processing.
##

proc start_slave {} {
    global port
    global master_name
    global tools
    
    if {[catch {connect $master_name $port} socket]} {
	ined acknowledge "Can not connect to $master_name:" $socket
	exit
    }

    while {![eof $socket]} {
	gets $socket line
	set tmp [split $line >]
	set cmd [lindex $tmp 0]
	set result [join [lindex $tmp 1]]
	process $cmd $result
    }

    ined acknowledge "Connection to $master_name closed."

    foreach id $tools { ined delete $id }
    exit

}

##
## Show the current connections. This is useful to keep track of
## all the guys looking on my net and it allows me to kick bad guys.
##

proc "show and kill" {list} {
    global slave_list
    global control_socket
    global connect_info

    if {![info exists control_socket]} {
        ined acknowledge "Master not started."
        return
    }

    if {[llength $slave_list] == 0} {
        ined acknowledge "No connections established yet."
        return
    }

    set res ""
    foreach slave $slave_list {
	set ip   [lindex $connect_info($slave) 0]
	set port [lindex $connect_info($slave) 1]
	if {[catch {nslook $ip} name]} {
	    set name ""
	}
	set name [lindex $name 0]
	lappend res [format "%5d %s %s" $port $ip $name]
    }

    set aa [ined list "Choose a connection to kill:" $res]
    if {$aa == ""} return

    set port [lindex [join $aa] 0]
    set ip [lindex [join $aa] 1]

    set res ""
    foreach slave $slave_list {
        if {([lindex $connect_info($slave) 0] == $ip)
	    && ([lindex $connect_info($slave) 1] == $port)} {
	    catch {shutdown $slave}
	    catch {close $slave}
	} else {
	    lappend res $slave
	}
    }
    set slave_list $res

    if {[llength $slave_list] == 0} {
	ined trace ""
    }
}

##
## Set the port number where the master listens for client 
## connections.
##

proc "Parameter" {list} {
    global master_accept
    global master_dump

    set result [ined request "Communication Parameter" \
		"{{Master Accept:} $master_accept radio automatically query} \
		 {{Master Dump:} $master_dump radio true false}" ]
    
    if {$result==""} return

    set master_accept [lindex $result 0]
}

##
## Display some help about this tool.
##

proc "help Communication" {list} {

    global role

    if {$role == "master"} {
	ined browse "Help about Comunication" {
	    "show and kill:" 
	    "    See who is connected to the master and kill one" 
	    "    connection if you like." 
	    "" 
	    "Parameter:" 
	    "    Examine and set the parameters." 
	}
    } else {
	ined browse "Help about Comunication" { 
	    "this help left intentionally blank" 
	}
    } 
}

##
## Delete the TOOL Manager and exit this interpreter.
##

proc "delete Communication" {list} {
    global role
    global slave_list
    global control_socket
    global accept_job
    global tools

    if {$role == "master"} {
	ined trace ""
	ined_kill_job $accept_job
	shutdown $control_socket all
	close $control_socket
	foreach slave $slave_list {
	    catch {shutdown $slave}
	    catch {close $slave}
	}
	set slave_list ""
	unset control_socket
    }

    foreach id $tools { ined delete $id }
    exit
}

##
## The main procedure issues a dialog which lets you define
## if we are a master or a slave and if we automatically
## accept connections.
##

proc main {} {
    global role
    global port
    global master_name
    global master_accept
    global master_dump
    global tools

    warning

    # Use any selected node as a default for the master_name

    foreach aa [ined select] {
	set comp [ined retrieve $aa]
	if {[ined_type $comp] == "NODE"} {
	    set master_name [lindex [ined_name $comp] 0]
	}
    }

    # Get the parameters and quit if there is nothing to do here.
    
    set res [ined request " \
	{Role: $role radio master slave} \
	{Port: $port scale [expr {$port-10}] [expr {$port+20}]} \
	{{Master Host:} $master_name} \
	{{Master Accept:} $master_accept radio automatically query} \
        {{Master Dump:} $master_dump radio true false}" ]

    if {$res == ""} exit

    set role [lindex $res 0]
    set port   [lindex $res 1]
    set master_name [lindex $res 2]
    set master_accept [lindex $res 3]

    # Create a tool that contains the master or slave commands.

    if {$role == "master"} {
	start_master
	set tools [ ined create TOOL "Master" \
		   "show and kill" "Parameter" "" \
		   "help Communication" "delete Communication" ]
    } else {
	start_slave
	set tools [ ined create TOOL "$master_name ($port)" \
		   "help Communication" "delete Communication" ]
    }
}

main
