/* tcl_dbm.c
 * (N)DBM interface for Tcl
 * 
 * Poul-Henning Kamp, phk@data.fls.dk
 * 921027 0.00
 * 921222 0.10	New syntax based on descriptors
 */


#include <tcl.h>
#include <tclHash.h>
#include <tclExtend.h>

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <memory.h>

#include <ndbm.h>

#define IFW(st) if(**argv==*st && !strcmp(*argv,st))
#define ERR_N_ARG(syntax) { Tcl_AppendResult(interp, "bad # arg: ", \
	"\nSyntax is: ", syntax, 0) ; return TCL_ERROR; }
#define CHKNARG(min,max,syntax) if(argc<min || argc>max) ERR_N_ARG(syntax)

/*
 * t_cldat
 * -------
 */

typedef struct {
    void_pt		handles;
    } t_cldat;

/*
 * t_desc
 * ------
 */

typedef struct {
    DBM			*dbm;
    Tcl_HashTable	*cache;
    } t_desc;

/*
 * DbmProc(cd,interp,argc,argv)
 * ============================
 *
 * Do the actual work.
 *
 */

static int
DbmProc(cd,interp,argc,argv)
    t_cldat	*cd;
    Tcl_Interp	*interp;
    int		argc;
    char	**argv;
    {
    int i;
    char *p;
    char buf[64];
    int flags = O_RDONLY;
    int mode = 0644;
    DBM *d;
    datum key,content;
    t_desc *td;
    Tcl_HashEntry *hash;

    argv++; argc--;

    if(!argc)
	ERR_N_ARG("dbm {open|close|fetch|store|firstkey|nextkey|forall}");

/*
!
! dbm open <file> [<flgs> [<mode>]]		# return <sess>
!     opens the specified <file> using <mode>.
!     <file> will be appended '.pag' and '.dir' before opening
!     <flgs> is a subset of these chars: (default: r)
!         r -- read mode           c -- create
!         w -- read-write mode     t -- truncate
!     <mode> is a octal integer used for creating files. (default 644)
*/
    IFW("open")
	{
	CHKNARG(2, 4, "dbm open <file> [<flgs> [<mode>]]");
	if(argc > 2)
	    for(p=argv[2];*p;p++)
		{
		switch(*p) 
		    {
		    case 'r': flags |= O_RDONLY; break;
		    case 'w': flags |= O_RDWR; break;
		    case 'c': flags |= O_CREAT; break;
		    case 't': flags |= O_TRUNC; break;
		    default:
			Tcl_AppendResult(interp, "what ??: ",
			    "\n<flgs> must be a subset of 'rwct'", 0);
			return TCL_ERROR;
		    }
		}
	if(argc > 3)
	    if(1 != sscanf(argv[3],"%o",&mode))
		{
		Tcl_AppendResult(interp, "what ??: ",
		    "\n<mode> must be an octal integer", 0);
		return TCL_ERROR;
		}
	mode &= 0777;

	d = dbm_open(Tcl_TildeSubst(interp,argv[1]), flags, mode);
	if(!d)
	    {
	    Tcl_AppendResult(interp, "couldn't open \"",argv[1],
		"\": ", Tcl_UnixError(interp),0);
	    return TCL_ERROR;
	    }
	td = Tcl_HandleAlloc(cd->handles,buf);
	memset(td,0,sizeof *td);
	td->dbm = d;
	Tcl_AppendResult(interp,buf,0);
	return TCL_OK;
	}

    /* everything from here needs a handle, but we want #args messages to come
     * from each one in turn to get the syntax presented
     */
    if(argc > 1 && !(td = Tcl_HandleXlate(interp,cd->handles,argv[1])))
	return TCL_ERROR;
/*
!
! dbm close <sess>
!     closes the specified dbm file.
*/
    IFW("close")
	{
	CHKNARG(2,2,"dbm close <sess>");
	dbm_close(td->dbm);
	Tcl_HandleFree(cd->handles,td);
	return TCL_OK;
	}
/*
!
! dbm cache <sess> 
!     cache the specified dbm file.
*/
    IFW("cache")
	{
	CHKNARG(2,2,"dbm cache <sess>");
	if(td->cache)
		return TCL_OK;
	td->cache = ckalloc(sizeof *(td->cache));
	Tcl_InitHashTable(td->cache,TCL_STRING_KEYS);
	return TCL_OK;
	}
/*
!
! dbm fetch <sess> <key> [<var>]	
!     will try to fetch the value addressed by <key>
!     if <var> is present it will contain the result, and the return
!         value will be a boolean: 1 on success, 0 on failure.
!     else the found value will be returned and failures will give an
!         error return;
*/
    IFW("fetch")
	{
	CHKNARG(3,4,"dbm fetch <sess> <key> [<var>]");
	if(td->cache)
	    {
	    hash=Tcl_FindHashEntry(td->cache,argv[2]);
	    if(hash)
		{
		content.dptr = hash->clientData;
		goto gotit;
		}
	    }
	key.dptr = argv[2]; key.dsize = strlen(key.dptr)+1;
	content = dbm_fetch(td->dbm,key);
	if(td->cache)
	    {
	    hash = Tcl_CreateHashEntry(td->cache,argv[2],&i);
	    if(content.dptr)
		{
		hash->clientData = ckalloc(strlen(content.dptr)+1);
		strcpy(hash->clientData,content.dptr);
		}
	    else
		hash->clientData = 0;
	    }
    gotit:
	if(argc > 3)
	    {
	    if(!content.dptr) 
		{
		Tcl_SetResult(interp,"0",TCL_VOLATILE);
		Tcl_SetVar(interp,argv[3],"",0);
		}
	    else
		{
		Tcl_SetResult(interp,"1",TCL_VOLATILE);
		Tcl_SetVar(interp,argv[3],content.dptr,0);
		}
	    }
	else
	    {
	    if(!content.dptr) 
		{
		Tcl_AppendResult(interp, "no match on key \"",argv[2],"\"",0);
		return TCL_ERROR;
		}
	    Tcl_SetResult(interp,content.dptr,TCL_VOLATILE);
	    }
	return TCL_OK;
	}
/*
!
! dbm store <sess> <key> <cont> [insert|replace]
!     stores the <cont> under the <key>. 'replace' is the default mode.
!     returns an error on failure.
*/
    IFW("store")
	{
	CHKNARG(4,5,"dbm store <sess> <key> <cont> [insert|replace]");
	flags = DBM_REPLACE;
	if(argc > 4)
	    {
	    if(!strcmp(argv[4],"insert"))
		flags = DBM_INSERT;
	    else if(!strcmp(argv[4],"replace"))
		flags = DBM_REPLACE;
	    else
		{
		Tcl_AppendResult(interp,
		   "what ?? either 'insert' or 'replace'",0);
	 	return TCL_ERROR;
		}
	    }
	key.dptr = argv[2]; key.dsize = strlen(key.dptr)+1;
	content.dptr = argv[3]; content.dsize = strlen(content.dptr)+1;
	i = dbm_store(td->dbm,key,content,flags);
	if(td->cache)
	    {
	    hash = Tcl_CreateHashEntry(td->cache,argv[2],&i);
	    hash->clientData = ckalloc(strlen(content.dptr)+1);
	    strcpy(hash->clientData,content.dptr);
	    }
	if(i<0) 
	    {
	    Tcl_AppendResult(interp,
		"Couldn't store: ", Tcl_UnixError(interp),0);
	    return TCL_ERROR;
	    }
	if(i) 
	    {
	    Tcl_AppendResult(interp,
		"Duplicate key: \"", argv[2], "\"", 0);
	    return TCL_ERROR;
	    }
	return TCL_OK;
	}
/*
!
! dbm delete <sess> <key>
!     Deletes the entry with the specified <key> 
!     If the record is deleted return value is "1" 
!     else it returns "0" 
!     probably because <key> wasn't there to begin with 
*/
    IFW("delete")
	{
	CHKNARG(3,3,"dbm delete <sess> <key>");
	if(td->cache)
	    {
	    hash=Tcl_FindHashEntry(td->cache,argv[2]);
	    if(hash->clientData)
		ckfree(hash->clientData);
	    if(hash) Tcl_DeleteHashEntry(hash);
	    }
	key.dptr = argv[2]; key.dsize = strlen(key.dptr)+1;
	i=dbm_delete(td->dbm,key);
	if(i) Tcl_SetResult(interp,"0",TCL_VOLATILE);
	else  Tcl_SetResult(interp,"1",TCL_VOLATILE);
	return TCL_OK;
	}
/*
!
! dbm firstkey <sess> [<var>]
!     	Finds the first key in <tag>.
!     	if <var> is specified:
!     	    if no keys found:
!     	        <var> is set to "" and "0" is returned
!     	    else
!     	        <var> is set to the key and "1" is returned.
!     	else 
!     	    if no keys found:
!     	        "" is returned
!     	    else
!     	        the key is returned
*/
    IFW("firstkey")
	{
	CHKNARG(2,3,"dbm firstkey <sess> [<var>]");
	key=dbm_firstkey(td->dbm);
	goto asnextkey;
	}
/*
!
! dbm nextkey <sess> [<var>]
!     	Finds the next key in <tag>.
!     	if <var> is specified:
!     	    if the end is reached:
!     	        <var> is set to "" and "0" is returned
!     	    else
!     	        <var> is set to the key and "1" is returned.
!     	else
!     	    if the end is reached:
!     	        "" is returned
!     	    else
!     	        the key is returned
*/
    IFW("nextkey")
	{
	CHKNARG(2,3,"dbm nextkey <sess> [<var>]");
	key=dbm_nextkey(td->dbm);
    asnextkey:
	if(argc > 2)
	    {
	    if(key.dptr)
		{
		Tcl_SetVar(interp,argv[2],key.dptr,0);
		Tcl_SetResult(interp,"1",TCL_VOLATILE);
		}
	    else
		{
		Tcl_SetVar(interp,argv[2],"",0);
		Tcl_SetResult(interp,"0",TCL_VOLATILE);
		}
	    }
	else
	    {
	    if(key.dptr)
		Tcl_SetResult(interp,key.dptr,TCL_VOLATILE);
	    }
	return TCL_OK;
	}
/*
! dbm forall <sess> <key_var> <proc>
!     	executes <proc> for all keys found in <tag>
!     	The actual key_value is available in $<key_var>
!     	Example:
!     		xx forall key {
!     			puts stdout "$key: [xx fetch $key]"
!     		}
!
*/
    IFW("forall")
	{
	CHKNARG(4,4,"dbm forall <sess> <key_var> <proc>");
	key=dbm_firstkey(td->dbm);
	if(!key.dptr) return TCL_OK;
	for(;;)
	    {
	    Tcl_SetVar(interp,argv[2],key.dptr,0);
	    i = Tcl_VarEval(interp,argv[3],0);
	    if(i == TCL_BREAK) break;
	    else if(i != TCL_OK && i != TCL_CONTINUE) return i;
	    key=dbm_nextkey(td->dbm);
	    if(!key.dptr) break;
	    }
	return TCL_OK;
	}

    Tcl_AppendResult(interp, "what ??: ",
	"\nSyntax is:", 
	"dbm {open|close|fetch|store|firstkey|nextkey|forall}",
	0 );
    return TCL_ERROR;
    }

/*
 * init_tcl_dbm(interp)
 * ====================
 *
 * Initialize the dbm interface for the interpreter 'interp'
 *
 */

void
init_tcl_dbm(interp)
    Tcl_Interp	*interp;
    {
    t_cldat *cd;

    cd = (t_cldat *)ckalloc(sizeof *cd);
    memset((void*)cd,0,sizeof *cd);
    cd->handles = Tcl_HandleTblInit("dbm",sizeof(t_desc),10);
    Tcl_CreateCommand(interp,"dbm",DbmProc,(ClientData)cd,0);
    }
