/*
 * objects.c
 *
 * This file contains all functions that manipulate the state
 * of tkined objects. These functions get called whenever a user
 * invokes a command of finishes a tool, or when a command is send
 * from an interpreter. Every command calls a corresponding tk 
 * procedure that is responsible to manipulate the display according
 * to the changes. The naming convention is that a command like `move'
 * is implemented as the method m_move (written in C) which calls
 * tkined_<type>_move to let tk take the appropriate action.
 *
 * Copyright (c) 1993, 1994
 *                    J. Schoenwaelder
 *                    TU Braunschweig, Germany
 *                    Institute for Operating Systems and Computer Networks
 *
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <netdb.h>

#include <pwd.h>

#ifdef DBMALLOC
#include <dbmalloc.h>
#endif

#include "tkined.h"
#include "Bitmaps/icon.bm"
#include "Bitmaps/machine.bm"
#include "Bitmaps/group.bm"

/* 
 * This interp is used whenever we receive a command from an application.
 * It would be nice if tk would give us an interpreter handle when
 * we are called via Tk_CreateFileHandler.
 */

static Tcl_Interp* the_interp;

/*
 * A general purpose sprintf buffer.
 */
static char buffer[1024];

/*
 * Our hashtable that maps object ids to the object structure.
 */

Tcl_HashTable ht_object;

/*
 * This variable gives the number of registered trace callbacks.
 */

int traceNum = 0;

/*
 * Forward declarations for procedures defined later in this file:
 */

static int tkined_retrieve       _ANSI_ARGS_((ClientData clientData,
					      Tcl_Interp* interp,
					      int argc, char** argv));
static int tkined_selection      _ANSI_ARGS_((ClientData clientData,
					      Tcl_Interp* interp,
					      int argc, char** argv));
static int tkined_mark_box       _ANSI_ARGS_((ClientData clientData,
					      Tcl_Interp* interp,
					      int argc, char** argv));
static int tkined_mark_points    _ANSI_ARGS_((ClientData clientData,
					      Tcl_Interp* interp,
					      int argc, char** argv));
static int tkined_object_command _ANSI_ARGS_((ClientData clientData,
					      Tcl_Interp* interp,
					      int argc, char** argv));
static int create                _ANSI_ARGS_((ClientData clientData,
					      Tcl_Interp* interp,
					      int argc, char** argv));
static void delete               _ANSI_ARGS_((ClientData clientData));


/* 
 * Find an object by its id.
 */

tkined_object* 
id_to_object (str)
     char *str;
{
    Tcl_HashEntry *ht_entry;

    if (str == NULL) return NULL;

    ht_entry = Tcl_FindHashEntry(&ht_object, str);
    if (ht_entry == NULL) return NULL;

    return (tkined_object *) Tcl_GetHashValue (ht_entry);
}

/*
 * This is  called to initialize this module and to register 
 * the new commands.
 */

int
Tkined_Init (interp)
    Tcl_Interp* interp;
{

#ifdef NCCICON 
    ncc_bitmaps (interp);
#endif

    Tk_DefineBitmap(interp, Tk_GetUid("icon"), icon_bits,
		    icon_width, icon_height);
    Tk_DefineBitmap(interp, Tk_GetUid("machine"), machine_bits,
		    machine_width, machine_height);
    Tk_DefineBitmap(interp, Tk_GetUid("group"), group_bits,
		    group_width, group_height);

    Tcl_CreateCommand (interp, "NODE", create, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "GROUP", create, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "NETWORK", create, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "LINK", create, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "TEXT", create, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "IMAGE", create, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "INTERPRETER", create, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "TOOL", create, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "LOG", create, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "tkined_retrieve", tkined_retrieve, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "tkined_selection", tkined_selection, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "tkined_mark_box", tkined_mark_box, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "tkined_mark_points", tkined_mark_points, 
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    Tcl_InitHashTable (&ht_object, TCL_STRING_KEYS);    

    /* this interp will be used in receive() */
    the_interp = interp;

    /* link the tcl variable tkined_debug to the C debug variable */
    return Tcl_LinkVar (interp, "tkined_debug", 
			(char *) &debug, TCL_LINK_BOOLEAN);
}

/*
 * This function gets called whenever a command is processed by
 * an object. It is used to write a trace to an interpreter which
 * can be used to drive client server interactions.
 *
 * Here is lots of room for optimizations: First, we should generate
 * the message once. Second, we should maintain our own list of trace
 * interpreters to avoid searching throught the whole hash table.
 */

void
trace (result, cmd)
    char *result;
    char *cmd;
{
    if (traceNum > 0) {
	Tcl_HashEntry *ht_entry;
	Tcl_HashSearch ht_search;

	ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
	while (ht_entry != NULL) {
	    tkined_object *obj = (tkined_object *) Tcl_GetHashValue (ht_entry);

	    if (obj->trace) {

		int len;
		Tcl_DString buf;

		Tcl_DStringInit (&buf);
		Tcl_DStringAppend (&buf, obj->member,-1);
		Tcl_DStringAppend (&buf, " ", -1);
		Tcl_DStringAppend (&buf, cmd, -1);
		if (result != NULL) {
		    Tcl_DStringAppend (&buf, " > ", -1);
		    Tcl_DStringAppendElement (&buf, result);
		}
		Tcl_DStringAppend (&buf, "\n", -1);

		len = Tcl_DStringLength (&buf);
		if (len != (write(obj->xv[0], Tcl_DStringValue (&buf), len))) {
		    fprintf (stderr, "trace: failed to write %s to %s\n",
			     Tcl_DStringValue (&buf), obj->id);
		};

		Tcl_DStringFree (&buf);

	    }
	    ht_entry = Tcl_NextHashEntry (&ht_search);
	}
    }
}

/*
 * This function gets called to write debug messages whenever a 
 * command is processed by an object.
 */

static void 
do_debug (object, interp, argc, argv, result)
    tkined_object *object;
    Tcl_Interp* interp;
    int argc;
    char** argv;
    char *result;
{
    int i;

    if (!debug) return;

    if (object != NULL) {
	printf ("#  %s ", object->id);
    } else {
	printf ("#  ");
    }
    for (i=0; i<argc; i++) {
	printf ("%s ", argv[i]);
    }
    if (result != NULL) {
	printf ("> %s\n", result);
    } else {
	printf ("\n");
    }
}

/*
 * Create a new object. The first argument is the type name e.g.
 * NODE and the secod argument is the keyword create. Further arguments
 * are type specific. Consult the tkined man page for more details
 * about this. The create functions initializes a new object and registers
 * a command that can be used later to query or modify the object state.
 */

static int 
create (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp* interp;
    int argc;
    char** argv;
{
    tkined_object *object;
    Tcl_HashEntry *ht_entry;
    int flag;

    if (argc<2) {
	interp->result = "wrong # args";
        return TCL_ERROR;
    }

    /* This is for backward compatibility. Please do not delete TOOL
       objects by name anymore. This will be removed in future versions. */

    if (argc > 2 
	&& (strcmp (argv[0], "TOOL") == 0) 
	&& (strcmp(argv[1], "delete") == 0)) {
	fprintf (stderr,"** ined TOOL delete is no longer a valid command!\n");
	return TCL_OK;
    }

    object = (tkined_object *) xmalloc(sizeof(tkined_object));

    object->type = string_to_type (argv[0]);
    if (object->type==TKINED_NONE) {
	free (object);
	interp->result = "illegal type";
        return TCL_ERROR;
    }

    object->id = xstrdup("");
    object->name = xstrdup("");
    object->address = xstrdup("");
    object->oid = 0;
    object->x = object->y = 0;
    object->icon = xstrdup("");
    object->font = xstrdup("");
    object->color = xstrdup("");
    object->label = xstrdup("");
    object->text = xstrdup("");
    object->canvas = xstrdup("");
    object->items = xstrdup("");    
    object->parent = xstrdup("");
    object->size = xstrdup("");
    object->member = xstrdup("");
    object->links = xstrdup("");
    object->ida = xstrdup("");
    object->idb = xstrdup("");
    object->done = 0;    
    object->queue = 0;
    object->trace = 0;
    object->selected = 0;
    object->collapsed = 0;
    object->stripchart = 0;
    object->barchart = 0;

    /* Call the create member function to do type 
       specific initialization */

    tkined_object_command ((ClientData) object, interp, argc, argv);

    /* throw the new object in the hash table */

    ht_entry = Tcl_CreateHashEntry (&ht_object, object->id, &flag);
    if (flag == 0) {
	sprintf (interp->result, "Tcl_CreateHashEntry failed for %s",
		 object->id);
	return TCL_ERROR;
    } else {
	Tcl_SetHashValue (ht_entry, (ClientData) object);
    }
    
    /* Create a tcl command for the new object */

    Tcl_CreateCommand(interp, object->id, tkined_object_command, 
		      (ClientData) object, delete);    

    /* Call the tk procedure to do initialization stuff */

    Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		 "_create ", object->id, (char *) NULL);

    do_debug ((tkined_object *) NULL, interp, argc, argv, object->id);

    interp->result = object->id;
    return TCL_OK;
}


/*
 * Delete an object. Free everything alloced before destroying
 * the structure.
 */

static void 
delete (clientData)
     ClientData clientData;
{
    Tcl_HashEntry *ht_entry;
    tkined_object *object = (tkined_object *) clientData;

    ht_entry = Tcl_FindHashEntry (&ht_object, object->id);
    if (ht_entry != NULL) Tcl_DeleteHashEntry (ht_entry);

    free (object->id);
    free (object->name);
    free (object->address);
    free (object->icon);
    free (object->font);
    free (object->color);
    free (object->label);
    free (object->text);
    free (object->canvas);
    free (object->items);
    free (object->parent);
    free (object->size);
    free (object->member);
    free (object->links);
    free (object->ida);
    free (object->idb);
    
    if (object->type == TKINED_INTERPRETER) {
	Tk_DeleteFileHandler (object->xv[0]);
	close (object->xv[0]);
	Tcl_DStringFree (&object->buffer);

	/* Hack against zombies - I dont know why Tcl_DetachPids not works. */
	kill (object->pid, SIGTERM);
	Tcl_ReapDetachedProcs();
    }
    
    free ((char*) object);
}

/*
 * Execute a received ined command. The interpreter object
 * is given by object. Commands that affect the user interface
 * are processed by calling the appropriate tk procedure. Commands
 * that change an object status are handled by calling the appropriate
 * member function.
 */

static int 
execute (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    static char *cmds[] = {
	"page",
        "status",

	"acknowledge",
        "confirm",
        "request",
	"browse",
	"list",
	0
    };

    char *cmd;
    char *tmp;
    int result;
    int i;
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;
    tkined_object *newobject;
    int update = 1;
    
    if (argc == 0) return TCL_RETURN;
    
    /* ignore everything not starting with the key word 'ined' */
    
    if (strcmp(argv[0], "ined") != 0) return TCL_CONTINUE;
    
    /* check for the -noupdate option */
    
    if (argc > 2 && (strcmp (argv[1], "-noupdate") == 0)) {
        update = 0;
	for ( argc--, i = 1; i < argc; i++) {
	    argv[i] = argv[i+1];
	}
    }
    
    /* process 'ined queue <n>' messages */
    
    if ((argc > 2) && (strcmp(argv[1], "queue") == 0)) {
        if (Tcl_GetInt (interp, argv[2], &result) == TCL_OK) {
	    object->queue = result;
	    sprintf (buffer, "tkined_%s_queue %s %d", 
		     type_to_string (object->type), 
		     object->id, object->queue);
	    Tcl_Eval (interp, buffer);
	}
	return TCL_RETURN;
    }
    
    /* Check for one of the commands given in the cmds table */
    
    for ( i = 0; cmds[i] != NULL; i++ ) {
        if (strcmp(cmds[i], argv[1]) == 0) {
	    cmd = Tcl_Merge (argc-2, argv+2);
	    result = Tcl_VarEval (interp, "tkined_", cmds[i], " ",
				  object->canvas, " ", cmd, (char *) NULL);
	    if (result == TCL_OK) {
		do_debug (object, interp, argc, argv, (char *) NULL);
	    }
	    free (cmd);
	    return result;
	}
    }

    /* Catch the retrieve command with no arguments. */

    if (argc == 2 && (strcmp (argv[1], "retrieve") == 0)) {
        Tcl_DString ds;
	Tcl_DStringInit (&ds);
	ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
	Tcl_ResetResult (interp);
	while (ht_entry != NULL) {
	    newobject = (tkined_object *) Tcl_GetHashValue (ht_entry);
	    if (strcmp (newobject->canvas, object->canvas) == 0) {
		result = Tcl_VarEval (interp, newobject->id, " retrieve", 
				      (char *) NULL);
		if (result == TCL_OK) {
		    Tcl_DStringAppendElement (&ds, interp->result);
		}
	    }
	    ht_entry = Tcl_NextHashEntry (&ht_search);
	}
	Tcl_DStringResult (interp, &ds);
	Tcl_DStringFree (&ds);
	return TCL_OK;
    }

    /* Catch the size command with no arguments */

    if (argc == 2 && (strcmp (argv[1], "size") == 0)) {
	result = Tcl_VarEval (interp, "tkined_size ",
			      object->canvas, (char *) NULL);
	if (result == TCL_OK) {
	    do_debug (object, interp, argc, argv, interp->result);
	}
	return result;
    }

    /* Catch the select command with no arguments */

    if (argc == 2 && (strcmp (argv[1], "select") == 0)) {
	result = Tcl_VarEval (interp, "tkined_selection ",
			      object->canvas, (char *) NULL);
	if (result == TCL_OK) {
	    do_debug (object, interp, argc, argv, interp->result);
	}
	return result;
    }

    /* Catch the trace command. I use the member pointer to
       save to name of the callback procedure. */

    if (argc == 3 && (strcmp (argv[1], "trace") == 0)) {
	free (object->member); object->member = xstrdup (argv[2]);
	if (strlen(argv[2]) > 0) {
	    traceNum++;
	    object->trace = 1;
	} else {
	    traceNum--;
	    object->trace = 0;
	}
	return TCL_OK;
    }
    
    /* Map the external syntax to internal syntax */
    
    if (cmds[i] == NULL && argc > 2) {
	tmp = argv[1]; argv[1] = argv[2]; argv[2] = tmp;
	cmd = Tcl_Merge (argc-1, argv+1);
	result = Tcl_Eval (interp, cmd);
	ht_entry = Tcl_FindHashEntry (&ht_object, argv[1]);
	if (result != TCL_OK && ht_entry == NULL) {
	    
	    if (debug) {
		fprintf (stderr, "** no such object: %s\n", argv[1]);
		fprintf (stderr, "** while doing: %s\n", cmd);
	    }
	    
	    /* For now we ignore invalid object references.
	       This may change if needed. */
	    
	    interp->result = "";
	    free (cmd);
	    return TCL_OK;
	}
	
	/* Check for create commands. Make sure that new created
	   objects get their canvas attribute setup to the canvas
	   of the interpreter object. */
	
	if (result == TCL_OK) {
	    if (strcmp(argv[2], "create") == 0) {
		ht_entry = Tcl_FindHashEntry (&ht_object, interp->result);
		if (ht_entry != NULL) {
		    newobject = (tkined_object *) Tcl_GetHashValue (ht_entry);
		    if (newobject->type == TKINED_TOOL) {
			STRCOPY (newobject->links, object->id);
		    }
		    if (newobject->type == TKINED_LOG) {
			STRCOPY (newobject->links, object->id);
		    }
		    m_canvas (interp, newobject, 1, &object->canvas);
		    interp->result = newobject->id;
		}
	    } else {
		do_debug (object, interp, argc-1, argv+1, (char *) NULL);
	    }
	}
	free (cmd);
    }
    
    if (update) {
	tmp = xstrdup (interp->result);
        Tcl_Eval (interp, "update idletask");
	Tcl_SetResult (interp, tmp, TCL_VOLATILE);
	free (tmp);	
    }

    return TCL_OK;
}

/*
 * This function gets called whenver an interpreter gets readable.
 * All available characters are read and added to the command buffer.
 * Complete commands are evaluated. 
 */

void 
receive(clientData, mask)
     ClientData clientData;    /* Describes command to execute. */
     int        mask;          /* Not used */
{
#define BUFFER_SIZE 4000
    tkined_object *object = (tkined_object*) clientData;
    char input[BUFFER_SIZE+1];
    char *cmd;
    char *line;
    char *p;
    int count, len, result;
    int argc;
    char **argv;
    Tcl_DString buf;
    
    if (!object->done) {
	Tcl_DStringFree (&object->buffer);
    }

    count = read(object->xv[0], input, BUFFER_SIZE);

    if (count <= 0) {
	if (!object->done) {   
	    m_interpreter_delete (the_interp, object, 0, (char **) NULL);
	    return;
	} else {
	    input[0] = 0;
	}
    } else {
	input[count] = 0;
    }
    cmd = Tcl_DStringAppend (&object->buffer, input, count);
    if (! Tcl_CommandComplete(cmd)) {
	object->done = 1;
        return;
    }
    object->done = 0;

    if (debug) {
	fprintf (stderr, ">> %s", cmd);
	fflush(stderr);
    }

    /* split the buffer with newlines and process each piece */
    for (line = cmd, p = cmd; *p != 0; p++) {
	if (*p!='\n') continue;

	*p = 0;
	Tcl_SplitList (the_interp, line, &argc, &argv);

	result = execute (the_interp, object, argc, argv);

	free ((char*) argv);

	/* write back an acknowledge and the result */

        Tcl_DStringInit (&buf);
	
	switch (result) {
            case TCL_OK:    Tcl_DStringAppend (&buf, "ined ok ", -1); break;
	    case TCL_ERROR: Tcl_DStringAppend (&buf, "ined error ", -1); break;
	    case TCL_CONTINUE: printf ("%s\n", line);
	}
	    
	if (Tcl_DStringLength (&buf) > 0) {
	    Tcl_DStringAppend (&buf, the_interp->result, -1);
	    Tcl_DStringAppend (&buf, "\n", 1);
	    
	    len = Tcl_DStringLength (&buf);
	    if (len != write (object->xv[0], Tcl_DStringValue (&buf), len)) {
		sprintf (the_interp->result, "write to %s failed",
			 object->id);
		return;
	    }
	    
	    if (debug) {
		fprintf (stderr, "<< %s", Tcl_DStringValue (&buf));
		fflush (stderr);
	    }
	}

	Tcl_DStringFree (&buf);

	line = p+1;
	cmd = p+1;
    }
}

/*
 * Retrieve the list of all known object ids. The optional argument
 * can be used to retrieve all object ids of a given canvas.
 */

static int 
tkined_retrieve (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp* interp;
    int argc;
    char** argv;
{
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;
    tkined_object *object;

    ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
    while (ht_entry != NULL) {
	object = (tkined_object *) Tcl_GetHashValue (ht_entry);
	if (argc == 1 
	    || (argc == 2 && (strcmp (argv[1], object->canvas) == 0))) {
	    Tcl_AppendElement (interp, object->id);
	}
	ht_entry = Tcl_NextHashEntry (&ht_search);
    }

    return TCL_OK;
}

/*
 * Retrieve the list of all selected object ids. The argument
 * is used to define a given canvas.
 */

static int 
tkined_selection (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp* interp;
    int argc;
    char** argv;
{
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;
    tkined_object *object;

    if (argc != 2) {
	interp->result = "wrong # of args: tkined_selection canvas";
	return TCL_ERROR;
    }

    ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
    while (ht_entry != NULL) {
	object = (tkined_object *) Tcl_GetHashValue (ht_entry);
	if (object->selected && strcmp (argv[1], object->canvas) == 0) {
	    Tcl_AppendElement (interp, object->id);
	}
	ht_entry = Tcl_NextHashEntry (&ht_search);
    }

    return TCL_OK;
}

/*
 * This one is just here because we need some simple arithmetics
 * that are very slow in TCL. So I rewrote this proc in C. It just
 * takes a canvas item and puts selection marks around it.
 */

static void 
mark_one_item (interp, x, y, canvas, item)
    Tcl_Interp* interp;
    double x,y;
    char *canvas;
    char *item;
{
    sprintf (buffer, 
	     "%s create rectangle %f %f %f %f -fill black -tags mark%s", 
	     canvas, x-1, y-1, x+1, y+1, item); 

    Tcl_Eval (interp, buffer);
}

static int 
tkined_mark_points (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp* interp;
    int argc;
    char** argv;
{
    int ret;
    int largc;
    char **largv;
    int i, n;
    double *x;
    double *y;

    if (argc != 3) {
	interp->result ="wrong number of args";
	return TCL_ERROR;
    }

    ret = Tcl_VarEval (interp, argv[1], " coords ", argv[2], (char *) NULL);
    if (ret != TCL_OK) return ret;

    Tcl_SplitList (interp, interp->result, &largc, &largv);

    x = (double *) xmalloc (largc * sizeof(double));
    y = (double *) xmalloc (largc * sizeof(double));

    if (x == NULL || y == NULL) {
	free ((char*) largv);
	interp->result = "setting selection marks failed";
	return TCL_ERROR;
    }

    for (n = 0, i = 0; i < largc; i++) {
	if ((i%2) == 0) {
	    Tcl_GetDouble (interp, largv[i], &x[n]);
	} else {
	    Tcl_GetDouble (interp, largv[i], &y[n]);
	    n++;
	}
    }

    if (x[0] > x[1]) x[0] += 4;
    if (x[0] < x[1]) x[0] -= 4;
    if (y[0] > y[1]) y[0] += 4;
    if (y[0] < y[1]) y[0] -= 4;

    if (x[n-1] > x[n-2]) x[n-1] += 4;
    if (x[n-1] < x[n-2]) x[n-1] -= 4;
    if (y[n-1] > y[n-2]) y[n-1] += 4;
    if (y[n-1] < y[n-2]) y[n-1] -= 4;

    mark_one_item (interp, x[0], y[0], argv[1], argv[2]);
    mark_one_item (interp, x[n-1], y[n-1], argv[1], argv[2]);

    free ((char *) x);
    free ((char *) y);
    free ((char*) largv);

    return TCL_OK;
}

static int 
tkined_mark_box (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp* interp;
    int argc;
    char** argv;
{
    int ret;
    int largc;
    char **largv;
    double x1, x2, y1, y2, xm, ym;

    if (argc != 3) {
	interp->result ="wrong number of args";
	return TCL_ERROR;
    }

    ret = Tcl_VarEval (interp, argv[1], " bbox ", argv[2], (char *) NULL);
    if (ret != TCL_OK) return ret;

    Tcl_SplitList (interp, interp->result, &largc, &largv);

    Tcl_GetDouble (interp, largv[0], &x1);
    Tcl_GetDouble (interp, largv[1], &y1);
    Tcl_GetDouble (interp, largv[2], &x2);
    Tcl_GetDouble (interp, largv[3], &y2);
    x1 -= 2; x2 += 2; y1 -= 2; y2 += 2;
    xm = (x1+x2)/2;
    ym = (y1+y2)/2;

    mark_one_item (interp, x1, y1, argv[1], argv[2]);
    mark_one_item (interp, x1, y2, argv[1], argv[2]);
    mark_one_item (interp, x2, y1, argv[1], argv[2]);
    mark_one_item (interp, x2, y2, argv[1], argv[2]);
    if (abs (x1-x2) > 15) {
	mark_one_item (interp, xm, y1, argv[1], argv[2]);
	mark_one_item (interp, xm, y2, argv[1], argv[2]);
    }
    if (abs (y1-y2) > 15) {
	mark_one_item (interp, x1, ym, argv[1], argv[2]);
	mark_one_item (interp, x2, ym, argv[1], argv[2]);
    }

    free ((char*) largv);

    return TCL_OK;
}

/*
 * All methods are dispatched using the following table. Depending
 * on the type and the name of an object, we choose the function
 * to call. The type TKINED_ALL matches any type.
 */

struct tkined_method {
    int type;
    char* cmd;
    int (*fnx)(); /* (Tcl_Interp*, tkined_object*, int, char**) */
};

static struct tkined_method tkined_methods[] = {

        { TKINED_NODE,        "create",      m_node_create },
        { TKINED_GROUP,       "create",      m_group_create },
        { TKINED_NETWORK,     "create",      m_network_create },
        { TKINED_LINK,        "create",      m_link_create },
        { TKINED_TEXT,        "create",      m_text_create },
        { TKINED_IMAGE,       "create",      m_image_create },
        { TKINED_INTERPRETER, "create",      m_interpreter_create },
        { TKINED_TOOL,        "create",      m_tool_create },
        { TKINED_LOG,         "create",      m_log_create },

        { TKINED_NODE,        "retrieve",    m_node_retrieve },
        { TKINED_GROUP,       "retrieve",    m_group_retrieve },
        { TKINED_NETWORK,     "retrieve",    m_network_retrieve },
        { TKINED_LINK,        "retrieve",    m_link_retrieve },
        { TKINED_TEXT,        "retrieve",    m_text_retrieve },
        { TKINED_IMAGE,       "retrieve",    m_image_retrieve },
        { TKINED_INTERPRETER, "retrieve",    m_interpreter_retrieve },
        { TKINED_TOOL,        "retrieve",    m_tool_retrieve },
        { TKINED_LOG,         "retrieve",    m_log_retrieve },

        { TKINED_ALL,         "type",        m_type },

        { TKINED_ALL,         "id",          m_id },

	{ TKINED_ALL,         "name",        m_name },

	{ TKINED_ALL,         "canvas",      m_canvas },

	{ TKINED_ALL,         "items",       m_items },

	{ TKINED_NODE,        "address",     m_address },
	{ TKINED_NETWORK,     "address",     m_address },	
	{ TKINED_LOG,         "address",     m_address },

        { TKINED_NODE,        "oid",         m_oid },
        { TKINED_GROUP,       "oid",         m_oid },
        { TKINED_NETWORK,     "oid",         m_oid },
        { TKINED_LINK,        "oid",         m_oid },
        { TKINED_TEXT,        "oid",         m_oid },
        { TKINED_IMAGE,       "oid",         m_oid },

        { TKINED_NODE,        "select",      m_select },
        { TKINED_GROUP,       "select",      m_select },
        { TKINED_NETWORK,     "select",      m_select },
        { TKINED_LINK,        "select",      m_select },
        { TKINED_TEXT,        "select",      m_select },
        { TKINED_IMAGE,       "select",      m_select },

        { TKINED_NODE,        "unselect",    m_unselect },
        { TKINED_GROUP,       "unselect",    m_unselect },
        { TKINED_NETWORK,     "unselect",    m_unselect },
        { TKINED_LINK,        "unselect",    m_unselect },
        { TKINED_TEXT,        "unselect",    m_unselect },
        { TKINED_IMAGE,       "unselect",    m_unselect },

        { TKINED_NODE,        "selected",    m_selected },
        { TKINED_GROUP,       "selected",    m_selected },
        { TKINED_NETWORK,     "selected",    m_selected },
        { TKINED_LINK,        "selected",    m_selected },
        { TKINED_TEXT,        "selected",    m_selected },
        { TKINED_IMAGE,       "selected",    m_selected },

	{ TKINED_NODE,        "icon",        m_icon },
        { TKINED_GROUP,       "icon",        m_icon },
        { TKINED_NETWORK,     "icon",        m_icon },
        { TKINED_LOG,         "icon",        m_icon },

	{ TKINED_NODE,        "label",       m_label },
	{ TKINED_GROUP,       "label",       m_label },
        { TKINED_NETWORK,     "label",       m_label },

	{ TKINED_NODE,        "font",        m_font },
	{ TKINED_GROUP,       "font",        m_font },
        { TKINED_NETWORK,     "font",        m_font },
	{ TKINED_TEXT,        "font",        m_font },

	{ TKINED_NODE,        "color",       m_color },
	{ TKINED_GROUP,       "color",       m_color },
        { TKINED_NETWORK,     "color",       m_color },
        { TKINED_LINK,        "color",       m_color },
	{ TKINED_TEXT,        "color",       m_color },
        { TKINED_IMAGE,       "color",       m_color },

        { TKINED_NODE,        "move",        m_move },
        { TKINED_GROUP,       "move",        m_group_move },
        { TKINED_NETWORK,     "move",        m_move },
        { TKINED_LINK,        "move",        m_link_move },
        { TKINED_TEXT,        "move",        m_move },
        { TKINED_IMAGE,       "move",        m_move },

	{ TKINED_NODE,        "raise",       m_raise },
	{ TKINED_GROUP,       "raise",       m_raise },
	{ TKINED_NETWORK,     "raise",       m_raise },
	{ TKINED_LINK,        "raise",       m_raise },
	{ TKINED_TEXT,        "raise",       m_raise },

	{ TKINED_NODE,        "lower",       m_lower },
	{ TKINED_GROUP,       "lower",       m_lower },
	{ TKINED_NETWORK,     "lower",       m_lower },
	{ TKINED_LINK,        "lower",       m_lower },
	{ TKINED_TEXT,        "lower",       m_lower },
	{ TKINED_IMAGE,       "lower",       m_lower },

	{ TKINED_NODE,        "size",        m_size },
	{ TKINED_GROUP,       "size",        m_size },
	{ TKINED_NETWORK,     "size",        m_size },
	{ TKINED_LINK,        "size",        m_size },
	{ TKINED_TEXT,        "size",        m_size },
	{ TKINED_IMAGE,       "size",        m_size },

	{ TKINED_NODE,        "dump",        m_node_dump },
	{ TKINED_GROUP,       "dump",        m_group_dump },
	{ TKINED_NETWORK,     "dump",        m_network_dump },
	{ TKINED_LINK,        "dump",        m_link_dump },
	{ TKINED_TEXT,        "dump",        m_text_dump },
	{ TKINED_IMAGE,       "dump",        m_image_dump },

	{ TKINED_LINK,        "ida",         m_ida },
	{ TKINED_LINK,        "idb",         m_idb },

	{ TKINED_TEXT,        "text",        m_text },

	{ TKINED_LOG,         "append",      m_append },
	{ TKINED_LOG,         "clear",       m_clear },
	{ TKINED_LOG,         "interpreter", m_interpreter },

	{ TKINED_NODE,        "stripchart",  m_stripchart },
	{ TKINED_NODE,        "barchart",    m_barchart },

	{ TKINED_GROUP,       "member",      m_member },
	{ TKINED_GROUP,       "collapse",    m_collapse },
	{ TKINED_GROUP,       "expand",      m_expand },
	{ TKINED_GROUP,       "collapsed",   m_collapsed },

	{ TKINED_NODE,        "links",       m_links },
	{ TKINED_NETWORK,     "links",       m_links },

	{ TKINED_LINK,        "points",      m_member },
	{ TKINED_NETWORK,     "points",      m_member },

	{ TKINED_NETWORK,     "labelxy",     m_network_labelxy },

        { TKINED_TOOL,        "interpreter", m_interpreter },

        { TKINED_INTERPRETER, "send",        m_send },

	{ TKINED_NODE,        "delete",      m_linked_delete },
	{ TKINED_NETWORK,     "delete",      m_linked_delete },
	{ TKINED_LINK,        "delete",      m_link_delete },
	{ TKINED_GROUP,       "delete",      m_group_delete },
	{ TKINED_INTERPRETER, "delete",      m_interpreter_delete },
        { TKINED_ALL,         "delete",      m_delete },

        { 0, 0, 0 }
    };

/*
 * Process a method of a tkined_object. Check the table for an
 * appropriate entry and call the desired function. We have an 
 * error if no entry matches.
 */

static int
tkined_object_command (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp* interp;
    int argc;
    char** argv;
{
    tkined_object *object = (tkined_object *) clientData;
    struct tkined_method *ds;

    if (argc<2) {
	interp->result = "wrong # args";
	return TCL_ERROR;
    }

    for (ds=tkined_methods; ds->cmd; ds++) {
	int res;

	if (ds->type != TKINED_ALL && object->type != ds->type) continue;
	if (strcmp(argv[1], ds->cmd) != 0) continue;

	res = (ds->fnx)(interp, object, argc-2, argv+2);
	if (res == TCL_OK) {
	    if (strcmp(argv[1], "create") != 0) {
		do_debug (object, interp, argc-1, argv+1, interp->result);
	    }
	}
	return res;
    }

    sprintf (interp->result, "%s can not %s", object->id, argv[1]);

    return TCL_ERROR;

}
