#! /usr/local/bin/tclnm
##
## Simple SNMP troubleshooting utilities for tkined.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

source [info library]/init.tcl
set auto_path "/usr/local/lib/ined [info library]"
set auto_path "/usr/local/lib/tkined [info library]"
if [info exists env(HOME)] {
    set auto_path "$env(HOME)/.tkined $env(HOME)/.ined $auto_path"
}
if [info exists env(INED_PATH)] {
    set auto_path "$env(INED_PATH) $auto_path"
}
if [info exists env(TKINED_PATH)] {
    set auto_path "$env(TKINED_PATH) $auto_path"
}

foreach dir $auto_path {
    if {[file readable $dir/ined.tcl]} {
        source $dir/ined.tcl
    }
}

if [catch {ined size}] {
    puts stderr "failed to load the INED interface module"
    exit 1
}

set snmp_community "public"
set snmp_timeout "5000000"
set snmp_retries "3"

##
## Test if we are at home.
##

proc local {} {
    if {[catch {exec nslook [exec hostname]} ip]} {
	return 0
    }
    return [string match "134.169.34.*" $ip]
}

##
## Write a report to a log object.
##

proc writeln {{txt ""}} {
    static log

    if {![info exists log]} {
        set log [ined -noupdate create LOG]
        ined -noupdate name $log "SNMP Trouble Log"
    }

    if {[ined -noupdate retrieve $log] == ""} {
        set log [ined -noupdate create LOG]
        ined -noupdate name $log "SNMP Trouble Log"
    }

    if {$txt == ""} { 
	ined append $log "" 
    } else {
	foreach line [split $txt "\n"] {
	    ined append $log [string trim $line "\r"]
	}
    }
}


## Get the IP Address of a node. Query the name server, if the
## address attribute is not set to something that looks like a
## valid IP address.

proc Get_IP_Address {node} {
    if {[ined_type $node]=="NODE"} {
        set host [ined_name $node]
        set ip [ined_address $node]
        if {[regexp "^\[0-9\]+\.\[0-9\]+\.\[0-9\]+\.\[0-9\]+$" $ip]>0} {
            return $ip
        }
        if {[catch {nslook $host} ip]==0} {
            return [lindex $ip 0]
        }
    }
    return ""
}

##
## Open a snmp session and initialize it to the defaults stored in
## the global variables snmp_community, snmp_timeout, snmp_retries.
##

proc snmp_open {ip} {
    global snmp_community snmp_timeout snmp_retries
    set s [snmp open $ip]
    snmp default $s community $snmp_community
    snmp default $s timeout $snmp_timeout
    snmp default $s retry $snmp_retries
    return $s
}

##
## Dump a hierarchy of the MIB and return the result as a list 
## of strings.
##

proc "dump" {ip path} {
    set s [snmp_open $ip]
    set op ""
    catch {
	snmp getbulk $s {$path} x {
	    set oid [lindex $x 0]
	    set typ [lindex $x 1]
	    set val [lindex $x 2]
	    # get the oid from the MIB
	    set l [mib oid $oid]
	    set p [lindex $l 0]
	    if {$p == $op} {
		writeln "\tvalue\[[lindex $l 1]\] =\t$val"
		continue;
	    }
	    set op $p
	    writeln
	    writeln [lindex $l 0]
	    writeln "$oid $typ"
	    writeln "\tvalue\[[lindex $l 1]\] =\t$val"
	}
    } error
    if {$error != ""} {
	ined acknowledge "Dumping MIB tree failed for $ip."
    }
    catch {snmp close $s}
}

##
## Try to figure out if the devices responds to snmp requests.
##

proc "ping" {list} {
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
	    set s [snmp_open $ip]
	    if [catch {snmp getnext $s .1.0}] {
		ined acknowledge \
		    "$host \[$ip\] does not respond to SNMP requests."
	    } else {
		ined acknowledge "$host \[$ip\] answers to SNMP requests."
	    }
	    catch {snmp close $s}
	}
    }
}

##
## Get the system information of MIB II.
##

proc "system information" {list} {
    foreach comp $list {
	if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
	    if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
	}
	writeln "System information for $host:"
	set s [snmp_open $ip]
	if {![catch {snmp get $s -l -v system.sysName.0} res]} {
	    writeln "Name:        [join $res]"
	}
	if {![catch {snmp get $s -l -v system.sysDescr.0} res]} {
	    writeln "Description: [join $res]"
	}
	if {![catch {snmp get $s -l -v system.sysUpTime.0} res]} {
	    writeln "Uptime:      [join $res]"
	}
	if {![catch {snmp get $s -l -v system.sysContact.0} res]} {
	    writeln "Contact:     [join $res]"
	}
	if {![catch {snmp get $s -l -v system.sysLocation.0} res]} {
	    writeln "Location:    [join $res]"
	}
	catch {snmp close $s}
	writeln
    }
}

##
## Get the list of interfaces and their status.
##

proc "interface status" {list} {
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
            set s [snmp_open $ip]
            snmp default $s prefix \
                [snmp default $s prefix].interfaces.ifTable.ifEntry
            writeln "Interface status of $host:"
            writeln "ifDescr            ifAdminStatus ifOperStatus (ifType)"
            snmp getbulk $s -l -v {
                ifDescr
		ifAdminStatus
		ifOperStatus
		ifType
            } x {
		set aa "format {%-16s %13s %12s   (%s)} [join $x]"
                writeln [eval "$aa"]
            }
	    catch {snmp close $s}
	    writeln
        }
    }
}

##
## We should check which interfaces are up and running.
##

proc "interface statistic" {list} {
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
            set s [snmp_open $ip]
            snmp default $s prefix \
                [snmp default $s prefix].interfaces.ifTable.ifEntry
            writeln "Interface statistics of $host:"
            writeln "ifDescr            ifInOctets ifInUcastPkts  ifOutOctets ifOutUcastPkts"
            snmp getbulk $s -l -v {
                ifDescr
		ifInOctets
		ifInUcastPkts
		ifOutOctets
		ifOutUcastPkts
            } x {
		set aa "format {%-16s %12s %14s %12s %14s} [join $x]"
                writeln [eval "$aa"]
            }
	    catch {snmp close $s}
	    writeln
        }
    }
}

##
## Show the routing tables of the agents.
##

proc "routing table" {list} {
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
	    if {$ip==""} {
		ined acknowledge "Can not lookup IP Address for $host."
		continue;
	    }
	    set s [snmp_open $ip]
	    snmp default $s prefix \
	        [snmp default $s prefix].ip.ipRouteTable.ipRouteEntry
	    writeln "routing table of $host:"
	    writeln "ipRouteDest      ipRouteNextHop ipRouteType ipRouteProto"
	    snmp getbulk $s -l -v {
		ipRouteDest
		ipRouteNextHop
		ipRouteType
		ipRouteProto
	    } x {
		set aa "format {%-16s %-16s %7s %10s} [join $x]"
                writeln [eval "$aa"]
	    }
	    catch {snmp close $s}
	    writeln
	}
    }
}

##
## Show the routing tables of the agents.
##

proc "arp table" {list} {
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
	    if {$ip==""} {
		ined acknowledge "Can not lookup IP Address for $host."
		continue;
	    }
	    set s [snmp_open $ip]
	    snmp default $s prefix \
	        [snmp default $s prefix].ip.ipNetToMediaTable.ipNetToMediaEntry
	    writeln "ARP table of $host:"
	    writeln "ipNetToMediaPhysAddress ipNetToMediaNetAddress"
	    snmp getbulk $s -l -v {
		ipNetToMediaPhysAddress
		ipNetToMediaNetAddress
	    } x {
		set aa "format {%-22s %-22s} [join $x]"
		writeln [eval "$aa"]
	    }
	    catch {snmp close $s}
	    writeln
	}
    }
}

##
## Show the tcp connection tables of the agents.
##

proc "tcp connections" {list} {
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
            set s [snmp_open $ip]
            snmp default $s prefix \
                [snmp default $s prefix].tcp.tcpConnTable.tcpConnEntry
	    writeln "TCP connections of $host:"
            writeln "State            LocalAddress LocalPort RemoteAddress RemotePort"
            snmp getbulk $s -l -v {
                tcpConnState
		tcpConnLocalAddress
		tcpConnLocalPort
		tcpConnRemAddress
		tcpConnRemPort
            } x {
		set aa [eval format "{%-12s %16s %6s %16s %6s }" [join $x]]
		writeln $aa
            }
	    catch {snmp close $s}
	    writeln
        }
    }
}

##
## Dump a complete hierarchy. The user may choose the hierarchy
## before we start our action. We store the last selected hierarchy
## in a static variable called dump_mib_tree_path.
##

proc "dump MIB" {list} {

    static dump_mib_tree_path

    if ![info exists dump_mib_tree_path] {
        set dump_mib_tree_path ".iso.org.dod.internet.mgmt.mib-2"
    }

    set path [ined request "Dump MIB Tree" \
	      "{ {MIB path:} $dump_mib_tree_path}"]
    if {$path==""} return
    set dump_mib_tree_path [lindex $path 0]

    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
	    writeln "MIB dump for $host starting at $path:"
	    dump $ip $path
	    writeln
        }
    }
}

##
## Set the parameters (community, timeout, retry) for snmp requests.
##

proc "set parameter" {list} {
    global snmp_community snmp_timeout snmp_retries

    set result [ined request "SNMP Parameter" \
		"{ {Community:} $snmp_community } \
                 { {Timeout \[us\]:} $snmp_timeout } \
                 { {Retries:} $snmp_retries scale 1 8} " ]

    if {$result==""} return

    set snmp_community [lindex $result 0]
    set snmp_timeout   [lindex $result 1]
    set snmp_retries   [lindex $result 2]
}

##
## Display some help about this tool.
##

proc "help SNMP Trouble" {list} {
    ined browse "Help about SNMP Trouble" {
	"ping:" 
	"    Test is a device responds to SNMP requests." 
	"" 
	"system information:" 
	"    Display the information of the system group." 
	"" 
	"interface status:" 
	"" 
	"interface statistic:" 
	"" 
	"routing table:" 
	"" 
	"arp table:" 
	"" 
	"tcp connections:" 
	"" 
	"dump MIB:" 
	"" 
	"set parameter:" 
	"    This dialog allows you to set SNMP parameters like retries, " 
	"    timeouts and community names. " 
    }
}

##
## Delete the tools created by this interpreter.
##

proc "delete SNMP Trouble" {list} {
    global tools
    foreach id $tools { ined delete $id }
    exit
}

if {![local]} {
    ined acknowledge "SNMP is still under construction. Use with care!!"
}

set tools [ ined create TOOL "SNMP Trouble" "ping" "system information" "" \
	    "interface status" "interface statistic" "" \
	    "routing table" "arp table" "tcp connections" "" \
	    "dump MIB" "" \
	    "set parameter" "" \
	    "help SNMP Trouble" "delete SNMP Trouble" ]

ined_main_loop
