/* 
 * barchart.c --
 *
 *	This file implements barchart items for canvas widgets.
 *      It's based on tkRectOval.c and tkCanvPoly.c from the 
 *      tk distribution by John Ousterhout.
 *
 *      Written by G. Hueske (hueske@ibr.cs.tu-bs.de)
 *                 TU Braunschweig, Germany
 */

#include <stdio.h>
#include <tkInt.h>
#include <tkCanvas.h>
#include <tkConfig.h>


/*
 * The structure below defines the record for each barchart item.
 */

typedef struct BarchartItem {
    Tk_Item header;             /* Generic stuff that's the same for all
                                 * types.  MUST BE FIRST IN STRUCTURE. */
    int numPoints;              /* Number of points in polygon (always >= 3).
                                 * Polygon is always closed. */
    double *valuePtr;           /* Pointer to malloc-ed array containing 
				 * the original values. */
    double *coordPtr;           /* Pointer to malloc-ed array containing
                                 * the computed coordinates. */
    int scale;                  /* Actual scale of the barchart. */
    int scalevalue;             /* Every scalevalue we scale up. */
    double bbox[4];             /* Coordinates of bounding box for rectangle
                                 * (x1, y1, x2, y2). */
    XColor *color;              /* Color for barchart bars. */
    GC fillGC;                  /* Graphics context for filling barchart. */
    int autocolor;              /* Boolean that indicates whether autocolormode
				 * is on or off. */
    XColor *rectColor;          /* Background color of the barchart. */
    GC rectGC;                  /* Graphics context of the rectangel around 
				 * the barchart. */
    XColor *borderlineColor;    /* Color of the rectangular outline 
				 * around the barchart. */
    int borderlineWidth;        /* Width of the rectangular outline around 
				 * the barchart. */
    GC borderlineGC;            /* Graphics context of the outline around 
				 * the barchart. */
    XColor *barlineColor;       /* Color of the outline of the bars. */
    int barlineWidth;           /* Width of the outline of the bars. */
    GC barlineGC;               /* Graphics context for outline of
				 * the bars. */
    int scalelineWidth;         /* Width of scaleline. */
    XColor *scalelineColor;     /* Color of scaleline. */
    int scalelineStyle;         /* Style of scaleline, 0 or negativ means 
				 * LineSolid, each other positive value 
				 * means the length of dashes using 
				 * LineOnOffDash. */
    GC scalelineGC;             /* Graphics context for scaleline. */
} BarchartItem;


/*
 * Information used for parsing configuration specs:
 */

static Tk_ConfigSpec configSpecs[] = {    
    {TK_CONFIG_INT, "-scalevalue", (char *) NULL, (char *) NULL,
        "100", Tk_Offset(BarchartItem, scalevalue),
        TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
        (char *) NULL, Tk_Offset(BarchartItem, color), TK_CONFIG_NULL_OK}, 
    {TK_CONFIG_BOOLEAN, "-autocolor", (char *) NULL, (char *) NULL,
        "0", Tk_Offset(BarchartItem, autocolor), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-background", (char *) NULL, (char *) NULL,
        (char *) NULL, Tk_Offset(BarchartItem, rectColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-borderline", (char *) NULL, (char *) NULL,
        "black", Tk_Offset(BarchartItem, borderlineColor), 
        TK_CONFIG_NULL_OK},   
    {TK_CONFIG_PIXELS, "-borderlinewidth", (char *) NULL, (char *) NULL,
        "1", Tk_Offset(BarchartItem, borderlineWidth), 
	TK_CONFIG_DONT_SET_DEFAULT},  
    {TK_CONFIG_COLOR, "-barline", (char *) NULL, (char *) NULL,
        "black", Tk_Offset(BarchartItem, barlineColor), 
        TK_CONFIG_NULL_OK},   
    {TK_CONFIG_PIXELS, "-barlinewidth", (char *) NULL, (char *) NULL,
        "1", Tk_Offset(BarchartItem, borderlineWidth), 
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-scaleline", (char *) NULL, (char *) NULL,
        "black", Tk_Offset(BarchartItem, scalelineColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_PIXELS, "-scalelinewidth", (char *) NULL, (char *) NULL,
        "1", Tk_Offset(BarchartItem, scalelineWidth), 
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_INT, "-scalelinestyle", (char *) NULL, (char *) NULL,
        "4", Tk_Offset(BarchartItem, scalelineStyle),
        TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
        (char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
        (char *) NULL, 0, 0}
};


/*
 * Prototypes for procedures defined in this file:
 */

static void             ComputeBarchartBbox _ANSI_ARGS_((
                            Tk_Canvas *canvasPtr, BarchartItem *barPtr));
static int              ConfigureBarchart _ANSI_ARGS_((
                            Tk_Canvas *canvasPtr, Tk_Item *itemPtr, int argc,
                            char **argv, int flags));
static int              CreateBarchart _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            struct Tk_Item *itemPtr, int argc, char **argv));
static void             DeleteBarchart _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr));
static void             DisplayBarchart _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, Drawable dst));
static int              BarchartCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, int argc, char **argv));
static int              BarchartToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, double *rectPtr));
static double           BarchartToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, double *pointPtr));
static int              BarchartToPostscript _ANSI_ARGS_((
                            Tk_Canvas *canvasPtr, Tk_Item *itemPtr, 
                            Tk_PostscriptInfo *psInfoPtr));
static void             ScaleBarchart _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, double originX, double originY,
                            double scaleX, double scaleY));
static void             TranslateBarchart _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * The structures below defines the barchart item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_ItemType TkBarchartType = {
    "barchart",                       /* name */
    sizeof(BarchartItem),             /* itemSize */
    CreateBarchart,                   /* createProc */
    configSpecs,                      /* configSpecs */
    ConfigureBarchart,                /* configureProc */
    BarchartCoords,                   /* coordProc */
    DeleteBarchart,                   /* deleteProc */
    DisplayBarchart,                  /* displayProc */
    0,                                /* alwaysRedraw */
    BarchartToPoint,                  /* pointProc */
    BarchartToArea,                   /* areaProc */
    BarchartToPostscript,             /* postscriptProc */
    ScaleBarchart,                    /* scaleProc */
    TranslateBarchart,                /* translateProc */
    (Tk_ItemIndexProc *) NULL,        /* indexProc */
    (Tk_ItemCursorProc *) NULL,       /* icursorProc */
    (Tk_ItemSelectionProc *) NULL,    /* selectionProc */
    (Tk_ItemInsertProc *) NULL,       /* insertProc */
    (Tk_ItemDCharsProc *) NULL,       /* dTextProc */
    (Tk_ItemType *) NULL              /* nextPtr */
};

/*
 * The definition below determines how large are static arrays
 * used to hold spline points (splines larger than this have to
 * have their arrays malloc-ed).
 */

#define MAX_STATIC_POINTS 50




/*
 *--------------------------------------------------------------
 *
 * CreateBarchart --
 *
 *	This procedure is invoked to create a new barchart item in
 *	a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	canvasPtr->interp->result;  in this case itemPtr is
 *	left uninitialized, so it can be safely freed by the
 *	caller.
 *
 * Side effects:
 *	A new barchart item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateBarchart(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing barchart. */
{
    register BarchartItem *barPtr = (BarchartItem *) itemPtr;
    int i;
    
    if (argc < 4) {
	Tcl_AppendResult(canvasPtr->interp, "wrong # args:  should be \"",
		Tk_PathName(canvasPtr->tkwin), "\" create ",
		itemPtr->typePtr->name,
		" x1 y1 x2 y2 ?y3 y4 ...? ?options?", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    barPtr->numPoints = 0;
    barPtr->coordPtr = NULL;
    barPtr->valuePtr = NULL;
    barPtr->scale = 1;
    barPtr->scalevalue = 100;
    barPtr->color = NULL;
    barPtr->fillGC = None;
    barPtr->autocolor = 0;
    barPtr->rectColor = NULL;
    barPtr->rectGC = None;
    barPtr->borderlineColor = NULL;
    barPtr->borderlineWidth = 1;
    barPtr->borderlineGC = None;
    barPtr->barlineColor = NULL;
    barPtr->barlineWidth = 1;
    barPtr->barlineGC = None;
    barPtr->scalelineWidth = 1;
    barPtr->scalelineColor = NULL;
    barPtr->scalelineStyle = 4;
    barPtr->scalelineGC = None;


    /*
     * Count the number of points and then parse them into a point
     * array.  Leading arguments are assumed to be points if they
     * start with a digit or a minus sign followed by a digit.
     */

    for (i = 4; i < argc; i++) {
	if ((!isdigit(UCHAR(argv[i][0]))) &&
		((argv[i][0] != '-') || (!isdigit(UCHAR(argv[i][1]))))) {
	    break;
	}
    }

    if (BarchartCoords(canvasPtr, itemPtr, i, argv) != TCL_OK) {
	goto error;
    }

    if (ConfigureBarchart(canvasPtr, itemPtr, argc-i, argv+i, 0) == TCL_OK) {
	return TCL_OK;
    }

    error:
    DeleteBarchart(canvasPtr, itemPtr);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * BarchartCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on barcharts.  See the user documentation for details
 *	on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets canvasPtr->interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
BarchartCoords(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    register BarchartItem *barPtr = (BarchartItem *) itemPtr;
    char buffer[TCL_DOUBLE_SPACE];
    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE];
    char c2[TCL_DOUBLE_SPACE], c3[TCL_DOUBLE_SPACE];
    int i, y1, y2, numPoints, height;
    double max = 0.0;

    if (argc == 0) {
        Tcl_PrintDouble(canvasPtr->interp, barPtr->bbox[0], c0);
        Tcl_PrintDouble(canvasPtr->interp, barPtr->bbox[1], c1);
        Tcl_PrintDouble(canvasPtr->interp, barPtr->bbox[2], c2);
        Tcl_PrintDouble(canvasPtr->interp, barPtr->bbox[3], c3);
        Tcl_AppendResult(canvasPtr->interp, c0, " ", c1, " ",
                c2, " ", c3, (char *) NULL);
        for (i = 0; i < barPtr->numPoints; i++) {
            Tcl_PrintDouble(canvasPtr->interp, barPtr->valuePtr[i], buffer);
            Tcl_AppendElement(canvasPtr->interp, buffer);
        }
    } else if (argc < 4) {
	Tcl_AppendResult(canvasPtr->interp,
		"too few coordinates for barchart:  must have at least 4",
		(char *) NULL);
	return TCL_ERROR;
    }  else {
	if ((TkGetCanvasCoord(canvasPtr, argv[0],
		    &barPtr->bbox[0]) != TCL_OK)
		|| (TkGetCanvasCoord(canvasPtr, argv[1],
		    &barPtr->bbox[1]) != TCL_OK)
		|| (TkGetCanvasCoord(canvasPtr, argv[2],
		     &barPtr->bbox[2]) != TCL_OK)
		|| (TkGetCanvasCoord(canvasPtr, argv[3],
                     &barPtr->bbox[3]) != TCL_OK)) {
	    return TCL_ERROR;
	}

	numPoints = argc - 4;

	if (barPtr->numPoints < numPoints) {
	    if (barPtr->valuePtr != NULL) {
		ckfree((char *) barPtr->valuePtr);
	    }
	    barPtr->valuePtr = (double *) ckalloc((unsigned)
		  (sizeof(double) * numPoints));
	    if (barPtr->coordPtr != NULL) {
		ckfree((char *) barPtr->coordPtr);
	    }
	    barPtr->coordPtr = (double *) ckalloc((unsigned)
		  (sizeof(double) * numPoints));
	}
	if (numPoints > 0) {
	    barPtr->numPoints = numPoints;
	
	    for (i = 0; i < numPoints; i++) {
		if (TkGetCanvasCoord(canvasPtr, argv[i+4], 
				     &barPtr->valuePtr[i]) != TCL_OK) {
		    return TCL_ERROR;
		}
	    }
	}

        /* 
         * Now we have to check if the actual scale is right. 
         */
        
        /* Find the maximal y-value of the polygon. */

        for (i = 0; i < numPoints; i++) {
            if (barPtr->valuePtr[i] > max) {
                    max = barPtr->valuePtr[i];
		}
        }
        
        /* Find the right scale. */

        barPtr->scale = 1;
        if (max/barPtr->scale > barPtr->scalevalue) {
            while (max/barPtr->scale > barPtr->scalevalue) {
                barPtr->scale++;
            }
        } 
   
        y1 = SCREEN_Y(canvasPtr, barPtr->bbox[1]);
        y2 = SCREEN_Y(canvasPtr, barPtr->bbox[3]);
        height = (y2-y1-1>0) ? y2-y1-1 : y1-y2-1;

	for (i = 0; i < barPtr->numPoints; i++) {
	    barPtr->coordPtr[i] = barPtr->valuePtr[i]/barPtr->scalevalue * 
		    height/barPtr->scale;
	}
    }
	    
    ComputeBarchartBbox(canvasPtr, barPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureBarchart --
 *
 *      This procedure is invoked to configure various aspects
 *      of a barchart item such as its background color.
 *
 * Results:
 *      A standard Tcl result code.  If an error occurs, then
 *      an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *      Configuration information, such as colors, may be set 
 *      for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureBarchart(canvasPtr, itemPtr, argc, argv, flags)
    Tk_Canvas *canvasPtr;       /* Canvas containing itemPtr. */
    Tk_Item *itemPtr;           /* Barchart item to reconfigure. */
    int argc;                   /* Number of elements in argv.  */
    char **argv;                /* Arguments describing things to configure. */
    int flags;                  /* Flags to pass to Tk_ConfigureWidget. */
{
    register BarchartItem *barPtr = (BarchartItem *) itemPtr;
    XGCValues gcValues;
    GC newGC;
    unsigned long mask;
 
    if (Tk_ConfigureWidget(canvasPtr->interp, canvasPtr->tkwin,
	    configSpecs, argc, argv, (char *) barPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    if (barPtr->color == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = barPtr->color->pixel;
	mask = GCForeground;
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (barPtr->fillGC != None) {
	Tk_FreeGC(canvasPtr->display, barPtr->fillGC);
    }
    barPtr->fillGC = newGC;

    if (barPtr->rectColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = barPtr->rectColor->pixel;
	mask = GCForeground;
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (barPtr->rectGC != None) {
	Tk_FreeGC(canvasPtr->display, barPtr->rectGC);
    }
    barPtr->rectGC = newGC;

    if (barPtr->borderlineColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = barPtr->borderlineColor->pixel;
	
	if (barPtr->borderlineWidth < 0) {
	    barPtr->borderlineWidth = 1;
	}
	gcValues.line_width = barPtr->borderlineWidth;	
	gcValues.cap_style = CapProjecting;
	mask = GCForeground|GCCapStyle|GCLineWidth;
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (barPtr->borderlineGC != None) {
	Tk_FreeGC(canvasPtr->display, barPtr->borderlineGC);
    }
    barPtr->borderlineGC = newGC;

    if (barPtr->barlineColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = barPtr->barlineColor->pixel;
	gcValues.cap_style = CapProjecting;
	if (barPtr->barlineWidth < 0) {
	    barPtr->barlineWidth = 1;
	}
	gcValues.line_width = barPtr->barlineWidth;
	mask = GCForeground|GCCapStyle|GCLineWidth;
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (barPtr->barlineGC != None) {
	Tk_FreeGC(canvasPtr->display, barPtr->barlineGC);
    }
    barPtr->barlineGC = newGC;

    if (barPtr->scalelineColor == NULL) {
        newGC = None;
    } else {
        gcValues.foreground = barPtr->scalelineColor->pixel;
        if (barPtr->scalelineWidth < 0) {
            barPtr->scalelineWidth = 1;
        }
        gcValues.line_width = barPtr->scalelineWidth;
        mask = GCForeground|GCLineWidth|GCLineStyle;
        if (barPtr->scalelineStyle < 0) {
            barPtr->scalelineStyle = 0;
        }
	    gcValues.line_style = LineSolid;
	if (barPtr->scalelineStyle > 0) { 
            gcValues.line_style = LineOnOffDash;
            gcValues.dashes = (char) barPtr->scalelineStyle;
            mask |= GCDashList;
        }
        newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (barPtr->scalelineGC != None) {
        Tk_FreeGC(canvasPtr->display, barPtr->scalelineGC);
    }
    barPtr->scalelineGC = newGC;
    
    ComputeBarchartBbox(canvasPtr, barPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteBarchart --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a barchart item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteBarchart(canvasPtr, itemPtr)
    Tk_Canvas *canvasPtr;		/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
{
    register BarchartItem *barPtr = (BarchartItem *) itemPtr;

    if (barPtr->coordPtr != NULL) {
	ckfree((char *) barPtr->coordPtr);
    }
    if (barPtr->valuePtr != NULL) {
	ckfree((char *) barPtr->valuePtr);
    }
    if (barPtr->color != NULL) {
	Tk_FreeColor(barPtr->color);
    }
    if (barPtr->fillGC != None) {
	Tk_FreeGC(canvasPtr->display, barPtr->fillGC);
    }
    if (barPtr->rectColor != NULL) {
	Tk_FreeColor(barPtr->rectColor);
    }
    if (barPtr->rectGC != None) {
	Tk_FreeGC(canvasPtr->display, barPtr->rectGC);
    }
    if (barPtr->borderlineColor != NULL) {
	Tk_FreeColor(barPtr->borderlineColor);
    }
    if (barPtr->borderlineGC != None) {
	Tk_FreeGC(canvasPtr->display, barPtr->borderlineGC);
    }
    if (barPtr->barlineColor != NULL) {
	Tk_FreeColor(barPtr->barlineColor);
    }
    if (barPtr->barlineGC != None) {
        Tk_FreeGC(canvasPtr->display, barPtr->barlineGC);
    }
    if (barPtr->scalelineColor != NULL) {
        Tk_FreeColor(barPtr->scalelineColor);
    }
    if (barPtr->scalelineGC != None) {
        Tk_FreeGC(canvasPtr->display, barPtr->scalelineGC);
    }

}

/*
 *--------------------------------------------------------------
 *
 * ComputeBarchartBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a barchart.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
ComputeBarchartBbox(canvasPtr, barPtr)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    BarchartItem *barPtr;		/* Item whose bbox is to be
					 * recomputed. */
{
    int bloat;

    /*
     * Make sure that the first coordinates are the lowest ones.
     */

    if (barPtr->bbox[1] > barPtr->bbox[3]) {
        double tmp;
        tmp = barPtr->bbox[3];
        barPtr->bbox[3] = barPtr->bbox[1];
        barPtr->bbox[1] = tmp;
    }
    if (barPtr->bbox[0] > barPtr->bbox[2]) {
        double tmp;
        tmp = barPtr->bbox[2];
        barPtr->bbox[2] = barPtr->bbox[0];
        barPtr->bbox[0] = tmp;
    }

    bloat = (barPtr->borderlineWidth+1)/2+1;
    barPtr->header.x1 = barPtr->bbox[0] - bloat;
    barPtr->header.y1 = barPtr->bbox[1] - bloat;
    barPtr->header.x2 = barPtr->bbox[2] + bloat;
    barPtr->header.y2 = barPtr->bbox[3] + bloat;

}


/*
 *--------------------------------------------------------------
 *
 * DisplayBarchart --
 *
 *	This procedure is invoked to draw a barchart item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvasPtr.
 *
 *--------------------------------------------------------------
 */

static void
DisplayBarchart(canvasPtr, itemPtr, drawable)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
{
    register BarchartItem *barPtr = (BarchartItem *) itemPtr;
    Display *display = Tk_Display(canvasPtr->tkwin);
    int i, x1, y1, x2, y2;
    XColor *newcolor;
    GC autocolorGC;
    char colorname[8];
    int red, green, blue;
    unsigned int width, height;

    red = green = blue = 175;
     
    /*
     * Compute the screen coordinates of the bounding box for the item.
     * Make sure that the bbox is at least one pixel large, since some
     * X servers will die if it isn't.
     */

    x1 = SCREEN_X(canvasPtr, barPtr->bbox[0]);
    y1 = SCREEN_Y(canvasPtr, barPtr->bbox[1]);
    x2 = SCREEN_X(canvasPtr, barPtr->bbox[2]);
    y2 = SCREEN_Y(canvasPtr, barPtr->bbox[3]);
    if (x2 <= x1) {
	x2 = x1+1;
    }
    if (y2 <= y1) {
	y2 = y1+1;
    }

  /* Display the barchart bounding box */

    if (barPtr->rectGC != None) {
	XFillRectangle(display, drawable, barPtr->rectGC,
		       x1-barPtr->borderlineWidth/2, 
		       y1-barPtr->borderlineWidth/2,
		       (unsigned int) (x2-x1+barPtr->borderlineWidth-1),
		       (unsigned int) (y2-y1+barPtr->borderlineWidth-1));
    }

    if (barPtr->borderlineGC != None) {
	XDrawRectangle(display, drawable, barPtr->borderlineGC,
		       x1, 
		       y1,
		       (unsigned int) (x2-x1-1),
		       (unsigned int) (y2-y1-1));
    }

    if (barPtr->numPoints > 0) {
	width = (x2-x1-1)/barPtr->numPoints;
    }

    autocolorGC = XCreateGC(display, drawable, 0, NULL);
    for (i = 0; i < barPtr->numPoints; i++) {
	if ((height = barPtr->coordPtr[i]) > 0) {
	    if (barPtr->fillGC != None) {
		XFillRectangle(display, drawable, barPtr->fillGC,
			       x1+i*width, y2-height-1, 
			       width, height);
	    }
	    
	    if (barPtr->autocolor > 0 && barPtr->fillGC == None) {
		
		switch(i%3) {
	            case 0:
		            red = i*80/barPtr->numPoints+175;
		            break;
	            case 1:
		            green = i*80/barPtr->numPoints+175;
		            break;
	            case 2:
		            blue = i*80/barPtr->numPoints+175;
		    }
		sprintf(colorname, "#%2x%2x%2x", red, green, blue);
		
		newcolor = Tk_GetColor(canvasPtr->interp, canvasPtr->tkwin,
				       None, Tk_GetUid(colorname));
		
		XSetForeground(display, autocolorGC, newcolor->pixel);
		
		XFillRectangle(display, drawable, autocolorGC,
			       x1+i*width, y2-height-1, 
			       width, height);

		/* need to free it somewhere :-( */
		/*if (newcolor != NULL) {
		  Tk_FreeColor(newcolor); 
		  }*/
	    }

	    if (barPtr->barlineGC != None) {
		XDrawRectangle(display, drawable, barPtr->barlineGC,
			       x1+i*width, y2-height-1, 
			       width, height);
	    }
	}
    }
    if (autocolorGC != None) {
	XFreeGC(display, autocolorGC);
    }
       
  
    /* Display the scaleline. */

    if (barPtr->scalelineGC != None){
        if (barPtr->scale > 1) {
            for (i = 1; i < barPtr->scale; i++) {
                XDrawLine(display, drawable, barPtr->scalelineGC,
                          x1+1, y2-1-i*(y2-y1-2)/barPtr->scale, 
                          x2-1, y2-1-i*(y2-y1-2)/barPtr->scale);
            }
        }
    }

}

/*
 *--------------------------------------------------------------
 *
 * BarchartToPoint --
 *
 *	Computes the distance from a given point to a given
 *	barchart, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the barchart.  If the
 *	point isn't inside the barchart then the return value is the
 *	distance from the point to the barchart.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
BarchartToPoint(canvasPtr, itemPtr, pointPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    register BarchartItem *barPtr = (BarchartItem *) itemPtr;
    double xDiff, yDiff, x1, y1, x2, y2, inc;

    /*
     * Generate a new larger rectangle that includes the border
     * width, if there is one.
     */

    x1 = barPtr->bbox[0];
    y1 = barPtr->bbox[1];
    x2 = barPtr->bbox[2];
    y2 = barPtr->bbox[3];
    if (barPtr->borderlineGC != None) {
	inc = barPtr->borderlineWidth/2.0;
	x1 -= inc;
	y1 -= inc;
	x2 += inc;
	y2 += inc;
    }

    /*
     * If the point is inside the rectangle, the distance is 0
     */

    if ((pointPtr[0] >= x1) && (pointPtr[0] < x2)
	        && (pointPtr[1] >= y1) && (pointPtr[1] < y2)) {
	return 0.0;
    }

    /*
     * Point is outside rectangle.
     */

    if (pointPtr[0] < x1) {
	xDiff = x1 - pointPtr[0];
    } else if (pointPtr[0] > x2)  {
	xDiff = pointPtr[0] - x2;
    } else {
	xDiff = 0;
    }

    if (pointPtr[1] < y1) {
	yDiff = y1 - pointPtr[1];
    } else if (pointPtr[1] > y2)  {
	yDiff = pointPtr[1] - y2;
    } else {
	yDiff = 0;
    }

    return hypot(xDiff, yDiff);
}

/*
 *--------------------------------------------------------------
 *
 * BarchartToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangular area.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by areaPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
BarchartToArea(canvasPtr, itemPtr, areaPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against barchart. */
    double *areaPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
    register BarchartItem *barPtr = (BarchartItem *) itemPtr;
    double halfWidth;

    halfWidth = barPtr->borderlineWidth/2.0;
    if (barPtr->borderlineGC == None) {
	halfWidth = 0.0;
    }

    if ((areaPtr[2] <= (barPtr->bbox[0] - halfWidth))
	    || (areaPtr[0] >= (barPtr->bbox[2] + halfWidth))
	    || (areaPtr[3] <= (barPtr->bbox[1] - halfWidth))
	    || (areaPtr[1] >= (barPtr->bbox[3] + halfWidth))) {
	return -1;
    }
    if ((areaPtr[0] <= (barPtr->bbox[0] - halfWidth))
	    && (areaPtr[1] <= (barPtr->bbox[1] - halfWidth))
	    && (areaPtr[2] >= (barPtr->bbox[2] + halfWidth))
	    && (areaPtr[3] >= (barPtr->bbox[3] + halfWidth))) {
	return 1;
    }
    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleBarchart --
 *
 *	This procedure is invoked to rescale a barchart item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The barchart referred to by itemPtr is rescaled so that the
 *	following transformation is applied to all point
 *	coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

static void
ScaleBarchart(canvasPtr, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas *canvasPtr;		/* Canvas containing barchart. */
    Tk_Item *itemPtr;			/* Barchart to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    BarchartItem *barPtr = (BarchartItem *) itemPtr;
    register double *coordPtr;
    int i;

    barPtr->bbox[0] = originX + scaleX*(barPtr->bbox[0] - originX);
    barPtr->bbox[1] = originY + scaleY*(barPtr->bbox[1] - originY);
    barPtr->bbox[2] = originX + scaleX*(barPtr->bbox[2] - originX);
    barPtr->bbox[3] = originY + scaleY*(barPtr->bbox[3] - originY);

    for (i = 0, coordPtr = barPtr->coordPtr; i < barPtr->numPoints;
	    i++, coordPtr += 1) {
	*coordPtr = originY + scaleY*(*coordPtr - originY);
    }
    ComputeBarchartBbox(canvasPtr, barPtr);
}

/*
 *--------------------------------------------------------------
 *
 * TranslateBarchart --
 *
 *	This procedure is called to move a barchart by a given
 *	amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the barchart is offset by (xDelta, yDelta),
 *	and the bounding box is updated in the generic part of the
 *	item structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateBarchart(canvasPtr, itemPtr, deltaX, deltaY)
    Tk_Canvas *canvasPtr;		/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    register BarchartItem *barPtr = (BarchartItem *) itemPtr;

    barPtr->bbox[0] += deltaX;
    barPtr->bbox[1] += deltaY;
    barPtr->bbox[2] += deltaX;
    barPtr->bbox[3] += deltaY;

    ComputeBarchartBbox(canvasPtr, barPtr);
}

/*
 *--------------------------------------------------------------
 *
 * BarchartToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	barchart items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in canvasPtr->interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	item is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
BarchartToPostscript(canvasPtr, itemPtr, psInfoPtr)
    Tk_Canvas *canvasPtr;		/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    Tk_PostscriptInfo *psInfoPtr;	/* Information about the Postscript;
					 * must be passed back to Postscript
					 * utility procedures. */
{
    register BarchartItem *barPtr = (BarchartItem *) itemPtr;
    char pathCmd[500], string[100];
    double x1, y1, y2;
    int dash, i;
    char colorname[8];
    XColor *newcolor;
    int red, green, blue;

    red = green = blue = 175;
    y1 = TkCanvPsY(psInfoPtr, barPtr->bbox[1]);
    y2 = TkCanvPsY(psInfoPtr, barPtr->bbox[3]);

    /*
     * Generate a string that creates a path for the barchart.
     * This is the only part of the procedure's code that is type-
     * specific.
     */


    sprintf(pathCmd, "%.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto %.15g 0 rlineto closepath\n",
                barPtr->bbox[0], y1,
                barPtr->bbox[2]-barPtr->bbox[0], y2-y1,
                barPtr->bbox[0]-barPtr->bbox[2]);
    
    /*
     * First draw the filled area of the barchart.
     */

    if (barPtr->rectColor != NULL) {
        Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
        if (TkCanvPsColor(canvasPtr, psInfoPtr, barPtr->rectColor)
                != TCL_OK) {
            return TCL_ERROR;
        }
        Tcl_AppendResult(canvasPtr->interp, "fill\n", (char *) NULL);
    }

    /*
     * Now draw the outline, if there is one.
     */

    if (barPtr->borderlineColor != NULL) {
        Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
        sprintf(string, "%d setlinewidth", barPtr->borderlineWidth);
        Tcl_AppendResult(canvasPtr->interp, string,
                " 0 setlinejoin 2 setlinecap\n", (char *) NULL);
        if (TkCanvPsColor(canvasPtr, psInfoPtr, barPtr->borderlineColor)
                != TCL_OK) {
            return TCL_ERROR;
        }
        Tcl_AppendResult(canvasPtr->interp, "stroke\n", (char *) NULL);
    }

    /*
     * Create a path for the bars.
     */

    for (i = 0; i < barPtr->numPoints; i++) {
	x1 = barPtr->bbox[0]+i*(barPtr->bbox[2]-barPtr->bbox[0]-1)
		/barPtr->numPoints;
	sprintf(pathCmd, "%.15g %.15g moveto 0 %.15g rlineto %.15g 0 rlineto 0 %.15g rlineto closepath\n", x1, y2, barPtr->coordPtr[i],
		(barPtr->bbox[2]-barPtr->bbox[0]-1)/barPtr->numPoints,
		-barPtr->coordPtr[i]);

	/* 
	 * Draw the inside of the bars, if necessary.
	 */

	/* Autocolor mode. */

	if (barPtr->autocolor > 0 && barPtr->fillGC == None) {
	    Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);

	    switch(i%3) {
	        case 0:
		        red = i*80/barPtr->numPoints+175;
		        break;
	        case 1:
		        green = i*80/barPtr->numPoints+175;
		        break;
	        case 2:
		        blue = i*80/barPtr->numPoints+175;
	    }
	    sprintf(colorname, "#%2x%2x%2x", red, green, blue);

	    newcolor = Tk_GetColor(canvasPtr->interp, canvasPtr->tkwin,
				   None, Tk_GetUid(colorname));

	    if (TkCanvPsColor(canvasPtr, psInfoPtr, newcolor) != TCL_OK) {
                return TCL_ERROR;
            }

	    Tk_FreeColor(newcolor);
	    Tcl_AppendResult(canvasPtr->interp, "fill\n", (char *) NULL);
	}

	/* Single color mode. */

	if (barPtr->color != NULL) {
	    Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
	    if (TkCanvPsColor(canvasPtr, psInfoPtr, barPtr->color)
                != TCL_OK) {
		return TCL_ERROR;
	    }
	    Tcl_AppendResult(canvasPtr->interp, "fill\n", (char *) NULL);
	}

	/*
	 * Draw the outline of the bars, if there is one.
	 */

	if (barPtr->barlineColor != NULL) {
	    Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
	    sprintf(string, "%d setlinewidth", barPtr->barlineWidth);
	    Tcl_AppendResult(canvasPtr->interp, string,
			     " 0 setlinejoin 2 setlinecap\n", (char *) NULL);
	    if (TkCanvPsColor(canvasPtr, psInfoPtr, barPtr->barlineColor)
                != TCL_OK) {
		return TCL_ERROR;
	    }
	    Tcl_AppendResult(canvasPtr->interp, "stroke\n", (char *) NULL);
	}
    }

    /*
     * Finally draw the scaleline, if there is one.
     */

    /*
     * Generate a path for the scaleline.
     */

    for (i = 1; i < barPtr->scale; i++) {
        y1 = TkCanvPsY(psInfoPtr, (barPtr->bbox[3]-1-i*
                 (barPtr->bbox[3]-barPtr->bbox[1]-2)/barPtr->scale));

        sprintf(pathCmd, "%.15g %.15g moveto %.15g 0 rlineto\n",
                barPtr->bbox[0]+barPtr->scalelineWidth/2, y1, 
                barPtr->bbox[2]-barPtr->bbox[0]-
                    barPtr->scalelineWidth);
    
        if (barPtr->scalelineColor != NULL) {
            Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
            sprintf(pathCmd, "%d setlinewidth", barPtr->scalelineWidth);
            if (barPtr->scalelineStyle > 0) {
                dash = barPtr->scalelineStyle;
                sprintf(string,
                        " 0 setlinejoin 2 setlinecap [%d] 0 setdash\n", 
			dash);
            } else {
                sprintf(string, " 0 setlinejoin 2 setlinecap [] 0 setdash\n");
            }
            Tcl_AppendResult(canvasPtr->interp, pathCmd, string, 
                             (char *) NULL);
            
            if (TkCanvPsColor(canvasPtr, psInfoPtr, barPtr->scalelineColor)
                != TCL_OK) {
                return TCL_ERROR;
            }
            Tcl_AppendResult(canvasPtr->interp, "stroke\n", (char *) NULL);
        }
    }
    return TCL_OK;
}
