##
## This is the really simple tcl interface to the ined editor.
## It is written in pure tcl so that every tcl program can talk
## to the [tk]ined editor.
##
## This file also contains some utility procedures shared between
## the INED editor and tcl interpreters that extend the editor.
##
## Copyright (c) 1993, 1994
##                    S. Schoek, J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

##
## The global variables below contain the actual jobid, command and
## intervall whenever a job is executed. This allows a job to easily
## access its parameters. Note: You must call ined_modify_job to change
## the parameter. Changing the global variable has no effect.
##

set ined_curjob_id  ""
set ined_curjob_cmd ""
set ined_curjob_itv ""
set ined_curjob_cnt ""

##
## Some global constants and variables needed by this interface module.
## These are for internal use and should not be modified outside of the
## module.
##

set ined_jobs(0,0)    0

set COMMAND      0  
set TIME         1  
set REM_TIME     2
set JOB_INTERVAL 3
set JOB_STATUS   4

set ined_maxjobs 0

set ined_main_time ""

##
## This nice procedure allows us to use static variables. It was
## posted on the net by Karl Lehenbauer.
##

proc static {args} {
    set procName [lindex [info level [expr [info level]-1]] 0]
    foreach varName $args {
        uplevel 1 "upvar #0 {$procName:$varName} $varName"
    }
}

##
## Test if the argument is a real number.
##

proc isnumber {num} {
    return [expr [catch {expr $num+1}] ? 0 : 1]
}

######################################################################

##
## Search for jobs that should be done now.
##

proc ined_search_jobs {} {

    global ined_jobs ined_maxjobs
    global COMMAND REM_TIME TIME JOB_STATUS JOB_INTERVAL

    set result ""
    
    for {set i 0} {$i < $ined_maxjobs} {incr i} {
        set res ""

	##  search suspended jobs with remaining time less or equal 0
	##  increment intervall and kill job if necessary

        if {($ined_jobs($i,$REM_TIME) <= 0) 
	    && ($ined_jobs($i,$JOB_STATUS) != "suspend")} {

            lappend res [list $i $ined_jobs($i,$COMMAND) \
			 $ined_jobs($i,$TIME) $ined_jobs($i,$JOB_INTERVAL)]
	    set ined_jobs($i,$REM_TIME) $ined_jobs($i,$TIME)

            if {$ined_jobs($i,$JOB_INTERVAL) >= 1} {
		incr ined_jobs($i,$JOB_INTERVAL) -1

		## if job i was executed k-times ($iteration) kill it from 
		## list of jobs

                if {$ined_jobs($i,$JOB_INTERVAL) == 0} {
                    ined_kill_job $i
                }
            }
        }
        if {$res != ""} {
            lappend result $res
        }
    }

    return $result
}

##
## Compute the time to wait until the next job need attention.
##

proc ined_search_wait_time {} {
    
    global ined_jobs ined_maxjobs
    global REM_TIME JOB_STATUS

    set wait_time ""
    
    # start with the remaining time of the first
    # job that is not suspended.

    for {set i 0} {$i < $ined_maxjobs} {incr i} {
        if {$ined_jobs($i,$JOB_STATUS) != "suspend"} {
            set wait_time $ined_jobs($i,$REM_TIME)
            break
        }
    }

    # check if we have a job with a remaining time
    # less than the one we already have.

    if {$wait_time != ""} {
        for {set j $i} {$j < $ined_maxjobs} {incr j} {
            if {($ined_jobs($j,$REM_TIME) < $wait_time) 
		&& ($ined_jobs($j,$JOB_STATUS) != "suspend") } {
		    set wait_time $ined_jobs($j,$REM_TIME)
            }
        }

	# Zero or negative waiting times confuse our scheduler

	if {$wait_time <= 0} { set wait_time 0.001 }

    }
    
    return $wait_time
}


##
## Subtract the given seconds from REM_TIME of all active jobs.
##

proc ined_sub_wait_time {secs} {
    
    global ined_jobs ined_maxjobs
    global REM_TIME
    global JOB_STATUS
    
    for {set i 0} {$i < $ined_maxjobs} {incr i} {
        if {$ined_jobs($i,$JOB_STATUS) != "suspend"} {
	    set ined_jobs($i,$REM_TIME) [expr {$ined_jobs($i,$REM_TIME)-$secs}]
        }
    }
}

##
## Schedule the jobs with a remaining time <= 0. Once all jobs have
## been done, wait for the next schedule time or the next input
## coming from the tkined editor.
##

proc ined_schedule {} {
    
    global errorInfo
    global ined_jobs ined_maxjobs
    global ined_queue
    global COMMAND
    global ined_main_time
    global ined_curjob_id
    global ined_curjob_cmd
    global ined_curjob_itv
    global ined_curjob_cnt

    set wait_time ""

    if {$ined_maxjobs != 0} {

	# set ined_main_time for the first time
        
        if {$ined_main_time == ""} {
            set ined_main_time [getclock]
        }

	## compute the time the interpreter was in ined_main_loop
        ## and set the new ined_main_time, this time we use it to
        ## compute the time the interpreter was in  proc schedule

        set res_time [expr {[getclock] - $ined_main_time}]
        set ined_main_time [getclock]

	## subtract the res_time from the time of all not suspended jobs
        
        ined_sub_wait_time $res_time

	## search all not suspended jobs with remaining time less or equal 0
        
        set list [ined_search_jobs]
        
        ## execute all jobs in list 

	foreach elem $list {
	    set elem [join $elem]
	    set ined_curjob_id  [lindex $elem 0]
	    set ined_curjob_cmd [lindex $elem 1]
	    set ined_curjob_itv [lindex $elem 2]
	    set ined_curjob_cnt [lindex $elem 3]
	    if {[catch [lindex $elem 1]]} {
		puts stderr $errorInfo
		flush stderr
	    }
        }

	## compute the time the interpreter needed to execute all jobs of list
        ## and subtract the time from all not suspended jobs
        
        set res_time [expr {[getclock] - $ined_main_time}]
        ined_sub_wait_time $res_time
        
        ## search the next time we must wait before we exec the next jobs

        set wait_time [ined_search_wait_time]

    }

    ## wait for input or until the wait time is over
    ## queue a job read from tkined. It will be done
    ## in the main loop.

    set ined_main_time [getclock]
    set rfile [select {stdin} {} {} $wait_time]
    if {$rfile != ""} {
        gets stdin job
        lappend ined_queue $job

	## here we calculate the real time we have been waiting for
	## input and set the new ined_main_time to compute later
	## the time the interpreter was in ined_main_loop

	set res_time [expr {[getclock] - $ined_main_time}]
	set wait_time [expr {($res_time <= $wait_time)
	    ? $wait_time - $res_time : $wait_time}]
    }

    if {$ined_maxjobs != 0} {

	set ined_main_time [getclock]
        
        ## sub the time we have been waiting for an input
        ## it is $wait_time or $res_time

        ined_sub_wait_time $wait_time

    } else {

        set ined_main_time ""

    }
}

######################################################################

##
## Append job to the job list. The delay argument defines the initial
## time before the job becomes active for the first time and the count
## argument gives the total number of activations.
##

proc ined_create_job {cmd interval {delay 0} {count 0}} {

    global COMMAND REM_TIME TIME JOB_STATUS JOB_INTERVAL
    global ined_jobs ined_maxjobs

    if {![isnumber $interval]} {
	error "ined_create_job: interval is not a number"
    }

    if {![isnumber $delay]} {
	error "ined_create_job: delay is not a number"
    }

    if {![isnumber $count]} {
	error "ined_create_job: count is not a number"
    }

    if {$interval <= 0} {
	error "ined_create_job: interval must be positive"
    }

    if {$delay < 0} {
	error "ined_create_job: delay must be positive"
    }

    set ined_jobs($ined_maxjobs,$COMMAND) $cmd
    set ined_jobs($ined_maxjobs,$TIME) $interval
    set ined_jobs($ined_maxjobs,$REM_TIME) $delay
    set ined_jobs($ined_maxjobs,$JOB_INTERVAL) $count
    set ined_jobs($ined_maxjobs,$JOB_STATUS) waiting

    incr ined_maxjobs
    return [expr {$ined_maxjobs-1}]
}

##
## Kill jobs given by their jobid's from the job list.
##

proc ined_kill_job {args} {

    global ined_jobs ined_maxjobs
    global JOB_STATUS

    if {$ined_maxjobs == 0} {
	error "ined_kill_job: no job to kill"
    } 

    foreach jobid $args {
	if {$jobid >= 0 && $jobid < $ined_maxjobs} {
	    set k [expr {$ined_maxjobs-1}]
	    if {$jobid != $k} {
		for {set j 0} {$j <= $JOB_STATUS} {incr j} {
		    set ined_jobs($jobid,$j) $ined_jobs($k,$j)
		    unset ined_jobs($k,$j) 
		} 
	    } else {
		for {set j 0} {$j <= $JOB_STATUS} {incr j} {
		    unset ined_jobs($k,$j) 
		}
	    }
	    set ined_maxjobs $k
	} else {
	    ined acknowledge "ERROR: (ined_kill_job)" "" \
		"job $jobid not found"
	}
    }
}

##
## Suspend a job (put it into suspend state).
##

proc ined_suspend_job {args} {
    
    global ined_jobs ined_maxjobs
    global JOB_STATUS

    if {$ined_maxjobs == 0} {
	error "ined_suspend_job: no job to kill"
    }
    
    foreach jobid $args {
	if {$jobid >= 0 && $jobid < $ined_maxjobs} {
	    set ined_jobs($jobid,$JOB_STATUS) suspend
	} else {
	    error "ined_suspend_job: job $jobid not found"
	}
    }
}

##
## Resume a job (put it back into waiting state).
##

proc ined_resume_job {args} {
    
    global ined_jobs ined_maxjobs
    global JOB_STATUS

    if {$ined_maxjobs == 0} {
	error "ined_resume_job: no job to resume"
    }
    
    foreach jobid $args {
	if {$jobid >= 0 && $jobid < $ined_maxjobs} {
	    set ined_jobs($jobid,$JOB_STATUS) waiting
	} else {
	    error "ined_resume_job: job $jobid not found"
	}
    }
}

##
## Set the given interval time to a new value.
##

proc ined_modify_job {jobid command interval} {

    global ined_jobs ined_maxjobs
    global COMMAND TIME REM_TIME
    
    if {$ined_maxjobs == 0} {
	error "ined_modify_job: no job to modify"
    }
    
    if {![isnumber $interval]} {
	error "ined_modify_job: interval is not a number"
    }

    if {$interval <= 0} {
	error "ined_modify_job: interval must be positive"
    }

    if {$jobid >= 0 && $jobid < $ined_maxjobs} {
	if {$command != ""} {
	    set ined_jobs($jobid,$COMMAND)  $command
	}
	if {$interval != ""} {
	    set ined_jobs($jobid,$TIME) $interval
	    if {$interval < $ined_jobs($jobid,$REM_TIME)} {
		set ined_jobs($jobid,$REM_TIME) $interval
	    }
	}
    } else {
	error "ined_modify_job: job $jobid not found"
    }
}

##
## Return a list of all elements of the job list.
##

proc ined_list_jobs {} {

    global ined_jobs ined_maxjobs
    global JOB_STATUS

    set inf ""
    set res ""

    for {set i 0} {$i < $ined_maxjobs} {incr i} {
        set inf $i
        for {set j 0} {$j <= $JOB_STATUS} {incr j} {
            lappend inf $ined_jobs($i,$j)
        }
        lappend res $inf
    }

    return $res
}

######################################################################

##
## The following procedures implement some simple commands to control
## the job scheduler. They are here so that application can include 
## them in their tools.
##

proc "create job" {list} {
    
    static jobname interval count

    if {![info exists jobname]}  { set jobname "" }
    if {![info exists interval]} { set interval 5 }
    if {![info exists count]}    { set count 0 }

    set result [ined request "Create a new job to execute." \
		" { {Job Command:} {$jobname} } \
                  { {Intervaltime \[s\]:} $interval } \
                  { {Interval:} $count } "]

    if {$result == ""}  return 
    set jobname  [lindex $result 0]
    set interval [lindex $result 1]
    set count    [lindex $result 2]

    if {![isnumber $interval]} {
        ined acknowledge "The interval time must be a positive number!"
        return
    }

    if {![isnumber $count]} {
        ined acknowledge "The number of repetitions must be a positive number!"
        return
    }
    
    if {$jobname != ""} {
	ined_create_job $jobname $interval $interval $count
    }
}

proc "suspend job" {list} {
    
    set jobnr [ined_select_job suspend]
    if {$jobnr == ""} return

    ined_suspend_job $jobnr
}

proc "resume job" {list} {
    
    set jobnr [ined_select_job resume]
    if {$jobnr == ""} return

    ined_resume_job $jobnr
}

proc "modify job" {list} {

    global ined_jobs ined_maxjobs
    global COMMAND TIME

    set jobnr [ined_select_job modify]
    if {$jobnr == ""} return 

    set jobname $ined_jobs($jobnr,$COMMAND)
    set jobtime $ined_jobs($jobnr,$TIME)

    set result [ined request "Enter new values for job $jobnr ($jobname)." \
		 " { {Job Name:} {$jobname}} \
                   { {Intervaltime \[ms\]:} $jobtime scale 1 300} "]
    if {$result == ""} return

    set jobname [lindex $result 0]
    set jobtime [lindex $result 1]

    if {![isnumber $jobtime]} {
	ined acknowledge "The interval time must be a positive number!"
	return
    }

    ined_modify_job $jobnr $jobname $jobtime  
}

proc "kill job" {list} {

     set jobnr [ined_select_job kill]
     if {$jobnr == ""} return

     ined_kill_job $jobnr
}

proc "job info" {list} {

    global ined_maxjobs

    if {$ined_maxjobs == 0} {
	ined acknowledge "Sorry, no jobs available."
	return
    } 

    set result ""
    set len 0
    foreach job [ined_list_jobs] {

	set line \
	    [format "%3d %6.1f %6.1f %3d %8s %s" \
	    [lindex $job 0] [lindex $job 2] [lindex $job 3] \
	    [lindex $job 4] [lindex $job 5] [lindex $job 1] ]

	if {[string length $line] > $len} {
            set len [string length $line]
        }

        lappend result $line
    }

    set header " ID   INTV   REM  CNT  STATUS      CMD"

    for {set i [string length $header]} {$i < $len} {incr i} {
	append header " "
    }

    ined browse $header $result
}

proc ined_select_job {action} {
    
    global ined_jobs ined_maxjobs
    global COMMAND JOB_STATUS

    set res ""

    if {$ined_maxjobs == 0} {
	ined acknowledge "Sorry, no job to $action."
	return
    } 

    for {set i 0} {$i < $ined_maxjobs} {incr i} {
	set status $ined_jobs($i,$JOB_STATUS)
	if {   ($action == "resume"  && $status == "suspend")
	    || ($action == "suspend" && $status == "waiting")
	    || ($action == "kill") || ($action == "modify")} {
	    lappend res [format "%3d %s" $i $ined_jobs($i,$COMMAND)]
	}
    }

    if {$res == ""} {
	ined acknowledge "Sorry, no job to $action."
	return
    } else {
	return [lindex [lindex [ined list "Choose a job to $action:" $res] 0] 0]
    }
}

######################################################################

##
## This is the client stub. It handles a very simple protocol
## by reading and writing from or to stdin or stdout. If the
## interpreter has the select command (like scotty or extended
## tcl), we start a small job scheduler.
##

proc ined {command args} {

    global ined_queue

    # Send the command to ined (very simple since we just write it
    # to stdout :-). ined will echo all stuff written to stdout 
    # unless it begins with the keyword ined.
    regsub -all "\n" $args "\\n" args
    puts stdout "ined $command $args"
    flush stdout

    # Wait for an answer of ined. Ignore all incoming messages that
    # do not start with the word ined followed by either ok or error.
    # additional words are returned as the result string. Incoming
    # requests are stored in a queue and to be handled later.
    while {1} {
	if {[eof stdin]==1} {
	    puts stderr "lost connection -- shuting down interpreter"
	    exit
        }
        gets stdin res
        set lres [split $res]
	if {[llength $lres]==0} continue
	if {[lindex $lres 0]=="ined"} {
	    if {[lindex $lres 1]=="ok"} {
		return [join [lrange $lres 2 end]]
	    }
	    if {[lindex $lres 1]=="error"} {
		error "ined: [join [lrange $lres 2 end]]"
	    }
	}
	lappend ined_queue $res
    }
}

##
## Ined main loop waits for requests from the INED editor and executes
## them. Incoming requests are buffered in the ined_queue and executed,
## as soon as the interpreter becomes idle again.
##

proc ined_main_loop {} {

    global ined_queue errorInfo
    lappend ined_queue ""

    while {![eof stdin]} {

	# test if we have can use the job scheduler
	if {[lsearch [info commands] select] < 0} {
	    # read a command and put it in the queue
	    gets stdin res
	    lappend ined_queue $res
	} else {
	    # let the scheduler read the command or process its jobs
	    ined_schedule
	}

	if {[llength $ined_queue] == 0} continue

	# Start the evaluation of the commands in the queue.
	while {[llength $ined_queue]>0} {
	    puts stdout "ined queue [llength $ined_queue]"; flush stdout
	    set cmd [lindex $ined_queue 0]
	    set ined_queue [lrange $ined_queue 1 end]
	    if {[catch {eval $cmd} eval_err]} {
		puts stderr $errorInfo
		flush stderr
	    }
	}
	puts stdout "ined queue 0"; flush stdout
    }
}

######################################################################

##
## The following functions allow the access of the information
## stored in the object lists (see tkined(1) for more about
## this.) We should always use these access procedures so that
## we can change the object representation without breaking any
## code. The current object representation has the following format:
##
##  { NETWORK <id> <name> <address> <oid> <list of links> }
##  { NODE <id> <name> <address> <oid> <list of links> }
##  { LINK <id> <ida> <idb> }
##  { GROUP <id> <name> <oid> <list of objects> }
##  { TEXT <id> <text> }
##  { IMAGE <id> <filename> }
##

proc ined_type {comp} { return [lindex $comp 0] }

proc ined_id {comp} { return [lindex $comp 1] }

proc ined_name {comp} {
    case [ined_type $comp] in {
	{LINK}               { return "" }
	{NODE NETWORK GROUP} { return [lindex $comp 2] }
    }
}

proc ined_address {comp} {
    case [ined_type $comp] in {
	{LINK GROUP}   { return "" }
	{NODE NETWORK} { return [lindex $comp 3] }
    }
}

proc ined_oid {comp} {
    case [ined_type $comp] in {
	{LINK}         { return "" }
	{GROUP}        { return [lindex $comp 3] }
	{NODE NETWORK} { return [lindex $comp 4] }
    }
}

proc ined_links {comp} {
    case [ined_type $comp] in {
	{LINK GROUP}   { return "" }
	{NODE NETWORK} { return [lindex $comp 5] }
    }
}

proc ined_member {comp} {
    case [ined_type $comp] in {
        {NODE NETWORK LINK} { return "" }
        {GROUP} { return [lindex $comp 4] }
    }
}

proc ined_elements {comp} {
    puts stderr "**** this is old stuff: dont use ined_elements"
    return [ined_member $comp]
}

proc ined_ida {comp} {
    case [ined_type $comp] in {
	{NODE NETWORK GROUP} { return "" }
	{LINK} { return [lindex $comp 2] }
    }
}

proc ined_idb {comp} {
    case [ined_type $comp] in {
	{NODE NETWORK GROUP} { return "" }
	{LINK} { return [lindex $comp 3] }
    }
}

proc ined_text {comp} {
    case [ined_type $comp] in {
        {NODE NETWORK LINK GROUP} { return "" }
       {TEXT} { return [lindex $comp 2] }
   }
}
