/*
 * methods.c
 *
 * This file contains all functions that manipulate the state
 * of tkined objects. These functions get called whenever a user
 * invokes a command when applying a tool, or when a command is send
 * from an interpreter. Every method calls a corresponding tk 
 * procedure that is responsible to manipulate the display according
 * to the changes. The naming convention is that a command like `move'
 * is implemented as the method m_move (written in C) which calls
 * tkined_<type>_move to let tk take the appropriate action.
 *
 * Copyright (c) 1993, 1994
 *                    J. Schoenwaelder
 *                    TU Braunschweig, Germany
 *                    Institute for Operating Systems and Computer Networks
 *
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <pwd.h>

#include "tkined.h"

/*
 * A general purpose sprintf buffer.
 */

static char buffer[1024];

/*
 * This utility function gets the coordinates of a network
 * and puts them into two arrays  named x and y. The next step 
 * is to find a network segment which allows us to draw a horizontal 
 * or vertical line. If this fails, we connect to the endpoint that 
 * is closest to our initial position.
 */

static void 
m_network_link_end (interp, network, sx, sy)
    Tcl_Interp* interp;
    tkined_object *network;
    double *sx;
    double *sy;
{
    int found = 0;
    int i, j, n;
    int largc;
    char **largv;
    double *x;
    double *y;
    double rx = 0, ry = 0;
    double d = -1;

    Tcl_SplitList (interp, network->member, &largc, &largv);

    x = (double *) xmalloc (largc * sizeof(double));
    y = (double *) xmalloc (largc * sizeof(double));

    if (x == NULL || y == NULL) {
	free ((char*) largv);
	return;
    }

    for (n = 0, i = 0; i < largc; i++) {
	if ((i%2) == 0) {
	    Tcl_GetDouble (interp, largv[i], &x[n]);
	    x[n] += network->x;
	} else {
	    Tcl_GetDouble (interp, largv[i], &y[n]);
	    y[n] += network->y;
	    n++;
	}
    }

    for (i = 1, j = 0, found = 0; i < n; i++, j++) {

	double lo_x = (x[i] < x[j]) ? x[i] : x[j];
	double up_x = (x[i] < x[j]) ? x[j] : x[i];
	double lo_y = (y[i] < y[j]) ? y[i] : y[j];
        double up_y = (y[i] < y[j]) ? y[j] : y[i];

	if ((lo_x <= *sx) && (up_x >= *sx)) {
	    double nd = (*sy > y[i]) ? *sy - y[i] : y[i] - *sy;
	    if (d < 0 || nd < d ) {
		rx = *sx; ry = y[i];
		d = nd;
		found++;
	    }
	}

	if ((lo_y <= *sy) && (up_y >= *sy)) {
	    double nd = (*sx > x[i]) ? *sx - x[i] : x[i] - *sx;
	    if (d < 0 || nd < d ) {
                rx = x[i]; ry = *sy;
                d = nd;
		found++;
	    }
        }
    }

    /* If we can not make a horizontal or vertical link
       or if one of the fixed points is much nearer, we 
       simply make a link to the nearest fixed point */
    
    for (i = 0; i < n; i++) {	
	double nd = ((x[i] > *sx) ? x[i] - *sx : *sx - x[i])
		  + ((y[i] > *sy) ? y[i] - *sy : *sy - y[i]);
	if (!found || (nd < d)) {
	    rx = x[i]; ry = y[i];
	    d = nd;
	    found++;
	}
    }
    
    free ((char *) x);
    free ((char *) y);
    free ((char *) largv);

    *sx = rx; *sy = ry;

    return;
}

/*
 * Update the link position. This must get called whenever the link
 * moves or one of the connected objects moves.
 */

static int 
m_link_update (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    char *tmp;
    char buf_a[20], buf_b[20];
    int n;
    double *x;
    double *y;
    tkined_object *ida = NULL, *idb = NULL;
    int selected = object->selected;
    double x_a, y_a, x_b, y_b;

    if (selected) {
        m_unselect (interp, object, 0, (char **) NULL);
    }

    /* Get the objects linked by this link object. Search for 
       parent objects if there is currently no canvas. */

    for (tmp = object->ida, ida = id_to_object (tmp);
	 ida != NULL && (strlen (ida->canvas) == 0);
	 ida = id_to_object (tmp))
      tmp = ida->parent;

    for (tmp = object->idb, idb = id_to_object (tmp);
	 idb != NULL && (strlen (idb->canvas) == 0);
	 idb = id_to_object (tmp))
	    tmp = idb->parent;

    if (ida == NULL || idb == NULL) {
	interp->result = "update link: can not find linked objects";
	return TCL_ERROR;
    }

    if (ida->type == TKINED_NETWORK) {
	x_a = idb->x;
	y_a = idb->y;
	m_network_link_end (interp, ida, &x_a, &y_a);
    } else {
	x_a = ida->x;
	y_a = ida->y;
    }

    if (idb->type == TKINED_NETWORK) {
	x_b = ida->x;
	y_b = ida->y;
	m_network_link_end (interp, idb, &x_b, &y_b);
    } else {
	x_b = idb->x;
	y_b = idb->y;
    }

    /* handle fixed points if any */

    tmp = NULL;
    if (strlen(object->member) > 0) {
	int i,largc;
	char **largv;
	Tcl_SplitList (interp, object->member, &largc, &largv);

	if (largc > 2) {
	    x = (double *) xmalloc (largc * sizeof(double));
	    y = (double *) xmalloc (largc * sizeof(double));
	
	    if (x == NULL || y == NULL) {
		free ((char*) largv);
		sprintf (interp->result, "%f %f", object->x, object->y);
		return TCL_OK;
	    }
	    
	    for (n = 0, i = 0; i < largc; i++) {
		if ((i%2) == 0) {
		    Tcl_GetDouble (interp, largv[i], &x[n]);
		    x[n] += object->x;
		} else {
		    Tcl_GetDouble (interp, largv[i], &y[n]);
		    y[n] += object->y;
		    n++;
		}
	    }
	    
	    if (x[0] == x[1]) {
		y[0] = y_a;
	    } else {
		x[0] = x_a;
	    }
	    
	    if (x[n-1] == x[n-2]) {
		y[n-1] = y_b;
	    } else {
		x[n-1] = x_b;
	    }

	    tmp = xmalloc (n*32);
	    *tmp = 0;
	    for (i = 0; i < n; i++) {
		sprintf (buffer, "%.2f %.2f ", x[i], y[i]);
		strcat (tmp, buffer);
	    }

	    free ((char *) x);
	    free ((char *) y);
	}
	free ((char *) largv);
    }

    sprintf (buf_a, "%.2f %.2f ", x_a, y_a);
    sprintf (buf_b, "%.2f %.2f ", x_b, y_b);

    Tcl_VarEval (interp, "foreach item [", object->id, " items] {",
		 "if {[", object->canvas, " type $item]==\"line\"} break }; ",
		 "eval ", object->canvas, " coords $item ", 
		 buf_a, (tmp == NULL) ? "" : tmp, buf_b,
		 (char *) NULL);

    if (tmp != NULL) free (tmp);

    if (selected) {
        m_select (interp, object, 0, (char **) NULL);
    }
    return TCL_OK;
}


/*
 * Process a method of a tkined_object. Most of them simply set
 * or retrieve attribute values.
 */


/*
 * Create a NODE object. Initialize the id and name fields.
 */

int 
m_node_create (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    static unsigned lastid = 0;

    sprintf(buffer, "node%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, buffer);

    trace (object->id, "ined create NODE");

    return TCL_OK;
}

/*
 * Create a GROUP object. Initialize the id and name fields.
 * Set the member to the ids given in argv.
 */

int 
m_group_create (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    static unsigned lastid = 0;
    char *buf;

    sprintf(buffer, "group%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, buffer);
    STRCOPY (object->member, Tcl_Merge (argc, argv));
    object->collapsed = 0;

    buf = xmalloc(strlen(object->member)+30);
    sprintf (buf, "ined create GROUP %s", object->member);
    trace (object->id, buf);
    free (buf);
    
    return TCL_OK;
}

/*
 * Create a NETWORK object. Initialize the id and name fields.
 * Store the points in the member field and set the position
 * to the first x and y coordinates.
 */

int 
m_network_create (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    static unsigned lastid = 0;
    char *buf;

    sprintf(buffer, "network%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, buffer);

    if (argc>1) {
	int i;
	Tcl_GetDouble (interp, argv[0], &(object->x));
	Tcl_GetDouble (interp, argv[1], &(object->y));
	*buffer = 0;
	for (i = 0; i < (argc/2)*2; i++) {
	    char tmp[10];
	    double val;
	    Tcl_GetDouble (interp, argv[i++], &val);
	    sprintf (tmp, "%8f ", val-object->x);
	    strcat (buffer, tmp);
	    Tcl_GetDouble (interp, argv[i], &val);
	    sprintf (tmp, "%8f ", val-object->y);
	    strcat (buffer, tmp);
	}
	STRCOPY (object->member, buffer);
    } else {
	/* default length and position */
	STRCOPY (object->member, "0 0 130 0");
	object->x = 50;
	object->y = 50;
    }
    
    buf = xmalloc(strlen(object->member)+30);
    sprintf (buf, "ined create NETWORK %s", object->member);
    trace (object->id, buf);
    free (buf);
    sprintf (buffer, "ined move %s %f %f",
	     object->id, object->x, object->y);
    trace ((char *) NULL, buffer);

    return TCL_OK;
}

/*
 * Create a LINK object. Initialize the id and name fields.
 * Initialize the fields that contain the id of the connected
 * objects.
 */

int 
m_link_create (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    static unsigned lastid = 0;
    char *buf;

    if (argc<2) {
	interp->result = "wrong # args";
	return TCL_ERROR;
    }

    sprintf(buffer, "link%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, buffer);
    STRCOPY (object->ida, argv[0]);
    STRCOPY (object->idb, argv[1]);
    if (argc>3) {
	STRCOPY (object->member, Tcl_Merge (argc-2, argv+2));
    }

    sprintf (buffer, "%s links \"[%s links] %s\"; %s links \"[%s links] %s\"",
	     object->ida, object->ida, object->id,
	     object->idb, object->idb, object->id);
    Tcl_Eval (interp, buffer);

    buf = xmalloc(strlen(object->member)+40);
    sprintf (buf, "ined create LINK %s %s %s", 
	     object->ida, object->idb, object->member);
    trace (object->id, buf);
    free (buf);
    
    return TCL_OK;
}

/*
 * Create a TEXT object. Initialize the id and name fields.
 * Map any newline characters to the sequence \n.
 */

int 
m_text_create (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    static unsigned lastid = 0;
    char *buf;

    sprintf(buffer, "text%d", lastid++);
    STRCOPY (object->id, buffer);

    m_text (interp, object, 1, &argv[0]);

    buf = xmalloc(strlen(argv[0])+30);
    sprintf (buf, "ined create TEXT \"%s\"", argv[0]);
    trace (object->id, buf);
    free (buf);

    return TCL_OK;
}

/*
 * Create an IMAGE object. Initialize the id and name fields,
 * which contains the path to the bitmap file.
 */

int 
m_image_create (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    static unsigned lastid = 0;
    char *buf;
    char *file;

    if (argc<1) {
	interp->result = "wrong # args";
	return TCL_ERROR;
    }

    if ((file = findfile(argv[0])) == (char *) NULL) {
        sprintf (interp->result, "%s not found", argv[0]);
        return TCL_ERROR;
    }

    sprintf(buffer, "image%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, file);

    buf = xmalloc(strlen(argv[0])+30);
    sprintf (buf, "ined create IMAGE %s", argv[0]);
    trace (object->id, buf);
    free (buf);

    return TCL_OK;
}

/*
 * Create an INTERPRETER object. Initialize the id and name fields.
 * Initialize a command buffer and fork a process to do the real work.
 */

int 
m_interpreter_create (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    static unsigned lastid = 0;

    int pid;
    FILE *in;
    char *p;
    char *argw[4];
    char *file;

    if (argc<1) {
	interp->result = "wrong # args";
	return TCL_ERROR;
    }

    if ((file = findfile(argv[0])) == (char *) NULL) {
        sprintf (interp->result, "%s not found", argv[0]);
	m_delete (interp, object, 0, (char **) NULL);
        return TCL_ERROR;
    }

    sprintf(buffer, "interpreter%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, file);
    Tcl_DStringInit (&object->buffer);
    
    if (socketpair (AF_UNIX, SOCK_STREAM, 0, object->xv) < 0) {
	object->xv[0] = object->xv[1] = -1;
	interp->result = "unable to get socket pair";
	return TCL_ERROR;
    }

    pid = fork();
    if (pid < 0) {
	object->xv[0] = object->xv[1] = -1;
	interp->result = "unable to fork process";
	return TCL_ERROR;
    }

    object->pid = pid;

    if (pid == 0) {                                        /* child */
	close (object->xv [0]);
	dup2  (object->xv [1], 0);
	dup2  (object->xv [1], 1);
	
	/* start #! scripts by hand to allow long pathes */
	
	if (((in = fopen(object->name, "r")) != NULL) 
	    && (buffer == fgets(buffer,512,in))) {
	    if ( buffer[0] == '#' && buffer[1] == '!') {
		for (p=buffer+2; isspace(*p); p++) ;
		argw[0] = p;
		while (*p && !isspace(*p)) p++;
		while (*p && isspace(*p)) *p++ = 0;
		argw[1] = p;
		while (*p && !isspace(*p)) p++;
		*p = 0;
		argw[2] = object->name;
		argw[3] = NULL;

		if (strlen(argw[1]) == 0) {
		    argw[1] = argw[2]; argw[2] = NULL;
		}

		execv (argw[0], argw);
		
		fprintf (stderr, "ined acknowledge {Can not execute %s}\n", argw[0]);
		printf ("ined acknowledge {Can not execute %s}\n", argw[0]);
		fflush (stdout);
		fgetc(stdin);
		exit (1);
	    }
	    fclose (in);
	}
	
	execl (object->name, object->name, (char *) 0);
	fprintf (stdout, "ined acknowledge {Can not execute %s}\n", 
		 object->name);
	fflush (stdout);
	fgetc(stdin);
	exit (1);
    } else {                                            /* parent */
	close (object->xv [1]);

	Tk_CreateFileHandler(object->xv[0], TK_READABLE,
			     receive, (ClientData) object);

	Tcl_DetachPids (1, &pid);
    }

    return TCL_OK;
}

/*
 * Create a TOOL object. Initialize the id and name fields.
 * The items field contains the commands of the tool and 
 * the links field contains the id of the interpreter object.
 */

int 
m_tool_create (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    static unsigned lastid = 0;

    if (argc<2) {
	interp->result = "wrong # args";
	return TCL_ERROR;
    }

    sprintf(buffer, "tool%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, argv[0]);
    STRCOPY (object->items, Tcl_Merge (argc-1, argv+1));

    return TCL_OK;
}

/*
 * Create a LOG object (window). Initialize the id and name fields.
 */

int 
m_log_create (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    struct passwd *pwd;
    static unsigned lastid = 0;

    sprintf(buffer, "log%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, buffer);

    pwd = getpwuid(getuid());
    STRCOPY (object->address, pwd->pw_name);

    trace (object->id, "ined create LOG");

    return TCL_OK;
}

/*
 * Retrieve the external representation of a NODE object.
 */

int 
m_node_retrieve (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    sprintf (buffer, "%u", object->oid);
    Tcl_AppendElement (interp, "NODE");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->address);
    Tcl_AppendElement (interp, buffer);
    Tcl_AppendElement (interp, object->links);
    return TCL_OK;
}

/*
 * Retrieve the external representation of a GROUP object.
 */

int 
m_group_retrieve (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{    
    sprintf (buffer, "%u", object->oid);
    Tcl_AppendElement (interp, "GROUP");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, buffer);
    Tcl_AppendElement (interp, object->member);
    return TCL_OK;
}

/*
 * Retrieve the external representation of a NETWORK object.
 */

int 
m_network_retrieve (interp, object, argc, argv)

    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{    
    sprintf (buffer, "%u", object->oid);
    Tcl_AppendElement (interp, "NETWORK");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->address);
    Tcl_AppendElement (interp, buffer);
    Tcl_AppendElement (interp, object->links);
    return TCL_OK;
}

/*
 * Retrieve the external representation of a LINK object.
 */

int
m_link_retrieve (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    Tcl_AppendElement (interp, "LINK");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->ida);
    Tcl_AppendElement (interp, object->idb);
    return TCL_OK;
}

/*
 * Retrieve the external representation of a TEXT object.
 */

int 
m_text_retrieve (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    Tcl_AppendElement (interp, "TEXT");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->text);
    return TCL_OK;
}

/*
 * Retrieve the external representation of a IMAGE object.
 */

int 
m_image_retrieve (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    Tcl_AppendElement (interp, "IMAGE");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    return TCL_OK;
}

/*
 * Retrieve the external representation of a INTERPRETER object.
 */

int 
m_interpreter_retrieve (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    Tcl_AppendElement (interp, "INTERPRETER");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    return TCL_OK;
}

/*
 * Retrieve the external representation of a TOOL object.
 */

int 
m_tool_retrieve (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    Tcl_AppendElement (interp, "TOOL");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->items);
    return TCL_OK;
}

/*
 * Retrieve the external representation of a LOG object.
 */

int 
m_log_retrieve (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    Tcl_AppendElement (interp, "LOG");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->address);
    return TCL_OK;
}

/*
 * Return the type of an object.
 */

int
m_type (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    interp->result = type_to_string (object->type);

    return TCL_OK;
}

/*
 * Return the id of an object.
 */

int 
m_id (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    interp->result = object->id;
    return TCL_OK;
}

/*
 * Get and set the name of an object. Call the tk callback if the label 
 * is set to show the name.
 */

int
m_name (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc==1) {
        STRCOPY (object->name, argv[0]);

	if (object->type == TKINED_LOG) {
	    sprintf (buffer, "tkined_%s_name %s",
		     type_to_string (object->type), object->id);
	    Tcl_Eval (interp, buffer);
	}

	if (strcmp(object->label, "name") == 0) {
	    m_label (interp, object, 1, &object->label);
	}

	sprintf (buffer, "ined name %s \"%s\"", object->id, object->name);
	trace (object->name, buffer);
    }

    interp->result = object->name;
    return TCL_OK;
}

/*
 * Return the canvas of an object.
 */

int 
m_canvas (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc > 0 ) {
        STRCOPY (object->canvas, argv[0]);

	if (strlen(object->canvas) > 0) {
	    Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
			 "_canvas ", object->id, (char *) NULL);
	    if (object->type == TKINED_LINK) {
		m_link_update (interp, object, 0, (char **) NULL);
	    }
	}

	if (object->type == TKINED_NODE) {
	    char *tmp = "create";
	    if (object->stripchart) {
		object->stripchart = 0;
		m_stripchart (interp, object, 1, &tmp);
	    } else if (object->barchart) {
		object->barchart = 0;
		m_barchart (interp, object, 1, &tmp);
	    }
	}

	if (object->type == TKINED_LINK) {
	    m_lower (interp, object, 0, (char **) NULL);
	}

	/* Update all links connected to this NODE or NETWORK object */

	if (object->type == TKINED_NODE || object->type == TKINED_NETWORK) {
	    int largc, i;
	    char **largv;
	    tkined_object *link;
	    Tcl_SplitList (interp, object->links, &largc, &largv);
	    for (i=0; i<largc; i++) {
	        link = id_to_object (largv[i]);
		if (link != NULL) {
		    m_link_update (interp, link, 0, (char **) NULL);
		    
		}
	    }
	    free ((char*) largv);
	}

    }

    interp->result = object->canvas;
    return TCL_OK;
}

/*
 * Get and set the items belonging to an object.
 */

int
m_items (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc==1) {
        STRCOPY (object->items, argv[0]);
    }
    
    interp->result = object->items;
    return TCL_OK;
}

/*
 * Get and set the address of an object. Call the tk callback if the
 * label is set to show the address.
 */

int
m_address (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc==1) {
        STRCOPY (object->address, argv[0]);

	if (strcmp(object->label, "address") == 0) {
	    m_label (interp, object, 1, &object->label);
	}
	sprintf (buffer, "ined address %s \"%s\"", 
		 object->id, object->address);
	trace (object->address, buffer);
    }

    interp->result = object->address;
    return TCL_OK;
}


/*
 * Get and set the oid of an object. There is no callback here.
 */

int
m_oid (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    int result;
    
    if (argc==1) {
	if (Tcl_GetInt (interp, argv[0], &result) != TCL_OK) {
	    return TCL_ERROR;
	}
	object->oid = result;
	sprintf (buffer, "ined oid %s \"%d\"", object->id, object->oid);
	trace (argv[0], buffer);
    }

    sprintf (interp->result, "%d", object->oid);
    return TCL_OK;
}

/*
 * Select on object. Mark it as selected and call the tk
 * procedure to actually do the selection.
 */

int
m_select (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (!object->selected) {
	object->selected = 1;

	Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		     "_select ", object->id, (char *) NULL);
    }
	
    return TCL_OK;
}

/*
 * Unselect on object. Mark it as unselected and call the tk
 * procedure to actually do the unselection.
 */

int
m_unselect (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (object->selected) {
	object->selected = 0;
	
	Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		     "_unselect ", object->id, (char *) NULL);
    }

    return TCL_OK;
}

/*
 * Return a boolean indicating if the object is selected or not.
 */

int
m_selected (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    sprintf (interp->result, "%d", object->selected);

    return TCL_OK;
}

/*
 * Get and set the icon of an object. Call tkined_<TYPE>_icon
 * to let tk update the canvas.
 */

#if FOO
extern Tk_Window w;
#endif

int
m_icon (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
#if FOO
    Tk_Window tkwin;
    Tk_Uid uid;
#endif
    char *tmp = "reset";

    int selected = object->selected;

    if (argc==1) {

#if FOO
	tkwin = Tk_NameToWindow (interp, object->canvas, w);
	uid = Tk_GetUid(argv[0]);

	fprintf(stderr, "**** %s %x %s\n", object->canvas, tkwin, uid);

	if ((Tk_GetBitmap(interp, tkwin, Tk_GetUid(argv[0]))) == None) {
	    fprintf (stderr, "** need to load bitmap %s\n", argv[0]);
	}
#endif

        STRCOPY (object->icon, argv[0]);
	
	if (selected) {
	    m_unselect (interp, object, 0, (char **) NULL);
	}
	
        Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		     "_icon ", object->id, (char *) NULL);

	m_label (interp, object, 1, &tmp);

	if (selected) {
	    m_select (interp, object, 0, (char **) NULL);
	}
	sprintf (buffer, "ined icon %s \"%s\"", object->id, object->icon);
	trace (object->icon, buffer);
    }
    
    interp->result = object->icon;
    return TCL_OK;
}

/*
 * Get and set the label of an object. Call tkined_<TYPE>_label
 * to let tk update the canvas.
 */

int
m_label (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc>0) {
	if ( strcmp(argv[0], "clear") == 0 ) {
	    STRCOPY (object->label, argv[0]);
	    sprintf (buffer, "tkined_%s_label_clear %s", 
		     type_to_string (object->type), object->id);
	    Tcl_Eval (interp, buffer);
	    sprintf (buffer, "ined label %s clear", object->id);
	    trace ((char *) NULL, buffer);
	} else if ( strcmp(argv[0], "reset") == 0) {
	    sprintf (buffer, "tkined_%s_label_clear %s", 
		     type_to_string (object->type), object->id);
	    Tcl_Eval (interp, buffer);
	    m_label (interp, object, 1, &object->label);
	} else {
	    char *tmp = NULL;
	    if (strcmp(argv[0], "name") == 0) { 
		tmp = object->name;
		sprintf (buffer, "ined label %s name", object->id);
		trace ((char *) NULL, buffer);
	    } else if (strcmp(argv[0], "address") == 0 ) {
		tmp = object->address;
		sprintf (buffer, "ined label %s address", object->id);
		trace ((char *) NULL, buffer);
	    } else if (strcmp(argv[0], "text") == 0) {
		char *buf;
	        if (argc == 2) {
		    STRCOPY (object->text, argv[1]);
		}
		tmp = object->text;
		buf = xmalloc (strlen(tmp)+40);
		sprintf (buf, "ined label %s text \"%s\"", object->id, tmp);
		trace ((char *) NULL, buf);
		free (buf);
	    }
	    if (tmp != NULL) {
	        STRCOPY (object->label, argv[0]);
		sprintf (buffer, "tkined_%s_label %s \"", 
			 type_to_string (object->type), object->id);
		Tcl_VarEval (interp, buffer, tmp, "\"", (char *) NULL);	    
	    }
	}
    }

    interp->result = object->label;

    return TCL_OK;
}

/*
 * Get and set the font of an object. Call tkined_<TYPE>_font
 * to let tk update the canvas.
 */

int
m_font (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    int selected = (object->selected && object->type == TKINED_TEXT);

    if (argc==1) {
        STRCOPY (object->font, argv[0]);
	
	if (selected) {
	    m_unselect (interp, object, 0, (char **) NULL);
	}
	
	Tcl_VarEval (interp, "tkined_", type_to_string (object->type), 
		  "_font ", object->id, (char *) NULL);

	if (selected) {
            m_select (interp, object, 0, (char **) NULL);
        }
	sprintf (buffer, "ined font %s \"%s\"", object->id, object->font);
	trace (object->font, buffer);
    }

    interp->result = object->font;
    return TCL_OK;
}

/*
 * Get and set the color of an object. Call tkined_<TYPE>_font
 * to let tk update the canvas.
 */

int
m_color (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc==1) {
        STRCOPY (object->color, argv[0]);
	
	Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		     "_color ", object->id, (char *) NULL);	    
	sprintf (buffer, "ined color %s \"%s\"", object->id, argv[0]);
	trace (object->color, buffer);
    }

    interp->result = object->color;
    return TCL_OK;
}

/*
 * Move an object. Return the new position and after calling
 * the tk callback tkined_<TYPE>_move.
 */

int
m_move (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc==2) {
	double x, y;

	if (Tcl_GetDouble (interp, argv[0], &x) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Tcl_GetDouble (interp, argv[1], &y) != TCL_OK) {
	    return TCL_ERROR;
	}
	object->x += x;
	object->y += y;

	if ( *(object->canvas) != '\0') {
	    int i;
	    int largc;
	    char **largv;

	    sprintf (buffer, "tkined_%s_move %s %f %f", 
		     type_to_string (object->type), object->id, x, y);
	    Tcl_Eval (interp, buffer);
	    
	    Tcl_SplitList (interp, object->links, &largc, &largv);
	    for (i=0; i<largc; i++) {
	        tkined_object *link;
	        link = id_to_object (largv[i]);
		if (link != NULL) {
		    m_link_update (interp, link, 0, (char **) NULL);
		}
	    }
	    free ((char*) largv);
	}
	sprintf (buffer, "ined move %s %f %f", object->id, x, y);
	trace ((char *) NULL, buffer);
    }

    sprintf (buffer, "%f %f", object->x, object->y);
    interp->result = buffer;
    return TCL_OK;
}

/*
 * Move a link object. Link objects just update their position, This
 * need only be done when the link is visible.
 */

int
m_link_move (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc==2) {
	double x, y;
    
        if (Tcl_GetDouble (interp, argv[0], &x) != TCL_OK) {
            return TCL_ERROR;
	  }
        if (Tcl_GetDouble (interp, argv[1], &y) != TCL_OK) {
            return TCL_ERROR;
	  }
        object->x += x;
        object->y += y;

	if ( *(object->canvas) != '\0') {
	    m_link_update (interp, object, 0, (char **) NULL);
        }

	sprintf (buffer, "ined move %s %f %f", object->id, x, y);
	trace ((char *) NULL, buffer);
    }

    sprintf (buffer, "%f %f", object->x, object->y);
    interp->result = buffer;
    return TCL_OK;
}

/*
 * Move a group object. Moving a group means moving all its members.
 # There is some space for optimization here, since members currently
 # not visible could be handled directly here, if they are not groups
 * themself.
 */

int
m_group_move (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{ 

    if (argc==2) {
	double x, y;
	int i, margc;
	char **margv;

        if (Tcl_GetDouble (interp, argv[0], &x) != TCL_OK) {
            return TCL_ERROR;
	  }
        if (Tcl_GetDouble (interp, argv[1], &y) != TCL_OK) {
            return TCL_ERROR;
	  }
        object->x += x;
        object->y += y;

	if ( *(object->canvas) != '\0') {
	    sprintf (buffer, "tkined_%s_move %s %f %f", 
		     type_to_string (object->type), object->id, x, y);
	    Tcl_Eval (interp, buffer);
	}
	    
	Tcl_SplitList (interp, object->member, &margc, &margv);
	for (i=0; i<margc; i++) {
	    tkined_object *member;
	    member = id_to_object (margv[i]);
	    if (member != NULL) {
		if ( ( *(member->canvas) != '\0') ||

		    (member->type == TKINED_GROUP)) {
		    sprintf (buffer, "%s move %f %f", margv[i], x, y);
		    Tcl_Eval (interp, buffer);

		} else {

		    member->x += x;
		    member->y += y;

		    /* Dont forget links pointing from collapsed node or
		       network objects to the outside world. */

		    if (member->type == TKINED_NODE 
			|| member->type == TKINED_NETWORK) {
		        int l, largc;
		        char **largv;
			Tcl_SplitList (interp, member->links, &largc, &largv);
			for (l=0; l<largc; l++) {
			    tkined_object *link;
			    link = id_to_object (largv[l]);
			    if (link != NULL) {
				if ( *(link->canvas) != '\0') {
				    m_link_update (interp, link, 
						   0, (char **) NULL);
				}
			    }
			}
			free ((char*) largv);
		    }
		}
	    }
	}
	free ((char*) margv);

	sprintf (buffer, "ined move %s %f %f", object->id, x, y);
	trace ((char *) NULL, buffer);
    }

    sprintf (buffer, "%f %f", object->x, object->y);
    interp->result = buffer;
    return TCL_OK;
}


/*
 * Raise all items belonging to an object.
 */

int
m_raise (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		 "_raise ", object->id, (char *) NULL);

    return TCL_OK;
}

/*
 * Lower all items belonging to an object. Make sure that images
 * are always in the background. The loop through all objects
 * should be optimized.
 */

int
m_lower (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{

    Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		 "_lower ", object->id, (char *) NULL);

    if (object->type != TKINED_IMAGE) {
	Tcl_HashEntry *ht_entry;
	Tcl_HashSearch ht_search;

	ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
	while (ht_entry != NULL) {
	    tkined_object *any;
	    any = (tkined_object *) Tcl_GetHashValue (ht_entry);
	    if (any->type == TKINED_IMAGE) {
		m_lower (interp, any, 0, (char **) NULL);
	    }
	    ht_entry = Tcl_NextHashEntry (&ht_search);
	}
    }

    return TCL_OK;
}

/*
 * Return the size of an object by the coordinates of the bounding box.
 */

int
m_size (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    int ret;

    ret = Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		       "_size ", object->id, (char *) NULL);

    if (ret == TCL_OK && strlen(interp->result) > 0) {
	STRCOPY (object->size, interp->result);
    }

    interp->result = object->size;

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the node
 * object.
 */

int 
m_node_dump (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{

    sprintf (buffer, "set %s [ined -noupdate create NODE]\n",
	     object->id);
    Tcl_AppendResult (interp, buffer, (char *)NULL);

    if (object->x != 0 && object->y != 0) {
	sprintf (buffer, "ined -noupdate move $%s %f %f\n",
		 object->id, object->x, object->y);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }
    sprintf (buffer, "ined -noupdate icon $%s \"%s\"\n",
	     object->id, object->icon);
    Tcl_AppendResult (interp, buffer, (char *)NULL);
    sprintf (buffer, "ined -noupdate name $%s \"%s\"\n",
	     object->id, object->name);
    Tcl_AppendResult (interp, buffer, (char *)NULL);

    if ( strcmp(object->address, "") != 0) {
	sprintf (buffer, "ined -noupdate address $%s \"%s\"\n",
		 object->id, object->address);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }

    if (object->oid != 0) {
	sprintf (buffer, "ined -noupdate oid $%s \"%d\"\n",
		 object->id, object->oid);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }

    if ( (strcmp(object->label, "clear") != 0)
	&& (strcmp(object->label, "text") != 0)) {
	sprintf (buffer, "ined -noupdate label $%s %s\n",
		 object->id, object->label);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }

    sprintf (buffer, "ined -noupdate font $%s \"%s\"\n",
	     object->id, object->font);
    Tcl_AppendResult (interp, buffer, (char *)NULL);

    if (strlen(object->color) > 0 && strcmp(object->color, "Black") != 0) {
	sprintf (buffer, "ined -noupdate color $%s %s\n",
                 object->id, object->color);
        Tcl_AppendResult (interp, buffer, (char *)NULL);
    }
    
    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the group
 * object. The user of the string  must ensure that the variables 
 * for the member objects exist.
 */

int
m_group_dump (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    int i;
    int largc;
    char **largv;

    sprintf (buffer, "set %s [eval ined -noupdate create GROUP ", object->id);
    Tcl_AppendResult (interp, buffer, (char *)NULL);
    Tcl_SplitList (interp, object->member, &largc, &largv);
    for (i=0; i < largc; i++) {
	Tcl_AppendResult (interp, " $", largv[i], (char *)NULL);
    }
    free ((char*) largv);
    Tcl_AppendResult (interp, "]\n", (char *)NULL);

    /* Save the position if the group has no members. Otherwise,
       we get the position from the position of the group members */

    if (largc == 0) {
	if (object->x != 0 && object->y != 0) {
	    sprintf (buffer, "ined -noupdate move $%s %f %f\n",
		     object->id, object->x, object->y);
	    Tcl_AppendResult (interp, buffer, (char *)NULL);
	}
    }

    sprintf (buffer, "ined -noupdate icon $%s \"%s\"\n",
	     object->id, object->icon);
    Tcl_AppendResult (interp, buffer, (char *)NULL);
    sprintf (buffer, "ined -noupdate name $%s \"%s\"\n",
	     object->id, object->name);
    Tcl_AppendResult (interp, buffer, (char *)NULL);

    if (object->oid != 0) {
	sprintf (buffer, "ined -noupdate oid $%s \"%d\"\n",
		 object->id, object->oid);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }

    if ( (strcmp(object->label, "clear") != 0)
	&& (strcmp(object->label, "text") != 0)) {
	sprintf (buffer, "ined -noupdate label $%s %s\n",
		 object->id, object->label);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }

    sprintf (buffer, "ined -noupdate font $%s \"%s\"\n",
	     object->id, object->font);
    Tcl_AppendResult (interp, buffer, (char *)NULL);

    if (strlen(object->color) > 0 && strcmp(object->color, "Black") != 0) {
	sprintf (buffer, "ined -noupdate color $%s %s\n",
                 object->id, object->color);
        Tcl_AppendResult (interp, buffer, (char *)NULL);
    }
    
    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the node
 * object.
 */

int
m_network_dump (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    sprintf (buffer, "set %s [ined -noupdate create NETWORK %s]\n",
	     object->id, object->member);
    Tcl_AppendResult (interp, buffer, (char *)NULL);

    if (object->x != 0 && object->y != 0) {
	sprintf (buffer, "ined -noupdate move $%s %f %f\n",
		 object->id, object->x, object->y);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }
    sprintf (buffer, "ined -noupdate icon $%s \"%s\"\n",
	     object->id, object->icon);
    Tcl_AppendResult (interp, buffer, (char *)NULL);
    sprintf (buffer, "ined -noupdate name $%s \"%s\"\n",
	     object->id, object->name);
    Tcl_AppendResult (interp, buffer, (char *)NULL);

    if ( strcmp(object->address, "") != 0) {
	sprintf (buffer, "ined -noupdate address $%s \"%s\"\n",
		 object->id, object->address);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }

    if (object->oid != 0) {
	sprintf (buffer, "ined -noupdate oid $%s \"%d\"\n",
		 object->id, object->oid);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }

    if ( (strcmp(object->label, "clear") != 0)
	&& (strcmp(object->label, "text") != 0)) {
	sprintf (buffer, "ined -noupdate label $%s %s\n",
		 object->id, object->label);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }

    sprintf (buffer, "ined -noupdate font $%s \"%s\"\n",
	     object->id, object->font);
    Tcl_AppendResult (interp, buffer, (char *)NULL);

    if (strlen(object->color) > 0 && strcmp(object->color, "Black") != 0) {
	sprintf (buffer, "ined -noupdate color $%s %s\n",
                 object->id, object->color);
        Tcl_AppendResult (interp, buffer, (char *)NULL);
    }
    
    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the node
 * object.
 */

int
m_link_dump (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    sprintf (buffer, "set %s [ined -noupdate create LINK $%s $%s %s]\n",
	     object->id, object->ida, object->idb, object->member);
    Tcl_AppendResult (interp, buffer, (char *)NULL);

    if (strlen(object->color) > 0 && strcmp(object->color, "Black") != 0) {
	sprintf (buffer, "ined -noupdate color $%s %s\n",
                 object->id, object->color);
        Tcl_AppendResult (interp, buffer, (char *)NULL);
    }
    
    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the text.
 */

int
m_text_dump (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    sprintf (buffer, "set %s [ined -noupdate create TEXT \"%s\"", 
	     object->id, object->text);
    Tcl_AppendResult (interp, buffer, object->name, "]\n", (char *)NULL);

    if (object->x != 0 && object->y != 0) {
	sprintf (buffer, "ined -noupdate move $%s %f %f\n",
		 object->id, object->x, object->y);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }
    sprintf (buffer, "ined -noupdate font $%s \"%s\"\n",
	     object->id, object->font);
    Tcl_AppendResult (interp, buffer, (char *)NULL);

    if (strlen(object->color) > 0 && strcmp(object->color, "Black") != 0) {
	sprintf (buffer, "ined -noupdate color $%s %s\n",
                 object->id, object->color);
        Tcl_AppendResult (interp, buffer, (char *)NULL);
    }
    
    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the image.
 */

int
m_image_dump (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    sprintf (buffer, "set %s [ined -noupdate create IMAGE %s]\n",
	     object->id, object->name);
    Tcl_AppendResult (interp, buffer, (char *)NULL);

    if (object->x != 0 && object->y != 0) {
	sprintf (buffer, "ined -noupdate move $%s %f %f\n",
		 object->id, object->x, object->y);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }

    if (strlen(object->color) > 0 && strcmp(object->color, "Black") != 0) {
	sprintf (buffer, "ined -noupdate color $%s %s\n",
                 object->id, object->color);
        Tcl_AppendResult (interp, buffer, (char *)NULL);
    }
    
    return TCL_OK;
}

/*
 * Get the starting point (object) of a link.
 */

int
m_ida (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    interp->result = object->ida;

    return TCL_OK;
}

/*
 * Get the end point (object) of a link.
 */

int
m_idb (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    interp->result = object->idb;

    return TCL_OK;
}

/*
 * Get and set the text of a text object. This calls
 * tkined_TEXT_text to update tkined.
 */

int
m_text (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc==1) {
	char *p, *t;
	int n = 2;

	for (p = argv[0]; *p != 0; p++) {
	    if (*p == '\n') n++;
	}
	n += (p - argv[0]);
	free (object->text);  object->text  = xmalloc (n);
	for (t = object->text, p = argv[0]; *p != 0; p++, t++) {
	    if (*p == '\n') {
		*t++ = '\\'; 
		*t = 'n';
	    } else {
		*t = *p;
	    }
	}
	*t = 0;
	
	Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		     "_text ", object->id, (char *) NULL);
    }

    interp->result = object->text;

    return TCL_OK;
}

/*
 * Append some text to the LOG window. The tk callback tkined_LOG_append
 * will be called to let the GUI do its actions.
 */

int
m_append (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    int i;

    for (i = 0; i < argc; i++) {
	Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		     "_append ", object->id, " {", argv[i], "}",
		     (char *) NULL);
    }

    return TCL_OK;
}

/*
 * Get the interpreter of a log or tool object. Its name is stored in 
 * the links attribute.
 */

int
m_interpreter (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    interp->result = object->links;
    return TCL_OK;
}

/*
 * Clear the text inside of a LOG window. The tk callback tkined_LOG_clear
 * will be called to let the GUI do its actions.
 */

int
m_clear (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		 "_clear ", object->id, (char *) NULL);

    return TCL_OK;
}

/*
 * Switch a node object to show values in a stripchart.
 */

int
m_stripchart (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    double value;
    int selected = object->selected;
    char *tmp = "reset";
    
    if (argc > 0) {
	if (strcmp(argv[0], "create") == 0) {
	    if (!object->stripchart && !object->barchart) {
		if (selected) {
		    m_unselect (interp, object, 0, (char **) NULL);
		}
		Tcl_VarEval (interp, "tkined_stripchart_create ",
			     object->id, (char *) NULL);
		object->stripchart = 1;
		if (selected) {
		    m_select (interp, object, 0, (char **) NULL);
		}
		m_label (interp, object, 1, &tmp);
		sprintf (buffer, "ined stripchart %s create", object->id);
		trace ((char *) NULL, buffer);
	    }
	} else if (strcmp(argv[0], "delete") == 0 && object->stripchart) {
	    if (selected) {
		m_unselect (interp, object, 0, (char **) NULL);
	    }
	    object->stripchart = 0;
	    Tcl_VarEval (interp, "tkined_stripchart_delete ",
			 object->id, (char *) NULL);
	    if (selected) {
		m_select (interp, object, 0, (char **) NULL);
	    }
	    sprintf (buffer, "ined stripchart %s delete", object->id);
	    trace ((char *) NULL, buffer);
	} else if (Tcl_GetDouble (interp, argv[0], &value) == TCL_OK) {
	    if (!object->stripchart && !object->barchart) {
		object->stripchart = 1;
	    }
	    if (object->stripchart) {
		char *args = Tcl_Merge (argc, argv);
		Tcl_VarEval (interp, "tkined_stripchart_values ",
			     object->id, " ", args, (char *) NULL);
		sprintf (buffer, "ined stripchart %s %s", object->id, args);
		trace ((char *) NULL, buffer);
		free (args);
	    }
	}
    }
    
    return TCL_OK;
}

/*
 * Switch a node object to show values in a barchart.
 */

int
m_barchart (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    int selected = object->selected;
    char *tmp = "reset";
    
    if (argc > 0) {
	if (strcmp(argv[0], "create") == 0) {
	    if (!object->stripchart && !object->barchart) {
		if (selected) {
		    m_unselect (interp, object, 0, (char **) NULL);
		}
		Tcl_VarEval (interp, "tkined_barchart_create ",
			     object->id, (char *) NULL);
		object->barchart = 1;
		if (selected) {
		    m_select (interp, object, 0, (char **) NULL);
		}
		m_label (interp, object, 1, &tmp);
		sprintf (buffer, "ined barchart %s create", object->id);
		trace ((char *) NULL, buffer);
	    }
	} else if (strcmp(argv[0], "delete") == 0 && object->barchart) {
	    if (selected) {
		m_unselect (interp, object, 0, (char **) NULL);
	    }
	    object->barchart = 0;
	    Tcl_VarEval (interp, "tkined_barchart_delete ",
			 object->id, (char *) NULL);
	    if (selected) {
		m_select (interp, object, 0, (char **) NULL);
	    }
	    sprintf (buffer, "ined barchart %s delete", object->id);
	    trace ((char *) NULL, buffer);
	} else {
	    if (!object->stripchart && !object->barchart) {
		object->barchart = 1;
	    }
	    if (object->barchart) {
		char *args = Tcl_Merge (argc, argv);
		Tcl_VarEval (interp, "tkined_barchart_values ",
			     object->id, " ", args, (char *) NULL);
		sprintf (buffer, "ined barchart %s %s", object->id, args);
		trace ((char *) NULL, buffer);
		free (args);
	    }
	}
    }
  
    return TCL_OK;
}

/*
 * Get and set the member variable of an object. member is used to
 * store the members of a group and to hold the fixed points of link
 * or network objects.
 */

int
m_member (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc==1) {
        STRCOPY (object->member, argv[0]);
    }

    interp->result = object->member;

    return TCL_OK;
}

/*
 * Collapse a group to an icon.
 */

int
m_collapse (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (!object->collapsed) {
	int i, largc;
	char **largv;
	double x1 = 0, y1 = 0, x2 = 0, y2 = 0;
        int selected = object->selected;

        object->collapsed = 1;
      
        if (selected) {
	    m_unselect (interp, object, 0, (char **) NULL);
	}

	Tcl_SplitList (interp, object->member, &largc, &largv);
	for (i = 0; i < largc; i++) {
	    tkined_object *member;
	    member = id_to_object (largv[i]);
	    if (member != NULL) {
	        int sargc;
		char **sargv;

		if (member->selected) {
		    m_unselect (interp, member, 0, (char **) NULL);
		}

		STRCOPY (member->parent, object->id);
		if (member->type == TKINED_GROUP && !member->collapsed) {
		    m_collapse (interp, member, 0, (char **) NULL);
		}

		/* Calculate the position of the group icon. */

		m_size (interp, member, 0, (char **) NULL);
		Tcl_SplitList (interp, member->size, &sargc, &sargv);
		if (sargc == 4) {
		  double mx1, my1, mx2, my2;
		  Tcl_GetDouble (interp, sargv[0], &mx1);
		  Tcl_GetDouble (interp, sargv[1], &my1);
		  Tcl_GetDouble (interp, sargv[2], &mx2);
		  Tcl_GetDouble (interp, sargv[3], &my2);
		  if (x1 == 0 || mx1 < x1) x1 = mx1;
		  if (y1 == 0 || my1 < y1) y1 = my1;
		  if (mx2>x2) x2=mx2;
		  if (my2>y2) y2=my2;
		}

		STRCOPY (member->canvas, "");	        
	    }
	}
	free ((char*) largv);

	if (largc > 0) {
	    char *tmp = "0";
	    char * foo[2];
	    foo[0] = tmp;
	    foo[1] = tmp;
	    object->x = x1+(x2-x1)/2;
	    object->y = y1+(y2-y1)/2;
	    m_group_move (interp, object, 2, foo);
	}

	Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		     "_collapse ", object->id, (char *) NULL);

        m_icon   (interp, object, 1, &object->icon);
        m_color  (interp, object, 1, &object->color);
        m_font   (interp, object, 1, &object->font);
        m_label  (interp, object, 1, &object->label);

	if (selected) {
	    m_select (interp, object, 0, (char **) NULL);
	}
	
	sprintf (buffer, "ined collapse %s", object->id);
	trace ((char *) NULL, buffer);
    }

    return TCL_OK;
}

/*
 * Expand a group to show its members.
 */

int
m_expand (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (object->collapsed) {
	int i, largc;
	char **largv;
        int selected = object->selected;
	object->collapsed = 0;
      
	if (selected) {
	    m_unselect (interp, object, 0, (char **) NULL);
	}
      
	Tcl_SplitList (interp, object->member, &largc, &largv);
	for (i = 0; i < largc; i++) {
	    tkined_object *member;
	    member = id_to_object (largv[i]);
	    if (member != NULL) {
		STRCOPY (member->parent, "");
		if (member->type == TKINED_GROUP && member->collapsed) {
		    member->collapsed = 0;
		}
		m_canvas (interp, member, 1, &object->canvas);
		m_color  (interp, member, 1, &member->color);
		m_icon   (interp, member, 1, &member->icon);
		m_font   (interp, member, 1, &member->font);
		m_label  (interp, member, 1, &member->label);
	    }
	}
	free ((char*) largv);
      
	Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		     "_expand ", object->id, (char *) NULL);

	m_color  (interp, object, 1, &object->color);
	m_font   (interp, object, 1, &object->font);
	m_label  (interp, object, 1, &object->label);
      
	if (selected) {
	    m_select (interp, object, 0, (char **) NULL);
	}

	sprintf (buffer, "ined expand %s", object->id);
	trace ((char *) NULL, buffer);
    }

    return TCL_OK;
}

/*
 * Return a boolean indicating if the group is collapsed or not.
 */

int
m_collapsed (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    sprintf (interp->result, "%d", object->collapsed);

    return TCL_OK;
}

/*
 * Get and set the links that are connected to a node or network
 * object.
 */

int
m_links (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc==1) {
        STRCOPY (object->links, argv[0]);
    }

    interp->result = object->links;

    return TCL_OK;
}

/*
 * Computer the label coordinates of a network object.
 * This is done in C because computations are complicated
 * and slow in TCL.
 *
 * m_network_labelxy searches for the longest horizontalnetwork segment 
 * that is at least longer than 100 points. If there is no such segment, 
 * we return the coordinates of the lowest endpoint of the vertical line
 * endpoint.
 */

int 
m_network_labelxy (interp, network, argc, argv)
    Tcl_Interp* interp;
    tkined_object *network;
    int argc;
    char** argv;
{
    int found = 0;
    int i, j, n;
    int largc;
    char **largv;
    double *x;
    double *y;
    double lx, ly;
    double sx = 0, sy = 0, slen = 0;

    Tcl_SplitList (interp, network->member, &largc, &largv);

    x = (double *) xmalloc (largc * sizeof(double));
    y = (double *) xmalloc (largc * sizeof(double));

    if (x == NULL || y == NULL) {
	free ((char*) largv);
	sprintf (interp->result, "%f %f", network->x, network->y);
	return TCL_OK;
    }

    for (n = 0, i = 0; i < largc; i++) {
	if ((i%2) == 0) {
	    Tcl_GetDouble (interp, largv[i], &x[n]);
	    x[n] += network->x;
	} else {
	    Tcl_GetDouble (interp, largv[i], &y[n]);
	    y[n] += network->y;
	    n++;
	}
    }

    for (i = 1, j = 0; i < n; i++, j++) {
	lx = (x[i]>x[j]) ? x[i]-x[j] : x[j]-x[i];
	ly = (y[i]>y[j]) ? y[i]-y[j] : y[j]-y[i];
	if (!found) {
            if (y[i] > sy) {
		sx = (x[i]+x[j])/2;
		sy = y[i];
	    }
	    if (y[j] > sy) {
                sx = (x[i]+x[j])/2;
                sy = y[j];
            }
	}
	if (lx > slen) {
            sx = (x[i]+x[j])/2;
	    sy = (y[i]+y[j])/2;
	    slen = lx;
	    found = (slen > 100);
	}
    }
    sy += 3;

    free ((char *) x);
    free ((char *) y);
    free ((char *) largv);

    sprintf (interp->result, "%f %f", sx, sy+1);
    return TCL_OK;
}

/*
 * Send a message (usually a command) to an interpreter.
 * Append a list of the current selection to it.
 */

int
m_send (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (argc>0) {
	int len;
	Tcl_DString buf;
	Tcl_DStringInit (&buf);

	/* Assemble the string in a dynamic string. start with the 
	   command to be invoked */

	Tcl_DStringAppendElement (&buf, argv[0]);

	if (argc > 1) {
	    char *args;

	    /* join the given arguments to a single list */

	    args = Tcl_Merge (argc, argv);
	    Tcl_DStringAppendElement (&buf, args);
	    free (args);

	} else {
	    
	    Tcl_HashEntry *ht_entry;
	    Tcl_HashSearch ht_search;
	    tkined_object *any;

	    /* Build a list of all selected objects */

	    Tcl_DStringStartSublist (&buf);

	    ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
	    while (ht_entry != NULL) {
		any = (tkined_object *) Tcl_GetHashValue (ht_entry);
		if (any->selected
		    && strcmp(object->canvas, any->canvas) == 0) {
		    if (Tcl_VarEval (interp, any->id, " retrieve", 
				     (char *) NULL) == TCL_OK) {
			Tcl_DStringAppendElement (&buf, interp->result);
		    }
		}
		ht_entry = Tcl_NextHashEntry (&ht_search);
	    }
	    
	    Tcl_DStringEndSublist (&buf);
	    Tcl_DStringAppend (&buf, "\n", 1);
	}

	len = Tcl_DStringLength (&buf);
	if (len != write (object->xv[0], Tcl_DStringValue (&buf), len)) {
	    sprintf (interp->result, "write to %s failed",
                     object->id);
            return TCL_ERROR;
	}
	
	if (debug) fprintf (stderr, "<< ined %s\n", Tcl_DStringValue (&buf));

	Tcl_DStringFree (&buf);
    }

    Tcl_ResetResult(interp);
    return TCL_OK;
}

/*
 * This is called for network and node objects. Before they
 * get deleted, they must delete all links connected to them.
 */

int
m_linked_delete (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    int i, largc;
    char **largv;

    Tcl_SplitList (interp, object->links, &largc, &largv);
    for (i=0; i<largc; i++) {
	Tcl_VarEval (interp, largv[i], " delete", (char *) NULL);
    }
    free ((char*) largv);

    return m_delete (interp, object, argc, argv);
}

/*
 * When deleting a link object, we have to update the links
 * stored by the objects connected by this link.
 */

int
m_link_delete (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    sprintf (buffer, "set lks [%s links]; set idx [lsearch $lks %s]; %s links [lreplace $lks $idx $idx]",
	     object->ida, object->id, object->ida);
    Tcl_Eval (interp, buffer);

    sprintf (buffer, "set lks [%s links]; set idx [lsearch $lks %s]; %s links [lreplace $lks $idx $idx]",
	     object->idb, object->id, object->idb);
    Tcl_Eval (interp, buffer);

    return m_delete (interp, object, argc, argv);
}

/*
 * When deleting a group object, we make sure that it is expanded.
 */

int
m_group_delete (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (object->collapsed) {
	m_expand (interp, object, argc, argv);
    }

    return m_delete (interp, object, argc, argv);
}

/*
 * Make sure to delete all tools associated with this interpreter.
 */

int
m_interpreter_delete (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (object->trace) traceNum--;

    if (object->type == TKINED_INTERPRETER) {
	Tcl_HashEntry *ht_entry;
	Tcl_HashSearch ht_search;
	ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
	while (ht_entry != NULL) {
	    tkined_object *obj;
	    obj = (tkined_object *) Tcl_GetHashValue (ht_entry);
	    if ((obj->type == TKINED_TOOL) && 
		(strcmp(obj->address, object->id) == 0)) {
		m_delete (interp, obj, 0, (char **) NULL);
	    }
	    ht_entry = Tcl_NextHashEntry (&ht_search);
	}
    }

    return m_delete (interp, object, argc, argv);
}

/*
 * Delete an object. This is understood by all objects.
 */

int
m_delete (interp, object, argc, argv)
    Tcl_Interp* interp;
    tkined_object *object;
    int argc;
    char** argv;
{
    if (object->selected) {
	m_unselect (interp, object, 0, (char **) NULL);
    }

    Tcl_VarEval (interp, "tkined_", type_to_string (object->type),
		 "_delete ", object->id, (char *) NULL);

    /* Remove the reference to this object if it is a member
       of a group (that is it has a valid parent) */

    if ( strlen(object->parent) > 0) {
	tkined_object *parent = id_to_object (object->parent);
	if (parent != NULL) {
	    int largc, i;
	    char **largv;	    
	    Tcl_SplitList (interp, parent->member, &largc, &largv);
	    *parent->member = 0;
	    for (i = 0; i< largc; i++) {
		if (strcmp (object->id, largv[i]) != 0) {
		    strcat (parent->member, largv[i]);
		    strcat (parent->member, " ");
		}
	    }
	    free ((char*) largv);
	}
    }

    Tcl_DeleteCommand (interp, object->id);

    sprintf (buffer, "ined delete %s", object->id);
    trace ((char *) NULL, buffer);

    return TCL_OK;
}

