/* 
 * stripchart.c --
 *
 *	This file implements stripchart items for canvas widgets.
 *      It's based on tkRectOval.c and tkCanvPoly.c from the 
 *      tk distribution by John Ousterhout.
 *
 *      Written by G. Hueske (hueske@ibr.cs.tu-bs.de)
 *                 TU Braunschweig, Germany
 */

#include <stdio.h>
#include <tkInt.h>
#include <tkCanvas.h>
#include <tkConfig.h>

/*
 * The structure below defines the record for each stripchart item.
 */

typedef struct StripchartItem {
    Tk_Item header;             /* Generic stuff that's the same for all
                                 * types.  MUST BE FIRST IN STRUCTURE. */
    int numPoints;              /* Number of points in polygon (always >= 3).
                                 * Polygon is always closed. */
    double *coordPtr;           /* Pointer to malloc-ed array containing
                                 * x- and y-coords of all points in polygon.
                                 * X-coords are even-valued indices, y-coords
                                 * are corresponding odd-valued indices. */
    double *shiftPtr;           /* Pointer to malloc-ed array containing 
				 * the y-coords of all points in polygon. */
    int scale;                  /* Actual scale of the stripchart. */
    int scalevalue;             /* Every scalevalue we scale up. */
    int jump;                   /* Number of pixels jumped with a 
				 * jumpscroll. */
    int numValues;              /* Number of values that are actually 
				 * stored in shiftPtr. */
    XColor *stripColor;         /* Foreground color for stripchart. */
    GC fillGC;                  /* Graphics context for filling stripchart. */
    double bbox[4];             /* Coordinates of bounding box for rectangle
                                 * (x1, y1, x2, y2). */
    XColor *rectColor;          /* Background color. */
    GC rectGC;                  /* Graphics context for rectangular area. */
    int outlineWidth;           /* Width of rectangular outline. */
    XColor *outlineColor;       /* Color of rectangular outline. */
    GC outlineGC;               /* Graphics context for outline. */
    int scalelineWidth;         /* Width of scaleline. */
    XColor *scalelineColor;     /* Color of scaleline. */
    int scalelineStyle;         /* Style of scaleline, 0 means LineSolid, 
				 * each other positive value means the
				 * length of dashes using LineOnOffDash. */
    GC scalelineGC;             /* Graphics context for scaleline. */
} StripchartItem;


/*
 * Information used for parsing configuration specs:
 */

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_INT, "-jumpscroll", (char *) NULL, (char *) NULL,
	"5", Tk_Offset(StripchartItem, jump), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_INT, "-scalevalue", (char *) NULL, (char *) NULL,
	"1", Tk_Offset(StripchartItem, scalevalue), 
	TK_CONFIG_DONT_SET_DEFAULT},  
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
        "black", Tk_Offset(StripchartItem, stripColor), TK_CONFIG_NULL_OK}, 
    {TK_CONFIG_COLOR, "-background", (char *) NULL, (char *) NULL,
        (char *) NULL, Tk_Offset(StripchartItem, rectColor), 
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_PIXELS, "-outlinewidth", (char *) NULL, (char *) NULL,
	"1", Tk_Offset(StripchartItem, outlineWidth), 
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-outline", "outlineColor", (char *) NULL,
        "black", Tk_Offset(StripchartItem, outlineColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_SYNONYM, "-oc", "outlineColor", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_PIXELS, "-scalelinewidth", (char *) NULL, (char *) NULL,
        "1", Tk_Offset(StripchartItem, scalelineWidth), 
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_COLOR, "-scaleline", (char *) NULL, (char *) NULL,
        "black", Tk_Offset(StripchartItem, scalelineColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_INT, "-scalelinestyle", (char *) NULL, (char *) NULL,
	"4", Tk_Offset(StripchartItem, scalelineStyle),
        TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
        (char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
        (char *) NULL, 0, 0}
};


/*
 * Prototypes for procedures defined in this file:
 */

static void             ComputeStripchartBbox _ANSI_ARGS_((
                            Tk_Canvas *canvasPtr, StripchartItem *stripPtr));
static int              ConfigureStripchart _ANSI_ARGS_((
                            Tk_Canvas *canvasPtr, Tk_Item *itemPtr, int argc,
                            char **argv, int flags));
static int              CreateStripchart _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            struct Tk_Item *itemPtr, int argc, char **argv));
static void             DeleteStripchart _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr));
static void             DisplayStripchart _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, Drawable dst));
static int              StripchartCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, int argc, char **argv));
static int              StripchartToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, double *rectPtr));
static double           StripchartToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, double *pointPtr));
static int              StripchartToPostscript _ANSI_ARGS_((
                            Tk_Canvas *canvasPtr, Tk_Item *itemPtr, 
                            Tk_PostscriptInfo *psInfoPtr));
static void             ScaleStripchart _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, double originX, double originY,
                            double scaleX, double scaleY));
static void             TranslateStripchart _ANSI_ARGS_((Tk_Canvas *canvasPtr,
                            Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * The structures below defines the stripchart item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_ItemType TkStripchartType = {
    "stripchart",                       /* name */
    sizeof(StripchartItem),             /* itemSize */
    CreateStripchart,                   /* createProc */
    configSpecs,                        /* configSpecs */
    ConfigureStripchart,                /* configureProc */
    StripchartCoords,                   /* coordProc */
    DeleteStripchart,                   /* deleteProc */
    DisplayStripchart,                  /* displayProc */
    0,                                  /* alwaysRedraw */
    StripchartToPoint,                  /* pointProc */
    StripchartToArea,                   /* areaProc */
    StripchartToPostscript,             /* postscriptProc */
    ScaleStripchart,                    /* scaleProc */
    TranslateStripchart,                /* translateProc */
    (Tk_ItemIndexProc *) NULL,          /* indexProc */
    (Tk_ItemCursorProc *) NULL,         /* icursorProc */
    (Tk_ItemSelectionProc *) NULL,      /* selectionProc */
    (Tk_ItemInsertProc *) NULL,         /* insertProc */
    (Tk_ItemDCharsProc *) NULL,         /* dTextProc */
    (Tk_ItemType *) NULL                /* nextPtr */
};

/*
 * The definition below determines how large are static arrays
 * used to hold spline points (splines larger than this have to
 * have their arrays malloc-ed).
 */

#define MAX_STATIC_POINTS 200




/*
 *--------------------------------------------------------------
 *
 * CreateStripchart --
 *
 *	This procedure is invoked to create a new stripchart item in
 *	a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	canvasPtr->interp->result;  in this case itemPtr is
 *	left uninitialized, so it can be safely freed by the
 *	caller.
 *
 * Side effects:
 *	A new stripchart item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateStripchart(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing stripchart. */
{
    register StripchartItem *stripPtr = (StripchartItem *) itemPtr;
    int i;
    
    if (argc < 4) {
	Tcl_AppendResult(canvasPtr->interp, "wrong # args:  should be \"",
		Tk_PathName(canvasPtr->tkwin), "\" create ",
		itemPtr->typePtr->name,
		" x1 y1 x2 y2 ?y3 y4 ...? ?options?", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    stripPtr->numPoints = 0;
    stripPtr->coordPtr = NULL;
    stripPtr->shiftPtr = NULL;
    stripPtr->scale = 1;
    stripPtr->scalevalue = 100;
    stripPtr->jump = 5;
    stripPtr->numValues = 0;
    stripPtr->stripColor = NULL;
    stripPtr->fillGC = None;
    stripPtr->rectColor = NULL;
    stripPtr->rectGC = None;
    stripPtr->outlineWidth = 1;
    stripPtr->outlineColor = NULL;
    stripPtr->outlineGC = None;
    stripPtr->scalelineWidth = 1;
    stripPtr->scalelineColor = NULL;
    stripPtr->scalelineStyle = 4;
    stripPtr->scalelineGC = None;

    /*
     * Count the number of points and then parse them into a point
     * array.  Leading arguments are assumed to be points if they
     * start with a digit or a minus sign followed by a digit.
     */

    for (i = 4; i < argc; i++) {
	if ((!isdigit(UCHAR(argv[i][0]))) &&
		((argv[i][0] != '-') || (!isdigit(UCHAR(argv[i][1]))))) {
	    break;
	}
    }

    if (StripchartCoords(canvasPtr, itemPtr, i, argv) != TCL_OK) {
	goto error;
    }

    if (ConfigureStripchart(canvasPtr, itemPtr, argc-i, argv+i, 0) == TCL_OK) {
	return TCL_OK;
    }

    error:
    DeleteStripchart(canvasPtr, itemPtr);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * StripchartCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on stripcharts.  See the user documentation for details
 *	on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets canvasPtr->interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
StripchartCoords(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
    register StripchartItem *stripPtr = (StripchartItem *) itemPtr;
    char buffer[TCL_DOUBLE_SPACE];
    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE];
    char c2[TCL_DOUBLE_SPACE], c3[TCL_DOUBLE_SPACE];
    int i, j, x1, x2, y1, y2;
    int numPoints, numCoords, width, height;
    double value = 0.0;
    double *savePtr = NULL;

    if (argc == 0) {
	Tcl_PrintDouble(canvasPtr->interp, stripPtr->bbox[0], c0);
        Tcl_PrintDouble(canvasPtr->interp, stripPtr->bbox[1], c1);
        Tcl_PrintDouble(canvasPtr->interp, stripPtr->bbox[2], c2);
        Tcl_PrintDouble(canvasPtr->interp, stripPtr->bbox[3], c3);
        Tcl_AppendResult(canvasPtr->interp, c0, " ", c1, " ",
                c2, " ", c3, (char *) NULL);
	for (i = 0; i < stripPtr->numValues; i++) {
	    Tcl_PrintDouble(canvasPtr->interp, stripPtr->shiftPtr[i], buffer);
	    Tcl_AppendElement(canvasPtr->interp, buffer);
	}
    } else if (argc < 4) {
	Tcl_AppendResult(canvasPtr->interp,
		"too few coordinates for stripchart:  must have at least 4",
		(char *) NULL);
	return TCL_ERROR;
    }  else {
	if ((TkGetCanvasCoord(canvasPtr, argv[0],
		    &stripPtr->bbox[0]) != TCL_OK)
		|| (TkGetCanvasCoord(canvasPtr, argv[1],
		    &stripPtr->bbox[1]) != TCL_OK)
		|| (TkGetCanvasCoord(canvasPtr, argv[2],
			&stripPtr->bbox[2]) != TCL_OK)
		|| (TkGetCanvasCoord(canvasPtr, argv[3],
			&stripPtr->bbox[3]) != TCL_OK)) {
	    return TCL_ERROR;
	}

	x1 = SCREEN_X(canvasPtr, stripPtr->bbox[0]);
	x2 = SCREEN_X(canvasPtr, stripPtr->bbox[2]);
	width = (x2-x1-1>0) ? x2-x1-1 : x1-x2-1;
	numPoints = width+2;
	numCoords = 2*numPoints;

	/* 
	 * If we enlarge the bouding box, we have to allocate
	 * more space for the point arrays.
	 */

	if (stripPtr->numPoints < numPoints) {
            if (stripPtr->shiftPtr != NULL) {
		savePtr = stripPtr->shiftPtr;
            }
            stripPtr->shiftPtr = (double *) ckalloc((unsigned)
				     (sizeof(double) * (width)));
            for (i = 0; i < width; i++) {
                stripPtr->shiftPtr[i] = -1;
            }
	    for (i = 0; i < stripPtr->numValues; i++) {
		stripPtr->shiftPtr[i] = savePtr[i];
	    }
	    if (savePtr != NULL) {
		ckfree ((char *) savePtr);
	    }
	    
	    if (stripPtr->coordPtr != NULL) {
		ckfree((char *) stripPtr->coordPtr);
	    }
	    stripPtr->coordPtr = (double *) ckalloc((unsigned)
                                     (sizeof(double) * numCoords));
	    stripPtr->numPoints = numPoints;
	} 
	    /* The bounding box gets smaller. */

	    else if (stripPtr->numPoints > numPoints) {
		if (stripPtr->numValues > width-1) {
		    for (i = 0; i < width; i++) {
			stripPtr->shiftPtr[i] = 
			    stripPtr->shiftPtr[i+stripPtr->numValues-width+1];
		    }
		    stripPtr->numValues = width-1;
		}
		stripPtr->numPoints = numPoints;
	    }

	/* There are new polygon coordinates. */

	if (stripPtr->jump <= 0) {
	    stripPtr->jump = 1;
	}
	for (i = 4; i < argc; i++) {

	    /* Check if we have to jump. */

	    if (stripPtr->numValues == width-1) {
		for (j = 0; j < width-stripPtr->jump; j++) {
		    stripPtr->shiftPtr[j] = 
			    stripPtr->shiftPtr[j+stripPtr->jump];
		}
		stripPtr->numValues -= stripPtr->jump;
		for (j = width-stripPtr->jump; j < width-1; j++) {
		    stripPtr->shiftPtr[j] = 0;
		}
	    }
	    if (TkGetCanvasCoord(canvasPtr, argv[i],
		    &stripPtr->shiftPtr[stripPtr->numValues]) != TCL_OK) {
		return TCL_ERROR;
	    }
	    stripPtr->numValues++;
	}
			
	/* Pass the coordinates into coordPtr. */

	stripPtr->coordPtr[0] = 1;
	stripPtr->coordPtr[1] = -1;
	for (i = 2; i < numCoords-2; i+=2) {
	    stripPtr->coordPtr[i] = i/2;
	    stripPtr->coordPtr[i+1] = stripPtr->shiftPtr[i/2-1];
	}
	stripPtr->coordPtr[numCoords-2] = width;
	stripPtr->coordPtr[numCoords-1] = -1;

	/* Adapt the x-coordinates to the actual bounding box. */

	for (i = 2; i < numCoords; i+=2) {
	    stripPtr->coordPtr[i] += stripPtr->bbox[0];
	}
	stripPtr->coordPtr[0] = stripPtr->bbox[0];

	/* 
	 * Now we have to check if the actual scale is right. 
	 */
	
	/* Find the maximal y-value of the polygon. */

	for (i = 0; i < numCoords; i+=2) {
	    if (stripPtr->coordPtr[i+1] > value)
		    value = stripPtr->coordPtr[i+1];
	}
	
	/* Find the right scale. */

	stripPtr->scale = 1;
	if (value/stripPtr->scale > stripPtr->scalevalue) {
	    while (value/stripPtr->scale > stripPtr->scalevalue) {
		stripPtr->scale++;
	    }
	} 
   
	y1 = SCREEN_Y(canvasPtr, stripPtr->bbox[1]);
	y2 = SCREEN_Y(canvasPtr, stripPtr->bbox[3]);
	height = (y2-y1-1>0) ? y2-y1-1 : y1-y2-1;

	for (i = 0; i < numCoords; i+=2) {
	    stripPtr->coordPtr[i+1] = 
		    -stripPtr->coordPtr[i+1]/stripPtr->scalevalue * 
			    height/stripPtr->scale + stripPtr->bbox[3]-1;
	}
    }
	    
    ComputeStripchartBbox(canvasPtr, stripPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureStripchart --
 *
 *      This procedure is invoked to configure various aspects
 *      of a stripchart item such as its background color.
 *
 * Results:
 *      A standard Tcl result code.  If an error occurs, then
 *      an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *      Configuration information, such as colors, may be set 
 *      for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureStripchart(canvasPtr, itemPtr, argc, argv, flags)
    Tk_Canvas *canvasPtr;       /* Canvas containing itemPtr. */
    Tk_Item *itemPtr;           /* Stripchart item to reconfigure. */
    int argc;                   /* Number of elements in argv.  */
    char **argv;                /* Arguments describing things to configure. */
    int flags;                  /* Flags to pass to Tk_ConfigureWidget. */
{
    register StripchartItem *stripPtr = (StripchartItem *) itemPtr;
    XGCValues gcValues;
    GC newGC;
    unsigned long mask;

    if (Tk_ConfigureWidget(canvasPtr->interp, canvasPtr->tkwin,
	    configSpecs, argc, argv, (char *) stripPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    if (stripPtr->stripColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = stripPtr->stripColor->pixel;
	mask = GCForeground;
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (stripPtr->fillGC != None) {
	Tk_FreeGC(canvasPtr->display, stripPtr->fillGC);
    }
    stripPtr->fillGC = newGC;

    if (stripPtr->outlineColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = stripPtr->outlineColor->pixel;
	gcValues.cap_style = CapProjecting;
	if (stripPtr->outlineWidth < 0) {
	    stripPtr->outlineWidth = 1;
	}
	gcValues.line_width = stripPtr->outlineWidth;
	mask = GCForeground|GCCapStyle|GCLineWidth;
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (stripPtr->outlineGC != None) {
	Tk_FreeGC(canvasPtr->display, stripPtr->outlineGC);
    }
    stripPtr->outlineGC = newGC;
    
    if (stripPtr->rectColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = stripPtr->rectColor->pixel;
	mask = GCForeground;
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (stripPtr->rectGC != None) {
	Tk_FreeGC(canvasPtr->display, stripPtr->rectGC);
    }
    stripPtr->rectGC = newGC;


    if (stripPtr->scalelineColor == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = stripPtr->scalelineColor->pixel;
	if (stripPtr->scalelineWidth < 0) {
	    stripPtr->scalelineWidth = 1;
	}
	gcValues.line_width = stripPtr->scalelineWidth;
	mask = GCForeground|GCLineWidth;
	if (stripPtr->scalelineStyle < 0) {
	    stripPtr->scalelineStyle = 0;
	}
	gcValues.line_style = LineSolid;
	if (stripPtr->scalelineStyle > 0) {
	    gcValues.line_style = LineOnOffDash;
	    gcValues.dashes = (char) stripPtr->scalelineStyle;
	    mask |= GCLineStyle|GCDashList;
	}
	newGC = Tk_GetGC(canvasPtr->tkwin, mask, &gcValues);
    }
    if (stripPtr->scalelineGC != None) {
	Tk_FreeGC(canvasPtr->display, stripPtr->scalelineGC);
    }
    stripPtr->scalelineGC = newGC;
    
    ComputeStripchartBbox(canvasPtr, stripPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteStripchart --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a stripchart item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteStripchart(canvasPtr, itemPtr)
    Tk_Canvas *canvasPtr;		/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
{
    register StripchartItem *stripPtr = (StripchartItem *) itemPtr;

    if (stripPtr->coordPtr != NULL) {
	ckfree((char *) stripPtr->coordPtr);
    }
    if (stripPtr->shiftPtr != NULL) {
	ckfree((char *) stripPtr->shiftPtr);
    }
    if (stripPtr->stripColor != NULL) {
	Tk_FreeColor(stripPtr->stripColor);
    }
    if (stripPtr->fillGC != None) {
	Tk_FreeGC(canvasPtr->display, stripPtr->fillGC);
    }
    if (stripPtr->rectColor != NULL) {
	Tk_FreeColor(stripPtr->rectColor);
    }
    if (stripPtr->rectGC != None) {
	Tk_FreeGC(canvasPtr->display, stripPtr->rectGC);
    }
    if (stripPtr->outlineColor != NULL) {
	Tk_FreeColor(stripPtr->outlineColor);
    }
    if (stripPtr->outlineGC != None) {
        Tk_FreeGC(canvasPtr->display, stripPtr->outlineGC);
    } 
    if (stripPtr->scalelineColor != NULL) {
	Tk_FreeColor(stripPtr->scalelineColor);
    }
    if (stripPtr->scalelineGC != None) {
	Tk_FreeGC(canvasPtr->display, stripPtr->scalelineGC);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputeStripchartBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a stripchart.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

static void
ComputeStripchartBbox(canvasPtr, stripPtr)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    StripchartItem *stripPtr;		/* Item whose bbox is to be
					 * recomputed. */
{
    int bloat;

    /*
     * Make sure that the first coordinates are the lowest ones.
     */

    if (stripPtr->bbox[1] > stripPtr->bbox[3]) {
        double tmp;
        tmp = stripPtr->bbox[3];
        stripPtr->bbox[3] = stripPtr->bbox[1];
        stripPtr->bbox[1] = tmp;
    }
    if (stripPtr->bbox[0] > stripPtr->bbox[2]) {
        double tmp;
        tmp = stripPtr->bbox[2];
        stripPtr->bbox[2] = stripPtr->bbox[0];
        stripPtr->bbox[0] = tmp;
    }

    bloat = (stripPtr->outlineWidth+1)/2 + 1;
    stripPtr->header.x1 = stripPtr->bbox[0] - bloat;
    stripPtr->header.y1 = stripPtr->bbox[1] - bloat;
    stripPtr->header.x2 = stripPtr->bbox[2] + bloat;
    stripPtr->header.y2 = stripPtr->bbox[3] + bloat;

}

/*
 *--------------------------------------------------------------
 *
 * TkFillStripchart --
 *
 *	This procedure is invoked to convert a stripchart to screen
 *	coordinates and display it using a particular GC.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvasPtr.
 *
 *--------------------------------------------------------------
 */

void
TkFillStripchart(canvasPtr, coordPtr, numPoints, drawable, gc)
    register Tk_Canvas *canvasPtr;	/* Canvas whose coordinate system
					 * is to be used for drawing. */
    double *coordPtr;			/* Array of coordinates for polygon:
					 * x1, y1, x2, y2, .... */
    int numPoints;			/* Twice this many coordinates are
					 * present at *coordPtr. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * stripchart. */
    GC gc;				/* Graphics context for drawing. */
{
    XPoint staticPoints[MAX_STATIC_POINTS];
    XPoint *pointPtr;
    register XPoint *pPtr;
    int i;

    /*
     * Build up an array of points in screen coordinates.  Use a
     * static array unless the stripchart has an enormous number of points;
     * in this case, dynamically allocate an array.
     */

    if (numPoints <= MAX_STATIC_POINTS) {
	pointPtr = staticPoints;
    } else {
	pointPtr = (XPoint *) ckalloc((unsigned) (numPoints * sizeof(XPoint)));
    }

    for (i = 0, pPtr = pointPtr; i < numPoints; i += 1, coordPtr += 2, pPtr++) {
	pPtr->x = SCREEN_X(canvasPtr, coordPtr[0]);
	pPtr->y = SCREEN_Y(canvasPtr, coordPtr[1]);
    }

    /*
     * Display stripchart, then free up stripchart storage if it was 
     * dynamically allocated.
     */

    XFillPolygon(Tk_Display(canvasPtr->tkwin), drawable, gc, pointPtr,
	    numPoints, Nonconvex, CoordModeOrigin);
    if (pointPtr != staticPoints) {
	ckfree((char *) pointPtr);
    }

}

/*
 *--------------------------------------------------------------
 *
 * DisplayStripchart --
 *
 *	This procedure is invoked to draw a stripchart item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvasPtr.
 *
 *--------------------------------------------------------------
 */

static void
DisplayStripchart(canvasPtr, itemPtr, drawable)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
{
    register StripchartItem *stripPtr = (StripchartItem *) itemPtr;
    Display *display = Tk_Display(canvasPtr->tkwin);
    int i, x1, y1, x2, y2;

    /*
     * Compute the screen coordinates of the bounding box for the item.
     * Make sure that the bbox is at least one pixel large, since some
     * X servers will die if it isn't.
     */

    x1 = SCREEN_X(canvasPtr, stripPtr->bbox[0]);
    y1 = SCREEN_Y(canvasPtr, stripPtr->bbox[1]);
    x2 = SCREEN_X(canvasPtr, stripPtr->bbox[2]);
    y2 = SCREEN_Y(canvasPtr, stripPtr->bbox[3]);
    if (x2 <= x1) {
	x2 = x1+1;
    }
    if (y2 <= y1) {
	y2 = y1+1;
    }
    /* Display the background of the stripchart bounding box */

    if (stripPtr->rectGC != None) {
	XFillRectangle(display, drawable, stripPtr->rectGC,
		       x1, y1, (unsigned int) (x2-x1), (unsigned int) (y2-y1));
    }
		   
    if (stripPtr->fillGC != None) {
	TkFillStripchart(canvasPtr, stripPtr->coordPtr, stripPtr->numPoints,
			 drawable, stripPtr->fillGC);
    }
    
    /* Display the border of the bounding box. */

    if (stripPtr->outlineGC != None) {
	XDrawRectangle(display, drawable, stripPtr->outlineGC,
		       x1, y1, (x2-x1-1), (y2-y1-1));
    }

    /* Display the scaleline. */

    if (stripPtr->scalelineGC != NULL){
	if (stripPtr->scale > 1) {
	    for (i = 1; i < stripPtr->scale; i++) {
		XDrawLine(display, drawable, stripPtr->scalelineGC,
			  x1+1, y2-1-i*(y2-y1-2)/stripPtr->scale, 
			  x2-1, y2-1-i*(y2-y1-2)/stripPtr->scale);
	    }
	}
    }
}

/*
 *--------------------------------------------------------------
 *
 * StripchartToPoint --
 *
 *	Computes the distance from a given point to a given
 *	stripchart, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are pointPtr[0] and pointPtr[1] is inside the stripchart.  If the
 *	point isn't inside the stripchart then the return value is the
 *	distance from the point to the stripchart.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
StripchartToPoint(canvasPtr, itemPtr, pointPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *pointPtr;		/* Pointer to x and y coordinates. */
{
    register StripchartItem *stripPtr = (StripchartItem *) itemPtr;
    double xDiff, yDiff, x1, y1, x2, y2, inc;

    /*
     * Generate a new larger rectangle that includes the border
     * width, if there is one.
     */

    x1 = stripPtr->bbox[0];
    y1 = stripPtr->bbox[1];
    x2 = stripPtr->bbox[2];
    y2 = stripPtr->bbox[3];
    if (stripPtr->outlineGC != None) {
	inc = stripPtr->outlineWidth/2.0;
	x1 -= inc;
	y1 -= inc;
	x2 += inc;
	y2 += inc;
    }

    /*
     * If the point is inside the rectangle, the distance is 0
     */

    if ((pointPtr[0] >= x1) && (pointPtr[0] < x2)
		&& (pointPtr[1] >= y1) && (pointPtr[1] < y2)) {
	return 0.0;
    }

    /*
     * Point is outside rectangle.
     */

    if (pointPtr[0] < x1) {
	xDiff = x1 - pointPtr[0];
    } else if (pointPtr[0] > x2)  {
	xDiff = pointPtr[0] - x2;
    } else {
	xDiff = 0;
    }

    if (pointPtr[1] < y1) {
	yDiff = y1 - pointPtr[1];
    } else if (pointPtr[1] > y2)  {
	yDiff = pointPtr[1] - y2;
    } else {
	yDiff = 0;
    }

    return hypot(xDiff, yDiff);
}

/*
 *--------------------------------------------------------------
 *
 * StripchartToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangular area.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by areaPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
StripchartToArea(canvasPtr, itemPtr, areaPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against stripchart. */
    double *areaPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
    register StripchartItem *stripPtr = (StripchartItem *) itemPtr;
    double halfWidth;

    halfWidth = stripPtr->outlineWidth/2.0;
    if (stripPtr->outlineGC == None) {
	halfWidth = 0.0;
    }

    if ((areaPtr[2] <= (stripPtr->bbox[0] - halfWidth))
	    || (areaPtr[0] >= (stripPtr->bbox[2] + halfWidth))
	    || (areaPtr[3] <= (stripPtr->bbox[1] - halfWidth))
	    || (areaPtr[1] >= (stripPtr->bbox[3] + halfWidth))) {
	return -1;
    }
    if ((areaPtr[0] <= (stripPtr->bbox[0] - halfWidth))
	    && (areaPtr[1] <= (stripPtr->bbox[1] - halfWidth))
	    && (areaPtr[2] >= (stripPtr->bbox[2] + halfWidth))
	    && (areaPtr[3] >= (stripPtr->bbox[3] + halfWidth))) {
	return 1;
    }
    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleStripchart --
 *
 *	This procedure is invoked to rescale a stripchart item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The stripchart referred to by itemPtr is rescaled so that the
 *	following transformation is applied to all point
 *	coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

static void
ScaleStripchart(canvasPtr, itemPtr, originX, originY, scaleX, scaleY)
    Tk_Canvas *canvasPtr;		/* Canvas containing stripchart. */
    Tk_Item *itemPtr;			/* Stripchart to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
{
    StripchartItem *stripPtr = (StripchartItem *) itemPtr;
    register double *coordPtr;
    int i;

    stripPtr->bbox[0] = originX + scaleX*(stripPtr->bbox[0] - originX);
    stripPtr->bbox[1] = originY + scaleY*(stripPtr->bbox[1] - originY);
    stripPtr->bbox[2] = originX + scaleX*(stripPtr->bbox[2] - originX);
    stripPtr->bbox[3] = originY + scaleY*(stripPtr->bbox[3] - originY);

    for (i = 0, coordPtr = stripPtr->coordPtr; i < stripPtr->numPoints;
	    i++, coordPtr += 2) {
	*coordPtr = originX + scaleX*(*coordPtr - originX);
	coordPtr[1] = originY + scaleY*(coordPtr[1] - originY);
    }
    ComputeStripchartBbox(canvasPtr, stripPtr);
}

/*
 *--------------------------------------------------------------
 *
 * TranslateStripchart --
 *
 *	This procedure is called to move a stripchart by a given
 *	amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the stripchart is offset by (xDelta, yDelta),
 *	and the bounding box is updated in the generic part of the
 *	item structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateStripchart(canvasPtr, itemPtr, deltaX, deltaY)
    Tk_Canvas *canvasPtr;		/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    register StripchartItem *stripPtr = (StripchartItem *) itemPtr;
    register double *coordPtr;
    int i;

    stripPtr->bbox[0] += deltaX;
    stripPtr->bbox[1] += deltaY;
    stripPtr->bbox[2] += deltaX;
    stripPtr->bbox[3] += deltaY;

    for (i = 0, coordPtr = stripPtr->coordPtr; i < stripPtr->numPoints;
	    i++, coordPtr += 2) {
	*coordPtr += deltaX;
	coordPtr[1] += deltaY;
    }
    ComputeStripchartBbox(canvasPtr, stripPtr);
}

/*
 *--------------------------------------------------------------
 *
 * StripchartToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	stripchart items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in canvasPtr->interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	item is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
StripchartToPostscript(canvasPtr, itemPtr, psInfoPtr)
    Tk_Canvas *canvasPtr;		/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    Tk_PostscriptInfo *psInfoPtr;	/* Information about the Postscript;
					 * must be passed back to Postscript
					 * utility procedures. */
{
    register StripchartItem *stripPtr = (StripchartItem *) itemPtr;
    char pathCmd[500], string[100];
    double y1, y2;
    int dash, i;

    y1 = TkCanvPsY(psInfoPtr, stripPtr->bbox[1]);
    y2 = TkCanvPsY(psInfoPtr, stripPtr->bbox[3]);

    /*
     * Generate a string that creates a path for the stripchart.
     * This is the only part of the procedure's code that is type-
     * specific.
     */


    sprintf(pathCmd, "%.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto %.15g 0 rlineto closepath\n",
                stripPtr->bbox[0], y1,
                stripPtr->bbox[2]-stripPtr->bbox[0], y2-y1,
                stripPtr->bbox[0]-stripPtr->bbox[2]);
    
    /*
     * First draw the filled area of the stripchart.
     */

    if (stripPtr->rectColor != NULL) {
        Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
        if (TkCanvPsColor(canvasPtr, psInfoPtr, stripPtr->rectColor)
                != TCL_OK) {
            return TCL_ERROR;
        }
      	Tcl_AppendResult(canvasPtr->interp, "fill\n", (char *) NULL);
    }

    /*
     * Now draw the outline, if there is one.
     */

    if (stripPtr->outlineColor != NULL) {
        Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
        sprintf(string, "%d setlinewidth", stripPtr->outlineWidth);
        Tcl_AppendResult(canvasPtr->interp, string,
                " 0 setlinejoin 2 setlinecap\n", (char *) NULL);
        if (TkCanvPsColor(canvasPtr, psInfoPtr, stripPtr->outlineColor)
                != TCL_OK) {
            return TCL_ERROR;
        }
        Tcl_AppendResult(canvasPtr->interp, "stroke\n", (char *) NULL);
    }

    /* 
     * Now draw the stripchart polygon, if there is one.
     */

    if (stripPtr->stripColor == NULL) {
        return TCL_OK;
    }

    /*
     * Generate a path for the polygon's outline.
     */

    TkCanvPsPath(canvasPtr->interp, stripPtr->coordPtr,
		 stripPtr->numPoints, psInfoPtr);

    /*
     * Fill the area of the polygon.
     */

    if (TkCanvPsColor(canvasPtr, psInfoPtr, stripPtr->stripColor) != TCL_OK) {
        return TCL_ERROR;
    }
    Tcl_AppendResult(canvasPtr->interp, "eofill\n", (char *) NULL);
    
     
    /*
     * Finally draw the scaleline, if there is one.
     */

    
    /*
     * Generate a path for the scaleline.
     */

    for (i = 1; i < stripPtr->scale; i++) {
	y1 = TkCanvPsY(psInfoPtr, (stripPtr->bbox[3]-1-i*
	         (stripPtr->bbox[3]-stripPtr->bbox[1]-2)/stripPtr->scale));

	sprintf(pathCmd, "%.15g %.15g moveto %.15g 0 rlineto\n",
		stripPtr->bbox[0]+stripPtr->scalelineWidth/2, y1, 
		stripPtr->bbox[2]-stripPtr->bbox[0]-
		    stripPtr->scalelineWidth);
    
	if (stripPtr->scalelineColor != NULL) {
	    Tcl_AppendResult(canvasPtr->interp, pathCmd, (char *) NULL);
	    sprintf(pathCmd, "%d setlinewidth", stripPtr->scalelineWidth);
	    if (stripPtr->scalelineStyle > 0) {
		dash = stripPtr->scalelineStyle;
		sprintf(string,
			" 0 setlinejoin 2 setlinecap [%d] 0 setdash\n", dash);
	    } else {
		sprintf(string, " 0 setlinejoin 2 setlinecap [] 0 setdash\n");
	    }
	    Tcl_AppendResult(canvasPtr->interp, pathCmd, string, 
			     (char *) NULL);
	    
	    if (TkCanvPsColor(canvasPtr, psInfoPtr, stripPtr->scalelineColor)
                != TCL_OK) {
		return TCL_ERROR;
	    }
	    Tcl_AppendResult(canvasPtr->interp, "stroke\n", (char *) NULL);
	}
    }
    return TCL_OK;
}

