#! /usr/local/bin/tclnm
##
## Some CISCO specific proc to dump the accounting table.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

source [info library]/init.tcl
set auto_path "/usr/local/lib/ined [info library]"
set auto_path "/usr/local/lib/tkined [info library]"
if [info exists env(HOME)] {
    set auto_path "$env(HOME)/.tkined $env(HOME)/.ined $auto_path"
}
if [info exists env(INED_PATH)] {
    set auto_path "$env(INED_PATH) $auto_path"
}
if [info exists env(TKINED_PATH)] {
    set auto_path "$env(TKINED_PATH) $auto_path"
}

foreach dir $auto_path {
    if {[file readable $dir/ined.tcl]} {
        source $dir/ined.tcl
    }
}

if [catch {ined size}] {
    puts stderr "failed to load the INED interface module"
    exit 1
}

set snmp_community "public"
set snmp_timeout "5000000"
set snmp_retries "3"

##
## Test if we are at home.
##

proc local {} {
    if {[catch {exec nslook [exec hostname]} ip]} {
	return 0
    }
    return [string match "134.169.34.*" $ip]
}

##
## Write a report to a log object.
##

proc writeln {{txt ""}} {
    static log

    if {![info exists log]} {
        set log [ined -noupdate create LOG]
        ined -noupdate name $log "SNMP CISCO Log"
    }

    if {[ined -noupdate retrieve $log] == ""} {
        set log [ined -noupdate create LOG]
        ined -noupdate name $log "SNMP CISCO Log"
    }

    if {$txt == ""} { 
	ined append $log "" 
    } else {
	foreach line [split $txt "\n"] {
	    ined append $log [string trim $line "\r"]
	}
    }
}


## Get the IP Address of a node. Query the name server, if the
## address attribute is not set to something that looks like a
## valid IP address.

proc Get_IP_Address {node} {
    if {[ined_type $node]=="NODE"} {
        set host [ined_name $node]
        set ip [ined_address $node]
        if {[regexp "^\[0-9\]+\.\[0-9\]+\.\[0-9\]+\.\[0-9\]+$" $ip]>0} {
            return $ip
        }
        if {[catch {nslook $host} ip]==0} {
            return [lindex $ip 0]
        }
    }
    return ""
}

##
## Open a snmp session and initialize it to the defaults stored in
## the global variables snmp_community, snmp_timeout, snmp_retries.
##

proc snmp_open {ip} {
    global snmp_community snmp_timeout snmp_retries
    set s [snmp open $ip]
    snmp default $s community $snmp_community
    snmp default $s timeout $snmp_timeout
    snmp default $s retry $snmp_retries
    return $s
}

##
## Dump a hierarchy of the MIB and return the result as a list 
## of strings.
##

proc "dump" {ip path} {
    set s [snmp_open $ip]
    set op ""
    catch {
	snmp getbulk $s {$path} x {
	    set oid [lindex $x 0]
	    set typ [lindex $x 1]
	    set val [lindex $x 2]
	    # get the oid from the MIB
	    set l [mib oid $oid]
	    set p [lindex $l 0]
	    if {$p == $op} {
		writeln "\tvalue\[[lindex $l 1]\] =\t$val"
		continue;
	    }
	    set op $p
	    writeln
	    writeln [lindex $l 0]
	    writeln "$oid $typ"
	    writeln "\tvalue\[[lindex $l 1]\] =\t$val"
	}
    } error
    if {$error != ""} {
	ined acknowledge "Dumping MIB tree failed for $ip."
    }
    catch {snmp close $s}
}

##
## Sort a list in numerical order. The first argument is the list
## to be sorted and the second argument gives the index of the 
## element that will be used as the key.
##

proc numsort {list index} {
    set result ""
    foreach elem $list {
	set key [lindex $elem $index]
	set i 0
	foreach aa $result {
	    if {[lindex $aa $index]>=$key} break
	    incr i
	}
	set result [linsert $result $i $elem]
    }
    return $result
}

##
## Display the system MIB of CISCO devices.
##

proc "system information" {list} {

    set path ".iso.org.dod.internet.private.enterprises.cisco.local.lsystem"

    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
	    writeln "MIB dump for $host starting at $path:"
	    dump $ip $path
	    writeln
        }
    }
}

##
## Dump the accounting table of the cisco MIB.
##

proc "accounting table" {list} {
    set ciscomib .iso.org.dod.internet.private.enterprises.cisco
    foreach comp $list {
        if {[lindex $comp 0]=="NODE"} {
            set host [lindex [lindex $comp 2] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
            set s [snmp_open $ip]
	    if {[catch {snmp get $s -l -v $ciscomib.local.lip.actAge.0} age]} {
		ined acknowledge "Can not get accounting information of $host."
		continue
	    }
            snmp default $s prefix \
                "$ciscomib.local.lip.lipAccountingTable.lipAccountEntry"
	    writeln "CISCO accounting statistics of $host ([join $age]):"
	    set table ""
	    catch {
		snmp getbulk $s -l -v {
		    actSrc
		    actDst
		    actPkts
		    actByts
		} row {
		    append table [join $row]
		}
	    } error
	    if {$error != ""} {
		ined acknowledge "Reading accounting table failed for $host."
	    } else {
		writeln "before numsort"
		set table [numsort $table 3]
		writeln "Source       Destination      Packets      Bytes"
		foreach row $table {
		    writeln [eval "format {%-16s %-16s %16s %16s} $row"]
		}
	    }
	    catch {snmp close $s}
	    writeln
        }
    }
}

proc "checkpoint accounting table" {list} {
    set ciscomib .iso.org.dod.internet.private.enterprises.cisco
    foreach comp $list {
        if {[lindex $comp 0]=="NODE"} {
            set host [lindex [lindex $comp 2] 0]
            set ip [Get_IP_Address $comp]
            if {$ip==""} {
                ined acknowledge "Can not lookup IP Address for $host."
                continue;
            }
            set s [snmp_open $ip]
	    if {[catch {snmp get $s -l -v $ciscomib.local.lip.ckactAge.0} age]} {
		ined acknowledge "Can not get accounting information of $host."
		continue
	    }
            snmp default $s prefix \
                "$ciscomib.local.lip.lipCkAccountingTable.lipCkAccountEntry"
	    writeln "CISCO checkpoint accounting statistics of $host ([join $age]):"
	    set table ""
	    catch {
		snmp getbulk $s -l -v {
		    ckactSrc
		    ckactDst
		    ckactPkts
		    ckactByts
		} row {
		    append table [join $row]
		}
	    } error
	    if {$error != ""} {
		ined acknowledge "Reading accounting table failed for $host."
	    } else {
		writeln "before numsort"
		set table [numsort $table 3]
		writeln "Source       Destination      Packets      Bytes"
		foreach row $table {
		    writeln [eval "format {%-16s %-16s %16s %16s} $row"]
		}
	    }
	    catch {snmp close $s}
	    writeln
        }
    }
}

##
## Set the parameters (community, timeout, retry) for snmp requests.
##

proc "set parameter" {list} {
    global snmp_community snmp_timeout snmp_retries

    set result [ined request "SNMP Parameter" \
		"{ {Community:} $snmp_community } \
                 { {Timeout \[us\]:} $snmp_timeout } \
                 { {Retries:} $snmp_retries scale 1 8} " ]

    if {$result==""} return

    set snmp_community [lindex $result 0]
    set snmp_timeout   [lindex $result 1]
    set snmp_retries   [lindex $result 2]
}

##
## Delete the tools created by this interpreter.
##

proc "delete SNMP CISCO" {list} {
    global tools
    foreach id $tools { ined delete $id }
    exit
}

if {![local]} {
    ined acknowledge "CISCO is still under construction. Use with care!!"
}

set tools [ ined create TOOL "SNMP CISCO" "system information" "" \
	    "accounting table" "checkpoint accounting table" "" \
	    "set parameter" "" \
	    "delete SNMP CISCO"]

ined_main_loop
