#! /usr/local/bin/tclnm
##
## Simple SNMP monitoring utilities for tkined.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

source [info library]/init.tcl
set auto_path "/usr/local/lib/ined [info library]"
set auto_path "/usr/local/lib/tkined [info library]"
if [info exists env(HOME)] {
    set auto_path "$env(HOME)/.tkined $env(HOME)/.ined $auto_path"
}
if [info exists env(INED_PATH)] {
    set auto_path "$env(INED_PATH) $auto_path"
}
if [info exists env(TKINED_PATH)] {
    set auto_path "$env(TKINED_PATH) $auto_path"
}

foreach dir $auto_path {
    if {[file readable $dir/ined.tcl]} {
        source $dir/ined.tcl
    }
}

if [catch {ined size}] {
    puts stderr "failed to load the INED interface module"
    exit 1
}

set snmp_community public
set snmp_timeout 5000000
set snmp_retries 3

set intervall 10
set clone true
set clone_ids ""

##
## Test if we are at home.
##

proc local {} {
    if {[catch {exec nslook [exec hostname]} ip]} {
	return 0
    }
    return [string match "134.169.34.*" $ip]
}

##
## Write a report to a log object.
##

proc writeln {{txt ""}} {
    static log

    if {![info exists log]} {
        set log [ined -noupdate create LOG]
        ined -noupdate name $log "SNMP Monitor Log"
    }

    if {[ined -noupdate retrieve $log] == ""} {
        set log [ined -noupdate create LOG]
        ined -noupdate name $log "SNMP Monitor Log"
    }

    if {$txt == ""} { 
	ined append $log "" 
    } else {
	foreach line [split $txt "\n"] {
	    ined append $log [string trim $line "\r"]
	}
    }
}


## Get the IP Address of a node. Query the name server, if the
## address attribute is not set to something that looks like a
## valid IP address.

proc Get_IP_Address {node} {
    if {[ined_type $node]=="NODE"} {
        set host [ined_name $node]
        set ip [ined_address $node]
        if {[regexp "^\[0-9\]+\.\[0-9\]+\.\[0-9\]+\.\[0-9\]+$" $ip]>0} {
            return $ip
        }
        if {[catch {nslook $host} ip]==0} {
            return [lindex $ip 0]
        }
    }
    return ""
}

##
## Clone a node object. Return the new id.
##

proc clone {id {offset_x 20} {offset_y 20}} {
    global clone
    global clone_ids

    if {$clone != "true"} { return $id }

    set new [ined -noupdate create NODE]
    ined -noupdate name $new    [ined -noupdate name $id]
    ined -noupdate color $new   [ined -noupdate color $id]
    ined -noupdate font $new    [ined -noupdate font $id]
    ined -noupdate label $new   [ined -noupdate label $id]
    ined -noupdate icon $new    [ined -noupdate icon $id]
    set xy [ined -noupdate move $id]
    set x [expr {[lindex $xy 0]+$offset_x}]
    set y [expr {[lindex $xy 1]+$offset_y}]
    ined -noupdate move $new $x $y

    lappend clone_ids $new

    return $new
}

##
## Open a snmp session and initialize it to the defaults stored in
## the global variables snmp_community, snmp_timeout, snmp_retries.
##

proc snmp_open {ip} {
    global snmp_community snmp_timeout snmp_retries
    set s [snmp open $ip]
    snmp default $s community $snmp_community
    snmp default $s timeout $snmp_timeout
    snmp default $s retry $snmp_retries
    return $s
}

##
## Monitor a SNMP variable in a stripchart. Not complete yet.
##

proc "monitor variable" {list} {

    set res [ined request "Tell what I should show you:" \
	     "{ {SNMP Variable:} interfaces.ifTable.ifEntry.ifInOctets.1 } \
              { {Intervall \[s\]:} 10 } \
              { {Scaling Factor:} 1.0 } " ]
    if {$res == ""} return
    set varname   [lindex $res 0]
    set intervall [lindex $res 1]
    set scale     [lindex $res 2]

    set at_leat_one 0
    foreach comp $list {
        if {[ined_type $comp]=="NODE"} {
	    set id [ined_id $comp]
            set host [lindex [ined_name $comp] 0]
            set ip [Get_IP_Address $comp]
	    if {$ip==""} {
		ined acknowledge "Can not lookup IP Address for $host."
		continue;
	    }
	    set id [clone $id 30 30]
	    set snmp($id) [snmp_open $ip]
	    snmp get $snmp($id) $varname snmpvar
	    set type [lindex $snmpvar 1]
	    if {[lsearch "Gauge Counter INTEGER" $type] < 0} {
		ined acknowledge \
		    "Sorry, can not display SNMP objects of type $type."
		return
	    }
	    set value($id) [lindex $snmpvar 2]
	    set time($id) [getclock]
	    ined stripchart $id create
	    ined label $id text $varname
	    incr at_leat_one
	}
    }

    while {$at_leat_one>0} {
	sleep $intervall
	foreach id [array names snmp] {
	    snmp get $snmp($id) $varname snmpvar
	    set now [getclock]
	    set type [lindex $snmpvar 1]
	    if {$type == "GAUGE"} {
		set val [lindex $snmpvar 2]
	    } else {
		set val [expr {([lindex $snmpvar 2] - $value($id))}]
	    }
	    set val [expr {$scale * $val / ( $now-$time($id) )}]
	    ined stripchart $id $val
	    set value($id) [lindex $snmpvar 2]
	    set time($id) $now
	}
    }
}

##
## Display the jobs currently running.
##

proc "monitor job info" {list} {

    set jobs [ined_list_jobs]

    if {$jobs == ""} {
	ined acknowledge "Sorry, no jobs available."
	return
    }
    
    set result ""
    set len 0
    foreach job $jobs {

	set jobcmd [lindex $job 1]

	# Convert the id's to hostnames for readability.
	
	set hosts ""
	foreach id [lindex $jobcmd 1] {
	    lappend hosts [lindex [ined name $id] 0]
	}
	if {[llength $hosts] > 3} {
	    set hosts "[lrange $hosts 0 2] ..."
	}
	
	set line \
	     [format "%3d %6.1f %6.1f %3d %8s %s %s" \
	     [lindex $job 0] [lindex $job 2] [lindex $job 3] \
	     [lindex $job 4] [lindex $job 5] [lindex $jobcmd 0] $hosts ]

	if {[string length $line] > $len} {
	    set len [string length $line]
	}

	lappend result $line
    }

    set header " ID   INTV   REM  CNT  STATUS    CMD"

    for {set i [string length $header]} {$i < $len} {incr i} {
	append header " "
    }

    ined browse $header $result

}

##
## Modify the state or the intervall of a running job.
##

proc "modify monitor job" {list} {

    # Ask for the job to modify.

    for {
	set jobnr [ined_select_job modify]
    } {
	$jobnr != ""
    } {
	set jobnr [ined_select_job modify]
    } {

	# Get the details about the selected job.

	set jobcmd ""
	set jobitv ""
	foreach job [ined_list_jobs] {
	    if {$jobnr == [lindex $job 0]} {
		set jobcmd [lindex $job 1]
		set jobitv [lindex $job 2]
		set jobstat [lindex $job 5]
	    }
	}
	
	# Convert the id's to hostnames for readability.
	
	set hosts ""
	foreach id [lindex $jobcmd 1] {
	    lappend hosts [lindex [ined -noupdate name $id] 0]
	}
	if {[llength $hosts] > 3} {
	    set hosts "[lrange $hosts 0 2] ..."
	}
	
	# Request for changes.
	
	set res [ined request "Modify job 0 ([lindex $jobcmd 0] on $hosts)" \
		 "{{Intervaltime \[s\]:} $jobitv scale 1 120} \
                  {{Job Status:} $jobstat radio waiting suspend} "]
	if {$res == ""} continue

	set jobitv  [lindex $res 0]
	set jobstat [lindex $res 1]
	
	if {$jobstat == "waiting"} { ined_resume_job  $jobnr }
	if {$jobstat == "suspend"} { ined_suspend_job $jobnr }
	
	ined_modify_job $jobnr $jobcmd $jobitv
    }
}

##
## Set the parameters (community, timeout, retry) for snmp requests.
##

proc "set parameter" {list} {
    global snmp_community snmp_timeout snmp_retries intervall clone

    set result [ined request "SNMP Parameter" \
		"{ {Community:} $snmp_community } \
                 { {Timeout \[us\]:} $snmp_timeout } \
                 { {Retries:} $snmp_retries scale 1 8} \
                 { {Intervall \[s\]:} $intervall scale 1 120} \
                 { {Clone nodes:} $clone radio true false} "]

    if {$result==""} return

    set snmp_community [lindex $result 0]
    set snmp_timeout   [lindex $result 1]
    set snmp_retries   [lindex $result 2]
    set intervall      [lindex $result 3]
    set clone          [lindex $result 3]
}

##
## Display some help about this tool.
##

proc "help SNMP Monitor" {list} {
    ined browse "Help about SNMP Monitor" {
	"monitor variable:" 
	"    Monitor a SNMP variable in a stripchart." 
	"" 
	"monitor job info:" 
	"    This command display information about all monitoring jobs" 
	"    started by this monitor script." 
	"" 
	"modify monitor job:" 
	"    Select one of the monitoring jobs and modify it. You can change" 
	"    the sampling interval and switch the state from waiting (which" 
	"    means active) to suspended." 
	"" 
	"set parameter:" 
	"    This dialog allows you to set the sampling interval and " 
	"    whether the monitoring commands should clone the selected " 
	"    nodes to display the status in a graphical chart. You can" 
	"    also set SNMP parameter like retries and community names." 
    }
}

##
## Delete the tools created by this interpreter.
##

proc "delete SNMP Monitor" {list} {
    global tools

    if {[ined_list_jobs] != ""} {
	if {[ined confirm "Kill running monitoring jobs?"] != "yes"} return
    }

    foreach id $tools { ined delete $id }
    exit
}

if {![local]} {
    ined acknowledge "SNMP is still under construction. Use with care!!"
}

set tools [ ined create TOOL "SNMP Monitor" "monitor variable" "" \
	   "monitor job info" "modify monitor job" "" \
	   "set parameter" "" \
	   "help SNMP Monitor" "delete SNMP Monitor" ]

ined_main_loop
