/* 
 * This file contains the main program for "tkined". It is heavily
 * based on main.c from the tk distribution by John Ousterhout.
 *
 * Copyright (c) 1993, 1994
 *                    J. Schoenwaelder
 *                    TU Braunschweig, Germany
 *                    Institute for Operating Systems and Computer Networks
 *
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>

#include "tkined.h"

#include "tkined_editor.c"
#include "tkined_commands.c"
#include "tkined_tools.c"
#include "tkined_tcl.c"
#include "tkined_dialog.c"
#include "tkined_help.c"
#include "tkined_graph.c"
#include "tkined_init.c"

/*
 * Global variables. The Interpreter and a window that is not used :-)
 */

Tk_Window w;		/* The main window for the application.  If
				 * NULL then the application no longer
				 * exists. */
static Tcl_Interp *interp;	/* Interpreter for this application. */

/*
 * Command-line options:
 */

int synchronize = 0;
int debug = 0;
char *display = NULL;
char *geometry = NULL;

Tk_ArgvInfo argTable[] = {
    {"-geometry", TK_ARGV_STRING, (char *) NULL, (char *) &geometry,
	"Initial geometry for window"},
    {"-display", TK_ARGV_STRING, (char *) NULL, (char *) &display,
	"Display to use"},
    {"-sync", TK_ARGV_CONSTANT, (char *) 1, (char *) &synchronize,
	"Use synchronous mode for display server"},
    {"-debug", TK_ARGV_CONSTANT, (char *) 1, (char *) &debug,
	"Turn on the debugging output"},
    {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL,
	(char *) NULL}
};

/*
 * The Tcl_AppInit() function as suggested by J. Ousterhout.
 */

int
Tcl_AppInit(interp)
    Tcl_Interp *interp;
{
    if (Tcl_Init(interp) == TCL_ERROR) {
        return TCL_ERROR;
    }

    if (Tk_Init(interp) == TCL_ERROR) {
        return TCL_ERROR;
    }

    if (Tkined_Init(interp) == TCL_ERROR) {
        return TCL_ERROR;
    }

    return TCL_OK;
}

/*
 * Main program for tkined.
 */

int
main(argc, argv)
    int argc;				/* Number of arguments. */
    char **argv;			/* Array of argument strings. */
{
    char *msg;
    char buf[20];
    Tk_3DBorder border;
    int done = 0;
    char *file;

    interp = Tcl_CreateInterp();

    /* Initialize the stripchart and barchart canvas objects. */

    Tk_CreateItemType((Tk_ItemType *) &TkStripchartType);
    Tk_CreateItemType((Tk_ItemType *) &TkBarchartType);

    /*
     * Parse command-line arguments.
     */

    if (Tk_ParseArgv(interp, (Tk_Window) NULL, &argc, argv, argTable, 0)
	    != TCL_OK) {
	fprintf(stderr, "%s\n", interp->result);
	exit(1);
    }

    /*
     * If a display was specified, put it into the DISPLAY
     * environment variable so that it will be available for
     * any sub-processes created by us.
     */

    if (display != NULL) {
        Tcl_SetVar2(interp, "env", "DISPLAY", display, TCL_GLOBAL_ONLY);
    }

    /*
     * Initialize the Tk application and arrange to map the main window
     * after the startup script has been executed, if any.  This way
     * the script can withdraw the window so it isn't ever mapped
     * at all.
     */

    w = Tk_CreateMainWindow(interp, display, "tkined", "tkined");
    if (w == NULL) {
	fprintf(stderr, "%s\n", interp->result);
	exit(1);
    }
    Tk_SetClass(w, "Tkined");
    if (synchronize) {
	XSynchronize(Tk_Display(w), True);
    }
    Tk_GeometryRequest(w, 200, 200);
    border = Tk_Get3DBorder(interp, w, None, "#ffe4c4");
    if (border == NULL) {
	Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);
	Tk_SetWindowBackground(w, WhitePixelOfScreen(Tk_Screen(w)));
    } else {
	Tk_SetBackgroundFromBorder(w, border);
    }
    XSetForeground(Tk_Display(w), DefaultGCOfScreen(Tk_Screen(w)),
	    BlackPixelOfScreen(Tk_Screen(w)));

    /*
     * Set up the global tkined variables tkined_version, tkined_path
     * and tkined_debug.
     */

    Tcl_SetVar(interp, "tkined_version", TKINED_VERSION, TCL_GLOBAL_ONLY);
    Tcl_SetVar(interp, "tkined_path", "./ ./Bitmaps", TCL_GLOBAL_ONLY);
    if ((msg = getenv("TKINED_PATH")) != NULL) {
	Tcl_SetVar(interp, "tkined_path", msg, 
		   TCL_GLOBAL_ONLY | TCL_APPEND_VALUE | TCL_LIST_ELEMENT);
    }
    Tcl_SetVar(interp, "tkined_path", "~/.tkined", 
	       TCL_GLOBAL_ONLY | TCL_APPEND_VALUE | TCL_LIST_ELEMENT);
    Tcl_SetVar(interp, "tkined_path", TKINEDLIB, 
	       TCL_GLOBAL_ONLY | TCL_APPEND_VALUE | TCL_LIST_ELEMENT);
    sprintf(buf, "%d", debug);
    Tcl_SetVar(interp, "tkined_debug", buf, TCL_GLOBAL_ONLY);
    if (geometry != NULL) {
	Tcl_SetVar(interp, "geometry", geometry, TCL_GLOBAL_ONLY);
    }

    /* Adjust the auto_path to take care of $HOME/.tkined
       and TKINED_PATH. */

    Tcl_Eval (interp, 
	      "set auto_path \"/usr/local/lib/tkined [info library]\"");
    Tcl_VarEval (interp, "set auto_path \"", TKINEDLIB, " $auto_path\"",
		 (char *) NULL);
    Tcl_VarEval (interp, "if [info exists env(HOME)] { ",
		 "set auto_path \"$env(HOME)/.tkined $auto_path\"}",
		 (char *) NULL);
    Tcl_VarEval (interp, "if [info exists env(TKINED_PATH)] {",
		 "set auto_path \"$env(TKINED_PATH) $auto_path\"}",
		 (char *) NULL);

    if (Tcl_AppInit(interp) != TCL_OK) {
	fprintf (stderr, "%s\n", interp->result);
        return 1;
    }

    /*
     * Execute the initialization scripts.
     */

    if (Tcl_Eval(interp, tkined_editor_tcl) != TCL_OK) 
	    fprintf (stderr, "%s\n", interp->result);
    if (Tcl_Eval(interp, tkined_commands_tcl) != TCL_OK)
            fprintf (stderr, "%s\n", interp->result);
    if (Tcl_Eval(interp, tkined_tools_tcl) != TCL_OK)
            fprintf (stderr, "%s\n", interp->result);
    if (Tcl_Eval(interp, tkined_tcl_tcl) != TCL_OK)
            fprintf (stderr, "%s\n", interp->result);
    if (Tcl_Eval(interp, tkined_dialog_tcl) != TCL_OK)
            fprintf (stderr, "%s\n", interp->result);
    if (Tcl_Eval(interp, tkined_help_tcl) != TCL_OK)
            fprintf (stderr, "%s\n", interp->result);
    if (Tcl_Eval(interp, tkined_graph_tcl) != TCL_OK)
            fprintf (stderr, "%s\n", interp->result);
    if (Tcl_Eval(interp, tkined_init_tcl) != TCL_OK)
            fprintf (stderr, "%s\n", interp->result);

    (void) Tcl_Eval(interp, "wm withdraw .");
    (void) Tcl_Eval(interp, "update");

    /*
     * Open a view for every readable file given on the command line.
     */

    for (file = *(++argv); file != NULL; file = *(++argv)) {
	if (access(file, R_OK) < 0) {
	    fprintf(stderr, "tkined: unable to open %s\n", file);
	    continue;
	}
	(void) Tcl_Eval(interp, "tkined_new_view_command");
	(void) Tcl_VarEval(interp, "tkined_open_command ",
			   interp->result, " ", file, (char *) NULL);
	done++;
    }
    fflush(stdout);

    if (!done) {
	(void) Tcl_Eval(interp, "tkined_new_view_command");
    }

    /*
     * Loop infinitely, waiting for commands to execute.  When there
     * are no windows left, Tk_MainLoop returns and we clean up and
     * exit.
     */

    Tk_MainLoop();
    Tcl_DeleteInterp(interp);
    return 0;
}
