##
## tkined_commands.tcl
##
## This file contains the commands of the ined editor. Commands are
## invoked from the pulldown menus or by the editor tools. Most
## commands simply call the corresponding proc for every selected
## object. Commands never change anything on the canvas directly.
## For most commands, the undo/redo buffer is modified with an
## inverse command.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

##
## Reinitialise the editor. Retrieve all objects of the canvas
## and delete them (ignore INTERPRETER and TOOL objects). Make
## sure that grouped objects get deleted since a delete on a group
## object just deletes the group and its members.
##

proc tkined_new_command {w} {
    set done 0
    while {! $done} {
	set done 1
	foreach id [tkined_retrieve $w.canvas] {
	    if {[catch {$id type} type]} continue
	    if {$type == "GROUP"} { set done 0 }
	    if {$type != "INTERPRETER" && $type != "TOOL"} {
		catch {$id delete}
	    }
	}
    }
    tkined_editor_attribute $w filename "unknown"
    wm title $w "tkined: [tkined_editor_attribute $w filename]"
}

##
## Read in a saved tkined map. The file just contains a series of
## ined commands that will rebuild the map. Perhaps we should take
## some care not to evaluate everything we get ...
##

proc tkined_merge_command {w {fname {}}} {
    global current_canvas
    set current_canvas $w.canvas
    if {$fname==""} {
	set dir [tkined_editor_attribute $w dirname]
	set fname [tkined_file_select $w.canvas "Select a file to open:" $dir]
	if {$fname==""} return
    }
    tkined_editor_attribute $w dirname [file dirname $fname]
    tkined_editor_attribute $w filename [file tail $fname]
    if {[catch {open $fname} fh]} {
	tkined_acknowledge $w.canvas \
	    "Can not open file [file tail $fname]."
	return
    }
    $w.menu.file configure -state disabled; update idletasks;
    while {![eof $fh]} {
	gets $fh line
	if {[catch {eval $line} err]} {
	    tkined_acknowledge $w.canvas \
		"Can not load file [file tail $fname]." "$err"
	    break
	}
    }
    close $fh
    wm title $w "tkined: [tkined_editor_attribute $w filename]"
    $w.menu.file configure -state normal
}

proc tkined_open_command {w {fname {}}} {
    tkined_new_command $w
    tkined_merge_command $w $fname
}

##
## Save the current map. The filename is found in the attribute
## named filename.
##

proc tkined_save_command {w} {
    global tkined_version
    set fname [tkined_editor_attribute $w dirname]
    append fname "/" [tkined_editor_attribute $w filename]
    if {[file exists $fname]} {
	if {![file writable $fname]} {
	    tkined_acknowledge $w.canvas "Can not write $fname"
	    return
	}
	set res [tkined_confirm $w.canvas "Replace file $fname?"]
	if {($res == "cancel") || ($res == "no")} return
    }
    set file [open $fname w]
    puts -nonewline $file \
	"## This file was created by tkined version $tkined_version."
    puts $file "      >> DO NOT EDIT <<"
    puts $file ""
    puts $file "ined page [tkined_page $w.canvas]"
    puts $file ""
    set idlist [tkined_retrieve $w.canvas]
    set collids ""
    foreach id $idlist {
	if {[$id type] == "GROUP"} {
	    if {[$id collapsed]} {
		lappend collids $id
	    }
	}
    }
    foreach id $collids { $id expand }
    set idlist [tkined_retrieve $w.canvas]
    foreach id $idlist {
	if {[lsearch {NODE NETWORK TEXT IMAGE} [$id type]] >= 0} {
	    puts $file [$id dump]
	}
    }
    foreach id $idlist {
	if {[$id type] == "LINK"} {
	    puts $file [$id dump]
	}
    }
    foreach id $idlist {
	if {[$id type] == "GROUP"} {
	    puts $file [$id dump]
	}
    }
    close $file
    foreach id $collids { $id collapse }
}

##
## Save the map under a new filename. Get a new filename
## and call tkined_save_command to do the job.
##

proc tkined_save_as_command {w} {
    set dir [tkined_editor_attribute $w dirname]
    set fname [tkined_file_select $w.canvas "Save as file:" $dir]
    if {$fname==""} return
    tkined_editor_attribute $w dirname [file dirname $fname]
    tkined_editor_attribute $w filename [file tail $fname]
    wm title $w "tkined: [file tail $fname]"
    tkined_save_command $w
}

##
## Dump the current map in postscript format. Before we start,
## we must set the background of all bitmaps to white. Otherwise
## we would get transparent bitmaps :-<.
##

proc tkined_postscript_command {w {fname {}}} {

    global tkined_ps_map
    set tkined_ps_map(fixed) [list Courier 10]

    if {$fname == ""} {

	set dir [tkined_editor_attribute $w psdirname]
	set fname [tkined_file_select $w.canvas \
		   "Write PostScript to file:" $dir]
	if {$fname == ""} return
	while {[file exists $fname] && ![file writable $fname]} {
	    set fname [tkined_file_select $w.canvas \
		       "Write PostScript to file:" $dir]
	    if {$fname == ""} return
	}
	tkined_editor_attribute $w psdirname [file dirname $fname]
	
	if {[file exists $fname]} {
	    if {![file writable $fname]} {
		tkined_acknowledge $w.canvas "Can not write $fname"
		return
	    }
	    set res [tkined_confirm $w.canvas "Replace file $fname?"]
	    if {($res == "cancel") || ($res == "no")} return
	}
    }

    set size [tkined_editor_attribute $w pageSize]
    set orientation [tkined_editor_attribute $w pageOrientation]
    foreach option [$w.canvas config] {
	if {[lindex $option 0] == "-scrollregion"} {
	    set i [llength $option]
	    incr i -1
	    set scrolloption [lindex $option $i]
	    set width  [lindex $scrolloption 2]
	    set height [lindex $scrolloption 3]
	}
    }
    foreach item [$w.canvas find all] {
	switch [$w.canvas type $item] {
	    bitmap {
		$w.canvas itemconfigure $item -background White
	    }
	    stripchart {
		$w.canvas itemconfigure $item -background White
	    }
	    barchart {
		$w.canvas itemconfigure $item -background White
	    }
	}
	if {[lsearch [$w.canvas gettags $item] labelclipbox]>=0} {
	    $w.canvas itemconfigure $item -fill White -outline White
	}
    }
    update
    if {[catch {$w.canvas postscript -file $fname \
            -fontmap tkined_ps_map \
	    -pageheight 210m -x 0 -y 0 \
	    -height $height -width $width \
            -rotate [expr {$orientation == "landscape"}]} err]} {
        puts stderr "generating postscript failed: $err"
	flush stderr
    }
    set color [lindex [$w.canvas configure -background] 4]
    foreach item [$w.canvas find all] {
	switch [$w.canvas type $item] {
	    bitmap {
		$w.canvas itemconfigure $item -background ""
	    }
	    stripchart {
		$w.canvas itemconfigure $item -background $color
	    }
	    barchart {
		$w.canvas itemconfigure $item -background $color
	    }
	}
	if {[lsearch [$w.canvas gettags $item] labelclipbox]>=0} {
	    $w.canvas itemconfigure $item -fill $color -outline $color
	}
    }
}

##
## Print a postscript dump. Use the proc above to get a postscript dump
## and push it to the printer.
##

proc tkined_print_command {w} {

    set fname "/tmp/tkined.ps"
    catch {exec /bin/rm -f $fname}
    if {[file exists $fname] && ![file writable $fname]} {
	tkined_acknowledge $w.canvas "Can not write temporary file $fname"
	return
    }

    tkined_postscript_command $w $fname

    tkined_print $w.canvas $fname

    catch {exec /bin/rm -f $fname}
}

##
## Import a X11 bitmap as a background image. Since tk can not
## store bitmaps as strings, we must handle the filename.
##

proc tkined_import_command {w} {
    set dir [tkined_editor_attribute $w dirname]
    set fname [tkined_file_select $w.canvas "Import file:" $dir]
    if {$fname!=""} {
	if {[catch {IMAGE create $fname} image]} return
	$image canvas $w.canvas
	$image color [tkined_editor_attribute $w color]
    }
}

##
## This should better be called kill, since we do not care
## about anything... Ok. We should try to delete all active
## interpreters first.
##

proc tkined_quit_command {w} {
    set killme ""
    foreach id [tkined_retrieve] {
	if {[$id type] == "INTERPRETER"} {
	    lappend killme $id
	}
    }
    foreach id $killme {
	$id delete
    }
    destroy .
}

##
## Here are all commands used in the edit menu.
##

proc tkined_undo_command {w {body ""}} {
    static undolist
    if {![info exists undolist($w)]} { set undolist($w) "{}" }
    if {$body!=""} {
	set undolist($w) "{$body} $undolist($w)"
    } else {
	if {[llength $undolist($w)]>1} {
	    set cmd [lindex $undolist($w) 0]
	    set undolist($w) [lrange $undolist($w) 1 end]
	    if {![catch {eval [lindex $cmd 0]}]} {
		tkined_redo_command $w $cmd
	    }
	}
    }
}

proc tkined_redo_command {w {body ""}} {
    static redolist
    if {![info exists redolist($w)]} { set redolist($w) "{}" }
    if {$body!=""} {
        set redolist($w) "{$body} $redolist($w)"
    } else {
        if {[llength $redolist($w)]>1} {
	    set cmd [lindex $redolist($w) 0]
            set redolist($w) [lrange $redolist($w) 1 end]
            if {![catch {eval [lindex $cmd 1]}]} {
		tkined_undo_command $w $cmd
	    }
        }
    }
}

proc tkined_delete_command {w} {
    foreach id [tkined_selection $w.canvas] {
	catch {$id delete}
    }
}

proc tkined_cut_command {w} {
    tkined_copy_command $w
    tkined_delete_command $w
}

proc tkined_copy_command {w} {
    global tkined_clip
    set tkined_clip ""
    set selection [tkined_selection $w.canvas]
    foreach id $selection {
	if {[lsearch {NODE NETWORK TEXT} [$id type]] >= 0} {
	    append tkined_clip [$id dump]
	}
    }
    foreach id $selection {
	if {[$id type] == "LINK"} {
	    append tkined_clip [$id dump]
	}
    }
    foreach id $selection {
	if {[$id type] == "GROUP"} {
	    foreach m [$id member] {
		append tkined_clip \
		    "if \[catch {$m id} $m\] \{ set $m \"\" \}\n"
	    }
	    append tkined_clip [$id dump]
	}
    }
}

proc tkined_paste_command {w} {
    global tkined_clip
    global current_canvas
    set current_canvas $w.canvas
    if {![info exists tkined_clip]} return
    eval $tkined_clip
}

##
## The label sub-menu contains commands to set the label contents.
##

proc tkined_label_command {w what} {
    foreach id [tkined_selection $w.canvas] {
	catch {$id label $what}
    }
}

##
## The select menu contains a set of commands to modify 
## the current selection.
##

proc tkined_select_all_command {w} {
    foreach id [tkined_retrieve $w.canvas] {
	if {[$id type] == "IMAGE"} continue
	catch {$id select}
    }
}

proc tkined_select_neighbours_command {w} {
    foreach id [tkined_selection $w.canvas] { 
	set type [$id type]
	if {($type == "NODE") || ($type == "NETWORK")} {
	    foreach link_id [$id links] {
		[$link_id ida] select
		[$link_id idb] select
	    }
	}
    }
}

proc tkined_select_member_command {w} {
    foreach id [tkined_selection $w.canvas] {
        if {[$id type] != "GROUP"} continue
        foreach m [$id member] {
	    $m select
        }
    }
}

proc tkined_select_type_command {w type} {
    foreach id [tkined_retrieve $w.canvas] {
        if {[$id type] == $type} {
	    $id select
        }
    }
}

proc tkined_select_name_command {w} {
    static regex
    if {![info exists regex]} { set regex "" }
    set regex [tkined_request $w.canvas "Select objects by name." {} \
               "Enter a regular expression:" "{{} $regex}"]
    if {$regex==""} return

    foreach id [tkined_retrieve $w.canvas] {
        if {[regexp -nocase $regex [$id name]]} {
	    $id select
        }
    }
}

proc tkined_select_address_command {w} {
    static regex
    if {![info exists regex]} { set regex "" }
    set regex [tkined_request $w.canvas "Select objects by address." {} \
               "Enter a regular expression:" "{{} $regex}"]
    if {$regex==""} return

    foreach id [tkined_retrieve $w.canvas] {
        if {[regexp -nocase $regex [$id name]]} {
	    $id select
        }
    }
}


##
## The structure menu allows to bring object to the front/back
## and to group/ungroup objects. A group can be collapsed or
## expanded.
##

proc tkined_front_command {w} {
    set redo_cmd ""
    set undo_cmd ""
    foreach id [tkined_selection $w.canvas] {
	if {[$id type] != "IMAGE"} {
	    append redo_cmd "$id raise; "
	    append undo_cmd "$id lower; "
	    $id raise
	}
    }
    tkined_undo_command $w [list $undo_cmd $redo_cmd]
}

proc tkined_back_command {w} {
    set redo_cmd ""
    set undo_cmd ""
    foreach id [tkined_selection $w.canvas] {
	append redo_cmd "$id lower; "
	append undo_cmd "$id raise; "
        $id lower
    }
    tkined_undo_command $w [list $undo_cmd $redo_cmd]
}

proc tkined_group_command {w} {
    set idlist [tkined_selection $w.canvas]
    if {$idlist != ""} {
	set group [eval GROUP create $idlist]
	$group canvas $w.canvas
	$group icon  [tkined_editor_attribute $w groupicon]
	$group font  [tkined_editor_attribute $w font]
	$group color [tkined_editor_attribute $w color]
	$group label name	      
    }
}

proc tkined_ungroup_command {w} {
    foreach id [tkined_selection $w.canvas] {
        if {[$id type]=="GROUP"} {
	    $id expand
	    $id delete
        }
    }
}

proc tkined_collapse_command {w} {
    set redo_cmd ""
    set undo_cmd ""
    foreach id [tkined_selection $w.canvas] {
	if {[$id type]=="GROUP"} {
	    append redo_cmd "$id collapse; "
	    append undo_cmd "$id expand; "
	    $id collapse
	}
    }
    tkined_undo_command $w [list $undo_cmd $redo_cmd]
}

proc tkined_expand_command {w} {
    set redo_cmd ""
    set undo_cmd ""
    foreach id [tkined_selection $w.canvas] {
	if {[$id type]=="GROUP"} {
	    append redo_cmd "$id expand; "
	    append undo_cmd "$id collapse; "
	    $id expand
	}
    }
    tkined_undo_command $w [list $undo_cmd $redo_cmd]
}

##
## Command to change the icon of the selected nodes.
##

proc tkined_icon_command {w type name} {
    set redo_cmd ""
    set undo_cmd ""
    foreach id [tkined_selection $w.canvas] {
	if {[$id type]=="$type"} {
	    append redo_cmd "$id icon \"$name\"; "
	    append undo_cmd "$id icon \"[$id icon]\"; "
	    $id icon $name
	}
    }
    tkined_undo_command $w [list $undo_cmd $redo_cmd]
    # set the default icon
    switch $type {
	NODE {
	    set fname [tkined_editor_attribute $w "icon$name"]
	    if {$fname!=""} {
		set fname [tkined_find_file $fname]
		if {[file isfile $fname]} {
		    $w.tools.node configure -bitmap @$fname
		}
	    } else {
		$w.tools.node configure -bitmap $name
	    }
	    tkined_editor_attribute $w icon $name
	}
	GROUP {
	    set fname [tkined_editor_attribute $w "groupicon$name"]
	    if {$fname!=""} {
		set fname [tkined_find_file $fname]
		if {[file isfile $fname]} {
		    $w.tools.group configure -bitmap @$fname
		}
	    } else {
		$w.tools.group configure -bitmap $name
	    }
	    tkined_editor_attribute $w groupicon $name
	}
	NETWORK {
	    tkined_editor_attribute $w network $name
	}
    }
}

##
## Change the font of the current selection (labels and text objects).
##

proc tkined_font_command {w fontname} {
    set redo_cmd ""
    set undo_cmd ""
    foreach id [tkined_selection $w.canvas] {
	if {[$id type] != "LINK"} {
	    append redo_cmd "$id font \"$fontname\"; "
	    append undo_cmd "$id font \"[$id font]\"; "
	    $id font $fontname 
	}
    }
    tkined_editor_attribute $w font $fontname
    tkined_undo_command $w [list $undo_cmd $redo_cmd]
}

##
## Change the color of the current selection.
##

proc tkined_color_command {w colorname} {
    set redo_cmd ""
    set undo_cmd ""
    foreach id [tkined_selection $w.canvas] {
	append redo_cmd "$id color \"$colorname\"; "
	append undo_cmd "$id color \"[$id color]\"; "
	$id color $colorname
    }
    tkined_editor_attribute $w color $colorname
    tkined_undo_command $w [list $undo_cmd $redo_cmd]
}

##
## Here are all commands used in the view menu.
##

proc tkined_new_view_command {{w .ined}} {
    static count
    if [catch {incr count}] { set count 0 }

    set widget "$w$count"
    catch {tkined_new_editor $widget} err

    tkined_editor_attribute $widget filename "unknown"
    if {[catch {exec pwd} dir]} { set dir "/" }
    tkined_editor_attribute $widget dirname $dir
    return $widget
}

proc tkined_close_view_command {w} {
    destroy $w
    if {[llength [winfo children .]]==0} { tkined_quit_command $w }
}

##
## Here is the implementation of the page command from the page menu.
## Set the editor attributes pageSize and pageOrientation to their
## new values and call tkined_page.
##

proc tkined_page_command {w size orientation} {
    if {$size!=""} {
	tkined_editor_attribute $w pageSize $size
    }
    if {$orientation!=""} {
	tkined_editor_attribute $w pageOrientation $orientation 
    }
    tkined_page $w.canvas $size $orientation
}

##
## Emulate the ined command. We first convert to the internal
## object driven syntax. If it fails, we try the old command.
## After each create command, we set the canvas to the global
## variable current_canvas. This is set by the open command.
##

proc ined {args} {
    global current_canvas
    if {[lindex $args 0] == "-noupdate"} { set args [lrange $args 1 end] }
    if {[lindex $args 0] == "page"} {
	set cmd $args
    } else {
	set cmd "[lindex $args 1] [lindex $args 0] [lrange $args 2 end]"
    }
    if {[catch {eval $cmd} res]} {
	set cmd "tkined_[lindex $args 0] $current_canvas [lrange $args 1 end]"
	if {[catch {eval $cmd}]} return 
    }
    if {[lindex $cmd 1] == "create"} {
	catch {$res canvas $current_canvas}
    }
    return $res
}
