##
## tkined_dialog.tcl
##
## This file contains the dialogs used by the tkined editor. Note
## that the help dialogs are located in tkined_help.tcl, since we
## need them not so frequently.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

##
## Compute the geometry of a dialog window w based on the geometry
## of the toplevel window top. Unfortunately we can not get the
## size before a window is mapped.
##

proc tkined_position {w top} {
    wm withdraw $w
    update idletasks
    set top [winfo toplevel $top]

    set rx [expr [winfo rootx $top]+[winfo vrootx $top]]
    set ry [expr [winfo rooty $top]+[winfo vrooty $top]]

    set cx [expr $rx+[winfo width $top]/2]
    set cy [expr $ry+[winfo height $top]/2]

    set x  [expr $cx-[winfo reqwidth $w]/2]
    set y  [expr $cy-[winfo reqheight $w]/2]

    if {$x < 0} { set x 0 }
    if {$y < 0} { set y 0 }

    wm geometry $w +$x+$y
    wm deiconify $w    
}

##
## Make a dialog toplevel window with the given dialog name.
##

proc tkined_dialog_toplevel {w} {
    catch {destroy $w}
    toplevel $w
    wm title $w [winfo name $w]
    wm transient $w [winfo toplevel [winfo parent $w]]
}

##
## Make a title for a dialog. Create a message containing the lines
## given in the lines argument.
##

proc tkined_dialog_title {w lines} {
    set text ""
    foreach line $lines { append text "$line\n" }
    message $w -aspect 25000 -text $text
    pack $w -side top -fill both    
}

##
## An acknowledge dialog takes its arguments, puts them into a
## message and displays them until the user hits the dismiss 
## button.
##

proc tkined_acknowledge {c args} {
    set w "$c.acknowledge"

    tkined_dialog_toplevel $w
    tkined_dialog_title $w.title $args

    button $w.ok -padx 5 -text dismiss -command "destroy $w"
    pack $w.ok -side bottom -padx 10 -pady 10 -expand true
    tkined_position $w $c
    grab set $w
    tkwait window $w
}

##
## A confirm dialog takes its arguments, puts them into a
## message and displays them until the user hits the yes, no
## or cancel button.
##

proc tkined_confirm {c args} {
    global tkined_confirm_result

    set tkined_confirm_result ""
    set w "$c.confirm"

    tkined_dialog_toplevel $w
    tkined_dialog_title $w.title $args

    button $w.ok -padx 5 -text YES \
	-command "global tkined_confirm_result; 
                  set tkined_confirm_result yes;
                  destroy $w"
    button $w.no -padx 5 -text NO \
	-command "global tkined_confirm_result;
                  set tkined_confirm_result no;
                  destroy $w"
    button $w.cancel -padx 5 -text Cancel \
	-command "global tkined_confirm_result;
                  set tkined_confirm_result cancel;
                  destroy $w"
    pack $w.cancel -side left  -padx 10 -pady 10 -expand true
    pack $w.ok     -side right -padx 10 -pady 10 -expand true
    tkined_position $w $c
    grab set $w
    tkwait window $w
    return $tkined_confirm_result
}

##
## A request dialog displays a message and asks the user to enter
## something in a request box. The last argument is a default for
## the entry widget.
##

proc tkined_request {c args} {
    global tkined_request_result

    set w "$c.request"
    set tkined_request_result ""

    tkined_dialog_toplevel $w
    set idx [llength $args]; incr idx -1
    set lastarg [lindex $args $idx]; incr idx -1
    tkined_dialog_title $w.title [lrange $args 0 $idx]

    set idx 1
    set cmd "global tkined_request_result; set tkined_request_result {}; "
    frame $w.ia
    frame $w.ia.msg
    frame $w.ia.entry
    foreach elem $lastarg {
	message $w.ia.msg.$idx -aspect 25000 -text [lindex $elem 0]
	switch [lindex $elem 2] {
	    scale {
		frame $w.ia.entry.$idx
		label $w.ia.entry.$idx.l -width 4 -font fixed
		scale $w.ia.entry.$idx.s -orient h -label "" -showvalue false \
		    -command "$w.ia.entry.$idx.l configure -text"
		append cmd \
		    "lappend tkined_request_result \[$w.ia.entry.$idx.s get\]; "
		set from [expr round([lindex $elem 3])]
		set to   [expr round([lindex $elem 4])]
		if {[lindex $elem 3] != ""} {
		    catch {$w.ia.entry.$idx.s configure -from $from}
		}
		if {[lindex $elem 4] != ""} {
		    catch {$w.ia.entry.$idx.s configure -to $to}
		}
		set val [expr round([lindex $elem 1])]
		catch {$w.ia.entry.$idx.s set $val} err
		pack $w.ia.entry.$idx.l -side left
		pack $w.ia.entry.$idx.s -side right -fill x -expand true
	    }
	    logscale {
		frame $w.ia.entry.$idx
		label $w.ia.entry.$idx.l -width 4 -font fixed
		scale $w.ia.entry.$idx.s -orient h -label "" -showvalue false \
		    -command "$w.ia.entry.$idx.l configure -text"
		append cmd \
		    "lappend tkined_request_result \[expr log(\[$w.ia.entry.$idx.s get\])\]; "
		set from [expr round(exp([lindex $elem 3]))]
		set to [expr round(exp([lindex $elem 4]))]
		if {[lindex $elem 3] != ""} {
		    catch {$w.ia.entry.$idx.s configure -from $from}
		}
		if {[lindex $elem 4] != ""} {
		    catch {$w.ia.entry.$idx.s configure -to $to}
		}
		set val [expr round(exp([lindex $elem 1]))]
		catch {$w.ia.entry.$idx.s set $val}
		pack $w.ia.entry.$idx.l -side left
		pack $w.ia.entry.$idx.s -side right -fill x -expand true
	    }
	    radio {
		frame $w.ia.entry.$idx
		foreach but [lrange $elem 3 end] {
		    radiobutton $w.ia.entry.$idx.$but -text $but -relief flat \
			-variable tkined_request_value_$idx
		    pack $w.ia.entry.$idx.$but -side left -fill x -expand true
		}
		catch {$w.ia.entry.$idx.[lindex $elem 1] invoke}
		append cmd \
		    "lappend tkined_request_result \[set tkined_request_value_$idx\]; "
	    }
	    default {
		entry $w.ia.entry.$idx -width 40 -relief sunken
		$w.ia.entry.$idx insert 0 [lindex $elem 1]
		bind $w.ia.entry.$idx <Return> "$w.ok invoke"
		append cmd "lappend tkined_request_result \[$w.ia.entry.$idx get\]; "
	    }
	}

	pack $w.ia.entry.$idx -side top -fill both -expand true -padx 1 -pady 1
	pack $w.ia.msg.$idx -side top
	incr idx
    }
    pack $w.ia.msg -side left
    pack $w.ia.entry -side right -padx 10
    append cmd "destroy $w"
    bind $w <Return> "$w.ok invoke"
    button $w.ok -padx 5 -text OK -command $cmd
    button $w.cancel -padx 5 -text Cancel -command "destroy $w"
    pack $w.ia -side top
    pack $w.cancel -side left -padx 10 -pady 10 -expand true
    pack $w.ok -side right -padx 10 -pady 10 -expand true
    tkined_position $w $c
    focus $w
    grab set $w
    tkwait window $w
    return $tkined_request_result
}

##
## Browse a long list of text. This dialog opens a text widget
## with a scrollbar. The dialog is finished when the dismiss
## button is invoked.
##

proc tkined_browse {c args} {
    set w "$c.browse"

    tkined_dialog_toplevel $w
    set idx [llength $args]; incr idx -1
    set lastarg [lindex $args $idx]; incr idx -1
    tkined_dialog_title $w.title [lrange $args 0 $idx]
    regsub "\n" $lastarg " " lastarg

    frame $w.b
    set width 20
    set height 1
    set msg ""
    foreach line $lastarg {
	if {$line == "\n"} continue
	set line [string trimright $line]
	append msg "$line\n"
	set len [string length $line]
	if {$len>$width} { set width $len }
	incr height
    }
    if {$height>24} { set height 24 }
    
    set scroll [expr {$height > 10}]

    if {$scroll} {
	text $w.b.txt -width $width -height $height \
	    -yscrollcommand "$w.b.scrollbar set" -setgrid true \
	    -borderwidth 2
	scrollbar $w.b.scrollbar -command "$w.b.txt yview" -relief sunken
    } else {
	text $w.b.txt -width $width -height $height -setgrid true \
	    -borderwidth 2
    }
    $w.b.txt insert 0.0 $msg

    button $w.ok -padx 5 -text dismiss -command "destroy $w"
    pack $w.b.txt -side left -expand true
    if {$scroll} {
	pack $w.b.scrollbar -side right -fill y -expand true 
    }
    pack $w.b -side top -padx 10 -pady 10
    pack $w.ok -side bottom -padx 10 -pady 10 -expand true
    tkined_position $w $c
    grab set $w
    tkwait window $w
}

##
## This dialog displays a list and lets the users select items from
## the list. It returns the list of selected items.
##

proc tkined_list {c args} {
    global tkined_list_result
    set tkined_list_result ""
    set w "$c.list"

    tkined_dialog_toplevel $w
    set idx [llength $args]; incr idx -1
    set lastarg [lindex $args $idx]; incr idx -1
    tkined_dialog_title $w.title [lrange $args 0 $idx]

    # compute the width and height of the listbox
    set width 24
    set height 8
    foreach elem $lastarg {
	set l [string length $elem]
	incr l
	if {($l > $width) && ($l <= 80)} {
	    set width $l
	}
    }
    set h [llength $lastarg]
    if {($h > $height)} {
	set height [expr {($h <= 24) ? $h : 24}]
    }
    set geom $width
    append geom x
    append geom $height

    frame $w.box
    scrollbar $w.box.scroll -command "$w.box.list yview" -relief sunken
    listbox $w.box.list -yscroll "$w.box.scroll set" -relief sunken \
	-geometry $geom
    foreach elem $lastarg {
	$w.box.list insert end $elem
    }
    tk_listboxSingleSelect $w.box.list
    pack $w.box.scroll -side right -fill y
    pack $w.box.list -side left -expand true -fill both

    button $w.cancel -padx 5 -text Cancel -command "destroy $w"
    button $w.ok -padx 5 -text "OK" -command \
	"global tkined_list_result;
         foreach i \[$w.box.list curselection\] {
             lappend tkined_list_result \[$w.box.list get \$i\]
         };
         destroy $w"

    pack $w.box -side top -padx 10 -pady 10 -expand true
    pack $w.cancel -side left -padx 10 -pady 10 -expand true
    pack $w.ok -side right -padx 10 -pady 10 -expand true

    bind $w.box.list <Double-Button-1> \
        "%W select from \[%W nearest %y\];
         %W select to \[%W nearest %y\];
         $w.ok invoke"

    tkined_position $w $c
    grab set $w
    tkwait window $w
    return $tkined_list_result
}

##
## The (ugly) fileselector. Perhaps I should borrow one from the net.
## tkined_file_select builds the browser and fires tkined_file_select_browse
## which does the real job.
##

proc tkined_file_select {c {title {File:}} {dir {}} {filename {}}} {
    global tkined_file_select_result
    set tkined_file_select_result ""

    catch {destroy $c.fs}
    toplevel $c.fs
    wm transient $c.fs [winfo toplevel $c]
    focus $c.fs
    grab set $c.fs

    message $c.fs.title -aspect 25000 -text $title
    entry $c.fs.entry -relief sunken
    if {$dir==""} { if {[catch {exec pwd} dir]} { set dir "/" } }    
    button $c.fs.ok -text "OK" -padx 8 
    button $c.fs.cancel -text "Cancel" -padx 8 -command "destroy $c.fs"
    frame $c.fs.box
    scrollbar $c.fs.box.scroll -relief sunken \
	-command "$c.fs.box.filelist yview"
    listbox $c.fs.box.filelist -yscroll "$c.fs.box.scroll set" \
	-relief sunken -geometry 24x12
    tk_listboxSingleSelect $c.fs.box.filelist

    pack $c.fs.box.scroll -side right -fill y
    pack $c.fs.box.filelist -side left -fill both -expand true

    pack $c.fs.title -side top
    pack $c.fs.entry -side top -padx 10 -pady 10 -expand true
    pack $c.fs.box -side top -padx 10 -expand true
    pack $c.fs.cancel -side left -padx 10 -pady 10 -expand true
    pack $c.fs.ok -side right -padx 10 -pady 10 -expand true
    tkined_position $c.fs $c
    
    bind $c.fs <Control-q> {destroy %W}
    bind $c.fs <Control-c> {destroy %W}
    bind $c.fs.entry <Return> "$c.fs.ok invoke"
    bind $c.fs.box.filelist <Button-1> \
	"%W select from \[%W nearest %y\];
         %W select to \[%W nearest %y\];
         $c.fs.entry delete 0 end;
         $c.fs.entry insert 0 \[%W get \[%W nearest %y\]\]"
    bind $c.fs.box.filelist <Double-Button-1> \
	"%W select from \[%W nearest %y\];
	 %W select to \[%W nearest %y\];
         $c.fs.ok invoke"

    tkined_file_select_browse $c.fs $dir $filename

    tkwait window $c.fs
    return $tkined_file_select_result
}

##
## tkined_file_select_browse gets called whenever something is
## selected in the filebrowser. This can happen by pressing the
## ok button, by double clicking in the listbox or by typing
## Return in the entry widget.
##

proc tkined_file_select_browse {top dir file} {
    if {$file==".."} {
	set dir [split $dir "/"]
	set len [llength $dir]
	set dir [join [lrange $dir 0 [expr $len-2]] "/"]
    } elseif {[string match /* $file]} {
	set dir $file
    } else {
	append dir "/$file"
    }
    regsub "//" $dir "/" dir
    $top.entry delete 0 end
    if {([file isfile $dir]) || (![file exists $dir])} {
	global tkined_file_select_result
	set tkined_file_select_result $dir
	destroy [winfo toplevel $top.box.filelist]
	return
    }
    $top.box.filelist select clear
    $top.box.filelist delete 0 end
    if {$dir!="/"} { $top.box.filelist insert end ".." }
    if [catch {glob $dir/*} files] { set files "" }
    set files [lsort $files]
    foreach f $files {
	$top.box.filelist insert end [file tail $f]
    }
    $top.ok configure \
	-command "tkined_file_select_browse $top $dir \[$top.entry get\]"
}
