##
## tkined_editor.tcl
##
## This file contains the definition of an tkined editor.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##


##
## Set up a new editor. Create all the menus and the canvas with
## the scollbars.
##

proc tkined_new_editor {w} {

    global tkined_version

    catch {destroy $w}
    toplevel $w -class tkined

    wm withdraw $w
    wm iconbitmap $w icon
    wm title $w "tkined: unknown"
    wm iconname $w "tkined"
    wm minsize $w 600 400

    # read the default settings
    tkined_load_defaults $w tkined.defaults

    # set up the canvas for the graphic
    canvas $w.canvas -borderwidth 1 -relief raised -width 600 -height 400
    
    # set up the tool box
    frame $w.tools -borderwidth 1 -relief raised
    
    button $w.tools.select  -text "Select" \
	-relief flat \
	-command "tkined_select_tool $w.tools.select $w.canvas"
    button $w.tools.move    -text "Move" \
	-relief flat \
	-command "tkined_select_tool $w.tools.move $w.canvas"
    button $w.tools.text    -text "Text" \
	-relief flat \
	-command "tkined_select_tool $w.tools.text $w.canvas"
    button $w.tools.label -text "Label" \
	-relief flat \
	-command "tkined_select_tool $w.tools.label $w.canvas"
    button $w.tools.node -bitmap machine \
	-relief flat \
	-command "tkined_select_tool $w.tools.node $w.canvas"
    button $w.tools.network -text Network \
	-relief flat \
	-command "tkined_select_tool $w.tools.network $w.canvas"
    button $w.tools.link -text Link \
	-relief flat \
	-command "tkined_select_tool $w.tools.link $w.canvas"
    button $w.tools.group -bitmap group \
	-relief flat  \
	-command "tkined_select_tool $w.tools.group $w.canvas"
    tkined_select_tool $w.tools.select $w.canvas

    label $w.tools.dummy -text "tkined $tkined_version"
    pack $w.tools.dummy -side top -pady 1 -ipadx 5
    pack $w.tools.select  -fill x
    pack $w.tools.move    -fill x
    pack $w.tools.text    -fill x
    pack $w.tools.label   -fill x
    pack $w.tools.node    -fill x
    pack $w.tools.network -fill x
    pack $w.tools.link    -fill x
    pack $w.tools.group   -fill x

    # set up the menu bar
    frame $w.menu -borderwidth 1 -relief raised
    
    # add some scrollbars to the canvas
    scrollbar $w.vscroll -relief sunken -command "$w.canvas yview"
    scrollbar $w.hscroll -orient horiz -relief sunken -command "$w.canvas xview"

    # set up the application
    pack $w.tools -side left -fill y
    pack $w.menu -side top -fill x
    pack $w.hscroll -side bottom -fill x
    pack $w.canvas -side left -fill both -expand true
    pack $w.vscroll -side right -fill y

    $w.canvas config -xscroll "$w.hscroll set" -yscroll "$w.vscroll set"

    # set up the file menu
    menubutton $w.menu.file -text " File " -menu $w.menu.file.m
    menu $w.menu.file.m
    $w.menu.file.m add command -label "New" \
	-accelerator "  ^N" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  tkined_new_command $w; \
                  $w.menu.file configure -state normal"
    bind $w <Control-n> "$w.menu.file.m invoke 0"
    $w.menu.file.m add command -label "Open" \
	-accelerator "  ^O" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  tkined_open_command $w; \
                  $w.menu.file configure -state normal"
    bind $w <Control-o> "$w.menu.file.m invoke 1"
    $w.menu.file.m add command -label "Merge" \
	-accelerator "  ^M" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  tkined_merge_command $w; \
                  $w.menu.file configure -state normal"
    bind $w <Control-m> "$w.menu.file.m invoke 2"
    $w.menu.file.m add command -label "Save" \
	-accelerator "  ^S" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  tkined_save_command $w; \
                  $w.menu.file configure -state normal"
    bind $w <Control-s> "$w.menu.file.m invoke 3"
    $w.menu.file.m add command -label "Save As" \
	-accelerator "  ^A" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  tkined_save_as_command $w; \
                  $w.menu.file configure -state normal"
    bind $w <Control-a> "$w.menu.file.m invoke 4"
    $w.menu.file.m add sep
    $w.menu.file.m add command -label "PostScript" \
	-accelerator "  ^P" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  tkined_postscript_command $w; \
                  $w.menu.file configure -state normal"
    bind $w <Control-p> "$w.menu.file.m invoke 6"
    $w.menu.file.m add command -label "Print" \
	-accelerator "  ^R" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  tkined_print_command $w; \
                  $w.menu.file configure -state normal"
    bind $w <Control-r> "$w.menu.file.m invoke 7"
    $w.menu.file.m add sep
    $w.menu.file.m add command -label "Import" \
	-accelerator "  ^I" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  tkined_import_command $w; \
                  $w.menu.file configure -state normal"
    bind $w <Control-i> "$w.menu.file.m invoke 9"
    $w.menu.file.m add sep
    $w.menu.file.m add command -label "Quit" \
	-accelerator "  ^Q" \
	-command "tkined_quit_command $w"
    bind $w <Control-q> "$w.menu.file.m invoke 11"
    
    # set up the edit menu
    menubutton $w.menu.edit -text "Edit " \
	-menu $w.menu.edit.m
    menu $w.menu.edit.m
    $w.menu.edit.m add cascade -label "Label" -menu $w.menu.edit.m.l
    $w.menu.edit.m add sep
    $w.menu.edit.m add command -label "Undo" \
	-accelerator "  U" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  tkined_undo_command $w; \
                  $w.menu.edit configure -state normal"
    bind $w <U> "$w.menu.edit.m invoke 2"
    $w.menu.edit.m add command -label "Redo" \
	-accelerator "  R" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  tkined_redo_command $w; \
                  $w.menu.edit configure -state normal" 
    bind $w <R> "$w.menu.edit.m invoke 3"
    $w.menu.edit.m add sep
    $w.menu.edit.m add command -label "Delete" \
	-accelerator " ^D" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  tkined_delete_command $w; \
                  $w.menu.edit configure -state normal"
    bind $w <Control-d> "$w.menu.edit.m invoke 5"
    $w.menu.edit.m add command -label "Cut" \
	-accelerator "  x" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  tkined_cut_command $w; \
                  $w.menu.edit configure -state normal"
    bind $w <x> "$w.menu.edit.m invoke 6"
    $w.menu.edit.m add command -label "Copy" \
	-accelerator "  w" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  tkined_copy_command $w; \
                  $w.menu.edit configure -state normal"
    bind $w <w> "$w.menu.edit.m invoke 7"
    $w.menu.edit.m add command -label "Paste" \
	-accelerator "  v" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  tkined_paste_command $w; \
                  $w.menu.edit configure -state normal"
    bind $w <v> "$w.menu.edit.m invoke 8"

    menu $w.menu.edit.m.l
    $w.menu.edit.m.l add command -label name \
	-command "tkined_label_command $w name"
    $w.menu.edit.m.l add command -label address \
	-command "tkined_label_command $w address"
    $w.menu.edit.m.l add command -label text \
	-command "tkined_label_command $w text"
    $w.menu.edit.m.l add command -label clear \
	-command "tkined_label_command $w clear"
    
    # set up the select menu
    menubutton $w.menu.select -text "Select " \
        -menu $w.menu.select.m
    menu $w.menu.select.m
    $w.menu.select.m add command -label "Select all" \
	-accelerator "  a" \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  tkined_select_all_command $w; \
                  $w.menu.select configure -state normal"
    bind $w <a> "$w.menu.select.m invoke 0"
    $w.menu.select.m add sep
    $w.menu.select.m add command -label "Select neighbours" \
	-accelerator "  n" \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  tkined_select_neighbours_command $w; \
                  $w.menu.select configure -state normal"
    bind $w <n> "$w.menu.select.m invoke 2"
    $w.menu.select.m add command -label "Select member" \
	-accelerator "  m" \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  tkined_select_member_command $w; \
                  $w.menu.select configure -state normal"
    bind $w <m> "$w.menu.select.m invoke 3"
    $w.menu.select.m add sep
    $w.menu.select.m add cascade -label "Select by type" \
	-menu $w.menu.select.m.s
    $w.menu.select.m add command -label "Select by name" \
	-accelerator "  N" \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  tkined_select_name_command $w; \
                  $w.menu.select configure -state normal"
    bind $w <N> "$w.menu.select.m invoke 6"
    $w.menu.select.m add command -label "Select by address" \
	-accelerator "  A" \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  tkined_select_address_command $w; \
                  $w.menu.select configure -state normal"
    bind $w <A> "$w.menu.select.m invoke 7"

    menu $w.menu.select.m.s
    $w.menu.select.m.s add command -label node \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  tkined_select_type_command $w NODE; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label network \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  tkined_select_type_command $w NETWORK; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label link \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  tkined_select_type_command $w LINK; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label group \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  tkined_select_type_command $w GROUP; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label text \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  tkined_select_type_command $w TEXT; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label image \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  tkined_select_type_command $w IMAGE; \
                  $w.menu.select configure -state normal"

    # set up the structure menu
    menubutton $w.menu.structure -text "Structure " \
        -menu $w.menu.structure.m
    menu $w.menu.structure.m
    $w.menu.structure.m add command -label "Bring to Front" \
	-accelerator "  f" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  tkined_front_command $w; \
                  $w.menu.structure configure -state normal"
    bind $w <f> "$w.menu.structure.m invoke 0"
    $w.menu.structure.m add command -label "Send to Back" \
        -accelerator "  b" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  tkined_back_command $w; \
                  $w.menu.structure configure -state normal"
    bind $w <b> "$w.menu.structure.m invoke 1"
    $w.menu.structure.m add sep
    $w.menu.structure.m add command -label "Group" \
        -accelerator "  g" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  tkined_group_command $w; \
                  $w.menu.structure configure -state normal"
    bind $w <g> "$w.menu.structure.m invoke 3"
    $w.menu.structure.m add command -label "Ungroup" \
        -accelerator "  u" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  tkined_ungroup_command $w; \
                  $w.menu.structure configure -state normal"
    bind $w <u> "$w.menu.structure.m invoke 4"
    $w.menu.structure.m add command -label "Collapse" \
        -accelerator "  c" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  tkined_collapse_command $w; \
                  $w.menu.structure configure -state normal"
    bind $w <c> "$w.menu.structure.m invoke 5"
    $w.menu.structure.m add command -label "Expand" \
        -accelerator "  e" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  tkined_expand_command $w; \
                  $w.menu.structure configure -state normal"
    bind $w <e> "$w.menu.structure.m invoke 6"

    # set up the node menu
    menubutton $w.menu.node -text "Node " \
	-menu $w.menu.node.m
    menu $w.menu.node.m
    set i 1
    while {[set val [tkined_editor_attribute $w icon$i]]!=""} {
	set name [lrange $val 1 end]
	if {$name == ""} {
            $w.menu.node.m add sep
        } else {
	    $w.menu.node.m add radio -label " $name" \
		-variable node$w -value "$name" \
		-command "$w.menu.node configure -state disabled; \
                      update idletasks; \
                      tkined_icon_command $w NODE {$name}; \
                      $w.menu.node configure -state normal"
	    tkined_editor_attribute $w icon$name [lindex $val 0]
	}
	incr i
    }
    $w.menu.node.m add radio -label " Machine" \
	-variable node$w -value "machine" \
	-command "$w.menu.node configure -state disabled; \
                  update idletasks; \
                  tkined_icon_command $w NODE machine; \
                  $w.menu.node configure -state normal"
    $w.menu.node.m invoke $i

    # set up the network menu
    menubutton $w.menu.network -text "Network " \
	-menu $w.menu.network.m
    menu $w.menu.network.m
    set i 1
    while {[set val [tkined_editor_attribute $w network$i]]!=""} {
	set name [lrange $val 1 end]
	if {$name == ""} {
            $w.menu.network.m add sep
        } else {
	    $w.menu.network.m add radio -label " $name" \
		-variable network$w -value "$name" \
		-command "$w.menu.network configure -state disabled; \
                      update idletasks; \
                      tkined_icon_command $w NETWORK {$name}; \
                      $w.menu.network configure -state normal"
	    tkined_editor_attribute $w network$name [lindex $val 0]
	}
        incr i
    }
    $w.menu.network.m add radio -label " Network" \
	-variable network$w -value "network" \
	-command "$w.menu.network configure -state disabled;
                  update idletasks; \
                  tkined_icon_command $w NETWORK network; \
                  $w.menu.network configure -state normal"
    $w.menu.network.m invoke $i

    # set up the group menu
    menubutton $w.menu.group -text "Group " \
	-menu $w.menu.group.m
    menu $w.menu.group.m
    set i 1
    while {[set val [tkined_editor_attribute $w groupicon$i]]!=""} {
	set name [lrange $val 1 end]
	if {$name == ""} {
            $w.menu.group.m add sep
        } else {
	    $w.menu.group.m add radio -label " $name" \
		-variable group$w -value "$name" \
		-command "$w.menu.group configure -state disabled;
                      update idletasks; \
                      tkined_icon_command $w GROUP {$name}; \
                      $w.menu.group configure -state normal"
	    tkined_editor_attribute $w groupicon$name [lindex $val 0]
	}
        incr i
    }
    $w.menu.group.m add radio -label " Group" \
	-variable group$w -value "group" \
	-command "$w.menu.group configure -state disabled;
                  update idletasks; \
                  tkined_icon_command $w GROUP group; \
                  $w.menu.group configure -state normal"
    $w.menu.group.m invoke $i

    # set up the font menu
    menubutton $w.menu.font -text "Font " \
        -menu $w.menu.font.m
    menu $w.menu.font.m
    set i 1
    while {[set val [tkined_editor_attribute $w font$i]]!=""} {
	set name [lrange $val 1 end]
	if {$name == ""} {
	    $w.menu.font.m add sep
	} else {
	    $w.menu.font.m add radio -label " $name" \
		-variable font$w -value "$name" \
		-command "$w.menu.font configure -state disabled;
                          update idletasks; \
                          tkined_font_command $w {$name}; \
                          $w.menu.font configure -state normal"
	    tkined_editor_attribute $w font$name [lindex $val 0]
	}
	incr i
    }
    $w.menu.font.m add radio -label " Default" \
	-variable font$w -value "default" \
	-command "$w.menu.font configure -state disabled;
                  update idletasks; \
                  tkined_font_command $w default; \
                  $w.menu.font configure -state normal"
    $w.menu.font.m invoke $i

    # set up the color menu
    if {[tk colormodel .]=="color"} { 
	menubutton $w.menu.color -text "Color " \
	    -menu $w.menu.color.m
	menu $w.menu.color.m
	set i 1
	while {[set val [tkined_editor_attribute $w color$i]]!=""} {
	    set name [lrange $val 1 end]
	    if {$name == ""} {
		$w.menu.color.m add sep
	    } else {
		$w.menu.color.m add radio -label " $name" \
		    -variable color$w -value "$name" \
		    -command "$w.menu.color configure -state disabled;
                          update idletasks; \
                          tkined_color_command $w {$name}; \
                          $w.menu.color configure -state normal"
		tkined_editor_attribute $w color$name [lindex $val 0]
	    }
	    incr i
	}
	$w.menu.color.m add radio -label " Black" \
	    -variable color$w -value "Black" \
	    -command "$w.menu.color configure -state disabled;
                  update idletasks; \
                  tkined_color_command $w Black; \
                  $w.menu.color configure -state normal"
	$w.menu.color.m invoke $i
    }

    # set up the view menu
    menubutton $w.menu.view -text "View " \
	-menu $w.menu.view.m
    menu $w.menu.view.m
    $w.menu.view.m add command -label "New View" \
	-accelerator "  V" \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  tkined_new_view_command; \
                  $w.menu.view configure -state normal"
    bind $w <V> "$w.menu.view.m invoke 0"
    $w.menu.view.m add command -label "Close View" \
	-accelerator "  K" \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  tkined_close_view_command $w"
    bind $w <K> "$w.menu.view.m invoke 1"

    # set up the page menu
    menubutton $w.menu.page -text "Page " \
	-menu $w.menu.page.m
    menu $w.menu.page.m
    global tkined_pageSize$w
    global tkined_orientation$w
    $w.menu.page.m add radio -label "DINA4" \
	-variable tkined_pageSize$w -value A4 \
	-command "$w.menu.page configure -state disabled; update idletasks; \
                  tkined_page_command $w DINA4 \
                  \[tkined_editor_attribute $w pageOrientation\]; \
                  $w.menu.page configure -state normal"
    $w.menu.page.m add radio -label "DINA3" \
	-variable tkined_pageSize$w -value A3 \
	-command "$w.menu.page configure -state disabled; update idletasks; \
                  tkined_page_command $w DINA3 \
                  \[tkined_editor_attribute $w pageOrientation\]; \
                  $w.menu.page configure -state normal"
    $w.menu.page.m add radio -label "DINA2" \
	-variable tkined_pageSize$w -value A2 \
	-command "$w.menu.page configure -state disabled; update idletasks; \
                  tkined_page_command $w DINA2 \
	          \[tkined_editor_attribute $w pageOrientation\]; \
                  $w.menu.page configure -state normal"
    $w.menu.page.m add radio -label "DINA1" \
	-variable tkined_pageSize$w -value A1 \
	-command "$w.menu.page configure -state disabled; update idletasks; \
                  tkined_page_command $w DINA1 \
	          \[tkined_editor_attribute $w pageOrientation\]; \
                  $w.menu.page configure -state normal"
    $w.menu.page.m add sep
    $w.menu.page.m add radio -label "Portrait" \
	-variable tkined_orientation$w -value portrait \
	-command "$w.menu.page configure -state disabled; update idletasks; \
                  tkined_page_command $w \
                  \[tkined_editor_attribute $w pageSize\] portrait; \
                  $w.menu.page configure -state normal"
    $w.menu.page.m add radio -label "Landscape" \
	-variable tkined_orientation$w -value landscape \
	-command "$w.menu.page configure -state disabled; update idletasks; \
                  tkined_page_command $w \
                  \[tkined_editor_attribute $w pageSize\] landscape; \
                  $w.menu.page configure -state normal"
    $w.menu.page.m invoke 0
    $w.menu.page.m invoke 6
    tkined_editor_attribute $w pageSize DINA4
    tkined_editor_attribute $w pageOrientation landscape

    # set up the help menu
    menubutton $w.menu.help -text "Help " \
	-menu $w.menu.help.m
    menu $w.menu.help.m
    $w.menu.help.m add command -label "General" \
	-command "$w.menu.help configure -state disabled; update idletasks; \
                  tkined_help_general $w; \
	          $w.menu.help configure -state normal"
    $w.menu.help.m add command -label "Status" \
	-command "$w.menu.help configure -state disabled; update idletasks; \
                  tkined_help_status $w; \
                  $w.menu.help configure -state normal"
    $w.menu.help.m add command -label "Key Bindings" \
	-command "$w.menu.help configure -state disabled; update idletasks; \
                  tkined_help_key_bindings $w; \
                  $w.menu.help configure -state normal"
    
    pack $w.menu.file -side left
    pack $w.menu.edit -side left
    pack $w.menu.select -side left
    pack $w.menu.structure -side left
    pack $w.menu.node -side left
    pack $w.menu.network -side left
    pack $w.menu.group -side left
    pack $w.menu.font -side left
    if {[tk colormodel .]=="color"} {
	pack $w.menu.color -side left
    }
    pack $w.menu.view -side left
    pack $w.menu.page -side left
    pack $w.menu.help -side right

    # bind the select tool on the left mouse button
    bind $w.canvas <3> \
	"tkined_clear_selection $w.canvas; \
         tkined_selectMark $w.canvas \
            \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <B3-Motion> \
	"tkined_selectDrag $w.canvas \
            \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <ButtonRelease-3> \
	"tkined_selectDone $w.canvas"
    bind $w.canvas <Shift-Button-3> \
	"tkined_selectMark $w.canvas \
            \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <Shift-B3-Motion> \
	"tkined_selectDrag $w.canvas \
            \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <Shift-ButtonRelease-3> \
	"tkined_selectDone $w.canvas"

    # bind the move tool on the middle mouse button
    bind $w.canvas <2> "tkined_moveMark $w.canvas \
                \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <B2-Motion> "tkined_moveDrag $w.canvas \
                \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <ButtonRelease-2> "tkined_moveDone $w.canvas \
                \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"

    # these bindings allow drag operations on the canvas
    bind $w.canvas <Shift-2> "tkined_scrollMark $w.canvas %x %y"
    bind $w.canvas <Shift-B2-Motion> "tkined_scrollDrag $w.canvas %x %y"
    bind $w.canvas <Shift-ButtonRelease-2> "tkined_scrollDone $w.canvas"

    # Key binding to turn on debugging mode
    bind $w <D> "global tkined_debug; set tkined_debug 1"
    bind $w <Q> "global tkined_debug; set tkined_debug 0"

    # Set up for keyboard-based menu traversal
    tk_bindForTraversal $w
    bind $w <Any-Enter> "focus $w"
    if {[tk colormodel .]=="color"} {
	tk_menuBar $w.menu $w.menu.file $w.menu.edit \
            $w.menu.select $w.menu.structure \
	    $w.menu.node $w.menu.network $w.menu.group $w.menu.font \
	    $w.menu.color \
	    $w.menu.view $w.menu.page $w.menu.help
    } else {
	tk_menuBar $w.menu $w.menu.file $w.menu.edit \
            $w.menu.select $w.menu.structure \
	    $w.menu.node $w.menu.network $w.menu.group $w.menu.font \
	    $w.menu.view $w.menu.page $w.menu.help
    }

    # now its time to map the whole thing on the screen
    update
    wm deiconify $w
    update

    # fire up all interpreters for this editor
    set i 1
    while {[set interp [tkined_editor_attribute $w interpreter$i]]!=""} {
        incr i
	if {[catch {INTERPRETER create $interp} interpreter]} {
	    tkined_acknowledge $w.canvas \
		"Sorry, $interp not found !!" "" \
		"Check your tkined.defaults and your TKINEDPATH."
	} else {
	    $interpreter canvas $w.canvas
	}
    }
}

##
## The following procedures enable the scroll operation on the canvas.
##

proc tkined_scrollMark {c x y} {
    catch {destroy $c.popup}
    $c configure -cursor fleur
    $c scan mark $x $y
}

proc tkined_scrollDrag {c x y} {
    $c scan dragto $x $y
}

proc tkined_scrollDone {c} {
    $c configure -cursor top_left_arrow
}

