##
## This is the initialization file for the tkined network editor.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

##
## This is ined implemented on top of tk. This file contains only
## some basic stuff. Most stuff is organized in modules kept in the
## following files:
##
##  tkined_editor.tcl	The editor abstraction. Generates a toplevel
##	           	window with the menubar, the toolbox and the
##			canvas. The editor just defines the interior.
##
##  tkined_tools.tcl	The tools that you can select from the tool box
##                      and apply on the canvas. These tools provide
##                      some graphical animation and finally call some
##                      of the low level tcl commands.
##
##  tkined_commands.tcl	The commands that implement all the operation
##                      that may be triggered by the menus and tools.
##                      Most commands just break complex operations on
##                      a selection down on low level operations on a
##                      single object.
##
##  tkined_tcl.tcl      This is the implementation of the callback
##                      routines. They really implement how tkined
##			objects are drawn.
##
##  tkined_dialog.tcl	Some useful dialogs used by tkined. This
##			file also contains the (ugly) fileselector.
##
##  tkined_graph.tcl	The graphic extensions that may be used to
##			display status information in graphical form.
##
##  tkined_help.tcl	Some help dialogs.
##
##  tkined_init.tcl     The initialization and customization file.
##                      This file also contains some utility proc's.
##
## The object handling code is written in as ordinary C code.
## All operations are performed on the C objects. Callback to tk
## procedures are used to actually modify the picture shown by 
## tkined.
##

##
## Editor specific information is stored in editor attributes.
## The file tkined_editor contains a procedure to get and set
## editor attributes.
##
## The following variables control where we search for ined
## source code (auto_path) and where we search for bitmaps,
## default settings and other runtime stuff.
##

source [info library]/init.tcl
source $tk_library/tk.tcl

append auto_path " $tkined_path"

##
## Add some nice key bindings to the entry widgets. I would like
## some more but I do not know how to move the icursor...
## Perhaps someone can help? I know of the emacs key bindings
## postet in comp.lang.tcl but I would like small ones.
## (Do we really need prefixes in entry widgets?)
##

bind Entry <Control-a> { %W icursor 0 }	
bind Entry <Control-e> { %W icursor end }	
bind Entry <Control-f> { %W icursor [expr {[%W index @%x]+1}] }	
bind Entry <Control-b> { %W icursor [expr {[%W index @%x]-1}] }	
bind Entry <Control-k> { %W delete @%x end }

# remove tk's default binding to get good old X11 paste working
# use button three to do the motion stuff

bind Entry <2> {%W insert insert [selection get]; tk_entrySeeCaret %W}
bind Entry <B2-Motion> ""
bind Entry <3> {%W scan mark %x}
bind Entry <B3-Motion> {%W scan dragto %x}

option add Tkined*exportSelection true startupFile

##
## Set the colors to grey. You might change it.
##

if {[tk colormodel .] == "color"} {
    option add Tkined*background "#EAEAEA" startupFile
    option add Tkined*activeBackground "#DADADA" startupFile
    option add Tkined*selectBackground "#CACACA" startupFile
    option add Tkined*foreground "#000000" startupFile
    option add Tkined*Scrollbar*foreground "#AAAAAA" startupFile
    option add Tkined*Scrollbar*activeForeground "#9A9A9A" startupFile
    option add Tkined*Scrollbar*selectForeground "#8A8A8A" startupFile
    option add Tkined*Scale*foreground "#AAAAAA" startupFile
    option add Tkined*Scale*activeForeground "#9A9A9A" startupFile
    option add Tkined*Scale*selectForeground "#8A8A8A" startupFile
    option add Tkined*Scale*selectBackground "#8A8A8A" startupFile
    option add Tkined*Scale*sliderForeground "#AAAAAA" startupFile
}

##
## I dont like the fat font tk uses per default. So here is good
## old fixed font.
##

option add Tkined*Entry.font fixed startupFile
option add Tkined*Text.font fixed startupFile
option add Tkined*Message.font fixed startupFile
option add Tkined*Listbox.font fixed startupFile

##
## This nice procedure allows us to use static variables. It was
## posted to the net by Karl Lehenbauer.
##

proc static {args} {
    set procName [lindex [info level [expr [info level]-1]] 0]
    foreach varName $args {
        uplevel 1 "upvar #0 {$procName:$varName} $varName"
    }
}

##
## Check if debug output is welcome.
##

proc debug {args} {
    global tkined_debug
    return $tkined_debug
}

##
## Handle background errors here.
##

proc tkerror {message} {
    if [debug] {
	puts stderr "TKERROR: BACKGROUND ERROR: $message"
    }
}

##
## Search for a file following tkined_path. Return the complete
## filename.
##

proc tkined_find_file {fname} {
    global tkined_path
    if [file exists $fname] { 
	return [file dirname $fname]/[file tail $fname]
    }
    foreach dir $tkined_path {
	if [file readable $dir/$fname] { 
	    return [file dirname $dir/$fname]/$fname 
	}
    }
    return ""
}

##
## Print a given file.
##

proc tkined_print {w fname} {
    global env

    set lpr [tkined_editor_attribute [winfo parent $w] printcmd]
    if {$lpr == ""} {
	if {[info exists env(PATH)]} { 
	    foreach dir [split $env(PATH) ":"] {
		set lpr $dir/lpr
		if {[file executable $lpr]} break
	    }
	}
    }
    if {![file executable [lindex $lpr 0]]} {
	tkined_acknowledge $w "How do you print on your system?"
	return
    }

    set res [tkined_confirm $w "Saved to temporary file $fname." \
	     "" "Should I really do the following command?" "" "$lpr $fname"]

    if {$res == "yes"} {
	if {[catch {eval exec $lpr $fname &} err]} {
	    tkined_acknowledge $w "$lpr $fname failed:" "" $err
	}
    }
}

##
## Search for files following the tkined_path and load their contents
## into the status attributes of an editor.
##

proc tkined_load_defaults {w name} {
    global tkined_path
    set reverse_path ""
    foreach dir $tkined_path {
	set reverse_path "$dir $reverse_path"
    }
    foreach dir $reverse_path {
        if [file readable $dir/$name] {
	    set fh [open $dir/$name r]
	    while {![eof $fh]} {
		gets $fh line
		set line [string trim $line]
		if {($line=="") || ([regexp "^#|^!" $line])} continue
		if {[string match tkined.* $line]} {
		    set line [split $line ":"]
		    set attname [lindex [split [lindex $line 0] "."] 1]
		    set attval  [string trim [lindex $line 1]]
		    tkined_editor_attribute $w $attname $attval
		}
	    }
	    close $fh
	}
    }
    return ""
}

##
## Access procedure for the editor attributes. Every editor
## dependent variable is stored as an editor attribute. This
## procedure maintains a static array indexed by the editor
## window path and the attribute name.
##

proc tkined_editor_attribute {w name args} {
    static tkined_status
    set index "$w-$name"
    if {!([info exists tkined_status($index)]) || ([llength $args]>0)} {
	set tkined_status($index) [join $args]
    }

    return $tkined_status($index)
}

##
## Return the size as the bounding box of the canvas.
##

proc tkined_size {c} {
    return [lindex [$c configure -scrollregion] 4]
}


##
## Get or set the size of the canvas.
##

proc tkined_page {c args} {

    set w [winfo parent $c]    
    set argc [llength $args]

    set size [tkined_size $c]
    set width [lindex $size 2]
    set height [lindex $size 3]

    if {$argc>0} {
	tkined_editor_attribute $w pageSize [lindex $args 0]
	if {$argc>1} {
	    tkined_editor_attribute $w pageOrientation [lindex $args 1]
	}
	global tkined_pageSize$w
	switch -glob [tkined_editor_attribute $w pageSize] {
	    *[Aa]4 {
		set tkined_pageSize$w A4
		set width 210m
		set height 297m
	    }
	    *[Aa]3 {
		set tkined_pageSize$w A3
		set width 297m
		set height 510m
	    }
	    *[Aa]2 {
		set tkined_pageSize$w A2
		set width 510m
		set height 594m
	    }
	    *[Aa]1 {
		set tkined_pageSize$w A1
                set width 594m
                set height 1020m
            }
	    default {
		set tkined_pageSize$w A4
		set width 210m
		set height 297m
	    }
	}

	set width  [winfo pixels $c $width]
	set height [winfo pixels $c $height]

	global tkined_orientation$w
	if {[tkined_editor_attribute $w pageOrientation]=="portrait"} {
	    set tkined_orientation$w "portrait"
	    $c configure -scrollregion "0 0 $width $height"
	} else {
	    $c configure -scrollregion "0 0 $height $width"
	    set tkined_orientation$w "landscape"
	}
    }

    return "[tkined_editor_attribute $w pageSize] \
            [tkined_editor_attribute $w pageOrientation]"
}

##
## Get some status information from the editor. Currently
## we just return the size of the canvas. This proc is
## often used as a ping command. This is really obsolete
## since we now have the size command.
##

proc tkined_status {c} {
    puts stderr "** obsolete command tkined_status called"
    set width [lindex [tkined_size $c] 2]
    set height [lindex [tkined_size $c] 3]
    return "{[winfo width $c] [winfo height $c] $width $height}"
}
