##
## tkined_tcl.tcl
##
## This file contains the tcl interface of the ined editor. The editor
## is based on this core set of functionality. Every command or tool
## simply calls a set of these procs representing the tcl interface.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##


##
## The following set of procedures handle node objects.
##

proc tkined_NODE_canvas { node } {
    set c [$node canvas]
    set x [lindex [$node move] 0]
    set y [lindex [$node move] 1]
    set item [$c create bitmap $x $y -bitmap machine \
	      -tags [list NODE "id $node"] ]
    $node items $item
}

proc tkined_NODE_delete { node } {
    eval [$node canvas] delete [$node items]
}

proc tkined_NODE_move { node dx dy } {
    set c [$node canvas]
    foreach item [$node items] {
	$c move $item $dx $dy
	$c move mark$item $dx $dy
    }
}

proc tkined_NODE_label_clear { node } {
    set c [$node canvas]
    set itemlist ""
    foreach item [$node items] {
	set tags [$c gettags $item]
	if {[lsearch $tags LABEL]>=0  
	    || [lsearch $tags labelclipbox]>=0} {
		$c delete $item
	    } else {
		lappend itemlist $item
	    }
    }
    $node items $itemlist
}

proc tkined_NODE_label { node {text ""} } {
    set c [$node canvas]
    set color [lindex [$c configure -background] 4]
    foreach item [$node items] {
	set tags [$c gettags $item]
        if {[lsearch $tags LABEL]>=0} { set label $item }
        if {[lsearch $tags labelclipbox]>=0} { set clipbox $item }
    }
    if {[info exist label]} {
        $c itemconfigure $label -text $text
        if {[info exist clipbox]} {
            eval $c coords $clipbox [$c bbox $label]
        }
	return
    }
    # create a new label with and a box behind it
    foreach item [$node items] {
	if {[lsearch [$c gettags $item] NODE]>=0} break
    }
    set bb [$c bbox $item]
    set x1 [lindex $bb 0]
    set x2 [lindex $bb 2]
    set y [expr {[lindex $bb 3]+1}]
    set x [expr {$x1+(($x2-$x1)/2)}]
    set label [$c create text $x $y -anchor n -text $text -font fixed \
	       -tags [list LABEL "id $node"] ]
    set tags [list labelclipbox "id $node"]
    set clipbox [eval $c create rectangle [$c bbox $label] \
		 -tags {$tags} -fill $color \
		     -outline $color -width 0 ]
    $c lower $clipbox $label
    $node items "[$node items] $label $clipbox"
    $node font [$node font]
}

proc tkined_NODE_icon { node } {
    set c [$node canvas]
    set name [$node icon]
    set fname [tkined_editor_attribute [winfo parent $c] icon$name]
    if {$fname==""} {
	set fname "$name.bm"
	if {[tkined_find_file $fname]==""} {
	    set fname "machine.bm"
	}
    }
    if {[set fname [tkined_find_file $fname]]==""} return
    foreach item [$node items] {
	if {[$c type $item]=="bitmap"} {
	    $c itemconfigure $item -bitmap @$fname
	    break
	}
    }
}

proc tkined_NODE_font { node } {
    set c [$node canvas]
    set fontname [$node font]
    set x11fontname [tkined_editor_attribute [winfo parent $c] font$fontname]
    if {$x11fontname == ""} { 
	set x11fontname fixed
    }
    foreach item [$node items] {
	if {[lsearch [$c gettags $item] LABEL] >= 0} {
	    $c itemconfigure $item -font $x11fontname
	}
    }
}

proc tkined_NODE_color { node } {
    set c [$node canvas]
    set w [winfo parent $c]
    set colorname [$node color]
    if {[tk colormodel .]=="color"} {
	set x11colorname [tkined_editor_attribute $w color$colorname]
	if {$x11colorname == ""} { set x11colorname $colorname }
    } else {
	set x11colorname black
    }
    foreach item [$node items] {
	if {([lsearch [$c gettags $item] NODE] >= 0) 
	    && ([$c type $item] == "bitmap")} {
		$c itemconfigure $item -foreground $x11colorname
	} elseif {[$c type $item] == "stripchart"} {
	    $c itemconfigure $item -fill $x11colorname
	}
    }
}

proc tkined_NODE_raise { node } {
    set c [$node canvas]
    foreach item [$node items] {
	set tags [$c gettags $item]
	if {[lsearch $tags LABEL] >= 0} { set label $item }
	if {[lsearch $tags labelclipbox] >= 0} { set box $item }
	$c raise $item
    }    
    catch {$c lower $box $label}
}

proc tkined_NODE_lower { node } {
    set c [$node canvas]
    foreach item [$node items] {
	set tags [$c gettags $item]
	if {[lsearch $tags LABEL] >= 0} { set label $item }
	if {[lsearch $tags labelclipbox] >= 0} { set box $item }
	$c lower $item
    }    
    catch {$c lower $box $label}
}

proc tkined_NODE_select { node } {
    set c [$node canvas]
    foreach item [$node items] {
	if {[lsearch [$c gettags $item] NODE] >=0 } {
	    $c addtag selected withtag $item
	    tkined_mark_box $c $item
	    break
	}
    }
}

proc tkined_NODE_unselect { node } {
    set c [$node canvas]
    foreach item [$node items] {
	$c dtag $item selected
	$c delete mark$item
    }
}

proc tkined_NODE_size { node } {
    set c [$node canvas]
    foreach item [$node items] {
	if {[lsearch [$c gettags $item] NODE]>=0} { 
	    return [$c bbox $item] 
	}
    }
    return ""
}

##
## The following set of procedures handle group objects.
##

proc tkined_GROUP_canvas { group } {
    $group collapse
}

proc tkined_GROUP_delete { group } {
    eval [$group canvas] delete [$group items]
}

proc tkined_GROUP_move { group dx dy } {
    set c [$group canvas]
    foreach item [$group items] {
	$c move $item $dx $dy
	foreach ii [$c find withtag mark$item] {
	    $c move $ii $dx $dy
	}
    }
}

proc tkined_GROUP_icon { group } {
    set c [$group canvas]
    set name [$group icon]
    set fname [tkined_editor_attribute [winfo parent $c] groupicon$name]
    if {$fname==""} {
	set fname "$name.bm"
	if {[tkined_find_file $fname]==""} {
	    set fname "group.bm"
	}
    }
    if {[set fname [tkined_find_file $fname]]==""} return
    foreach item [$group items] {
	if {[$c type $item]=="bitmap"} {
	    $c itemconfigure $item -bitmap @$fname
	    break
	}
    }
}

proc tkined_GROUP_label_clear { group } {
    set c [$group canvas]
    set itemlist ""
    foreach item [$group items] {
	set tags [$c gettags $item]
	if {[lsearch $tags LABEL]>=0 
	    || [lsearch $tags labelclipbox]>=0} {
		$c delete $item
	    } else {
		lappend itemlist $item
	    }
    }
    $group items $itemlist
}

proc tkined_GROUP_label { group {text ""}} {
    set c [$group canvas]
    set color [lindex [$c configure -background] 4]
    foreach item [$group items] {
	set tags [$c gettags $item]
	if {[lsearch $tags LABEL]>=0} { set label $item }
	if {[lsearch $tags labelclipbox]>=0} { set clipbox $item }
    }
    if {[info exist label]} {
	$c itemconfigure $label -text $text
	if {[info exist clipbox]} {
	    eval $c coords $clipbox [$c bbox $label]
	}
	return
    }
    # create a new label with and a box behind it
    foreach item [$group items] {
	if {[lsearch [$c gettags $item] GROUP]>=0} break
    }
    set bb [$c bbox $item]
    set x1 [lindex $bb 0]
    set x2 [lindex $bb 2]
    set y [expr {[lindex $bb 3]+1}]
    set x [expr {$x1+(($x2-$x1)/2)}]
    set label [$c create text $x $y -anchor n -text $text -font fixed \
	       -tags [list LABEL "id $group"] ]

    set tags [list labelclipbox "id $group"]
    set clipbox [eval $c create rectangle [$c bbox $label] \
		 -tags {$tags} -fill $color \
		     -outline $color -width 0 ]
    $c lower $clipbox $label
    $group items "[$group items] $label $clipbox"
    $group font [$group font]
}

proc tkined_GROUP_font { group } {
    set c [$group canvas]
    set fontname [$group font]
    set x11fontname [tkined_editor_attribute [winfo parent $c] font$fontname]
    if {$x11fontname == ""} { 
	set x11fontname fixed
    }
    foreach item [$group items] {
	if {[lsearch [$c gettags $item] LABEL] >= 0} {
	    $c itemconfigure $item -font $x11fontname
	}
    }
}

proc tkined_GROUP_color { group } {
    set c [$group canvas]
    set w [winfo parent $c]
    set colorname [$group color]
    if {[tk colormodel .]=="color"} {
	set x11colorname [tkined_editor_attribute $w color$colorname]
	if {$x11colorname == ""} { set x11colorname $colorname }
    } else {
	set x11colorname black
    }
    foreach item [$group items] {
	if {[lsearch [$c gettags $item] GROUP] >= 0} {
	    switch [$c type $item] {
		bitmap { $c itemconfigure $item -foreground $x11colorname }
		line   { $c itemconfigure $item -fill $x11colorname }
	    }
	}
    }
}

proc tkined_GROUP_raise { group } {
    set c [$group canvas]
    foreach item [$group items] {
	set tags [$c gettags $item]
	if {[lsearch $tags LABEL] >= 0} { set label $item }
	if {[lsearch $tags labelclipbox] >= 0} { set box $item }
	$c raise $item
    }    
    catch {$c lower $box $label}
}

proc tkined_GROUP_lower { group } {
    set c [$group canvas]
    foreach item [$group items] {
	set tags [$c gettags $item]
	if {[lsearch $tags LABEL] >= 0} { set label $item }
	if {[lsearch $tags labelclipbox] >= 0} { set box $item }
	$c lower $item
    }    
    catch {$c lower $box $label}
}

proc tkined_GROUP_collapse { group } {
    set c [$group canvas]
    eval $c delete [$group items]

    set x [lindex [$group move] 0]
    set y [lindex [$group move] 1]
    
    foreach id [$group member] {
	eval $c delete [$id items]
	$id items ""
    }

    set item [$c create bitmap $x $y -bitmap group \
	      -tags [list GROUP "id $group"] ]
    $group items $item
}

proc tkined_GROUP_expand { group } {
    set c [$group canvas]
    eval $c delete [$group items]

    set memberitems ""
    foreach id [$group member] {
	lappend memberitems [$id items]
    }
   
    if {$memberitems != ""} {
	set bb [eval $c bbox [join $memberitems]]
	set x1 [expr {[lindex $bb 0]-3}]
	set y1 [expr {[lindex $bb 1]-3}]
	set x2 [expr {[lindex $bb 2]+3}]
	set y2 [expr {[lindex $bb 3]+3}]
    } else {
	set bb [$group move]
	set x1 [expr {[lindex $bb 0]-30}]
        set y1 [expr {[lindex $bb 1]-30}]
        set x2 [expr {[lindex $bb 0]+30}]
        set y2 [expr {[lindex $bb 1]+30}]
    }

    set tags [list GROUP "id $group"]
    set item [eval $c create line $x1 $y1 $x1 $y2 $x2 $y2 $x2 $y1 $x1 $y1 \
	      -width 2 -fill Black -stipple gray50 -joinstyle miter \
		  -tags {$tags}]
    $group items $item
}

proc tkined_GROUP_select { group } {
    set c [$group canvas]
    foreach item [$group items] {
	if {[lsearch [$c gettags $item] GROUP] >=0 } {
	    $c addtag selected withtag $item
	    tkined_mark_box $c $item
	    break
	}
    }
}

proc tkined_GROUP_unselect { group } {
    set c [$group canvas]
    foreach item [$group items] {
	$c dtag $item selected
	$c delete mark$item
    }
}

proc tkined_GROUP_size { group } {
    set c [$group canvas]
    foreach item [$group items] {
	if {[lsearch [$c gettags $item] GROUP]>=0} {
	    return [$c bbox $item]
	}
    }
    return ""
}


##
## The following set of procedures handle network objects.
##

proc tkined_NETWORK_canvas { network } {
    set c [$network canvas]
    set points [join [$network points]]
    set tags [list NETWORK "id $network"]
    set item [eval $c create line $points -width 3 -fill black -tags {$tags} ]
    eval $c move $item [$network move]
    $network items $item
}

proc tkined_NETWORK_delete { network } {
    eval [$network canvas] delete [$network items]
}

proc tkined_NETWORK_move { network dx dy } {
    set c [$network canvas]
    foreach item [$network items] {
	$c move $item $dx $dy
	$c move mark$item $dx $dy
    }
}

proc tkined_NETWORK_label_clear { network } {
    set c [$network canvas]
    set itemlist ""
    foreach item [$network items] {
	set tags [$c gettags $item]
	if {[lsearch $tags LABEL]>=0 
	    || [lsearch $tags labelclipbox]>=0} {
		$c delete $item
	    } else {
		lappend itemlist $item
	    }
    }
    $network items $itemlist
}

proc tkined_NETWORK_label { network {text ""}} {
    set c [$network canvas]
    set color [lindex [$c configure -background] 4]
    foreach item [$network items] {
	set tags [$c gettags $item]
        if {[lsearch $tags LABEL]>=0} { set label $item }
        if {[lsearch $tags labelclipbox]>=0} { set clipbox $item }
    }
    if {[info exist label]} {
        $c itemconfigure $label -text $text
        if {[info exist clipbox]} {
            eval $c coords $clipbox [$c bbox $label]
        }
	return
    }
    # create a new label with and a box behind it
    foreach item [$network items] {
	set tags [$c gettags $item]
	if {[lsearch $tags NETWORK]>=0} break
    }
    set xy [$network labelxy]
    set tags [list LABEL "id $network"]
    set label [eval $c create text $xy -anchor n -text {$text} -font fixed \
	       -tags {$tags} ]
    set tags [list labelclipbox "id $network"]
    set clipbox [eval $c create rectangle [$c bbox $label] \
		 -tags {$tags} -fill $color \
		     -outline $color -width 0 ]
    $c lower $clipbox $label
    $network items "[$network items] $label $clipbox"
    $network font [$network font]
}

proc tkined_NETWORK_icon { network } {
    set c [$network canvas]
    set name [$network icon]
    set width [tkined_editor_attribute [winfo parent $c] network$name]
    if {$width == ""} { set width "3" }
    foreach item [$network items] {
	if {[$c type $item]=="line"} {
	    $c itemconfigure $item -width $width
	    break
	}
    }
}

proc tkined_NETWORK_font { network } {
    set c [$network canvas]
    set fontname [$network font]
    set x11fontname [tkined_editor_attribute [winfo parent $c] font$fontname]
    if {$x11fontname == ""} { 
	set x11fontname fixed
    }
    foreach item [$network items] {
	if {[lsearch [$c gettags $item] LABEL] >= 0} {
	    $c itemconfigure $item -font $x11fontname
	}
    }
}

proc tkined_NETWORK_color { network } {
    set c [$network canvas]
    set w [winfo parent $c]
    set colorname [$network color]
    if {[tk colormodel .]=="color"} {
	set x11colorname [tkined_editor_attribute $w color$colorname]
	if {$x11colorname == ""} { set x11colorname $colorname }
    } else {
	set x11colorname black
    }
    foreach item [$network items] {
	if {([lsearch [$c gettags $item] NETWORK] >= 0) 
	    && ([$c type $item] == "line")} {
		$c itemconfigure $item -fill $x11colorname
	}
    }
}

proc tkined_NETWORK_raise { network } {
    set c [$network canvas]
    foreach item [$network items] {
	set tags [$c gettags $item]
	if {[lsearch $tags LABEL] >= 0} { set label $item }
	if {[lsearch $tags labelclipbox] >= 0} { set box $item }
	$c raise $item
    }    
    catch {$c lower $box $label}
}

proc tkined_NETWORK_lower { network } {
    set c [$network canvas]
    foreach item [$network items] {
	set tags [$c gettags $item]
	if {[lsearch $tags LABEL] >= 0} { set label $item }
	if {[lsearch $tags labelclipbox] >= 0} { set box $item }
	$c lower $item
    }    
    catch {$c lower $box $label}
}

proc tkined_NETWORK_select { network } {
    set c [$network canvas]
    foreach item [$network items] {
	if {[lsearch [$c gettags $item] NETWORK] >=0 } {
	    $c addtag selected withtag $item
	    tkined_mark_points $c $item
	    break
	}
    }
}

proc tkined_NETWORK_unselect { network } {
    set c [$network canvas]
    foreach item [$network items] {
	$c dtag $item selected
	$c delete mark$item
    }
}

proc tkined_NETWORK_size { network } {
    set c [$network canvas]
    foreach item [$network items] {
	if {[lsearch [$c gettags $item] NETWORK]>=0} {
	    return [$c bbox $item]
	}
    }
    return ""
}


##
## The following set of procedures handle link objects.
##

proc tkined_LINK_canvas {link} {
    set c [$link canvas]
    set points [join [$link points]]
    set len [llength $points]
    if {$len%2!=0} { 
	incr len -2
	set points [join [lrange $points 1 len]]
	inr len
    }
    set xya [[$link ida] move]
    set xyb [[$link idb] move]
    set tags [list LINK "id $link"]
    set item [eval $c create line $xya $points $xyb -fill black -tags {$tags}]
    $link items $item
}

proc tkined_LINK_delete { link } {
    eval [$link canvas] delete [$link items]
}

proc tkined_LINK_move { link dx dy } {
    set c [$link canvas]
    foreach item [$link items] {
	$c move $item $dx $dy
	$c move mark$item $dx $dy
    }
}

proc tkined_LINK_color { link } {
    set c [$link canvas]
    set w [winfo parent $c]
    set colorname [$link color]
    if {[tk colormodel .]=="color"} {
	set x11colorname [tkined_editor_attribute $w color$colorname]
	if {$x11colorname == ""} { set x11colorname $colorname }
    } else {
	set x11colorname black
    }
    foreach item [$link items] {
	if {([lsearch [$c gettags $item] LINK] >= 0) 
	    && ([$c type $item] == "line")} {
		$c itemconfigure $item -fill $x11colorname
	}
    }
}

proc tkined_LINK_raise { link } {
    set c [$link canvas]
    foreach item [$link items] {
	$c raise $item
    }    
}

proc tkined_LINK_lower { link } {
    set c [$link canvas]
    foreach item [$link items] {
	$c lower $item
    }    
}

proc tkined_LINK_select { link } {
    set c [$link canvas]
    foreach item [$link items] {
	if {[lsearch [$c gettags $item] LINK] >=0 } {
	    $c addtag selected withtag $item
	    tkined_mark_points $c $item
	    break
	}
    }
}

proc tkined_LINK_unselect { link } {
    set c [$link canvas]
    foreach item [$link items] {
	$c dtag $item selected
	$c delete mark$item
    }
}

proc tkined_LINK_size { link } {
    set c [$link canvas]
    foreach item [$link items] {
	if {[lsearch [$c gettags $item] LINK]>=0} {
	    return [$c bbox $item]
	}
    }
    return ""
}


##
## The following set of procedures handle text objects.
##

proc tkined_TEXT_canvas { text } {
    set c [$text canvas]
    set w [winfo parent $c]
    set x [lindex [$text move] 0]
    set y [lindex [$text move] 1]
    set txt [$text text]
    regsub -all "\\\\n" $txt "\n" txt
    set item [$c create text $x $y -anchor nw -text $txt -font fixed \
	      -tags [list TEXT "id $text"]  ]
    $text items $item
}

proc tkined_TEXT_delete { text } {
    eval [$text canvas] delete [$text items]
}

proc tkined_TEXT_move { text dx dy } {
    set c [$text canvas]
    foreach item [$text items] {
	$c move $item $dx $dy
	$c move mark$item $dx $dy
    }
}

proc tkined_TEXT_font { text } {
    set c [$text canvas]
    set fontname [$text font]
    set x11fontname [tkined_editor_attribute [winfo parent $c] font$fontname]
    if {$x11fontname == ""} { 
	set x11fontname fixed
    }
    foreach item [$text items] {
	if {[lsearch [$c gettags $item] TEXT] >= 0} {
	    $c itemconfigure $item -font $x11fontname
	}
    }
}

proc tkined_TEXT_color { text } {
    set c [$text canvas]
    set w [winfo parent $c]
    set colorname [$text color]
    if {[tk colormodel .]=="color"} {
	set x11colorname [tkined_editor_attribute $w color$colorname]
	if {$x11colorname == ""} { set x11colorname $colorname }
    } else {
	set x11colorname black
    }
    foreach item [$text items] {
	if {([lsearch [$c gettags $item] TEXT] >= 0) 
	    && ([$c type $item] == "text")} {
		$c itemconfigure $item -fill $x11colorname
	}
    }
}

proc tkined_TEXT_raise { text } {
    set c [$node canvas]
    foreach item [$text items] {
	$c raise $item
    }    
}

proc tkined_TEXT_lower { text } {
    set c [$node canvas]
    foreach item [$text items] {
	$c lower $item
    }    
}

proc tkined_TEXT_select { text } {
    set c [$text canvas]
    foreach item [$text items] {
	if {[lsearch [$c gettags $item] TEXT] >=0 } {
	    $c addtag selected withtag $item
	    tkined_mark_box $c $item
	    break
	}
    }
}

proc tkined_TEXT_unselect { text } {
    set c [$text canvas]
    foreach item [$text items] {
	$c dtag $item selected
	$c delete mark$item
    }
}

proc tkined_TEXT_size { text } {
    set c [$text canvas]
    foreach item [$text items] {
	if {[lsearch [$c gettags $item] TEXT]>=0} {
	    return [$c bbox $item]
	}
    }
    return ""
}


##
## The following set of procedures handle image objects.
##

proc tkined_IMAGE_canvas { image } {
    set c [$image canvas]
    set fname [$image name]
    set x [lindex [$image move] 0]
    set y [lindex [$image move] 1]
    set tags [list IMAGE "id $image"]
    if {[catch {$c create bitmap $x $y -bitmap @$fname -tags $tags} item]} {
	tkined_acknowledge $c "Image file not readable!"
	$image canvas ""
	return
    }
    $image items $item
    $image lower
}

proc tkined_IMAGE_delete { image } {
    eval [$image canvas] delete [$image items]
}

proc tkined_IMAGE_move { image dx dy } {
    set c [$image canvas]
    foreach item [$image items] {
	$c move $item $dx $dy
	$c move mark$item $dx $dy
    }
}

proc tkined_IMAGE_color { image } {
    set c [$image canvas]
    set w [winfo parent $c]
    set colorname [$image color]
    if {[tk colormodel .]=="color"} {
	set x11colorname [tkined_editor_attribute $w color$colorname]
	if {$x11colorname == ""} { set x11colorname $colorname }
    } else {
	set x11colorname black
    }
    foreach item [$image items] {
	if {([lsearch [$c gettags $item] IMAGE] >= 0) 
	    && ([$c type $item] == "bitmap")} {
		$c itemconfigure $item -foreground $x11colorname
	}
    }
}

proc tkined_IMAGE_lower { image } {
    set c [$image canvas]
    foreach item [$image items] {
	$c lower $item
    }    
}

proc tkined_IMAGE_select { image } {
    set c [$image canvas]
    foreach item [$image items] {
	if {[lsearch [$c gettags $item] IMAGE] >=0 } {
	    $c addtag selected withtag $item
	    tkined_mark_box $c $item
	    break
	}
    }
}

proc tkined_IMAGE_unselect { image } {
    set c [$image canvas]
    foreach item [$image items] {
	$c dtag $item selected
	$c delete mark$item
    }
}

proc tkined_IMAGE_size { image } {
    set c [$image canvas]
    foreach item [$image items] {
	if {[lsearch [$c gettags $item] IMAGE]>=0} {
	    return [$c bbox $item]
	}
    }
    return ""
}


##
## The following set of procedures handle tool objects.
##

proc tkined_TOOL_canvas { tool } {
    set interpreter [$tool interpreter]
    set c [$tool canvas]
    set w [winfo parent $c]
    set name $w.tools.[$tool id]
    menubutton $name -text [$tool name] -menu $name.m
    menu $name.m
    foreach cmd [$tool items] {
	if {$cmd==""} {
	    $name.m add separator
	} else {
	    $name.m add command -label $cmd \
		-command "$interpreter send \"$cmd\""
	}
    }
    pack $name -side top -fill x
    $interpreter items "$tool [$interpreter items]"
    set tools [tkined_editor_attribute $w tools]
    lappend tools [list [$tool name] [$interpreter $name]]
    tkined_editor_attribute $w tools $tools
}

proc tkined_TOOL_delete { tool } {
    set c [$tool canvas]
    set w [winfo parent $c]
    destroy $w.tools.[$tool id]
}



##
## The following set of procedures handle interpreter objects.
##

proc tkined_INTERPRETER_queue { interpreter qlen } {
    set c [$interpreter canvas]
    set w [winfo parent $c]
    if {$qlen > 0} { set state disabled } { set state normal }
    foreach tool [$interpreter items] {
	$w.tools.$tool configure -state $state
	set last [$w.tools.$tool.m index last]
	if {$last == "none"} continue
	for {set idx 0} {$idx <= $last} {incr idx} {
	    catch {$w.tools.$tool.m entryconfigure $idx -state $state}
	}
    }
}


##
## The following set of procedures handle log objects.
##

proc tkined_LOG_canvas { log } {

    # The offset used to position log windows automatically.
    static offset
    if {![info exists offset]} {
	set offset 0
    } else {
	incr offset 10
    }

    set c [$log canvas]
    set w [winfo parent $c]
    toplevel $w.$log

    frame $w.$log.button 
    button $w.$log.button.ok    -padx 5 -text dismiss \
	-command "$log delete"
    button $w.$log.button.clear -padx 5 -text clear \
	-command "$log clear"
    button $w.$log.button.save  -padx 5 -text save \
	-command "tkined_LOG_save $log"
    button $w.$log.button.load  -padx 5 -text load \
	-command "tkined_LOG_load $log"
    button $w.$log.button.print  -padx 5 -text print \
	-command "tkined_LOG_print $log"
    button $w.$log.button.email -padx 5 -text email \
	-command "tkined_LOG_email $log"
    button $w.$log.button.up -padx 5 -text up \
	-command "tkined_LOG_up $log"
    button $w.$log.button.down -padx 5 -text down \
	-command "tkined_LOG_down $log"
    pack $w.$log.button.clear -fill both -pady 5 -padx 2 -side left
    pack $w.$log.button.up    -fill both -pady 5 -padx 2 -side left
    pack $w.$log.button.down  -fill both -pady 5 -padx 2 -side left
    pack $w.$log.button.load  -fill both -pady 5 -padx 2 -side left
    pack $w.$log.button.save  -fill both -pady 5 -padx 2 -side left
    pack $w.$log.button.print -fill both -pady 5 -padx 2 -side left
    pack $w.$log.button.email -fill both -pady 5 -padx 2 -side left
    pack $w.$log.button.ok    -fill both -pady 5 -padx 2 -side left
    pack $w.$log.button -anchor w

    scrollbar $w.$log.scrollbar -command "$w.$log.text yview" -relief sunken
    text $w.$log.text -height 24 -width 80 -setgrid true \
	-relief sunken -borderwidth 2 \
	-yscrollcommand "$w.$log.scrollbar set" 
    pack $w.$log.text -side left -padx 2 -pady 2 -fill y -expand true
    pack $w.$log.scrollbar -side left -fill y -expand true

    $log items $w.$log

    # This special purpose binding makes it possible to send
    # complete lines back to the interpreter that created this
    # window. This allows us to use a log window as a simple
    # command frontend.

    bind $w.$log.text <Shift-Return> "tkined_LOG_process $log"
    bind $w.$log.text <Control-n> "tkined_LOG_clear $log"
    bind $w.$log.text <Control-o> "tkined_LOG_load $log"
    bind $w.$log.text <Control-s> "tkined_LOG_save $log"
    bind $w.$log.text <Control-q> "$log delete"

    # Position the log window on the screen.

    wm withdraw $w.$log
    update idletasks
    set top [winfo toplevel $w]

    set rx [expr [winfo rootx $top]+[winfo vrootx $top]]
    set ry [expr [winfo rooty $top]+[winfo vrooty $top]]

    set cx [expr $rx+[winfo width $top]/4]
    set cy [expr $ry+[winfo height $top]/4]

    set x  [expr $cx+$offset]
    set y  [expr $cy+$offset]

    if {$x < 0} { set x 0 }
    if {$y < 0} { set y 0 }

    wm geometry $w.$log +$x+$y
    wm deiconify $w.$log    
}

proc tkined_LOG_process { log } {
    set w [$log items].text
    set i1 [$w index "insert linestart"]
    set i2 [$w index "insert"]

    puts stderr \
	"** [$log interpreter] send ined append $log \[[$w get $i1 $i2]\]"
    catch {eval [$log interpreter] send ined append $log \[[$w get $i1 $i2]\]} err
    puts stderr "** with result $err"
    $w insert insert "\n"
}

proc tkined_LOG_name { log } {
    wm title [$log items] [$log name]
    wm iconname [$log items] [$log name]
}

proc tkined_LOG_icon { log } {
    wm iconbitmap [$log items] [$log icon]
}

proc tkined_LOG_append { log line } {
    [$log items].text insert end "$line\n"
    [$log items].text yview -pickplace end
}

proc tkined_LOG_clear { log } {
    [$log items].text delete 0.0 end
}

proc tkined_LOG_delete { log } {
    destroy [$log items]
}

proc tkined_LOG_up { log } {
    [$log items].text yview "@0,0 - 24 lines"
}

proc tkined_LOG_down { log } {
    [$log items].text yview "@0,0 + 24 lines"
}

proc tkined_LOG_save { log } {
    set fname [tkined_file_select [$log items] "Write to file:"]
    if {$fname==""} return

    if {[file exists $fname]} {
	if {[tkined_confirm [$log items] "Replace file $fname?"] != "yes"} {
	    return
	}
    }

    if {[catch {open $fname w+} file]} {
	tkined_acknowledge [$log items] "Unable to write to $fname"
	return
    }

    puts $file [[$log items].text get 1.0 end]
    close $file
}

proc tkined_LOG_load { log } {
    set fname [tkined_file_select [$log items] "Read from file:"]
    if {$fname==""} return

    if {[catch {open $fname r} file]} {
	tkined_acknowledge [$log items] "Unable to read from $fname"
	return
    }

    $log clear
    while {![eof $file]} {
	gets $file line
	$log append $line
	[$log items].text yview 1.0
    }
    close $file
}

proc tkined_LOG_print { log } {

    set fname "/tmp/tkined.asc"
    catch {exec /bin/rm -f $fname}
    if {[file exists $fname] && ![file writable $fname]} {
	tkined_acknowledge [$log items] "Can not write temporary file $fname"
	return
    }

    set file [open $fname w]
    puts $file [[$log items].text get 1.0 end]
    close $file

    tkined_print [$log items] $fname

    catch {exec /bin/rm -f $fname}
}

proc tkined_LOG_email { log } {
    global env

    set result [tkined_request [$log items] "Please enter the email address:" \
		"{To: [$log address]} {Subject: {[$log name]}}"]
    if {$result == ""} return

    set to [lindex $result 0]
    $log address $to
    set subject [lindex $result 1]

    if {[catch {split $env(PATH) :} path]} {
	set path "/usr/bin /bin /usr/ucb /usr/local/bin"
    }

    set mprog ""
    foreach mailer "Mail mail" {
	foreach dir $path {
	    if [file executable $dir/Mail] {
		set mprog $dir/$mailer
		break
	    }
	}
	if {$mprog != ""} break
    }

    if {$mprog == ""} {
	tkined_acknowledge [$log items] "Sorry, can not find mail program."
	return
    }

    if {[catch {open "|$mprog -s \"$subject\" $to" w} file]} {
        tkined_acknowledge [$log items] "Unable to write to $mprog $to"
        return
    }

    puts $file [[$log items].text get 1.0 end]
    close $file
}

