##
## tkined_tools.tcl
##
## This file contains the tools of the tkined editor.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##


##
## This procedure is called whenever a tool is seleed in the tool
## panel. It sets the relief of the button to sunken and creates new 
## key bindings for the selected tool.
##

proc tkined_select_tool {button c} {
    static tool
    foreach child [winfo children [winfo parent $button]] {
	$child configure -relief flat
    }
    $button configure -relief groove
    catch {if {$tool=="text"} {tkined_textDone $c}}

    $c focus {}; focus [winfo parent $c]

##    catch {if {$tool=="label"} {puts "should finish label"}}
    set tool [winfo name $button]
    bind $c <1> ""
    bind $c <B1-Motion> ""
    bind $c <ButtonRelease-1> ""
    bind $c <Shift-Button-1> ""
    bind $c <Motion> ""
    bind $c <Shift-Motion> ""
    switch $tool {
        select {
	    bind $c <1> \
		"tkined_clear_selection $c; \
                    tkined_apply_select_tool $c \
                    \[$c canvasx %x\] \[$c canvasy %y\]"
	    bind $c <Shift-Button-1> \
		"tkined_apply_select_tool $c \
		    \[$c canvasx %x\] \[$c canvasy %y\]"
	}
	move {
	    bind $c <1> "tkined_moveMark $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
	    bind $c <B1-Motion> "tkined_moveDrag $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
	    bind $c <ButtonRelease-1> "tkined_moveDone $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
	}
	text {
	    bind $c <1> "tkined_textMark $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
	}
	label {
	    bind $c <1> "tkined_labelMark $c \
                \[$c canvasx %x\] \[$c canvasy %y\] %X %Y"
	}
	node {
	    bind $c <1> "tkined_nodeDone $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
	}
	network {
	    bind $c <1> "tkined_netMark $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
	}
	link {
	    bind $c <1> "tkined_linkMark $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
	}
	group {
	    bind $c <1> "tkined_groupDone $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
	}
    }
}

##
## Implementation of the select tool.
##

proc tkined_apply_select_tool {c x y} {
    set x1 [expr {$x-5}]
    set y1 [expr {$y-5}]
    set x2 [expr {$x+5}]
    set y2 [expr {$y+5}]
    foreach item [$c find overlapping $x1 $y1 $x2 $y2] {
	set item_id [tkined_getid $c $item]
	if {$item_id == ""} continue
	set id $item_id
    }
    if {![catch {$id selected} stat]} {
	if {$stat} { $id unselect } else { $id select }
    }
}

proc tkined_selectMark {c x y} {
    global tkined_areaX1 tkined_areaY1 tkined_areaX2 tkined_areaY2 \
	tkined_cursor tkined_valid
    set x1 [expr {$x-5}]
    set y1 [expr {$y-5}]
    set x2 [expr {$x+5}]
    set y2 [expr {$y+5}]
    set tkined_valid 1
    foreach item [$c find overlapping $x1 $y1 $x2 $y2] {
	set tkined_valid 0
	set tags [$c gettags $item]
	if {[lsearch $tags IMAGE]<0} break
	set tkined_valid 1
    }
    if {$tkined_valid} {
	set tkined_areaX1 $x
	set tkined_areaY1 $y
	set tkined_areaX2 $x
	set tkined_areaY2 $y
	set tkined_cursor nope
	$c delete area
    } else {
	tkined_apply_select_tool $c $x $y
    }
}

proc tkined_selectDrag {c x y} {
    global tkined_areaX1 tkined_areaY1 tkined_areaX2 tkined_areaY2 \
	tkined_cursor tkined_valid
    if {!$tkined_valid} return
    if {($tkined_areaX1 != $x) && ($tkined_areaY1 != $y)} {
	$c delete area
	$c create rect $tkined_areaX1 $tkined_areaY1 \
	    $x $y -outline black -tags "area"
	set tkined_areaX2 $x
	set tkined_areaY2 $y
    }
    if {$tkined_areaX2>$tkined_areaX1} {
	if {$tkined_areaY2>$tkined_areaY1} {
	    set new_cursor bottom_right_corner
	} else {
	    set new_cursor top_right_corner
	}
    } else {
	if {$tkined_areaY2>$tkined_areaY1} {
	    set new_cursor bottom_left_corner
	} else {
	    set new_cursor top_left_corner
	}
    }
    if {$new_cursor!="$tkined_cursor"} {
	set tkined_cursor $new_cursor
	$c configure -cursor $tkined_cursor
    }
}

proc tkined_selectDone {c} {
    global tkined_areaX1 tkined_areaY1 tkined_areaX2 tkined_areaY2 \
	tkined_valid
    if {!$tkined_valid} return
    set w [winfo parent $c]
    $w.tools.select configure -state disabled
    update idletask
    set area [$c find withtag area]
    foreach item [$c find enclosed $tkined_areaX1 $tkined_areaY1 \
		  $tkined_areaX2 $tkined_areaY2] {
        set id [tkined_getid $c $item]
        if {$id == ""} continue
	set table($id) ""
    }
    if {[info exist table]} {
	foreach id [array names table] {
	    if {[catch {$id selected} sel]} continue
	    if {$sel} { $id unselect } else { $id select }
	}
    }
    $c delete area
    $c configure -cursor top_left_arrow
    $w.tools.select configure -state normal
}

##
## Implementation of the move tool.
##

proc tkined_moveMark {c x y} {
    global tkined_startX tkined_startY tkined_oldX tkined_oldY tkined_valid
    set x1 [expr {$x-5}]
    set y1 [expr {$y-5}]
    set x2 [expr {$x+5}]
    set y2 [expr {$y+5}]
    set tkined_valid 0
    foreach item [$c find overlapping $x1 $y1 $x2 $y2] {
	set id [tkined_getid $c $item]
	if {$id == ""} continue
	if {[catch {$id selected} sel]} continue
	if {[set tkined_valid $sel]} {
	    break
	}
    }
    if {!$tkined_valid} return
    set tkined_startX $x
    set tkined_startY $y
    set tkined_oldX $x
    set tkined_oldY $y
    set selection [$c find withtag selected]
    set tkined_valid [expr {$selection!=""}]
    foreach item [$c find overlapping $x $y $x $y] {
	if [expr {[lsearch [$c gettags $item] selected]>=0}] {
	    set tkined_valid 1
	    break
	}
    }
    if {$tkined_valid} {
	set area [eval $c bbox $selection]
	eval $c create rect $area -outline black -tags "area"
    }
}

proc tkined_moveDrag {c x y} {
    global tkined_startX tkined_startY tkined_oldX tkined_oldY tkined_valid
    static X Y
    if {!$tkined_valid} return
    set dx [expr $x-$tkined_oldX]
    set dy [expr $y-$tkined_oldY]
    if {($dx != 0) && ($dy != 0)} {
	$c move area $dx $dy
	set tkined_oldX $x
	set tkined_oldY $y
    }
}

proc tkined_moveDone {c x y} {
    global tkined_startX tkined_startY tkined_valid
    set redo_cmd ""
    set undo_cmd ""
    if {!$tkined_valid} return
    set w [winfo parent $c]
    $w.tools.move configure -state disabled
    update idletask
    set dx [expr $x-$tkined_startX]
    set dy [expr $y-$tkined_startY]
    set rx [expr {-1*$dx}]
    set ry [expr {-1*$dy}]
    set idlist ""
    if {($dx != 0) && ($dy != 0)} {
	foreach item [$c find withtag selected] {
	    set id [tkined_getid $c $item]
	    if {$id == ""} continue
	    lappend idlist $id
	    if {[$id type] == "GROUP"} {
		foreach m [$id member] {
		    set ignore($m) ""
		}
	    }
	}
    }
    foreach id $idlist {
	if {![info exists ignore($id)]} {
	    $id move $dx $dy
	    append redo_cmd "$id move $dx $dy; "
	    append undo_cmd "$id move $rx $ry; "
	}
    }
    if {$undo_cmd != ""} {
	tkined_undo_command [winfo parent $c] [list $undo_cmd $redo_cmd]
    }
    $c delete area
    $w.tools.move configure -state normal
}

##
## Implementation of the text tool. It can be used to enter text
## on the canvas (creating a text item). We currently do not support
## the selection mechanisms.
##

proc tkined_textBs {w item} {
    set char [expr {[$w index $item insert] - 1}]
    if {$char >= 0} {$w dchar $item $char}
}

proc tkined_textMark {c x y} {
    set w [winfo parent $c]
    set fontname [tkined_editor_attribute $w font]
    set x11fontname [tkined_editor_attribute $w font$fontname]
    if {$x11fontname == ""} {
	set x11fontname fixed
    }
    set colorname [tkined_editor_attribute $w color]
    set x11colorname [tkined_editor_attribute $w color$colorname]
    if {$x11colorname == ""} { set x11colorname black }  
    set item [$c create text $x $y -anchor nw -tags "text" \
	      -font $x11fontname -fill $x11colorname]
    $c bind $item <KeyPress>       "$c insert $item insert %A"
    $c bind $item <Shift-KeyPress> "$c insert $item insert %A"
    $c bind $item <Return>         "$c insert $item insert \\n"
    $c bind $item <BackSpace>      "tkined_textBs $c $item"
    $c bind $item <Control-h>      "tkined_textBs $c $item"
    $c bind $item <Delete>         "tkined_textBs $c $item"
    $c icursor $item @$x,$y
    $c focus $item
    focus $c
    $c select from $item @$x,$y
}

proc tkined_textDone {c} {
    set w [winfo parent $c]
    foreach item [$c find withtag text] {
	set xy [$c coords $item]
	foreach opt [$c itemconfigure $item] {
	    if {[lindex $opt 0]=="-text"} {
		set text [lindex $opt 4]
	    }
	}
	set text [TEXT create $text]
	$text canvas $c
	$text move  [lindex $xy 0] [lindex $xy 1]
	$text font  [tkined_editor_attribute $w font]
	$text color [tkined_editor_attribute $w color]
	$c delete $item
    }
    $c focus ""
    focus [winfo parent $c]
}

##
## The Label tool can be used to get and set informations about
## an object. The various attributes are shown in the label of
## the object.
##

proc tkined_labelMark {c cx cy xx xy} {
    set x1 [expr {$cx-5}]
    set y1 [expr {$cy-5}]
    set x2 [expr {$cx+5}]
    set y2 [expr {$cy+5}]
    set item ""
    foreach item [$c find overlapping $x1 $y1 $x2 $y2] {
	set id [tkined_getid $c $item]
	set tags [$c gettags $item]
	set t ""
	foreach aa "NODE GROUP NETWORK LABEL" {
	    if {[lsearch $tags $aa]>=0} { set t $aa }
	}
	if {$t!=""} break
	set item ""
    }
    if {$item==""} return
    case $t in {
	{NODE GROUP NETWORK} {
	    catch {destroy $c.popup}
	    menu $c.popup
	    $c.popup add command -label "name" \
		-command "$id label name; destroy $c.popup"
	    if {$t!="GROUP"} {
		$c.popup add command -label "address" \
		    -command "$id label address; destroy $c.popup"
	    }
	    $c.popup add command -label "text" \
		-command "$id label text; destroy $c.popup"
	    $c.popup add command -label "clear" \
		-command "$id label clear; focus [winfo parent $c]; destroy $c.popup"
	    $c.popup post [incr xx -20] [incr xy -20]
	    bind $c.popup <Any-Leave> {destroy %W}
	}
	LABEL {
            $c bind $item <Return>         "focus [winfo parent $c]"
	    $c bind $item <KeyPress>       "$c insert $item insert %A"
	    $c bind $item <Shift-KeyPress> "$c insert $item insert %A"
	    $c bind $item <BackSpace>      "tkined_textBs $c $item"
	    $c bind $item <Control-h>      "tkined_textBs $c $item"
	    $c bind $item <Control-a>      "$c icursor $item 0"
	    $c bind $item <Control-e>      "$c icursor $item end"
	    $c bind $item <Delete>         "tkined_textBs $c $item"
	    $c bind $item <2> "$c insert $item insert \[selection get\]"
	    # save the new label when we leave the current focus
	    bind $c <FocusOut>             "tkined_labelDone $c $id $item"
	    $c icursor $item @$cx,$cy
	    $c focus $item
	    focus $c
	}
    }
}

proc tkined_labelDone {c id item} {
    set txt [lindex [$c itemconfigure $item -text] 4]
    switch [$id label] {
	name {
	    $id name $txt
	}
	address {
	    $id address $txt
	}
	text {
	    $id label text $txt
	}
    }
}

##
## Implementation of the node tool which instantiates a new
## node object.
##

proc tkined_nodeDone {c x y} {
    set w [winfo parent $c]
    set node [NODE create]
    $node canvas $c
    $node move $x $y
    $node icon  [tkined_editor_attribute $w icon]
    $node font  [tkined_editor_attribute $w font]
    $node color [tkined_editor_attribute $w color]
    $node label name
}

##
## Implementation of the network tool.
##

proc tkined_netMark {c x y} {
    global tkined_points
    set tkined_points "$x $y"
    bind $c <1> "tkined_netDone $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
    bind $c <Shift-Button-1> "tkined_netAddMark $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
    bind $c <Motion> "tkined_netDrag $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
    bind $c <Shift-Motion> "tkined_netDrag $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
}

proc tkined_netDrag {c x y} {
    global tkined_points
    if ![info exists tkined_points] return
    if {$tkined_points==""} return
    $c delete network
    set len [llength $tkined_points]
    incr len -1
    set lasty [lindex $tkined_points $len]
    incr len -1
    set lastx [lindex $tkined_points $len]
    set dx [expr {$x>$lastx ? $x-$lastx : $lastx-$x}]
    set dy [expr {$y>$lasty ? $y-$lasty : $lasty-$y}]
    if {$dx>$dy} {set y $lasty} else {set x $lastx}
    eval $c create line "$tkined_points $x $y" -fill black -tags "network"
}

proc tkined_netAddMark {c x y} {
    global tkined_points
    if {$tkined_points==""} return
    $c delete network
    set len [llength $tkined_points]
    incr len -1
    set lasty [lindex $tkined_points $len]
    incr len -1
    set lastx [lindex $tkined_points $len]
    set dx [expr {$x>$lastx ? $x-$lastx : $lastx-$x}]
    set dy [expr {$y>$lasty ? $y-$lasty : $lasty-$y}]
    if {$dx>$dy} {set y $lasty} else {set x $lastx}
    append tkined_points " $x $y"
    eval $c create line $tkined_points -fill black -tags "network"
}

proc tkined_netDone {c x y} {
    global tkined_points
    if {$tkined_points==""} return
    set w [winfo parent $c]
    $c delete network
    set len [llength $tkined_points]
    incr len -1
    set lasty [lindex $tkined_points $len]
    incr len -1
    set lastx [lindex $tkined_points $len]
    set dx [expr {$x>$lastx ? $x-$lastx : $lastx-$x}]
    set dy [expr {$y>$lasty ? $y-$lasty : $lasty-$y}]
    if {$dx>$dy} {set y $lasty} else {set x $lastx}
    append tkined_points " $x $y"

    set network [eval NETWORK create $tkined_points]
    $network canvas $c
    $network icon  [tkined_editor_attribute $w network]
    $network font  [tkined_editor_attribute $w font]
    $network color [tkined_editor_attribute $w color]
    $network label name

    set tkined_points ""
    bind $c <1> "tkined_netMark $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
    bind $c <Motion> ""
    bind $c <Shift-Motion> ""
}

##
## Implementation of the link tool. It can be used to link two node
## objects or to link a node object and a network object.
##

proc tkined_linkMark {c x y} {
    global tkined_points
    set tkined_points ""
    set type ""
    set x1 [expr {$x-5}]
    set y1 [expr {$y-5}]
    set x2 [expr {$x+5}]
    set y2 [expr {$y+5}]
    foreach item [$c find overlapping $x1 $y1 $x2 $y2] {
	set tags [$c gettags $item]
	if {[lsearch $tags NODE]>=0} {
	    set type NODE
	    break
	}
	if {[lsearch $tags NETWORK]>=0} {
            set type NETWORK
            break
        }
    }
    set tkined_points "$x $y"
    if {($type=="NODE") || ($type=="NETWORK")} {
	bind $c <1> "tkined_linkDone $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
	bind $c <Shift-Button-1> "tkined_linkAddMark $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
	bind $c <Motion> "tkined_linkDrag $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
	bind $c <Shift-Motion> "tkined_linkShiftDrag $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
    }
}

proc tkined_linkDrag {c x y} {
    global tkined_points
    if ![info exists tkined_points] return
    if {$tkined_points==""} return
    $c delete line
    eval $c create line "$tkined_points $x $y" -fill black -tags "line"
}

proc tkined_linkShiftDrag {c x y} {
    global tkined_points
    if ![info exists tkined_points] return
    if {$tkined_points==""} return
    set len [llength $tkined_points]
    if {$len>2} {
	incr len -1
	set lasty [lindex $tkined_points $len]
	incr len -1
	set lastx [lindex $tkined_points $len]
	set dx [expr {$x>$lastx ? $x-$lastx : $lastx-$x}]
	set dy [expr {$y>$lasty ? $y-$lasty : $lasty-$y}]
	if {$dx>$dy} {set y $lasty} else {set x $lastx}
    }
    tkined_linkDrag $c $x $y
}

proc tkined_linkAddMark {c x y} {
    global tkined_points
    if {$tkined_points==""} return
    set len [llength $tkined_points]
    if {$len>2} {
	incr len -1
	set lasty [lindex $tkined_points $len]
	incr len -1
	set lastx [lindex $tkined_points $len]
	set dx [expr {$x>$lastx ? $x-$lastx : $lastx-$x}]
	set dy [expr {$y>$lasty ? $y-$lasty : $lasty-$y}]
	if {$dx>$dy} {set y $lasty} else {set x $lastx}
    }
    $c delete line
    append tkined_points " $x $y"
    eval $c create line $tkined_points -fill black -tags "line"
}

proc tkined_linkDone {c x y} {
    global tkined_points
    if {$tkined_points==""} return
    $c delete line
    set dst_type ""
    set x1 [expr {$x-5}]
    set y1 [expr {$y-5}]
    set x2 [expr {$x+5}]
    set y2 [expr {$y+5}]
    foreach item [$c find overlapping $x1 $y1 $x2 $y2] {
	set dst_id   [tkined_getid $c $item]
	set dst_type [$dst_id type]
        if {$dst_type == "NODE" || $dst_type == "NETWORK"} {
	    append tkined_points " [$dst_id move]"
	    break
	}
    }
    set sx [lindex $tkined_points 0]
    set sy [lindex $tkined_points 1]
    set src_type ""
    set x1 [expr {$sx-5}]
    set y1 [expr {$sy-5}]
    set x2 [expr {$sx+5}]
    set y2 [expr {$sy+5}]
    foreach item [$c find overlapping $x1 $y1 $x2 $y2] {
	set src_id   [tkined_getid $c $item]
	set src_type [$src_id type]
	if {$src_type == "NODE" || $src_type == "NETWORK"} break
    }
    if {($dst_type=="NODE") || 
	(($dst_type=="NETWORK") && ($src_type=="NODE"))} {
	    set len [llength $tkined_points]
	    incr len -3
	    set link [eval LINK create $src_id $dst_id \
		      [lrange $tkined_points 2 $len]]
	    $link canvas $c
	    $link color [tkined_editor_attribute [winfo parent $c] color]
    }
    set tkined_points ""
    bind $c <1> "tkined_linkMark $c \
                \[$c canvasx %x\] \[$c canvasy %y\]"
    bind $c <Motion> ""
    bind $c <Shift-Motion> ""
}

##
## Create a new group object that does not have any members.
##

proc tkined_groupDone {c x y} {
    set w [winfo parent $c]
    set group [GROUP create]
    $group canvas $c
    $group move $x $y
    $group icon  [tkined_editor_attribute $w groupicon]
    $group font  [tkined_editor_attribute $w font]
    $group color [tkined_editor_attribute $w color]
    $group label name
}

##
## Return the id of the object that this item belongs to.
## We just scan through the tags for entry of the format
## "id <id>".
##

proc tkined_getid {c item} {
    set id ""
    foreach tag [$c gettags $item] {
	if {[lindex $tag 0]=="id"} {
	    set id [lindex $tag 1]
	    break
	}
    }
    return $id
}

##
## Clear the whole selection. Get all items marked selected and
## call unselect for each of them.
##

proc tkined_clear_selection {c} {
    foreach id [tkined_selection $c] {
	$id unselect
    }
}
