#==============================================================================
#                                  util.tcl
#------------------------------------------------------------------------------
# Utility for parsing the command line and later rendering the current world
# space according to the parsed parameters.
#------------------------------------------------------------------------------
# $Id: util.tcl,v 3.0 1993/01/20 06:40:02 markd Rel $
#------------------------------------------------------------------------------

#
# Print a series of lines to stderr and exit.
#
proc ArgError {args} {
    foreach msg $args {
        puts stderr $msg
    }
    exit 1
}

#
# Print a usage error and exit.
#
proc Usage {msg} {
    ArgError $msg {... [-mode mode] [-size size] [-oversample n] -ppm -rle}
}

#
# Procedure parse the arguments and saves them for rendering time.
#
# Arguments:
#    o oversample - The oversampling value to use if not specified on command
#      line.  (optional, default is 2)
# Globals:
#   G_mode - PHONG, GOURAUD, FLAT or LINE
#   G_oversample - Oversampling factor.
#   G_size - The size of image to render (size X size).
#   G_rle - 1 if render to an RLE file, 0 if render to a PBM file.
#   G_suffix - Suffix to use for rendering.
#
proc ParseArgs {{oversample 2}} {
    global argv G_mode G_size G_oversample G_rle G_suffix
    set G_mode PHONG
    set G_size 256
    set G_oversample $oversample
    set G_rle [SippInfo RLE]

    while {![lempty $argv]} {
        set option [lvarpop argv]
        case $option {
            {-mode} {
                set G_mode [lvarpop argv]
                if {[lsearch {PHONG GOURAUD FLAT LINE} $G_mode] < 0} {
                    ArgError "Invalid mode `$G_mode'" \
                             "expected one of PHONG GOURAUD FLAT LINE"
                }
            }
            {-size} {
                set G_size [lvarpop argv]
                if {$G_size <= 0 || [catch {expr 0+$G_size}] != 0} {
                    ArgError "Invalid size `$G_size'"
                }
            }
            {-oversample} {
                set G_oversample [lvarpop argv]
                if {$G_oversample <= 0 || ![ctype digit $G_oversample] } {
                    ArgError "Invalid oversampling `$G_oversample'"
                }
            }
            {-ppm}  {
                set G_rle 0
            }
            {-rle}  {
                set G_rle 1
            }
            default {
              Usage "Invalid option `$option'"
            }
        }
    }
    if $G_rle {
        set G_suffix ".rle"
    } else {
        if {$G_mode == "LINE"} {
            set G_suffix ".pbm"
        } else {
            set G_suffix ".ppm"
        }
    }
}


#
# Do rendering based on the command line arguments that were parsed.
# baseName is the name of the file to render, less extensions.  For RLE files,
# Set a default image for viewing with getx11, since they were designed for
# a ppm viewer.
#
proc DoRendering {baseName} {
    global G_mode G_size G_oversample G_rle G_suffix

    set imfile_name "${baseName}${G_suffix}"

    puts stdout "Rendering $G_mode image to $imfile_name, wait..." nonewline
    flush stdout

    if $G_rle {
        set rle [SippRLEOpen $imfile_name "w"]
        SippRLEPutCom $rle image_gamma 0.4
        SippRLEPutCom $rle image_title $baseName
        SippRender $rle $G_size $G_size $G_mode $G_oversample
        SippRLEClose $rle
    } else {
        set fh [open $imfile_name "w"]
        SippPPMRender $fh $G_size $G_size $G_mode $G_oversample
        close $fh
    }

    puts stdout "Done."
}
