/*
 * photo.c --
 *	Image display widget for use with Tk.
 *
 * Copyright 1992 The Australian National University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, provided that the above copyright
 * notice appears in all copies.  This software is provided without any
 * warranty, express or implied. The Australian National University
 * makes no representations about the suitability of this software for
 * any purpose.
 *
 * Author: Paul Mackerras (paulus@cs.anu.edu.au)
 *
 * SCCS info: @(#)photo.c	2.3	last modified: 6/29/92
 */
#include <math.h>
#include <stdlib.h>
#include <tk.h>
#include <tclHash.h>
#include <X11/Xutil.h>
#include "photo.h"

#define FALSE	0
#define TRUE	1

#define MIN(a, b)	((a) < (b)? (a): (b))
#define MAX(a, b)	((a) > (b)? (a): (b))

/* When dithering, update screen after calculating DITHER_PIXELS pixels */
#define DITHER_PIXELS	20000

/* Number of bits in a byte */
#ifndef NBBY
#define NBBY		8
#endif

/*
 * Definition of data needed to map 24-bit colour values to
 * pixel values for a particular display and screen.
 * This information is shared among all photo widgets on the same
 * X display and screen.
 */

/* Information identifying a particular palette on a particular screen */
typedef struct {
    Display	*display;	/* display the window's on */
    int		screen;		/* screen number */
    Tk_Uid	palette;	/* # shades of each color required */
    double	gamma;		/* display gamma to correct for */
} ColorTableId;

/*
 * Stores color -> pixel value mapping information.
 */
typedef struct {
    ColorTableId id;		/* display & screen this applies to */
    Colormap	cmap;		/* colormap for displaying image */
    int		flags;		/* NEED_ALLOC, OWN_CMAP */
    int		refcnt;		/* # photos using this map */
    int		ncolors;	/* # colors allocated for this map */

    /* Mapping tables for converting RGB 8-bit values to pixel values */
    unsigned long	red_values[256];
    unsigned long	green_values[256];
    unsigned long	blue_values[256];
    unsigned long	*pixel_map;	/* list of pixel values allocated */

    unsigned char	color_quant[3][256];
} ColorTable;

/*
 * Definition of the data associated with each photo widget.
 */
typedef struct {
    int		flags;		/* see below */
    Tcl_Interp	*interp;	/* interpreter associated with object */

    Tk_Window	tkwin;		/* window for the photo to appear in */
    Display	*display;	/* display the window's on */
    Tk_Uid	palette;	/* specifies color allocations */
    ColorTable	*colors;	/* points to info about mapping colours */
    int		depth;		/* bits/pixel for the window */
    Visual	*visual;	/* X visual for the display window */
    XVisualInfo	vis_info;	/* information about visual */
    double	gamma;		/* correct for this display gamma */
    GC		monoGC;		/* GC for putting monochrome images */
    Tk_Window	main;		/* main window ancestor of this window */
    Colormap	mainCmap;	/* and its colormap */
    Cursor	cursor;		/* cursor for this window */

    Tk_3DBorder	bgBorder;	/* used for drawing a frame around image */
    Tk_3DBorder	activeBorder;	/* used for frame when mouse is in window */
    int 	width;		/* actual width of image display area */
    int 	height;		/* actual height of image display area */
    int		borderWidth;	/* width of border frame */
    int		relief;		/* appearance of border */
    int		reqWidth;	/* requested width of display area */
    int		reqHeight;	/* requested height */
    int		padx;		/* extra width between image and border */
    int		pady;		/* extra height */
    XColor	*blankColor;	/* colour for blank areas of image */
    GC		blankGC;	/* GC for filling in blankColor */
    char	*xScrollCmd;	/* command for X scrolling */
    char	*yScrollCmd;	/* command for Y scrolling */
    Tk_Uid	geometry;	/* string geometry spec for window */
    int		xorg;		/* X origin of image in window */
    int		yorg;		/* Y origin */
    int		xshift;		/* X image coord at window xorg */
    int		yshift;		/* Y image coord at window yorg */
    int		scanX;		/* X coord given in scan mark cmd */
    int		scanY;		/* Y coord given in scan mark cmd */
    int		scanSpeed;	/* amt to move window for unit mouse mvt */

    int		xscr_pos;	/* position we last told the x scrollbar */
    int		xscr_wid;	/* width ditto */
    int		xscr_pwid;	/* pixelWidth ditto */
    int		yscr_pos;	/* position we last told the y scrollbar */
    int		yscr_ht;	/* height ditto */
    int		yscr_pht;	/* pixelHeight ditto */

    Pixmap	pixels;		/* backing pixmap */
    int		pixelWidth;	/* width of backing pixmap */
    int		pixelHeight;	/* height of backing pixmap */

    int		ditherLevel;	/* 0=don't, 1=when asked, 2=each block */
    unsigned char	*pix24;		/* local storage for 24-bit image */
    char	*error;		/* error image used in dithering */
    int		ditherX;	/* image is correctly dithered up to */
    int		ditherY;	/* (not including) pixel (ditherX, ditherY) */

    Tk_Uid	imageGeom;	/* string giving widthxheight of image */
    int		imageWidth;	/* user-declared width of image */
    int		imageHeight;	/* user-declared height */

    XImage	*image;		/* image structure for converted pixels */
    int		imageSize;	/* # bytes available at image->data */

} Photo;

/*
 * Values for flags
 */
#define MAP_COLORS	1	/* using a pseudo-colour display */
#define SET_COLORMAP	2	/* need to set colormap when window exists */
#define OWN_CMAP	4	/* a colormap was allocated for this window */
#define EXPOSED		8	/* seen expose event => can draw in window */
#define REDRAW_FRAME	0x10	/* some part of frame needs redrawing */
#define REDRAW_SCROLL	0x20	/* image needs redrawing due to scroll */
#define USER_GEOMETRY	0x40	/* geometry was specified by user */
#define USER_IMAGE_GEOM	0x80	/* image size was specified by user */
#define ACTIVE		0x200	/* the mouse is in the window */
#define MONOCHROME	0x400	/* we're displaying a monochrome image */
#define RESTORE_CMAP	0x800	/* restore main window's cmap when leaving */
#define NEED_ALLOC	0x1000	/* need to allocate colors in ColorTable */
#define REDRAW_PENDING	(REDRAW_FRAME|REDRAW_SCROLL)

/*
 * Values for ditherLevel
 */
#define DITHER_NONE	0
#define DITHER_REQUEST	1
#define DITHER_BLOCKS	2

/*
 * Colour values for defaults
 */
#define BLACK		"#000000"
#define GREY50		"#808080"
#define BISQUE2		"#eed5b7"

/*
 * Default configuration
 */
#define DEF_PHOTO_ACTIVE_BG	BISQUE2
#define DEF_PHOTO_BG		GREY50
#define DEF_PHOTO_BLANK		BLACK
#define DEF_PHOTO_BORDER_WIDTH	"6"
#define DEF_PHOTO_CURSOR	((char *) NULL)
#define DEF_PHOTO_DITHER_LEVEL	"-1"
#define DEF_PHOTO_GAMMA		"1"
#define DEF_PHOTO_GEOMETRY	""
#define DEF_PHOTO_IMAGE_GEOM	""
#define DEF_PHOTO_PADX		"0"
#define DEF_PHOTO_PADY		"0"
#define DEF_PHOTO_PALETTE	""
#define DEF_PHOTO_RELIEF	"sunken"
#define DEF_PHOTO_SCANSPEED	"10"

#define DEFAULT_WIDTH		256
#define DEFAULT_HEIGHT		256

/*
 * Information used for parsing configuration options for photo widget
 */
static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_BORDER, "-activebackground", "activeBackground", "Foreground",
	 DEF_PHOTO_ACTIVE_BG, Tk_Offset(Photo, activeBorder), 0},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	 DEF_PHOTO_BG, Tk_Offset(Photo, bgBorder), 0},
    {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *) NULL,
	 (char *) NULL, 0, 0},
    {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL,
	 (char *) NULL, 0, 0},
    {TK_CONFIG_COLOR, "-blank", "blank", "Blank",
	 DEF_PHOTO_BLANK, Tk_Offset(Photo, blankColor), 0},
    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	 DEF_PHOTO_BORDER_WIDTH, Tk_Offset(Photo, borderWidth), 0},
    {TK_CONFIG_ACTIVE_CURSOR, "-cursor", "cursor", "Cursor",
	 DEF_PHOTO_CURSOR, Tk_Offset(Photo, cursor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_INT, "-ditherlevel", "ditherLevel", "DitherLevel",
	 DEF_PHOTO_DITHER_LEVEL, Tk_Offset(Photo, ditherLevel), 0},
    {TK_CONFIG_DOUBLE, "-gamma", "gamma", "Gamma",
	 DEF_PHOTO_GAMMA, Tk_Offset(Photo, gamma), 0},
    {TK_CONFIG_UID, "-geometry", "geometry", "Geometry",
	 DEF_PHOTO_GEOMETRY, Tk_Offset(Photo, geometry), 0},
    {TK_CONFIG_UID, "-imagesize", "imageSize", "Geometry",
	 DEF_PHOTO_IMAGE_GEOM, Tk_Offset(Photo, imageGeom), 0},
    {TK_CONFIG_PIXELS, "-padx", "padX", "Pad",
	 DEF_PHOTO_PADX, Tk_Offset(Photo, padx), 0},
    {TK_CONFIG_PIXELS, "-pady", "padY", "Pad",
	 DEF_PHOTO_PADY, Tk_Offset(Photo, pady), 0},
    {TK_CONFIG_UID, "-palette", "palette", "Palette",
	 DEF_PHOTO_PALETTE, Tk_Offset(Photo, palette), 0},
    {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
	 DEF_PHOTO_RELIEF, Tk_Offset(Photo, relief), 0},
    {TK_CONFIG_INT, "-scanspeed", "scanSpeed", "ScanSpeed",
	 DEF_PHOTO_SCANSPEED, Tk_Offset(Photo, scanSpeed), 0},
    {TK_CONFIG_STRING, "-xscrollcommand", "xScrollCommand", "ScrollCommand",
	 (char *) NULL, Tk_Offset(Photo, xScrollCmd), 0},
    {TK_CONFIG_STRING, "-yscrollcommand", "yScrollCommand", "ScrollCommand",
	 (char *) NULL, Tk_Offset(Photo, yScrollCmd), 0},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	 (char *) NULL, 0, 0}
};

/*
 * Hash table used to provide access to photos from C code.
 */
static Tcl_HashTable photoHash;
static int hash_inited;		/* set when Tcl_InitHashTable done */

/*
 * Hash table used to hash from display:screen to ColorTable info.
 */
static Tcl_HashTable colorHash;
static int color_hash_inited;
#define N_COLOR_HASH	(sizeof(ColorTableId) / sizeof(int))

/*
 * Forward declarations
 */
static Photo *MakePhotoWindow();
static int  PhotoWidgetCmd();
static int  ConfigurePhoto();
static int  ComputePhotoGeometry();
static void PhotoReallocatePixmap();
static void DestroyPhoto();
static int  GetColorInfo();
static int  PhotoSetPalette();
static void PhotoGetColorTable();
static void PhotoInstallCmap();
static void PhotoAllocColors();
static int  PhotoReclaimColors();
static void PhotoDoScroll();
static void PhotoDoScrollCmds();
static void RedrawImage();
static void RedrawPhoto();
static void RedrawPhotoFrame();
static void DisplayPhoto();
static void PhotoEventProc();
static void PhotoAllocImage();
static void PhotoDither();

/*
 *--------------------------------------------------------------
 *
 * Tk_PhotoCmd --
 *
 *	This procedure is invoked to process the "photo" Tcl
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 */
int
Tk_PhotoCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* main window of interpreter */
    Tcl_Interp *interp;		/* current interpreter */
    int argc;			/* number of arguments */
    char **argv;		/* arguments */
{
    Photo *php;

    if( argc < 2 ){
	Tcl_AppendResult(interp, "usage: ", argv[0], " pathName ?options?",
			 (char *) NULL);
	return TCL_ERROR;
    }

    php = MakePhotoWindow(interp, (Tk_Window) clientData, argv[1]);
    if( php == NULL )
	return TCL_ERROR;

    if( ConfigurePhoto(interp, php, argc-2, argv+2, 0) != TCL_OK ){
	Tk_DestroyWindow(php->tkwin);
	return TCL_ERROR;
    }

    interp->result = Tk_PathName(php->tkwin);
    return TCL_OK;
}

/*
 * Make a new photo window.
 */
static Photo *
MakePhotoWindow(interp, tkwin, path)
    Tcl_Interp *interp;		/* for error reporting */
    Tk_Window tkwin;		/* interpret path wrt this */
    char *path;			/* name for new window */
{
    register Photo *php;
    Tk_Window new;
    int i;
    Tcl_HashEntry *entry;
    int hnew;
    Tk_Window top;
    union {			/* for determining big/little endianness */
	int	i;
	char	c[sizeof(int)];
    } kludge;

    /* Create the new window */
    new = Tk_CreateWindowFromPath(interp, tkwin, path, (char *) NULL);
    if( new == NULL )
	return NULL;
    Tk_SetClass(new, "Photo");

    /* Allocate a data structure and set default values */
    php = (Photo *) ckalloc(sizeof(Photo));
    bzero(php, sizeof(Photo));
    php->interp = interp;
    php->tkwin = new;
    php->display = Tk_Display(new);
    php->palette = NULL;
    php->gamma = 1.0;
    php->colors = NULL;
    php->monoGC = None;
    php->cursor = None;
    php->reqWidth = DEFAULT_WIDTH;
    php->reqHeight = DEFAULT_HEIGHT;
    php->bgBorder = NULL;
    php->activeBorder = NULL;
    php->blankColor = NULL;
    php->blankGC = None;
    php->xScrollCmd = NULL;
    php->yScrollCmd = NULL;
    php->pixels = None;
    php->pix24 = NULL;
    php->error = NULL;
    php->geometry = Tk_GetUid(DEF_PHOTO_GEOMETRY);
    php->imageGeom = Tk_GetUid(DEF_PHOTO_IMAGE_GEOM);

    /* get visual info */
    if( GetColorInfo(interp, php) != TCL_OK ){
	/* should never happen */
	Tk_DestroyWindow(new);
	ckfree((char *) php);
	return NULL;
    }

    /* create an image structure for sending image data to the X server */
    php->image = XCreateImage(php->display, php->visual, php->depth,
			      php->depth > 1? ZPixmap: XYBitmap,
			      0, (char *) NULL, 1, 1, 32, 0);
    if( php->image == NULL ){
	Tcl_AppendResult(interp, "allocation failure in XCreateImage",
			 (char *) NULL);
	Tk_DestroyWindow(new);
	ckfree((char *) php);
	return NULL;
    }

    /* work out depth and endianness of image */
    php->image->bits_per_pixel = php->depth == 1? 1: php->depth <= 8? 8: 32;
    kludge.i = 0;
    kludge.c[0] = 1;
    php->image->byte_order = (kludge.i == 1)? LSBFirst: MSBFirst;
    php->image->bitmap_unit = sizeof(unsigned long) * NBBY;
    _XInitImageFuncPtrs(php->image);

    Tk_CreateEventHandler(new, ExposureMask | StructureNotifyMask
			       | EnterWindowMask | LeaveWindowMask,
			  PhotoEventProc, (ClientData) php);
    Tcl_CreateCommand(interp, Tk_PathName(new), PhotoWidgetCmd,
		      (ClientData) php, (void (*)()) NULL);

    /* find our top-level window ancestor */
    top = new;
    do {
	top = Tk_Parent(top);
    } while( !Tk_IsTopLevel(top) );
    php->main = top;

    /* enter this one in the hash table */
    if( !hash_inited ){
	Tcl_InitHashTable(&photoHash, TCL_STRING_KEYS);
	hash_inited = 1;
    }
    entry = Tcl_CreateHashEntry(&photoHash, path, &hnew);
    Tcl_SetHashValue(entry, php);

    return php;
}

/*
 * This procedure handles the Tcl command corresponding to a photo widget.
 */
static int
PhotoWidgetCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    register Photo *php;
    char *option;
    int c, length, result, i;
    int x, y, w, h, dw, dh;
    XColor color;
    int lac;
    char **lav;
    unsigned char *imp;
    Colormap cmap;
    char string[24];
    XWindowAttributes attrs;
    PhotoImage block;

    if( argc < 2 ){
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " option ?arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    php = (Photo *) clientData;
    Tk_Preserve((ClientData) php);

    option = argv[1];
    c = option[0];
    length = strlen(option);
    result = TCL_OK;

    if( c == 'b' && strncmp(option, "blank", length) == 0 ){
	/* blank the image */
	if( argc == 2 )
	    PhotoBlank(php);
	else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
			     argv[0], " blank\"", (char *) NULL);
	    result = TCL_ERROR;
	}

    } else if( c == 'c' && strncmp(option, "configure", length) == 0 ){
	/* change or report the photo window configuration */
	if( argc == 2 )
	    result = Tk_ConfigureInfo(interp, php->tkwin, configSpecs,
				      (char *) php, (char *) NULL, 0);
	else if( argc == 3 )
	    result = Tk_ConfigureInfo(interp, php->tkwin, configSpecs,
				      (char *) php, argv[2], 0);
	else
	    result = ConfigurePhoto(interp, php, argc-2, argv+2,
					     TK_CONFIG_ARGV_ONLY);

    } else if( c == 'd' && length >= 2
	      && strncmp(option, "dither", length) == 0 ){
	/* dither a portion of the image */
	if( argc == 2 || argc == 6 ){
	    if( argc == 2 ){
		x = y = 0;
		w = php->pixelWidth;
		h = php->pixelHeight;
	    } else {
		if( Tcl_GetInt(interp, argv[2], &x) != TCL_OK
		   || Tcl_GetInt(interp, argv[3], &y) != TCL_OK
		   || Tcl_GetInt(interp, argv[4], &w) != TCL_OK
		   || Tcl_GetInt(interp, argv[5], &h) != TCL_OK )
		    result = TCL_ERROR;
		else if( x < 0 || y < 0 || w <= 0 || h <= 0
			|| x + w > php->pixelWidth
			|| y + h > php->pixelHeight ){
		    Tcl_AppendResult(interp, argv[0], " dither: coordinates",
				     " not contained in image", (char *) NULL);
		    result = TCL_ERROR;
		}
	    }
	    if( result == TCL_OK ){
		if( php->ditherLevel == DITHER_NONE ){
		    Tcl_AppendResult(interp, argv[0], ": ",
				     "Dithering not available", (char *) NULL);
		    result = TCL_ERROR;
		} else
		    PhotoDither(php, x, y, w, h);
	    }
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " dither ?x y width height?", (char *) NULL);
	    result = TCL_ERROR;
	}

    } else if( c == 'p' && strncmp(option, "put", length) == 0 ){
	/* put pixels into the image */
	if( argc == 9 ){
	    if( Tcl_GetInt(interp, argv[2], &x) != TCL_OK
	       || Tcl_GetInt(interp, argv[3], &y) != TCL_OK
	       || Tcl_GetInt(interp, argv[4], &w) != TCL_OK
	       || Tcl_GetInt(interp, argv[5], &h) != TCL_OK
	       || Tcl_GetInt(interp, argv[6], &dw) != TCL_OK
	       || Tcl_GetInt(interp, argv[7], &dh) != TCL_OK
	       || Tcl_SplitList(interp, argv[8], &lac, &lav) != TCL_OK )
		result = TCL_ERROR;
	    else if( lac != dw * dh ){
		Tcl_AppendResult(interp, "color list has wrong # elements: ",
				 argv[6], "*", argv[7], " elements required",
				 (char *) NULL);
		free((char *) lav);
		result = TCL_ERROR;
	    } else {

		block.ptr = imp = (unsigned char *) ckalloc(lac * 3);
		cmap = DefaultColormap(php->display,
				       Tk_ScreenNumber(php->tkwin));
		for( i = 0; i < lac; ++i ){
		    if( !XParseColor(php->display, cmap, lav[i], &color) ){
			Tcl_AppendResult(interp, "can't parse color \"",
					 lav[i], "\"", (char *) NULL);
			result = TCL_ERROR;
			break;
		    }
		    *imp++ = color.red >> 8;
		    *imp++ = color.green >> 8;
		    *imp++ = color.blue >> 8;
		}

		if( result == TCL_OK ){
		    block.width = dw;
		    block.height = dh;
		    block.pitch = dw * 3;
		    block.pixel_size = 3;
		    block.comp_off[0] = 0;
		    block.comp_off[1] = 1;
		    block.comp_off[2] = 2;
		    PhotoPutBlock((ClientData)php, &block, x, y, w, h);
		}

		ckfree((char *) block.ptr);
		free((char *) lav);
	    }

	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " put x y w h data-width data-height {colors}",
			     (char *) NULL);
	    result = TCL_ERROR;
	}

    } else if( c == 'r' && strncmp(option, "redither", length) == 0 ){
	/* redither as much of the whole image as necessary */
	if( argc == 2 ){
	    if( php->ditherLevel > DITHER_NONE ){
		if( php->ditherX != 0 )
		    PhotoDither(php, php->ditherX, php->ditherY,
				php->pixelWidth - php->ditherX, 1);
		if( php->ditherY < php->pixelHeight )
		    PhotoDither(php, 0, php->ditherY, php->pixelWidth,
				php->pixelHeight - php->ditherY);
	    }
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " redither", (char *) NULL);
	    result = TCL_ERROR;
	}

    } else if( c == 's' && strncmp(option, "scan", length) == 0 ){
	/* move the image within the window */
	if( argc != 5 ){
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " scan mark|dragto x y\"", (char *) NULL);
	    result = TCL_ERROR;
	} else if( Tcl_GetInt(interp, argv[3], &x) != TCL_OK
		  || Tcl_GetInt(interp, argv[4], &y) != TCL_OK ){
	    result = TCL_ERROR;
	} else if( argv[2][0] == 'm'
		  && strncmp(argv[2], "mark", strlen(argv[2])) == 0 ){
	    php->scanX = x;
	    php->scanY = y;
	} else if( argv[2][0] == 'd'
		  && strncmp(argv[2], "dragto", strlen(argv[2])) == 0 ){
	    PhotoDoScroll(php, php->xshift - (x - php->scanX) * php->scanSpeed,
			  php->yshift - (y - php->scanY) * php->scanSpeed);
	    php->scanX = x;
	    php->scanY = y;
	} else {
	    Tcl_AppendResult(interp, "bad scan option \"", argv[2],
		    "\":  must be mark or dragto", (char *) NULL);
	    result = TCL_ERROR;
	}

    } else if( c == 'x' && strncmp(option, "xview", length) == 0 ){
	/* scroll to a given image X coord at left window edge */
	if( argc == 3 ){
	    if( Tcl_GetInt(interp, argv[2], &x) == TCL_OK ){
		php->xscr_pos = -1; /* make sure we do xscroll cmd */
		PhotoDoScroll(php, x, php->yshift);
	    } else
		result = TCL_ERROR;
	} else if( argc == 2 ){
	    sprintf(string, "%d", php->xshift);
	    Tcl_AppendResult(interp, string, (char *) NULL);
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " xscroll ?x-coord?\"", (char *) NULL);
	    result = TCL_ERROR;
	}

    } else if( c == 'y' && strncmp(option, "yview", length) == 0 ){
	/* scroll to a given image Y coord at left window edge */
	if( argc == 3 ){
	    if( Tcl_GetInt(interp, argv[2], &y) == TCL_OK ){
		php->yscr_pos = -1; /* make sure we do yscroll cmd */
		PhotoDoScroll(php, php->xshift, y);
	    } else
		result = TCL_ERROR;
	} else if( argc == 2 ){
	    sprintf(string, "%d", php->yshift);
	    Tcl_AppendResult(interp, string, (char *) NULL);
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " yscroll ?y-coord?\"", (char *) NULL);
	    result = TCL_ERROR;
	}

    } else {
	Tcl_AppendResult(interp, "bad option \"", option,
			 "\": must be blank, configure, dither, redither,",
			 " scan, xview or yview", (char *) NULL);
	result = TCL_ERROR;
    }

    Tk_Release((ClientData) php);
    return result;
}

/*
 * Configure a photo widget from information in an argv list,
 * plus the Tk option database (if TK_CONFIG_ARGV_ONLY is not set in flags).
 */
static int
ConfigurePhoto(interp, php, argc, argv, flags)
    Tcl_Interp *interp;
    register Photo *php;
    int argc;
    char **argv;
    int flags;
{
    int w, h, result;
    GC newGC;
    XColor *oldBlank;
    Tk_Uid oldGeom, oldImageGeom, oldPalette;
    double oldGamma;
    XGCValues gcValues;
    char geomString[64];

    oldBlank = php->blankColor;
    oldGeom = php->geometry;
    oldImageGeom = php->imageGeom;
    oldPalette = php->palette;
    oldGamma = php->gamma;

    if( Tk_ConfigureWidget(interp, php->tkwin, configSpecs, argc, argv,
			   (char *) php, flags) != TCL_OK )
	return TCL_ERROR;

    if( php->blankGC == None || php->blankColor != oldBlank ){
	gcValues.foreground = php->blankColor->pixel;
	newGC = Tk_GetGC(php->tkwin, GCForeground, &gcValues);
	if( php->blankGC != None )
	    Tk_FreeGC(php->display, php->blankGC);
	php->blankGC = newGC;
	Tk_SetWindowBackground(php->tkwin, php->blankColor->pixel);
    }

    result = TCL_OK;
    if( php->geometry[0] == 0 ){
	php->flags &= ~USER_GEOMETRY;
	if( (php->flags & USER_IMAGE_GEOM) != 0 ){
	    php->reqWidth = php->imageWidth;
	    php->reqHeight = php->imageHeight;
	} else if( php->pixels != None ){
	    php->reqWidth = php->pixelWidth;
	    php->reqHeight = php->pixelHeight;
	}
    } else if( php->geometry != oldGeom ){
	if( sscanf(php->geometry, "%dx%d", &w, &h) == 2 ){
	    if( w > 0 && h > 0 ){
		php->reqWidth = w;
		php->reqHeight = h;
		php->flags |= USER_GEOMETRY;
	    } else {
		Tcl_AppendResult(interp, "bad geometry \"", php->geometry,
				 "\": width and height must be positive",
				 (char *) NULL);
		result = TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, "bad geometry \"", php->geometry,
			     "\": expected widthxheight", (char *) NULL);
	    result = TCL_ERROR;
	}
    }

    if( php->imageGeom[0] == 0 )
	php->flags &= ~USER_IMAGE_GEOM;
    else if( php->imageGeom != oldImageGeom ){
	if( sscanf(php->imageGeom, "%dx%d", &w, &h) == 2 ){
	    if( w > 0 && h > 0 ){
		php->imageWidth = w;
		php->imageHeight = h;
		php->flags |= USER_IMAGE_GEOM;
		if( (php->flags & USER_GEOMETRY) == 0 ){
		    php->reqWidth = w;
		    php->reqHeight = h;
		}
	    } else {
		Tcl_AppendResult(interp, "bad image geometry \"",
				 php->imageGeom,
				 "\": width and height must be positive",
				 (char *) NULL);
		result = TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, "bad image geometry \"", php->imageGeom,
			     "\": expected widthxheight", (char *) NULL);
	    result = TCL_ERROR;
	}
    }

    if( php->gamma <= 0 ){
	Tcl_AppendResult(interp, "gamma value must be positive",
			 (char *) NULL);
	php->gamma = oldGamma;
	result = TCL_ERROR;
    }

    if( oldPalette == NULL || php->palette != oldPalette
       || php->gamma != oldGamma )
	if( PhotoSetPalette(php) != TCL_OK )
	    result = TCL_ERROR;

    if( php->ditherLevel < DITHER_NONE || php->ditherLevel > DITHER_BLOCKS )
	if( (php->flags & MAP_COLORS) != 0 || (php->flags & MONOCHROME) != 0 )
	    php->ditherLevel = DITHER_BLOCKS;
	else
	    php->ditherLevel = DITHER_NONE;

    ComputePhotoGeometry(php);
    RedrawPhotoFrame(php);

    return TCL_OK;
}

/*
 * Compute & request dimensions of photo window
 */
static int
ComputePhotoGeometry(php)
Photo *php;
{
    int b, w, h;

    Tk_GeometryRequest(php->tkwin,
		       php->reqWidth + 2 * php->borderWidth + php->padx,
		       php->reqHeight + 2 * php->borderWidth + php->pady);
    Tk_SetInternalBorder(php->tkwin, php->borderWidth);
    php->width = Tk_Width(php->tkwin) - 2 * php->borderWidth;
    if( php->width > php->reqWidth )
	php->width = php->reqWidth;
    php->height = Tk_Height(php->tkwin) - 2 * php->borderWidth;
    if( php->height > php->reqHeight )
	php->height = php->reqHeight;
    php->xorg = (Tk_Width(php->tkwin) - php->width) / 2;
    php->yorg = (Tk_Height(php->tkwin) - php->height) / 2;

    /* compute necessary size of pixmap */
    w = MAX(php->reqWidth, php->imageWidth);
    h = MAX(php->reqHeight, php->imageHeight);
    if( (php->flags & USER_IMAGE_GEOM) == 0 ){
	if( w < php->pixelWidth )
	    w = php->pixelWidth;
	if( h < php->pixelHeight )
	    h = php->pixelHeight;
    }
    if( php->pixels == None || w != php->pixelWidth || h != php->pixelHeight
       || php->ditherLevel > DITHER_NONE && php->pix24 == NULL
       || php->ditherLevel == DITHER_NONE && php->pix24 != NULL )
	/* reallocate the pixmap and dithering arrays */
	PhotoReallocatePixmap(php, w, h);

    /* adjust xshift and yshift if necessary */
    PhotoDoScroll(php, php->xshift, php->yshift);

    PhotoDoScrollCmds(php);
}

/*
 * Set the pixmap associated with `php' to be w pixels wide
 * and h pixels high.  Also reallocates the `pix24' and `error' storage
 * if necessary.
 */
static void
PhotoReallocatePixmap(php, w, h)
    Photo *php;
    int w, h;
{
    Window root;
    int clr_x, clr_y, n;
    register int i0, i1, j, w3, x3, pw3;
    Pixmap newPixmap;
    ColorTable *ctp;
    unsigned char *newPix24;
    char *newError;

    if( w != php->pixelWidth || h != php->pixelHeight ){
	/* Create new pixmap using the root window,
	   since Tk windows might not yet exist, as far as X is concerned. */
	ctp = php->colors;
	root = RootWindow(php->display, Tk_ScreenNumber(php->tkwin));
	newPixmap = XCreatePixmap(php->display, root, w, h,
				  php->vis_info.depth);
	
	if( php->pixels != None ){
	    /* copy bits from the old pixmap and free it */
	    clr_x = MIN(w, php->pixelWidth);
	    clr_y = MIN(h, php->pixelHeight);
	    XCopyArea(php->display, php->pixels, newPixmap, php->blankGC,
		      0, 0, clr_x, clr_y, 0, 0);
	    XFreePixmap(php->display, php->pixels);
	} else
	    clr_x = clr_y = 0;
	
	/* clear the new areas of the new pixmap */
	if( clr_x < w )
	    XFillRectangle(php->display, newPixmap, php->blankGC,
			   clr_x, 0, w - clr_x, h);
	if( clr_y < h && clr_x > 0 )
	    XFillRectangle(php->display, newPixmap, php->blankGC,
			   0, clr_y, clr_x, h - clr_y);

	php->pixels = newPixmap;

    } else {
	/* Not changing size; must need to allocate/deallocate dither stuff */
	clr_x = w;
	clr_y = h;
    }

    if( php->ditherLevel > DITHER_NONE ){
	/* reallocate pix24 and error as well */
	if( (php->flags & MONOCHROME) == 0 ){
	    w3 = w * 3;
	    x3 = clr_x * 3;
	    pw3 = php->pixelWidth * 3;
	} else {
	    /* store 1 byte per pixel for monochrome */
	    w3 = w;
	    x3 = clr_x;
	    pw3 = php->pixelWidth;
	}
	n = w3 * h;
	newPix24 = (unsigned char *) ckalloc(n);
	newError = (char *) ckalloc(n);
	/* copy stuff from old to new */
	if( php->pix24 != NULL ){
	    i0 = i1 = 0;
	    for( j = 0; j < clr_y; ++j ){
		bcopy(php->pix24 + i0, newPix24 + i1, x3);
		bcopy(php->error + i0, newError + i1, x3);
		i0 += pw3;
		i1 += w3;
	    }
	    ckfree((char *) php->pix24);
	    ckfree((char *) php->error);
	} else
	    clr_x = clr_y = 0;
	/* clear out new areas */
	if( clr_x == 0 ){
	    bzero(newPix24, n);
	    bzero(newError, n);
	} else {
	    if( clr_x < w ){
		i1 = x3;
		n = w3 - x3;
		for( j = 0; j < clr_y; ++j ){
		    bzero(newPix24 + i1, n);
		    bzero(newError + i1, n);
		    i1 += w3;
		}
	    }
	    if( clr_y < h ){
		n = (h - clr_y) * w3;
		bzero(newPix24 + clr_y * w3, n);
		bzero(newError + clr_y * w3, n);
	    }
	}
	php->pix24 = newPix24;
	php->error = newError;
	/* dithering is correct up to the end of the last pre-existing
	   complete scanline. */
	if( clr_x == w ){
	    if( clr_y <= php->ditherY ){
		php->ditherX = 0;
		php->ditherY = clr_y;
	    }
	} else {
	    php->ditherX = clr_x;
	    php->ditherY = 0;
	}

    } else if( php->pix24 != NULL ){
	ckfree((char *) php->pix24);
	ckfree((char *) php->error);
	php->pix24 = NULL;
	php->error = NULL;
    }

    php->pixelWidth = w;
    php->pixelHeight = h;

}

/*
 * Release all resources associated with a photo widget.
 */
static void
DestroyPhoto(clientData)
    ClientData clientData;
{
    Photo *php;

    php = (Photo *) clientData;
    if( php->bgBorder != NULL )
	Tk_Free3DBorder(php->bgBorder);
    if( php->activeBorder != NULL )
	Tk_Free3DBorder(php->activeBorder);
    if( php->blankColor != NULL )
	Tk_FreeColor(php->blankColor);
    if( php->blankGC != None )
	Tk_FreeGC(php->display, php->blankGC);
    if( php->cursor != None )
	Tk_FreeCursor(php->display, php->cursor);
    if( php->xScrollCmd != NULL )
	free(php->xScrollCmd);
    if( php->yScrollCmd != NULL )
	free(php->yScrollCmd);
    if( php->pixels != None )
	XFreePixmap(php->display, php->pixels);
    if( php->pix24 != NULL ){
	ckfree((char *) php->pix24);
	ckfree((char *) php->error);
    }
    if( php->image != NULL ){
	if( php->image->data != NULL )
	    ckfree((char *) php->image->data);
	XFree((char *) php->image);
    }
    ckfree((char *) php);
}

/*
 * Get information about the visual we'll be using.
 */
static int
GetColorInfo(interp, php)
    Tcl_Interp *interp;		/* for error reporting */
    Photo *php;
{
    int i, n_visi, screen;
    XVisualInfo *visi_list, vis_info;
    XWindowAttributes attrs;
    XGCValues gcValues;

    /*
     * When Tk creates a window, it uses CopyFromParent for the depth
     * and visual attributes.  So we need to query the parent's values
     * for depth and visual to see what we'll get.  In fact the top-level
     * parent will inherit them from the root, so we query the root's
     * values, since none of the Tk windows may exist yet as far as the
     * X server is concerned.
     */
    screen = Tk_ScreenNumber(php->tkwin);
    XGetWindowAttributes(php->display,
			 RootWindow(php->display, screen), &attrs);

    /* Get visual info for visual---have to do it the hard way */
    vis_info.screen = screen;
    visi_list = XGetVisualInfo(php->display, VisualScreenMask,
			       &vis_info, &n_visi);
    for( i = 0; i < n_visi; ++i )
	if( visi_list[i].visual == attrs.visual ){
	    vis_info = visi_list[i];
	    break;
	}
    XFree((char *) visi_list);
    if( i >= n_visi ){
	/* should never happen */
	Tcl_AppendResult(interp, "BUG: can't get visual information",
			 (char *) NULL);
	return TCL_ERROR;
    }

    switch( vis_info.class ){
    case PseudoColor:
    case StaticColor:
	php->flags |= MAP_COLORS;
	break;
    case GrayScale:
    case StaticGray:
	php->flags |= MONOCHROME;
	break;
    }
    php->depth = vis_info.depth;
    php->visual = attrs.visual;
    php->vis_info = vis_info;

    /* Make a GC with background = black and foreground = white */
    gcValues.foreground = WhitePixel(php->display, screen);
    gcValues.background = BlackPixel(php->display, screen);
    php->monoGC = Tk_GetGC(php->tkwin, GCForeground|GCBackground, &gcValues);

    return TCL_OK;
}

/*
 * The palette specification for a window has been changed.
 * Parse it, check that we have enough colors, and request that the color
 * allocation be done later when we're idle.
 */
static int
PhotoSetPalette(php)
    Photo *php;
{
    int res, mono, f;
    char *p;
    int n, nred, ngreen, nblue;
    double g;
    char spec[32];

    if( php->palette[0] != 0 ){
	/* user has given a specification; parse and check it. */
	res = TCL_ERROR;
	p = php->palette;
	if( sscanf(p, "%d%n", &nred, &n) == 1 ){
	    /* got a number so far; any more following? */
	    p += n;
	    if( p[0] == 0 ){
		/* single number => monochrome */
		mono = TRUE;
		ngreen = nblue = 1;
		res = TCL_OK;
	    } else if( sscanf(p, "/%d/%d%n", &ngreen, &nblue, &n) == 2
		      && p[n] == 0 ){
		mono = FALSE;
		res = TCL_OK;
	    }
	}
	if( res != TCL_OK )
	    Tcl_AppendResult(php->interp, "can't parse palette specification",
			     (char *) NULL);

	if( res == TCL_OK ){
	    /* check validity of numbers */
	    if( nred > 256 )	/* only have 8 bits to select shades with */
		nred = 256;
	    if( ngreen > 256 )
		ngreen = 256;
	    if( nblue > 256 )
		nblue = 256;
	    if( nred < 2 || !mono && (ngreen < 2 || nblue < 2) ){
		Tcl_AppendResult(php->interp,
				 "each number in palette must be >= 2",
				 (char *) NULL);
		res = TCL_ERROR;
	    }
	}

	if( res == TCL_OK ){
	    /* check that we have enough colors */
	    switch( php->vis_info.class ){
	    case DirectColor:
	    case TrueColor:
		n = 1 << php->vis_info.bits_per_rgb;
		if( nred > n || ngreen > n || nblue > n )
		    res = TCL_ERROR;
		break;
	    case PseudoColor:
	    case StaticColor:
		if( nred * ngreen * nblue > 1 << php->vis_info.depth )
		    res = TCL_ERROR;
		break;
	    case GrayScale:
	    case StaticGray:
		if( !mono || nred > 1 << php->vis_info.depth )
		    res = TCL_ERROR;
		break;
	    }
	    if( res != TCL_OK )
		Tcl_AppendResult(php->interp, "palette requires more colors ",
				 "than display provides", (char *) NULL);
	}
    } else
	res = TCL_OK;

    if( php->palette[0] == 0 || res != TCL_OK ){
	/* no palette or invalid spec given; choose default. */
	nred = 2;		/* worst-case default */
	mono = TRUE;
	switch( php->vis_info.class ){
	case DirectColor:
	case TrueColor:
	    nred = ngreen = nblue = 1 << php->vis_info.bits_per_rgb;
	    mono = FALSE;
	    break;
	case PseudoColor:
	case StaticColor:
	    if( php->vis_info.depth >= 8 ){
		nred = ngreen = 7;
		nblue = 4;
		mono = FALSE;
	    } else if( php->vis_info.depth >= 3 ){
		nred = ngreen = nblue = 2;
		mono = FALSE;
	    }
	    break;
	case GrayScale:
	case StaticGray:
	    nred = 1 << php->vis_info.depth;
	    break;
	}
    }

    /* Convert spec back to canonical form */
    sprintf(spec, mono? "%d": "%d/%d/%d", nred, ngreen, nblue);
    php->palette = Tk_GetUid(spec);

    /* If we are changing to/from monochrome, we need to reallocate the
       pix24 and error arrays, because they've changed size. */
    f = php->flags;
    if( mono )
	php->flags |= MONOCHROME;
    else
	php->flags &= ~MONOCHROME;
    if( f != php->flags && php->pix24 != NULL ){
	ckfree((char *) php->pix24);
	ckfree((char *) php->error);
	php->pix24 = NULL;
	php->error = NULL;
	/* A subsequent call to ComputePhotoGeometry will alloc new arrays. */
	/* Should we blank the image here? or whenever changing palettes? */
    }

    if( mono && nred == 2 ){
	php->depth = 1;
	php->colors = NULL;
	PhotoInstallCmap(php);
    } else {
	php->depth = php->vis_info.depth;
	/* Get a ColorTable for this display screen and palette */
	PhotoGetColorTable(php);
    }

    return res;
}

/*
 * This code maintains a table of colormap information for each display,
 * screen and palette used by the Tk application, so that the application
 * need only request a set of colors from the X server once for each palette
 * on each display screen.  We assume that the palette spec is valid.
 * The colors are actually allocated later in a DoWhenIdle handler.
 */
static void
PhotoGetColorTable(php)
    Photo *php;
{
    ColorTable *ctp;
    Tcl_HashEntry *entry;
    ColorTableId id;
    int new;

    id.display = php->display;
    id.screen = Tk_ScreenNumber(php->tkwin);
    id.palette = php->palette;
    id.gamma = php->gamma;
    if( !color_hash_inited ){
	Tcl_InitHashTable(&colorHash, N_COLOR_HASH);
	color_hash_inited = TRUE;
    }
    entry = Tcl_CreateHashEntry(&colorHash, (char *) &id, &new);

    if( !new ){
	ctp = (ColorTable *) Tcl_GetHashValue(entry);
	if( ctp == php->colors )
	    return;
	++ctp->refcnt;

    } else {
	/* No color table currently available; need to make one. */
	ctp = (ColorTable *) ckalloc(sizeof(ColorTable));
	ctp->id = id;
	ctp->flags = NEED_ALLOC;
	ctp->refcnt = 1;
	ctp->ncolors = 0;
	ctp->pixel_map = NULL;
	Tcl_SetHashValue(entry, ctp);
    }

    if( php->colors != NULL )
	--php->colors->refcnt;
    php->colors = ctp;

    Tk_DoWhenIdle(PhotoInstallCmap, (ClientData) php);
}

/*
 * If the cursor is in php's window, update the colormap as required.
 */
static void
PhotoInstallCmap(php)
    Photo *php;
{
    ColorTable *ctp;
    Colormap cmap;
    XWindowAttributes attrs;

    ctp = php->colors;
    php->flags &= ~SET_COLORMAP;
    cmap = DefaultColormap(php->display, Tk_ScreenNumber(php->tkwin));

    if( ctp != NULL ){
	if( (ctp->flags & NEED_ALLOC) != 0 )
	    PhotoAllocColors(php);
	if( (ctp->flags & OWN_CMAP) != 0 )
	    php->flags |= SET_COLORMAP;
	cmap = ctp->cmap;
    }

    if( (php->flags & ACTIVE) != 0 ){
	if( (php->flags & SET_COLORMAP) != 0 ){
	    /* install our colormap on the top-level window */
	    if( (php->flags & RESTORE_CMAP) == 0 ){
		Tk_MakeWindowExist(php->main);
		XGetWindowAttributes(php->display, Tk_WindowId(php->main),
				     &attrs);
		php->mainCmap = attrs.colormap;
		php->flags |= RESTORE_CMAP;
	    }
	    XSetWindowColormap(php->display, Tk_WindowId(php->main), cmap);

	} else if( (php->flags & RESTORE_CMAP) != 0 ){
	    /* restore the default colormap on the top-level window */
	    XSetWindowColormap(php->display, Tk_WindowId(php->main),
			       php->mainCmap);
	    php->flags &= ~RESTORE_CMAP;
	}
    }

    Tk_MakeWindowExist(php->tkwin);
    XSetWindowColormap(php->display, Tk_WindowId(php->tkwin), cmap);
}

/*
 * Allocate colors for the ColorTable for a photo.
 */
#define CFRAC(i, n)	((i) * 65535 / (n))
#define CGFRAC(i, n, g)	((int)(65535 * pow((double)(i) / (n), (g))))

static void
PhotoAllocColors(clientData)
    ClientData clientData;
{
    Photo *php;
    ColorTable *ctp;
    int i, r, g, b, rmult, mono;
    int ncol, nred, ngreen, nblue, nctot;
    double fr, fg, fb, igam;
    XColor *colors;
    Colormap cmap, dflt;
    Display *disp;
    Window root;

    php = (Photo *) clientData;
    ctp = php->colors;
    if( (ctp->flags & NEED_ALLOC) == 0 )
	return;
    ctp->flags &= ~NEED_ALLOC;

    disp = ctp->id.display;
    cmap = dflt = DefaultColormap(disp, ctp->id.screen);

    mono = sscanf(ctp->id.palette, "%d/%d/%d", &nred, &ngreen, &nblue) <= 1;
    igam = 1.0 / ctp->id.gamma;

    if( php->vis_info.class == DirectColor
       || php->vis_info.class == TrueColor ){
	nctot = 1 << php->vis_info.bits_per_rgb;
	colors = (XColor *) ckalloc(nctot * sizeof(XColor));
	if( mono )
	    ncol = ngreen = nblue = nred;
	else
	    ncol = MAX(MAX(nred, ngreen), nblue);

	for( i = 0; i < ncol; ++i ){
	    if( igam == 1.0 ){
		colors[i].red = CFRAC(i, nred - 1);
		colors[i].green = CFRAC(i, ngreen - 1);
		colors[i].blue = CFRAC(i, nblue - 1);
	    } else {
		colors[i].red = CGFRAC(i, nred - 1, igam);
		colors[i].green = CGFRAC(i, ngreen - 1, igam);
		colors[i].blue = CGFRAC(i, nblue - 1, igam);
	    }
	}

    } else {
	/* PseudoColor, StaticColor, GrayScale or StaticGray */
	nctot = 1 << php->vis_info.depth;
	colors = (XColor *) ckalloc(nctot * sizeof(XColor));
	rmult = mono? 1: ngreen * nblue;
	ncol = nred * rmult;

	if( !mono ){
	    i = 0;
	    for( r = 0; r < nred; ++r )
		for( g = 0; g < ngreen; ++g )
		    for( b = 0; b < nblue; ++b ){
			if( igam == 1.0 ){
			    colors[i].red = CFRAC(r, nred - 1);
			    colors[i].green = CFRAC(g, ngreen - 1);
			    colors[i].blue = CFRAC(b, nblue - 1);
			} else {
			    colors[i].red = CGFRAC(r, nred - 1, igam);
			    colors[i].green = CGFRAC(g, ngreen - 1, igam);
			    colors[i].blue = CGFRAC(b, nblue - 1, igam);
			}
			++i;
		    }
	} else {
	    /* monochrome case */
	    for( i = 0; i < ncol; ++i ){
		if( igam == 1.0 )
		    r = CFRAC(i, ncol - 1);
		else
		    r = CGFRAC(i, ncol - 1, igam);
		colors[i].red = colors[i].green = colors[i].blue = r;
	    }
	}
    }

    /* now allocate the colors we've calculated */
    ctp->pixel_map = (unsigned long *) ckalloc(ncol * sizeof(unsigned long));
    for( i = 0; i < ncol; ++i ){
	if( !XAllocColor(disp, cmap, &colors[i]) ){
	    /* can't get all the colors we want in the default colormap;
	       first try freeing colors from other unused palettes */
	    if( !PhotoReclaimColors(&ctp->id, ncol - i)
	       || !XAllocColor(disp, cmap, &colors[i]) ){
		/* still can't allocate it; we'll have to use our own cmap */
		if( i > 0 )
		    XFreeColors(disp, cmap, ctp->pixel_map, i, 0);
		break;
	    }
	}
	ctp->pixel_map[i] = colors[i].pixel;
    }
    
    if( i < ncol ){
	/* allocate a private colormap */
	root = RootWindow(disp, ctp->id.screen);
	/* We use the root window because maybe the widget's window
	   doesn't exist yet. */
	cmap = XCreateColormap(disp, root, php->visual, AllocAll);
	ctp->flags |= OWN_CMAP;
	for( i = 0; i < ncol; ++i ){
	    colors[i].flags = DoRed | DoGreen | DoBlue;
	    colors[i].pixel = ctp->pixel_map[i] = i + nctot - ncol;
	}
	for( ; i < nctot; ++i ){
	    colors[i].flags = DoRed | DoGreen | DoBlue;
	    colors[i].pixel = i - ncol;
	}		
	XQueryColors(disp, dflt, &colors[ncol], nctot - ncol);
	XStoreColors(disp, cmap, colors, nctot);
    }

    ctp->ncolors = ncol;
    ctp->cmap = cmap;

    /* Set up quantization tables */
    for( i = 0; i < 256; ++i ){
	r = (i * (nred - 1) + 127) / 255;
	if( mono ){
	    fr = colors[r].red / 65535.0;
	    if( php->gamma != 1.0 )
		fr = pow(fr, php->gamma);
	    ctp->color_quant[0][i] = (int)(fr * 255.99);
	    ctp->red_values[i] = colors[r].pixel;
	} else {
	    g = (i * (ngreen - 1) + 127) / 255;
	    b = (i * (nblue - 1) + 127) / 255;
	    if( php->vis_info.class == DirectColor
	       || php->vis_info.class == TrueColor ){
		ctp->red_values[i] = colors[r].pixel
				& php->vis_info.red_mask;
		ctp->green_values[i] = colors[g].pixel
				& php->vis_info.green_mask;
		ctp->blue_values[i] = colors[b].pixel
				& php->vis_info.blue_mask;
	    } else {
		r *= rmult;
		g *= nblue;
		ctp->red_values[i] = r;
		ctp->green_values[i] = g;
		ctp->blue_values[i] = b;
	    }
	    fr = colors[r].red / 65535.0;
	    fg = colors[g].green / 65535.0;
	    fb = colors[b].blue / 65535.0;
	    if( php->gamma != 1.0 ){
		fr = pow(fr, php->gamma);
		fg = pow(fg, php->gamma);
		fb = pow(fb, php->gamma);
	    }
	    ctp->color_quant[0][i] = (int)(fr * 255.99);
	    ctp->color_quant[1][i] = (int)(fg * 255.99);
	    ctp->color_quant[2][i] = (int)(fb * 255.99);
	}

    }

}

/*
 * If possible, reclaim colors in the default colormap for the given
 * display and screen.  Return TRUE iff some colors were freed.
 * If there's no way we could get ncol colors, don't bother trying.
 */
static int
PhotoReclaimColors(id, ncol)
    ColorTableId *id;
    int ncol;
{
    Tcl_HashSearch srch;
    Tcl_HashEntry *entry;
    ColorTable *ctp;
    int navail;

    /* first scan through to see how many we may possibly be able to get */
    navail = 0;
    entry = Tcl_FirstHashEntry(&colorHash, &srch);
    while( entry != NULL ){
	ctp = (ColorTable *) Tcl_GetHashValue(entry);
	if( ctp->id.display == id->display && ctp->id.screen == id->screen
	   && (ctp->id.palette != id->palette || ctp->id.gamma != id->gamma)
	   && ctp->refcnt == 0 && ctp->ncolors != 0
	   && (ctp->flags & OWN_CMAP) == 0 )
	    navail += ctp->ncolors;
	entry = Tcl_NextHashEntry(&srch);
    }

    /* navail is an overestimate of the number of colors we could get
       if we try. */
    if( navail < ncol )
	return FALSE;		/* don't disturb other maps unnecessarily */

    /* scan through a second time freeing colors */
    entry = Tcl_FirstHashEntry(&colorHash, &srch);
    while( entry != NULL && ncol > 0 ){
	ctp = (ColorTable *) Tcl_GetHashValue(entry);
	if( ctp->id.display == id->display && ctp->id.screen == id->screen
	   && (ctp->id.palette != id->palette || ctp->id.gamma != id->gamma)
	   && ctp->refcnt == 0 && ctp->ncolors != 0
	   && (ctp->flags & OWN_CMAP) == 0 ){
	    /* free the colors for this map */
	    XFreeColors(ctp->id.display, ctp->cmap, ctp->pixel_map,
			ctp->ncolors, 0);
	    ncol -= ctp->ncolors;
	    ctp->flags |= NEED_ALLOC;
	    ctp->ncolors = 0;
	    ckfree((char *) ctp->pixel_map);
	    ctp->pixel_map = NULL;
	}
	entry = Tcl_NextHashEntry(&srch);
    }
    return TRUE;		/* we freed some colors */
}

/*
 * Scroll so that (x, y) in the image is at the top-left corner
 * of the window.
 */
static void
PhotoDoScroll(php, x, y)
    Photo *php;
    int x, y;
{
    if( x < 0 )
	x = 0;
    else if( x > php->pixelWidth - php->width )
	x = php->pixelWidth - php->width;
    if( y < 0 )
	y = 0;
    else if( y > php->pixelHeight - php->height )
	y = php->pixelHeight - php->height;
    if( x != php->xshift || y != php->yshift ){
	php->xshift = x;
	php->yshift = y;
	if( (php->flags & REDRAW_PENDING) == 0 )
	    Tk_DoWhenIdle(DisplayPhoto, (ClientData) php);
	php->flags |= REDRAW_SCROLL;
    }
    PhotoDoScrollCmds(php);
}

/*
 * Execute the scrolling commands to update related scrollbars (if any),
 * but only if the parameters have changed.
 */
static void
PhotoDoScrollCmds(php)
    Photo *php;
{
    int res;
    char string[64];

    if( php->width <= 0 || php->height <= 0 )
	return;			/* window not set up yet */
    if( php->xScrollCmd != NULL
       && (php->xscr_pos != php->xshift || php->xscr_wid != php->width
	   || php->xscr_pwid != php->pixelWidth) ){
	sprintf(string, " %d %d %d %d", php->pixelWidth, php->width,
		php->xshift, php->xshift + php->width - 1);
	res = Tcl_VarEval(php->interp, php->xScrollCmd, string, (char *) NULL);
	if( res != TCL_OK )
	    Tk_BackgroundError(php->interp);
	php->xscr_pos = php->xshift;
	php->xscr_wid = php->width;
	php->xscr_pwid = php->pixelWidth;
    }
    if( php->yScrollCmd != NULL
       && (php->yscr_pos != php->yshift || php->yscr_ht != php->height
	   || php->yscr_pht != php->pixelHeight) ){
	sprintf(string, " %d %d %d %d", php->pixelHeight, php->height,
		php->yshift, php->yshift + php->height - 1);
	res = Tcl_VarEval(php->interp, php->yScrollCmd, string, (char *) NULL);
	if( res != TCL_OK )
	    Tk_BackgroundError(php->interp);
	php->yscr_pos = php->yshift;
	php->yscr_ht = php->height;
	php->yscr_pht = php->pixelHeight;
    }
}

/*
 * Redraw the window because some of the photo image has changed.
 */
static void
RedrawImage(php, x, y, w, h)
    Photo *php;
    int x, y, w, h;
{
    if( (x -= php->xshift) < 0 ){
	w += x;
	x = 0;
    }
    if( (y -= php->yshift) < 0 ){
	h += y;
	y = 0;
    }
    if( x + w > php->width )
	w = php->width - x;
    if( y + h > php->height )
	h = php->height - y;
    if( w <= 0 || h <= 0 )
	return;
    RedrawPhoto(php, x + php->xorg, y + php->yorg, w, h);
}

/*
 * Redraw the screen image for a block of the photo widget
 */
static void
RedrawPhoto(php, x, y, w, h)
    Photo *php;
    int x, y, w, h;
{
    int xi, yi, wi, hi;

    if( php->tkwin == NULL || Tk_WindowId(php->tkwin) == None
       || (php->flags & EXPOSED) == 0 )
	return;

    /* Redraw any necessary part of the image */
    xi = MAX(x - php->xorg, 0);
    yi = MAX(y - php->yorg, 0);
    wi = MIN(w, php->width - xi);
    hi = MIN(h, php->height - yi);
    if( wi > 0 && hi > 0 )
	XCopyArea(php->display, php->pixels, Tk_WindowId(php->tkwin),
		  php->blankGC, xi + php->xshift, yi + php->yshift, wi, hi,
		  xi + php->xorg, yi + php->yorg);

    if( x < php->xorg || y < php->yorg
       || x + w > php->xorg + php->width
       || y + h > php->yorg + php->height )
	/* need to redraw the frame around the photo - do it later */
	RedrawPhotoFrame(php);

}

/*
 * Request that the frame around a photo widget be redrawn when there's
 * nothing else to do.
 */
static void
RedrawPhotoFrame(php)
    Photo *php;
{
    if( (php->flags & REDRAW_PENDING) == 0 )
	Tk_DoWhenIdle(DisplayPhoto, (ClientData) php);
    php->flags |= REDRAW_FRAME;
}

/*
 * Draw the frame around a photo widget and/or scroll it to its present
 * position.
 */
static void
DisplayPhoto(clientData)
    ClientData clientData;
{
    Photo *php;
    Tk_Window win;
    int t, o, w, h, x1, y1;
    Tk_3DBorder border;

    php = (Photo *) clientData;
    win = php->tkwin;

    if( win == NULL || !Tk_IsMapped(win) ){
	php->flags &= ~REDRAW_PENDING;
	return;
    }

    if( (php->flags & REDRAW_FRAME) != 0 ){
	o = (php->relief == TK_RELIEF_FLAT)? 0: php->borderWidth;
	w = Tk_Width(win);
	h = Tk_Height(win);
	border = (php->flags & ACTIVE)? php->activeBorder: php->bgBorder;
	
	if( o > 0 )
	    Tk_Draw3DRectangle(php->display, Tk_WindowId(win),
			       border, 0, 0, w, h,
			       php->borderWidth, php->relief);
	
	if( (t = php->yorg - o) > 0 )
	    Tk_Fill3DRectangle(php->display, Tk_WindowId(win),
			       border, o, o, w - 2 * o, t,
			       0, TK_RELIEF_FLAT);
	
	y1 = php->yorg + php->height;
	if( (t = h - o - y1) > 0 )
	    Tk_Fill3DRectangle(php->display, Tk_WindowId(win),
			       border, o, y1, w - 2 * o, t,
			       0, TK_RELIEF_FLAT);
	
	if( (t = php->xorg - o) > 0 )
	    Tk_Fill3DRectangle(php->display, Tk_WindowId(win),
			       border, o, php->yorg, t, php->height,
			       0, TK_RELIEF_FLAT);
	
	x1 = php->xorg + php->width;
	if( (t = w - o - x1) > 0 )
	    Tk_Fill3DRectangle(php->display, Tk_WindowId(win),
			       border, x1, php->yorg, t, php->height,
			       0, TK_RELIEF_FLAT);
    }

    if( (php->flags & REDRAW_SCROLL) != 0 )
	RedrawPhoto(php, php->xorg, php->yorg, php->width, php->height);

    php->flags &= ~REDRAW_PENDING;
}

/*
 * Event procedure for various events on a photo widget:
 * configuration changes, expose and destroy.
 */
static void
PhotoEventProc(clientData, event)
ClientData clientData;
XEvent *event;
{
    register Photo *php;
    XWindowAttributes attrs;

    php = (Photo *) clientData;

    if( event->type == Expose ){
	php->flags |= EXPOSED;
	RedrawPhoto(php, event->xexpose.x, event->xexpose.y,
		    event->xexpose.width, event->xexpose.height);
    } else if( event->type == DestroyNotify ){
	Tcl_DeleteCommand(php->interp, Tk_PathName(php->tkwin));
	php->tkwin = NULL;
	if( (php->flags & REDRAW_PENDING) != 0 )
	    Tk_CancelIdleCall(DisplayPhoto, (ClientData) php);
	Tk_EventuallyFree((ClientData) php, DestroyPhoto);
    } else if( event->type == ConfigureNotify ){
	ComputePhotoGeometry(php);
    } else if( event->type == EnterNotify ){
	php->flags |= ACTIVE;
	if( (php->flags & SET_COLORMAP) != 0 ){
	    /* we actually set the colormap for our top-level ancestor */
	    XGetWindowAttributes(php->display, Tk_WindowId(php->main), &attrs);
	    php->mainCmap = attrs.colormap;
	    XSetWindowColormap(php->display, Tk_WindowId(php->main),
			       php->colors->cmap);
	    php->flags |= RESTORE_CMAP;
	}
	RedrawPhotoFrame(php);
    } else if( event->type == LeaveNotify ){
	if( (php->flags & ACTIVE) != 0 && (php->flags & RESTORE_CMAP) != 0 )
	    XSetWindowColormap(php->display, Tk_WindowId(php->main),
			       php->mainCmap);
	php->flags &= ~(ACTIVE | RESTORE_CMAP);
	RedrawPhotoFrame(php);
    }
}

/*
 * Allocate local image storage for an image block to be displayed.
 */
static void
PhotoAllocImage(php, width, height)
    Photo *php;
    int width, height;
{
    XImage *im;
    int bpp, n;

    im = php->image;
    bpp = php->depth == 1? 1: php->depth <= 8? 8: 32;
    if( im->bits_per_pixel != bpp ){
	im->bits_per_pixel = bpp;
	im->format = bpp == 1? XYBitmap: ZPixmap;
	im->depth = php->depth;
	_XInitImageFuncPtrs(im);
    }
    im->width = width;
    im->height = height;
    im->bytes_per_line = ((bpp * width + 31) >> 3) & ~3;
    n = im->bytes_per_line * height;
    if( im->data == NULL || n > php->imageSize ){
	if( im->data != NULL )
	    ckfree((char *) im->data);
	im->data = ckalloc(n);
	php->imageSize = n;
    }
}

/*
 * Return an opaque key (actually a Photo *) for a photo window named by
 * its window pathname, or NULL if not found.
 */
PhotoHandle
FindPhoto(name)
char *name;
{
    Tcl_HashEntry *entry;

    if( !hash_inited
       || (entry = Tcl_FindHashEntry(&photoHash, name)) == NULL )
	return NULL;
    return (PhotoHandle) Tcl_GetHashValue(entry);
}

/*
 * Procedure to store a block of data in the image.
 * Called from client code using a handle provided by FindPhoto.
 */
void
PhotoPutBlock(handle, block, x, y, w, h)
    PhotoHandle *handle;
    register struct image_block *block;
    int x, y, w, h;
{
    register Photo *php;
    register unsigned long *ldest, mask, word;
    register unsigned char *cdest, *src;
    register ColorTable *ctp;
    register int goff, boff, big_endian;
    XImage *im;
    unsigned long first_bit;
    int bw, bh, n, xx, yy;
    int ww, hh, iw, ih;
    int nb_line, pw3;
    unsigned char *src0, *dst0;
    unsigned char *p24, *q24;

    php = (Photo *) handle;
    ctp = php->colors;
    im = php->image;

    xx = x + w;
    yy = y + h;
    if( (php->flags & USER_IMAGE_GEOM) != 0 ){
	if( xx > php->imageWidth )
	    w = php->imageWidth - x;
	if( yy > php->imageHeight )
	    h = php->imageHeight - y;
    } else if( xx > php->pixelWidth || yy > php->pixelHeight )
	PhotoReallocatePixmap(php, MAX(xx, php->pixelWidth),
			      MAX(yy, php->pixelHeight));
    if( w <= 0 || h <= 0 )
	return;

    if( y < php->ditherY || y == php->ditherY && x < php->ditherX ){
	/* the dithering isn't correct past the start of this block */
	php->ditherX = x;
	php->ditherY = y;
    }

    if( ctp != NULL && (ctp->flags & NEED_ALLOC) != 0 )
	PhotoAllocColors((ClientData) php);

    goff = block->comp_off[1] - block->comp_off[0];
    boff = block->comp_off[2] - block->comp_off[0];

    /*
     * First work out how many bytes we'll need for pixel storage,
     * and allocate more if necessary.
     */
    bw = MIN(block->width, w);
    bh = MIN(block->height, h);
    if( php->ditherLevel != DITHER_BLOCKS )
	PhotoAllocImage(php, bw, bh);

    /* Convert image data to pixel values */
    if( php->ditherLevel > DITHER_NONE ){
	/* Dithering - copy 24-bit data in, converting to monochrome
	   if necessary; then if required, call dither routine */
	if( (php->flags & MONOCHROME) == 0 ){
	    pw3 = php->pixelWidth * 3;
	    p24 = php->pix24 + y * pw3 + x * 3;
	} else {
	    pw3 = php->pixelWidth;
	    p24 = php->pix24 + y * pw3 + x;
	}
	for( hh = h; hh > 0; ){
	    src0 = block->ptr + block->comp_off[0];
	    ih = MIN(hh, bh);
	    hh -= ih;
	    for( ; ih > 0; --ih ){
		q24 = p24;
		for( ww = w; ww > 0; ){
		    iw = MIN(bw, ww);
		    ww -= iw;
		    src = src0;
		    for( ; iw > 0; --iw ){
			if( (php->flags & MONOCHROME) != 0 ){
			    *q24++ = (src[0]*11 + src[goff]*16 + src[boff]*5)
					>> 5;
			} else {
			    *q24++ = src[0];
			    *q24++ = src[goff];
			    *q24++ = src[boff];
			}
			src += block->pixel_size;
		    }
		}
		src0 += block->pitch;
		p24 += pw3;
	    }
	}
	if( php->ditherLevel == DITHER_BLOCKS ){
	    PhotoDither(php, x, y, w, h);
	    return;
	}
    }

    src0 = block->ptr + block->comp_off[0];
    dst0 = (unsigned char *) im->data;
    nb_line = im->bytes_per_line;

    if( (php->flags & MONOCHROME) != 0 ){
	if( php->depth == 1 ){
	    /* monochrome 1-bit per pixel (not dithering at this stage) */
	    ldest = (unsigned long *) dst0;
	    big_endian = im->bitmap_bit_order == MSBFirst;
	    first_bit = big_endian? (1 << (im->bitmap_unit - 1)): 1;
	    for( yy = 0; yy < bh; ++yy ){
		src = src0;
		mask = first_bit;
		word = 0;
		for( xx = 0; xx < bw; ++xx ){
		    if( mask == 0 ){
			*ldest++ = word;
			mask = first_bit;
			word = 0;
		    }
		    if( src[0]*11 + src[goff]*16 + src[boff]*5 > 128*32 )
			word |= mask;
		    mask = big_endian? (mask >> 1): (mask << 1);
		    src += block->pixel_size;
		}
		*ldest++ = word;
		src0 += block->pitch;
	    }

	} else {
	    /* multibit monochrome display */
	    if( im->bits_per_pixel <= 8 ){
		/* monochrome, 2 to 8 bits/pixel */
		for( yy = 0; yy < bh; ++yy ){
		    src = src0;
		    cdest = dst0;
		    for( xx = 0; xx < bw; ++xx ){
			n = (src[0]*11 + src[goff]*16 + src[boff]*5 + 31) >> 5;
			*cdest++ = ctp->red_values[n];
			src += block->pixel_size;
		    }
		    src0 += block->pitch;
		    dst0 += nb_line;
		}
	    } else {
		/* monochrome, 9 to 32 bits/pixel */
		for( yy = 0; yy < bh; ++yy ){
		    src = src0;
		    ldest = (unsigned long *) dst0;
		    for( xx = 0; xx < bw; ++xx ){
			n = (src[0]*11 + src[goff]*16 + src[boff]*5 + 31) >> 5;
			*ldest++ = ctp->red_values[n];
			src += block->pixel_size;
		    }
		    src0 += block->pitch;
		    dst0 += nb_line;
		}
	    }

	}

    } else if( (php->flags & MAP_COLORS) != 0 ){
	if( im->bits_per_pixel <= 8 ){
	    /* PseudoColor or StaticColor with <= 8 bits/pixel */
	    for( yy = 0; yy < bh; ++yy ){
		src = src0;
		cdest = dst0;
		for( xx = 0; xx < bw; ++xx ){
		    n = ctp->red_values[src[0]] + ctp->green_values[src[goff]]
			+ ctp->blue_values[src[boff]];
		    *cdest++ = ctp->pixel_map[n];
		    src += block->pixel_size;
		}
		src0 += block->pitch;
		dst0 += nb_line;
	    }
	} else {
	    /* PseudoColor or StaticColor with 9 to 32 bits/pixel */
	    for( yy = 0; yy < bh; ++yy ){
		src = src0;
		ldest = (unsigned long *) dst0;
		for( xx = 0; xx < bw; ++xx ){
		    n = ctp->red_values[src[0]] + ctp->green_values[src[goff]]
			+ ctp->blue_values[src[boff]];
		    *ldest++ = ctp->pixel_map[n];
		    src += block->pixel_size;
		}
		src0 += block->pitch;
		dst0 += nb_line;
	    }
	}

    } else {
	if( im->bits_per_pixel <= 8 ){
	    /* DirectColor or TrueColor display with <= 8 bits/pixel */
	    for( yy = 0; yy < bh; ++yy ){
		src = src0;
		cdest = dst0;
		for( xx = 0; xx < bw; ++xx ){
		    *cdest++ = ctp->red_values[src[0]]
			     + ctp->green_values[src[goff]]
			     + ctp->blue_values[src[boff]];
		    src += block->pixel_size;
		}
		src0 += block->pitch;
		dst0 += nb_line;
	    }
	} else {
	    /* DirectColor or TrueColor display with 9 to 32 bits/pixel */
	    for( yy = 0; yy < bh; ++yy ){
		src = src0;
		ldest = (unsigned long *) dst0;
		for( xx = 0; xx < bw; ++xx ){
		    *ldest++ = ctp->red_values[src[0]]
			     + ctp->green_values[src[goff]]
			     + ctp->blue_values[src[boff]];
		    src += block->pixel_size;
		}
		src0 += block->pitch;
		dst0 += nb_line;
	    }
	}
    }

    yy = y;
    for( hh = h; hh > 0; hh -= ih ){
	ih = MIN(hh, bh);
	xx = x;
	for( ww = w; ww > 0; ww -= iw ){
	    iw = MIN(bw, ww);
	    XPutImage(php->display, php->pixels, php->monoGC,
		      im, 0, 0, xx, yy, iw, ih);
	    xx += iw;
	}
	yy += ih;
    }

    RedrawImage(php, x, y, w, h);
}

/*
 * Dither a region of the image
 */
static void
PhotoDither(php, x, y, w, h)
    Photo *php;
    int x, y, w, h;
{
    ColorTable *ctp;
    XImage *im;
    int n, nb_line, ll;
    register int i, c, big_endian;
    int xx, yy, col[3];
    register unsigned char *cdest, *src;
    register char *err;
    register unsigned long *ldest, mask, word;
    unsigned char *src0, *dst0;
    char *err0;
    unsigned long first_bit;
    int nlines;

    if( w <= 0 || h <= 0 )
	return;
    ctp = php->colors;
    im = php->image;

    if( ctp != NULL && (ctp->flags & NEED_ALLOC) != 0 )
	PhotoAllocColors((ClientData) php);

    /*
     * Work out if this stuff will be correctly dithered and if it
     * will extend the correctly dithered region.
     */
    if( (y < php->ditherY || y == php->ditherY && x <= php->ditherX)
       && y + h > php->ditherY ){
	/* This block starts inside (or immediately after) the correctly
	   dithered region, so the first scan line at least will be right.
	   Furthermore this block extends into scanline php->ditherY. */
	if( x == 0 && w == php->pixelWidth ){
	    /* doing the full width => correct dithering to the end */
	    php->ditherX = 0;
	    php->ditherY = y + h;
	} else {
	    /* doing partial scanlines => extend correct region by
	       at most one scan line */
	    if( x <= php->ditherX ){
		if( (php->ditherX = x + w) >= php->pixelWidth ){
		    php->ditherX = 0;
		    ++php->ditherY;
		}
	    }
	}
    }

    /*
     * First work out how many lines to do at a time,
     * then how many bytes we'll need for pixel storage,
     * and allocate more if necessary.
     */
    nlines = DITHER_PIXELS / w;	/* # lines to do before updating screen */
    if( nlines < 1 )
	nlines = 1;
    if( nlines > h )
	nlines = h;
    PhotoAllocImage(php, w, nlines);
    nb_line = im->bytes_per_line;

    if( (php->flags & MONOCHROME) == 0 ){
	ll = php->pixelWidth * 3;
	src0 = php->pix24 + y * ll + x * 3;
	err0 = php->error + y * ll + x * 3;
    } else {
	ll = php->pixelWidth;
	src0 = php->pix24 + y * ll + x;
	err0 = php->error + y * ll + x;
    }

    for( ; h > 0; h -= nlines, y += nlines ){
	if( nlines > h )
	    nlines = h;
	dst0 = (unsigned char *) im->data;
	if( php->depth == 1 ){
	    /* set up for 1-bit/pixel monochrome */
	    big_endian = im->bitmap_bit_order == MSBFirst;
	    first_bit = big_endian? (1 << (im->bitmap_unit - 1)): 1;
	}
	for( yy = y; yy < y + nlines; ++yy ){
	    src = src0;
	    err = err0;
	    cdest = dst0;
	    ldest = (unsigned long *) dst0;
	    if( (php->flags & MONOCHROME) == 0 ){
		/* color display */
		for( xx = x; xx < x + w; ++xx ){
		    for( i = 0; i < 3; ++i ){
			c = xx > 0? err[-3] * 7: 0;
			if( yy > 0 ){
			    if( xx > 0 )
				c += err[-ll-3];
			    c += err[-ll] * 5;
			    if( xx + 1 < php->pixelWidth )
				c += err[-ll+3] * 3;
			}
			c = ((c + 8) >> 4) + *src++;
			if( c < 0 )
			    c = 0;
			else if( c > 255 )
			    c = 255;
			col[i] = ctp->color_quant[i][c];
			*err++ = c - col[i];
		    }
		    i = ctp->red_values[col[0]] + ctp->green_values[col[1]]
			+ ctp->blue_values[col[2]];
		    if( (php->flags & MAP_COLORS) != 0 )
			i = ctp->pixel_map[i];
		    if( im->bits_per_pixel <= 8 )
			*cdest++ = i;
		    else
			*ldest++ = i;
		}

	    } else if( im->bits_per_pixel > 1 ){
		/* multibit monochrome */
		for( xx = x; xx < x + w; ++xx ){
		    c = xx > 0? err[-1] * 7: 0;
		    if( yy > 0 ){
			if( xx > 0 )
			    c += err[-ll-1];
			c += err[-ll] * 5;
			if( xx + 1 < php->pixelWidth )
			    c += err[-ll+1] * 3;
		    }
		    c = ((c + 8) >> 4) + *src++;
		    if( c < 0 )
			c = 0;
		    else if( c > 255 )
			c = 255;
		    i = ctp->color_quant[0][c];
		    *err++ = c - i;
		    if( im->bits_per_pixel <= 8 )
			*cdest++ = ctp->red_values[i];
		    else
			*ldest++ = ctp->red_values[i];
		}

	    } else {
		/* 1-bit monochrome */
		word = 0;
		mask = first_bit;
		for( xx = x; xx < x + w; ++xx ){
		    if( mask == 0 ){
			*ldest++ = word;
			mask = first_bit;
			word = 0;
		    }
		    c = xx > 0? err[-1] * 7: 0;
		    if( yy > 0 ){
			if( xx > 0 )
			    c += err[-ll-1];
			c += err[-ll] * 5;
			if( xx + 1 < php->pixelWidth )
			    c += err[-ll+1] * 3;
		    }
		    c = ((c + 8) >> 4) + *src++;
		    if( c < 0 )
			c = 0;
		    else if( c > 255 )
			c = 255;
		    if( c >= 128 ){
			word |= mask;
			*err++ = c - 255;
		    } else
			*err++ = c;
		    mask = big_endian? (mask >> 1): (mask << 1);
		}
		*ldest = word;
	    }
	    src0 += ll;
	    err0 += ll;
	    dst0 += nb_line;
	}
	
	XPutImage(php->display, php->pixels, php->monoGC, im,
		  0, 0, x, y, w, nlines);
	
	RedrawImage(php, x, y, w, nlines);
    }
}

/*
 * Clear the entire image to the blank colour.
 */
void
PhotoBlank(handle)
    PhotoHandle handle;
{
    Photo *php;
    int n;

    php = (Photo *) handle;
    XFillRectangle(php->display, php->pixels, php->blankGC,
		   0, 0, php->pixelWidth, php->pixelHeight);
    if( php->ditherLevel > DITHER_NONE ){
	n = php->pixelWidth * php->pixelHeight;
	if( (php->flags & MONOCHROME) == 0 )
	    n *= 3;
	bzero(php->pix24, n);
	bzero(php->error, n);
	php->ditherX = php->ditherY = 0;
    }
    RedrawImage(php, 0, 0, php->pixelWidth, php->pixelHeight);
}

/*
 * If permitted (i.e. if the user hasn't declared a definite image size),
 * expand the image to be at least widthxheight pixels.
 */
void
PhotoExpand(handle, width, height)
    PhotoHandle handle;
    int width, height;
{
    Photo *php;

    php = (Photo *) handle;

    if( width <= 0 || height <= 0 || (php->flags & USER_IMAGE_GEOM) != 0 )
	return;

    if( php->imageWidth < width )
	php->imageWidth = width;
    if( php->imageHeight < height )
	php->imageHeight = height;
    if( (php->flags & USER_GEOMETRY) == 0 ){
	php->reqWidth = php->imageWidth;
	php->reqHeight = php->imageHeight;
    }
    ComputePhotoGeometry(php);
    RedrawPhotoFrame(php);
}

/*
 * Report the current size of a photo (the image display area,
 * not the window).
 */
void
PhotoGetSize(handle, widthp, heightp)
    PhotoHandle handle;
    int *widthp, *heightp;
{
    Photo *php;

    php = (Photo *) handle;
    if( (php->flags & USER_IMAGE_GEOM) != 0 ){
	*widthp = php->imageWidth;
	*heightp = php->imageHeight;
    } else {
	*widthp = php->pixelWidth;
	*heightp = php->pixelHeight;
    }
}

/*
 * Set the size of a photo to w x h pixels, as if the user had done
 * "photo-name config -image wxh"
 */
void
PhotoSetSize(handle, w, h)
    PhotoHandle handle;
    int w, h;
{
    Photo *php;

    php = (Photo *) handle;

    if( w == 0 && h == 0 )
	php->flags &= ~USER_IMAGE_GEOM;
    else if( w > 0 && h > 0 ){
	php->imageWidth = w;
	php->imageHeight = h;
	php->flags |= USER_IMAGE_GEOM;
	if( (php->flags & USER_GEOMETRY) == 0 ){
	    php->reqWidth = w;
	    php->reqHeight = h;
	}
    }

    ComputePhotoGeometry(php);
    RedrawPhotoFrame(php);
}
