/*
 *=============================================================================
 *                                  tSippPhoto.c
 *-----------------------------------------------------------------------------
 * Rendering callbacks for the Tk photo widget.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippPhoto.c,v 3.0 1993/01/20 06:43:54 markd Rel $
 *=============================================================================
 */

#include "tSippInt.h"
#include "sipp_pixmap.h"
#include "sipp_bitmap.h"

#include "photo.h"

/*
 * Client-data for output callback used to hold a block of data.  The actual
 * set of scanlines to buffer follows this structure.  The structure is setup
 * so it can be rendered in either scan direction.
 *
 */
typedef struct {
    tSippGlob_pt     tSippGlobPtr;
    char            *photoPath;
    PhotoHandle      photoHandle;
    int              xSize;
    int              ySize;
    scanDirection_t  scanDirection;
    PhotoImage       block;
} renderData_t, *renderData_pt;

/*
 * Internal prototypes.
 */
static bool
SetBlankColor _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                           char         *path,
                           u_char       *color));

static bool
ReditherPhoto _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                           char         *path));

static void *
PhotoOutputStart _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                              tSippOutputParms_pt   renderParmsPtr,
                              char                 *handle,
                              char                **comments));

static void
PhotoOutputLine _ANSI_ARGS_((renderData_pt  renderDataPtr,
                             int            y,
                             u_char        *rowPtr));

static void
PhotoOutputBitMap _ANSI_ARGS_((renderData_pt   renderDataPtr,
                               Sipp_bitmap    *bitMapPtr));

static bool
PhotoOutputEnd _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                            tSippOutputParms_pt   renderParmsPtr,
                            renderData_pt         renderDataPtr));

/*=============================================================================
 * SetBlankColor --
 *
 * Set the blank color for an instance of the photo widget.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o path (I) - The photo widget path.
 *   o color (I) - Integer background color.
 * Returns:
 *   TRUE if all is OK, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
SetBlankColor (tSippGlobPtr, path, color)
    tSippGlob_pt  tSippGlobPtr;
    char         *path;
    u_char       *color;
{
    char  colorBuf [10];
    char *command;

    sprintf (colorBuf, "#%02x%02x%02x", color [TSIPP_RED],
             color [TSIPP_GREEN], color [TSIPP_BLUE]);

    command = (char *) alloca (strlen (path) + 30);
    strcpy (command, path);
    strcat (command, " configure -blank ");
    strcat (command, colorBuf);

    if (Tcl_Eval (tSippGlobPtr->interp, command, 0,
                  (char **) NULL) == TCL_ERROR)
        return FALSE;

    return TRUE;

}

/*=============================================================================
 * ReditherPhoto --
 *
 * Redither an instance of the photo widget.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o path (I) - The photo widget path.
 * Returns:
 *   TRUE if all is OK, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
ReditherPhoto (tSippGlobPtr, path)
    tSippGlob_pt  tSippGlobPtr;
    char         *path;
{
    char *command;

    command = (char *) alloca (strlen (path) + 10);
    strcpy (command, path);
    strcat (command, " redither");

    if (Tcl_Eval (tSippGlobPtr->interp, command, 0,
                  (char **) NULL) == TCL_ERROR)
        return FALSE;

    return TRUE;

}

/*=============================================================================
 * PhotoOutputStart --
 *   Start output to a photo widget.  Sets the widget's blank color to be the
 * same as the background color and clears the widget if requested. This
 * routine is pointed to by the photo widget image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o handle (I) - Pointer to the Photo handle for the file to write.
 *   o comments (I) - Comments are not useful with the photo widget, so this
 *     argument is ignored.
 * Returns:
 *   A pointer to be passed back into the other Photo output routines or NULL
 *   an a message in tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static void *
PhotoOutputStart (tSippGlobPtr, renderParmsPtr, handle, comments)
    tSippGlob_pt          tSippGlobPtr;
    tSippOutputParms_pt   renderParmsPtr;
    char                 *handle;
    char                **comments;
{
    PhotoHandle    photoHandle;
    renderData_pt  renderDataPtr;

    photoHandle = FindPhoto (handle);
    if (photoHandle == NULL) {
        Tcl_AppendResult (tSippGlobPtr->interp, "bad photo widget path \"",
                          handle, "\"", (char *) NULL);
        return NULL;
    }
    if (!SetBlankColor (tSippGlobPtr, handle,
                        renderParmsPtr->imgData.backgroundColor))
        return NULL;

    if (renderParmsPtr->clear)
        PhotoBlank (photoHandle);

    /*
     * Set up the client-data and fill in image block information.
     * The image block scanline is not actually allocated here.
     */
    renderDataPtr = (renderData_pt) smalloc (sizeof (renderData_t));

    renderDataPtr->tSippGlobPtr       = tSippGlobPtr;
    renderDataPtr->photoPath          = strdup (handle);
    renderDataPtr->photoHandle        = photoHandle;
    renderDataPtr->xSize              = renderParmsPtr->imgData.xSize;
    renderDataPtr->ySize              = renderParmsPtr->imgData.ySize;
    renderDataPtr->scanDirection      = renderParmsPtr->scanDirection;
    renderDataPtr->block.ptr          = NULL;
    renderDataPtr->block.width        = renderParmsPtr->imgData.ySize;
    renderDataPtr->block.height       = 1;
    renderDataPtr->block.pitch        = 3 * renderParmsPtr->imgData.xSize;
    renderDataPtr->block.pixel_size   = 3;
    renderDataPtr->block.comp_off [0] = 0;
    renderDataPtr->block.comp_off [1] = 1;
    renderDataPtr->block.comp_off [2] = 2;

    return renderDataPtr;

}

/*=============================================================================
 * PhotoOutputLine --
 *   Output a rendered line to a photo widget.  This routine is pointed to by
 * the photo widget image storage class table.
 *
 * Parameters:
 *   o renderDataPtr (I) - A pointer to the Photo rendering clientdata.
 *   o y (I) - The scan line that was just rendered.
 *   o rowPtr (I) - The pixels for the scanline that was just rendered.
 *-----------------------------------------------------------------------------
 */
static void
PhotoOutputLine (renderDataPtr, y, rowPtr)
    renderData_pt  renderDataPtr;
    int            y;
    u_char        *rowPtr;
{
    renderDataPtr->block.ptr = rowPtr;

    PhotoPutBlock (renderDataPtr->photoHandle,
                   &renderDataPtr->block,
                   0, y,
                   renderDataPtr->xSize, 1);

}

/*=============================================================================
 * PhotoOutputBitMap --
 *   Output a SIPP bit map to a photo widget.  This routine is pointed to by
 * the photo widget image storage class table.
 *
 * Parameters:
 *   o renderDataPtr (I) - A pointer to the Photo rendering clientdata.
 *   o bitMapPtr (I) - Pointer to the SIPP bit map structure.
 *-----------------------------------------------------------------------------
 */
static void
PhotoOutputBitMap (renderDataPtr, bitMapPtr)
    renderData_pt   renderDataPtr;
    Sipp_bitmap    *bitMapPtr;
{
    tSippGlob_pt  tSippGlobPtr = renderDataPtr->tSippGlobPtr;
    int           xSize        = renderDataPtr->xSize;
    int           yStart, yEnd, yIncr;
    u_char        lineColor [3], backgroundColor [3];
    int           x, y;
    u_char       *bitRowPtr;
    unsigned      bit;
    u_char       *rowPtr;

    lineColor [TSIPP_RED]   = tSippGlobPtr->lineColor.red * 255;
    lineColor [TSIPP_GREEN] = tSippGlobPtr->lineColor.grn * 255;
    lineColor [TSIPP_BLUE]  = tSippGlobPtr->lineColor.blu * 255;

    backgroundColor [TSIPP_RED]   = tSippGlobPtr->backgroundColor.red * 255;
    backgroundColor [TSIPP_GREEN] = tSippGlobPtr->backgroundColor.grn * 255;
    backgroundColor [TSIPP_BLUE]  = tSippGlobPtr->backgroundColor.blu * 255;


    if (renderDataPtr->scanDirection == TSIPP_SCAN_TOP_DOWN) {
        yStart = renderDataPtr->ySize - 1;
        yEnd   = 0;
        yIncr  = -1;
    } else {
        yStart = 0;
        yEnd   = renderDataPtr->ySize - 1;
        yIncr  = 1;
    }

    renderDataPtr->block.ptr = (u_char *) alloca (renderDataPtr->block.pitch);

    /*
     * Loop setting pixels based on the bit map and outputting rows to 
     * the widget.
     */

    for (y = yStart; y != yEnd; y += yIncr) {
        bitRowPtr = &bitMapPtr->buffer [y * bitMapPtr->width_bytes];
        rowPtr    = renderDataPtr->block.ptr;

        for (x = 0; x < xSize; x++) {
            bit = (bitRowPtr [x >> 3] >> (7-(x & 7))) & 1;
            if (bit) {
                rowPtr [TSIPP_RED]   = lineColor [TSIPP_RED];
                rowPtr [TSIPP_GREEN] = lineColor [TSIPP_GREEN];
                rowPtr [TSIPP_BLUE]  = lineColor [TSIPP_BLUE];
            } else {
                rowPtr [TSIPP_RED]   = backgroundColor [TSIPP_RED];
                rowPtr [TSIPP_GREEN] = backgroundColor [TSIPP_GREEN];
                rowPtr [TSIPP_BLUE]  = backgroundColor [TSIPP_BLUE];
            }
            rowPtr += 3;
        }
        PhotoPutBlock (renderDataPtr->photoHandle,
                       &renderDataPtr->block,
                       0, y,
                       renderDataPtr->xSize, 1);
    }

}

/*=============================================================================
 * PhotoOutputEnd --
 *   Finish up output to a photo widget.  This routine is pointed to by the
 * photo widget image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o renderDataPtr (I) - A pointer to the Photo rendering clientdata.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in
 *  tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
PhotoOutputEnd (tSippGlobPtr, renderParmsPtr, renderDataPtr)
    tSippGlob_pt          tSippGlobPtr;
    tSippOutputParms_pt   renderParmsPtr;
    renderData_pt         renderDataPtr;
{
    bool  status = TRUE;
    
    /*
     * Final updates are handled by the tSippRender code.  If we were scanning
     * from bottom-up, the photo is not dithered, so dither it here.
     */
    if (renderParmsPtr->scanDirection == TSIPP_SCAN_BOTTOM_UP)
        status = ReditherPhoto (tSippGlobPtr, renderDataPtr->photoPath);

    sfree (renderDataPtr->photoPath);
    sfree (renderDataPtr);

    return status;

}

/*=============================================================================
 * TSippPhotoInit --
 *   Initialized the photo widget rendering commands.
 *   table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippPhotoInit (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    static tSippStorageClass_t storageClass = {
        ".",                        /* handlePrefix  */
        1,                          /* prefixSize    */
        TSIPP_SCAN_BOTH,            /* scanDirection */
        FALSE,                      /* bitMapOptimal */
        PhotoOutputStart,
        PhotoOutputLine,
        PhotoOutputBitMap,
        PhotoOutputEnd,
        NULL
    };

    TSippAddStorageClass (tSippGlobPtr, &storageClass);

}


